<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sites;
use App\Models\SiteDocument;
use App\Models\DocumentType;
use App\Models\LinkManagement;
use App\Models\Company;
use App\Models\Project;
use App\Models\EmpCompanyDetails;
use App\Models\SiteHistory;
use App\Models\EmpPersonalDetails;
use App\Models\User;
use App\Models\Role;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\QueryException;


class SiteController extends Controller
{
    public function __construct()
    {
        $this->middleware('Permissions:Sites Maintain')->only(['create', 'edit']);
    }
    public function index(Request $request)
    {
        $query = Sites::where('del', 0);

        // When active=0 is passed, return ALL sites (both active and inactive) without filtering by active status
        // When active is not provided or active=1, return only active sites
        if($request->has('active') && (int)$request->active === 0){
            // Return all sites without active status filter
        }else{
            // Default: only return active sites
            $query->where('active', 1);
        }
        
        $query = $this->applyCustomerWorkspaceFilter($query);
        $query->with('supervisor')->with('Company')->with('project');
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', '%' . $searchTerm . '%')
                ->orWhere('site_state', 'like', '%' . $searchTerm . '%')
                ->orWhere('street_address', 'like', '%' . $searchTerm . '%')
                ->orWhere(function ($nameQuery) use ($searchTerm) {
                    $this->applyNameSearch($nameQuery, $searchTerm, 'supervisor');
                })
                ->orWhereHas('company', function ($subquery) use ($searchTerm) {
                    $subquery->where('name', 'like', '%' . $searchTerm . '%');
                })
                ->orWhereHas('project', function ($subquery) use ($searchTerm) {
                    $subquery->where('title', 'like', '%' . $searchTerm . '%');
                });
            });
        }
        return $this->withCount($query, 'Data get Successfully');
    }
    
 

    public function create(Request $request)
    {
        $get_employes = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->where('access_role', 'SPV')->get();
        $get_companies = Company::where('del', 0)->where('active', 1)->get();
        return view('Sites.create', compact('get_employes', 'get_companies'));
    }

    public function ssoList()
    {
        $userTable = $this->getUserTable();
        // Base query for employee list
        $query = EmpCompanyDetails::where('compeleted', '1')
            ->where('approved', '1')
            ->where('status', '1')
            ->where('del', '0')
            ->with([
                'empPersonalDetails' => function ($query) {
                    $query->select('emp_id', 'first_name', 'middle_name', 'last_name', 'customer_id', 'workspace_id');
                },
                'accessRole' => function ($query) {
                    $query->select('id', 'title', 'code');
                },
                'empTier' => function ($query) {
                    $query->select('id', 'title');
                },
                'accessTier' => function ($query) {
                    $query->select('id', 'title', 'tier_key');
                },
            ])
            ->select('id', 'access_role', 'tier_id');
        // Add Customer and Employee Access Checks
        if ($userTable === 'customer') {
            $query->whereHas('empPersonalDetails', function ($q) {
                $q->where('customer_id', auth()->id())
                    ->where('workspace_id', auth()->user()->current_workspace_id);
            });
        }
        if ($userTable === 'emp') {
            $query->whereHas('empPersonalDetails', function ($q) {
                $q->where('customer_id', auth()->user()->customer_id)
                    ->where('workspace_id', auth()->user()->workspace_id);
            });
        }
        // Execute query
        $empList = $query->get();
        // Filter only SSO roles
        $data = $empList->where('access_role', 'SSO')->values()->toArray();
        // Return response
        return $this->success($data, 'SSO List fetched successfully');
    }

    public function store(Request $request)
    {
       
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'state' => 'required',
            'external_id' => 'nullable',
            'supervisor' => 'required',
            'company_client' => 'required',
            'description' => 'nullable',
            'address' => 'required',
            'latitude' => 'required',
            'longitude' => 'required',
            'area_radius' => 'required',
            'project' => 'nullable',
            'forman_id' => 'required',
            'sso_id'=>'required'
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        try {
            $userTable = $this->getUserTable();
            $auth_id = 0;
            $workspace_id = 0;
            $created_by = 0;
            if ($userTable === "customer") {
                $auth_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            }
            if ($userTable === "emp") {
                $auth_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
                $created_by = auth()->user()->id;
            }
            $validatedData = $validator->validated();
            $Data = Sites::create([
                'title' => $validatedData['title'],
                'site_state' => $validatedData['state'],
                'external_id' => $validatedData['external_id'],
                'supervisor_id' => $validatedData['supervisor'],
                'company_client' => $validatedData['company_client'],
                'forman_id' => $validatedData['forman_id'],
                'sso_id' => $validatedData['sso_id'],   
                'description' => $validatedData['description'],
                'street_address' => $validatedData['address'],
                'latitude' => $validatedData['latitude'],
                'longitude' => $validatedData['longitude'],
                'area_radius' => $validatedData['area_radius'],
                'project_id' => $validatedData['project'],
                'customer_id' => $auth_id,
                'workspace_id' => $workspace_id,
                'created_by' => $created_by
            ]);
            return $this->success($Data, 'Site Created Successfully');
        } catch (QueryException $exception) {
            $errorMessage = $exception->getMessage();
            preg_match("/'([^']+)'/", $errorMessage, $matches);
            $errorField = count($matches) > 1 ? $matches[1] : 'unknown';
            return $this->message('The ' . $errorField . ' field is required.', 500);
        }
    }


    public function siteEdit($id)
    {
        $site = Sites::with(['emppersonaldetails', 'company','siteSafetyOfficer'])->where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not Found.', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        return $this->success($site, 'Site data retrieved successfully');
    }

    public function siteUpdate(Request $request, $id)
    {

        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'state' => 'required',
            'external_id' => 'nullable',
            'supervisor' => 'required',
            'company_client' => 'required',
            'description' => 'nullable',
            'address' => 'required',
            'latitude' => 'required',
            'longitude' => 'required',
            'area_radius' => 'required',
            'project' => 'nullable',
            'forman_id' => 'required',
            'sso_id'=>'required'
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        } else {

            $validatedData = $validator->validated();
            $site = Sites::where('id', $id)->where('del', '0')->first();
            if (!$site) {
                return $this->message('Site not found or already deleted', 404);
            }
            $updateData = [
                'title' => $validatedData['title'],
                'site_state' => $validatedData['state'],
                'external_id' => $validatedData['external_id'],
                'supervisor_id' => $validatedData['supervisor'],
                'company_client' => $validatedData['company_client'],
                'description' => $validatedData['description'],
                'street_address' => $validatedData['address'],
                'latitude' => $validatedData['latitude'],
                'longitude' => $validatedData['longitude'],
                'area_radius' => $validatedData['area_radius'],
                'forman_id' => $validatedData['forman_id'],
                'sso_id' => $validatedData['sso_id'],
                // 'project_id' => $validatedData['project']
            ];
            $userTable = $this->getUserTable();
            if (
                $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
            ) {
                return $this->message('You do not have access to this Site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('You do not have access to this Site', 403);
            }
            $site->update($updateData);
            $getuserdetail = DB::table('users')->where('id', auth()->user()->id)->first();
            $description = "<a href='" . url('/user-profile/' . Auth::id()) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'>" . e($getuserdetail->name ?? '') . "</a> changed the site detail: '" . e($site->title) . "'.";
            $history = [
                'site_id' => $site->id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);
            return $this->success($updateData, 'Site Updated Successfully');
        }
    }

    public function updateStatus($id)
    {
        $site = Sites::where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not found', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        // Toggle the active status
        $site->active = $site->active == 1 ? 0 : 1;
        $site->save();
        return $this->success($site, 'Status Updated Successfully');
    }

    public function destroy($id)
    {
        $site = Sites::where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not found or already deleted', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        Sites::where('id', $id)->update(['del' => '1']);
        $getuserdetail = DB::table('users')->where('id', auth()->user()->id)->first();
        if (!$getuserdetail) {
            return $this->message('User details not found');
        }
        $description = "<a href='" . url('/') . "/user-profile/" . Auth::user()->id . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$getuserdetail->name} </a>  Deleted that site: '{$site->title}'.";

        $history = [
            'site_id' => $site->id,
            'description' => $description,
            'updated_by' => auth()->user()->id,
        ];
        SiteHistory::create($history);
        return $this->message('User details not found', 200);
    }


    public function view($id)
    {
        $site = Sites::with('siteSafetyOfficer','supervisor','forman')->where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not found or deleted', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        // $site = Sites::where('id', $id)->first();
        $sitehistories = SiteHistory::where('site_id', $id)->get();
        $company = Company::where('id', $site->company_client)->first();
        $project = Project::where('id', $site->project_id)->first();

        if ($site->created_by == 0) {
            $created_by = User::where('id', $site->customer_id)->select('name as name')->first();
        }else {
            $created_by = EmpPersonalDetails::where('emp_id', $site->created_by)->first();
        }
        
        $empsuper = EmpPersonalDetails::where('emp_id', $site->supervisor_id)->first();

        $site['company'] = $company;
        $site['project'] = $project;
        $site['created_by'] = $created_by;
        $site['sitehistories'] = $sitehistories;
        
        // Load site documents with roles
        $siteDocuments = SiteDocument::where('site_id', $id)
            ->orderBy('created_at', 'desc')
            ->get();
        
            // Load role details, document type, and subcontractor for each document
        foreach ($siteDocuments as $doc) {
            if ($doc->role_ids && is_array($doc->role_ids)) {
                $doc->roles = Role::whereIn('id', $doc->role_ids)->get();
            } else {
                $doc->roles = collect([]);
            }
            
            // Load document type relationship
            if ($doc->document_type) {
                $doc->documentTypeDetail = DocumentType::find($doc->document_type);
            }

            // Load subcontractor relationship
            if ($doc->subcontractors) {
                $doc->subcontractorDetail = LinkManagement::with(['roleId', 'tierId'])->find($doc->subcontractors);
            }
        }
        
        $site['documents'] = $siteDocuments;

        return response()->json([
            'data' => $site
        ]);
        // return view('Sites.view', compact('site','company','emppersonal','empsuper','sitehistories'));
    }

    /**
     * Create/Store a site document
     */
    public function siteDocument(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255',
                'site_id' => 'required|integer|exists:sites,id',
                'document_type' => 'nullable|integer|exists:document_types,id',
                'roles' => 'nullable|array', // Array of role IDs
                'roles.*' => 'integer|exists:roles,id',
                'subcontractors' => 'nullable|integer|exists:link_management,id',
                'sign_requires' => 'nullable|integer|in:0,1',
                'signature_timing' => 'nullable|integer|in:0,1'
            ]);

            if ($validator->fails()) {
                return $this->message($validator->errors(), 422);
            }

            $userTable = $this->getUserTable();
            $site = Sites::find($request->site_id);
            
            if (!$site) {
                return $this->message('Site not found', 404);
            }

            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Create site document
            $siteDocument = new SiteDocument();
            $siteDocument->title = $request->title;
            $siteDocument->site_id = $request->site_id;
            $siteDocument->document_type = $request->document_type ?? null;
            $siteDocument->uploaded_by = auth()->user()->id;
            $siteDocument->uploaded_by_type = $userTable; // Set 'customer' or 'emp'
            $siteDocument->subcontractors = $request->subcontractors ?? null;
            $siteDocument->sign_requires = $request->sign_requires ?? null;
            $siteDocument->signature_timing = $request->signature_timing ?? null;

            // Handle file upload
            $document = null;
            if ($request->hasFile('file')) {
                $document = $this->handleFileImageUpload($request, 'SiteDocuments');
                $siteDocument->file = $document['path'] ?? null;
            }

            // Store role IDs as JSON array
            if ($request->has('roles') && is_array($request->roles)) {
                $siteDocument->role_ids = $request->roles;
            }

            $siteDocument->save();

            // Get uploaded by user info
            if ($userTable === 'emp') {
                $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                    ->select('first_name', 'middle_name', 'last_name')
                    ->first();
                $fullName = trim(($added_by->first_name ?? '') . " " . ($added_by->middle_name ?? '') . " " . ($added_by->last_name ?? ''));
            } else {
                $added_by = User::where('id', Auth::user()->id)->select('name')->first();
                $fullName = $added_by->name ?? '';
            }

            // Create site history entry
            $description = "<a href='" . url('/user-profile/' . Auth::user()->id) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$fullName} </a> uploaded a new document: '{$siteDocument->title}' to site: {$site->title}.";
            $history = [
                'site_id' => $siteDocument->site_id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);

            // Load role details, document type, and subcontractor for the document
            if ($siteDocument->role_ids && is_array($siteDocument->role_ids)) {
                $siteDocument->roles = Role::whereIn('id', $siteDocument->role_ids)->get();
            } else {
                $siteDocument->roles = collect([]);
            }
            
            // Load document type relationship
            if ($siteDocument->document_type) {
                $siteDocument->documentTypeDetail = DocumentType::find($siteDocument->document_type);
            }

            // Load subcontractor relationship
            if ($siteDocument->subcontractors) {
                $siteDocument->subcontractorDetail = LinkManagement::with(['roleId', 'tierId'])->find($siteDocument->subcontractors);
            }

            return $this->success($siteDocument, 'Site document saved successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error saving site document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get all documents for a site
     */
    public function siteDocuments(Request $request, $site_id)
    {
        try {
            $site = Sites::find($site_id);
            
            if (!$site) {
                return $this->message('Site not found', 404);
            }

            $userTable = $this->getUserTable();
            
            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Get documents with uploaded by info and roles
            $documents = SiteDocument::where('site_id', $site_id)
                ->orderBy('created_at', 'desc')
                ->get();

            // Add uploaded_by user information and roles
            foreach ($documents as $doc) {
                // Get user info based on uploaded_by_type
                if ($doc->uploaded_by_type == 'customer') {
                    $user = User::find($doc->uploaded_by);
                    if ($user) {
                        $doc->uploaded_by_name = $user->name;
                    }
                } else {
                    $emp = EmpPersonalDetails::where('emp_id', $doc->uploaded_by)->first();
                    if ($emp) {
                        $doc->uploaded_by_name = trim(($emp->first_name ?? '') . " " . ($emp->middle_name ?? '') . " " . ($emp->last_name ?? ''));
                    }
                }

                // Load roles from role_ids JSON array
                if ($doc->role_ids && is_array($doc->role_ids)) {
                    $doc->roles = Role::whereIn('id', $doc->role_ids)->get();
                } else {
                    $doc->roles = collect([]);
                }
                
                // Load document type relationship
                if ($doc->document_type) {
                    $doc->documentTypeDetail = DocumentType::find($doc->document_type);
                }

                // Load subcontractor relationship
                if ($doc->subcontractors) {
                    $doc->subcontractorDetail = LinkManagement::with(['roleId', 'tierId'])->find($doc->subcontractors);
                }
            }

            return $this->success($documents, 'Site documents retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error retrieving site documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Update a site document
     */
    public function updateSiteDocument(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'document_id' => 'required|exists:site_documents,id',
                'title' => 'required|string|max:255',
                'document_type' => 'nullable|integer|exists:document_types,id',
                'roles' => 'nullable|array',
                'roles.*' => 'integer|exists:roles,id'
            ]);

            if ($validator->fails()) {
                return $this->message($validator->errors(), 422);
            }

            $siteDocument = SiteDocument::find($request->document_id);
            
            if (!$siteDocument) {
                return $this->message('Site document not found', 404);
            }

            $site = Sites::find($siteDocument->site_id);
            
            if (!$site) {
                return $this->message('Site not found for the document', 404);
            }

            $userTable = $this->getUserTable();
            
            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            $oldTitle = $siteDocument->title;

            // Update document fields
            $siteDocument->title = $request->title;
            if ($request->has('document_type')) {
                $siteDocument->document_type = $request->document_type;
            }
            if ($request->has('subcontractors')) {
                $siteDocument->subcontractors = $request->subcontractors;
            }
            if ($request->has('sign_requires')) {
                $siteDocument->sign_requires = $request->sign_requires;
            }
            if ($request->has('signature_timing')) {
                $siteDocument->signature_timing = $request->signature_timing;
            }

            // Handle file upload if provided
            if ($request->hasFile('file')) {
                // Delete old file if exists
                if ($siteDocument->file && file_exists(public_path($siteDocument->file))) {
                    unlink(public_path($siteDocument->file));
                }
                $document = $this->handleFileImageUpload($request, 'SiteDocuments');
                $siteDocument->file = $document['path'] ?? null;
            }

            // Update role IDs as JSON array if provided
            if ($request->has('roles') && is_array($request->roles)) {
                $siteDocument->role_ids = $request->roles;
            }

            $siteDocument->save();

            // Get user info for history
            if ($userTable === 'emp') {
                $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                    ->select('first_name', 'middle_name', 'last_name')
                    ->first();
                $fullName = trim(($added_by->first_name ?? '') . " " . ($added_by->middle_name ?? '') . " " . ($added_by->last_name ?? ''));
            } else {
                $added_by = User::where('id', Auth::user()->id)->select('name')->first();
                $fullName = $added_by->name ?? '';
            }

            // Create site history entry
            $description = "<a href='" . url('/user-profile/' . Auth::user()->id) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$fullName} </a> updated the document title from: '{$oldTitle}' to: '{$request->title}'.";
            $history = [
                'site_id' => $site->id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);

            // Load role details, document type, and subcontractor for the document
            if ($siteDocument->role_ids && is_array($siteDocument->role_ids)) {
                $siteDocument->roles = Role::whereIn('id', $siteDocument->role_ids)->get();
            } else {
                $siteDocument->roles = collect([]);
            }
            
            // Load document type relationship
            if ($siteDocument->document_type) {
                $siteDocument->documentTypeDetail = DocumentType::find($siteDocument->document_type);
            }

            // Load subcontractor relationship
            if ($siteDocument->subcontractors) {
                $siteDocument->subcontractorDetail = LinkManagement::with(['roleId', 'tierId'])->find($siteDocument->subcontractors);
            }

            return $this->success($siteDocument, 'Site document updated successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error updating site document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Delete a site document
     */
    public function deleteSiteDocument(Request $request)
    {
        try {
            $siteDocument = SiteDocument::find($request->id);
            
            if (!$siteDocument) {
                return $this->message('Document not found', 404);
            }

            $site = Sites::find($siteDocument->site_id);
            
            if (!$site) {
                return $this->message('Site not found for the document', 404);
            }

            $userTable = $this->getUserTable();
            
            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Get user info for history
            if ($userTable === 'emp') {
                $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                    ->select('first_name', 'middle_name', 'last_name')
                    ->first();
                $fullName = trim(($added_by->first_name ?? '') . " " . ($added_by->middle_name ?? '') . " " . ($added_by->last_name ?? ''));
            } else {
                $added_by = User::where('id', Auth::user()->id)->select('name')->first();
                $fullName = $added_by->name ?? '';
            }

            // Create site history entry
            $description = "<a href='" . url('/user-profile/' . Auth::user()->id) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$fullName} </a> deleted the document: '{$siteDocument->title}' from site: {$site->title}.";
            $history = [
                'site_id' => $site->id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);

            // Delete the file if it exists
            if ($siteDocument->file && file_exists(public_path($siteDocument->file))) {
                unlink(public_path($siteDocument->file));
            }

            // Delete the document (roles are stored in JSON, no need to detach)
            $siteDocument->delete();

            return $this->message('Document deleted successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error deleting site document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get all document types for dropdown
     */
    public function getDocumentTypes()
    {
        try {
            $documentTypes = DocumentType::where('del', '0')
                ->orderBy('title', 'asc')
                ->select('id', 'title')
                ->get();

            return $this->success($documentTypes, 'Document types retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error retrieving document types: ' . $e->getMessage(), 500);
        }
    }
}
