<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use Illuminate\Http\Request;
use Stripe\Stripe;
use Stripe\Customer;
use Stripe\Subscription;
use Stripe\Exception\ApiErrorException;
use App\Models\User;
use GrahamCampbell\ResultType\Success;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Stripe\PaymentMethod;
use Stripe\Invoice;
use App\Models\EmpCompanyDetails;
use Carbon\Carbon;
class SubscriptionController extends Controller
{
    public function __construct()
    {
        Stripe::setApiKey(config('stripe.secret'));
    }

    public function subscribe(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'stripe_token'  => 'required|string',
            'plan_id'       => 'required|integer|exists:plans,id',
            'customer_id'   => 'required|integer|exists:users,id',
            'interval'       => 'required|in:monthly,yearly',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors(),
            ], 422);
        }

        $user = User::find($request->input('customer_id'));
        $plan = Plan::find($request->input('plan_id'));
        $planType = $request->input('interval');

        $priceId = null;

        if ($planType == 'monthly') {
            $priceId = $plan->price_key;
        } elseif ($planType == 'yearly') {
            $priceId = $plan->price_key_yearly;
        }

        try {
            if ($user->cus_id) {
                $customer = \Stripe\Customer::retrieve($user->cus_id);
            } else {
                $paymentMethod = \Stripe\PaymentMethod::create([
                    'type' => 'card',
                    'card' => [
                        'token' => $request->input('stripe_token'),
                    ],
                ]);

                $customer = $this->findOrCreateCustomer(
                    $user->email,
                    $user->name,
                    $paymentMethod->id
                );

                $paymentMethod->attach(['customer' => $customer->id]);
                \Stripe\Customer::update($customer->id, [
                    'invoice_settings' => [
                        'default_payment_method' => $paymentMethod->id,
                    ],
                ]);

                $user->update(['cus_id' => $customer->id]);
            }

            $subscription = \Stripe\Subscription::create([
                'customer'         => $customer->id,
                'items'            => [['price' => $priceId]],  
                'payment_behavior' => 'default_incomplete',
                'payment_settings' => [
                    'save_default_payment_method' => 'on_subscription',
                ],
            ]);

            $invoiceId = $subscription->latest_invoice;
            $invoice   = \Stripe\Invoice::retrieve(
                $invoiceId,
                ['expand' => ['payment_intent']] 
            );

            $clientSecret = $invoice->payment_intent->client_secret ?? null;

            return $this->success([
                'subscription_id'   => $subscription->id,
                'client_secret'     => $clientSecret,
                'status'            => $subscription->status,
                'latest_invoice_id' => $invoiceId,
            ], 'Subscription Subscribed Successfully');
        } catch (ApiErrorException $e) {
            Log::error('Stripe Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error'   => $e->getMessage(),
            ], 400);
        } catch (\Exception $e) {
            Log::error('Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error'   => 'Subscription processing failed',
            ], 500);
        }
    }


    private function findOrCreateCustomer($email, $name, $paymentMethodId)
    {
        $customers = Customer::all([
            'email' => $email,
            'limit' => 1
        ]);

        if (!empty($customers->data)) {
            $customer = $customers->data[0];

            $this->attachPaymentMethod($customer->id, $paymentMethodId);

            return $customer;
        }

        return Customer::create([
            'email' => $email,
            'name' => $name,
            'payment_method' => $paymentMethodId,
        ]);
    }

    private function attachPaymentMethod($customerId, $paymentMethodId)
    {
        try {
            $paymentMethod = \Stripe\PaymentMethod::retrieve($paymentMethodId);
            $paymentMethod->attach(['customer' => $customerId]);

            Customer::update($customerId, [
                'invoice_settings' => [
                    'default_payment_method' => $paymentMethodId
                ]
            ]);
        } catch (ApiErrorException $e) {
            Log::warning('Error attaching payment method: ' . $e->getMessage());
        }
    }



    public function getSubscription(Request $request)
    {
        $validated = $request->validate([
            'subscription_id' => 'required|string'
        ]);

        try {
            $subscription = Subscription::retrieve([
                'id' => $validated['subscription_id'],
                'expand' => ['customer', 'items.data.price.product']
            ]);

            return response()->json($subscription);
        } catch (ApiErrorException $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 400);
        }
    }

    public function checkUserExpiry()
    {
        $now = Carbon::now();
    
        // 1. Get expired admins
        $expiredAdmins = User::whereNotNull('plan_expire_date')
            ->where('plan_expire_date', '<', $now)
            ->get();
    
        $expiredAdminEmails = [];
        $expiredAdminIds = [];
    
        foreach ($expiredAdmins as $admin) {
            $admin->tokens()->delete(); 
            $admin->active_plan_id = null;
            $admin->plan_expire_date = null;
            $admin->save();
    
            $expiredAdminEmails[] = $admin->email;
            $expiredAdminIds[] = $admin->id;
        }
    
        $expiredEmployees = EmpCompanyDetails::whereIn('customer_id', $expiredAdminIds)->get();
        $expiredEmployeeEmails = [];
    
        foreach ($expiredEmployees as $employee) {
            $employee->tokens()->delete();
            $expiredEmployeeEmails[] = $employee->employee_email;
        }
    
        return response()->json([
            'message' => 'Expired users logged out successfully.',
            'expired_admins_count' => count($expiredAdminEmails),
            'expired_admins_emails' => $expiredAdminEmails,
            'expired_employees_count' => $expiredEmployees->count(),
            'expired_employees_emails' => $expiredEmployeeEmails,
        ]);
    }
    
    // public function updateSubscription(Request $request)
    // {
    //     $validated = $request->validate([
    //         'subscription_id' => 'required|string',
    //         'price_id' => 'required|string'
    //     ]);

    //     try {
    //         $subscription = Subscription::retrieve($validated['subscription_id']);

    //         $updatedSubscription = Subscription::update($validated['subscription_id'], [
    //             'items' => [
    //                 [
    //                     'id' => $subscription->items->data[0]->id,
    //                     'price' => $validated['price_id'],
    //                 ],
    //             ],
    //         ]);

    //         return response()->json($updatedSubscription);
    //     } catch (ApiErrorException $e) {
    //         return response()->json([
    //             'error' => $e->getMessage()
    //         ], 400);
    //     }
    // }
}
