<?php

namespace App\Models;

class Defect extends BaseModel
{
    protected $fillable = [
        'project_id',
        'site_id',
        'assigned_emp_id',
        'assignment_type',
        'subcontractor_id',
        'assigned_subcontractor_emp_id',
        'customer_id',
        'workspace_id',
        'title',
        'description',
        'priority',
        'status',
        'due_date',
        'completion_description',
        'completed_at',
        'reject_reason',
        'created_by',
        'created_by_type',
        'del',
    ];

    protected $casts = [
        'due_date' => 'date',
        'completed_at' => 'datetime',
    
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
        // Hide the individual relations to prevent duplication
        'createdByEmployee',
        'createdByCustomer',
    ];

    // Add these to appends to automatically include in JSON
    protected $appends = [
        'creator_info',
        'creator_type',
        'assigned_employee'
    ];

    // Existing relations remain the same...
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id')->select('id', 'title', 'description');
    }

    public function site()
    {
        return $this->belongsTo(Sites::class, 'site_id')->select('id', 'title', 'description', 'longitude', 'latitude');
    }

    public function employee()
    {
        return $this->belongsTo(EmpCompanyDetails::class, 'assigned_emp_id')->select('id', 'employee_email');
    }

    public function employeePersonalDetails()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'assigned_emp_id', 'emp_id')
            ->select('emp_id', 'first_name', 'middle_name', 'last_name');
    }

    public function assignedSubcontractorEmployeePersonalDetails()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'assigned_subcontractor_emp_id', 'emp_id')
            ->select('emp_id', 'first_name', 'middle_name', 'last_name');
    }

    public function images()
    {
        return $this->hasMany(DefectImage::class, 'defect_id');
    }

    public function histories()
    {
        return $this->hasMany(DefectHistory::class, 'defect_id');
    }

    public function createdByCustomer()
    {
        return $this->belongsTo(User::class, 'created_by')->select('id', 'name' , 'email');
    }

    public function createdByEmployee()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'created_by', 'emp_id')
            ->select('emp_id', 'first_name', 'middle_name', 'last_name');
    }

    public function subcontractor()
    {
        return $this->belongsTo(\App\Models\LinkManagement::class, 'subcontractor_id')
            ->select('id', 'name', 'email', 'phone', 'company_name');
    }

    public function assignedSubcontractorEmployee()
    {
        return $this->belongsTo(EmpCompanyDetails::class, 'assigned_subcontractor_emp_id')->select('id', 'employee_email');
    }

    // NEW ACCESSORS - These will automatically be included in JSON output
    public function getCreatorInfoAttribute()
    {
        if ($this->created_by_type === 'employee') {
            return $this->createdByEmployee;
        } else {
            return $this->createdByCustomer;
        }
    }

    public function getCreatorTypeAttribute()
    {
        return $this->created_by_type;
    }

    public function getAssignedEmployeeAttribute()
    {
        return $this->employeePersonalDetails ?? $this->assignedSubcontractorEmployeePersonalDetails;
    }

    // Keep your existing helper methods...
    public function getAssignedEntity()
    {
        switch ($this->assignment_type) {
            case 'internal_employee':
                return [
                    'type' => 'internal_employee',
                    'employee' => $this->employee,
                    'employee_details' => $this->employeePersonalDetails
                ];
            case 'subcontractor':
                return [
                    'type' => 'subcontractor',
                    'subcontractor' => $this->subcontractor
                ];
            case 'subcontractor_employee':
                return [
                    'type' => 'subcontractor_employee',
                    'subcontractor' => $this->subcontractor,
                    'employee' => $this->assignedSubcontractorEmployee,
                    'employee_details' => $this->assignedSubcontractorEmployeePersonalDetails
                ];
            default:
                return null;
        }
    }

    public function getAssigneeName()
    {
        $assigned = $this->getAssignedEntity();
        if (!$assigned) return 'Not Assigned';

        switch ($assigned['type']) {
            case 'internal_employee':
                if ($assigned['employee_details']) {
                    return $assigned['employee_details']->first_name . ' ' . $assigned['employee_details']->last_name;
                }
                return 'Internal Employee';
            case 'subcontractor':
                return $assigned['subcontractor'] ? $assigned['subcontractor']->name : 'Subcontractor';
            case 'subcontractor_employee':
                $name = $assigned['subcontractor'] ? $assigned['subcontractor']->name : 'Subcontractor';
                if ($assigned['employee_details']) {
                    $name .= ' - ' . $assigned['employee_details']->first_name . ' ' . $assigned['employee_details']->last_name;
                }
                return $name;
            default:
                return 'Unknown';
        }
    }
}
