<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class FAQ extends BaseModel
{
    use HasFactory;

    protected $table = 'faqs';

    protected $fillable = [
        'title',
        'description',
        'module_id',
        'page_slug',
        'sort_order',
        'status'
    ];

    protected $casts = [
        'status' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Scope to get active FAQs
     */
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    /**
     * Scope to order by sort order
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order', 'asc')->orderBy('created_at', 'asc');
    }

    /**
     * Scope to get inactive FAQs
     */
    public function scopeInactive($query)
    {
        return $query->where('status', 0);
    }

    /**
     * Check if FAQ is active
     */
    public function isActive()
    {
        return $this->status == 1;
    }

    /**
     * Check if FAQ is inactive
     */
    public function isInactive()
    {
        return $this->status == 0;
    }

    /**
     * Get the module that this FAQ belongs to
     */
    public function module()
    {
        return $this->belongsTo(PermissionsModules::class, 'module_id', 'id');
    }

    /**
     * Get all FAQ items for this FAQ group
     */
    public function items()
    {
        return $this->hasMany(FAQItem::class, 'faq_id', 'id');
    }

    /**
     * Get only active FAQ items for this FAQ group
     */
    public function activeItems()
    {
        return $this->hasMany(FAQItem::class, 'faq_id', 'id')->active()->ordered();
    }

    /**
     * Get the count of active FAQ items
     */
    public function getActiveItemsCountAttribute()
    {
        return $this->activeItems()->count();
    }

    /**
     * Check if FAQ has any active items
     */
    public function hasActiveItems()
    {
        return $this->activeItems()->exists();
    }
} 