<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Subscription extends BaseModel
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'user_id',
        'order_id',
        'plan_id',
        'name',
        'stripe_id',
        'stripe_status',
        'stripe_price',
        'stripe_product',
        'stripe_customer',
        'quantity',
        'trial_ends_at',
        'current_period_start',
        'current_period_end',
        'ends_at',
        'cancel_at_period_end',
        'interval',
        'amount',
        'currency',
        'is_recurring',
        'metadata',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'trial_ends_at' => 'datetime',
        'current_period_start' => 'datetime',
        'current_period_end' => 'datetime',
        'ends_at' => 'datetime',
        'cancel_at_period_end' => 'boolean',
        'is_recurring' => 'boolean',
        'metadata' => 'array',
    ];

    protected $hidden = [
        'stripe_price',
        'stripe_customer',
        'stripe_product',
        'stripe_id',
        'metadata'
    ];

    /**
     * Get the user that owns the subscription.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the plan associated with the subscription.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function plan()
    {
        return $this->belongsTo(Plan::class);
    }

    /**
     * Get the order associated with the subscription.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Determine if the subscription is active.
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->stripe_status === 'active' || $this->stripe_status === 'trialing';
    }

    /**
     * Determine if the subscription is on trial.
     *
     * @return bool
     */
    public function onTrial()
    {
        return $this->stripe_status === 'trialing';
    }

    /**
     * Determine if the subscription is past due.
     *
     * @return bool
     */
    public function pastDue()
    {
        return $this->stripe_status === 'past_due';
    }

    /**
     * Determine if the subscription is canceled.
     *
     * @return bool
     */
    public function canceled()
    {
        return !is_null($this->ends_at);
    }

    /**
     * Determine if the subscription has ended.
     *
     * @return bool
     */
    public function ended()
    {
        return $this->canceled() && $this->ends_at->isPast();
    }

    /**
     * Determine if the subscription is set to cancel at period end.
     *
     * @return bool
     */
    public function willCancelAtPeriodEnd()
    {
        return $this->cancel_at_period_end === true;
    }

    /**
     * Determine if the subscription can be modified (not canceled or set to cancel).
     *
     * @return bool
     */
    public function canBeModified()
    {
        return $this->isActive() && !$this->willCancelAtPeriodEnd();
    }

    /**
     * Get days remaining in the current period.
     *
     * @return int
     */
    public function daysRemaining()
    {
        if (!$this->current_period_end) {
            return 0;
        }

        return Carbon::now()->diffInDays($this->current_period_end, false);
    }
}
