<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class WHSQEContentOption extends BaseModel
{
    use HasFactory;

    protected $table = 'whsqe_content_options';

    protected $fillable = [
        'content_meta_id',
        'option_id',
        'option_value',
        'option_type',
        'option_label',
        'main_heading',
        'group_order',
        'level',
        'level_id',
        'customer_id',
        'workspace_id'
    ];

    protected $casts = [
        'group_order' => 'integer',
        'level_id' => 'integer'
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];

    /**
     * Get the content meta this option belongs to
     */
    public function contentMeta()
    {
        return $this->belongsTo(WHSQETableOfContentMeta::class, 'content_meta_id');
    }

    /**
     * Scope to filter by customer and workspace
     */
    public function scopeByCustomerAndWorkspace($query, $customerId, $workspaceId)
    {
        return $query->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId);
    }

    /**
     * Scope to filter by option type
     */
    public function scopeByType($query, $type)
    {
        return $query->where('option_type', $type);
    }

    /**
     * Scope to filter by main heading
     */
    public function scopeByMainHeading($query, $heading)
    {
        return $query->where('main_heading', $heading);
    }

    /**
     * Scope to filter by level
     */
    public function scopeByLevel($query, $level)
    {
        return $query->where('level', $level);
    }

    /**
     * Scope to filter by level and level ID
     */
    public function scopeByLevelAndId($query, $level, $levelId)
    {
        return $query->where('level', $level)->where('level_id', $levelId);
    }

    /**
     * Scope to order by group order
     */
    public function scopeOrderByGroup($query)
    {
        return $query->orderBy('main_heading')->orderBy('group_order');
    }

    /**
     * Check if option is checked/selected
     */
    public function isChecked()
    {
        return in_array($this->option_value, ['checked', 'selected', 'true', '1']);
    }

    /**
     * Get human readable option value
     */
    public function getDisplayValueAttribute()
    {
        switch ($this->option_value) {
            case 'checked':
            case 'true':
            case '1':
                return 'Yes';
            case 'unchecked':
            case 'false':
            case '0':
                return 'No';
            default:
                return $this->option_value;
        }
    }

    /**
     * Get grouped options by main heading
     */
    public static function getGroupedOptions($contentMetaId)
    {
        return self::where('content_meta_id', $contentMetaId)
            ->orderBy('main_heading')
            ->orderBy('group_order')
            ->get()
            ->groupBy('main_heading');
    }

    /**
     * Get options by level
     */
    public static function getOptionsByLevel($contentMetaId, $level)
    {
        return self::where('content_meta_id', $contentMetaId)
            ->where('level', $level)
            ->orderBy('main_heading')
            ->orderBy('group_order')
            ->get();
    }

    /**
     * Get options by level and level ID
     */
    public static function getOptionsByLevelAndId($contentMetaId, $level, $levelId)
    {
        return self::where('content_meta_id', $contentMetaId)
            ->where('level', $level)
            ->where('level_id', $levelId)
            ->orderBy('main_heading')
            ->orderBy('group_order')
            ->get();
    }
} 