<?php

namespace Database\Seeders;

use App\Models\EmpTeam;
use App\Models\User;
use App\Models\Workspace;
use App\Models\Role;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class CustomerConstructionRolesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        // Get all customers (user_type = 2) that are active and not deleted
        $customers = User::where('user_type', config('constants.user_types.customer'))
            ->where('active_status', 1)
            ->where('del', '0')
            ->get();

        if ($customers->isEmpty()) {
            echo "No customers found to seed roles for.\n";
            return;
        }

        echo "Found {$customers->count()} customers. Starting to seed construction roles...\n";

        foreach ($customers as $customer) {
            try {
                // Get the customer's workspace
                $workspace = Workspace::where('customer_id', $customer->id)->first();

                if (!$workspace) {
                    echo "Warning: No workspace found for customer {$customer->name} (ID: {$customer->id}). Skipping...\n";
                    continue;
                }

                // Create/update construction roles for this customer
                $createdRoles = $this->createConstructionRoles($customer->id, $workspace->id);

                echo "Successfully created/updated " . count($createdRoles) . " roles for customer {$customer->name} (ID: {$customer->id})\n";
            } catch (\Exception $e) {
                echo "Error processing customer {$customer->name} (ID: {$customer->id}): " . $e->getMessage() . "\n";
            }
        }

        echo "Finished seeding construction roles for all customers.\n";
    }

    /**
     * Create construction roles for a specific customer and workspace
     * This function mirrors the createConstructionRoles function from HelperTrait
     */
    public function createConstructionRoles($customer_id, $workspace_id, $created_by = null)
    {
        $roles = [
            // Existing roles with original codes
            ['title' => 'Admin', 'code' => 'A-1'],
            ['title' => 'HR Manager', 'code' => 'HR'],
            ['title' => 'Employee', 'code' => 'EMP'],
            ['title' => 'Supervisor', 'code' => 'SPV'],
            ['title' => 'Director', 'code' => 'DIR'],
            // Construction roles requested by user
            ['title' => 'Safety Manager', 'code' => 'SM'],
            ['title' => 'Site Safety Officer', 'code' => 'SSO'],
            ['title' => 'Leading Hand', 'code' => 'LH'],
            ['title' => 'Foreman', 'code' => 'FM'],
            ['title' => 'Tradesman', 'code' => 'TM'],
            ['title' => 'Labourer', 'code' => 'LAB'],
            ['title' => 'Visitor', 'code' => 'VIS'],
            ['title' => 'Head Office', 'code' => 'HO'],
            ['title' => 'Formworker', 'code' => 'FW'],
            ['title' => 'Subcontractor', 'code' => 'SUB'],
            ['title' => 'Administrative Officer', 'code' => 'AO'],
            ['title' => 'Patcher', 'code' => 'PAT'],
            ['title' => 'Educated Skilled Labourer', 'code' => 'ESL'],
            ['title' => 'Jumpform Formworker', 'code' => 'JFW'],
            ['title' => 'Formworker Stripper m2', 'code' => 'FWS'],
            ['title' => 'Formwork Stripping P/H', 'code' => 'FSP'],
            ['title' => 'Principal Contractor', 'code' => 'PC'],
            ['title' => 'Principal Contractor Scaffolder', 'code' => 'PCS'],
            ['title' => 'Principal Contractor Steelfixer', 'code' => 'PCSF'],
            ['title' => 'Principal Contractor PT', 'code' => 'PCPT'],
            ['title' => 'Principal Contractor Other Trades', 'code' => 'PCOT'],
            ['title' => 'Other Contractors', 'code' => 'OC'],

            // Additional Australian construction roles
            ['title' => 'Site Manager', 'code' => 'SMGR'],
            ['title' => 'Project Manager', 'code' => 'PM'],
            ['title' => 'Construction Manager', 'code' => 'CM'],
            ['title' => 'Quality Assurance Manager', 'code' => 'QAM'],
            ['title' => 'Site Engineer', 'code' => 'SE'],
            ['title' => 'Quantity Surveyor', 'code' => 'QS'],
            ['title' => 'Health and Safety Coordinator', 'code' => 'HSC'],
            ['title' => 'Plant Operator', 'code' => 'PO'],
            ['title' => 'Crane Operator', 'code' => 'CO'],
            ['title' => 'Excavator Operator', 'code' => 'EO'],
            ['title' => 'Dozer Operator', 'code' => 'DO'],
            ['title' => 'Grader Operator', 'code' => 'GO'],
            ['title' => 'Loader Operator', 'code' => 'LO'],
            ['title' => 'Roller Operator', 'code' => 'RO'],
            ['title' => 'Concrete Pumper', 'code' => 'CP'],
            ['title' => 'Concrete Finisher', 'code' => 'CF'],
            ['title' => 'Steel Fixer', 'code' => 'SF'],
            ['title' => 'Carpenter', 'code' => 'CAR'],
            ['title' => 'Bricklayer', 'code' => 'BL'],
            ['title' => 'Plasterer', 'code' => 'PL'],
            ['title' => 'Tiler', 'code' => 'TI'],
            ['title' => 'Painter', 'code' => 'PA'],
            ['title' => 'Electrician', 'code' => 'EL'],
            ['title' => 'Plumber', 'code' => 'PLU'],
            ['title' => 'Welder', 'code' => 'WE'],
            ['title' => 'Scaffolder', 'code' => 'SC'],
            ['title' => 'Roofer', 'code' => 'RF'],
            ['title' => 'Glazier', 'code' => 'GL'],
            ['title' => 'Insulation Worker', 'code' => 'IW'],
            ['title' => 'Drywaller', 'code' => 'DW'],
            ['title' => 'Flooring Installer', 'code' => 'FI'],
            ['title' => 'HVAC Technician', 'code' => 'HVAC'],
            ['title' => 'Surveyor', 'code' => 'SUR'],
            ['title' => 'Demolition Worker', 'code' => 'DEM'],
            ['title' => 'Traffic Controller', 'code' => 'TC'],
            ['title' => 'Rigger', 'code' => 'RIG'],
            ['title' => 'Boilermaker', 'code' => 'BM'],
            ['title' => 'Mechanic', 'code' => 'MEC'],
            ['title' => 'Truck Driver', 'code' => 'TD'],
            ['title' => 'Delivery Driver', 'code' => 'DD'],
            ['title' => 'Warehouse Worker', 'code' => 'WW'],
            ['title' => 'Material Handler', 'code' => 'MH'],
            ['title' => 'Site Cleaner', 'code' => 'SCA'],
            ['title' => 'Landscaper', 'code' => 'LA'],
            ['title' => 'Fencer', 'code' => 'FE'],
            ['title' => 'Concreter', 'code' => 'CON'],
            ['title' => 'Driller', 'code' => 'DR'],
            ['title' => 'Blaster', 'code' => 'BLA'],
            ['title' => 'Tunneling Worker', 'code' => 'TW'],
            ['title' => 'Bridge Builder', 'code' => 'BB'],
            ['title' => 'Marine Construction Worker', 'code' => 'MCW'],
            ['title' => 'Pipeline Worker', 'code' => 'PW'],
            ['title' => 'Utility Worker', 'code' => 'UW'],
            ['title' => 'Civil Construction Worker', 'code' => 'CCW'],
            ['title' => 'Road Worker', 'code' => 'RW'],
            ['title' => 'Piling Operator', 'code' => 'PIU'],
            ['title' => 'Asbestos Removalist', 'code' => 'AR'],
            ['title' => 'Environmental Officer', 'code' => 'EO2'],
            ['title' => 'Site Clerk', 'code' => 'SCL'],
            ['title' => 'Storeman', 'code' => 'STR'],
            ['title' => 'Security Guard', 'code' => 'SG'],
            ['title' => 'First Aid Officer', 'code' => 'FAO'],
            ['title' => 'Workplace Health & Safety Representative', 'code' => 'WHSR'],
        ];

        $createdRoles = [];
        $created_by = $created_by ?? $customer_id;

        foreach ($roles as $roleData) {
            // Normalize code to uppercase
            $upperCaseCode = strtoupper($roleData['code']);
            $lowerCaseCode = strtolower($roleData['code']);

            // First, check if there's an existing role with lowercase code
            $existingRole = Role::where('customer_id', $customer_id)
                ->where('workspace_id', $workspace_id)
                ->where(function($query) use ($upperCaseCode, $lowerCaseCode, $roleData) {
                    $query->where('code', $upperCaseCode)
                          ->orWhere('code', $lowerCaseCode)
                          ->orWhere('code', $roleData['code']); // original case
                })
                ->first();

            if ($existingRole) {
                // Update the existing role to uppercase and update title
                $existingRole->update([
                    'code' => $upperCaseCode,
                    'title' => $roleData['title'],
                    'del' => '0',
                ]);
                $createdRoles[] = $existingRole;
                // echo "Updated existing role: {$existingRole->title} (Code: {$existingRole->code})\n";
            } else {
                // Create new role with uppercase code
                $newRole = Role::create([
                    'title' => $roleData['title'],
                    'code' => $upperCaseCode,
                    'customer_id' => $customer_id,
                    'workspace_id' => $workspace_id,
                    'del' => '0',
                ]);
                $createdRoles[] = $newRole;
                // echo "Created new role: {$newRole->title} (Code: {$newRole->code})\n";
            }
        }

        // Clean up any duplicate roles that might have been created
        $this->cleanupDuplicateRoles($customer_id, $workspace_id);

                // Create a default team if it doesn't exist
        $this->createDefaultTeam($customer_id, $workspace_id, $created_by);

        return $createdRoles;
    }

    /**
     * Clean up duplicate roles (same title/code combination)
     */
    private function cleanupDuplicateRoles($customer_id, $workspace_id)
    {
        $duplicates = Role::where('customer_id', $customer_id)
            ->where('workspace_id', $workspace_id)
            ->select('code', 'title', DB::raw('COUNT(*) as count'))
            ->groupBy('code', 'title')
            ->having('count', '>', 1)
            ->get();

        foreach ($duplicates as $duplicate) {
            $roles = Role::where('customer_id', $customer_id)
                ->where('workspace_id', $workspace_id)
                ->where('code', $duplicate->code)
                ->where('title', $duplicate->title)
                ->orderBy('id', 'asc')
                ->get();

            // Keep the first one, delete the rest
            $roles->skip(1)->each(function($role) {
                $role->delete();
                echo "Deleted duplicate role: {$role->title} (Code: {$role->code})\n";
            });
        }
    }

    private function createDefaultTeam($customer_id, $workspace_id, $created_by = null)
    {
        $created_by = $created_by ?? $customer_id;

        // Check if a default team already exists for this customer/workspace
        $existingTeam = EmpTeam::where('customer_id', $customer_id)
            ->where('workspace_id', $workspace_id)
            ->where('del', '0')
            ->first();

        if (!$existingTeam) {
            // Create a default team with null supervisor (since employees might not exist yet)
            EmpTeam::create([
                'title' => 'Default Team',
                'supervisor' => null, // Will be updated later when employees are created
                'description' => 'Default team created automatically during role setup',
                'customer_id' => $customer_id,
                'workspace_id' => $workspace_id,
                'created_by' => $created_by,
                'del' => '0',
            ]);
        }
    }
}