<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use App\Models\EmpCompanyDetails;
use App\Models\EmpPersonalDetails;
use App\Models\EmployeeSubcontractor;
use App\Models\SubcontractorCompany;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class AssetController extends Controller
{
    /**
     * Display a listing of assets with optional filters.
     */
    public function index(Request $request)
    {
        try {
            $query = Asset::query();
            $query = $this->applyCustomerWorkspaceFilter($query);
            if ($request->has('category')) {
                $query->where('category', $request->category);
            }
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('asset_name', 'like', "%{$search}%")
                      ->orWhere('model', 'like', "%{$search}%")
                      ->orWhere('serial_number', 'like', "%{$search}%")
                      ->orWhere('vendor', 'like', "%{$search}%")
                      ->orWhere('location', 'like', "%{$search}%")
                      ->orWhere('assigned_to', 'like', "%{$search}%");
                });
            }
            $assets = $query->with(['site'])->orderBy('created_at', 'desc');
            
            // Use withCount which handles pagination automatically
            // We'll process the data in prepareData or after
            $result = $this->withCount($assets, 'Assets fetched successfully');
            
            // Get the response data
            $responseData = json_decode($result->getContent(), true);
            $assetsData = $responseData['data'] ?? [];
            
            // Process assets to add assigned_to information based on assigned_to_type
            $processedAssets = collect($assetsData)->map(function ($asset) {
                // Handle both array and object formats
                $assignedTo = is_array($asset) ? ($asset['assigned_to'] ?? null) : ($asset->assigned_to ?? null);
                $assignedToType = is_array($asset) ? ($asset['assigned_to_type'] ?? null) : ($asset->assigned_to_type ?? null);
                $customerId = is_array($asset) ? ($asset['customer_id'] ?? null) : ($asset->customer_id ?? null);
                $workspaceId = is_array($asset) ? ($asset['workspace_id'] ?? null) : ($asset->workspace_id ?? null);
                $subcontractorId = is_array($asset) ? ($asset['subcontractor_id'] ?? null) : ($asset->subcontractor_id ?? null);
                
                $assignedToData = null;
                
                if ($assignedTo && $assignedToType) {
                    switch ($assignedToType) {
                        case 'internal':
                            // Get internal employee details
                            $empCompanyDetails = EmpCompanyDetails::find($assignedTo);
                            if ($empCompanyDetails) {
                                $empPersonalDetails = EmpPersonalDetails::where('emp_id', $assignedTo)->first();
                                $name = 'Unknown';
                                if ($empPersonalDetails) {
                                    $firstName = $empPersonalDetails->first_name ?? '';
                                    $middleName = $empPersonalDetails->middle_name ?? '';
                                    $lastName = $empPersonalDetails->last_name ?? '';
                                    $name = trim($firstName . ' ' . $middleName . ' ' . $lastName) ?: ($empCompanyDetails->employee_email ?? 'Unknown');
                                } else {
                                    $name = $empCompanyDetails->employee_email ?? 'Unknown';
                                }
                                
                                $assignedToData = [
                                    'id' => $assignedTo,
                                    'name' => $name,
                                    'type' => 'internal',
                                ];
                            }
                            break;
                            
                        case 'subcontractor':
                            // Get subcontractor details
                            $subcontractor = SubcontractorCompany::where('user_id', $assignedTo)
                                ->where('customer_id', $customerId)
                                ->where('workspace_id', $workspaceId)
                                ->first();
                            
                            if ($subcontractor) {
                                $name = $subcontractor->name ?? $subcontractor->company_name ?? 'Unknown';
                                $assignedToData = [
                                    'id' => $assignedTo,
                                    'name' => $name,
                                    'type' => 'subcontractor',
                                ];
                            } else {
                                // Fallback to User table
                                $user = User::find($assignedTo);
                                if ($user) {
                                    $assignedToData = [
                                        'id' => $assignedTo,
                                        'name' => $user->name ?? $user->email ?? 'Unknown',
                                        'type' => 'subcontractor',
                                    ];
                                }
                            }
                            break;
                            
                        case 'subcontractor_employee':
                            // Get subcontractor employee details
                            $subcontractorEmployee = EmployeeSubcontractor::find($assignedTo);
                            if ($subcontractorEmployee) {
                                $firstName = $subcontractorEmployee->first_name ?? '';
                                $middleName = $subcontractorEmployee->middle_name ?? '';
                                $lastName = $subcontractorEmployee->last_name ?? '';
                                $employeeName = trim($firstName . ' ' . $middleName . ' ' . $lastName) ?: ($subcontractorEmployee->email ?? 'Unknown');
                                
                                // Get subcontractor name if subcontractor_id is set
                                $subcontractorName = null;
                                if ($subcontractorId) {
                                    $subcontractor = SubcontractorCompany::where('user_id', $subcontractorId)
                                        ->where('customer_id', $customerId)
                                        ->where('workspace_id', $workspaceId)
                                        ->first();
                                    
                                    if ($subcontractor) {
                                        $subcontractorName = $subcontractor->name ?? $subcontractor->company_name ?? null;
                                    } else {
                                        // Fallback to User table
                                        $user = User::find($subcontractorId);
                                        if ($user) {
                                            $subcontractorName = $user->name ?? $user->email ?? null;
                                        }
                                    }
                                }
                                
                                $assignedToData = [
                                    'id' => $assignedTo,
                                    'name' => $employeeName,
                                    'type' => 'subcontractor_employee',
                                    'subcontractor_id' => $subcontractorId,
                                    'subcontractor_name' => $subcontractorName,
                                ];
                            }
                            break;
                    }
                }
                
                // Convert asset to array and add assigned_to_data
                $assetArray = is_array($asset) ? $asset : $asset->toArray();
                $assetArray['assigned_to_data'] = $assignedToData;
                
                return $assetArray;
            })->values()->toArray();
            
            // Update the response data with processed assets
            $responseData['data'] = $processedAssets;
            
            return response()->json($responseData);
        } catch (\Exception $e) {
            return $this->error($e->getMessage(),500);
        }
    }

    /** 
     * Store a newly created asset.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'asset_name' => 'required|string|max:255',
                'category' => 'integer|required',
                'model' => 'required|string|max:255',
                'serial_number' => 'required|string|max:255|unique:assets,serial_number',
                'purchase_date' => 'required|date',
                'purchase_price' => 'required|numeric|min:0',
                'vendor' => 'nullable|string|max:255',
                'warranty_expiry' => 'nullable|date|after_or_equal:purchase_date',
                'location' => 'nullable|string|max:255',
                'assigned_to' => 'nullable|integer',
                'status' => 'required|integer',
                'description' => 'nullable|string',
                'site_id' => 'nullable|integer',
                'subcontractor_id' => 'nullable|integer',
                'assigned_to_type' => 'nullable|string',
            ]);
            $userTable = $this->getUserTable();
            $workspace_id = 0;
            $customer_id = 0;
            if ($userTable === "customer") {
                $customer_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            }
            if ($userTable === "emp") {
                $customer_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
            }
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
            
            $data = $validator->validated();
            $data['customer_id'] = $customer_id;
            $data['workspace_id'] = $workspace_id;


            $asset = Asset::create($data);

            return $this->success($asset,'Asset created successfully');

        } catch (\Exception $e) {
            return $this->error($e->getMessage(),500);
        }
    }

    /**
     * Display the specified asset.
     */
    public function show($id)
    {
        try {
            $userTable = $this->getUserTable();
            $workspace_id = 0;
            $customer_id = 0;
            if ($userTable === "customer") {
                $customer_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            }
            if ($userTable === "emp") {
                $customer_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
            }
            $asset = Asset::where('customer_id', $customer_id)->where('workspace_id', $workspace_id)->find($id);
            if (!$asset) return $this->error('Asset not found',404);
            return $this->success($asset,'Asset fetched successfully');

        } catch (\Exception $e) {
            return $this->error($e->getMessage(),404);
        }
    }

    /**
     * Update the specified asset.
     */
    public function update(Request $request, $id)
    {
        try {
            $userTable = $this->getUserTable();
            $workspace_id = 0;
            $customer_id = 0;
            if ($userTable === "customer") {
                $customer_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            }
            if ($userTable === "emp") {
                $customer_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
            }
            $asset = Asset::where('customer_id', $customer_id)->where('workspace_id', $workspace_id)->find($id);
            if (!$asset) return $this->error('Asset not found',404);
            $validator = Validator::make($request->all(), [
                'asset_name' => 'sometimes|required|string|max:255',
                'category' => 'sometimes|required|integer',
                'model' => 'sometimes|required|string|max:255',
                'serial_number' => [
                    'sometimes',
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('assets', 'serial_number')->ignore($id)
                ],
                'purchase_date' => 'sometimes|required|date',
                'purchase_price' => 'sometimes|required|numeric|min:0',
                'vendor' => 'nullable|string|max:255',
                'warranty_expiry' => 'nullable|date|after_or_equal:purchase_date',
                'location' => 'nullable|string|max:255',
                'assigned_to' => 'nullable|integer',
                'status' => 'sometimes|required|integer',
                'description' => 'nullable|string',
                'site_id' => 'nullable|integer',
                'subcontractor_id' => 'nullable|integer',
                'assigned_to_type' => 'nullable|string',
            ]);
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }

            $asset->update($validator->validated());

            return $this->success($asset->fresh(),'Asset updated successfully');

        } catch (\Exception $e) {
            return $this->error($e->getMessage(),500);
        }
    }

    /**
     * Remove the specified asset.
     */
    public function destroy($id)
    {
        try {
            $userTable = $this->getUserTable();
            $workspace_id = 0;
            $customer_id = 0;
            if ($userTable === "customer") {
                $customer_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            }
            if ($userTable === "emp") {
                $customer_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
            }
            $asset = Asset::where('customer_id', $customer_id)->where('workspace_id', $workspace_id)->find($id);
            if (!$asset) return $this->error('Asset not found',404);
            $asset->delete();
            return $this->success(1,'Asset deleted successfully');
        } catch (\Exception $e) {
            return $this->error($e->getMessage(),500);
        }
    }
    public function getDashboard()
    {
        try {
            $userTable = $this->getUserTable();
            $workspace_id = 0;
            $customer_id = 0;
            
            if ($userTable === "customer") {
                $customer_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            }
            if ($userTable === "emp") {
                $customer_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
            }
    
            $thirtyDaysFromNow = Carbon::now()->addDays(30);
    
            // Get statistics - using the correct status values based on your system
            $statistics = [
                'total' => Asset::where('customer_id', $customer_id)
                               ->where('workspace_id', $workspace_id)
                               ->count(),
                'inUse' => Asset::where('customer_id', $customer_id)
                               ->where('workspace_id', $workspace_id)
                               ->where('status', 2) // In Use
                               ->count(),
                'available' => Asset::where('customer_id', $customer_id)
                                   ->where('workspace_id', $workspace_id)
                                   ->where('status', 1) // Available
                                   ->count(),
                'maintenance' => Asset::where('customer_id', $customer_id)
                                     ->where('workspace_id', $workspace_id)
                                     ->where('status', 3) // Under Maintenance
                                     ->count(),
                'retired' => Asset::where('customer_id', $customer_id)
                ->where('workspace_id', $workspace_id)
                ->where('status', 4) // Under Maintenance
                ->count(),
                'lost' => Asset::where('customer_id', $customer_id)
                ->where('workspace_id', $workspace_id)
                ->where('status', 5) // Under Maintenance
                ->count(),
                'damaged' => Asset::where('customer_id', $customer_id)
                ->where('workspace_id', $workspace_id)
                ->where('status', 6) // Under Maintenance
                ->count(),
            ];
    
            // Get warranty alerts
            $warrantyExpiring = Asset::where('customer_id', $customer_id)
                                    ->where('workspace_id', $workspace_id)
                                    ->whereNotNull('warranty_expiry')
                                    ->where('warranty_expiry', '<=', $thirtyDaysFromNow)
                                    ->where('warranty_expiry', '>=', Carbon::now())
                                    ->select('id', 'asset_name', 'warranty_expiry', 'serial_number', 'category')
                                    ->get()
                                    ->map(function ($asset) {
                                        $daysUntilExpiry = Carbon::now()->diffInDays($asset->warranty_expiry, false);
                                        return [
                                            'id' => $asset->id,
                                            'asset_name' => $asset->asset_name,
                                            'serial_number' => $asset->serial_number,
                                            'category' => $asset->category,
                                            'warranty_expiry' => $asset->warranty_expiry,
                                            'days_until_expiry' => $daysUntilExpiry,
                                            'alert_type' => 'warranty_expiring'
                                        ];
                                    });

    
            // Get recent assets
            $recentAssets = Asset::where('customer_id', $customer_id)
                                ->where('workspace_id', $workspace_id)
                                ->orderBy('created_at', 'desc')
                                ->limit(5)
                                ->select('id', 'asset_name', 'category', 'status', 'created_at')
                                ->get();
    
            // Get assets by category distribution
            $categoryDistribution = Asset::where('customer_id', $customer_id)
                                        ->where('workspace_id', $workspace_id)
                                        ->selectRaw('category, COUNT(*) as count')
                                        ->groupBy('category')
                                        ->get();
    
            // Get status summary
            $statusSummary = Asset::where('customer_id', $customer_id)
                                 ->where('workspace_id', $workspace_id)
                                 ->selectRaw('status, COUNT(*) as count')
                                 ->groupBy('status')
                                 ->get()
                                 ->map(function ($item) {
                                     $statusLabels = [
                                         1 => 'Available',
                                         2 => 'In Use',
                                         3 => 'Under Maintenance',
                                         4 => 'Retired',
                                         5 => 'Lost',
                                         6 => 'Damaged'
                                     ];
                                     
                                     return [
                                         'status_id' => $item->status,
                                         'status_label' => $statusLabels[$item->status] ?? 'Unknown',
                                         'count' => $item->count
                                     ];
                                 });
    
            $dashboard = [
                'statistics' => $statistics,
                'alerts' => [
                    'warranty_expiring' => $warrantyExpiring,
                    'total_alerts' => $warrantyExpiring->count()
                ],
                'recent_assets' => $recentAssets,
                'category_distribution' => $categoryDistribution,
                'status_summary' => $statusSummary,
                'last_updated' => Carbon::now()->format('Y-m-d H:i:s')
            ];
    
            return $this->success($dashboard, 'Dashboard data fetched successfully');
    
        } catch (\Exception $e) {
            return $this->error($e->getMessage(), 500);
        }
    }
}
