<?php


namespace App\Http\Controllers;

use App\Models\Fund;
use App\Models\Role;
use App\Models\User;
use App\Models\EmpType;
use App\Models\EmpAccess;
use App\Models\EmpPermit;
use App\Models\FundsMeta;
use App\General\MetaClass;
use App\Models\EmpDocuments;
use App\Services\XeroService;
use App\Models\XeroToken;
use Illuminate\Http\Request;
use App\Models\EmpTeamsMember;
use App\Models\EmpListChangeCol;
use App\Models\RequiredDocument;
use App\Models\EmpCompanyDetails;
use App\Models\EmpPersonalDetails;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\SubcontractorCompany;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\SubcontractorEmployeeDocument;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use App\Models\LeavePackage;
use App\Models\EmployeeAttendance;
use App\Models\Overtime;
use App\Models\RosterAssign;
use App\Models\Sites;
use App\Models\Project;
class EmployeeController extends Controller
{
  // Employee type constants for better maintainability
  const EMPLOYEE_TYPE_INTERNAL = 0;
  const EMPLOYEE_TYPE_EXTERNAL = 1;


  public function index(Request $request)
  {
    $user = Auth::user();
    if (!$user) {
      return $this->error('Unauthorized', 401);
    }
    $query = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class);
    $query = $this->applyCustomerWorkspaceFilter($query);
    if ($request->filled('invited')) {
      $query->where('invited', $request->invited);
    }
    // Filter by xero_emp_id - if true, only show employees with Xero Employee ID
    if ($request->filled('xero_emp_id') && ($request->xero_emp_id == true)) {
      $query->whereNotNull('xero_emp_id')
            ->where('xero_emp_id', '!=', '');
    }
    // Apply employee type filtering
    $this->applyEmployeeTypeFilter($query, $request);
    if ($request->filled('tierKyes')) {
      $query = $this->getTierEmployees($request->tierKyes, $query);
    }
    // Apply filters from request parameters
    $requestFilters = $request->only(['first_name', 'middle_name', 'last_name', 'employee_email', 'mobile', 'tax_file_no', 'access_role', 'status', 'completed', 'user_type', 'link_key', 'teams', 'streat_address', 'bank_name', 'account_number']);
    // Also support JSON-based filters for backward compatibility
    if ($request->filled('filter')) {
      $jsonFilters = json_decode($request->filter, true);
      $requestFilters = array_merge($requestFilters, $jsonFilters);
    }
    if ($request->filled('link_key')) {
      $query->where('user_type', 1);
    }
    // Apply search if provided - search across multiple columns
    if ($request->has('search') && !empty($request->search)) {
      $searchTerm = $request->search;
      $query->where(function($q) use ($searchTerm) {
        // Search in employee_email
        $q->where('employee_email', 'like', '%' . $searchTerm . '%')
          // Search in empPersonalDetails (first_name, middle_name, last_name)
          ->orWhereHas('empPersonalDetails', function($subquery) use ($searchTerm) {
            $subquery->where('first_name', 'like', '%' . $searchTerm . '%')
                     ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                     ->orWhere('last_name', 'like', '%' . $searchTerm . '%');
          })
          // Search in accessRole title
          ->orWhereHas('accessRole', function($subquery) use ($searchTerm) {
            $subquery->where('title', 'like', '%' . $searchTerm . '%');
          });
      });
    }
    
    // Apply all filters in one call
    $query = $this->filter($requestFilters, $query);
    $query->with([
      'empPersonalDetails',
      'empDocuments',
      'empEmergencyContacts',
      'empAccess',
      'accessRole',
      'empTeamsMembers',
      'empMetaData',
      'funds_meta'
    ]);
    // No sorting at query level - will sort after merging all data
    $users = $query;
    $array_filter = $request->filled('filter') ? json_decode($request->filter, true) : [];
    $filters = [
      'first_name'     => $array_filter['first_name'] ?? '',
      'middle_name'    => $array_filter['middle_name'] ?? '',
      'last_name'      => $array_filter['last_name'] ?? '',
      'employee_email' => $array_filter['employee_email'] ?? '',
      'completed'      => $array_filter['completed'] ?? '',
      'status'         => $array_filter['status'] ?? '',
      'access_role'    => $array_filter['access_role'] ?? '',
    ];
    $users_collection = $users->get();
    
    // If sub_employee=1 is passed, merge subcontractor employees with regular employees
    if ($request->filled('sub_employee') && $request->sub_employee == 1) {
      $subcontractor_employees = $this->getSubcontractorEmployeesForCustomer($request, $request->filled('attendance') && $request->attendance == 1, $request->filled('leave_request') && $request->leave_request == 1);
      
      // Convert regular employees to array and add employee_type flag
      $regular_employees = $users_collection->map(function($emp) {
        $empArray = $emp->toArray();
        $empArray['employee_type'] = 'regular';
        // Normalize created_at to string format for consistent sorting
        if ($emp->created_at instanceof \Carbon\Carbon) {
          $empArray['created_at'] = $emp->created_at->toDateTimeString();
        } elseif (!isset($empArray['created_at']) || empty($empArray['created_at'])) {
          $empArray['created_at'] = $emp->created_at ? (string)$emp->created_at : now()->toDateTimeString();
        }
        return $empArray;
      })->toArray();
      
      // Merge both collections
      $merged_employees = array_merge($regular_employees, $subcontractor_employees->toArray());
      
      // Apply search filter on merged collection if search parameter is provided
      if ($request->has('search') && !empty($request->search)) {
        $searchTerm = strtolower($request->search);
        $merged_employees = array_filter($merged_employees, function($emp) use ($searchTerm) {
          // Initialize search fields
          $firstName = '';
          $middleName = '';
          $lastName = '';
          $email = '';
          $roleTitle = '';
          
          // For regular employees, check emp_personal_details structure
          if (isset($emp['employee_type']) && $emp['employee_type'] === 'regular') {
            $firstName = isset($emp['emp_personal_details']['first_name']) ? strtolower($emp['emp_personal_details']['first_name']) : '';
            $middleName = isset($emp['emp_personal_details']['middle_name']) ? strtolower($emp['emp_personal_details']['middle_name']) : '';
            $lastName = isset($emp['emp_personal_details']['last_name']) ? strtolower($emp['emp_personal_details']['last_name']) : '';
            $email = isset($emp['employee_email']) ? strtolower($emp['employee_email']) : '';
            // Get role title from accessRole relationship
            $roleTitle = isset($emp['access_role']['title']) ? strtolower($emp['access_role']['title']) : '';
          } 
          // For subcontractor employees, check emp_personal_details structure (they also have this)
          else {
            // Check if emp_personal_details exists (subcontractor employees also have this structure)
            if (isset($emp['emp_personal_details'])) {
              $firstName = isset($emp['emp_personal_details']['first_name']) ? strtolower($emp['emp_personal_details']['first_name']) : '';
              $middleName = isset($emp['emp_personal_details']['middle_name']) ? strtolower($emp['emp_personal_details']['middle_name']) : '';
              $lastName = isset($emp['emp_personal_details']['last_name']) ? strtolower($emp['emp_personal_details']['last_name']) : '';
            }
            $email = isset($emp['employee_email']) ? strtolower($emp['employee_email']) : '';
            // Get role title from accessRole relationship (subcontractor employees have default EMP role)
            $roleTitle = isset($emp['access_role']['title']) ? strtolower($emp['access_role']['title']) : '';
          }
          
          return strpos($firstName, $searchTerm) !== false ||
                 strpos($middleName, $searchTerm) !== false ||
                 strpos($lastName, $searchTerm) !== false ||
                 strpos($email, $searchTerm) !== false ||
                 strpos($roleTitle, $searchTerm) !== false;
        });
        // Re-index array after filtering
        $merged_employees = array_values($merged_employees);
      }
      
      // Apply access_role filter on merged collection if provided
      $accessRoleFilter = $request->input('access_role');
      if (empty($accessRoleFilter) && $request->filled('filter')) {
        $array_filter = json_decode($request->filter, true);
        $accessRoleFilter = $array_filter['access_role'] ?? '';
      }
      
      if ($accessRoleFilter && !empty($accessRoleFilter)) {
        $accessRoleFilterLower = strtolower($accessRoleFilter);
        $merged_employees = array_filter($merged_employees, function($emp) use ($accessRoleFilterLower) {
          $accessRoleCode = '';
          $accessRoleTitle = '';
          
          // Extract access_role code and title
          if (isset($emp['access_role'])) {
            if (is_array($emp['access_role'])) {
              // access_role is an object/array with code and title
              $accessRoleCode = isset($emp['access_role']['code']) ? strtolower($emp['access_role']['code']) : '';
              $accessRoleTitle = isset($emp['access_role']['title']) ? strtolower($emp['access_role']['title']) : '';
            } else {
              // access_role is a string (code)
              $accessRoleCode = strtolower($emp['access_role']);
            }
          }
          
          // Check if filter matches either code or title
          return strpos($accessRoleCode, $accessRoleFilterLower) !== false ||
                 strpos($accessRoleTitle, $accessRoleFilterLower) !== false;
        });
        // Re-index array after filtering
        $merged_employees = array_values($merged_employees);
      }
      
      // Clean and simple sorting: by created_at (descending), then by id (descending) for stability
      usort($merged_employees, function($a, $b) {
        // Get created_at timestamp - both regular and subcontractor employees have it at root level
        $timestampA = isset($a['created_at']) && !empty($a['created_at']) ? strtotime($a['created_at']) : 0;
        $timestampB = isset($b['created_at']) && !empty($b['created_at']) ? strtotime($b['created_at']) : 0;
        
        // Primary sort by created_at (newest first)
        if ($timestampB != $timestampA) {
          return $timestampB - $timestampA;
        }
        
        // Secondary sort by id (higher id first) when created_at is equal
        $idA = isset($a['id']) ? (int)$a['id'] : 0;
        $idB = isset($b['id']) ? (int)$b['id'] : 0;
        return $idB - $idA;
      });
      
      if ($request->filled('chat')) {
        $customer = $this->getCustomerForChat($users_collection);
        $response = [
          'message' => "success",
          'statusCode' => 200,
          'data' => $merged_employees,
          'customer' => $customer,
        ];
        return response()->json($response, 200);
      }
      
      // Use WithFilter for pagination support
      return $this->WithFilter($merged_employees, $filters, 'Get Employees list Successfully');
    }
    
    // Sort regular employees by created_at (descending), then by id (descending)
    $regular_employees_array = $users_collection->toArray();
    usort($regular_employees_array, function($a, $b) {
      $timestampA = isset($a['created_at']) && !empty($a['created_at']) ? strtotime($a['created_at']) : 0;
      $timestampB = isset($b['created_at']) && !empty($b['created_at']) ? strtotime($b['created_at']) : 0;
      if ($timestampB != $timestampA) {
        return $timestampB - $timestampA;
      }
      $idA = isset($a['id']) ? (int)$a['id'] : 0;
      $idB = isset($b['id']) ? (int)$b['id'] : 0;
      return $idB - $idA;
    });
    $sorted_regular_employees = collect($regular_employees_array);
    
    if ($request->filled('chat')) {
      $customer = $this->getCustomerForChat($users_collection);
      $response = [
        'message' => "success",
        'statusCode' => 200,
        'data' => $sorted_regular_employees,
        'customer' => $customer,
      ];
      return response()->json($response, 200);
    }
    return $this->WithFilter($sorted_regular_employees, $filters, 'Get Employees list Successfully');
  }
  
  /**
   * Get subcontractor employees who have accepted invitations for this customer
   * 
   * @param Request $request
   * @param bool $attendanceFilter If true, only return employees with completed induction and approved docs
   * @return \Illuminate\Support\Collection
   */
  private function getSubcontractorEmployeesForCustomer(Request $request, $attendanceFilter = false, $leaveRequestFilter = false)
  {
    $ids = $this->getCustomerAndWorkspaceIds();
    
    $customerId = $ids['customer_id'];
    $workspaceId = $ids['workspace_id'];
    
    // Get status filter if provided
    $statusFilter = $request->input('status');
    
    // Get project_id filter if provided
    $projectId = $request->input('project_id');
    
    // If project_id is provided, get all subcontractors assigned to this project
    $subcontractorIds = null;
    if ($projectId) {
      // Get all subcontractors assigned to this project from SubcontractorCompany
      $subcontractorCompanies = SubcontractorCompany::where('customer_id', $customerId)
        ->when($workspaceId, function($query) use ($workspaceId) {
          $query->where('workspace_id', $workspaceId);
        })
        ->where('del', '0')
        ->get();
      
      // Filter subcontractors whose project_ids array contains the project_id
      $subcontractorIds = $subcontractorCompanies->filter(function($company) use ($projectId) {
        $projectIds = is_array($company->project_ids) ? $company->project_ids : (is_string($company->project_ids) ? json_decode($company->project_ids, true) : []);
        return is_array($projectIds) && in_array($projectId, $projectIds);
      })->pluck('user_id')->unique()->toArray();
    }
    
    // Get unique employee IDs with accepted invitations for this customer
    $acceptedInvitations = SubcontractorEmployeeInvitation::where('customer_id', $customerId)
      ->where('invitation_status', 'accepted')
      ->when($workspaceId, function($query) use ($workspaceId) {
        $query->where('workspace_id', $workspaceId);
      })
      ->when($statusFilter !== null && $statusFilter !== '', function($query) use ($statusFilter) {
        $query->where('status', $statusFilter);
      })
      ->when($projectId, function($query) use ($projectId, $subcontractorIds) {
        // When project_id is provided, filter by project_id directly OR by subcontractor_ids assigned to the project
        // This ensures we get:
        // 1. Employees directly assigned to this project (via project_id)
        // 2. Employees from subcontractors assigned to this project (via subcontractor_id)
        $query->where(function($q) use ($projectId, $subcontractorIds) {
          $q->where('project_id', $projectId);
          // Also include employees from subcontractors assigned to this project
          if (!empty($subcontractorIds)) {
            $q->orWhereIn('subcontractor_id', $subcontractorIds);
          }
        });
      })
      ->select('employee_id', 'subcontractor_id', 'project_id', 'status', 'required_docs_status', 'induction_status')
      ->with(['employee', 'subcontractor:id,name,email,mobile_number,company_name,abn', 'project:id,title'])
      ->get();
    
    // Group invitations by employee to get unique employees
    // Note: We removed the duplicate filtering here (lines 217-231) as it's handled in the employee filter below (lines 297-314)
    $employeeIds = $acceptedInvitations->pluck('employee_id')->unique()->toArray();
    
    if (empty($employeeIds)) {
      return collect([]);
    }
    

    $employeeInvitations = $acceptedInvitations->groupBy('employee_id')->map(function($invitations) {
      return $invitations->first(); // Use first for status info
    });
    
    // Group all project_ids by employee for getting projects with sites
    $employeeProjectIds = $acceptedInvitations->groupBy('employee_id')->map(function($invitations) {
      return $invitations->pluck('project_id')->filter()->unique()->values()->toArray();
    });
    
    // Get employees with their details
    $employees = EmployeeSubcontractor::whereIn('id', $employeeIds)->get();
    // Get employee metadata (user_meta table)
    $employeeMetaData = DB::table('user_meta')
      ->whereIn('emp_id', $employeeIds)
      ->where('del', '0')
      ->get()
      ->groupBy('emp_id');
    
    // Get funds meta
    $fundsMeta = DB::table('funds_meta')
      ->whereIn('emp_id', $employeeIds)
      ->get()
      ->groupBy('emp_id');
    
    // Get default role for external employees (usually "EMP" code)
    $defaultRole = Role::where('code', 'EMP')->where('del', '0')->first();
    
    // Get leave packages for EMP role if leave_request filter is enabled
    $leavePackages = collect([]);
    if ($leaveRequestFilter && $defaultRole) {
      $leavePackages = LeavePackage::whereRaw("FIND_IN_SET(?, employee_roles)", [$defaultRole->id])
        ->where('customer_id', $customerId)
        ->where('workspace_id', $workspaceId)
        ->get();
    }
    
    // Apply name filters if provided
    $firstNameFilter = $request->input('first_name');
    $lastNameFilter = $request->input('last_name');
    $emailFilter = $request->input('employee_email');
    $searchTerm = $request->input('search');
    
    // Format the response to match regular employee structure
    return $employees->filter(function($employee) use ($firstNameFilter, $lastNameFilter, $emailFilter, $searchTerm, $defaultRole, $employeeInvitations, $attendanceFilter, $leaveRequestFilter, $projectId) {
      // Apply name/email filters
      if ($firstNameFilter && stripos($employee->first_name, $firstNameFilter) === false) {
        return false;
      }
      if ($lastNameFilter && stripos($employee->last_name, $lastNameFilter) === false) {
        return false;
      }
      if ($emailFilter && stripos($employee->email, $emailFilter) === false) {
        return false;
      }
      
      // Apply search filter if provided - search across first_name, middle_name, last_name, email, and role title
      if ($searchTerm && !empty($searchTerm)) {
        $searchLower = strtolower($searchTerm);
        $firstName = strtolower($employee->first_name ?? '');
        $middleName = strtolower($employee->middle_name ?? '');
        $lastName = strtolower($employee->last_name ?? '');
        $email = strtolower($employee->email ?? '');
        
        // Get role title for subcontractor employees (default role is EMP)
        $roleTitle = '';
        if ($defaultRole) {
          $roleTitle = strtolower($defaultRole->title ?? '');
        }
        
        if (strpos($firstName, $searchLower) === false &&
            strpos($middleName, $searchLower) === false &&
            strpos($lastName, $searchLower) === false &&
            strpos($email, $searchLower) === false &&
            strpos($roleTitle, $searchLower) === false) {
          return false;
        }
      }
      
      // If attendance or leave_request filter is enabled, verify invitation exists and statuses are complete
      if ($attendanceFilter || $leaveRequestFilter || $projectId) {
        $invitation = $employeeInvitations->get($employee->id);
        
        // Check if invitation exists
        if (!$invitation) {
          return false;
        }
        
        // Check induction completion (Status 1 = Complete)
        if ($invitation->induction_status != 1) {
          return false;
        }
        
        // Check document approval (Status 1 = Complete)
        if ($invitation->required_docs_status != 1) {
          return false;
        }
      }
      
      return true;
    })->map(function($employee) use ($employeeMetaData, $fundsMeta, $defaultRole, $customerId, $workspaceId, $employeeInvitations, $employeeProjectIds, $leavePackages, $attendanceFilter, $projectId) {
      $meta = $employeeMetaData->get($employee->id, collect());
      $funds = $fundsMeta->get($employee->id, collect());
      $invitation = $employeeInvitations->get($employee->id);
      
      $projectIds = $employeeProjectIds->get($employee->id, []);
      
      // Get subcontractor information when project_id is provided
      $subcontractorInfo = null;
      if ($projectId && $invitation) {
        // Get the subcontractor from the invitation relationship
        $subcontractor = $invitation->subcontractor ?? null;
        if ($subcontractor) {
          $subcontractorInfo = [
            'id' => $subcontractor->id,
            'name' => $subcontractor->name,
            'email' => $subcontractor->email,
            'mobile_number' => $subcontractor->mobile_number ?? null,
            'company_name' => $subcontractor->company_name ?? null,
            'abn' => $subcontractor->abn ?? null,
          ];
        }
      }
      
      // Only get sites if attendance filter is enabled
      $sites = $attendanceFilter ? $this->getSitesFromProjects($projectIds, $customerId, $workspaceId) : collect([]);
      
      // Format metadata to match user_meta structure
      $formattedMeta = $meta->map(function($item) {
        return [
          'id' => $item->id,
          'emp_id' => (string)$item->emp_id,
          'option' => $item->option,
          'value' => $item->value,
          'del' => (string)$item->del,
          'created_at' => $item->created_at,
          'updated_at' => $item->updated_at,
        ];
      })->values()->toArray();
      
      // Format funds meta
      $formattedFunds = $funds->map(function($item) {
        return [
          'id' => $item->id,
          'emp_id' => (string)$item->emp_id,
          'fund_id' => $item->fund_id,
          'usi_number' => $item->usi_number ?? null,
          'member_number' => $item->member_number ?? null,
          'del' => (string)($item->del ?? '0'),
          'created_at' => $item->created_at,
          'updated_at' => $item->updated_at,
        ];
      })->values()->toArray();
      
      // Format access role
      $accessRole = null;
      if ($defaultRole) {
        $accessRole = [
          'id' => $defaultRole->id,
          'title' => $defaultRole->title,
          'code' => $defaultRole->code,
          'del' => (string)$defaultRole->del,
          'is_seeded' => $defaultRole->is_seeded,
          'created_at' => $defaultRole->created_at ? date('d-m-Y', strtotime($defaultRole->created_at)) : null,
          'updated_at' => $defaultRole->updated_at ? date('d-m-Y', strtotime($defaultRole->updated_at)) : null,
        ];
      }
      
      // Format personal details (from employee_subcontractor table)
      $empPersonalDetails = [
        'id' => null,
        'emp_id' => (string)$employee->id,
        'first_name' => $employee->first_name,
        'middle_name' => $employee->middle_name,
        'last_name' => $employee->last_name,
        'streat_address' => $employee->address,
        'suburb' => $employee->suburb,
        'state' => $employee->state,
        'mobile' => $employee->mobile,
        'phone' => $employee->phone,
        'dob' => $employee->dob,
        'image' => $employee->profile_image,
        'del' => '0',
        'subcontractors' => $employee->subcontractors,
        'created_at' => $employee->created_at,
        'updated_at' => $employee->updated_at,
      ];
      
      // Get status from invitations (each company manages independently)
      $employeeStatus = $invitation ? (string)($invitation->status ?? 1) : '1';
      
      // Return structure matching regular employee
      return [
        'id' => $employee->id,
        'employee_email' => $employee->email,
        'employment_type' => null,
        'work_permit_type' => null,
        'tax_file_no' => null,
        'tier_id' => null,
        'employment_start_date' => null,
        'employment_end_date' => null,
        'attendance_effective_from' => null,
        'status' => $employeeStatus, // From subcontractor_employee_invitations.status
        'required_docs_status' => $invitation ? $invitation->required_docs_status : 0,
        'induction_status' => $invitation ? $invitation->induction_status : 0,
        'induction_status_label' => $invitation ? $invitation->induction_status_label : 'not_signed',
        'invited' => '1',
        'approved' => '1',
        'rejected' => '0',
        'approved_by' => null,
        'approved_status' => null,
        'approved_at' => null,
        'is_submitted' => '0',
        'compeleted' => '1',
        'credentials' => '0',
        'created_by' => (string)$customerId,
        'del' => '0',
        'created_at' => $employee->created_at ? $employee->created_at->toDateTimeString() : now()->toDateTimeString(),
        'updated_at' => $employee->updated_at ? $employee->updated_at->toDateTimeString() : null,
        'two_factor' => 0,
        'force_reset' => $employee->force_reset ?? 0,
        'user_type' => self::EMPLOYEE_TYPE_EXTERNAL, // 1 = external/subcontractor employee
        'job_title' => null,
        'trade_qualified' => $employee->trade_qualified ?? null,
        'trade_qualified_year' => $employee->trade_qualified_year ?? null,
        'trade_licensed' => $employee->trade_licensed ?? null,
        'trade_licensed_year' => $employee->trade_licensed_year ?? null,
        'work_experience' => $employee->work_experience ?? null,
        'temporary_student_visa' => null,
        'worker_type' => null,
        'year_commenced' => $employee->year_commenced ?? null,
        'classified_high_risk' => null,
        'citizenship_status' => $employee->citizenship_status ?? null,
        'citizenship_file' => null,
        'allergies' => $employee->allergies ?? null,
        'previous_injuries' => $employee->previous_injuries ?? null,
        'medical_condition' => $employee->medical_condition ?? null,
        'employer_name' => null,
        'legally_australia' => null,
        'link_key' => null,
        'xero_emp_id' => $employee->xero_emp_id ?? null,
        'signature' => $employee->signature ?? null,
        'details_allergies' => $employee->allergies ?? null,
        'details_previous_injuries' => $employee->previous_injuries ?? null,
        'emp_personal_details' => $empPersonalDetails,
        'emp_documents' => [],
        'emp_emergency_contacts' => [],
        'emp_access' => null,
        'emp_teams_members' => [],
        'emp_meta_data' => $formattedMeta,
        'funds_meta' => $formattedFunds,
        'access_role' => $accessRole, // This is the relationship object, not the code
        'employee_type' => 'subcontractor_employee',
        'sites' => $sites->toArray(), // Sites array (id and title only) from all assigned projects
        'subcontractor' => $subcontractorInfo, // Subcontractor information when project_id is provided
        'leave_packages' => $leavePackages->map(function($package) {
          return [
            'id' => $package->id,
            'title' => $package->title,
            'leave_type_id' => $package->leave_type_id,
            'leave_number' => $package->leave_number,
            'is_paid' => $package->is_paid,
            'color_code' => $package->color_code,
            'renew_on' => $package->renew_on,
            'employee_roles' => $package->employee_roles,
            'created_at' => $package->created_at,
            'updated_at' => $package->updated_at,
          ];
        })->toArray(), // Leave packages assigned to EMP role for this customer
      ];
    })->values();
  }

  public function getCustomerForChat($users_collection)
  {
    $ids = $this->getCustomerAndWorkspaceIds();
    // Get the customer user details
    $customer = User::find($ids['customer_id']);
    if ($customer) {
      $customer->type = 'customer'; // just add the extra property
    }
    return $customer;
  }

  public function filter($filters, $query)
  {

    foreach ($filters as $filterName => $filterValue) {
      if ($filterValue != null ||  $filterValue != "") {
        switch ($filterName) {
          case 'first_name':
            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
              $subquery->where('first_name', 'like', '%' . $filterValue . '%');
            });
            break;
          case 'middle_name':
            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
              $subquery->where('middle_name', 'like', '%' . $filterValue . '%');
            });
            break;
          case 'last_name':
            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
              $subquery->where('last_name', 'like', '%' . $filterValue . '%');
            });
            break;
          case 'employee_email':
            $query->where('employee_email', 'like', '%' . $filterValue . '%');
            break;
          case 'streat_address':
            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
              $subquery->where('streat_address', 'like', '%' . $filterValue . '%');
            });
            break;
          case 'mobile':
            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
              $subquery->where('mobile', 'like', '%' . $filterValue . '%');
            });
            break;
          case 'tax_file_no':
            $query->where('tax_file_no', 'like', '%' . $filterValue . '%');
            break;
          case 'bank_name':
            $query->whereHas('empMetaData', function ($subquery) use ($filterValue) {
              $subquery->where('option', 'bank_name')
                ->where('value', 'like', '%' . $filterValue . '%');
            });
            break;
          case 'account_number':
            $query->whereHas('empMetaData', function ($subquery) use ($filterValue) {
              $subquery->where('option', 'account_number')
                ->where('value', 'like', '%' . $filterValue . '%');
            });
            break;

          case 'teams':
            $emp_teams_members = EmpTeamsMember::where('team_id', $filterValue)->pluck('emp_id')->toArray();
            $query->whereIn('id', $emp_teams_members);
            break;
          case 'access_role':
            // Filter by access_role code OR role title
            $query->where(function($q) use ($filterValue) {
              $q->where('access_role', 'like', '%' . $filterValue . '%')
                ->orWhereHas('accessRole', function($subquery) use ($filterValue) {
                  $subquery->where('title', 'like', '%' . $filterValue . '%');
                });
            });
            break;
          case 'completed':
            $filterValue =   $filterValue ? 1 : 0;
            $query->where('compeleted', 'like', '%' . $filterValue . '%');
            break;
          case 'status':
            $filterValue =   $filterValue ? 1 : 0;
            $query->where('status', 'like', '%' . $filterValue . '%');
            break;
          case 'user_type':
            $filterValue =   $filterValue ? 1 : 0;
            $query->where('user_type', 'like', '%' . $filterValue . '%');
            break;
          case 'link_key':
            $query->where('link_key', 'like', '%' . $filterValue . '%');
            break;
        }
      }
    }

    return $query;
  }

  public function empDownload(Request $request)
  {
    $query = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->with(['empPersonalDetails', 'empDocuments', 'empEmergencyContacts', 'empAccess', 'accessRole', 'empTeamsMembers']);

    if ($request->filled('invited')) {
      $query->where('invited', $request->invited);
    }

    if ($request->filled('filter')) {
      $query = $this->filter(json_decode($request->filter, true), $query);
    }
    $userTable = $this->getUserTable();
    $auth_id = 0;
    $workspace_id = 0;
    if ($userTable === "customer") {
      $auth_id = Auth::user()->id;
      $workspace_id = Auth::user()->current_workspace_id;
      $authPersonalDetails = User::where('id', Auth::user()->id)->first();
    }
    if ($userTable === "emp") {
      $auth_id = auth()->user()->customer_id;
      $workspace_id = auth()->user()->workspace_id;
      $authPersonalDetails = EmpPersonalDetails::where('emp_id', Auth::user()->id)->first();
    }
    $query->where('customer_id', $auth_id);
    $query->where('workspace_id', $workspace_id);
    $query->where('del', '0');
    $query->orderBy('id', 'desc');
    $emps = $query->get();

    $cols = DB::table('emp_list_change_cols')->pluck('col')->toArray();

    $columns = [
      'First Name',
      'Middle Name',
      'Last Name',
      'Employee Email',
      'Teams',
      'Role',
      'Mobile',
      'Address',
      'Tax File No',
      'Account Number',
      'BSB',
      'Registration',
      'Status',
      'Type',
    ];

    $col_list[] = "Employee Id";
    $col_list[] = "External Id";
    $indexs = [];
    $i = 2;
    foreach ($columns as $column) {
      if (in_array($column, $cols)) {
        $col_list[] = $column;
      } else {
        $indexs[] = $i;
      }
      $i++;
    }

    $csvData[] = $col_list;

    $data_list  = [];

    foreach ($emps as $emp) {
      $team_list = "";
      foreach ($emp->empTeamsMembers as $et) {
        $team_list .= $et->empTeamsList->title . ",";
      }

      $team_list = rtrim($team_list, ',');

      $user_meta = DB::table('user_meta')
        ->where('emp_id', $emp->id)
        ->get();

      $data_list = [
        $emp->id,
        $this->searchInMeta($user_meta, 'employee_payroll_or_external_id'),
        $emp->empPersonalDetails ? $emp->empPersonalDetails->first_name : '',
        $emp->empPersonalDetails ? $emp->empPersonalDetails->middle_name : '',
        $emp->empPersonalDetails ? $emp->empPersonalDetails->last_name : '',
        $emp->employee_email,
        $team_list,
        $emp->accessRole ? $emp->accessRole->title : '',
        $emp->empPersonalDetails ? $emp->empPersonalDetails->mobile : '',
        $emp->empPersonalDetails ? $emp->empPersonalDetails->streat_address : '',
        $emp->tax_file_no,
        $this->searchInMeta($user_meta, 'account_number'),
        $this->searchInMeta($user_meta, 'bank_name'),
        $emp->registration == 0 ? 'InProcess' : 'Completed',
        $emp->status == 0 ? 'InActive' : 'Active',
        $emp->user_type == 0 ? 'Internal' : 'External'
      ];

      $csvData[] = array_values(array_diff_key($data_list, array_flip($indexs)));
    }
    $headers = [
      'Content-Type' => 'text/csv',
      'Content-Disposition' => 'attachment; filename="emp.csv"',
    ];


    $callback = function () use ($csvData) {
      $file = fopen('php://output', 'w');
      foreach ($csvData as $row) {
        fputcsv($file, $row);
      }
      fclose($file);
    };

    return Response::stream($callback, 200, $headers);
  }

  function searchInMeta($dataArray, $option)
  {
    $value = '';
    foreach ($dataArray as $element) {
      if ($element->option === $option) {
        $value = $element->value;
        break;
      }
    }
    return $value;
  }
  public function profile(Request $request)
  {
    // Check existence without global scope
    $exists = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
      ->where('id', $request->id)
      ->exists();

    if (!$exists) {
      return $this->message('Employee not found');
    }

    $userTable = $this->getUserTable();

    // Get full employee record without global scope
    $employee = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
      ->find($request->id);

    if (
      $userTable === "customer" &&
      ($employee->workspace_id != auth()->user()->current_workspace_id || $employee->customer_id != auth()->user()->id)
    ) {
      return $this->message('You do not have access to this Employee Profile', 403);
    }

    if (
      $userTable === "emp" &&
      ($employee->customer_id != auth()->user()->customer_id || $employee->workspace_id != auth()->user()->workspace_id)
    ) {
      return $this->message('You do not have access to this Employee Profile', 403);
    }

    // Build profile query without global scope
    $query = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
      ->where('id', $request->id)
      ->with([
        'empPersonalDetails',
        'empEmergencyContacts',
        'empAccess',
        'empworkExperience',
        'workerType'
      ]);

    $query_result = $query->first();

    // Add related data
    $query_result['empType'] = EmpType::where('id', $query_result->employment_type)
      ->select('id', 'title')
      ->first();

    $query_result['empWorkPermit'] = EmpPermit::where('id', $query_result->work_permit_type)
      ->select('id', 'title')
      ->first();

    $query_result['empAccessRole'] = Role::where('code', $query_result->access_role)
      ->select('id', 'title', 'code')
      ->first();

    $data['profile'] = $query_result;

    // Funds
    $data['funds_meta'] = FundsMeta::with('fund')
      ->where('emp_id', $request->id)
      ->get();

    // Medical files
    $data['medicalAttachFiles'] = DB::table('medical_attach_files')
      ->where('emp_id', $request->id)
      ->get();

    // Employee documents
    $data['emp_documents'] = EmpDocuments::where('emp_id', $request->id)
      ->where('del', '0')
      ->get();
    
    // Required documents - filter by employee type using boolean flags
    $isInternal = ($employee->user_type == 0);
    $requiredDocsQuery = RequiredDocument::with('requiredDocumentField')
      ->where('customer_id', $employee->customer_id)
      ->where('workspace_id', $employee->workspace_id)
      ->where('del', '0');
    
    if ($isInternal) {
      $requiredDocsQuery->where('for_internal', true);
    } else {
      $requiredDocsQuery->where('for_external', true);
    }
    
    $data['required_documents'] = $requiredDocsQuery->get();

    // Meta
    $user_meta = DB::table('user_meta')
      ->where('emp_id', $request->id)
      ->get();

    $employee_payroll_or_external_id = MetaClass::getOptionValue('employee_payroll_or_external_id', $user_meta);
    $data['meta']['employee_payroll_or_external_id'] = $employee_payroll_or_external_id;

    // Get induction documents (signed and not signed) for this employee
    $data['induction_documents'] = $this->getEmployeeInductionDocuments($request->id);

    // Calculate hours worked (sum of working_hours from attendance records in minutes, convert to hours)
    $hoursWorkedMinutes = EmployeeAttendance::where('employee_id', $request->id)
      ->where('customer_id', $employee->customer_id)
      ->where('workspace_id', $employee->workspace_id)
      ->where('status', 1) // Active/approved attendance
      ->whereNotNull('check_in')
      ->whereNotNull('check_out')
      ->whereNotNull('working_hours')
      ->sum('working_hours');
    $data['hours_worked'] = round($hoursWorkedMinutes / 60, 0); // Convert minutes to hours and round

    // Calculate overtime hours (sum of working_hours from approved overtime records in minutes, convert to hours)
    $overtimeMinutes = Overtime::where('employee_id', $request->id)
      ->where('customer_id', $employee->customer_id)
      ->where('workspace_id', $employee->workspace_id)
      ->where('status', 'approved')
      ->whereNotNull('working_hours')
      ->sum('working_hours');
    $data['overtime_hours'] = round($overtimeMinutes / 60, 0); // Convert minutes to hours and round

    // Get total sites (distinct sites from roster assigns and attendance records)
    $siteIdsFromRoster = RosterAssign::where('assign_to', $request->id)
      ->where('customer_id', $employee->customer_id)
      ->where('workspace_id', $employee->workspace_id)
      ->whereNotNull('site_id')
      ->distinct()
      ->pluck('site_id')
      ->toArray();
    
    $siteIdsFromAttendance = EmployeeAttendance::where('employee_id', $request->id)
      ->where('customer_id', $employee->customer_id)
      ->where('workspace_id', $employee->workspace_id)
      ->whereNotNull('site_id')
      ->distinct()
      ->pluck('site_id')
      ->toArray();
    
    $allSiteIds = array_unique(array_merge($siteIdsFromRoster, $siteIdsFromAttendance));
    $data['total_sites'] = count($allSiteIds);

    // Get total projects (distinct projects from roster assigns)
    $projectIds = RosterAssign::where('assign_to', $request->id)
      ->where('customer_id', $employee->customer_id)
      ->where('workspace_id', $employee->workspace_id)
      ->whereNotNull('project_id')
      ->distinct()
      ->pluck('project_id')
      ->toArray();
    $data['total_projects'] = count($projectIds);

    return $this->success($data, 'Get Profile Successfully');
  }


  public function show($id)
  {
    //
  }

  public function edit($id)
  {
    //
  }

  public function update(Request $request, $id)
  {
    //
  }

  public function store(Request $request)
  {
    //
  }

  public function employe_delete(Request $request)
  {
    $id = $request->id;
    $userTable = $this->getUserTable();
    $EmpCompanyDetails = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->find($id);

    if ($userTable == "customer" && ($EmpCompanyDetails->workspace_id != auth()->user()->current_workspace_id || $EmpCompanyDetails->customer_id != auth()->user()->id)) {
      return $this->message('You do not have access to this Employee ', 403);
    }

    if ($userTable == "emp" && ($EmpCompanyDetails->customer_id != auth()->user()->customer_id || $EmpCompanyDetails->workspace_id != auth()->user()->workspace_id)) {
      return $this->message('You do not have access to this Employee ', 403);
    }
    $EmpCompanyDetails->del = '1';
    $EmpCompanyDetails->save();

    return $this->message('Employee  Deleted Successfully');
  }

  public function updateStatus(Request $request)
  {
    $validator = Validator::make($request->all(), [
      'id' => 'required|integer',
      'user_type' => 'nullable|in:0,1', // 0 = regular employee, 1 = subcontractor employee
    ], [
      'id.required' => 'Employee ID is required.',
      'id.integer' => 'Employee ID must be an integer.',
      'user_type.in' => 'User type must be either 0 (regular) or 1 (subcontractor).',
    ]);

    if ($validator->fails()) {
      return $this->error($validator->errors()->first(), 422);
    }

    $id = $request->id;
    $userType = $request->input('user_type', 0); // Default to 0 (regular employee)
    $userTable = $this->getUserTable();
    $ids = $this->getCustomerAndWorkspaceIds();

    if (!$ids) {
      return $this->message('Unauthorized access', 401);
    }

    $customerId = $ids['customer_id'];
    $workspaceId = $ids['workspace_id'];

    // Handle subcontractor employee (user_type = 1)
    if ($userType == 1) {
      return $this->updateSubcontractorEmployeeStatus($id, $customerId, $workspaceId);
    }

    // Handle regular employee (user_type = 0 or not provided)
    $emp = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
      ->where('id', $id)
      ->first();

    if (!$emp) {
      return $this->message('Employee not found', 404);
    }

    // Validate access
    if ($emp->customer_id != $customerId || $emp->workspace_id != $workspaceId) {
      return $this->message('You do not have access to this Employee', 403);
    }

    // Toggle status
    $status = null;
    if ($emp->status == 1) {
      $status = 0;
    } elseif ($emp->status == 0) {
      $status = 1;
    } else {
      $status = 1; // Default to active
    }

    try {
      $updated = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
        ->where('id', $id)
        ->update(['status' => $status]);

      if ($updated === 0) {
        return $this->message('Failed to update employee status', 500);
      }

      return $this->success([
        'status' => $status,
        'employee_type' => 'regular'
      ], 'Employee status updated successfully');
    } catch (\Exception $e) {
      Log::error('Failed to update employee status: ' . $e->getMessage());
      return $this->message('Failed to update employee status', 500);
    }
  }

  /**
   * Update subcontractor employee status (active/inactive) for a specific company
   * Updates status in subcontractor_employee_invitations for all invitations of this employee for this customer
   * 
   * @param int $employeeId
   * @param int $customerId
   * @param int|null $workspaceId
   * @return \Illuminate\Http\JsonResponse
   */
  private function updateSubcontractorEmployeeStatus($employeeId, $customerId, $workspaceId)
  {
    // Verify the subcontractor employee exists
    $subcontractorEmp = EmployeeSubcontractor::where('id', $employeeId)->first();
    
    if (!$subcontractorEmp) {
      return $this->message('Subcontractor employee not found', 404);
    }

    // Get all accepted invitations for this employee and customer
    // This allows the company to manage status for all projects at once
    $invitations = SubcontractorEmployeeInvitation::where('employee_id', $employeeId)
      ->where('customer_id', $customerId)
      ->when($workspaceId, function($query) use ($workspaceId) {
        $query->where('workspace_id', $workspaceId);
      })
      ->where('invitation_status', 'accepted')
      ->get();

    if ($invitations->isEmpty()) {
      return $this->message('You do not have access to this Employee. The employee has not accepted any invitation for this company.', 403);
    }

    // Get current status from the first invitation (all should have same status for this customer)
    $currentStatus = $invitations->first()->status ?? 1;
    
    // Toggle status: 1 -> 0, 0 -> 1
    $newStatus = $currentStatus == 1 ? 0 : 1;

    try {
      // Update status for ALL accepted invitations of this employee for this customer
      $updated = SubcontractorEmployeeInvitation::where('employee_id', $employeeId)
        ->where('customer_id', $customerId)
        ->when($workspaceId, function($query) use ($workspaceId) {
          $query->where('workspace_id', $workspaceId);
        })
        ->where('invitation_status', 'accepted')
        ->update(['status' => $newStatus]);

      return $this->success([
        'status' => $newStatus,
        'invitations_updated' => $updated,
        'employee_type' => 'subcontractor_employee'
      ], 'Employee status updated successfully');
    } catch (\Exception $e) {
      Log::error('Failed to update subcontractor employee status: ' . $e->getMessage());
      return $this->message('Failed to update employee status', 500);
    }
  }


  public function changeCol(Request $request)
  {
    $arr = [];
    $col = $request->col;
    if ($col) {
      DB::table('emp_list_change_cols')->delete();
      for ($i = 0; $i < count($col); $i++) {
        $arr[$i]['col'] = $col[$i];
      }
      EmpListChangeCol::insert($arr);
    }
    return back();
  }

  private function applyEmployeeTypeFilter($query, $request)
  {
    if ($request->has('employee_type')) {
      $employeeType = $request->employee_type;
      if (in_array($employeeType, [self::EMPLOYEE_TYPE_INTERNAL, self::EMPLOYEE_TYPE_EXTERNAL])) {
        $query->where('user_type', $employeeType);
      }
    } elseif ($request->filled('external')) {
      // Legacy support for external parameter
      $query->where('user_type', self::EMPLOYEE_TYPE_EXTERNAL);
    } elseif ($request->filled('internal')) {
      // Legacy support for internal parameter
      $query->where('user_type', self::EMPLOYEE_TYPE_INTERNAL);
    }
  }

  /**
   * Change employee password
   */
  public function changePassword(Request $request)
  {
    $validator = $this->adminResetPasswordValidationRequest($request);
    if ($validator->fails()) {
      return $this->handleValidationFailure($validator);
    }
    $employeeId = Auth::id();
    if (!$employeeId) {
      return $this->message('Employee not authenticated', 401);
    }
    $updated = EmpCompanyDetails::where('id', $employeeId)->update([
      'password' => Hash::make($request->input('password')),
    ]);
    if (!$updated) {
      return $this->message('Employee not found', 404);
    }
    return $this->message('Password changed successfully');
  }

  /**
   * Get employees for starting new chats
   */
  public function getEmployeesForChat(Request $request)
  {
    $user = $request->user();
    if (!$user) {
      return response()->json(['error' => 'Unauthorized'], 401);
    }

    $customerId = $user->customer_id ?? $user->id;
    $workspaceId = $user->workspace_id ?? $user->current_workspace_id;

    $employees = EmpCompanyDetails::where('customer_id', $customerId)
      ->where('workspace_id', $workspaceId)
      ->where('is_active', true)
      ->with(['empPersonalDetails', 'tierEmpPersonalDetail'])
      ->get()
      ->map(function ($emp) {
        $name = 'Unknown User';
        $image = null;

        // Get name from personal details
        if ($emp->empPersonalDetails) {
          $firstName = $emp->empPersonalDetails->first_name ?? '';
          $lastName = $emp->empPersonalDetails->last_name ?? '';
          $name = trim($firstName . ' ' . $lastName) ?: 'Unknown User';
        }

        // Get image from tier personal details
        if ($emp->tierEmpPersonalDetail) {
          $image = $emp->tierEmpPersonalDetail->image;
        }

        return [
          'id' => $emp->id,
          'name' => $name,
          'email' => $emp->employee_email,
          'image' => $image,
          'type' => 'emp'
        ];
      });

    return response()->json([
      'success' => true,
      'data' => $employees
    ]);
  }

  /**
   * Sync employees to Xero
   * Accepts array of employee IDs and syncs them to Xero payroll
   */
  public function employeeSyncToXero(Request $request)
  {
    try {
      // Validation for new payload format
      $validator = Validator::make($request->all(), [
        'employees' => 'required|array|min:1',
        'employees.*.id' => 'required|integer',
        'employees.*.employee_type' => 'required|string|in:subcontractor_employee,regular',
      ]);

      if ($validator->fails()) {
        return $this->handleValidationFailure($validator);
      }
      
      $employees = $request->employees;
      $ids = $this->getCustomerAndWorkspaceIds();
      
      // Check if Xero token exists - use authenticated user's customer_id/workspace_id
      // Xero credentials (client_id/secret) are stored in adminsettings for this customer/workspace
      $xeroToken = XeroToken::where('customer_id', $ids['customer_id'])
        ->where('workspace_id', $ids['workspace_id'])
        ->where('is_active', true)
        ->orderBy('created_at', 'desc')
        ->first();
      if (!$xeroToken) {
        return $this->message('Please connect your Xero account first from settings', 400);
      }
      
      $xeroService = new XeroService($ids['customer_id'], $ids['workspace_id']);
      
      // Separate employees by type
      $subcontractorEmployeeIds = [];
      $regularEmployeeIds = [];
      
      foreach ($employees as $emp) {
        if ($emp['employee_type'] === 'subcontractor_employee') {
          $subcontractorEmployeeIds[] = $emp['id'];
        } else {
          $regularEmployeeIds[] = $emp['id'];
        }
      }
      
      $syncedEmployees = [];
      $failedEmployees = [];
      
      // Process subcontractor employees
      if (!empty($subcontractorEmployeeIds)) {
        $subcontractorEmployees = EmployeeSubcontractor::whereIn('id', $subcontractorEmployeeIds)->get();
        
        if ($subcontractorEmployees->isEmpty()) {
          return $this->message('No valid subcontractor employees found to sync', 404);
        }
        
        foreach ($subcontractorEmployees as $subcontractorEmployee) {
          try {
            // Check if employee already exists in Xero (has xero_emp_id in database)
            $isUpdate = !empty($subcontractorEmployee->xero_emp_id);
            $existingXeroEmpId = $subcontractorEmployee->xero_emp_id;
            // Map subcontractor employee data to Xero format
            $xeroEmployee = $this->mapSubcontractorEmployeeToXero($subcontractorEmployee);
            
            // For updates, include EmployeeID in the payload to prevent duplication
            if ($isUpdate && $existingXeroEmpId) {
              $xeroEmployee['EmployeeID'] = $existingXeroEmpId;
            }
            // Check payroll scopes from services config before making API request
            $currentScopes = config('services.xero.scopes', '');
            $requiredPayrollScopes = ['payroll.employees', 'payroll.payruns', 'payroll.settings'];
            $hasPayrollScope = false;
            
            // Check if any of the required payroll scopes are present in config
            foreach ($requiredPayrollScopes as $scope) {
              if (strpos($currentScopes, $scope) !== false) {
                $hasPayrollScope = true;
                break;
              }
            }
            
            if (!$hasPayrollScope) {
              return $this->message('Xero payroll API requires payroll scopes. Current scopes: ' . $currentScopes, 400);
            }
            
            // Make API request to Xero
            try {
              $response = $xeroService->makeRequest('POST', '/payroll.xro/1.0/Employees', [
                'json' => [$xeroEmployee]
              ]);
            } catch (\Exception $e) {
              // Check if it's a 401 error - likely means token was created without payroll scopes
              if (strpos($e->getMessage(), '401') !== false || strpos($e->getMessage(), 'Unauthorized') !== false) {
                return $this->message('Xero API Error: Unauthorized. Your Xero token was created without payroll scopes. Even though payroll scopes are now configured, you must re-authenticate your Xero connection to create a new token with the updated scopes. Please disconnect and reconnect Xero in your settings. Configured scopes: ' . $currentScopes, 400);
              }
              // Re-throw other errors
              return $this->message('Failed to sync subcontractor employee to Xero: ' . $e->getMessage(), 500);
            }

            // Extract Xero Employee ID from response
            $xeroEmpId = null;
            if (isset($response['Employees']) && is_array($response['Employees']) && count($response['Employees']) > 0) {
              $xeroEmpId = $response['Employees'][0]['EmployeeID'] ?? null;
            } elseif (isset($response['EmployeeID'])) {
              $xeroEmpId = $response['EmployeeID'];
            }

            // Save Xero Employee ID to database ONLY if it's a new employee (CREATE)
            // For updates, we already have xero_emp_id, so don't overwrite it
            if (!$isUpdate && $xeroEmpId) {
              $subcontractorEmployee->xero_emp_id = $xeroEmpId;
              $subcontractorEmployee->save();
              
            } elseif ($isUpdate) {
              // For updates, verify the returned ID matches what we have
              if ($xeroEmpId && $xeroEmpId !== $subcontractorEmployee->xero_emp_id) {
                Log::warning('Xero Employee ID mismatch during update', [
                  'employee_id' => $subcontractorEmployee->id,
                  'stored_xero_emp_id' => $subcontractorEmployee->xero_emp_id,
                  'returned_xero_emp_id' => $xeroEmpId,
                ]);
              }
            }
            
            // Use the existing xero_emp_id for updates, or the new one for creates
            $finalXeroEmpId = $isUpdate ? $subcontractorEmployee->xero_emp_id : ($xeroEmpId ?? null);

            $syncedEmployees[] = [
              'employee_id' => $subcontractorEmployee->id,
              'employee_email' => $subcontractorEmployee->email ?? '',
              'xero_emp_id' => $finalXeroEmpId,
              'action' => $isUpdate ? 'updated' : 'created',
              'xero_response' => $response,
            ];
          } catch (\Exception $e) {
            $errorMessage = $e->getMessage();
            
            // Check if this is a scope error - if so, return early since it affects all employees
            if (strpos($errorMessage, 'payroll scopes') !== false || strpos($errorMessage, 'payroll API requires') !== false) {
              return $this->message('Xero scope error detected during subcontractor employee sync: ' . $errorMessage, 400);
            }
            
            $failedEmployees[] = [
              'employee_id' => $subcontractorEmployee->id,
              'employee_email' => $subcontractorEmployee->email ?? '',
              'error' => $errorMessage,
            ];

            return $this->message('Failed to sync subcontractor employee to Xero: ' . $errorMessage, 500);
          }
        }
      }
      
      // Process regular employees
      if (!empty($regularEmployeeIds)) {
        $employees = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
          ->whereIn('id', $regularEmployeeIds)
          ->where('customer_id', $ids['customer_id'])
          ->where('workspace_id', $ids['workspace_id'])
          ->where('user_type', self::EMPLOYEE_TYPE_INTERNAL) // Only internal employees
          ->with([
            'empPersonalDetails',
            'empEmergencyContacts',
            'empMetaData',
            'accessRole', // Load access role for classification
          ])
          ->get();
          
        if ($employees->isEmpty()) {
          return $this->message('No valid internal employees found to sync', 404);
        }
        
        foreach ($employees as $employee) {
          try {
            // Check if employee already exists in Xero (has xero_emp_id in database)
          // If xero_emp_id exists, it means employee is already in Xero, so UPDATE only
          // If xero_emp_id is NULL/empty, it means employee is NOT in Xero, so CREATE only
          $isUpdate = !empty($employee->xero_emp_id);
          if ($isUpdate) {
            // Employee already exists in Xero - UPDATE only, do not create duplicate
            Log::info('Employee already exists in Xero, updating only', [
              'employee_id' => $employee->id,
              'employee_email' => $employee->employee_email,
              'xero_emp_id' => $employee->xero_emp_id,
            ]);
          } else {
            // Employee does NOT exist in Xero - CREATE new employee
            Log::info('Employee does not exist in Xero, creating new', [
              'employee_id' => $employee->id,
              'employee_email' => $employee->employee_email,
            ]);
          }
          // Map employee data to Xero format
          $xeroEmployee = $this->mapEmployeeToXero($employee);
          // For updates, include EmployeeID in the payload to prevent duplication
          // This tells Xero to UPDATE the existing employee instead of creating a new one
          if ($isUpdate) {
            $xeroEmployee['EmployeeID'] = $employee->xero_emp_id;
          }
          
          // Log the payload being sent for debugging
          Log::info(($isUpdate ? 'Updating' : 'Creating') . ' employee in Xero', [
            'employee_id' => $employee->id,
            'employee_email' => $employee->employee_email,
            'xero_emp_id' => $employee->xero_emp_id,
            'is_update' => $isUpdate,
            'xero_payload' => $xeroEmployee,
          ]);
          
          // Check payroll scopes from services config before making API request
          $currentScopes = config('services.xero.scopes', '');
          $requiredPayrollScopes = ['payroll.employees', 'payroll.payruns', 'payroll.settings'];
          $hasPayrollScope = false;
          
          // Check if any of the required payroll scopes are present in config
          foreach ($requiredPayrollScopes as $scope) {
            if (strpos($currentScopes, $scope) !== false) {
              $hasPayrollScope = true;
              break;
            }
          }
          
          if (!$hasPayrollScope) {
            return $this->message('Xero payroll API requires payroll scopes. Current scopes: ' . $currentScopes, 400);
          }
          
          // Make API request to Xero
          // Xero uses POST for both create and update
          // When EmployeeID is included in payload, it updates existing employee; otherwise it creates new
          try {
            $response = $xeroService->makeRequest('POST', '/payroll.xro/1.0/Employees', [
              'json' => [$xeroEmployee]
            ]);
          } catch (\Exception $e) {
            // Check if it's a 401 error - likely means token was created without payroll scopes
            if (strpos($e->getMessage(), '401') !== false || strpos($e->getMessage(), 'Unauthorized') !== false) {
              return $this->message('Xero API Error: Unauthorized. Your Xero token was created without payroll scopes. Even though payroll scopes are now configured, you must re-authenticate your Xero connection to create a new token with the updated scopes. Please disconnect and reconnect Xero in your settings. Configured scopes: ' . $currentScopes, 400);
            }
            // Re-throw other errors
            return $this->message('Failed to sync employee to Xero: ' . $e->getMessage(), 500);
          }

          // Extract Xero Employee ID from response
          $xeroEmpId = null;
          if (isset($response['Employees']) && is_array($response['Employees']) && count($response['Employees']) > 0) {
            $xeroEmpId = $response['Employees'][0]['EmployeeID'] ?? null;
          } elseif (isset($response['EmployeeID'])) {
            $xeroEmpId = $response['EmployeeID'];
          }

          // Save Xero Employee ID to database ONLY if it's a new employee (CREATE)
          // For updates, we already have xero_emp_id, so don't overwrite it
          if (!$isUpdate && $xeroEmpId) {
            $employee->xero_emp_id = $xeroEmpId;
            $employee->save();
            
          } elseif ($isUpdate) {
            // For updates, verify the returned ID matches what we have
            if ($xeroEmpId && $xeroEmpId !== $employee->xero_emp_id) {
              return $this->message('Xero Employee ID mismatch during update', 400);
            }
          }

          // Use the existing xero_emp_id for updates, or the new one for creates
          $finalXeroEmpId = $isUpdate ? $employee->xero_emp_id : ($xeroEmpId ?? null);

          $syncedEmployees[] = [
            'employee_id' => $employee->id,
            'employee_email' => $employee->employee_email,
            'xero_emp_id' => $finalXeroEmpId,
            'action' => $isUpdate ? 'updated' : 'created',
            'xero_response' => $response,
          ];

        } catch (\Exception $e) {
          $errorMessage = $e->getMessage();
          
          // Check if this is a scope error - if so, return early since it affects all employees
          if (strpos($errorMessage, 'payroll scopes') !== false || strpos($errorMessage, 'payroll API requires') !== false) {
            return $this->message('Xero scope error detected during employee sync: ' . $errorMessage, 400);
          }
          
          $failedEmployees[] = [
            'employee_id' => $employee->id,
            'employee_email' => $employee->employee_email,
            'xero_emp_id' => $employee->xero_emp_id ?? null,
            'error' => $errorMessage,
          ];

          return $this->message('Failed to sync employee to Xero: ' . $errorMessage, 500);
        }
      }
      }

      // Determine response status code
      $statusCode = 200;
      $message = 'Employee sync completed';
      
      if (count($failedEmployees) > 0 && count($syncedEmployees) === 0) {
        // All employees failed
        $statusCode = 400;
        $message = 'Employees failed to sync. Please check the errors.';
      } elseif (count($failedEmployees) > 0) {
        // Partial success
        $statusCode = 207; // Multi-Status
        $message = 'Employee sync completed with some failures.';
      } elseif (count($syncedEmployees) > 0) {
        // All succeeded
        $message = 'Employees synced successfully.';
      }

      return response()->json([
        'message' => $message,
        'statusCode' => $statusCode,
        'data' => [
          'synced_count' => count($syncedEmployees),
          'failed_count' => count($failedEmployees),
          'synced_employees' => $syncedEmployees,
          'failed_employees' => $failedEmployees,
        ],
      ], $statusCode);
    } catch (\Exception $e) {
      return $this->message('Failed to sync employees to Xero: ' . $e->getMessage(), 500);
    }
  }

  /**
   * Get a single employee from Xero by Xero Employee ID
   * Uses Xero API v2.0: GET /payroll.xro/2.0/employees/{employeeID}
   */
  public function getEmployeeFromXero($xeroEmployeeId)
  {
    try {
      $ids = $this->getCustomerAndWorkspaceIds();

      // Check if Xero token exists
      $xeroToken = XeroToken::where('customer_id', $ids['customer_id'])
        ->where('workspace_id', $ids['workspace_id'])
        ->where('is_active', true)
        ->orderBy('created_at', 'desc')
        ->first();

      if (!$xeroToken) {
        return $this->message('Please connect your Xero account first from settings', 400);
      }

      // Get XeroService with customer/workspace credentials
      $xeroService = new XeroService($ids['customer_id'], $ids['workspace_id']);

      // Validate xero_employee_id format (should be a UUID)
      if (empty($xeroEmployeeId)) {
        return $this->error('Xero Employee ID is required', 400);
      }

      Log::info('Fetching employee from Xero', [
        'xero_employee_id' => $xeroEmployeeId,
        'customer_id' => $ids['customer_id'],
        'workspace_id' => $ids['workspace_id'],
      ]);

      // Make API request to Xero - using v2.0 endpoint
      $response = $xeroService->makeRequest('GET', '/payroll.xro/1.0/Employees/' . $xeroEmployeeId);

      // Extract employee data from response
      $employeeData = null;
      if (isset($response['Employees']) && is_array($response['Employees']) && count($response['Employees']) > 0) {
        $employeeData = $response['Employees'][0];
      } elseif (isset($response['EmployeeID'])) {
        // If response is a single employee object
        $employeeData = $response;
      }

      if (!$employeeData) {
        return $this->error('Employee not found in Xero', 404);
      }

      Log::info('Employee fetched from Xero successfully', [
        'xero_employee_id' => $xeroEmployeeId,
        'employee_name' => ($employeeData['FirstName'] ?? '') . ' ' . ($employeeData['LastName'] ?? ''),
      ]);

      return $this->success($employeeData, 'Employee fetched from Xero successfully');
    } catch (\Exception $e) {
      $errorMessage = $e->getMessage();

      Log::error('Failed to fetch employee from Xero', [
        'xero_employee_id' => $xeroEmployeeId ?? null,
        'error' => $errorMessage,
        'trace' => $e->getTraceAsString(),
      ]);

      // Check if it's a 404 error (employee not found)
      if (strpos($errorMessage, '404') !== false || strpos($errorMessage, 'not found') !== false) {
        return $this->error('Employee not found in Xero', 404);
      }

      return $this->error('Failed to fetch employee from Xero: ' . $errorMessage, 500);
    }
  }

  /**
   * Map employee data to Xero format
   * Maps all available fields from emp_company_details and emp_personal_details tables
   */
  private function mapEmployeeToXero($employee)
  {
    $personalDetails = $employee->empPersonalDetails;
    
    if (!$personalDetails) {
      throw new \Exception('Employee personal details not found');
    }

    // Validate required fields
    if (empty($personalDetails->first_name)) {
      throw new \Exception('First name is required for Xero employee sync');
    }
    if (empty($personalDetails->last_name)) {
      throw new \Exception('Last name is required for Xero employee sync');
    }
    // Get raw attribute to avoid automatic date casting issues
    try {
      // Try multiple methods to get raw value before casting
      if (method_exists($personalDetails, 'getRawOriginal')) {
        $dateOfBirthRaw = $personalDetails->getRawOriginal('date_of_birth');
      } elseif (isset($personalDetails->getAttributes()['date_of_birth'])) {
        $dateOfBirthRaw = $personalDetails->getAttributes()['date_of_birth'];
      } else {
        // Last resort: access attribute and convert to string if Carbon
        $dateOfBirthRaw = $personalDetails->date_of_birth;
        if ($dateOfBirthRaw instanceof \Carbon\Carbon) {
          $dateOfBirthRaw = $dateOfBirthRaw->format('Y-m-d');
        }
      }
      
      // If still empty, try direct attribute access
      if (empty($dateOfBirthRaw)) {
        $dateOfBirthRaw = $personalDetails->date_of_birth;
        if ($dateOfBirthRaw instanceof \Carbon\Carbon) {
          $dateOfBirthRaw = $dateOfBirthRaw->format('Y-m-d');
        }
      }
    } catch (\Exception $e) {
      // If all methods fail, try accessing attribute directly
      Log::warning('Failed to get raw date_of_birth, trying direct access', [
        'employee_id' => $employee->id ?? null,
        'error' => $e->getMessage(),
      ]);
      $dateOfBirthRaw = $personalDetails->date_of_birth;
      if ($dateOfBirthRaw instanceof \Carbon\Carbon) {
        $dateOfBirthRaw = $dateOfBirthRaw->format('Y-m-d');
      }
    }
    $dateOfBirth = $this->formatDateForXero($dateOfBirthRaw);
    if (empty($dateOfBirth)) {
      throw new \Exception('Date of birth is required and must be a valid date for Xero employee sync');
    }
    
    // Required fields
    $xeroEmployee = [
      'FirstName' => substr(trim($personalDetails->first_name), 0, 35),
      'LastName' => substr(trim($personalDetails->last_name), 0, 35),
      'DateOfBirth' => $dateOfBirth,
      'HomeAddress' => [
        'AddressLine1' => substr(trim($personalDetails->streat_address ?? 'Not Provided'), 0, 50),
        'City' => substr(trim($personalDetails->suburb ?? 'Not Provided'), 0, 50),
        'Region' => $this->mapStateToRegion($personalDetails->state ?? 'NSW'),
        'PostalCode' => $this->validatePostcode($personalDetails->postcode ?? '2000'),
        'Country' => 'AUSTRALIA',
      ],
    ];

    // Optional fields from emp_personal_details - only include if data exists
    if (!empty($personalDetails->middle_name)) {
      $xeroEmployee['MiddleNames'] = substr($personalDetails->middle_name, 0, 35);
    }

    // Optional fields from emp_company_details
    if (!empty($employee->employee_email)) {
      $xeroEmployee['Email'] = substr($employee->employee_email, 0, 100);
    }

    if (!empty($personalDetails->mobile)) {
      $xeroEmployee['Mobile'] = substr($personalDetails->mobile, 0, 50);
    }

    if (!empty($employee->job_title)) {
      $xeroEmployee['JobTitle'] = substr($employee->job_title, 0, 100);
    }

    if (!empty($employee->employment_start_date)) {
      // Get raw attribute to avoid automatic date casting issues
      try {
        // Try multiple methods to get raw value before casting
        if (method_exists($employee, 'getRawOriginal')) {
          $startDateRaw = $employee->getRawOriginal('employment_start_date');
        } elseif (isset($employee->getAttributes()['employment_start_date'])) {
          $startDateRaw = $employee->getAttributes()['employment_start_date'];
        } else {
          // Last resort: access attribute and convert to string if Carbon
          $startDateRaw = $employee->employment_start_date;
          if ($startDateRaw instanceof \Carbon\Carbon) {
            $startDateRaw = $startDateRaw->format('Y-m-d');
          }
        }
        
        // If still empty, try direct attribute access
        if (empty($startDateRaw)) {
          $startDateRaw = $employee->employment_start_date;
          if ($startDateRaw instanceof \Carbon\Carbon) {
            $startDateRaw = $startDateRaw->format('Y-m-d');
          }
        }
      } catch (\Exception $e) {
        // If all methods fail, try accessing attribute directly
        Log::warning('Failed to get raw employment_start_date, trying direct access', [
          'employee_id' => $employee->id,
          'error' => $e->getMessage(),
        ]);
        $startDateRaw = $employee->employment_start_date;
        if ($startDateRaw instanceof \Carbon\Carbon) {
          $startDateRaw = $startDateRaw->format('Y-m-d');
        }
      }
      $xeroEmployee['StartDate'] = $this->formatDateForXero($startDateRaw);
    }

    // Employment Type - send simple "EMPLOYEE" value
    $xeroEmployee['EmploymentType'] = 'EMPLOYEE';

    // Classification - can use access_role title or employment type
    if ($employee->accessRole && !empty($employee->accessRole->title)) {
      $xeroEmployee['Classification'] = substr($employee->accessRole->title, 0, 100);
    }

    // // Termination Date - Only include if Status is ARCHIVED
    // // If TerminationDate is provided, Xero requires TerminationReason
    // if (!empty($employee->employment_end_date) && isset($xeroEmployee['Status']) && $xeroEmployee['Status'] === 'ARCHIVED') {
    //   $xeroEmployee['TerminationDate'] = $this->formatDateForXero($employee->employment_end_date);
    //   // Xero requires TerminationReason when TerminationDate is provided
    //   // Common values: RESIGNED, DISMISSED, REDUNDANCY, RETIREMENT, OTHER
    //   $xeroEmployee['TerminationReason'] = 'OTHER'; // Default, can be customized based on your data
    // } elseif (!empty($employee->employment_end_date) && (!isset($xeroEmployee['Status']) || $xeroEmployee['Status'] === 'ACTIVE')) {
    //   // If employee is ACTIVE but has end date, don't include TerminationDate
    //   // This prevents validation errors
    // }

    // Status - map from status field (0 = inactive, 1 = active)
    // Xero Status: ACTIVE, ARCHIVED
    if (isset($employee->status)) {
      $xeroEmployee['Status'] = ($employee->status == 1) ? 'ACTIVE' : 'ARCHIVED';
    }

    // Country of Residence - from citizenship_status or legally_australia
    // Xero requires ISO 3166-1 alpha-2 country codes (e.g., 'AU', 'US', 'NZ')
    if (!empty($employee->citizenship_status)) {
      $xeroEmployee['CountryOfResidence'] = $this->mapCountryOfResidence($employee->citizenship_status);
    } elseif (!empty($employee->legally_australia)) {
      // If legally_australia is false, might be from another country
      if ($employee->legally_australia === 'false' || $employee->legally_australia === false) {
        // Could be working holiday maker or other - would need more info
        // For now, default to AU (Australia) if not specified
        $xeroEmployee['CountryOfResidence'] = 'AU';
      } else {
        $xeroEmployee['CountryOfResidence'] = 'AU'; // Australia - ISO code
      }
    }

    // Bank account information from user_meta
    $userMeta = DB::table('user_meta')
      ->where('emp_id', $employee->id)
      ->get();

    $accountNumber = $this->searchInMeta($userMeta, 'account_number');
    $bankName = $this->searchInMeta($userMeta, 'bank_name');
    $bsb = $this->searchInMeta($userMeta, 'bsb');
    $accountHolderName = $personalDetails->account_holder_name ?? null;
    
    // If account_number is not in user_meta, try to get from empMetaData relation
    if (!$accountNumber && $employee->empMetaData) {
      foreach ($employee->empMetaData as $meta) {
        if ($meta->option === 'account_number') {
          $accountNumber = $meta->value;
        } elseif ($meta->option === 'bsb') {
          $bsb = $meta->value;
        } elseif ($meta->option === 'bank_name') {
          $bankName = $meta->value;
        }
      }
    }

    if ($accountNumber && $bsb) {
      $xeroEmployee['BankAccounts'] = [
        [
          'StatementText' => substr($accountHolderName ?? $personalDetails->first_name . ' ' . $personalDetails->last_name, 0, 18),
          'AccountName' => substr($accountHolderName ?? $personalDetails->first_name . ' ' . $personalDetails->last_name, 0, 32),
          'BSB' => substr($bsb, 0, 6),
          'AccountNumber' => substr($accountNumber, 0, 9),
          'Remainder' => true,
        ],
      ];
    }

    // // Tax file number and Tax Declaration
    // if (!empty($employee->tax_file_no)) {
    //   $taxDeclaration = [
    //     'TaxFileNumber' => $employee->tax_file_no,
    //     'EmploymentBasis' => $this->mapEmploymentBasis($employee->employment_type),
    //     'AustralianResidentForTaxPurposes' => ($employee->legally_australia === 'true' || $employee->legally_australia === true || $employee->legally_australia === '1'),
    //     'TaxFreeThresholdClaimed' => true, // Default to true, can be customized
    //   ];
      
    //   // Add residency status if available
    //   if (!empty($employee->citizenship_status)) {
    //     $taxDeclaration['ResidencyStatus'] = $this->mapResidencyStatus($employee->citizenship_status);
    //   }
      
    //   $xeroEmployee['TaxDeclaration'] = $taxDeclaration;
    // }

    return $xeroEmployee;
  }

  /**
   * Map employment type title from emp_types table to Xero EmploymentType
   * Xero valid values: FULLTIME, PARTTIME, CASUAL, LABOURHIRE, SUPERINCOMESTREAM, CONTRACTOR
   */


  /**
   * Map employment basis for Tax Declaration
   */
  private function mapEmploymentBasis($employmentTypeId)
  {
    if (empty($employmentTypeId)) {
      return 'FULLTIME';
    }
    
    $empType = EmpType::find($employmentTypeId);
    if ($empType) {
      $typeTitle = strtoupper(trim($empType->title ?? ''));
      
      // Map to Xero EmploymentBasis: FULLTIME, PARTTIME, CASUAL, LABOURHIRE, SUPERINCOMESTREAM, CONTRACTOR
      if (stripos($typeTitle, 'FULL') !== false) {
        return 'FULLTIME';
      } elseif (stripos($typeTitle, 'PART') !== false) {
        return 'PARTTIME';
      } elseif (stripos($typeTitle, 'CASUAL') !== false) {
        return 'CASUAL';
      } elseif (stripos($typeTitle, 'CONTRACT') !== false) {
        return 'CONTRACTOR';
      } elseif (stripos($typeTitle, 'LABOUR') !== false || stripos($typeTitle, 'LABOR') !== false) {
        return 'LABOURHIRE';
      }
    }
    
    return 'FULLTIME'; // Default
  }

  /**
   * Map residency status for Tax Declaration
   */
  private function mapResidencyStatus($citizenshipStatus)
  {
    $statusUpper = strtoupper(trim($citizenshipStatus));
    
    // Xero ResidencyStatus: AUSTRALIANRESIDENT, FOREIGNRESIDENT, WORKINGHOLIDAYMAKER
    if (stripos($statusUpper, 'AUSTRALIAN') !== false || stripos($statusUpper, 'RESIDENT') !== false) {
      return 'AUSTRALIANRESIDENT';
    } elseif (stripos($statusUpper, 'FOREIGN') !== false || stripos($statusUpper, 'NON') !== false) {
      return 'FOREIGNRESIDENT';
    } elseif (stripos($statusUpper, 'HOLIDAY') !== false || stripos($statusUpper, 'WORKING') !== false) {
      return 'WORKINGHOLIDAYMAKER';
    }
    
    return 'AUSTRALIANRESIDENT'; // Default
  }

  /**
   * Map country of residence
   */
  private function mapCountryOfResidence($citizenshipStatus)
  {
    // Xero requires ISO 3166-1 alpha-2 country codes (e.g., 'AU', 'US', 'NZ')
    // Not full country names like 'AUSTRALIA'
    // Handle both integer and string values for citizenship_status
    if (empty($citizenshipStatus) && $citizenshipStatus !== 0) {
      return 'AU'; // Default to Australia
    }
    
    // Convert to string if it's an integer
    $statusString = is_numeric($citizenshipStatus) ? (string)$citizenshipStatus : $citizenshipStatus;
    $statusUpper = strtoupper(trim($statusString));
    
    // Map common citizenship status values to ISO country codes
    // Handle integer values (common: 1=Australian, 2=New Zealand, etc.)
    // Add more mappings based on your data
    if (stripos($statusUpper, 'AUSTRALIA') !== false || stripos($statusUpper, 'AUSTRALIAN') !== false || $statusUpper === '1' || $statusUpper === 'AU') {
      return 'AU'; // Australia
    } elseif (stripos($statusUpper, 'NEW ZEALAND') !== false || stripos($statusUpper, 'KIWI') !== false || $statusUpper === '2' || $statusUpper === 'NZ') {
      return 'NZ'; // New Zealand
    } elseif (stripos($statusUpper, 'UNITED STATES') !== false || stripos($statusUpper, 'AMERICAN') !== false || stripos($statusUpper, 'USA') !== false || $statusUpper === '3' || $statusUpper === 'US') {
      return 'US'; // United States
    } elseif (stripos($statusUpper, 'UNITED KINGDOM') !== false || stripos($statusUpper, 'BRITISH') !== false || stripos($statusUpper, 'UK') !== false || $statusUpper === '4' || $statusUpper === 'GB') {
      return 'GB'; // United Kingdom
    } elseif (stripos($statusUpper, 'CANADA') !== false || stripos($statusUpper, 'CANADIAN') !== false || $statusUpper === '5' || $statusUpper === 'CA') {
      return 'CA'; // Canada
    }
    
    // Default to Australia (AU) - ISO code, not full name
    return 'AU';
  }

  /**
   * Map subcontractor employee data to Xero format
   * Similar to mapEmployeeToXero but for employees_subcontractors table
   */
  private function mapSubcontractorEmployeeToXero($subcontractorEmployee)
  {
    // Validate required fields
    if (empty($subcontractorEmployee->first_name)) {
      throw new \Exception('First name is required for Xero employee sync');
    }
    if (empty($subcontractorEmployee->last_name)) {
      throw new \Exception('Last name is required for Xero employee sync');
    }
    // Get date of birth
    $dateOfBirthRaw = $subcontractorEmployee->dob;
    if ($dateOfBirthRaw instanceof \Carbon\Carbon) {
      $dateOfBirthRaw = $dateOfBirthRaw->format('Y-m-d');
    }
    $dateOfBirth = $this->formatDateForXero($dateOfBirthRaw);
    if (empty($dateOfBirth)) {
      throw new \Exception('Date of birth is required and must be a valid date for Xero employee sync');
    }
    // Required fields
    $xeroEmployee = [
      'FirstName' => substr(trim($subcontractorEmployee->first_name), 0, 35),
      'LastName' => substr(trim($subcontractorEmployee->last_name), 0, 35),
      'DateOfBirth' => $dateOfBirth,
      'HomeAddress' => [
        'AddressLine1' => substr(trim($subcontractorEmployee->address ?? 'Not Provided'), 0, 50),
        'City' => substr(trim($subcontractorEmployee->suburb ?? 'Not Provided'), 0, 50),
        'Region' => $this->mapStateToRegion($subcontractorEmployee->state ?? 'NSW'),
        'PostalCode' => '2000', // Default postcode since it's not in the table
        'Country' => 'AUSTRALIA',
      ],
    ];
    // Optional fields - only include if data exists
    if (!empty($subcontractorEmployee->middle_name)) {
      $xeroEmployee['MiddleNames'] = substr($subcontractorEmployee->middle_name, 0, 35);
    }
    if (!empty($subcontractorEmployee->email)) {
      $xeroEmployee['Email'] = substr($subcontractorEmployee->email, 0, 100);
    }
    if (!empty($subcontractorEmployee->mobile)) {
      $xeroEmployee['Mobile'] = substr($subcontractorEmployee->mobile, 0, 50);
    } elseif (!empty($subcontractorEmployee->phone)) {
      $xeroEmployee['Mobile'] = substr($subcontractorEmployee->phone, 0, 50);
    }
    // Employment Type - send simple "EMPLOYEE" value
    $xeroEmployee['EmploymentType'] = 'EMPLOYEE';
    // Status - default to ACTIVE for subcontractor employees
    $xeroEmployee['Status'] = 'ACTIVE';
    // Country of Residence - from citizenship_status
    // Xero requires ISO 3166-1 alpha-2 country codes (e.g., 'AU', 'US', 'NZ')
    if (!empty($subcontractorEmployee->citizenship_status)) {
      $xeroEmployee['CountryOfResidence'] = $this->mapCountryOfResidence($subcontractorEmployee->citizenship_status);
    } else {
      $xeroEmployee['CountryOfResidence'] = 'AU'; // Australia - ISO code
    }
    return $xeroEmployee;
  }

  /**
   * Format date for Xero in YYYY-MM-DD format
   * This function ensures all dates sent to Xero API are in the correct format: YYYY-MM-DD (e.g., 2025-12-17)
   * 
   * @param mixed $date Date value (string, Carbon instance, or object)
   * @return string|null Date in YYYY-MM-DD format, or null if parsing fails
   */
  private function formatDateForXero($date)
  {
    if (empty($date)) {
      return null;
    }

    // If it's already a Carbon instance, just format it to YYYY-MM-DD
    if ($date instanceof \Carbon\Carbon) {
      return $date->format('Y-m-d'); // Returns YYYY-MM-DD format
    }

    // Convert to string if it's an object
    if (is_object($date)) {
      $date = (string) $date;
    }

    try {
      // Use safeCarbonParse to handle various date formats including m-d-Y (12-17-2025)
      $carbonDate = \App\Models\BaseModel::safeCarbonParse($date, 'EmployeeController formatDateForXero');
      
      if (!($carbonDate instanceof \Carbon\Carbon)) {
        Log::warning('Failed to parse date for Xero - not a valid Carbon instance', [
          'date' => $date,
          'date_type' => gettype($date),
        ]);
        return null;
      }

      // Return date in YYYY-MM-DD format for Xero API
      return $carbonDate->format('Y-m-d'); // Returns YYYY-MM-DD format (e.g., 2025-12-17)
    } catch (\Exception $e) {
      Log::warning('Failed to format date for Xero', [
        'date' => $date,
        'date_type' => gettype($date),
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
      ]);
      return null;
    }
  }

  /**
   * Map Australian state to Xero region format
   * Xero only accepts valid Australian state codes: NSW, VIC, QLD, WA, SA, TAS, ACT, NT
   */
  private function mapStateToRegion($state)
  {
    $stateMap = [
      'NSW' => 'NSW',
      'NEW SOUTH WALES' => 'NSW',
      'VIC' => 'VIC',
      'VICTORIA' => 'VIC',
      'QLD' => 'QLD',
      'QUEENSLAND' => 'QLD',
      'WA' => 'WA',
      'WESTERN AUSTRALIA' => 'WA',
      'SA' => 'SA',
      'SOUTH AUSTRALIA' => 'SA',
      'TAS' => 'TAS',
      'TASMANIA' => 'TAS',
      'ACT' => 'ACT',
      'AUSTRALIAN CAPITAL TERRITORY' => 'ACT',
      'NT' => 'NT',
      'NORTHERN TERRITORY' => 'NT',
    ];

    $stateUpper = strtoupper(trim($state));
    
    // Check if it's a valid state code
    if (isset($stateMap[$stateUpper])) {
      return $stateMap[$stateUpper];
    }
    
    // If not found, default to NSW (most common) to avoid validation errors
    // Log a warning for invalid states
    if (!empty($state)) {
      Log::warning('Invalid Australian state code for Xero', [
        'state' => $state,
        'defaulting_to' => 'NSW',
      ]);
    }
    
    return 'NSW'; // Default to NSW if state is invalid or empty
  }

  /**
   * Validate and format Australian postcode for Xero
   * Xero requires postcodes between 0200-9999
   */
  private function validatePostcode($postcode)
  {
    if (empty($postcode)) {
      return '2000'; // Default to Sydney postcode
    }
    
    // Remove any non-numeric characters
    $postcode = preg_replace('/[^0-9]/', '', $postcode);
    
    // Convert to integer for validation
    $postcodeInt = (int)$postcode;
    
    // Xero requires postcodes between 0200-9999
    if ($postcodeInt >= 200 && $postcodeInt <= 9999) {
      return str_pad($postcode, 4, '0', STR_PAD_LEFT);
    }
    
    // If invalid, default to 2000 and log warning
    Log::warning('Invalid Australian postcode for Xero, defaulting to 2000', [
      'original_postcode' => $postcode,
    ]);
    
    return '2000'; // Default to Sydney postcode
  }

  public function getSubcontractorEmployees(Request $request)
  {
    $subcontractorEmployees = EmployeeSubcontractor::whereHas('associations', function ($query) use ($request) {
      $query->where('subcontractor_id', $request->subcontractor_id);
    })
      ->get();
    return $this->withCount($subcontractorEmployees, 'Subcontractor employees fetched successfully');
  }
}
