<?php

namespace App\Http\Controllers;

use App\Models\Lead;
use App\Models\Referral;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LeadController extends Controller
{
    /**
     * Display a listing of leads.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $leads = Lead::with('referral')->get();
        return $this->withCount($leads, 'Leads retrieved successfully');
    }

    /**
     * Store a newly created lead through a referral link.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  string  $referralCode
     * @return \Illuminate\Http\Response
     */
    public function storeFromReferral(Request $request, $referralCode)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255',
            'contact_number' => 'required|string|max:20',
            'review' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation error', 422);
        }

        // Find the referral by code
        $referral = Referral::where('referral_code', $referralCode)->first();
        if (!$referral) {
            return $this->error('Invalid referral code', 404);
        }

        // Create the lead
        $lead = Lead::create([
            'name' => $request->name,
            'email' => $request->email,
            'contact_number' => $request->contact_number,
            'review' => $request->review,
            'referral_id' => $referral->id,
        ]);

        return $this->success($lead, 'Thank you for your submission!');
    }

    /**
     * Display the specified lead.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $lead = Lead::with('referral')->findOrFail($id);
        return $this->success($lead, 'Lead retrieved successfully');
    }

    /**
     * Update the specified lead.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255',
            'contact_number' => 'required|string|max:20',
            'review' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation error', 422);
        }

        $lead = Lead::findOrFail($id);
        $lead->update([
            'name' => $request->name,
            'email' => $request->email,
            'contact_number' => $request->contact_number,
            'review' => $request->review,
        ]);

        return $this->success($lead, 'Lead updated successfully');
    }

    /**
     * Remove the specified lead.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $lead = Lead::findOrFail($id);
        $lead->delete();

        return $this->message('Lead deleted successfully');
    }

    /**
     * Display the referral form page.
     *
     * @param  string  $referralCode
     * @return \Illuminate\Http\Response
     */
    public function showReferralForm($referralCode)
    {
        // Find the referral by code
        $referral = Referral::where('referral_code', $referralCode)->first();
        if (!$referral) {
            return $this->error('Invalid referral code', 404);
        }

        $data = [
            'referral_code' => $referralCode,
            'referred_by' => $referral->name
        ];
        
        return $this->success($data, 'Referral form loaded successfully');
    }
} 