<?php

namespace App\Http\Controllers;

use PDF;
use \Carbon\Carbon;
use App\Models\Sites;
use App\Models\Meeting;
use App\Models\MeetingUser;
use Illuminate\Http\Request;
use App\General\GeoLoacation;
use App\Models\MeetingHistory;
use App\Models\User;
use App\Models\MeetingDocument;
use App\Models\MeetingGuestUser;
use App\Models\MeetingNoteTable;
use App\Models\MeetingOrganiser;
use App\Models\EmpCompanyDetails;
use App\Models\MeetingAttendance;
use App\Models\EmployeeAttendance;
use App\Models\EmpPersonalDetails;
use Illuminate\Support\Facades\DB;
use App\Models\MeetingDocumentNote;
use App\Models\MeetingNoteSignature;
use Illuminate\Support\Facades\Auth;
use App\Models\EmployeeSignedDocument;
use App\Models\MeetingDocumentSetting;
use App\Models\MeetingNoteDocumentsPdf;
use Illuminate\Support\Facades\Validator;
use App\Models\MeetingDocumentNoteCheckpoint;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Models\Adminsettings;
use Illuminate\Support\Facades\Log;

class MeetingController extends Controller
{
    public $current_date;
    public function __construct()
    {
        $this->middleware('Permissions:Meetings Manager Maintain')->only(['create', 'edit']);
        $this->current_date = date('Y-m-d');
    }

    public function index(Request $request)
    {
        $query = Meeting::with('created_by', 'meeting_users', 'meeting_organisers', 'meeting_site')
            ->where('is_deleted', 0);
        $query = $this->applyCustomerWorkspaceFilter($query);
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('agenda', 'like', '%' . $searchTerm . '%')
                    ->orWhereHas('created_by', function ($subquery) use ($searchTerm) {
                        $subquery->where(function ($nameQuery) use ($searchTerm) {
                            $nameQuery->where('first_name', 'like', '%' . $searchTerm . '%')
                                      ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                                      ->orWhere('last_name', 'like', '%' . $searchTerm . '%');
                        });
                    });
            });
        }
        if($request->filled('today_meetings')){
            $today = date('Y-m-d');

            $query->where(function ($q) use ($today) {
                $q->whereDate('start_date', '<=', $today)
                  ->whereDate('end_date', '>=', $today);
            });
        }
        
        $query_result = $query->orderBy("id", "desc")->get();
        return $this->withCount($query_result, 'Meetings retrieved successfully.');
    }
 
    public function store(Request $request)
    {

        // Build validation rules
        $rules = [
            'agenda' => 'required|string',
            'organiser_ids' => 'required|array',
            'is_recurring' => 'required|in:0,1',
            'frequency' => 'nullable|required_if:is_recurring,1|integer|min:1',
            'days' => 'nullable',
            'start_date' => 'required|date|after_or_equal:today',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'start_time' => 'required',
            'end_time' => 'required|after:start_time',
            'description' => 'required|string',
            'employee_ids' => 'nullable|array',
            'employee_ids.*' => 'nullable|exists:emp_company_details,id,del,0',
            'subcontractor_employee_ids' => 'nullable|array',
            'subcontractor_employee_ids.*' => 'nullable|exists:employees_subcontractors,id',
            'site_id' => 'required|exists:sites,id,active,1,del,0',
            'project_id' => 'required|exists:projects,id,active,1',
            'allow_meeting_members' => 'nullable|integer',
            'is_before_meeting' => 'nullable|integer'
        ];

        // Add additional validation for specific meeting members
        if ($request->allow_meeting_members == config('constants.allow_meeting_members.specific.key')) {
            $rules['employee_ids'] = 'required_without:subcontractor_employee_ids|array';
            $rules['subcontractor_employee_ids'] = 'required_without:employee_ids|array';
        }

        $validator = Validator::make(
            $request->all(),
            $rules,
            [
                'agenda.required' => "Please add Meeting Title.",
                'organiser_ids.required' => "Please select an Organiser.",
                'site_id.required' => "Please select a site.",
                'project_id.required' => "Please select a project.",
                'frequency.required_if' => "Frequency is required when meeting is recurring.",
                'days.required_if' => "Days are required when meeting is recurring.",
                'days.regex' => "Days format must be comma-separated digits like 1,2,3.",
                'employee_ids.required_without' => 'Please select at least one employee or subcontractor employee for the meeting.',
                'subcontractor_employee_ids.required_without' => 'Please select at least one employee or subcontractor employee for the meeting.',
            ]
        );

        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }

        $validatedData = $validator->validated();
        $userTable = $this->getUserTable();

        // Apply customer and emp conditions
        $query = Meeting::query();

        if ($userTable === 'customer') {
            $customerId = auth()->user()->id;
            $workspaceId = auth()->user()->current_workspace_id;

            $query->where('created_by', $customerId)
                ->where('workspace_id', $workspaceId);
        }

        if ($userTable === 'emp') {
            $customerId = auth()->user()->customer_id;
            $workspaceId = auth()->user()->workspace_id;

            $query->where('created_by', $customerId)
                ->where('workspace_id', $workspaceId);
        }

        if ($userTable !== 'customer' && $userTable !== 'emp') {
            return response()->json(['message' => 'Unauthorized access.'], 403);
        }

        $end_date = $request->is_recurring ? $request->end_date : $request->start_date;
        // Create the meeting
        $meeting = Meeting::create([
            'created_by' => auth()->user()->id,
            'workspace_id' => $workspaceId,
            'customer_id' =>  $customerId,
            'site_id' => $validatedData['site_id'],
            'project_id' => $validatedData['project_id'],
            'agenda' => $validatedData['agenda'],
            'description' => $validatedData['description'],
            'status' => config('constants.meeting_status.pending.key'),
            'start_date' => $validatedData['start_date'],
            'end_date' => $end_date,
            'start_time' => $validatedData['start_time'],
            'end_time' => $validatedData['end_time'],
            'is_recurring' => $validatedData['is_recurring'],
            'allow_meeting_members' => $validatedData['allow_meeting_members'],
            'frequency' => $validatedData['frequency'] ?? null,
            'days' => $validatedData['days'] ?? null,
            'allow_guest' => $request->allow_guest ?? null,
            'is_before_meeting' => $request->is_before_meeting ?? 0
        ]);

        if (!$meeting) {
            return $this->message('Failed to create the meeting.', 500);
        }

        // Attach organisers
        foreach ($validatedData['organiser_ids'] as $organiser_id) {
            MeetingOrganiser::create([
                'meeting_id' => $meeting->id,
                'site_id' => $validatedData['site_id'],
                'organiser_id' => $organiser_id
            ]);
        }
        // Attach employees if specific members are allowed
        if ($request->allow_meeting_members == config('constants.allow_meeting_members.specific.key')) {
            // Store regular employees (user_type = 0)
            if (!empty($request->employee_ids)) {
                foreach ($request->employee_ids as $employee_id) {
                    MeetingUser::create([
                        'meeting_id' => $meeting->id,
                        'site_id' => $meeting->site_id,
                        'employee_id' => $employee_id,
                        'user_type' => 0, // Regular employee
                    ]);
                }
            }
            
            // Store subcontractor employees (user_type = 1)
            if (!empty($request->subcontractor_employee_ids)) {
                foreach ($request->subcontractor_employee_ids as $subcontractor_employee_id) {
                    MeetingUser::create([
                        'meeting_id' => $meeting->id,
                        'site_id' => $meeting->site_id,
                        'employee_id' => $subcontractor_employee_id,
                        'user_type' => 1, // Subcontractor employee
                    ]);
                }
            }
        }
        // Notifications and history
        if ($userTable === 'emp') {
            $added_by = EmpPersonalDetails::where('emp_id', auth()->user()->id)->first();
            $name = $added_by->first_name . ' ' . $added_by->middle_name . ' ' . $added_by->last_name;
        } else {
            $added_by = User::where('id', auth()->user()->id)->first();
            $name = $added_by->name;
        }

        $history_description = sprintf(
            "%s created this meeting at %s",
            strip_tags($name), // Strips any HTML tags from the name
            now()->format('d/m/Y H:i:s'),
            url('/'),
            auth()->user()->id
        );

        // Send notifications to organisers
        foreach ($validatedData['organiser_ids'] as $organiser_id) {
            $this->save_notifications(
                "New Meeting Created",
                "You have been assigned as an organiser for the meeting: {$validatedData['agenda']} on " . date('d/m/Y', strtotime($validatedData['start_date'])) . " at " . date('h:i A', strtotime($validatedData['start_time'])),
                config('constants.employee_types.customer'),
                auth()->user()->id, // sender_id (meeting creator)
                config('constants.employee_types.employee'),
                $organiser_id, // receiver_id (organiser)
                'meeting_created', // notification type
                $customerId,
                $workspaceId
            );
        }

        // Send notifications to meeting participants (if specific members are allowed)
        if ($request->allow_meeting_members == config('constants.allow_meeting_members.specific.key')) {
            // Send notifications to regular employees
            if (!empty($request->employee_ids)) {
                foreach ($request->employee_ids as $employee_id) {
                    $this->save_notifications(
                        "New Meeting Invitation",
                        "You have been invited to attend the meeting: {$validatedData['agenda']} on " . date('d/m/Y', strtotime($validatedData['start_date'])) . " at " . date('h:i A', strtotime($validatedData['start_time'])),
                        config('constants.employee_types.customer'),
                        auth()->user()->id, // sender_id (meeting creator)
                        config('constants.employee_types.employee'),
                        $employee_id, // receiver_id (meeting participant)
                        'meeting_invitation', // notification type
                        $customerId,
                        $workspaceId
                    );
                }
            }
            
            // Send notifications to subcontractor employees
            if (!empty($request->subcontractor_employee_ids)) {
                foreach ($request->subcontractor_employee_ids as $subcontractor_employee_id) {
                    $this->save_notifications(
                        "New Meeting Invitation",
                        "You have been invited to attend the meeting: {$validatedData['agenda']} on " . date('d/m/Y', strtotime($validatedData['start_date'])) . " at " . date('h:i A', strtotime($validatedData['start_time'])),
                        config('constants.employee_types.customer'),
                        auth()->user()->id, // sender_id (meeting creator)
                        config('constants.employee_types.employee'),
                        $subcontractor_employee_id, // receiver_id (subcontractor employee)
                        'meeting_invitation', // notification type
                        $customerId,
                        $workspaceId
                    );
                }
            }
        }

                        // Send notification to customer about meeting creation
                $this->save_notifications(
                    "New Meeting Created",
                    "A new meeting has been created: {$validatedData['agenda']} on " . date('d/m/Y', strtotime($validatedData['start_date'])) . " at " . date('h:i A', strtotime($validatedData['start_time'])),
                    config('constants.employee_types.customer'),
                    auth()->user()->id, // sender_id (meeting creator)
                    config('constants.employee_types.customer'),
                    $customerId, // receiver_id (customer)
                    'meeting_created', // notification type
                    $customerId,
                    $workspaceId
                );

        $this->storeMeetingHistory($meeting->id, $history_description);
        return $this->success($meeting, 'Meeting created successfully.');
    }

    public function show(Request $request)
    {
        // Fetch the authenticated user's type
        $userTable = $this->getUserTable();
        $meeting = Meeting::where('id', $request->id)
            ->where('is_deleted', 0)
            ->with([
                'meeting_users',
                'meeting_organisers',
                'meeting_site',
                'meeting_project',
                'meeting_documents',
                'meeting_note_documents',
                'meeting_histories',
                'meeting_guest_users',
                'meeting_note_documents_pdf'
            ])
            ->first();

        // Check if the meeting exists
        if (!$meeting) {
            return $this->message('Meeting not exists.', 404);
        }
        // Apply access restrictions based on user type
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('You do not have access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this meeting.', 403);
        }
        if ($meeting) {
            if ($userTable === 'emp') {
                $meeting->load('created_by');
            } elseif ($userTable === 'customer') {
                $customerUser = User::where('id', $meeting->created_by)
                    ->select('id', 'name', 'email', 'mobile_number')
                    ->first();
                $meeting->created_by = $customerUser;
            }
        }
        // Handle meeting frequency and days
        if (isset($meeting->frequency) && $meeting->frequency == config('constants.meeting_frequency.weekly.key')) {
            $daysOfWeek = explode(',', $meeting->days);
            $result = getDaysBetween($meeting->start_date, $meeting->end_date, $daysOfWeek);
            $meeting->week_days = $result;
        }
        
        // Copy day_of_the_week from meeting_note_documents to meeting_note_documents_pdf
        if ($meeting->meeting_note_documents && $meeting->meeting_note_documents_pdf) {
            foreach ($meeting->meeting_note_documents_pdf as $pdfDocument) {
                // Find corresponding note document based on meeting_note_id
                $noteDocument = $meeting->meeting_note_documents->where('id', $pdfDocument->meeting_note_id)->first();
                if ($noteDocument && isset($noteDocument->day_of_the_week)) {
                    $pdfDocument->day_of_the_week = $noteDocument->day_of_the_week;
                }
            }
        }
        
        // Map meeting_users to include both regular and subcontractor employees with consistent structure
        if ($meeting->meeting_users) {
            // Convert to plain array to prevent Laravel from serializing original relationships
            $mappedMeetingUsers = $meeting->meeting_users->map(function($meetingUser) {
                // Get employee data based on user_type
                $employeeData = null;
                
                if ($meetingUser->user_type == 1) {
                    // Subcontractor employee - map from meeting_subcontractor_employees to meeting_employees
                    if ($meetingUser->meeting_subcontractor_employees) {
                        $subcontractorEmp = $meetingUser->meeting_subcontractor_employees;
                        $employeeData = [
                            'id' => $subcontractorEmp->id ?? null,
                            'emp_id' => (string)($subcontractorEmp->id ?? $meetingUser->employee_id),
                            'first_name' => $subcontractorEmp->first_name ?? null,
                            'middle_name' => $subcontractorEmp->middle_name ?? null,
                            'last_name' => $subcontractorEmp->last_name ?? null,
                            'image' => $subcontractorEmp->profile_image ?? null,
                            'email' => $subcontractorEmp->email ?? null,
                            'employee_type' => 'subcontractor_employee',
                        ];
                    }
                } else {
                    // Regular employee (user_type = 0 or null)
                    if ($meetingUser->meeting_employees) {
                        $regularEmp = $meetingUser->meeting_employees;
                        $employeeData = [
                            'id' => $regularEmp->id ?? null,
                            'emp_id' => (string)($regularEmp->emp_id ?? $meetingUser->employee_id),
                            'first_name' => $regularEmp->first_name ?? null,
                            'middle_name' => $regularEmp->middle_name ?? null,
                            'last_name' => $regularEmp->last_name ?? null,
                            'image' => $regularEmp->image ?? null,
                            'employee_type' => 'regular',
                        ];
                    }
                }
                
                return [
                    'id' => $meetingUser->id,
                    'meeting_id' => $meetingUser->meeting_id,
                    'site_id' => $meetingUser->site_id,
                    'employee_id' => $meetingUser->employee_id,
                    'user_type' => $meetingUser->user_type ?? 0,
                    'created_at' => $meetingUser->created_at,
                    'updated_at' => $meetingUser->updated_at,
                    'meeting_employees' => $employeeData,
                    'meeting_attendances' => $meetingUser->meeting_attendances ? $meetingUser->meeting_attendances->toArray() : [],
                ];
            })->values()->toArray();
            
            // Unset the relation and set as plain array attribute to prevent Laravel from serializing original relationships
            $meeting->unsetRelation('meeting_users');
            $meeting->setAttribute('meeting_users', $mappedMeetingUsers);
        }
        
        // Map meeting_note_signatures to map subcontractor_employee to employee
        if ($meeting->meeting_note_documents) {
            $mappedNoteDocuments = $meeting->meeting_note_documents->map(function($noteDocument) {
                if ($noteDocument->meeting_note_signatures) {
                    $mappedSignatures = $noteDocument->meeting_note_signatures->map(function($signature) {
                        // If subcontractor_employee exists, map it to employee
                        $employeeData = null;
                        $userType = 0; // Default to regular employee
                        
                        if ($signature->subcontractor_employee) {
                            // Map subcontractor employee to employee
                            $userType = 1; // Subcontractor employee
                            $subcontractorEmp = $signature->subcontractor_employee;
                            $employeeData = [
                                'id' => $subcontractorEmp->id ?? null,
                                'emp_id' => (string)($subcontractorEmp->id ?? $signature->employee_id),
                                'first_name' => $subcontractorEmp->first_name ?? null,
                                'middle_name' => $subcontractorEmp->middle_name ?? null,
                                'last_name' => $subcontractorEmp->last_name ?? null,
                                'image' => $subcontractorEmp->profile_image ?? null,
                                'email' => $subcontractorEmp->email ?? null,
                            ];
                        } elseif ($signature->employee) {
                            // Regular employee - use as is
                            $userType = 0; // Regular employee
                            $regularEmp = $signature->employee;
                            $employeeData = [
                                'id' => $regularEmp->id ?? null,
                                'emp_id' => (string)($regularEmp->emp_id ?? $signature->employee_id),
                                'first_name' => $regularEmp->first_name ?? null,
                                'middle_name' => $regularEmp->middle_name ?? null,
                                'last_name' => $regularEmp->last_name ?? null,
                                'image' => $regularEmp->image ?? null,
                            ];
                        }
                        
                        // Build the mapped signature as plain array - only include employee, not subcontractor_employee
                        return [
                            'id' => $signature->id,
                            'meeting_id' => $signature->meeting_id,
                            'employee_id' => $signature->employee_id,
                            'employee_note_id' => $signature->employee_note_id,
                            'signature' => $signature->signature,
                            'date' => $signature->date,
                            'del' => $signature->del,
                            'created_at' => $signature->created_at,
                            'updated_at' => $signature->updated_at,
                            'user_type' => $userType,
                            'employee' => $employeeData,
                        ];
                    })->values()->toArray();
                    
                    // Unset the relation and set as plain array attribute to prevent Laravel from serializing original relationships
                    $noteDocument->unsetRelation('meeting_note_signatures');
                    $noteDocument->setAttribute('meeting_note_signatures', $mappedSignatures);
                }
                
                return $noteDocument;
            });
            
            // Unset the relation and set as mapped collection
            $meeting->unsetRelation('meeting_note_documents');
            $meeting->setAttribute('meeting_note_documents', $mappedNoteDocuments);
        }
        
        return $this->success($meeting);
    }



    public function updateMeeting(Request $request)
    {
        $userTable = $this->getUserTable();
        // Initial Validation
        $validator = Validator::make(
            $request->all(),
            [
                'agenda' => 'required|string',
                'organiser_ids' => 'required|array',
                'is_recurring' => 'required|in:0,1',
                'start_date' => 'required|date',
                'end_date' => 'date|after_or_equal:start_date',
                'start_time' => 'required',
                'end_time' => 'required|after:start_time',
                'description' => 'required|string',
                'employee_ids' => 'nullable|array',
                'employee_ids.*' => 'nullable|exists:emp_company_details,id,del,0',
                'subcontractor_employee_ids' => 'nullable|array',
                'subcontractor_employee_ids.*' => 'nullable|exists:employees_subcontractors,id',
                'site_id' => 'required',
                'project_id' => 'required',
                'allow_meeting_members' => 'required|integer',
                'is_before_meeting' => 'integer',
            ],
            [
                'agenda.required' => "Please add Meeting Title.",
                'organiser_ids.required' => "Please select an Organiser.",
                'site_id.required' => "Please select a site.",
                'project_id.required' => "Please select a project.",
            ]
        );

        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }

        // Additional Validation for Recurring Meetings
        if ($request->is_recurring == config('constants.meeting_recurring.recurring.key')) {
            $validator = Validator::make(
                $request->all(),
                [
                    'frequency' => 'required|in:1,2,3,4',
                    'end_date' => 'required',
                ],
                [
                    'frequency.required' => 'Please select anyone from Daily/Weekly/Monthly/Custom.',
                ]
            );

            if ($request->frequency == config('constants.meeting_frequency.weekly.key')) {
                $validator = Validator::make(
                    $request->all(),
                    [
                        'days' => 'required',
                    ],
                    [
                        'days.required' => 'Please select a day for meeting.',
                    ]
                );
            }
        }

        // Validate Specific Meeting Members
        if ($request->allow_meeting_members == config('constants.allow_meeting_members.specific.key')) {
            $validator = Validator::make(
                $request->all(),
                [
                    'employee_ids' => 'required_without:subcontractor_employee_ids',
                    'subcontractor_employee_ids' => 'required_without:employee_ids',
                ],
                [
                    'employee_ids.required_without' => 'Please select at least one employee or subcontractor employee for the meeting.',
                    'subcontractor_employee_ids.required_without' => 'Please select at least one employee or subcontractor employee for the meeting.',
                ]
            );
        }

        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }

        $meeting = Meeting::find($request->id);
        if (!$meeting) {
            return $this->message('Meeting not found', 404);
        }
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('You do not have access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this meeting.', 403);
        }
        $customerId = $userTable == "customer" ? auth()->id() : auth()->user()->customer_id;
        $workspaceId = $userTable == "customer" ? auth()->user()->current_workspace_id : auth()->user()->workspace_id;
        $end_date = $request->is_recurring == 1 ? $request->end_date : $request->start_date;

        // Update Meeting Details
        $meeting->update([
            'created_by' => auth()->id(),
            'customer_id' => $customerId,
            'workspace_id' => $workspaceId,
            'site_id' => $request->site_id,
            'project_id' => $request->project_id,
            'agenda' => $request->agenda,
            'description' => $request->description,
            'status' => config('constants.meeting_status.pending.key'),
            'start_date' => $request->start_date,
            'end_date' => $end_date,
            'start_time' => $request->start_time,
            'end_time' => $request->end_time,
            'is_recurring' => $request->is_recurring,
            'allow_meeting_members' => $request->allow_meeting_members,
            'frequency' => $request->frequency ?? null,
            'days' => $request->frequency == config('constants.meeting_frequency.weekly.key') ? $request->days : null,
            'allow_guest' => $request->allow_guest ?? null,
            'is_before_meeting' => $request->is_before_meeting ?? 0,
        ]);

        // Remove Existing Organisers and Users
        MeetingOrganiser::where('meeting_id', $request->id)->delete();
        MeetingUser::where('meeting_id', $request->id)->delete();

        // Add New Organisers
        foreach ($request->organiser_ids as $organiser_id) {
            MeetingOrganiser::create([
                'meeting_id' => $meeting->id,
                'site_id' => $meeting->site_id,
                'organiser_id' => $organiser_id,
            ]);
        }
        // Add Meeting Members
        if ($request->allow_meeting_members == config('constants.allow_meeting_members.specific.key')) {
            // Store regular employees (user_type = 0)
            if (!empty($request->employee_ids)) {
                foreach ($request->employee_ids as $employee_id) {
                    MeetingUser::create([
                        'meeting_id' => $meeting->id,
                        'site_id' => $meeting->site_id,
                        'employee_id' => $employee_id,
                        'user_type' => 0, // Regular employee
                    ]);
                }
            }
            
            // Store subcontractor employees (user_type = 1)
            if (!empty($request->subcontractor_employee_ids)) {
                foreach ($request->subcontractor_employee_ids as $subcontractor_employee_id) {
                    MeetingUser::create([
                        'meeting_id' => $meeting->id,
                        'site_id' => $meeting->site_id,
                        'employee_id' => $subcontractor_employee_id,
                        'user_type' => 1, // Subcontractor employee
                    ]);
                }
            }
        }
        if ($userTable === 'emp') {
            $added_by = EmpPersonalDetails::where('emp_id', auth()->id())->first();
            $added_by_name = $added_by->first_name . ' ' . $added_by->middle_name . ' ' . $added_by->last_name;
        }
        if ($userTable === 'customer') {
            $added_by = User::where('id', auth()->id())->first();
            $added_by_name = $added_by->name;
        }
        // Send Notifications and Store History
        $history_description = sprintf(
            "%s updated this meeting at %s",
            strip_tags($added_by_name),
            now()->format('d/m/Y H:i:s'),
            url('/'),
            auth()->user()->id
        );

        // Prepare notification message
        $message = "{$added_by_name} updated {$meeting->agenda} meeting ";
        

        $user_ids = array_merge($request->organiser_ids, $request->employee_ids ?? []);
        
        // Get customer_id and workspace_id based on user type
        $customer_id = null;
        $workspace_id = null;
        if ($userTable === 'customer') {
            $customer_id = auth()->id();
            $workspace_id = auth()->user()->current_workspace_id;
        } elseif ($userTable === 'emp') {
            $customer_id = auth()->user()->customer_id;
            $workspace_id = auth()->user()->workspace_id;
        }
        
//send notification to customer
        $this->save_notifications(
            "Meeting Updated",
             " {$meeting->agenda} meeting has been updated.",
              config('constants.employee_types.customer'),
              auth()->id(),
              config('constants.employee_types.customer'),
              auth()->id(),
                'update_meeting',
                 $customer_id,
                  $workspace_id);

//send notification to employee
        foreach ($user_ids as $user_id) {
            $this->save_notifications(
                "Meeting Updated",
                 $message,  
                 
                  config('constants.employee_types.customer'),
                  auth()->id(),
                  config('constants.employee_types.employee'),
                   $user_id,
                    'update_meeting',
                     $customer_id,
                      $workspace_id);
        }
        $this->storeMeetingHistory($meeting->id, $history_description);
        return $this->success($meeting, 'Meeting updated successfully.');
    }

    public function destroy($id)
    {
        try {
            $meeting = Meeting::find($id);

            if (!$meeting) {
                return $this->message('Meeting not found.', 404);
            }
            if ($meeting->is_deleted == 1) {
                return $this->message('Meeting has been deleted already.', 200);
            }
            $userTable = $this->getUserTable();
            if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('You do not have access to this meeting.', 403);
            }
            if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('You do not have access to this meeting.', 403);
            }
            if ($userTable !== 'customer' && $userTable !== 'emp') {
                return $this->message('Unauthorized access.', 403);
            }
            // if ($meeting->start_date < $this->current_date) {
            //     return $this->message('You cannot delete this Meeting. Meeting start date must be in the future.', 400);
            // }
            // Delete meeting-related records
            MeetingUser::where('meeting_id', $id)->delete();
            MeetingOrganiser::where('meeting_id', $id)->delete();
            // Delete associated documents
            $documents = MeetingDocument::where('meeting_id', $id)->select('document_path')->get();
            foreach ($documents as $document) {
                if (file_exists($document->document_path)) {
                    unlink($document->document_path);
                }
            }
            MeetingDocument::where('meeting_id', $id)->delete();
            $meeting->is_deleted = 1;
            $meeting->save();
            // Maintain history based on user type
            if ($userTable === 'emp') {
                $added_by = EmpPersonalDetails::where('emp_id', auth()->id())->first();
                $added_by_name = $added_by->first_name . ' ' . $added_by->middle_name . ' ' . $added_by->last_name;
            }
            if ($userTable === 'customer') {
                $added_by = User::where('id', auth()->id())->first();
                $added_by_name = $added_by->name;
            }
            // Store history
            $history_description = sprintf(
                "%s deleted this meeting at %s",
                strip_tags($added_by_name), // Strips any HTML tags from the name
                now()->format('d/m/Y H:i:s'),
                url('/'),
                auth()->id()
            );

            $this->storeMeetingHistory($id, $history_description);

            return $this->message('Meeting Deleted Successfully.', 200);
        } catch (\Exception $e) {
            return $this->message('Failed to delete meeting. Error: ' . $e->getMessage(), 500);
        }
    }

    function searchEmployees(Request $request)
    {
        $search = strtolower($request->search);
        $data = EmpPersonalDetails::where('first_name', 'LIKE', "%$search%")
            ->orWhere('middle_name', 'LIKE', "%$search%")
            ->orWhere('last_name', 'LIKE', "%$search%")
            ->get();
        if (!$data) {
            return $this->message('Record not found', 404);
        }
        return $this->success($data, 'Employees Search Successfully');
    }

    public function storeMeetingDocument(Request $request)
    {
        if ($request->select_meeting_date == 1) {
            $validator = Validator::make(
                $request->all(),
                [
                    'meeting_date' => 'required|date',
                ],
                [
                    'meeting_date.required' => 'Date field is required.',
                ]
            );
            if ($validator->fails()) {
                return $this->message($validator->errors()->first(), 422);
            }
        }
        $validator = Validator::make($request->all(), [
            'title' => 'required|string',
            'meeting_document' => 'required|mimes:jpeg,png,jpg,pdf|max:10240',
            'meeting_id' => 'required|exists:meetings,id',
        ], [
            'meeting_document.required' => 'A meeting document is required.',
            'meeting_document.mimes' => 'The meeting document must be a file of type: jpeg, png, jpg, pdf.',
            'meeting_document.max' => 'The meeting document may not be greater than 10MB.',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $meeting = Meeting::find($request->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        // Add customer and employee check
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable !== 'customer' && $userTable !== 'emp') {
            return $this->message('Unauthorized access.', 403);
        }
        // Handle meeting date with proper format conversion
        if ($meeting->is_recurring) {
            $meeting_date = $request->meeting_date ?? null;
        } else {
            // Convert meeting start_date from d-m-Y to Y-m-d format for database
            if ($meeting->start_date) {
                try {
                    $meeting_date = \Carbon\Carbon::createFromFormat('d-m-Y', $meeting->start_date)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $meeting_date = \Carbon\Carbon::parse($meeting->start_date)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $meeting_date = now()->format('Y-m-d');
                    }
                }
            } else {
                $meeting_date = null;
            }
        }
        $meeting_document = new MeetingDocument();
        if ($request->hasFile('meeting_document')) {
            $uploadResult = $this->handleFileImageUpload($request, 'MeetingDocuments');
            $meeting_document->document_path = $uploadResult['path'];
        }
        $meeting_document->meeting_id = $request->meeting_id;
        $meeting_document->title = $request->title;
        $meeting_document->meeting_date = $meeting_date;
        $meeting_document->uploaded_by = Auth::user()->id;
        $meeting_document->is_signed = $request->is_signed ?? 0;
        $meeting_document->save();
        // Maintain history based on user type
        if ($userTable === 'emp') {
            $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)->select('first_name', 'middle_name', 'last_name')->first();
            $fullName = trim($added_by->first_name . " " . $added_by->middle_name . " " . $added_by->last_name);
        } else {
            $added_by = User::where('id', Auth::user()->id)->select('name')->first();
            $fullName = $added_by->name;
        }
        $history_description = sprintf(
            "%s added a document to this meeting on %s",
            strip_tags($fullName), // Strips any HTML tags from the full name
            now()->format('d/m/Y H:i:s'),
            url('/'),
            Auth::user()->id
        );

        $this->storeMeetingHistory($request->meeting_id, $history_description);
        return $this->success($meeting_document, 'Meeting document uploaded successfully.');
    }

    public function deleteMeetingDocument(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'meeting_document_id' => 'required|exists:meeting_documents,id',
            ],
            [
                'meeting_document_id.required' => 'Meeting document ID is required.',
                'meeting_document_id.exists' => 'Meeting document not found.',
            ]
        );
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $meeting_document = MeetingDocument::find($request->meeting_document_id);
        if (!$meeting_document) {
            return $this->message('Meeting document not found.', 404);
        }
        $meeting = Meeting::find($meeting_document->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        // Add customer and employee check
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('You do not have access to this meeting document.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this meeting document.', 403);
        }

        if ($userTable !== 'customer' && $userTable !== 'emp') {
            return $this->message('Unauthorized access.', 403);
        }

        // Delete the meeting document and its file
        if (file_exists($meeting_document->document_path)) {
            unlink($meeting_document->document_path);
        }
        $meeting_document->delete();
        if ($userTable === 'emp') {
            $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)->select('first_name', 'middle_name', 'last_name')->first();
            $fullName = trim($added_by->first_name . " " . $added_by->middle_name . " " . $added_by->last_name);
        } else {
            $added_by = User::where('id', Auth::user()->id)->select('name')->first();
            $fullName = $added_by->name;
        }
        $history_description = sprintf(
            "%s deleted a document from this meeting on %s",
            strip_tags($fullName), // Strips any HTML tags from the full name
            now()->format('d/m/Y H:i:s'),
            url('/'),
            Auth::user()->id
        );

        $this->storeMeetingHistory($meeting->id, $history_description);
        return $this->message('Meeting document deleted successfully.', 200);
    }

    function storeMeetingHistory($meeting_id, $description)
    {
        MeetingHistory::create([
            'meeting_id' => $meeting_id,
            'updated_by' => Auth::user()->id,
            'description' => $description
        ]);
    }
    // ================== Meeting APIs ===================
    public function getEmployeeMeetings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|integer',
            'site_id' => 'required|integer'
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $validatedData = $validator->validated();
        $userTable = $this->getUserTable();
        // Access control for customer and emp
        if ($userTable === "customer") {
            $isValid = User::where('id', $validatedData['employee_id'])
                ->where('id', auth()->id())
                ->where('current_workspace_id', auth()->user()->current_workspace_id)
                ->exists();
            if (!$isValid) {
                return $this->message('Unauthorized access.', 403);
            }
        }
        if ($userTable === "emp") {
            $isValid = EmpCompanyDetails::where('id', $validatedData['employee_id'])
                ->where('customer_id', auth()->user()->customer_id)
                ->where('workspace_id', auth()->user()->workspace_id)
                ->exists();

            if (!$isValid) {
                return $this->message('Unauthorized access.', 403);
            }
        }
        if ($userTable !== 'customer' && $userTable !== 'emp') {
            return $this->message('Unauthorized access.', 403);
        }
        // Fetch meetings for the employee and site
        $data = $this->getAllEmployeeMeetings($validatedData['employee_id'], $validatedData['site_id']);
        if (!count($data) > 0) {
            return $this->success($data, 'There is no meeting for today.');
        }

        return $this->success($data, 'Employee meetings fetched successfully.');
    }

    function getAllEmployeeMeetings($employee_id, $site_id, $is_before_meeting = "")
    {
        $currentTime = now()->format('H:i:s');
        $dayNumber = Carbon::parse($this->current_date)->format('N');

        $data = [];
        $allowed_clock_in = 1;
        $meetings = Meeting::with('today_meeting_documents','today_meeting_document_notes','meeting_users')
            ->where('meetings.site_id', $site_id)
            ->where('meetings.is_deleted', 0)
            ->where('meetings.start_date', "<=", $this->current_date)
            ->where('meetings.end_date', ">=", $this->current_date)
            ->where('meetings.start_time', "<=", $currentTime)
            ->where('meetings.end_time', ">=", $currentTime)
            ->where('meetings.is_before_meeting', $is_before_meeting)
            ->select('meetings.*')
            ->get();
        foreach ($meetings as $key => $meeting) {
            $meeting_users =  $meeting->meeting_users->toArray();
            unset($meetings[$key]['meeting_users']);
            
            // Check if this meeting is for specific members only
            if ($meeting->allow_meeting_members == 1) {
                // This meeting is for specific members only
                $employee_ids = array_column($meeting_users, 'employee_id');
                
                // Check if the current employee is in the meeting members list
                if (!in_array($employee_id, $employee_ids)) {
                    // Employee is not in the specific members list, skip this meeting
                    continue;
                }
                
                // Get meeting employees for specific members
                $meetings[$key]['meeting_employees'] = DB::table('emp_company_details')
                    ->join('emp_personal_details', 'emp_company_details.id', '=', 'emp_personal_details.emp_id')
                    ->where('emp_company_details.status', 1)
                    ->where('emp_company_details.approved_by', 1)
                    ->where('emp_company_details.del', '!=', 1)
                    ->where(function ($query) {
                        $query->where('emp_company_details.invited', 1)
                            ->orWhere(function ($q) {
                                $q->where('emp_company_details.invited', 0)
                                    ->where('emp_company_details.approved', 1)
                                    ->where('emp_company_details.compeleted', 1)
                                    ->where('emp_company_details.status', 1);
                            });
                    })
                    ->whereIn('emp_company_details.id', $employee_ids)
                    ->select([
                        'emp_personal_details.id',
                        'emp_personal_details.emp_id',
                        'emp_personal_details.first_name',
                        'emp_personal_details.middle_name',
                        'emp_personal_details.last_name',
                        'emp_personal_details.image'
                    ])
                    ->get();
            } else {
                $meetings[$key]['meeting_employees'] = DB::table('emp_company_details')
                    ->join('emp_personal_details', 'emp_company_details.id', '=', 'emp_personal_details.emp_id')
                    ->where('emp_company_details.status', 1)
                    ->where('emp_company_details.approved_by', 1)
                    ->where('emp_company_details.del', '!=', 1)
                    ->where(function ($query) {
                        $query->where('emp_company_details.invited', 1)
                            ->orWhere(function ($q) {
                                $q->where('emp_company_details.invited', 0)
                                    ->where('emp_company_details.approved', 1)
                                    ->where('emp_company_details.compeleted', 1)
                                    ->where('emp_company_details.status', 1);
                            });
                    })
                    ->select([
                        'emp_personal_details.id',
                        'emp_personal_details.emp_id',
                        'emp_personal_details.first_name',
                        'emp_personal_details.middle_name',
                        'emp_personal_details.last_name',
                        'emp_personal_details.image'
                    ])
                    ->get();
            }
        
            // Loop through the meeting documents and check if any document is signed
            foreach ($meeting->today_meeting_documents as $doc_key => $value) {
                // Check if employee has signed this document today
                $is_document_signed_today = EmployeeSignedDocument::where('meeting_id', $value->meeting_id)
                    ->where('employee_id', $employee_id)
                    ->where('document_id', $value->id)
                    ->where('signed', 1)
                    ->where('date', $this->current_date)
                    ->exists();
                
                // Set is_signed based on signature status for today
                // is_signed = 1 means NOT signed (employee needs to sign)
                // is_signed = 0 means signed (employee has already signed)
                if($is_document_signed_today){
                    $meetings[$key]['today_meeting_documents'][$doc_key]['is_signed'] = 0;
                } else {
                    // $meetings[$key]['today_meeting_documents'][$doc_key]['is_signed'] = 1;
                }
        
                // Check if document requires signature
                if ($value->is_signed) {
                    // For daily documents, check if signed today
                    // For specific date documents, check if signed on that date
                    if ($value->is_daily == 1) {
                        // Daily document - must be signed every day
                        if (!$is_document_signed_today) {
                            $allowed_clock_in = 0;
                        }
                    } else {
                        // Specific date document - check if signed on the specific date
                        $is_sign_document = EmployeeSignedDocument::where('meeting_id', $value->meeting_id)
                            ->where('employee_id', $employee_id)
                            ->where('document_id', $value->id)
                            ->where('signed', 1)
                            ->where('date', $value->meeting_date ?? $this->current_date)
                            ->exists();
        
                        if (!$is_sign_document) {
                            $allowed_clock_in = 0;
                        }
                    }
                }
            }
            foreach ($meeting->today_meeting_documents as $value) {
                if (!is_null($value->meeting_note_documents_pdf)) {
                    $meeting->today_meeting_documents[] = $value->meeting_note_documents_pdf;
                }
            }
            foreach ($meeting->today_meeting_document_notes as $doc_key => $value) {
                // Check if employee has signed this meeting document note today
                $is_note_signed_today = DB::table('meeting_note_signatures')
                    ->where('employee_note_id', $value->id)
                    ->where('employee_id', $employee_id)
                    ->whereNotNull('signature')
                    ->where('date', $this->current_date)
                    ->exists();
                
                // Set is_signed based on signature status for today
                // is_signed = 1 means NOT signed (employee needs to sign)
                // is_signed = 0 means signed (employee has already signed)
                $is_signed_value = $is_note_signed_today ? 0 : 1;
                $meetings[$key]['today_meeting_document_notes'][$doc_key]['is_signed'] = $is_signed_value;
                
                // Also set is_signed for the related PDF document if it exists
                if (isset($meetings[$key]['today_meeting_document_notes'][$doc_key]['meeting_note_documents_pdf'])) {
                    $meetings[$key]['today_meeting_document_notes'][$doc_key]['meeting_note_documents_pdf']['is_signed'] = $is_signed_value;
                }
                
                // If document note is not signed today, don't allow clock in
                if (!$is_note_signed_today) {
                    $allowed_clock_in = 0;
                }
            }
            // Process meeting document notes PDFs
            $meeting_document_notes_with_pdfs = [];
            foreach ($meeting->today_meeting_document_notes as $value) {
                if (!is_null($value->meeting_note_documents_pdf)) {
                    // Check signature status for this document note today
                    $is_note_signed_today = DB::table('meeting_note_signatures')
                        ->where('employee_note_id', $value->id)
                        ->where('employee_id', $employee_id)
                        ->whereNotNull('signature')
                        ->where('date', $this->current_date)
                        ->exists();
                    
                    $pdf_document = $value->meeting_note_documents_pdf->toArray();
                    $pdf_document['is_signed'] = $is_note_signed_today ? 0 : 1;
                    
                    // Transform the document structure: move meeting_note_id to id and remove meeting_note_id
                    if (isset($pdf_document['meeting_note_id'])) {
                        $pdf_document['id'] = $pdf_document['meeting_note_id'];
                        unset($pdf_document['meeting_note_id']);
                    }
                    
                    // Add base URL to document path
                    if (isset($pdf_document['document_path']) && !empty($pdf_document['document_path'])) {
                        $pdf_document['document_path'] = url($pdf_document['document_path']);
                    }
                    
                    $meeting_document_notes_with_pdfs[] = $pdf_document;
                }
            }
            
            // Add base URL to today_meeting_documents paths
            $today_meeting_documents = $meeting->today_meeting_documents->toArray();
            foreach ($today_meeting_documents as &$document) {
                if (isset($document['document_path']) && !empty($document['document_path'])) {
                    $document['document_path'] = url($document['document_path']);
                }
            }
            
            // Add the PDFs to the meeting data
            // $meetings[$key]['today_meeting_document_notes_pdfs'] = $meeting_document_notes_with_pdfs;
            $meetings[$key]['meeting_documents'] = array_merge($today_meeting_documents, $meeting_document_notes_with_pdfs);
            
            // Remove the original fields to avoid duplication
            unset($meetings[$key]['today_meeting_documents']);
            unset($meetings[$key]['today_meeting_document_notes']);
            
            $employee_attendance = MeetingAttendance::where('meeting_id', $meeting->id)->where('employee_id', $employee_id)
                ->where('date', $this->current_date)->exists();
        
            if ($employee_attendance == 1) {
                $meetings[$key]['is_joined'] = 1;
            } else {
                $meetings[$key]['is_joined'] = 0;
            }
            $meetings[$key]['allowed_clock_in'] = $allowed_clock_in;
        
            // Add meeting to data array (no need for complex recurring logic since we already filtered by date/time)
            $data[] = $meeting;
        }
            
        return $data;
    }

    public function addMeetingAttendance(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:emp_company_details,id',
            'meeting_id' => 'required|exists:meetings,id',
            'attendance_type' => 'required'
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $userTable = $this->getUserTable();
        $meeting = Meeting::find($request->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting..', 403);
        }
        if ($userTable === "emp") {
            $empDetails = EmpCompanyDetails::find($request->employee_id);
            if (!$empDetails || $empDetails->customer_id != auth()->user()->customer_id || $empDetails->workspace_id != auth()->user()->workspace_id) {
                return $this->message('Unauthorized access to this meeting..', 403);
            }
        }
        if ($request->attendance_type == "in") {
            $employee_attendance = MeetingAttendance::where('meeting_id', $request->meeting_id)
                ->where('employee_id', $request->employee_id)
                ->where('date', date('Y-m-d'))
                ->exists();

            if ($employee_attendance) {
                return $this->message('You have already punched in for the meeting.', 200);
            }
            $attendance = new MeetingAttendance();
            $attendance->meeting_id = $request->meeting_id;
            $attendance->employee_id = $request->employee_id;
            $attendance->date = date('Y-m-d');
            $attendance->check_in = now()->format('H:i:s');
            $attendance->save();
            
            $responseData = $attendance->toArray();
            
            
            if (isset($responseData['check_in']) && $responseData['check_in']) {
                $rawCheckIn = $attendance->getRawOriginal('check_in');
                
                $responseData['check_in'] = $rawCheckIn ? \Carbon\Carbon::parse($rawCheckIn)->format('H:i:s') : $responseData['check_in'];
            }
            
            return $this->success($responseData, 'You punch-in the meeting successfully.');
        }
        $attendance = MeetingAttendance::where('meeting_id', $request->meeting_id)
            ->where('employee_id', $request->employee_id)
            ->where('date', date('Y-m-d'))
            ->first();
        if (!$attendance) {
            return $this->message('Please punch in before punching out of the meeting.', 422);
        }
        
        $attendance->check_out = now()->format('H:i:s');
        $attendance->save();
        
        // Convert to array and manually format check_in and check_out to ensure seconds are included
        $responseData = $attendance->toArray();
        if (isset($responseData['check_in']) && $responseData['check_in']) {
            $responseData['check_in'] = \Carbon\Carbon::parse($responseData['check_in'])->format('H:i:s');
        }
        if (isset($responseData['check_out']) && $responseData['check_out']) {
            $responseData['check_out'] = \Carbon\Carbon::parse($responseData['check_out'])->format('H:i:s');
        }
        
        return $this->success($responseData, 'You punch-out the meeting successfully.');
    }

    public function signDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:emp_company_details,id',
            'document_id' => 'required',
            'meeting_id' => 'required|exists:meetings,id',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $userTable = $this->getUserTable();
        $meeting = Meeting::find($request->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp") {
            $empDetails = EmpCompanyDetails::find($request->employee_id);
            if (!$empDetails || $empDetails->customer_id != auth()->user()->customer_id || $empDetails->workspace_id != auth()->user()->workspace_id) {
                return $this->message('Unauthorized access to this meeting.', 403);
            }
        }
        $data = [];
        // 🌟 Check if it's a custom (note-based) document
        $isCustomNoteDocument = $request->filled('type_of_setting');
    
        if ($isCustomNoteDocument) {
            
            // ✅ Save in meeting_note_signatures
            $existing = DB::table('meeting_note_signatures')
                ->where('employee_note_id', $request->document_id)
                ->where('employee_id', $request->employee_id)
                ->first();
    
            $insertData = [
                'meeting_id' => $request->meeting_id,
                'employee_note_id' => $request->document_id,
                'employee_id' => $request->employee_id,
                'date' => $this->current_date,
                'updated_at' => now()
            ];
    
            if ($request->hasFile('document')) {
                $uploadResult = $this->handleFileImageUpload($request, 'MeetingDocuments');
                $insertData['signature'] = $uploadResult['path'];
            } else {
                return $this->message('Please upload the signed custom document.', 422);
            }
    
            if ($existing) {
                // Update existing record
                DB::table('meeting_note_signatures')
                    ->where('employee_note_id', $request->document_id)
                    ->where('employee_id', $request->employee_id)
                    ->update($insertData);
            } else {
                // Create new record
                $insertData['created_at'] = now();
                DB::table('meeting_note_signatures')->insert($insertData);
            }

            // Send notification to customer about custom document signature
            $this->sendDocumentSignatureNotification(
                $request->meeting_id,
                $request->document_id,
                $request->employee_id,
                $meeting->customer_id,
                $meeting->workspace_id,
                'custom_note'
            );

        } else {
            // ✅ Save in employee_signed_documents - Create or Update
            $meeting_document = MeetingDocument::where('id', $request->document_id)
                ->where('meeting_id', $request->meeting_id)
                ->first();
    
            if (!$meeting_document) {
                return $this->message('No record found for this document.', 404);
            }
    
            // Check if a signed document record already exists
            $signed_document = EmployeeSignedDocument::where('meeting_id', $request->meeting_id)
                ->where('employee_id', $request->employee_id)
                ->where('date', $this->current_date)
                ->where('document_id', $request->document_id)
                ->first();
    
            if ($signed_document) {
                // Update existing record
                $signed_document->signed = 1;
                $signed_document->date = $this->current_date;
                if ($request->hasFile('document')) {
                    $uploadResult = $this->handleFileImageUpload($request, 'MeetingDocuments');
                    $signed_document->document = $uploadResult['path'];
                } else {
                    return $this->message('Please upload the signed document.', 422);
                }
                $signed_document->save();
            } else {
                // Create new record
                $signed_document = new EmployeeSignedDocument();
                $signed_document->meeting_id = $request->meeting_id;
                $signed_document->employee_id = $request->employee_id;
                $signed_document->document_id = $request->document_id;
                $signed_document->signed = 1;
                $signed_document->date = $this->current_date;
    
                if ($request->hasFile('document')) {
                    $uploadResult = $this->handleFileImageUpload($request, 'MeetingDocuments');
                    $signed_document->document = $uploadResult['path'];
                } else {
                    return $this->message('Please upload the signed document.', 422);
                }
    
                $signed_document->save();
            }

            // Send notification to customer about meeting document signature
            $this->sendDocumentSignatureNotification(
                $request->meeting_id,
                $request->document_id,
                $request->employee_id,
                $meeting->customer_id,
                $meeting->workspace_id,
                'meeting_document'
            );

        }
        $remaining_documents = $this->unSignedDocuments($request->meeting_id, $request->employee_id);
        $data['documents'] = $remaining_documents;
        $data['allowed_clock_in'] = count($remaining_documents) === 0 ? 1 : 0;
        return $this->success($data, 'Document signed successfully.');
    }
    
    function unSignedDocuments($meeting_id, $employee_id)
    {
        $data = MeetingDocument::leftJoin('employee_signed_documents', function ($join) use ($employee_id) {
            $join->on('meeting_documents.id', '=', 'employee_signed_documents.document_id')
                ->where('employee_signed_documents.employee_id', '=', $employee_id);
        })
            ->whereNull('employee_signed_documents.document_id')
            ->where('employee_signed_documents.date', '=', $this->current_date)
            ->where('meeting_documents.meeting_id', $meeting_id)
            ->where('meeting_documents.is_signed', 1)
            ->select('meeting_documents.*')
            ->get();
        return $data;
    }

    /**
     * Send notification to customer when a document is signed
     */
    private function sendDocumentSignatureNotification($meetingId, $documentId, $employeeId, $customerId, $workspaceId, $documentType = 'meeting_document')
    {
        try {
            $employee = EmpPersonalDetails::where('emp_id', $employeeId)->first();
            if (!$employee) {
                Log::warning('Employee not found for document signature notification: ' . $employeeId);
                return;
            }

            $meeting = Meeting::find($meetingId);
            if (!$meeting) {
                Log::warning('Meeting not found for document signature notification: ' . $meetingId);
                return;
            }

            $employeeName = trim($employee->first_name . ' ' . $employee->middle_name . ' ' . $employee->last_name);
            $meetingAgenda = $meeting->agenda ?? 'Meeting';
            
            $notificationTitle = "Document Signed - Meeting";
            $notificationMessage = "Employee {$employeeName} has signed a document for meeting: {$meetingAgenda}";
            
            if ($documentType === 'custom_note') {
                $notificationTitle = "Custom Document Signed - Meeting";
                $notificationMessage = "Employee {$employeeName} has signed a custom document for meeting: {$meetingAgenda}";
            }

            $this->save_notifications(
                $notificationTitle,
                $notificationMessage,
                config('constants.employee_types.employee'),
                $employeeId, // sender_id (employee who signed)
                config('constants.employee_types.customer'),
                $customerId, // receiver_id (customer)
                config('constants.notification_types.signature_added'),
                $customerId,
                $workspaceId
            );

            Log::info("Document signature notification sent to customer {$customerId} for meeting ID {$meetingId}");
        } catch (\Exception $e) {
            Log::error("Failed to send document signature notification: " . $e->getMessage());
        }
    }

    function addMeetingGuestUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'meeting_id' => 'required|exists:meetings,id',
            'name' => 'required',
            'email' => 'required|email',
            'number' => 'required',
            'occupation' => 'required'
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $userTable = $this->getUserTable();
        $meeting = Meeting::find($request->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $guest_user = new MeetingGuestUser();
        $guest_user->meeting_id = $request->meeting_id;
        $guest_user->name = $request->name;
        $guest_user->email = $request->email;
        $guest_user->phone_number = $request->number;
        $guest_user->occupation = $request->occupation;
        $guest_user->details = $request->details ?? "";
        $guest_user->added_by = Auth::user()->id;
        $guest_user->meeting_date = $request->meeting_date;
        $guest_user->save();
        if (!$guest_user) {
            return $this->message('Guest User not added.', 422);
        }
        if ($userTable === "emp") {
            $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                ->select('first_name', 'middle_name', 'last_name')
                ->first();
        } else {
            $added_by = User::where('id', Auth::user()->id)
                ->select('name as first_name') // Assuming User has a single name field
                ->first();
        }
        $history_description = sprintf(
            "%s %s %s added a guest user %s in meeting at %s",
            strip_tags($added_by->first_name ?? ''),
            strip_tags($added_by->middle_name ?? ''),
            strip_tags($added_by->last_name ?? ''),
            strip_tags($guest_user->name),
            now()->format('d/m/Y H:i:s'),
            url('/'),
            Auth::user()->id
        );

        $this->storeMeetingHistory($guest_user->meeting_id, $history_description);
        return $this->success($guest_user, 'Guest User added Successfully.');
    }

    function updateMeetingGuestUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'guest_user_id' => 'required|exists:meeting_guest_users,id',
            'meeting_id' => 'required|exists:meetings,id',
            'name' => 'required',
            'email' => 'required|email',
            'number' => 'required',
            'occupation' => 'required'
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $userTable = $this->getUserTable();
        $meeting = Meeting::find($request->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $guest_user = MeetingGuestUser::find($request->guest_user_id);
        if (!$guest_user) {
            return $this->message('Guest User not found.', 404);
        }
        $guest_user->meeting_id = $request->meeting_id;
        $guest_user->name = $request->name;
        $guest_user->email = $request->email;
        $guest_user->phone_number = $request->number;
        $guest_user->occupation = $request->occupation;
        $guest_user->details = $request->details ?? "";
        $guest_user->added_by = Auth::user()->id;
        $guest_user->save();
        // Maintain history based on user type
        if ($userTable === "emp") {
            $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                ->select('first_name', 'middle_name', 'last_name')
                ->first();
            $added_by_name = "{$added_by->first_name} {$added_by->middle_name} {$added_by->last_name}";
        } else {
            $added_by = User::where('id', Auth::user()->id)
                ->select('name as first_name') // Assuming User has a single name field
                ->first();
            $added_by_name = $added_by->first_name;
        }
        $history_description = sprintf(
            "%s updated a guest user %s in the meeting at %s",
            strip_tags($added_by_name), // Strips any HTML tags from the added_by_name
            strip_tags($guest_user->name), // Strips any HTML tags from the guest user's name
            now()->format('d/m/Y H:i:s'),
            url('/'),
            Auth::user()->id
        );

        $this->storeMeetingHistory($guest_user->meeting_id, $history_description);
        return $this->success($guest_user, 'Guest User updated successfully.');
    }

    function deleteMeetingGuestUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'guest_user_id' => 'required|exists:meeting_guest_users,id',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $userTable = $this->getUserTable();
        $guest = MeetingGuestUser::find($request->guest_user_id);
        if (!$guest) {
            return $this->message('Guest User not found.', 404);
        }
        $meeting = Meeting::find($guest->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $guest->delete();
        // Maintain history based on user type
        if ($userTable === "emp") {
            $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                ->select('first_name', 'middle_name', 'last_name')
                ->first();
            $added_by_name = "{$added_by->first_name} {$added_by->middle_name} {$added_by->last_name}";
        } else {
            $added_by = User::where('id', Auth::user()->id)
                ->select('name as first_name') // Assuming User has a single name field
                ->first();
            $added_by_name = $added_by->first_name;
        }
        $history_description = sprintf(
            "%s deleted a guest user %s from this meeting at %s",
            strip_tags($added_by_name), // Strips any HTML tags from the added_by_name
            strip_tags($guest->name), // Strips any HTML tags from the guest's name
            now()->format('d/m/Y H:i:s'),
            url('/'),
            Auth::user()->id
        );

        $this->storeMeetingHistory($guest->meeting_id, $history_description);
        return $this->message('Guest User deleted successfully.');
    }

    public function meetingPortal(Request $request)
    {
        // Normalize empty strings to null for validation
        $requestData = $request->all();
        if (isset($requestData['meeting_id']) && $requestData['meeting_id'] === '') {
            $requestData['meeting_id'] = null;
            $request->merge(['meeting_id' => null]);
        }
        if (isset($requestData['site_id']) && $requestData['site_id'] === '') {
            $requestData['site_id'] = null;
            $request->merge(['site_id' => null]);
        }
        
        $validator = Validator::make($requestData, [
            'from' => 'nullable|date',
            'to' => 'nullable|date|after_or_equal:from',
            'meeting_id' => 'nullable|exists:meetings,id',
            'site_id' => 'nullable|exists:sites,id',
        ], [
            'from.date' => 'The From filter must be a date.',
            'to.date' => 'The To filter must be a date.',
            'to.after_or_equal' => 'The To date filter must be equal to or greater than the From date filter.',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $userTable = $this->getUserTable();
        
        // Get customer and workspace IDs based on user type
        $customerId = $userTable === "customer" ? auth()->id() : auth()->user()->customer_id;
        $workspaceId = $userTable === "customer" ? auth()->user()->current_workspace_id : auth()->user()->workspace_id;
        
        $start_date = $request->from ?? Carbon::now()->subDays(7)->format('Y-m-d');
        $end_date = $request->to ?? $this->current_date;
        $limit = $request->limit ?? 10;
        $page = $request->page ?? 1;
        $skip = ($page - 1) * $limit;

        $query = MeetingAttendance::join('meetings', 'meeting_attendances.meeting_id', '=', 'meetings.id')
            ->join('emp_personal_details', 'meeting_attendances.employee_id', '=', 'emp_personal_details.emp_id')
            ->join('sites', 'meetings.site_id', '=', 'sites.id')
            ->where('meetings.is_deleted', 0)
            ->where(function ($query) use ($start_date, $end_date) {
                $query->whereBetween('meetings.start_date', [$start_date, $end_date])
                    ->orWhereBetween('meetings.end_date', [$start_date, $end_date])
                    ->orWhere(function ($query) use ($start_date, $end_date) {
                        $query->where('meetings.start_date', '<=', $start_date)
                            ->where('meetings.end_date', '>=', $end_date);
                    });
            });

        if ($userTable === "customer") {
            $query->where('meetings.customer_id', $customerId)
                ->where('meetings.workspace_id', $workspaceId);
        }
        if ($userTable === "emp") {
            $query->where('meetings.customer_id', $customerId)
                ->where('meetings.workspace_id', $workspaceId);
        }
        if ($userTable !== "customer" && $userTable !== "emp") {
            return $this->message('Unauthorized access.', 403);
        }
        
        // Handle empty strings as null (already normalized in validation, but ensure safety)
        $meetingId = !empty($request->meeting_id) ? $request->meeting_id : null;
        $siteId = !empty($request->site_id) ? $request->site_id : null;
        
        if ($meetingId) {
            $query->where('meetings.id', $meetingId);
        }
        if ($siteId) {
            $query->where('meetings.site_id', $siteId);
        }
        
        // Clone query before executing for meeting IDs
        $meetingIdsQuery = clone $query;
        $meetingIds = $meetingIdsQuery->select('meetings.id')->distinct()->pluck('meetings.id')->toArray();
        
        // Clone query for count to keep original query fresh for results
        $countQuery = clone $query;
        $total = $countQuery->count();

        // Get total meetings count with date, meeting_id, and site_id filters
        $totalMeetingsQuery = Meeting::where('is_deleted', 0)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where(function (
                $query
            ) use (
                $start_date,
                $end_date
            ) {
                $query->whereBetween('start_date', [$start_date, $end_date])
                    ->orWhereBetween('end_date', [$start_date, $end_date])
                    ->orWhere(function ($query) use ($start_date, $end_date) {
                        $query->where('start_date', '<=', $start_date)
                            ->where('end_date', '>=', $end_date);
                    });
            });
        if ($meetingId) {
            $totalMeetingsQuery->where('id', $meetingId);
        }
        if ($siteId) {
            $totalMeetingsQuery->where('site_id', $siteId);
        }
        $total_meetings = $totalMeetingsQuery->count();

        // Count total joined employees and guests
        $totalJoinedEmployees = 0;
        $totalGuestEmployees = 0;

        if (!empty($meetingIds)) {
            // Count total joined employees (distinct employees who attended)
            $employeeCountResult = MeetingAttendance::whereIn('meeting_id', $meetingIds)
                ->select(DB::raw('COUNT(DISTINCT employee_id) as count'))
                ->first();
            $totalJoinedEmployees = $employeeCountResult ? $employeeCountResult->count : 0;

            // Count total guests
            $totalGuestEmployees = MeetingGuestUser::whereIn('meeting_id', $meetingIds)
                ->count();
        }

        $meeting_users = $query->skip($skip)->take($limit)
            ->orderBy('meeting_attendances.id', 'desc')
            ->select(
                'emp_personal_details.first_name',
                'emp_personal_details.last_name',
                'emp_personal_details.image',
                'meetings.agenda',
                'sites.title as site_name',
                'meeting_attendances.*'
            )
            ->get();

        foreach ($meeting_users as $key => $user) {
            $meeting_document = MeetingDocument::where('meeting_documents.meeting_id', $user->meeting_id)
                ->join('employee_signed_documents', 'meeting_documents.id', '=', 'employee_signed_documents.document_id')
                ->where('employee_signed_documents.employee_id', $user->employee_id)
                ->where('employee_signed_documents.date', $user->date)
                ->select('meeting_documents.title', 'employee_signed_documents.*')
                ->get();
            $meeting_users[$key]['documents'] = $meeting_document;
        }
        
        // Calculate pagination metadata
        $lastPage = ceil($total / $limit);
        $nextPageUrl = $page < $lastPage ? request()->fullUrlWithQuery(array_merge(request()->query(), ['page' => $page + 1])) : null;
        $prevPageUrl = $page > 1 ? request()->fullUrlWithQuery(array_merge(request()->query(), ['page' => $page - 1])) : null;
        
        $data = [
            'total' => $total,
            'total_meetings' => $total_meetings,
            'total_joined_employees' => $totalJoinedEmployees,
            'total_guest_employees' => $totalGuestEmployees,
            'page' => $page,
            'limit' => $limit,
            'meeting_users' => $meeting_users,
        ];

        // Return direct JSON response to avoid prepareData pagination issues with custom structure
        return response()->json([
            'message' => 'Meeting portal data retrieved successfully.',
            'statusCode' => 200,
            'data' => $data,
            'pagination' => [
                'total' => $total,
                'per_page' => $limit,
                'current_page' => $page,
                'last_page' => $lastPage,
                'from' => $skip + 1,
                'to' => min($skip + $limit, $total),
                'next_page_url' => $nextPageUrl,
                'prev_page_url' => $prevPageUrl
            ]
        ], 200);
    }

    function getEmployeeMeetingBeforeCheckIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:emp_company_details,id',
            'longitude' => 'required',
            'latitude' => 'required',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $userTable = $this->getUserTable();
        if ($userTable === "emp") {
            $employee = EmpCompanyDetails::find($request->employee_id);
            $customer_id = $employee->customer_id;
            $workspace_id = $employee->workspace_id;
        } elseif ($userTable === "customer") {
            $employee = User::find($request->employee_id);
            $customer_id = $employee->id;
            $workspace_id = $employee->current_workspace_id;
        }
        if (!$employee) {
            return $this->message('Employee record not found.', 404);
        }
        // Check customer or employee permissions
        if (
            ($userTable === "customer" && ($employee->id != auth()->id() || $employee->current_workspace_id != auth()->user()->current_workspace_id)) ||
            ($userTable === "emp" && ($employee->customer_id != auth()->user()->customer_id || $employee->workspace_id != auth()->user()->workspace_id))
        ) {
            return $this->message('Unauthorized access to this employee record.', 403);
        }
        $point = [
            'longitude' => $request->longitude,
            'latitude' => $request->latitude,
        ];
        $sites = Sites::where('customer_id', $customer_id)->where('workspace_id', $workspace_id)->where('active', '1')->where('del', '0')->get();
        
        $foundSite = null;
        foreach ($sites as $site) {
            $center = [
                'longitude' => $site->longitude,
                'latitude' => $site->latitude,
            ];
            $radius = $site->area_radius;

            if (GeoLoacation::isPointWithinRadius($point, $center, $radius)) {
                $foundSite = $site;
                break;
            }
        }
        if (!$foundSite) {
            $message = 'The provided location is outside the permitted radius of any site.';
            storeAttendanceLogs([
                'employee_id' => $request->employee_id,
                'site_id' => null,
                'check_in' => null,
                'check_out' => null,
                'date' => now()->format('Y-m-d'),
                'longitude' => $request->longitude,
                'latitude' => $request->latitude,
                'message' => $message,
                'customer_id' => $customer_id,
                'workspace_id' => $workspace_id,
                'error_type' => 'invalid-location-for-meeting',
                'request_type' => 'meeting',
            ]);
            return $this->message($message, 422);
        }
        $attendance = EmployeeAttendance::where('employee_id', $request->employee_id)
            // ->where('date', now()->format('Y-m-d'))
            ->with('sites')
            ->orderBy('id', 'desc')
            ->first();
        if ($attendance && $attendance->site_id != $foundSite->id) {
            $message = 'You are trying to get meetings from the site: ' . $foundSite->title . '. Your assigned site is: ' . $attendance->sites->title . '.';
            storeAttendanceLogs([
                'employee_id' => $request->employee_id,
                'site_id' => $attendance->site_id,
                'check_in' => $attendance->check_in,
                'check_out' => null,
                'date' => now()->format('Y-m-d'),
                'longitude' => $request->longitude,
                'latitude' => $request->latitude,
                'message' => "Checked in at {$attendance->sites->title}, trying to access {$foundSite->title}.",
                'customer_id' => $customer_id,
                'workspace_id' => $workspace_id,
                'error_type' => 'on-other-site',
                'request_type' => 'meeting',
            ]);
            return $this->message($message, 422);
        }
        $meetings = $this->getAllEmployeeMeetings($request->employee_id, $foundSite->id, "1");
        if (count($meetings) > 0) {
            $message = 'Employee meetings retrieved successfully.';
            storeAttendanceLogs([
                'employee_id' => $request->employee_id,
                'site_id' => $foundSite->id,
                'check_in' => $attendance->check_in ?? null,
                'check_out' => null,
                'date' => now()->format('Y-m-d'),
                'longitude' => $request->longitude,
                'latitude' => $request->latitude,
                'message' => $message,
                'customer_id' => $customer_id,
                'workspace_id' => $workspace_id,
                'error_type' => 'success',
                'request_type' => 'meeting',
            ]);
            return $this->success($meetings, $message);
        }

        $message = "There are no meetings for today.";
        storeAttendanceLogs([
            'employee_id' => $request->employee_id,
            'site_id' => null,
            'check_in' => null,
            'check_out' => null,
            'date' => now()->format('Y-m-d'),
            'longitude' => $request->longitude,
            'latitude' => $request->latitude,
            'message' => $message,
            'customer_id' => $customer_id,
            'workspace_id' => $workspace_id,
            'error_type' => 'no-meetings',
            'request_type' => 'meeting',
        ]);
        return $this->success($meetings, $message);
    }
    function getMeetingGuest(Request $request, $id)
    {
        $meeting_id = base64_decode($id);
        $meeting_id_number = explode('|', $meeting_id)[0]; // Get first part
        $meeting = Meeting::where('id', $meeting_id_number)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        $customerSettings = Adminsettings::where('customer_id', $meeting->customer_id)
        ->where('workspace', $meeting->workspace_id)
        ->get()
        ->map(function ($setting) {
            return [
                'key' => $setting->key,
                'value' => $setting->value
            ];
        })
        ->toArray();
        $data = [
            'customerSettings' => $customerSettings,
        ];
        return $this->success($data, 'Meeting found successfully.');
    }

    function addGuestUser(Request $request)
    {
        $meeting_id = base64_decode($request->id);
        $guest_meeting_date = base64_decode($request->meeting_date);
        $meeting = Meeting::where('id', $meeting_id)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $data = [
            'meeting' => $meeting,
            'guest_meeting_date' => $guest_meeting_date,
        ];
        return $this->success($data, 'Meeting found successfully.');
    }

    public function addGuestUserDetailsByGuest(Request $request)
    {
        // Validate the input
        $validator = Validator::make($request->all(), [
            'meeting_id' => 'required|exists:meetings,id',
            'meeting_date' => 'required|date',
            'name' => 'required|string',
            'email' => 'required|email',
            'phone_number' => 'required',
            'occupation' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        // Find the meeting
        $meeting = Meeting::find($request->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found.', 404);
        }
        $guest_user = new MeetingGuestUser();
        $guest_user->meeting_id = $request->meeting_id;
        $guest_user->name = $request->name;
        $guest_user->email = $request->email;
        $guest_user->phone_number = $request->phone_number;
        $guest_user->occupation = $request->occupation;
        $guest_user->details = $request->details ?? "";
        $guest_user->meeting_date = $request->meeting_date;
        $guest_user->added_by = 0; // Use the correct user or employee ID
        $guest_user->save();
        if (!$guest_user) {
            return $this->message('Details are not added.', 422);
        }
        return $this->success($guest_user, 'Guest user details created successfully.');
    }

    function getMeetingAttendanceDetails(Request $request)
    {
        if (empty($request->meeting_id)) {
            return $this->message('Invalid Request', 422);
        }
        $meeting = Meeting::where('id', $request->meeting_id)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        // Add customer and employee checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        // Fetch meeting attendance details
        $meeting_users = MeetingAttendance::join('meetings', 'meeting_attendances.meeting_id', '=', 'meetings.id')
            ->where('meetings.id', $request->meeting_id)
            ->join('emp_personal_details', 'meeting_attendances.employee_id', '=', 'emp_personal_details.emp_id')
            ->select(
                'emp_personal_details.first_name',
                'emp_personal_details.last_name',
                'emp_personal_details.image',
                'meetings.agenda',
                'meeting_attendances.*'
            )
            ->orderBy('meeting_attendances.id', 'desc')
            ->get();
        // Attach signed documents to each attendance record
        foreach ($meeting_users as $key => $user) {
            $meeting_document = MeetingDocument::where('meeting_documents.meeting_id', $user->meeting_id)
                ->join('employee_signed_documents', 'meeting_documents.id', '=', 'employee_signed_documents.document_id')
                ->where('employee_signed_documents.employee_id', $user->employee_id)
                ->where('employee_signed_documents.date', $user->date)
                ->select('meeting_documents.title', 'meeting_documents.document_path', 'employee_signed_documents.*')
                ->get();

            $meeting_users[$key]['documents'] = $meeting_document;
        }
        // Fetch meeting details
        $meeting_details = Meeting::where('id', $request->meeting_id)
            ->where('is_deleted', 0)
            ->with('created_by', 'meeting_organisers', 'meeting_site', 'meeting_note_documents', 'meeting_documents')
            ->first();
        if (!$meeting_details) {
            return $this->message('Meeting details not found.', 404);
        }
        $data = [
            'meeting' => $meeting_details,
            'attendance_details' => $meeting_users,
        ];
        return $this->success($data, 'Meeting attendance details retrieved successfully.');
    }

    function generateSignedDocumentPDF(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'meeting_id' => 'required|exists:meetings,id',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $meeting = Meeting::where('id', $request->meeting_id)
            ->where('is_deleted', 0)
            ->with('created_by', 'meeting_organisers', 'meeting_site', 'meeting_documents_only')
            ->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        // Add customer and employee checks
        $userTable = $this->getUserTable();
        if (
            ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) ||
            ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id))
        ) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $meeting_users = MeetingAttendance::join('meetings', 'meeting_attendances.meeting_id', '=', 'meetings.id')
            ->where('meetings.id', $request->meeting_id)
            ->join('emp_personal_details', 'meeting_attendances.employee_id', '=', 'emp_personal_details.emp_id')
            ->select(
                'emp_personal_details.first_name',
                'emp_personal_details.last_name',
                'emp_personal_details.image',
                'meetings.agenda',
                'meeting_attendances.*'
            )
            ->orderBy('meeting_attendances.id', 'desc')
            ->get();
        // Fetch and map meeting documents to their respective users
        $documents = MeetingDocument::join('employee_signed_documents', 'meeting_documents.id', '=', 'employee_signed_documents.document_id')
            ->where('meeting_documents.meeting_id', $request->meeting_id)
            ->select('meeting_documents.title', 'meeting_documents.document_path', 'employee_signed_documents.*')
            ->get()
            ->groupBy('employee_id');
        foreach ($meeting_users as $user) {
            $user->documents = $documents->get($user->employee_id) ?? [];
        }
        // Generate PDF content
        $pdfContent = PDF::loadView('download', ['meeting' => $meeting, 'meeting_users' => $meeting_users])
            ->setOptions(['defaultFont' => 'DejaVu Sans', 'isRemoteEnabled' => true])
            ->output();
        // Save the PDF file
        $directory = 'MeetingDocuments';
        $fileName = 'meeting_attendance_' . $meeting->id . '_' . time() . '.pdf';
        $filePath = public_path("$directory/$fileName");
        if (!is_dir(public_path($directory))) {
            mkdir(public_path($directory), 0777, true);
        }
        file_put_contents($filePath, $pdfContent);
        if (!file_exists($filePath)) {
            return $this->message('Failed to save the generated PDF.', 500);
        }
        // Save the document details to the database
        MeetingDocument::create([
            'meeting_id' => $request->meeting_id,
            'title' => 'Signed Meeting Attendance Document',
            'document_path' => "$directory/$fileName",
            'uploaded_by' => auth()->user()->id,
        ]);
        $downloadUrl = url("$directory/$fileName");
        return $this->success($downloadUrl, 'PDF generated and saved successfully.');
    }

    function addMeetingNotes(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'meeting_id' => 'required|exists:meetings,id',
            'meeting_notes' => 'required|string'
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $meeting = Meeting::find($request->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        // Add customer and employee checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $meeting->notes = $request->meeting_notes ?? null;
        $meeting->save();
        return $this->success($meeting, 'Meeting notes added successfully.');
    }

    public function meetingSettings(Request $request)
    {
        // Fetch settings where 'del' is 0
        $query = MeetingDocumentSetting::where('del', '0');
        $query = $this->applyCustomerWorkspaceFilter($query);
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', '%' . $searchTerm . '%')
                    ->orWhere('description', 'like', '%' . $searchTerm . '%')
                    ->orWhere('type_of_setting', 'like', '%' . $searchTerm . '%');
            });
        }
        $query_result = $query->orderBy("id", "desc")->get();
        
        return $this->success($query_result, 'Settings retrieved successfully.');
    }

    public function meetingSettingsStore(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'setting_title' => 'required|string',
            'setting_description' => 'required|string',
            'type_of_setting' => 'required|string',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $validatedData = $validator->validated();
        // Add employee and customer checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer") {
            $customerId = auth()->id();
            $workspaceId = auth()->user()->current_workspace_id;
        } elseif ($userTable === "emp") {
            $customerId = auth()->user()->customer_id;
            $workspaceId = auth()->user()->workspace_id;
        } else {
            return $this->message('Unauthorized access.', 403);
        }
        $MeetingDocumentSetting = new MeetingDocumentSetting();
        $MeetingDocumentSetting->title = $validatedData['setting_title'];
        $MeetingDocumentSetting->description = $validatedData['setting_description'];
        $MeetingDocumentSetting->type_of_setting = $validatedData['type_of_setting'];
        $MeetingDocumentSetting->customer_id = $customerId; // Add customer ID
        $MeetingDocumentSetting->workspace_id = $workspaceId; // Add workspace ID
        if ($MeetingDocumentSetting->save()) {
            return $this->success($MeetingDocumentSetting, 'Setting created successfully.');
        }
        return $this->message('Setting not saved, please retry.', 500);
    }

    public function meetingSettingsDelete($id)
    {
        // Fetch the setting
        $setting = MeetingDocumentSetting::find($id);
        if (!$setting) {
            return $this->message('Setting not found.', 404);
        }
        if ($setting->del == 1) {
            return $this->message('Setting is already deleted.', 400);
        }
        $userTable = $this->getUserTable();
        if ($userTable === "customer") {
            if ($setting->customer_id != auth()->id() || $setting->workspace_id != auth()->user()->current_workspace_id) {
                return $this->message('Unauthorized access to this setting.', 403);
            }
        } elseif ($userTable === "emp") {
            if ($setting->customer_id != auth()->user()->customer_id || $setting->workspace_id != auth()->user()->workspace_id) {
                return $this->message('Unauthorized access to this setting.', 403);
            }
        } else {
            return $this->message('Unauthorized access.', 403);
        }
        $update_status = $setting->update(['del' => '1']);
        if ($update_status) {
            return $this->message('Setting deleted successfully.');
        }
        return $this->message('Setting not deleted, please retry.', 500);
    }

    public function meetingSettingsUpdate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'setting_id' => 'required|exists:meeting_document_settings,id',
            'setting_title_edit' => 'required|string',
            'type_of_setting_edit' => 'required|string',
            'setting_description_edit' => 'nullable|string',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $validatedData = $validator->validated();
        $setting = MeetingDocumentSetting::find($validatedData['setting_id']);
        if (!$setting) {
            return $this->message('Setting not found.', 404);
        }
        // Add employee and customer checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($setting->customer_id != auth()->id() || $setting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this setting.', 403);
        }
        if ($userTable === "emp" && ($setting->customer_id != auth()->user()->customer_id || $setting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this setting.', 403);
        }
        // Update the setting
        $status = $setting->update([
            'title' => $validatedData['setting_title_edit'],
            'type_of_setting' => $validatedData['type_of_setting_edit'],
            'description' => $validatedData['setting_description_edit'] ?? null,
        ]);
        if (!$status) {
            return $this->message('Setting not updated, please try again.', 500);
        }
        return $this->success($setting, 'Setting updated successfully.');
    }

    public function meetingDocumentStep1Store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'meeting_id' => 'required|exists:meetings,id',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $meeting = Meeting::where('id', $request->meeting_id)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        // Add employee and customer checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }

        $sequence = $request->sequence;
        $facilitated_by = $request->faciliated_by;
        $meeting_id = $request->meeting_id;
        // Ensure week_days is properly formatted as Y-m-d
        $week_days = date('Y-m-d', strtotime($request->week_days));
        $meeting_comments_notes = $request->meeting_comments_notes;
        $project_id = $request->project_id;
        $document_id = $request->document_id;

        if ($sequence == 1) {
            // If document_id is provided, update the existing document
            $MeetingDocumentNote = null;

            if ($document_id) {
                $MeetingDocumentNote = MeetingDocumentNote::find($document_id);
                if (!$MeetingDocumentNote) {
                    return $this->message('Meeting Document Not Found.', 404);
                }
            }

            // If document doesn't exist, create a new one
            if (!$MeetingDocumentNote) {
                $MeetingDocumentNote = MeetingDocumentNote::create([
                    'meeting_id' => $meeting_id,
                    'prestart_facilitated_by' => $facilitated_by,
                    'comments_on_weather' => $meeting_comments_notes,
                    'project_id' => $project_id,
                    'project_number' => $request->project_number ?? null,
                    'site_id' => $request->site_id,
                    'prestart_meeting_notes' => $request->meeting_prestart_notes,
                    'day_of_the_week' => $week_days,
                    'type_of_setting' => $request->meeting_type_of_setting,
                ]);
                $meeting_document_record = [
                    'meeting_id' => $meeting_id,
                    'meeting_note_id' => $MeetingDocumentNote->id,
                    'document_path' => null,
                    'title' => $meeting->agenda ?? 'Meeting Document',
                    'uploaded_by' => Auth::id(),
                    'status' => 0,

                ];
                // Update or Insert the record in the database
                DB::table('meeting_note_documents_pdfs')->updateOrInsert(
                    [
                        'meeting_id' => $meeting_id,
                        'meeting_note_id' => $MeetingDocumentNote->id,
                    ],
                    $meeting_document_record
                );
            } else {
                // If document exists, update it
                $MeetingDocumentNote->update([
                    'prestart_facilitated_by' => $facilitated_by,
                    'comments_on_weather' => $meeting_comments_notes,
                    'project_id' => $project_id,
                    'project_number' => $request->project_number ?? null,
                    'site_id' => $request->site_id,
                    'prestart_meeting_notes' => $request->meeting_prestart_notes,
                    'day_of_the_week' => $week_days,
                    'type_of_setting' => $request->meeting_type_of_setting,
                ]);
                $meeting_document_record = [
                    'meeting_id' => $meeting_id,
                    'meeting_note_id' => $MeetingDocumentNote->id,
                    'document_path' => null,
                    'title' => $meeting->agenda ?? 'Meeting Document',
                    'uploaded_by' => Auth::id(),
                    'status' => 0,

                ];
                // Update or Insert the record in the database
                DB::table('meeting_note_documents_pdfs')->updateOrInsert(
                    [
                        'meeting_id' => $meeting_id,
                        'meeting_note_id' => $MeetingDocumentNote->id,
                    ],
                    $meeting_document_record
                );
            }

            // Remove previous checkpoints
            MeetingDocumentNoteCheckpoint::where('meeting_id', $MeetingDocumentNote->id)->delete();
            if (!empty($request->meeting_ids)) {
                foreach ($request->meeting_ids as $setting_id) {
                    MeetingDocumentNoteCheckpoint::updateOrCreate(
                        [
                            'meeting_id' => $MeetingDocumentNote->id,
                            'meeting_seeting_id' => $setting_id,
                        ],
                        [
                            'is_note_checked' => '1',
                            'del' => '0',
                        ]
                    );
                }
            }
            return $this->success($MeetingDocumentNote->id, 'Records saved successfully.');
        }

        // If sequence is not 1, create a new document
        $MeetingDocumentNote = MeetingDocumentNote::create([
            'meeting_id' => $meeting_id,
            'prestart_facilitated_by' => $facilitated_by,
            'comments_on_weather' => $meeting_comments_notes,
            'project_id' => $project_id,
            'project_number' => $request->project_number ?? null,
            'site_id' => $request->site_id,
            'prestart_meeting_notes' => $request->meeting_prestart_notes,
            'day_of_the_week' => $week_days ?? null,
            'type_of_setting' => $request->meeting_type_of_setting,
        ]);

         $meeting_document_record = [
            'meeting_id' => $meeting_id,
            'meeting_note_id' => $MeetingDocumentNote->id,
            'document_path' => null,
            'title' => $meeting->agenda ?? 'Meeting Document',
            'uploaded_by' => Auth::id(),
            'status' => 0,
            
        ];
        // Update or Insert the record in the database
        DB::table('meeting_note_documents_pdfs')->updateOrInsert(
            [
                'meeting_id' => $meeting_id,
                'meeting_note_id' => $MeetingDocumentNote->id,
            ],
            $meeting_document_record
        );

        $newMeetingDocumentId = $MeetingDocumentNote->id;

        if (!empty($request->meeting_ids)) {
            foreach ($request->meeting_ids as $setting_id) {
                MeetingDocumentNoteCheckpoint::create([
                    'meeting_id' => $newMeetingDocumentId,
                    'meeting_seeting_id' => $setting_id,
                    'is_note_checked' => '1',
                    'del' => '0',
                ]);
            }
        }
        return $this->success($newMeetingDocumentId, 'New records created successfully.');
    }

    public function meetingDocumentStep1Edit(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'document_id' => 'required|exists:meeting_document_notes,id',
        ]);

        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }

        $meeting_note_id = $request->document_id;

        // Get meeting document note
        $meeting_document_note = MeetingDocumentNote::where('id', $meeting_note_id)->first();

        if (!$meeting_document_note) {
            return $this->message('Meeting document note not found.', 404);
        }

        $meeting_id = $meeting_document_note->meeting_id;

        // Get meeting with all relationships
        $meeting = Meeting::where('id', $meeting_id)
            ->where('is_deleted', 0)
            ->with('created_by', 'meeting_users', 'meeting_organisers', 'meeting_site', 'meeting_documents', 'meeting_histories', 'meeting_guest_users')
            ->first();

        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }

        // Authorization checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === 'customer') {
            $customerId = auth()->user()->id;
            $workspaceId = auth()->user()->current_workspace_id;
        }

        if ($userTable === 'emp') {
            $customerId = auth()->user()->customer_id;
            $workspaceId = auth()->user()->workspace_id;
        }

        // Handle weekly frequency
        if ($meeting && isset($meeting->frequency) && $meeting->frequency == config('constants.meeting_frequency.weekly.key')) {
            $daysOfWeek = explode(',', $meeting->days);
            $meeting->week_days = getDaysBetween($meeting->start_date, $meeting->end_date, $daysOfWeek);
        }

        // Step 1 Data - Meeting Settings with checkpoint status
        $meetings_settings = [];
        if ($meeting_document_note->type_of_setting) {
            $meetings_settings = MeetingDocumentSetting::where('type_of_setting', $meeting_document_note->type_of_setting)
                ->where('del', '0')
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->with(['check_status' => function ($query) use ($meeting_note_id) {
                    $query->where('meeting_id', $meeting_note_id);
                }])
                ->get();
        }

        // Step 2 Data - Signatures and Users
        $meeting_signature = MeetingNoteSignature::where('meeting_id', $meeting_id)->where('del', '0')->get();
        $meeting_user_ids = MeetingUser::where('meeting_id', $meeting_id)->pluck('employee_id');

        $guest_users = DB::table('meeting_guest_users')->where('meeting_id', $meeting_id)->get();

        // Step 3 Data - Framework Tables
        $erecting_framework = MeetingNoteTable::where('name', 'erecting_framework')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $stripping_framework = MeetingNoteTable::where('name', 'stripping_framework')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $jumping_framework = MeetingNoteTable::where('name', 'jumping_framework')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $installing_framework = MeetingNoteTable::where('name', 'installing_framework')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $dismantling_framework = MeetingNoteTable::where('name', 'dismantling_framework')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $stripping_no_framework = MeetingNoteTable::where('name', 'stripping_no_framework')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $preparing_walls = MeetingNoteTable::where('name', 'preparing_walls')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $bondek = MeetingNoteTable::where('name', 'bondek')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        $other = MeetingNoteTable::where('name', 'other')
            ->where('meeting_note_id', $meeting_note_id)
            ->where('del', '0')
            ->get();

        // Combine all data
        $data = [
            // Step 1 Data
            'meeting' => $meeting,
            'meeting_document_note' => $meeting_document_note,
            'meetings_settings' => $meetings_settings,

            // Step 2 Data
            'meeting_signature' => $meeting_signature,
            'guest_users' => $guest_users,

            // Step 3 Data
            'erecting_framework' => $erecting_framework,
            'stripping_framework' => $stripping_framework,
            'jumping_framework' => $jumping_framework,
            'installing_framework' => $installing_framework,
            'dismantling_framework' => $dismantling_framework,
            'stripping_no_framework' => $stripping_no_framework,
            'preparing_walls' => $preparing_walls,
            'bondek' => $bondek,
            'other' => $other,
            'sequence' => 1
        ];

        return $this->success($data, 'Record retrieved successfully.');
    }

    private function saveSignatures($meeting_document_id, $employee_ids, $signature_files, $dates, $modelClass)
    {
        try {
            $employee_count = count($employee_ids);

            for ($i = 0; $i < $employee_count; $i++) {
                $employee_id = $employee_ids[$i];
                $signature_date = isset($dates[$i]) ? $dates[$i] : null;
                $signature_file = isset($signature_files[$i]) ? $signature_files[$i] : null;

                if (empty($employee_id)) {
                    continue;
                }
                $signature_path = null;
                if ($signature_file && $signature_file->isValid()) {
                    $imageName = $employee_id . time() . '.' . $signature_file->getClientOriginalExtension();
                    $signature_file->move(public_path('uploads/inspection_document_signatures'), $imageName);
                    $signature_path = 'uploads/inspection_document_signatures/' . $imageName;
                }

                $signature = $modelClass::where('meeting_id', $meeting_document_id)
                    ->where('employee_id', $employee_id)
                    ->where('del', '0')
                    ->first();

                if ($signature) {
                    if ($signature_path) {
                        $signature->signature = $signature_path;
                    }
                    if ($signature_date) {
                        $signature->date = $signature_date;
                    }
                    $signature->save();
                } else {
                    $modelClass::create([
                        'meeting_id' => $meeting_document_id,
                        'employee_id' => $employee_id,
                        'signature' => $signature_path,
                        'date' => $signature_date,
                    ]);
                }
            }
        } catch (Exception $e) {
            $shortMessage = substr($e->getMessage(), 0, 200) . (strlen($e->getMessage()) > 200 ? '...' : '');
            $log = [
                'employee_id' => auth()->user()->id,
                'message' => 'SWMS Step 3 error:' . $shortMessage,
                'report_id' =>  $request->meeting_document_id,
                'report_type' => 'swms_report',
                'error_type' => 'Exception error'
            ];
            storeReportsLogs($log);
            return response()->json([
                'message' => 'An error occurred : ' . $shortMessage,
            ], 500);
        }
    }

    public function meetingDocumentStep2Store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'document_id' => 'required|exists:meeting_document_notes,id',
        ]);
        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }
        $meeting_note_document = MeetingDocumentNote::where('id', $request->document_id)->first();
        if (!$meeting_note_document) {
            return $this->message('Meeting document note not found.', 404);
        }
        $meeting_id = $meeting_note_document->meeting_id;
        $meeting = Meeting::where('id', $meeting_id)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        // Add customer and employee checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        // Generate PDF for the meeting note
        try {
            $response = $this->MeetingNotePDF($request->document_id);
            return $response;
        } catch (\Exception $e) {
            return $this->message('Failed to generate PDF for the meeting note. ' . $e->getMessage(), 500);
        }

        return $this->success($meeting_id, 'Records saved successfully.');
    }

    public function meetingDocumentStep2Edit(Request $request)
    {
        $sequence = $request->sequence;
        $meeting_note_id = $request->document_id;

        $meeting_note_document =   MeetingDocumentNote::where('id', $meeting_note_id)->first();
        $meeting_id = $meeting_note_document->meeting_id;
        $meeting_signature =    MeetingNoteSignature::where('meeting_id', $meeting_id)->where('del', '0')->get();
        // $meeting_document_note = MeetingDocumentNote::where('meeting_id', $meeting_id)->first();
        $meeting_user_ids = MeetingUser::where('meeting_id', $meeting_id)->pluck('employee_id');

        $authorised_by_list = EmpCompanyDetails::where('compeleted', '1')
            ->where('approved', '1')
            ->where('status', '1')
            ->where('del', '0')
            ->whereIn('id', $meeting_user_ids)
            ->with([
                'empPersonalDetails' => function ($query) {
                    $query->select('emp_id', 'first_name', 'middle_name', 'last_name');
                },
                'accessRole' => function ($query) {
                    $query->select('id', 'title', 'code');
                },
                'empTier' => function ($query) {
                    $query->select('id', 'title');
                },
                'accessTier' => function ($query) {
                    $query->select('id', 'title', 'tier_key');
                },
            ])
            ->select('id', 'access_role', 'tier_id')
            ->whereIn('tier_id', function ($query) {
                $query->select('id')
                    ->from('tiers')
                    ->whereIn('tier_key', ['B-1', 'B-2', 'B-3']);
            })
            ->get();
        $meeting = Meeting::where('id',  $meeting_id)->where('is_deleted', 0)->with('created_by', 'meeting_users', 'meeting_organisers', 'meeting_site', 'meeting_documents', 'meeting_histories', 'meeting_guest_users')->first();
        if ($meeting) {
            if (isset($meeting->frequency) && $meeting->frequency == config('constants.meeting_frequency.weekly.key')) {
                $daysOfWeek = explode(',', $meeting->days);
                $result = getDaysBetween($meeting->start_date, $meeting->end_date, $daysOfWeek);
                $meeting->week_days = $result;
            }
            $data['meeting_recuring'] = $meeting;
        }
        $guest_users = DB::table('meeting_guest_users')->where('meeting_id', $meeting_id)->get();
        $data['meeting_signature'] = $meeting_signature;
        $data['meeting_document_note'] = $meeting_note_document;
        $data['authorised_by_list'] = $authorised_by_list;
        $data['guest_users'] = $guest_users;

        return $this->success($data, 'Records saved successfully.');
    }

    public function meetingDocumentStep2Delete($id)
    {
        $meetingNoteSignature = MeetingNoteSignature::find($id);
        if (!$meetingNoteSignature) {
            return $this->message('Meeting note signature not found.', 404);
        }
        if ($meetingNoteSignature->del == 1) {
            return $this->message('Meeting note signature has been already deleted.', 433);
        }
        $meeting = Meeting::find($meetingNoteSignature->meeting_id);
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        // Check user access (customer or employee)
        $userTable = $this->getUserTable();
        $unauthorized = ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) ||
            ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id));
        if ($unauthorized) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        // Update deletion status
        $status = $meetingNoteSignature->update(['del' => '1']);
        if (!$status) {
            return $this->message('Failed to delete or already deleted.', 433);
        }
        return $this->message('Deleted successfully.', 200);
    }
    // change the fucniton name into step 2 because it is step not 3
    public function meetingDocumentStep3Store(Request $request)
    {
        $meeting_note_id = $request->document_id;
        $MeetingDocumentNote = MeetingDocumentNote::where('id', $meeting_note_id)->first();
        $sequence = $request->sequence;
        $meeting_id = $MeetingDocumentNote->meeting_id;
        if ($sequence == 0) {
            // logic
        } else {
            MeetingNoteTable::where('meeting_id', $meeting_id)->where('meeting_note_id', $meeting_note_id)->where('del', '0')->delete();
        }
        $meeting = Meeting::where('id', $meeting_id)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        // Add customer and employee checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $buildingInputs = $request->erectingFormwork_building_input;
        $levelsInputs = $request->erectingFormwork_levels_input;
        $areaInputs = $request->erectingFormwork_area_input;
        $workforceAllocations = $request->erectingFormwork_workforce_allocation;
        if ($buildingInputs) {
            foreach ($buildingInputs as $index => $building) {
                $level = $levelsInputs[$index] ?? null;
                $area = $areaInputs[$index] ?? null;
                $workforceAllocation = $workforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'erecting_framework';
                $meetingNote->building = $building;
                $meetingNote->level = $levelsInputs[$index] ?? null;
                $meetingNote->area = $areaInputs[$index] ?? null;
                $meetingNote->workfore_allocation = $workforceAllocations[$index] ?? null;
                $meetingNote->del = '0';
                $meetingNote->save();
            }
        }
        $strippingBuildingInputs = $request->strippingFormwork_building_input;
        $strippingLevelsInputs = $request->strippingFormwork_levels_input;
        $strippingAreaInputs = $request->strippingFormwork_area_input;
        $strippingWorkforceAllocations = $request->strippingFormwork_workforce_allocation;
        if ($strippingBuildingInputs) {
            foreach ($strippingBuildingInputs as $index => $building) {
                $level = $strippingLevelsInputs[$index] ?? null;
                $area = $strippingAreaInputs[$index] ?? null;
                $workforceAllocation = $strippingWorkforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'stripping_framework';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';
                $meetingNote->save();
            }
        }
        $jumpingBuildingInputs = $request->jumpformJumping_building_input;
        $jumpingLevelsInputs = $request->jumpformJumping_levels_input;
        $jumpingAreaInputs = $request->jumpformJumping_area_input;
        $jumpingWorkforceAllocations = $request->jumpformJumping_workforce_allocation;
        // Loop through the inputs
        if ($jumpingBuildingInputs) {
            foreach ($jumpingBuildingInputs as $index => $building) {
                $level = $jumpingLevelsInputs[$index] ?? null;
                $area = $jumpingAreaInputs[$index] ?? null;
                $workforceAllocation = $jumpingWorkforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'jumping_framework';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';
                // Save the record
                $meetingNote->save();
            }
        }
        // installing
        $installingBuildingInputs = $request->installing_building_input;
        $installingLevelsInputs = $request->installing_levels_input;
        $installingAreaInputs = $request->installing_area_input;
        $installingWorkforceAllocations = $request->installing_workforce_allocation;
        if ($installingBuildingInputs) {
            foreach ($installingBuildingInputs as $index => $building) {
                $level = $installingLevelsInputs[$index] ?? null;
                $area = $installingAreaInputs[$index] ?? null;
                $workforceAllocation = $installingWorkforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'installing_framework';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';
                $meetingNote->save();
            }
        }
        // dismantling
        $dismantlingBuildingInputs = $request->dismantling_building_input;
        $dismantlingLevelsInputs = $request->dismantling_levels_input;
        $dismantlingAreaInputs = $request->dismantling_area_input;
        $dismantlingWorkforceAllocations = $request->dismantling_workforce_allocation;
        if ($dismantlingBuildingInputs) {
            foreach ($dismantlingBuildingInputs as $index => $building) {
                $level = $dismantlingLevelsInputs[$index] ?? null;
                $area = $dismantlingAreaInputs[$index] ?? null;
                $workforceAllocation = $dismantlingWorkforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'dismantling_framework';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';
                // Save the record
                $meetingNote->save();
            }
        }
        // stripping_no_framwework
        $stripping_no_framweworkBuildingInputs = $request->stripping_building_input;
        $stripping_no_framweworkLevelsInputs = $request->stripping_levels_input;
        $stripping_no_framweworkAreaInputs = $request->stripping_area_input;
        $stripping_no_framweworkWorkforceAllocations = $request->stripping_workforce_allocation;
        if ($stripping_no_framweworkBuildingInputs) {
            foreach ($stripping_no_framweworkBuildingInputs as $index => $building) {
                $level = $stripping_no_framweworkLevelsInputs[$index] ?? null;
                $area = $stripping_no_framweworkAreaInputs[$index] ?? null;
                $workforceAllocation = $stripping_no_framweworkWorkforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'stripping_no_framework';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';
                // Save the record
                $meetingNote->save();
            }
        }
        // preparing_wall_
        $preparing_wall_framweworkBuildingInputs = $request->preparingWalls_building_input;
        $preparing_wall_framweworkLevelsInputs = $request->preparingWalls_levels_input;
        $preparing_wall_framweworkAreaInputs = $request->preparingWalls_area_input;
        $preparing_wall_framweworkWorkforceAllocations = $request->preparingWalls_workforce_allocation;
        if ($preparing_wall_framweworkBuildingInputs) {
            foreach ($preparing_wall_framweworkBuildingInputs as $index => $building) {
                $level = $preparing_wall_framweworkLevelsInputs[$index] ?? null;
                $area = $preparing_wall_framweworkAreaInputs[$index] ?? null;
                $workforceAllocation = $preparing_wall_framweworkWorkforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'preparing_walls';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';
                // Save the record
                $meetingNote->save();
            }
        }
        // bondek
        $bondekframweworkBuildingInputs = $request->bondek_building_input;
        $bondekframweworkLevelsInputs = $request->bondek_levels_input;
        $bondekframweworkAreaInputs = $request->bondek_area_input;
        $bondekframweworkWorkforceAllocations = $request->bondek_workforce_allocation;
        if ($bondekframweworkBuildingInputs) {
            foreach ($bondekframweworkBuildingInputs as $index => $building) {
                $level = $bondekframweworkLevelsInputs[$index] ?? null;
                $area = $bondekframweworkAreaInputs[$index] ?? null;
                $workforceAllocation = $bondekframweworkWorkforceAllocations[$index] ?? null;
                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'bondek';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';

                // Save the record
                $meetingNote->save();
            }
        }
        // other
        $otherframweworkBuildingInputs = $request->other_building_input;
        $otherframweworkLevelsInputs = $request->other_levels_input;
        $otherframweworkAreaInputs = $request->other_area_input;
        $otherframweworkWorkforceAllocations = $request->other_workforce_allocation;
        if ($otherframweworkBuildingInputs) {
            foreach ($otherframweworkBuildingInputs as $index => $building) {
                $level = $otherframweworkLevelsInputs[$index] ?? null;
                $area = $otherframweworkAreaInputs[$index] ?? null;
                $workforceAllocation = $otherframweworkWorkforceAllocations[$index] ?? null;

                if (empty($building) && empty($level) && empty($area) && empty($workforceAllocation)) {
                    continue;
                }
                $meetingNote = new MeetingNoteTable();
                $meetingNote->meeting_id = $meeting_id;
                $meetingNote->meeting_note_id = $meeting_note_id;
                $meetingNote->name = 'other';
                $meetingNote->building = $building;
                $meetingNote->level = $level;
                $meetingNote->area = $area;
                $meetingNote->workfore_allocation = $workforceAllocation;
                $meetingNote->del = '0';
                $meetingNote->save();
            }
        }
        return $this->success($meeting_note_id, 'Saved Successfully');
    }

    public function meetingDocumentStep3Edit(Request $request)
    {
        $sequence = $request->sequence;
        $meetingNoteId = $request->document_id;
        $meetingDocument = MeetingNoteTable::where('meeting_note_id', $meetingNoteId)->first();
        $meeting = Meeting::where('id', $meetingDocument->meeting_id)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($sequence == 1) {
            $erecting_framework = MeetingNoteTable::where('name', 'erecting_framework')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $stripping_framework = MeetingNoteTable::where('name', 'stripping_framework')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $jumping_framework = MeetingNoteTable::where('name', 'jumping_framework')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $installing_framework = MeetingNoteTable::where('name', 'installing_framework')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $dismantling_framework = MeetingNoteTable::where('name', 'dismantling_framework')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $stripping_no_framework = MeetingNoteTable::where('name', 'stripping_no_framework')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $preparing_walls = MeetingNoteTable::where('name', 'preparing_walls')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $bondek = MeetingNoteTable::where('name', 'bondek')->where('del', '0')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $other = MeetingNoteTable::where('name', 'other')->where('meeting_note_id', $meetingNoteId)->where('del', '0')->get();
            $meeting_document_note = MeetingDocumentNote::where('id', $request->document_id)->first();
            $data['meeting_document_note'] = $meeting_document_note;
            $data['erecting_framework'] = $erecting_framework;
            $data['stripping_framework'] = $stripping_framework;
            $data['jumping_framework'] = $jumping_framework;
            $data['installing_framework'] = $installing_framework;
            $data['dismantling_framework'] = $dismantling_framework;
            $data['stripping_no_framework'] = $stripping_no_framework;
            $data['preparing_walls'] = $preparing_walls;
            $data['bondek'] = $bondek;
            $data['other'] = $other;
        } else {
            $data['meeting_document_note'] = [];
            $data['stripping_framework'] = [];
            $data['jumping_framework'] = [];
            $data['installing_framework'] = [];
            $data['dismantling_framework'] = [];
            $data['stripping_no_framework'] = [];
            $data['preparing_walls'] = [];
            $data['bondek'] = [];
            $data['other'] = [];
        }
        return $this->success($data, 'get Successfully');
    }

    public function MeetingStep3Delete($type, $id)
    {
        $validTypes = [
            'erecting_framework',
            'stripping_framework',
            'installing_framework',
            'dismantling_framework',
            'stripping-no_framework',
            'preparing-walls',
            'bondek',
            'other',
        ];
        $meetingNoteTable = MeetingNoteTable::find($id);
        if ($meetingNoteTable->del == 1) {
            return $this->message('Already deleted.', 433);
        }
        $meeting = Meeting::find($meetingNoteTable->meeting_id);
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        // Validate the type
        if (!in_array($type, $validTypes)) {
            return $this->message('Invalid type specified.', 400);
        }
        // Perform deletion
        $deleteStatus = MeetingNoteTable::where('id', $id)->where('del', '0')->update(['del' => '1']);
        if (!$deleteStatus) {
            return $this->message('Delete failed  Please retry.', 433);
        }
        return $this->message('Deleted successfully.', 200);
    }

    public function MeetingNotePDF($id)
    {
        if (!$id) {
            return $this->message('Report ID is required.please retry...', 433);
        }

        $meeting_note_id = (int)$id;
        $MeetingDocumentNote = MeetingDocumentNote::where('id', $meeting_note_id)->first();

        if (!$MeetingDocumentNote) {
            return $this->message('Meeting document note not found.', 404);
        }

        $meeting_id = $MeetingDocumentNote->meeting_id;

        $meetings = DB::table('meetings')
            ->where('id', $meeting_id)
            ->first();

        $meeting_document_notes = DB::table('meeting_document_notes')
            ->where('id', $meeting_note_id)
            ->first();

        $facilaited_by_list = EmpCompanyDetails::where('approved', '1')
            ->where('status', '1')
            ->where('del', '0')
            ->where('id', $meeting_document_notes->prestart_facilitated_by ?? 0)
            ->with([
                'empPersonalDetails' => function ($query) {
                    $query->select('emp_id', 'first_name', 'middle_name', 'last_name');
                },
                'accessRole' => function ($query) {
                    $query->select('id', 'title', 'code');
                },
                'empTier' => function ($query) {
                    $query->select('id', 'title');
                },
                'accessTier' => function ($query) {
                    $query->select('id', 'title', 'tier_key');
                },
            ])
            ->select('id', 'access_role', 'tier_id')
            ->first();

        if ($facilaited_by_list && $facilaited_by_list->empPersonalDetails) {
            $faciliatedByfull_name = trim($facilaited_by_list->empPersonalDetails->first_name . ' ' . $facilaited_by_list->empPersonalDetails->last_name);

            if ($facilaited_by_list->empPersonalDetails->middle_name) {
                $faciliatedByfull_name = trim($facilaited_by_list->empPersonalDetails->first_name . ' ' . $facilaited_by_list->empPersonalDetails->middle_name . ' ' . $facilaited_by_list->empPersonalDetails->last_name);
            }
        } else {
            $faciliatedByfull_name = '';
        }

        $meeting_project = DB::table('meeting_document_notes')
            ->leftJoin('projects', 'meeting_document_notes.project_id', '=', 'projects.id')
            ->select('projects.title as project_title')
            ->where('meeting_document_notes.id', $meeting_note_id)
            ->first();

        $meeting_sites = DB::table('meeting_document_notes')
            ->leftJoin('sites', 'meeting_document_notes.site_id', '=', 'sites.id')
            ->select('sites.title as site_title')
            ->where('meeting_document_notes.id', $meeting_note_id)
            ->first();

        // step2 
        $meeting_document_note_checkpoints = DB::table('meeting_document_note_checkpoints')
            ->leftJoin('meeting_document_settings', 'meeting_document_note_checkpoints.meeting_seeting_id', '=', 'meeting_document_settings.id')
            ->select(
                'meeting_document_settings.title as meeting_document_settings_title',
                'meeting_document_settings.description as meeting_document_settings_description'
            )
            ->where('meeting_document_note_checkpoints.meeting_id', $meeting_note_id)
            ->get();


        // ////////////////////////////////////////////////////
        $users = EmpPersonalDetails::join('emp_company_details', 'emp_personal_details.emp_id', 'emp_company_details.id')
            ->where('emp_company_details.status', 1)->where('emp_company_details.del', 0)->where('emp_company_details.approved', 1)->select('emp_personal_details.*')->get();

        $meetings = DB::table('meetings')
            ->where('id', $meeting_document_notes->meeting_id)
            ->first();

        if ($meetings->allow_meeting_members == 1) {
            $meetings_users = DB::table('meeting_users')
                ->where('meeting_id', $meeting_document_notes->meeting_id)
                ->get();

            foreach ($meetings_users as $user) {
                $exists = DB::table('meeting_note_signatures')
                    ->where('meeting_id', $meeting_document_notes->meeting_id)
                    ->where('employee_note_id', $meeting_note_id)
                    ->where('employee_id', $user->employee_id)
                    ->exists();

                if (!$exists) {
                    DB::table('meeting_note_signatures')->insert([
                        'meeting_id' => $meeting_document_notes->meeting_id,
                        'employee_note_id' => $meeting_note_id,
                        'employee_id' => $user->employee_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        } else {
            $start_date = $meetings->start_date;
            $end_date = $meetings->end_date;
            $start_time = $meetings->start_time;
            $end_time = $meetings->end_time;
            $days = $meetings->days;

            $meetings_users = DB::table('employee_attendances')
                ->where('site_id', $meetings->site_id)
                ->whereBetween('date', [$start_date, $end_date])
                ->get();

            foreach ($meetings_users as $user) {
                $meeting_attandance_site_id = $user->site_id;
                $exists = DB::table('meeting_note_signatures')
                    ->where('meeting_id', $meeting_id)
                    ->where('employee_note_id', $meeting_note_id)
                    ->where('employee_id', $user->employee_id)
                    ->exists();

                if (!$exists) {
                    DB::table('meeting_note_signatures')->insert([
                        'meeting_id' => $meeting_id,
                        'employee_note_id' => $meeting_note_id,
                        'employee_id' => $user->employee_id,
                    ]);
                }
            }
        }

        // ////////////////////////////////////////////////////
        $meeting_signature_data = DB::table('meeting_note_signatures')
            ->where('del', '0')
            ->whereNotNull('signature')
            ->where('meeting_id', $meeting_id)
            ->where('employee_note_id', $meeting_note_id)
            ->get(['employee_id', 'date', 'signature']);

        if ($meeting_signature_data->isNotEmpty()) {
            $employee_ids = $meeting_signature_data->pluck('employee_id');
            $meeting_sign_by_list = EmpCompanyDetails::where('approved', '1')
                ->where('status', '1')
                ->where('del', '0')
                ->whereIn('id', $employee_ids)
                ->with([
                    'empPersonalDetails' => function ($query) {
                        $query->select('emp_id', 'first_name', 'middle_name', 'last_name');
                    },
                    'accessRole' => function ($query) {
                        $query->select('id', 'title', 'code');
                    },
                    'empTier' => function ($query) {
                        $query->select('id', 'title');
                    },
                    'accessTier' => function ($query) {
                        $query->select('id', 'title', 'tier_key');
                    },
                ])
                ->select('id', 'access_role', 'tier_id')
                ->get();

            $meeting_sign_by_list = $meeting_sign_by_list->map(function ($employee) use ($meeting_signature_data) {
                $signature_data = $meeting_signature_data->firstWhere('employee_id', $employee->id);
                if ($signature_data) {
                    $employee->date = $signature_data->date;
                    $employee->signature = $signature_data->signature;
                }
                return $employee;
            });
        } else {
            $meeting_sign_by_list = collect(); // Empty collection
        }

        // step3
        $erecting_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'erecting_framework')
            ->get();

        $stripping_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'stripping_framework')
            ->get();

        $jumping_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'jumping_framework')
            ->get();

        $installing_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'installing_framework')
            ->get();

        $dismantling_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'dismantling_framework')
            ->get();

        $stripping_no_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'stripping_no_framework')
            ->get();

        $preparing_walls = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'preparing_walls')
            ->get();

        $bondek = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'bondek')
            ->get();

        $other = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'other')
            ->get();
        $name = str_replace(' ', '_', $meetings->agenda ?? 'NoName');

        $type_of_setting = $MeetingDocumentNote->type_of_setting ?? 'Prestart';
        // Convert meeting date to proper format for database
        $meeting_date = null;
        if ($meetings->start_date) {
            try {
                $meeting_date = \Carbon\Carbon::createFromFormat('d-m-Y', $meetings->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $meeting_date = \Carbon\Carbon::parse($meetings->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $meeting_date = now()->format('Y-m-d');
                }
            }
        }
        $meetingDate = null;
        if (isset($meetings->week_days)) {
            // Recurring meeting logic (your existing code)
            $dayKey = $meeting_document_notes->day_of_the_week ?? null;
            if ($dayKey !== null && isset($meetings->week_days[$dayKey])) {
                $value = $meetings->week_days[$dayKey];
                $meetingDate = is_array($value) ? (reset($value) ?: null) : $value;
            } else {
                $meetingDate = collect($meetings->week_days)->flatten()->first();
            }
        } else {
            // Non-recurring meeting: use day_of_the_week directly
            $meetingDate = $meeting_document_notes->day_of_the_week ?? $meetings->start_date ?? null;
        }
        
        // Fill missing date for each signer
        if (isset($meeting_sign_by_list) && $meeting_sign_by_list) {
            $meeting_sign_by_list = $meeting_sign_by_list->map(function ($employee) use ($meetingDate) {
                if (empty($employee->date)) {
                    $employee->date = $meetingDate;
                }
                return $employee;
            });
        }
        
        $meeting_document_record = [
            'meeting_id' => $meeting_id,
            'type_of_setting' => $MeetingDocumentNote->type_of_setting ?? null,
            'meeting_note_id' => $meeting_note_id,
            'document_path' => null,
            'title' => $name,
            'meeting_date' => $meeting_date,
            'uploaded_by' => Auth::id(),
            'type' => 'uploaded',
            'created_at' => now(),
            'updated_at' => now()
        ];
        // Update or Insert the record in the database
        DB::table('meeting_note_documents_pdfs')->updateOrInsert(
            [
                'meeting_id' => $meeting_id,
                'meeting_note_id' => $meeting_note_id
            ],
            $meeting_document_record
        );
        // Prepare the data array that was being passed to the view
        $data = [
            'meetings' => $meetings,
            'meeting_document_notes' => $meeting_document_notes,
            'faciliatedByfull_name' => $faciliatedByfull_name,
            'meeting_project' => $meeting_project,
            'meeting_sites' => $meeting_sites,
            'meeting_document_note_checkpoints' => $meeting_document_note_checkpoints,
            'meeting_sign_by_list' => $meeting_sign_by_list,
            'erecting_framework' => $erecting_framework,
            'stripping_framework' => $stripping_framework,
            'jumping_framework' => $jumping_framework,
            'installing_framework' => $installing_framework,
            'dismantling_framework' => $dismantling_framework,
            'stripping_no_framework' => $stripping_no_framework,
            'preparing_walls' => $preparing_walls,
            'bondek' => $bondek,
            'type_of_setting' => $type_of_setting,
            'other' => $other,
        ];

        // Return the data instead of generating PDF
        return $this->success($data, 'Meeting note data retrieved successfully.');
    }
    // regenerate
    public function uploadMeetingDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'meeting_id' => 'required|exists:meetings,id',
            'document_id' => 'required',
            'file' => 'required|file|mimes:pdf,doc,docx,jpg,jpeg,png,gif',
        ]);

        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 422);
        }

        $meeting_id = $request->meeting_id;
        $document_id = $request->document_id;

        // Verify the meeting exists and is not deleted
        $meeting = Meeting::where('id', $meeting_id)->where('is_deleted', 0)->first();
        if (!$meeting) {
            return $this->message('Meeting not found or has been deleted.', 404);
        }

        // Verify the document belongs to the meeting
        $meeting_note_document = MeetingDocumentNote::where('id', $document_id)
            ->where('meeting_id', $meeting_id)
            ->first();
        if (!$meeting_note_document) {
            return $this->message('Meeting document note not found or does not belong to this meeting.', 404);
        }

        // Add customer and employee checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }

        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }

        try {
            $file = $request->file('file');

            // Check if a record already exists for this meeting_id and document_id
            $existingRecord = DB::table('meeting_note_documents_pdfs')
                ->where('meeting_id', $meeting_id)
                ->where('meeting_note_id', $document_id)
                ->first();

            // If existing record found, delete the old file
            if ($existingRecord && $existingRecord->document_path) {
                $oldFilePath = public_path($existingRecord->document_path);
                if (file_exists($oldFilePath)) {
                    unlink($oldFilePath);
                }
            }

            // Generate unique filename
            $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $file->getClientOriginalExtension();
            $filename = $originalName . '_' . time() . '_' . uniqid() . '.' . $extension;

            // Define directory path
            $directory = public_path('MeetingDocuments');

            // Ensure the directory exists
            if (!file_exists($directory)) {
                mkdir($directory, 0777, true);
            }

            // Move the file to the directory
            $file->move($directory, $filename);

            // Prepare the document path for database
            $publicFilePath = "MeetingDocuments/{$filename}";

            // Prepare the data for database insertion/update
            // Convert meeting date to proper format for database
            $meeting_date = null;
            if ($meeting->start_date) {
                try {
                    $meeting_date = \Carbon\Carbon::createFromFormat('d-m-Y', $meeting->start_date)->format('Y-m-d');
                } catch (\Exception $e) {
                    // If d-m-Y format fails, try to parse with Carbon's default parsing
                    try {
                        $meeting_date = \Carbon\Carbon::parse($meeting->start_date)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        // If all parsing fails, use current date as fallback
                        $meeting_date = now()->format('Y-m-d');
                    }
                }
            }
            
            $meeting_document_record = [
                'meeting_id' => $meeting_id,
                'type_of_setting' => $meeting_note_document->type_of_setting ?? null,
                'meeting_note_id' => $document_id,
                'document_path' => $publicFilePath,
                'title' => $originalName,
                'meeting_date' => $meeting_date,
                'status' => 1,
                'uploaded_by' => Auth::id(),
                'type' => 'uploaded',
                'updated_at' => now()
            ];

            // If existing record, update it; otherwise insert new record
            if ($existingRecord) {
                DB::table('meeting_note_documents_pdfs')
                    ->where('meeting_id', $meeting_id)
                    ->where('meeting_note_id', $document_id)
                    ->update($meeting_document_record);
            } else {
                $meeting_document_record['created_at'] = now();
                DB::table('meeting_note_documents_pdfs')->insert($meeting_document_record);
            }

            // Generate the full URL for the uploaded file
            $fileUrl = url($publicFilePath);

            return $this->success([
                'file_path' => $publicFilePath,
                'file_url' => $fileUrl,
                'filename' => $filename,
                'original_name' => $file->getClientOriginalName(),
                'meeting_id' => $meeting_id,
                'document_id' => $document_id,
                'action' => $existingRecord ? 'updated' : 'created'
            ], 'File uploaded and saved successfully.');
        } catch (\Exception $e) {
            return $this->message('Failed to upload file. ' . $e->getMessage(), 500);
        }
    }

    public function MeetingNotePDFRegenerate($id)
    {
        if (!$id) {
            return $this->message('Report ID is required.please retry...', 433);
        }
        $meeting_note_id = (int)$id;
        $MeetingDocumentNote = MeetingDocumentNote::where('id', $meeting_note_id)->first();
        $meeting = Meeting::find($MeetingDocumentNote->meeting_id);
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting.', 403);
        }
        $meeting_id = $MeetingDocumentNote->meeting_id;
        $meetings = DB::table('meetings')
            ->where('id', $meeting_id)
            ->first();

        $meeting_document_notes = DB::table('meeting_document_notes')
            ->where('id', $meeting_note_id)
            ->first();
        $facilaited_by_list = EmpCompanyDetails::where('approved', '1')
            ->where('status', '1')
            ->where('del', '0')
            ->where('id', $meeting_document_notes->prestart_facilitated_by ?? 0)
            ->with([
                'empPersonalDetails' => function ($query) {
                    $query->select('emp_id', 'first_name', 'middle_name', 'last_name');
                },
                'accessRole' => function ($query) {
                    $query->select('id', 'title', 'code');
                },
                'empTier' => function ($query) {
                    $query->select('id', 'title');
                },
                'accessTier' => function ($query) {
                    $query->select('id', 'title', 'tier_key');
                },
            ])
            ->select('id', 'access_role', 'tier_id')
            ->first();

        if ($facilaited_by_list && $facilaited_by_list->empPersonalDetails) {
            $faciliatedByfull_name = trim($facilaited_by_list->empPersonalDetails->first_name . ' ' . $facilaited_by_list->empPersonalDetails->last_name);

            if ($facilaited_by_list->empPersonalDetails->middle_name) {
                $faciliatedByfull_name = trim($facilaited_by_list->empPersonalDetails->first_name . ' ' . $facilaited_by_list->empPersonalDetails->middle_name . ' ' . $facilaited_by_list->empPersonalDetails->last_name);
            }
        } else {
            $faciliatedByfull_name = '';
        }
        $meeting_project = DB::table('meeting_document_notes')
            ->leftJoin('projects', 'meeting_document_notes.project_id', '=', 'projects.id')
            ->select('projects.title as project_title')
            ->where('meeting_document_notes.id', $meeting_note_id)
            ->first();
        $meeting_sites = DB::table('meeting_document_notes')
            ->leftJoin('sites', 'meeting_document_notes.site_id', '=', 'sites.id')
            ->select('sites.title as site_title')
            ->where('meeting_document_notes.id', $meeting_note_id)
            ->first();

        $meeting_document_note_checkpoints = DB::table('meeting_document_note_checkpoints')
            ->leftJoin('meeting_document_settings', 'meeting_document_note_checkpoints.meeting_seeting_id', '=', 'meeting_document_settings.id')
            ->select(
                'meeting_document_settings.title as meeting_document_settings_title',
                'meeting_document_settings.description as meeting_document_settings_description',

            )
            ->where('meeting_document_note_checkpoints.meeting_id', $meeting_note_id)
            ->get();
        // ////////////////////////////////////////////////////
        $users = EmpPersonalDetails::join('emp_company_details', 'emp_personal_details.emp_id', 'emp_company_details.id')
            ->where('emp_company_details.status', 1)->where('emp_company_details.del', 0)->where('emp_company_details.approved', 1)->select('emp_personal_details.*')->get();
        $meetings = DB::table('meetings')
            ->where('id', $meeting_document_notes->meeting_id)
            ->first();
        if ($meetings->allow_meeting_members == 1) {
            $meetings_users = DB::table('meeting_users')
                ->where('meeting_id', $meeting_document_notes->meeting_id)
                ->get();
            foreach ($meetings_users as $user) {
                $exists = DB::table('meeting_note_signatures')
                    ->where('meeting_id', $meeting_document_notes->meeting_id)
                    ->where('employee_note_id', $meeting_note_id)
                    ->where('employee_id', $user->employee_id)
                    ->exists();

                if (!$exists) {
                    DB::table('meeting_note_signatures')->insert([
                        'meeting_id' => $meeting_document_notes->meeting_id,
                        'employee_note_id' => $meeting_note_id,
                        'employee_id' => $user->employee_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        } else {
            $start_date = $meetings->start_date;
            $end_date = $meetings->end_date;
            $start_time = $meetings->start_time;
            $end_time = $meetings->end_time;
            $days = $meetings->days;

            $meetings_users = DB::table('employee_attendances')
                ->where('site_id', $meetings->site_id)
                ->whereBetween('date', [$start_date, $end_date])
                // ->where('check_in', '>=', $start_time)
                // ->where('check_in', '<=', $end_time)
                ->get();
            foreach ($meetings_users as $user) {
                $meeting_attandance_site_id = $user->site_id;
                $exists = DB::table('meeting_note_signatures')
                    ->where('meeting_id', $meeting_id)
                    ->where('employee_note_id', $meeting_note_id)
                    ->where('employee_id', $user->employee_id)
                    ->exists();

                if (!$exists) {
                    DB::table('meeting_note_signatures')->insert([
                        'meeting_id' => $meeting_id,
                        'employee_note_id' => $meeting_note_id,
                        'employee_id' => $user->employee_id,
                    ]);
                }
            }
        }
        // ////////////////////////////////////////////////////
        $meeting_signature_data = DB::table('meeting_note_signatures')
            ->where('del', '0')
            ->where('meeting_id', $meeting_id)
            ->where('employee_note_id', $meeting_note_id)
            ->get(['employee_id', 'date', 'signature']);

        if ($meeting_signature_data->isNotEmpty()) {
            $employee_ids = $meeting_signature_data->pluck('employee_id');
            $meeting_sign_by_list = EmpCompanyDetails::where('approved', '1')
                ->where('status', '1')
                ->where('del', '0')
                ->whereIn('id', $employee_ids)
                ->with([
                    'empPersonalDetails' => function ($query) {
                        $query->select('emp_id', 'first_name', 'middle_name', 'last_name');
                    },
                    'accessRole' => function ($query) {
                        $query->select('id', 'title', 'code');
                    },
                    'empTier' => function ($query) {
                        $query->select('id', 'title');
                    },
                    'accessTier' => function ($query) {
                        $query->select('id', 'title', 'tier_key');
                    },
                ])
                ->select('id', 'access_role', 'tier_id')
                ->get();

            $meeting_sign_by_list = $meeting_sign_by_list->map(function ($employee) use ($meeting_signature_data) {
                $signature_data = $meeting_signature_data->firstWhere('employee_id', $employee->id);
                if ($signature_data) {
                    $employee->date = $signature_data->date;
                    $employee->signature = $signature_data->signature;
                }
                return $employee;
            });
        } else {
            $meeting_sign_by_list = collect(); // Empty collection
        }
        // step3
        $erecting_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'erecting_framework')
            ->get();
        $stripping_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'stripping_framework')
            ->get();
        $jumping_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'jumping_framework')
            ->get();
        $installing_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'installing_framework')
            ->get();
        $dismantling_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'dismantling_framework')
            ->get();
        $stripping_no_framework = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'stripping_no_framework')
            ->get();
        $preparing_walls = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'preparing_walls')
            ->get();
        $bondek = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'bondek')
            ->get();
        $other = DB::table('meeting_note_tables')
            ->where('meeting_note_tables.meeting_id', $meeting_id)
            ->where('meeting_note_tables.meeting_note_id', $meeting_note_id)
            ->where('meeting_note_tables.name', 'other')
            ->get();


        $name = str_replace(' ', '_', $meetings->agenda ?? 'NoName');
        $date = str_replace('-', '_', $meetings->start_date ?? 'NoDate');
        $added_date = $name . "_" . $date;
        $type_of_setting = $MeetingDocumentNote->type_of_setting ?? 'Prestart';
        $current_pdfs = DB::table('meeting_note_documents_pdfs')->where('meeting_id', $meeting_id)->count();
        $filename = $added_date . "_" . $type_of_setting . "_" . $current_pdfs . ".pdf";
        // 
        $viewPath = 'meeting-pdf';
        $html = view($viewPath, compact(
            'meetings',
            'meeting_document_notes',
            'faciliatedByfull_name',
            'meeting_project',
            'meeting_sites',
            'meeting_document_note_checkpoints',
            'meeting_sign_by_list',
            'erecting_framework',
            'stripping_framework',
            'jumping_framework',
            'installing_framework',
            'dismantling_framework',
            'stripping_no_framework',
            'preparing_walls',
            'bondek',
            'other',
            'type_of_setting'
        ))->render();
        $datas = [
            'meetings' => $meetings,
            'meeting_document_notes' => $meeting_document_notes,
            'faciliatedByfull_name' => $faciliatedByfull_name,
            'meeting_project' => $meeting_project,
            'meeting_sites' => $meeting_sites,
            'meeting_document_note_checkpoints' => $meeting_document_note_checkpoints,
            'meeting_sign_by_list' => $meeting_sign_by_list,
            'erecting_framework' => $erecting_framework,
            'stripping_framework' => $stripping_framework,
            'jumping_framework' => $jumping_framework,
            'installing_framework' => $installing_framework,
            'dismantling_framework' => $dismantling_framework,
            'stripping_no_framework' => $stripping_no_framework,
            'preparing_walls' => $preparing_walls,
            'bondek' => $bondek,
            'other' => $other,
            'type_of_setting' => $type_of_setting,
        ];


        $filename = "meeting_notes_document_{$meeting_id}.pdf";
        $directory = public_path('MeetingDocuments');
        $filePath = "{$directory}/{$filename}";

        // Ensure the directory exists
        if (!file_exists($directory)) {
            mkdir($directory, 0777, true);
        }

        // Check if a record already exists
        $existingRecord = DB::table('meeting_note_documents_pdfs')
            ->where('meeting_id', $meeting_id)
            ->where('meeting_note_id', $meeting_note_id)
            ->first();

        if ($existingRecord) {
            // Delete existing files
            if (file_exists(public_path($existingRecord->document_path))) {
                unlink(public_path($existingRecord->document_path));
            }
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            // Delete existing record from the database
            DB::table('meeting_note_documents_pdfs')
                ->where('meeting_id', $meeting_id)
                ->where('meeting_note_id', $meeting_note_id)
                ->delete();
        }

        // Generate the PDF from the HTML
        $pdf = PDF::loadHTML($html)
            ->setPaper('A4', 'portrait');

        $pdf->save($filePath);

        // Prepare the document path for the database
        $publicFilePath = "MeetingDocuments/{$filename}";

        // Convert meeting date to proper format for database
        $meeting_date = null;
        if ($meetings->start_date) {
            try {
                $meeting_date = \Carbon\Carbon::createFromFormat('d-m-Y', $meetings->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $meeting_date = \Carbon\Carbon::parse($meetings->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $meeting_date = now()->format('Y-m-d');
                }
            }
        }
        
        // Prepare the data to insert into the database
        $meetingDocumentRecord = [
            'meeting_id' => $meeting_id,
            'type_of_setting' => $MeetingDocumentNote->type_of_setting ?? null,
            'meeting_note_id' => $meeting_note_id,
            'document_path' => $publicFilePath,
            'title' => $name,
            'meeting_date' => $meeting_date,
            'uploaded_by' => Auth::id(),
            'type' => 'uploaded',
            'created_at' => now(),
            'updated_at' => now(),
        ];

        // Insert or update the database record
        DB::table('meeting_note_documents_pdfs')->updateOrInsert(
            [
                'meeting_id' => $meeting_id,
                'meeting_note_id' => $meeting_note_id,
            ],
            $meetingDocumentRecord
        );

        // Return the download URL
        $downloadUrl = url($publicFilePath);
        return $this->success($downloadUrl, 'PDF generated and saved successfully.');
    }
    // 
    public function meetingReportList(Request $request)
    {
        $perPage = 10;
        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $query = DB::table('meeting_note_documents_pdfs')
            ->orderBy('created_at', 'desc');
        $totalReports = $query->count();
        $currentReports = $query->skip(($currentPage - 1) * $perPage)
            ->take($perPage)
            ->get();
        // Extract meeting_ids
        $meetingIds = $currentReports->pluck('meeting_id')->unique();
        // Retrieve the meetings
        $meetings = Meeting::whereIn('id', $meetingIds)->get();
        // Apply customer and employee checks
        $userTable = $this->getUserTable();
        foreach ($meetings as $meeting) {
            if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to some meetings.', 403);
            }
            if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to some meetings.', 403);
            }
        }
        $paginator = new LengthAwarePaginator(
            $currentReports,
            $totalReports,
            $perPage,
            $currentPage,
            ['path' => LengthAwarePaginator::resolveCurrentPath()]
        );
        return $this->success($paginator, 'Meeting reports retrieved successfully.');
    }

    public function download(MeetingDocument $document)
    {
        // Check if file exists
        if (!Storage::exists($document->document_path)) {
            abort(404, 'Document not found');
        }

        // Get file content
        $file = Storage::get($document->document_path);

        // Get file mime type
        $mimeType = Storage::mimeType($document->document_path);

        // Generate filename for download
        $filename = $document->title ?? basename($document->document_path);

        // Ensure the filename has an extension
        if (!pathinfo($filename, PATHINFO_EXTENSION)) {
            $extension = pathinfo($document->document_path, PATHINFO_EXTENSION);
            $filename .= '.' . $extension;
        }

        // Return the file as a download response
        return response($file, 200, [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    public function MeetingNoteDocumentDelete(Request $request)
    {
        $meeting_document = MeetingNoteDocumentsPdf::where('meeting_note_id', $request->id)->first();
        if (!$meeting_document) {
            return $this->message('Meeting document not found.', 404);
        }
        // Fetch the associated meeting
        $meeting = Meeting::find($meeting_document->meeting_id);
        if (!$meeting) {
            return $this->message('Associated meeting not found.', 404);
        }
        // Add customer and employee checks
        $userTable = $this->getUserTable();
        if ($userTable === "customer" && ($meeting->customer_id != auth()->id() || $meeting->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this meeting document.', 403);
        }

        if ($userTable === "emp" && ($meeting->customer_id != auth()->user()->customer_id || $meeting->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this meeting document.', 403);
        }
        // Delete the document file
        $path = $meeting_document->document_path;
        if (!$path) {
            return $this->message('Document not find in system .', 404);
        }
        if (file_exists(public_path($path))) {
            unlink(public_path($path));
        }
        // Delete the document record
        $meeting_document->delete();
        $data = "Deleted";
        return $this->success($data, 'Meeting document deleted successfully.');
    }
}
