<?php

namespace App\Http\Controllers\Mobilev2;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\EmployeeAttendance;
use App\Models\EmployeeBreak;
use App\Models\LeaveRequest;
use App\Models\RosterAssign;
use App\Models\Sites;
use App\Models\SiteDocument;
use App\Models\SiteDocumentSignature;
use App\General\GeoLoacation;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\EmployeeSubcontractor;
class AttendanceController extends Controller
{

    /**
     * Check-in for subcontractor employees
     * Similar to regular employee check-in but handles subcontractor_id
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRosterList(Request $request)
{
    $validator = Validator::make($request->all(), [
        'customer_id' => 'required|integer',
        'workspace_id' => 'required|integer',
        'date' => 'nullable|date',
        'from_date' => 'nullable|date',
        'to_date' => 'nullable|date|after_or_equal:from_date',
    ]);

    if ($validator->fails()) {
        return $this->error($validator->errors()->first(), 422);
    }

    $employee = $request->user();
    if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
        return $this->error('Unauthorized access', 401);
    }

    $validatedData = $validator->validated();
    $customerId = $validatedData['customer_id'];
    $workspaceId = $validatedData['workspace_id'];
    $currentDate = now()->format('Y-m-d');
    $currentTime = now()->format('H:i:s');

    // Verify employee has an active invitation for this customer/workspace
    $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
        ->where('customer_id', $customerId)
        ->where('workspace_id', $workspaceId)
        ->where('invitation_status', 'accepted')
        ->where('status', 1)
        ->first();

    if (!$invitation) {
        return $this->error('No active invitation found for this customer and workspace.', 404);
    }

    $subcontractorId = $invitation->subcontractor_id;

    // Build the query for roster assignments
    $query = RosterAssign::join('roster_templates', 'roster_assigns.roster_template_id', 'roster_templates.id')
        ->leftJoin('sites', 'roster_assigns.site_id', 'sites.id')
        ->where('roster_assigns.assign_to', $employee->id)
        ->where('roster_assigns.subcontractor_id', $subcontractorId)
        ->where('roster_assigns.customer_id', $customerId)
        ->where('roster_assigns.workspace_id', $workspaceId);

    // Apply date filters
    if (isset($validatedData['date'])) {
        $query->where('roster_assigns.schedule_date', Carbon::parse($validatedData['date'])->format('Y-m-d'));
    } elseif (isset($validatedData['from_date']) && isset($validatedData['to_date'])) {
        $query->whereBetween('roster_assigns.schedule_date', [
            Carbon::parse($validatedData['from_date'])->format('Y-m-d'),
            Carbon::parse($validatedData['to_date'])->format('Y-m-d')
        ]);
    } else {
        // Default: get today's and future rosters
        $query->where('roster_assigns.schedule_date', '>=', $currentDate);
    }

    $rosters = $query->select(
        'roster_assigns.id as roster_assign_id',
        'roster_assigns.roster_template_id',
        'roster_assigns.schedule_date',
        'roster_assigns.site_id',
        'roster_assigns.customer_id',
        'roster_assigns.workspace_id',
        'roster_assigns.subcontractor_id',
        // 'roster_templates.title as roster_title',
        'roster_templates.start_time',
        'roster_templates.end_time',
        'roster_templates.shift_notes',
        'roster_templates.break_minutes',
        'sites.title as site_name',
        'sites.latitude as site_latitude',
        'sites.longitude as site_longitude',
        'sites.area_radius as site_radius',
        'sites.street_address as site_address'
    )
    ->orderBy('roster_assigns.schedule_date', 'asc')
    ->orderBy('roster_templates.start_time', 'asc')
    ->get();

    // Add check-in status for each roster
    $rostersWithStatus = $rosters->map(function ($roster) use ($employee, $subcontractorId, $currentDate, $currentTime) {
        // Get attendance record for this roster
        $attendance = EmployeeAttendance::where('employee_id', $employee->id)
            ->where('subcontractor_id', $subcontractorId)
            ->where('date', $roster->schedule_date)
            ->where('site_id', $roster->site_id)
            ->first();

        // Determine status
        $roster->is_checked_in = $attendance && $attendance->check_in && !$attendance->check_out;
        $roster->is_completed = $attendance && $attendance->check_in && $attendance->check_out;
        $roster->attendance_id = $attendance ? $attendance->id : null;
        $roster->check_in_time = $attendance ? $attendance->check_in : null;
        $roster->check_out_time = $attendance ? $attendance->check_out : null;

        // Check if this is today's roster and within time window
        $isToday = $roster->schedule_date === $currentDate;
        $isWithinTimeWindow = $currentTime >= $roster->start_time && $currentTime <= $roster->end_time;
        
        $roster->is_today = $isToday;
        $roster->is_within_time_window = $isToday && $isWithinTimeWindow;
        $roster->can_check_in = $isToday && $isWithinTimeWindow && !$roster->is_checked_in && !$roster->is_completed;
        $roster->can_check_out = $roster->is_checked_in;

        // Status label for easy display
        if ($roster->is_completed) {
            $roster->status = 'completed';
        } elseif ($roster->is_checked_in) {
            $roster->status = 'checked_in';
        } elseif ($roster->schedule_date < $currentDate) {
            $roster->status = 'missed';
        } elseif ($roster->schedule_date > $currentDate) {
            $roster->status = 'upcoming';
        } elseif ($currentTime < $roster->start_time) {
            $roster->status = 'not_started';
        } elseif ($currentTime > $roster->end_time) {
            $roster->status = 'expired';
        } else {
            $roster->status = 'available';
        }

        return $roster;
    });

    return $this->success($rostersWithStatus, 'Roster list retrieved successfully');
}
    public function checkIn(Request $request)
    {
        // dd($request->all());
        $validator = Validator::make($request->all(), [
            'longitude' => 'required|numeric',
            'latitude' => 'required|numeric',
            'date' => 'nullable|date',
            'check_in' => 'nullable|date_format:H:i',
            'site_id' => 'nullable',
            'image' => 'nullable|mimes:jpeg,png,jpg,gif|max:7168',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $currentTime = $request->input('check_in') ? Carbon::parse($request->input('check_in'))->format('H:i:s') : now()->format('H:i:s');
        $currentDate = $request->input('date') ? Carbon::parse($request->input('date'))->format('Y-m-d') : now()->format('Y-m-d');
        $validatedData = $validator->validated();
        $validatedData['date'] = $currentDate;

        // Get employee's active invitation to get customer_id, workspace_id, and subcontractor_id
        $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
            ->where('invitation_status', 'accepted')
            ->where('status', 1)
            ->first();

        if (!$invitation) {
            return $this->error('No active invitation found for this employee.', 404);
        }

        $customerId = $invitation->customer_id;
        $workspaceId = $invitation->workspace_id;
        $subcontractorId = $invitation->subcontractor_id;

        // Check if employee is on leave
        $checkLeave = LeaveRequest::where('employee_id', $employee->id)
            ->where('subcontractor_id', $subcontractorId)
            ->where('status', 1) // Approved leave
            ->where('from', '<=', $currentDate)
            ->where('to', '>=', $currentDate)
            ->exists();

        if ($checkLeave) {
            return $this->error('You are on leave. You are not allowed to check-in.', 422);
        }

        // Check roster assignment for today
        $checkRoster = RosterAssign::join('roster_templates', 'roster_assigns.roster_template_id', 'roster_templates.id')
            ->where('roster_assigns.assign_to', $employee->id)
            ->where('roster_assigns.subcontractor_id', $subcontractorId)
            ->where('roster_assigns.schedule_date', $currentDate)
            ->where('roster_assigns.customer_id', $customerId)
            ->where('roster_assigns.workspace_id', $workspaceId)
            ->where('roster_templates.start_time', '<=', $currentTime)
            ->where('roster_templates.end_time', '>=', $currentTime)
            ->exists();

        if (!$checkRoster) {
            return $this->error('Please clock in according to your scheduled roster time.', 422);
        }

        // Validate location (geofencing)
        $point = [
            'longitude' => (float) $validatedData['longitude'],
            'latitude' => (float) $validatedData['latitude'],
        ];

        $sites = Sites::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('active', 1)
            ->where('del', 0)
            ->get();

        $foundSite = null;
        foreach ($sites as $site) {
            $center = [
                'longitude' => (float) $site->longitude,
                'latitude' => (float) $site->latitude,
            ];
            $radius = (float) $site->area_radius;
            if (GeoLoacation::isPointWithinRadius($point, $center, $radius)) {
                $foundSite = $site;
                break;
            }
        }

        // Get roster assignment to check site
        $rosterAssign = RosterAssign::join('roster_templates', 'roster_assigns.roster_template_id', 'roster_templates.id')
            ->where('roster_assigns.assign_to', $employee->id)
            ->where('roster_assigns.subcontractor_id', $subcontractorId)
            ->where('roster_assigns.schedule_date', $currentDate)
            ->where('roster_assigns.customer_id', $customerId)
            ->where('roster_assigns.workspace_id', $workspaceId)
            ->where('roster_templates.start_time', '<=', $currentTime)
            ->where('roster_templates.end_time', '>=', $currentTime)
            ->where('roster_assigns.site_id', '!=', null)
            ->select('roster_assigns.site_id')
            ->first();

        if (!$foundSite) {
            $assignedSite = Sites::where('id', $rosterAssign->site_id ?? null)->select('title')->first();
            $assignedSiteName = $assignedSite->title ?? 'any';
            return $this->error('The provided location is outside the permitted radius of the ' . $assignedSiteName . ' site.', 422);
        }

        // Check if assigned to correct site
        if ($rosterAssign && $foundSite->id != $rosterAssign->site_id) {
            $assignedSite = Sites::where('id', $rosterAssign->site_id)->select('title')->first();
            $attemptedSiteName = $foundSite->title ?? 'Unknown Site';
            $assignedSiteName = $assignedSite->title ?? 'Unknown Site';
            return $this->error('You are not assigned to this site. You are trying to check in at: ' . $attemptedSiteName . '. Your assigned site is: ' . $assignedSiteName . '.', 422);
        }

        // Check if already checked in
        $previousRecord = EmployeeAttendance::where('employee_id', $employee->id)
            ->where('subcontractor_id', $subcontractorId)
            ->where('date', $currentDate)
            ->whereNull('check_out')
            ->latest()
            ->first();

        if ($previousRecord) {
            return $this->error('You are already checked in at a site. Please check out first before checking in again.', 422);
        }

        // Check for required documents that need to be signed before check-in
        $requiredDocuments = $this->checkRequiredDocumentsForCheckIn(
            $foundSite->id,
            $employee->id,
            $customerId,
            $workspaceId,
            $subcontractorId
        );

        if (!empty($requiredDocuments)) {
            return $this->error([
                'requires_signature' => true,
                'documents' => $requiredDocuments,
                'message' => 'Please sign the required documents before checking in.'
            ], 422);
        }

        // Handle image upload if provided
        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $employee->id . '_' . $image->getClientOriginalName();
            $image->move(public_path('upload/AttendanceLog'), $imageName);
            $imagePath = 'upload/AttendanceLog/' . $imageName;
        }

        // Create attendance record
        $data = [
            'employee_id' => $employee->id,
            'date' => $currentDate,
            'check_in' => $currentTime,
            'longitude' => $validatedData['longitude'],
            'latitude' => $validatedData['latitude'],
            'status' => 1,
            'site_id' => $foundSite->id,
            'added_by' => $employee->id,
            'image' => $imagePath,
            'customer_id' => $customerId,
            'workspace_id' => $workspaceId,
            'subcontractor_id' => $subcontractorId,
        ];

        $checkin = EmployeeAttendance::create($data);
        $checkin->load('sites');

        return $this->success($checkin, 'Check-in Successfully');
    }

    /**
     * Check required documents for check-in (for subcontractor employees)
     */
    private function checkRequiredDocumentsForCheckIn($siteId, $employeeId, $customerId, $workspaceId, $subcontractorId)
    {
        // For subcontractor employees, use default EMP role
        $defaultRole = \App\Models\Role::where('code', 'EMP')->where('del', '0')->first();
        $employeeRoleId = $defaultRole ? $defaultRole->id : null;

        // Get documents that require signature at check-in
        $documents = SiteDocument::where('site_id', $siteId)
            ->where('sign_requires', 1) // Signature required
            ->where('signature_timing', 0) // Must sign at check-in
            ->get();

        $requiredDocuments = [];

        foreach ($documents as $document) {
            // Check if document has role_ids defined
            if (!$document->role_ids || !is_array($document->role_ids) || count($document->role_ids) == 0) {
                continue;
            }

            // Check if employee's role matches any role in document's role_ids
            if (!$employeeRoleId || !in_array($employeeRoleId, $document->role_ids)) {
                continue;
            }

            // Only return documents that haven't been signed yet
            $hasSigned = SiteDocumentSignature::where('site_document_id', $document->id)
                ->where('employee_id', $employeeId)
                ->where('site_id', $siteId)
                ->exists();

            if (!$hasSigned) {
                $requiredDocuments[] = [
                    'id' => $document->id,
                    'title' => $document->title,
                    'file' => $document->file,
                    'document_type' => $document->document_type,
                    'site_id' => $document->site_id,
                ];
            }
        }

        return $requiredDocuments;
    }

    /**
     * Break-in for subcontractor employees
     * Records when a subcontractor employee starts a break
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function breakIn(Request $request)
    {
        $currentTime = now()->format('H:i:s');
        $validator = Validator::make($request->all(), [
            'emp_attendance_id' => 'required|exists:employee_attendances,id',
            'break_in' => 'nullable|date_format:H:i',
            'date' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();
        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $validatedData = $validator->validated();
        $breakInTime = $request->input('break_in') ? Carbon::parse($request->input('break_in'))->format('H:i:s') : $currentTime;

        $currentDate = $request->input('date') ? Carbon::parse($request->input('date'))->format('Y-m-d') : now()->format('Y-m-d');
        $validatedData['date'] = $currentDate;
        // Get attendance record and verify it belongs to this employee
        $attendanceRecord = EmployeeAttendance::where('id', $validatedData['emp_attendance_id'])
            ->where('employee_id', $employee->id)
            ->whereNotNull('subcontractor_id')
            ->first();

        if (!$attendanceRecord) {
            return $this->error('Attendance record not found or unauthorized', 404);
        }

        // Check if employee is checked in (has check_in but no check_out)
        if (!$attendanceRecord->check_in || $attendanceRecord->check_out) {
            return $this->error('You must be checked in to take a break', 422);
        }

        // Check if there's an open break (break_in without break_out)
        $openBreak = EmployeeBreak::where('emp_attendance_id', $validatedData['emp_attendance_id'])
            ->whereNull('break_out')
            ->latest()
            ->first();

        if ($openBreak) {
            return $this->error('You already have an open break. Please complete your current break before starting a new one.', 422);
        }

        $data = [
            'emp_attendance_id' => $validatedData['emp_attendance_id'],
            'break_in' => $breakInTime,
            'date' => Carbon::parse($validatedData['date'])->format('Y-m-d'),
        ];

        $breakIn = EmployeeBreak::create($data);
        return $this->success($breakIn, 'Break In successfully');
    }

    /**
     * Get last break-in for subcontractor employees
     * Returns the last break record, especially if it's still open (no break_out)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function lastBreakIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'emp_attendance_id' => 'required|exists:employee_attendances,id',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();
        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $validatedData = $validator->validated();

        // Verify attendance record belongs to this employee
        $attendanceRecord = EmployeeAttendance::where('id', $validatedData['emp_attendance_id'])
            ->where('employee_id', $employee->id)
            ->whereNotNull('subcontractor_id')
            ->first();

        if (!$attendanceRecord) {
            return $this->error('Attendance record not found or unauthorized', 404);
        }

        $lastBreakIn = EmployeeBreak::where('emp_attendance_id', $validatedData['emp_attendance_id'])
            ->latest()
            ->first();

        if ($lastBreakIn) {
            if (is_null($lastBreakIn->break_out)) {
                return $this->success($lastBreakIn, 'Last open break found.', 200);
            } else {
                return $this->error('No open break found.', 200);
            }
        }

        return $this->error('No break records found for this attendance.', 200);
    }

    /**
     * Break-out for subcontractor employees
     * Records when a subcontractor employee ends a break
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function breakOut(Request $request)
    {
        $currentTime = now()->format('H:i:s');
        $validator = Validator::make($request->all(), [
            'breakin_id' => 'required|exists:employee_breaks,id',
            'break_out' => 'nullable|date_format:H:i',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();
        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $validatedData = $validator->validated();
        $breakOutTime = $request->input('break_out') ? Carbon::parse($request->input('break_out'))->format('H:i:s') : $currentTime;

        $breakRecord = EmployeeBreak::find($validatedData['breakin_id']);

        if (!$breakRecord) {
            return $this->error('Break record not found', 404);
        }

        // Verify the break belongs to this employee's attendance
        $attendanceRecord = $breakRecord->employeeAttendance;
        if (!$attendanceRecord) {
            return $this->error('Associated attendance record not found', 404);
        }

        if ($attendanceRecord->employee_id != $employee->id || !$attendanceRecord->subcontractor_id) {
            return $this->error('Unauthorized access to this record', 403);
        }

        // Check if break_out is already set
        if ($breakRecord->break_out) {
            return $this->error('This break has already been completed', 422);
        }

        // Update the break-out time
        $breakRecord->update([
            'break_out' => $breakOutTime,
        ]);

        return $this->success($breakRecord, 'Break Out successfully');
    }

    /**
     * Check-out for subcontractor employees
     * Records when a subcontractor employee ends their shift
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkout(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'checkin_id' => 'required|exists:employee_attendances,id',
            'check_out' => 'nullable|date_format:H:i',
            'longitude' => 'required|numeric',
            'latitude' => 'required|numeric',
            'date' => 'nullable|date',
            'image' => 'nullable|mimes:jpeg,png,jpg,gif|max:7168',
        ]);
        $currentDate = $request->input('date') ? Carbon::parse($request->input('date'))->format('Y-m-d') : now()->format('Y-m-d');
        $currentTime = $request->input('check_out') ? Carbon::parse($request->input('check_out'))->format('H:i:s') : now()->format('H:i:s');
        $validatedData['date'] = $currentDate;

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();
        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $validatedData = $validator->validated();

        // Get attendance record and verify it belongs to this employee
        $attendanceRecord = EmployeeAttendance::where([
            ['id', $validatedData['checkin_id']],
            ['employee_id', $employee->id],
            ['subcontractor_id', '!=', null]
        ])->first();

        if (!$attendanceRecord) {
            $attendanceRecordWithCheckout = EmployeeAttendance::where('id', $validatedData['checkin_id'])
                ->where('employee_id', $employee->id)
                ->whereNotNull('check_out')
                ->whereNotNull('subcontractor_id')
                ->first();
            
            if ($attendanceRecordWithCheckout) {
                $checkOutTime = $attendanceRecordWithCheckout->check_out;
                $formattedTime = is_string($checkOutTime) ? $checkOutTime : Carbon::parse($checkOutTime)->format('H:i');
                $message = 'Your shift has been auto logout from the system at ' . $formattedTime;
                return $this->error($message, 422);
            } else {
                return $this->error('Check-in required before check-out. Please complete your check-in before proceeding with check-out.', 422);
            }
        }

        // Check if already checked out
        if ($attendanceRecord->check_out) {
            $checkOutTime = $attendanceRecord->check_out;
            $formattedTime = is_string($checkOutTime) ? $checkOutTime : Carbon::parse($checkOutTime)->format('H:i');
            return $this->error('You have already checked out at ' . $formattedTime, 422);
        }

        // Validate location (geofencing)
        $point = [
            'longitude' => (float) $validatedData['longitude'],
            'latitude' => (float) $validatedData['latitude'],
        ];

        $site = Sites::find($attendanceRecord->site_id);
        if (!$site) {
            return $this->error('The site you are trying to check out from is not available now.', 422);
        }

        $center = [
            'longitude' => (float) $site->longitude,
            'latitude' => (float) $site->latitude,
        ];
        $radius = (float) $site->area_radius;

        if (!GeoLoacation::isPointWithinRadius($point, $center, $radius)) {
            return $this->error('The provided location is outside the permitted radius of the ' . $site->title . ' site.', 422);
        }

        // Handle image upload if provided
        $imagePath = $attendanceRecord->image; // Keep existing image if no new one
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $employee->id . '_checkout_' . $image->getClientOriginalName();
            $image->move(public_path('upload/AttendanceLog'), $imageName);
            $imagePath = 'upload/AttendanceLog/' . $imageName;
        }

        // Close any open breaks
        $openBreaks = EmployeeBreak::where('emp_attendance_id', $attendanceRecord->id)
            ->whereNull('break_out')
            ->get();

        foreach ($openBreaks as $break) {
            $break->update(['break_out' => $currentTime]);
        }

        // Calculate working hours
        $checkInTime = Carbon::parse($attendanceRecord->check_in);
        $checkOutTime = Carbon::parse($currentTime);
        $totalMinutes = $checkOutTime->diffInMinutes($checkInTime);

        // Calculate total break time
        $breaks = EmployeeBreak::where('emp_attendance_id', $attendanceRecord->id)->get();
        $totalBreakMinutes = 0;
        foreach ($breaks as $break) {
            if ($break->break_in && $break->break_out) {
                $breakInTime = Carbon::parse($break->break_in);
                $breakOutTime = Carbon::parse($break->break_out);
                $totalBreakMinutes += $breakOutTime->diffInMinutes($breakInTime);
            }
        }

        $workingMinutes = $totalMinutes - $totalBreakMinutes;

        // Update attendance record
        $attendanceRecord->update([
            'check_out' => $currentTime,
            'longitude_out' => $validatedData['longitude'],
            'latitude_out' => $validatedData['latitude'],
            'working_hours' => $workingMinutes,
            'image' => $imagePath,
        ]);

        return $this->success($attendanceRecord, 'Check-out successfully');
    }
    public function last_check_in(Request $request)
    {
        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        // Get employee's active invitation to get subcontractor_id
        $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
            ->where('invitation_status', 'accepted')
            ->where('status', 1)
            ->first();

        if (!$invitation) {
            return $this->error('No active invitation found for this employee.', 404);
        }

        $subcontractorId = $invitation->subcontractor_id;

        // Get the last attendance record for the employee
        $lastCheckin = EmployeeAttendance::with('breaks')->where('employee_id', $employee->id)
            ->where('subcontractor_id', $subcontractorId)
            ->latest()
            ->first();

        if (!$lastCheckin) {
            return $this->message('No attendance record found for this employee.', 200);
        }

        // Return the last attendance record with appropriate message
        if ($lastCheckin->check_in && !$lastCheckin->check_out) {
            return $this->success($lastCheckin, 'Employee is currently checked in');
        } elseif ($lastCheckin->check_in && $lastCheckin->check_out) {
            return $this->success($lastCheckin, 'Last attendance record - employee has checked out');
        } else {
            return $this->success($lastCheckin, 'Last attendance record');
        }
    }
}
