<?php

namespace App\Http\Controllers\MobileV2;

use App\Http\Controllers\Controller;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\SubcontractorEmployeeDocument;
use App\Models\User;
use App\Models\SubcontractorCompany;
use App\Models\Project;
use App\Models\RequiredDocument;
use App\Models\RequiredDocumentField;
use App\Models\Role;
use App\Models\FundsMeta;
use App\Models\EmpDocuments;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;
use App\General\MetaClass;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class SubcontractorEmployeeController extends Controller
{
    /**
     * Get all subcontractors associated with the authenticated employee
     * Returns subcontractors with their projects including customer_id and workspace_id
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSubcontractors(Request $request)
    {
        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Get all associations for this employee with raw timestamps
            $associationsRaw = DB::table('employees_subcontractors_metas')
                ->where('emp_id', $employee->id)
                ->get();

            // Get associations with relationships for other data
            $associations = EmployeeSubcontractorMeta::where('emp_id', $employee->id)
                ->with(['subcontractor'])
                ->get()
                ->keyBy('id');

            // Get all invitations for this employee grouped by customer (for checking if already accepted)
            $invitationsByCustomer = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('invitation_status', 'accepted')
                ->get()
                ->groupBy('customer_id');

            // Get all invitations grouped by subcontractor and project (for specific project status)
            $invitations = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->get()
                ->groupBy('subcontractor_id');

            $subcontractors = $associationsRaw->map(function ($associationRaw) use ($associations, $invitations, $invitationsByCustomer, $employee) {
                // Get the model instance for relationships
                $association = $associations->get($associationRaw->id);
                
                if (!$association) {
                    return null;
                }

                $subcontractor = $association->subcontractor;
                
                if (!$subcontractor) {
                    return null;
                }

                // Get subcontractor company details
                $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
                    ->where('del', '0')
                    ->with('trade')
                    ->first();

                // Get company information
                $companyInfo = null;
                $customerId = null;
                $workspaceId = null;
                
                if ($subcontractorCompany && $subcontractorCompany->customer_id) {
                    $customerId = $subcontractorCompany->customer_id;
                    $workspaceId = $subcontractorCompany->workspace_id ?? null;
                    
                    $company = User::find($subcontractorCompany->customer_id);
                    if ($company) {
                        $companyInfo = [
                            'id' => $company->id,
                            'name' => $company->name,
                            'company_name' => $company->company_name,
                        ];
                    }
                }

                // Parse project_ids from JSON if needed
                $projectIds = [];
                if ($associationRaw->project_ids) {
                    $projectIds = is_string($associationRaw->project_ids) 
                        ? json_decode($associationRaw->project_ids, true) 
                        : $associationRaw->project_ids;
                }

                // Get projects with customer_id and workspace_id
                $projects = [];
                if (!empty($projectIds)) {
                    $projectList = DB::table('projects')
                        ->whereIn('id', $projectIds)
                        ->where('is_deleted', '0')
                        ->get();

                    // Get invitations for this subcontractor
                    $subcontractorInvitations = $invitations->get($subcontractor->id, collect())
                        ->keyBy('project_id');

                    $projects = $projectList->map(function ($project) use ($subcontractorInvitations) {
                        $invitation = $subcontractorInvitations->get($project->id);
                        
                        // Only mark as accepted if there's an explicit invitation record for this project
                        $invitationStatus = 'pending';
                        $invitationId = null;
                        $requiredDocsStatus = 0; // Default: 0 (not uploaded)
                        $inductionStatus = 0; // Default: 0 (not signed)
                        // Default labels for default status 0
                        $requiredDocsStatusLabel = 'not_uploaded'; 
                        $inductionStatusLabel = 'not_signed';
                        
                        if ($invitation) {
                            $invitationStatus = $invitation->invitation_status;
                            $invitationId = $invitation->id;
                            $requiredDocsStatus = $invitation->required_docs_status;
                            $inductionStatus = $invitation->induction_status;
                        }
                        
                        return [
                            'id' => $project->id,
                            'title' => $project->title,
                            'description' => $project->description,
                            'address' => $project->address,
                            'state' => $project->state,
                            'start_date' => $project->start_date,
                            'end_date' => $project->end_date,
                            'thumbnail' => $project->thumbnail ? url($project->thumbnail) : null,
                            'customer_id' => $project->customer_id,
                            'workspace_id' => $project->workspace_id,
                            'invitation_status' => $invitationStatus,
                            'invitation_id' => $invitationId,
                            'required_docs_status' => $requiredDocsStatus,
                            'induction_status' => $inductionStatus,
                            'required_docs_status_label' => $invitation ? $invitation->required_docs_status_label : 'not_uploaded',
                            'induction_status_label' => $invitation ? $invitation->induction_status_label : 'not_signed',
                        ];
                    })->values()->toArray();
                }

                return [
                    'id' => $subcontractor->id,
                    'name' => $subcontractor->name,
                    'company_name' => $subcontractor->company_name,
                    'email' => $subcontractor->email,
                    'mobile_number' => $subcontractor->mobile_number,
                    'abn' => $subcontractor->abn,
                    'company_logo' => $subcontractor->company_logo ? url($subcontractor->company_logo) : null,
                    'trade_id' => $subcontractorCompany ? $subcontractorCompany->trade_id : null,
                    'trade_name' => $subcontractorCompany && $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null,
                    'company' => $companyInfo,
                    // 'customer_id' => $customerId,
                    // 'workspace_id' => $workspaceId,
                    'association_id' => $associationRaw->id,
                    'projects' => $projects,
                    'invited_at' => $associationRaw->created_at,
                ];
            })->filter()->values()->toArray();

            return $this->success($subcontractors, 'Subcontractors retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while retrieving subcontractors: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Accept or respond to a project invitation
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function respondToInvitation(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'subcontractor_id' => 'required|integer|exists:users,id',
            'project_id' => 'required|integer|exists:projects,id',
            'status' => 'required|in:accepted,rejected',
            'rejection_reason' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Verify subcontractor exists and is valid
            $subcontractor = User::where('id', $request->subcontractor_id)
                ->where('user_type', config('constants.user_types.subcontractor'))
                ->where('del', '0')
                ->first();

            if (!$subcontractor) {
                return $this->error('Invalid subcontractor', 404);
            }

            // Verify project exists
            $project = DB::table('projects')
                ->where('id', $request->project_id)
                ->where('is_deleted', '0')
                ->first();

            if (!$project) {
                return $this->error('Invalid project', 404);
            }

            // Verify the employee is associated with this subcontractor and project
            $association = DB::table('employees_subcontractors_metas')
                ->where('emp_id', $employee->id)
                ->where('subcontractor_id', $request->subcontractor_id)
                ->first();

            if (!$association) {
                return $this->error('You are not associated with this subcontractor', 404);
            }

            // Check if project is in the association's project_ids
            $projectIds = [];
            if ($association->project_ids) {
                $projectIds = is_string($association->project_ids) 
                    ? json_decode($association->project_ids, true) 
                    : $association->project_ids;
            }

            if (!in_array($request->project_id, $projectIds)) {
                return $this->error('You are not assigned to this project', 404);
            }

            // Get customer_id from project
            $customerId = $project->customer_id;
            $workspaceId = $project->workspace_id;

            // Check if there are any existing invitations for this employee and customer with non-zero statuses
            // We want to reuse these statuses if they exist (sync across projects)
            $existingStatus = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where(function($query) {
                    $query->where('required_docs_status', '!=', 0)
                        ->orWhere('induction_status', '!=', 0);
                })
                ->select('required_docs_status', 'induction_status')
                ->orderBy('updated_at', 'desc') // Get the most recent status update
                ->first();

            $docsStatus = $existingStatus ? $existingStatus->required_docs_status : 0;
            $inductionStatus = $existingStatus ? $existingStatus->induction_status : 0;

            // Find or create invitation record for this specific project
            $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('subcontractor_id', $request->subcontractor_id)
                ->where('project_id', $request->project_id)
                ->first();

            if ($invitation) {
                // Check if already responded
                if ($invitation->invitation_status !== 'pending') {
                    return $this->error('Invitation already responded to', 400);
                }

                // Update existing invitation
                $updateData = [
                    'invitation_status' => $request->status,
                    'rejection_reason' => $request->status === 'rejected' ? $request->rejection_reason : null,
                    'responded_at' => now(),
                ];

                // Apply existing statuses if improved (or just sync them)
                // If we found a status from another project, we trust it's the current "customer-level" status
                if ($existingStatus) {
                    $updateData['required_docs_status'] = $docsStatus;
                    $updateData['induction_status'] = $inductionStatus;
                }

                $invitation->update($updateData);
            } else {
                // Create new invitation record with response
                $createData = [
                    'employee_id' => $employee->id,
                    'subcontractor_id' => $request->subcontractor_id,
                    'project_id' => $request->project_id,
                    'customer_id' => $customerId,
                    'workspace_id' => $workspaceId,
                    'invitation_status' => $request->status,
                    'rejection_reason' => $request->status === 'rejected' ? $request->rejection_reason : null,
                    'responded_at' => now(),
                    'required_docs_status' => $docsStatus,
                    'induction_status' => $inductionStatus,
                ];

                $invitation = SubcontractorEmployeeInvitation::create($createData);
            }

            // If accepting, create accepted invitations for all other projects from this customer
            if ($request->status === 'accepted') {
                // Get all projects for this employee from this customer
                $allCustomerProjects = DB::table('employees_subcontractors_metas')
                    ->where('emp_id', $employee->id)
                    ->where('subcontractor_id', $request->subcontractor_id)
                    ->get();

                $allProjectIds = [];
                foreach ($allCustomerProjects as $assoc) {
                    $projIds = $assoc->project_ids ? 
                        (is_string($assoc->project_ids) ? json_decode($assoc->project_ids, true) : $assoc->project_ids) 
                        : [];
                    $allProjectIds = array_merge($allProjectIds, $projIds);
                }

                // Get all projects from this customer
                $customerProjects = DB::table('projects')
                    ->whereIn('id', $allProjectIds)
                    ->where('customer_id', $customerId)
                    ->where('is_deleted', '0')
                    ->pluck('id')
                    ->toArray();

                // Create accepted invitations for all customer projects
                foreach ($customerProjects as $projId) {
                    if ($projId != $request->project_id) {
                        SubcontractorEmployeeInvitation::updateOrCreate(
                            [
                                'employee_id' => $employee->id,
                                'subcontractor_id' => $request->subcontractor_id,
                                'project_id' => $projId,
                            ],
                            [
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                                'invitation_status' => 'accepted',
                                'rejection_reason' => null,
                                'responded_at' => now(),
                                'required_docs_status' => $docsStatus,
                                'induction_status' => $inductionStatus,
                            ]
                        );
                    }
                }
            }

            $message = $request->status === 'accepted' 
                ? 'Invitation accepted successfully. All projects from this customer are now accepted.' 
                : 'Invitation rejected successfully';

            return $this->success([
                'invitation_id' => $invitation->id,
                'status' => $invitation->invitation_status,
                'responded_at' => $invitation->responded_at,
            ], $message, 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while responding to invitation: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get required documents for a project by customer
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRequiredDocuments(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer|exists:users,id',
            'workspace_id' => 'required|integer',
            'project_id' => 'required|integer|exists:projects,id',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Get required documents for external/subcontractor employees
            $requiredDocuments = RequiredDocument::with(['requiredDocumentField' => function ($query) {
                    $query->where('status', 1)->where('del', 0);
                }])
                ->where('del', '0')
                ->where('customer_id', $request->customer_id)
                ->where('workspace_id', $request->workspace_id)
                ->where(function ($q) {
                    $q->where('for_external', true)
                        ->orWhere('for_subcontractor', true);
                })
                ->get();

            // Get already submitted documents for this employee and customer (any project)
            // Once approved for a customer, documents are considered approved for all projects
            $submittedDocuments = SubcontractorEmployeeDocument::where('employee_id', $employee->id)
                ->where('customer_id', $request->customer_id)
                ->where('workspace_id', $request->workspace_id)
                ->where('del', '0')
                ->with(['approver'])
                ->get()
                ->groupBy('required_document_id');

            $data = $requiredDocuments->map(function ($document) use ($submittedDocuments) {
                $documentSubmissions = $submittedDocuments->get($document->id, collect());

                // Calculate document-level approval status
                $docApprovalStatus = null;
                $docRejectionReason = null;
                $docApprovedBy = null;
                $docApprovedAt = null;
                
                if ($documentSubmissions->isNotEmpty()) {
                    $allApproved = $documentSubmissions->every(fn($s) => $s->approval_status === 'approved');
                    $anyRejected = $documentSubmissions->contains(fn($s) => $s->approval_status === 'rejected');
                    
                    if ($allApproved) {
                        $docApprovalStatus = 'approved';
                        $firstApproved = $documentSubmissions->first();
                        $docApprovedBy = $firstApproved->approver ? $firstApproved->approver->name : null;
                        $docApprovedAt = $firstApproved->approved_at;
                    } elseif ($anyRejected) {
                        $docApprovalStatus = 'rejected';
                        $rejectedDoc = $documentSubmissions->firstWhere('approval_status', 'rejected');
                        $docRejectionReason = $rejectedDoc->rejection_reason;
                        $docApprovedBy = $rejectedDoc->approver ? $rejectedDoc->approver->name : null;
                        $docApprovedAt = $rejectedDoc->approved_at;
                    } else {
                        $docApprovalStatus = 'pending';
                    }
                }

                return [
                    'id' => $document->id,
                    'title' => $document->title,
                    'identity_text' => $document->identity_text,
                    'identity_file' => $document->identity_file,
                    'priority' => $document->priority,
                    'approval_status' => $docApprovalStatus,
                    'rejection_reason' => $docRejectionReason,
                    'approved_by' => $docApprovedBy,
                    'approved_at' => $docApprovedAt,
                    'fields' => $document->requiredDocumentField->map(function ($field) use ($documentSubmissions) {
                        $submission = $documentSubmissions->firstWhere('required_document_field_id', $field->id);
                        
                        return [
                            'id' => $field->id,
                            'field_name' => $field->field_name,
                            'field_type' => $field->field_type,
                            'field_required' => $field->field_required,
                            'priority' => $field->priority,
                            'submitted_value' => $submission ? $submission->value : null,
                            'submission_id' => $submission ? $submission->id : null,
                        ];
                    })->values()->toArray(),
                ];
            })->values()->toArray();

            return $this->success($data, 'Required documents retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while retrieving required documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Store/submit documents for the subcontractor employee
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function storeDocuments(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'subcontractor_id' => 'required|integer|exists:users,id',
            'project_id' => 'required|integer|exists:projects,id',
            'customer_id' => 'required|integer|exists:users,id',
            'workspace_id' => 'required|integer',
            'documents' => 'required|array',
            'documents.*.required_document_id' => 'required|integer|exists:required_documents,id',
            'documents.*.required_document_field_id' => 'required|integer|exists:required_document_fields,id',
            'documents.*.value' => 'nullable|string',
            'documents.*.file' => 'nullable|file|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Verify the employee is associated with this subcontractor and project
            $association = DB::table('employees_subcontractors_metas')
                ->where('emp_id', $employee->id)
                ->where('subcontractor_id', $request->subcontractor_id)
                ->first();

            if (!$association) {
                return $this->error('You are not associated with this subcontractor', 404);
            }

            // Ensure upload directory exists
            $uploadDirectory = public_path('upload/subcontractor_employee_documents');
            if (!file_exists($uploadDirectory)) {
                mkdir($uploadDirectory, 0755, true);
            }

            $savedDocuments = [];
            $i = 0;

            foreach ($request->documents as $index => $documentData) {
                $value = null;

                // Handle file upload
                if ($request->hasFile("documents.{$index}.file")) {
                    $file = $request->file("documents.{$index}.file");
                    if ($file && $file->isValid()) {
                        $fileName = $employee->id . '_' . $i . '_' . time() . '.' . $file->getClientOriginalExtension();
                        $file->move($uploadDirectory, $fileName);
                        $value = 'upload/subcontractor_employee_documents/' . $fileName;
                    }
                } elseif (isset($documentData['value'])) {
                    $value = $documentData['value'];
                }

                // Check if document is already approved for this customer (any project)
                $approvedDocument = SubcontractorEmployeeDocument::where('employee_id', $employee->id)
                    ->where('customer_id', $request->customer_id)
                    ->where('workspace_id', $request->workspace_id)
                    ->where('required_document_id', $documentData['required_document_id'])
                    ->where('required_document_field_id', $documentData['required_document_field_id'])
                    ->where('approval_status', 'approved')
                    ->where('del', '0')
                    ->first();

                // Find existing document for this specific project
                $existingDocument = SubcontractorEmployeeDocument::where('employee_id', $employee->id)
                    ->where('subcontractor_id', $request->subcontractor_id)
                    ->where('project_id', $request->project_id)
                    ->where('customer_id', $request->customer_id)
                    ->where('workspace_id', $request->workspace_id)
                    ->where('required_document_id', $documentData['required_document_id'])
                    ->where('required_document_field_id', $documentData['required_document_field_id'])
                    ->where('del', '0')
                    ->first();

                if ($existingDocument) {
                    // If already approved for this customer, keep approval status
                    $approvalStatus = $approvedDocument ? 'approved' : 'pending';
                    
                    // Update existing document
                    $existingDocument->update([
                        'value' => $value ?: $existingDocument->value, // Keep existing value if new value is empty
                        'approval_status' => $approvalStatus,
                        'rejection_reason' => null,
                        'approved_by' => $approvedDocument ? $approvedDocument->approved_by : null,
                        'approved_at' => $approvedDocument ? $approvedDocument->approved_at : null,
                    ]);
                    $savedDocuments[] = $existingDocument;
                } else {
                    // Create new document - if already approved for customer, mark as approved
                    $approvalStatus = $approvedDocument ? 'approved' : 'pending';
                    
                    $newDocument = SubcontractorEmployeeDocument::create([
                        'employee_id' => $employee->id,
                        'subcontractor_id' => $request->subcontractor_id,
                        'project_id' => $request->project_id,
                        'customer_id' => $request->customer_id,
                        'workspace_id' => $request->workspace_id,
                        'required_document_id' => $documentData['required_document_id'],
                        'required_document_field_id' => $documentData['required_document_field_id'],
                        'value' => $value,
                        'approval_status' => $approvalStatus,
                        'approved_by' => $approvedDocument ? $approvedDocument->approved_by : null,
                        'approved_at' => $approvedDocument ? $approvedDocument->approved_at : null,
                    ]);
                    $savedDocuments[] = $newDocument;
                }

                $i++;
            }

            // Update invitation status to pending if documents were saved
            if (!empty($savedDocuments)) {
                // Check current status first to avoid overwriting if already complete/approved
                // actually, for submission we should probably respect the workflow.
                // If they submit a document, it might need re-approval.
                // Based on SubcontractorCompany model logic (status=4 is pending)
                
                SubcontractorEmployeeInvitation::updateEmployeeDocsStatus(
                    $employee->id,
                    $request->customer_id,
                    $request->workspace_id,
                    SubcontractorEmployeeInvitation::DOCS_STATUS_PENDING
                );
            }

            return $this->success([
                'documents_saved' => count($savedDocuments),
                'documents' => collect($savedDocuments)->map(function ($doc) {
                    // Only apply url() to file paths, not to text/date values
                    $value = null;
                    if ($doc->value) {
                        // Check if value is a file path (starts with 'upload/')
                        if (strpos($doc->value, 'upload/') === 0) {
                            $value = url($doc->value);
                        } else {
                            // Return text/date values as-is
                            $value = $doc->value;
                        }
                    }
                    
                    return [
                        'id' => $doc->id,
                        'required_document_id' => $doc->required_document_id,
                        'required_document_field_id' => $doc->required_document_field_id,
                        'value' => $value,
                        'approval_status' => $doc->approval_status,
                    ];
                })->values()->toArray(),
            ], 'Documents submitted successfully', 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while storing documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get subcontractor details with assigned projects
     * 
     * @param Request $request
     * @param int $subcontractorId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSubcontractorDetails(Request $request, $subcontractorId)
    {
        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Verify the association exists using raw query to get original timestamps
            $associationRaw = DB::table('employees_subcontractors_metas')
                ->where('emp_id', $employee->id)
                ->where('subcontractor_id', $subcontractorId)
                ->first();

            if (!$associationRaw) {
                return $this->error('Subcontractor not found or not associated with you', 404);
            }

            // Get subcontractor details
            $subcontractor = User::where('id', $subcontractorId)
                ->where('user_type', config('constants.user_types.subcontractor'))
                ->where('del', '0')
                ->with('contactPersons')
                ->first();

            if (!$subcontractor) {
                return $this->error('Subcontractor not found', 404);
            }

            // Get subcontractor company details
            $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
                ->where('del', '0')
                ->with('trade')
                ->first();

            // Get company information
            $companyInfo = null;
            $customerId = null;
            $workspaceId = null;
            
            if ($subcontractorCompany && $subcontractorCompany->customer_id) {
                $customerId = $subcontractorCompany->customer_id;
                $workspaceId = $subcontractorCompany->workspace_id ?? null;
                
                $company = User::find($subcontractorCompany->customer_id);
                if ($company) {
                    $companyInfo = [
                        'id' => $company->id,
                        'name' => $company->name,
                        'company_name' => $company->company_name,
                        'email' => $company->email,
                    ];
                }
            }

            // Get projects assigned to this employee by this subcontractor
            $projectIds = [];
            if ($associationRaw->project_ids) {
                $projectIds = is_string($associationRaw->project_ids) 
                    ? json_decode($associationRaw->project_ids, true) 
                    : $associationRaw->project_ids;
            }

            // Get invitations for this subcontractor's projects
            $invitations = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('subcontractor_id', $subcontractorId)
                ->get()
                ->keyBy('project_id');

            // Check if invitation is already accepted for this customer (any project)
            $isCustomerAccepted = $customerId && SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('customer_id', $customerId)
                ->where('invitation_status', 'accepted')
                ->exists();

            $projects = [];
            if (!empty($projectIds)) {
                $projectList = DB::table('projects')
                    ->whereIn('id', $projectIds)
                    ->where('is_deleted', '0')
                    ->get();

                $projects = $projectList->map(function ($project) use ($invitations, $isCustomerAccepted, $customerId) {
                    $invitation = $invitations->get($project->id);
                    
                    // If customer already accepted for any project, mark all projects as accepted
                    // But only if this project belongs to the same customer
                    $invitationStatus = 'pending';
                    if ($isCustomerAccepted && $project->customer_id == $customerId) {
                        $invitationStatus = 'accepted';
                    } elseif ($invitation) {
                        $invitationStatus = $invitation->invitation_status;
                    }
                    
                    return [
                        'id' => $project->id,
                        'title' => $project->title,
                        'description' => $project->description,
                        'address' => $project->address,
                        'state' => $project->state,
                        'start_date' => $project->start_date,
                        'end_date' => $project->end_date,
                        'thumbnail' => $project->thumbnail ? url($project->thumbnail) : null,
                        'customer_id' => $project->customer_id,
                        'workspace_id' => $project->workspace_id,
                        'invitation_status' => $invitationStatus,
                        'invitation_id' => $invitation ? $invitation->id : null,
                        'responded_at' => $invitation ? $invitation->responded_at : null,
                    ];
                })->values()->toArray();
            }

            // Prepare response data
            $data = [
                'id' => $subcontractor->id,
                'name' => $subcontractor->name,
                'company_name' => $subcontractor->company_name,
                'email' => $subcontractor->email,
                'mobile_number' => $subcontractor->mobile_number,
                'abn' => $subcontractor->abn,
                'company_logo' => $subcontractor->company_logo ? url($subcontractor->company_logo) : null,
                'trade_id' => $subcontractorCompany ? $subcontractorCompany->trade_id : null,
                'trade_name' => $subcontractorCompany && $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null,
                'company' => $companyInfo,
                'customer_id' => $customerId,
                'workspace_id' => $workspaceId,
                'contact_persons' => $subcontractor->contactPersons ? $subcontractor->contactPersons->map(function ($person) {
                    return [
                        'id' => $person->id,
                        'name' => $person->name,
                        'email' => $person->email,
                        'phone' => $person->phone,
                        'role' => $person->role,
                    ];
                })->values()->toArray() : [],
                'invited_at' => $associationRaw->created_at,
                'projects' => $projects,
            ];

            return $this->success($data, 'Subcontractor details retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while retrieving subcontractor details: ' . $e->getMessage(), 500);
        }
    }
        /**
     * Get detailed information about a subcontractor employee
     * For customer use
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getEmployeeDetails(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:employees_subcontractors,id',
        ], [
            'employee_id.required' => 'Employee ID is required.',
            'employee_id.exists' => 'Employee not found.',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        try {
            $user = $request->user();
            
            if (!$user) {
                return $this->error('Unauthorized access', 401);
            }

            $customerId = $user->id;
            $workspaceId = $user->current_workspace_id ?? null;
            $employeeId = $request->employee_id;

            // Verify the employee has accepted invitation for this customer
            $invitations = SubcontractorEmployeeInvitation::where('employee_id', $employeeId)
                ->where('customer_id', $customerId)
                ->when($workspaceId, function($query) use ($workspaceId) {
                    $query->where('workspace_id', $workspaceId);
                })
                ->where('invitation_status', 'accepted')
                ->first();

            if (!$invitations) {
                return $this->error('You do not have access to this employee. The employee has not accepted any invitation for this company.', 403);
            }

            // Get employee details
            $employee = EmployeeSubcontractor::find($employeeId);
            if (!$employee) {
                return $this->error('Employee not found', 404);
            }

            // Get status from invitation
            $employeeStatus = (string)($invitations->status ?? 1);
            $requiredDocsStatus = $invitations ? $invitations->required_docs_status : 0;
            $inductionStatus = $invitations ? $invitations->induction_status : 0;
            $requiredDocsStatusLabel = $invitations ? $invitations->required_docs_status_label : 'not_uploaded';
            $inductionStatusLabel = $invitations ? $invitations->induction_status_label : 'not_signed';

            // Build profile object matching regular employee structure
            $profile = [
                'id' => $employee->id,
                'employee_email' => $employee->email,
                'employment_type' => null,
                'work_permit_type' => null,
                'tax_file_no' => null,
                'tier_id' => null,
                'employment_start_date' => null,
                'employment_end_date' => null,
                'attendance_effective_from' => null,
                'status' => $employeeStatus,
                'required_docs_status' => $requiredDocsStatus,
                'induction_status' => $inductionStatus,
                'required_docs_status_label' => $requiredDocsStatusLabel,
                'induction_status_label' => $inductionStatusLabel,
                'invited' => '1',
                'approved' => '1',
                'rejected' => '0',
                'approved_by' => null,
                'approved_status' => null,
                'approved_at' => null,
                'is_submitted' => '0',
                'compeleted' => '0',
                'credentials' => '0',
                'created_by' => (string)$customerId,
                'del' => '0',
                'created_at' => $employee->created_at ? $employee->created_at->toDateTimeString() : now()->toDateTimeString(),
                'updated_at' => $employee->updated_at ? $employee->updated_at->toDateTimeString() : null,
                'two_factor' => 0,
                'force_reset' => $employee->force_reset ?? 0,
                'user_type' => 1, // External/subcontractor employee
                'job_title' => null,
                'trade_qualified' => $employee->trade_qualified ?? null,
                'trade_qualified_year' => $employee->trade_qualified_year ?? null,
                'trade_licensed' => $employee->trade_licensed ?? null,
                'trade_licensed_year' => $employee->trade_licensed_year ?? null,
                'work_experience' => $employee->work_experience ?? null,
                'temporary_student_visa' => null,
                'worker_type' => null,
                'year_commenced' => $employee->year_commenced ?? null,
                'classified_high_risk' => null,
                'citizenship_status' => $employee->citizenship_status ?? null,
                'citizenship_file' => null,
                'allergies' => $employee->allergies ?? null,
                'previous_injuries' => $employee->previous_injuries ?? null,
                'medical_condition' => $employee->medical_condition ?? null,
                'employer_name' => null,
                'legally_australia' => null,
                'link_key' => null,
                'xero_emp_id' => $employee->xero_emp_id ?? null,
                'signature' => $employee->signature ?? null,
                'details_allergies' => $employee->allergies ?? null,
                'details_previous_injuries' => $employee->previous_injuries ?? null,
            ];

            // Add empPersonalDetails (matching regular employee structure)
            $profile['empPersonalDetails'] = [
                'id' => null,
                'emp_id' => (string)$employee->id,
                'first_name' => $employee->first_name,
                'middle_name' => $employee->middle_name,
                'last_name' => $employee->last_name,
                'streat_address' => $employee->address,
                'suburb' => $employee->suburb,
                'state' => $employee->state,
                'mobile' => $employee->mobile,
                'phone' => $employee->phone,
                'dob' => $employee->dob,
                'image' => $employee->profile_image,
                'del' => '0',
                'created_at' => $employee->created_at,
                'updated_at' => $employee->updated_at,
            ];

            $profile['emp_personal_details'] = $profile['empPersonalDetails'];
            // Add empEmergencyContacts (empty for subcontractor employees)
            $profile['empEmergencyContacts'] = [];

            // Add empAccess (null for subcontractor employees)
            $profile['empAccess'] = null;

            // Add empworkExperience (null for subcontractor employees)
            $profile['empworkExperience'] = null;

            // Add workerType (null for subcontractor employees)
            $profile['workerType'] = null;

            // Add empType (null for subcontractor employees)
            $profile['empType'] = null;

            // Add empWorkPermit (null for subcontractor employees)
            $profile['empWorkPermit'] = null;

            // Add empAccessRole (matching regular employee)
            $defaultRole = Role::where('code', 'EMP')->where('del', '0')->first();
            $profile['empAccessRole'] = $defaultRole ? [
                'id' => $defaultRole->id,
                'title' => $defaultRole->title,
                'code' => $defaultRole->code,
            ] : null;

            // Build data array matching profile method structure
            $data = [];
            $data['profile'] = $profile;

            // Funds meta with fund relationship
            $data['funds_meta'] = FundsMeta::with('fund')
                ->where('emp_id', $employeeId)
                ->get();

            // Medical files
            $data['medicalAttachFiles'] = DB::table('medical_attach_files')
                ->where('emp_id', $employeeId)
                ->get();

            // Meta
            $user_meta = DB::table('user_meta')
                ->where('emp_id', $employeeId)
                ->get();

            $employee_payroll_or_external_id = MetaClass::getOptionValue('employee_payroll_or_external_id', $user_meta);
            $data['meta']['employee_payroll_or_external_id'] = $employee_payroll_or_external_id;

            // Get required documents for external/subcontractor employees
            $requiredDocuments = RequiredDocument::with(['requiredDocumentField' => function ($query) {
                    $query->where('status', 1)->where('del', 0);
                }])
                ->where('del', '0')
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where(function ($q) {
                    $q->where('for_external', true)
                        ->orWhere('for_subcontractor', true);
                })
                ->get();

            // Get submitted documents for this employee
            $submittedDocuments = SubcontractorEmployeeDocument::where('employee_id', $employeeId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->get()
                ->groupBy('required_document_field_id');

            // Build required documents response
            $data['required_documents'] = $requiredDocuments->map(function ($document) use ($submittedDocuments) {
                // Format dates safely
                $createdAt = null;
                if ($document->created_at) {
                    $createdAt = is_string($document->created_at) 
                        ? \Carbon\Carbon::parse($document->created_at)->format('d-m-Y')
                        : $document->created_at->format('d-m-Y');
                }
                
                $updatedAt = null;
                if ($document->updated_at) {
                    $updatedAt = is_string($document->updated_at) 
                        ? \Carbon\Carbon::parse($document->updated_at)->format('d-m-Y')
                        : $document->updated_at->format('d-m-Y');
                }
                
                return [
                    'id' => $document->id,
                    'title' => $document->title,
                    'customer_id' => $document->customer_id,
                    'workspace_id' => $document->workspace_id,
                    'priority' => $document->priority,
                    'del' => $document->del,
                    'created_at' => $createdAt,
                    'updated_at' => $updatedAt,
                    'for_internal' => $document->for_internal ?? false,
                    'for_external' => $document->for_external ?? false,
                    'for_subcontractor' => $document->for_subcontractor ?? false,
                    'required_document_field' => $document->requiredDocumentField->map(function ($field) use ($submittedDocuments) {
                        $submissionCollection = $submittedDocuments->get($field->id);
                        $submission = $submissionCollection ? $submissionCollection->first() : null;
                        $hasData = $submission ? 1 : 0;
                        
                        // Format field dates safely
                        $fieldCreatedAt = null;
                        if ($field->created_at) {
                            $fieldCreatedAt = is_string($field->created_at) 
                                ? \Carbon\Carbon::parse($field->created_at)->format('d-m-Y')
                                : $field->created_at->format('d-m-Y');
                        }
                        
                        $fieldUpdatedAt = null;
                        if ($field->updated_at) {
                            $fieldUpdatedAt = is_string($field->updated_at) 
                                ? \Carbon\Carbon::parse($field->updated_at)->format('d-m-Y')
                                : $field->updated_at->format('d-m-Y');
                        }
                        
                        return [
                            'id' => $field->id,
                            'doc_id' => (string)$field->doc_id,
                            'field_type' => $field->field_type,
                            'field_name' => $field->field_name,
                            'priority' => $field->priority,
                            'del' => $field->del,
                            'created_at' => $fieldCreatedAt,
                            'updated_at' => $fieldUpdatedAt,
                            'field_required' => $field->field_required,
                            'status' => $field->status,
                            'hasData' => $hasData,
                        ];
                    })->values()->toArray(),
                ];
            })->values()->toArray();

            // Get induction documents
            $allDocuments = InductionDocument::where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->where('is_active', true)
                ->orderBy('title')
                ->orderBy('version', 'desc')
                ->get();

            // Group by title and get only the latest version for each document
            $documents = [];
            foreach ($allDocuments as $doc) {
                $title = $doc->title;
                if (!isset($documents[$title])) {
                    $documents[$title] = $doc;
                }
            }
            $documents = array_values($documents);

            $signedDocuments = [];
            $notSignedDocuments = [];
            $validSignatures = [];
            $invalidSignatures = [];

            foreach ($documents as $doc) {
                $roleTypes = $doc->role_types ?? ['all'];

                // Check if employee matches role types (external, subcontractor_employee, or all)
                if (!in_array('all', $roleTypes) 
                    && !in_array('external', $roleTypes)
                    && !in_array('subcontractor_employee', $roleTypes)
                    && !in_array('subcontractor', $roleTypes)) {
                    continue; // Skip if doesn't match role types
                }

                // Check for all signatures (both valid and invalid) for this document version
                $allSignatures = InductionDocumentSignature::where('induction_document_id', $doc->id)
                    ->where('document_version', $doc->version)
                    ->where('employee_id', $employeeId)
                    ->where('del', '0')
                    ->get();

                // Get valid signature
                $validSignature = $allSignatures->where('is_valid', true)->first();

                // Get invalid signatures
                $invalidSignature = $allSignatures->where('is_valid', false)->first();

                // Format created_at safely
                $docCreatedAt = null;
                if ($doc->created_at) {
                    $docCreatedAt = is_string($doc->created_at) 
                        ? \Carbon\Carbon::parse($doc->created_at)->format('d-m-Y')
                        : $doc->created_at->format('d-m-Y');
                }

                $documentData = [
                    'id' => $doc->id,
                    'title' => $doc->title,
                    'document_type' => $doc->document_type,
                    'version' => $doc->version,
                    'file_path' => $doc->file_path,
                    'description' => $doc->description,
                    'role_types' => $doc->role_types ?? [],
                    'created_at' => $docCreatedAt,
                ];

                if ($validSignature) {
                    $signedDocuments[] = $documentData;
                    $validSignatures[] = $documentData;
                } elseif ($invalidSignature) {
                    // Document has invalid signature
                    $invalidSignatures[] = $documentData;
                    // Also add to not_signed since it's not valid
                    $notSignedDocuments[] = $documentData;
                } else {
                    // No signature at all
                    $notSignedDocuments[] = $documentData;
                }
            }

            $totalValidSignatures = count($validSignatures);
            $totalInvalidSignatures = count($invalidSignatures);
            $totalSigned = count($signedDocuments);
            $totalNotSigned = count($notSignedDocuments);
            $totalDocuments = $totalSigned + $totalNotSigned;

            $data['induction_documents'] = [
                'signed' => $signedDocuments,
                'not_signed' => $notSignedDocuments,
                'valid_signatures' => $validSignatures,
                'invalid_signatures' => $invalidSignatures,
                'total_signed' => $totalSigned,
                'total_not_signed' => $totalNotSigned,
                'total_documents' => $totalDocuments,
                'total_valid_signatures' => $totalValidSignatures,
                'total_invalid_signatures' => $totalInvalidSignatures,
                'all_signatures_valid' => $totalInvalidSignatures === 0 && $totalValidSignatures > 0,
                'has_invalid_signatures' => $totalInvalidSignatures > 0,
                'all_documents_signed' => $totalNotSigned === 0,
            ];

            return $this->success($data, 'Get Profile Successfully');
        } catch (\Exception $e) {
            return $this->error('An error occurred while retrieving employee details: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get induction documents for a customer/workspace
     * Similar to getRequiredDocuments but for induction documents
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getInductionDocuments(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer|exists:users,id',
            'workspace_id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Verify the employee has accepted invitation for this customer
            $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('customer_id', $request->customer_id)
                ->where('workspace_id', $request->workspace_id)
                ->where('invitation_status', 'accepted')
                ->first();

            if (!$invitation) {
                return $this->error('You have not accepted any invitation for this company.', 403);
            }

            // Get all active induction documents for this customer/workspace
            $documents = InductionDocument::where('customer_id', $request->customer_id)
                ->where('workspace_id', $request->workspace_id)
                ->where('del', '0')
                ->where('is_active', true)
                ->get();

            // Filter documents that include 'subcontractor_employee' or 'external' or 'all' in role_types
            $filteredDocuments = $documents->filter(function ($doc) {
                $roleTypes = $doc->role_types ?? [];
                return in_array('all', $roleTypes) 
                    || in_array('subcontractor_employee', $roleTypes)
                    || in_array('external', $roleTypes);
            });

            // Get signature status for each document
            $customerId = $request->customer_id;
            $workspaceId = $request->workspace_id;
            $documentsWithStatus = $filteredDocuments->map(function ($doc) use ($employee, $customerId, $workspaceId) {
                // Find the root document (traverse up using parent_document_id)
                $rootDocumentId = $doc->id;
                $currentDoc = $doc;
                while ($currentDoc && $currentDoc->parent_document_id) {
                    $rootDocumentId = $currentDoc->parent_document_id;
                    $currentDoc = InductionDocument::where('id', $rootDocumentId)
                        ->where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('del', '0')
                        ->first();
                    if (!$currentDoc) {
                        break;
                    }
                }

                // Get all document IDs in the chain (root and all children) - get all versions first
                $allDocumentIdsInChain = InductionDocument::where(function($q) use ($rootDocumentId) {
                        $q->where('id', $rootDocumentId)
                          ->orWhere('parent_document_id', $rootDocumentId);
                    })
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->pluck('id')
                    ->toArray();
                
                // If no chain found, use the document itself
                if (empty($allDocumentIdsInChain)) {
                    $allDocumentIdsInChain = [$doc->id];
                }
                
                // Filter to get only documents with the same version
                $documentIdsInChain = InductionDocument::whereIn('id', $allDocumentIdsInChain)
                    ->where('version', $doc->version)
                    ->pluck('id')
                    ->toArray();
                
                // Fallback to document ID if no matching version found
                if (empty($documentIdsInChain)) {
                    $documentIdsInChain = [$doc->id];
                }

                // Check if employee has signed this document version (check across entire document chain)
                // Use whereIn with array to handle empty array case
                if (empty($documentIdsInChain)) {
                    $documentIdsInChain = [$doc->id];
                }
                $hasSigned = InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
                    ->where('document_version', $doc->version)
                    ->where('employee_id', $employee->id)
                    ->where('is_valid', true)
                    ->where('del', '0')
                    ->exists();

                // Get signature details if signed
                $signature = null;
                if ($hasSigned) {
                    $signature = InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
                        ->where('document_version', $doc->version)
                        ->where('employee_id', $employee->id)
                        ->where('is_valid', true)
                        ->where('del', '0')
                        ->first();
                }

                return [
                    'id' => $doc->id,
                    'title' => $doc->title,
                    'document_type' => $doc->document_type,
                    'version' => $doc->version,
                    'file_path' => $doc->file_path,
                    'file_url' => $doc->file_path ? url($doc->file_path) : null,
                    'description' => $doc->description,
                    'role_types' => $doc->role_types ?? [],
                    'trades' => $doc->trades->map(function($trade) {
                        return [
                            'id' => $trade->id,
                            'title' => $trade->title,
                        ];
                    })->toArray(),
                    'is_signed' => $hasSigned,
                    'signed_at' => $signature ? $signature->signed_at : null,
                    'signature_path' => $signature ? $signature->signature_path : null,
                    'signature_url' => $signature && $signature->signature_path ? url($signature->signature_path) : null,
                    'signed_file' => $signature ? $signature->signed_file : null,
                    'signed_file_url' => $signature && $signature->signed_file ? url($signature->signed_file) : null,
                    'created_at' => $doc->created_at,
                ];
            })->values()->toArray();

            return $this->success([
                'customer_id' => $request->customer_id,
                'workspace_id' => $request->workspace_id,
                'documents' => $documentsWithStatus,
                'total_documents' => count($documentsWithStatus),
                'signed_count' => collect($documentsWithStatus)->where('is_signed', true)->count(),
                'pending_count' => collect($documentsWithStatus)->where('is_signed', false)->count(),
            ], 'Induction documents retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while retrieving induction documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Sign an induction document
     * Similar to how subcontractors sign but for subcontractor employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function signInductionDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer|exists:users,id',
            'workspace_id' => 'required|integer',
            'document_id' => 'required|integer|exists:induction_documents,id',
            'version' => 'nullable|string',
            'signature' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'signed_file' => 'nullable|file|mimes:pdf|max:20480', // Signed PDF with signature embedded (max 20MB)
            'notes' => 'nullable|string',
        ], [
            'customer_id.required' => 'Customer ID is required.',
            'workspace_id.required' => 'Workspace ID is required.',
            'document_id.required' => 'Document ID is required.',
            'signature.required' => 'Signature image is required.',
            'signature.image' => 'Signature must be an image file.',
            'signature.mimes' => 'Signature must be jpeg, png, jpg, gif, or svg.',
            'signature.max' => 'Signature image must not exceed 2MB.',
            'signed_file.file' => 'Signed file must be a valid file.',
            'signed_file.mimes' => 'Signed file must be a PDF.',
            'signed_file.max' => 'Signed file must not exceed 20MB.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Verify the employee has accepted invitation for this customer
            $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('customer_id', $request->customer_id)
                ->where('workspace_id', $request->workspace_id)
                ->where('invitation_status', 'accepted')
                ->first();

            if (!$invitation) {
                return $this->error('You have not accepted any invitation for this company.', 403);
            }

            $customerId = $request->customer_id;
            $workspaceId = $request->workspace_id;
            $documentId = $request->document_id;

            // Get document
            $document = InductionDocument::where('id', $documentId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->where('is_active', true)
                ->first();

            if (!$document) {
                return $this->error('Document not found or not active', 404);
            }

            // Check if document is for subcontractor employees
            $roleTypes = $document->role_types ?? [];
            if (!in_array('all', $roleTypes) 
                && !in_array('subcontractor_employee', $roleTypes)
                && !in_array('external', $roleTypes)) {
                return $this->error('You are not authorized to sign this document', 403);
            }

            // Use provided version or default to document's current version
            $version = $request->version ?? $document->version;

            // Find the root document (traverse up using parent_document_id)
            $rootDocumentId = $document->id;
            $currentDoc = $document;
            while ($currentDoc && $currentDoc->parent_document_id) {
                $rootDocumentId = $currentDoc->parent_document_id;
                $currentDoc = InductionDocument::where('id', $rootDocumentId)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->first();
                if (!$currentDoc) {
                    break;
                }
            }

            // Verify version exists - check in the same document chain (root or children of root)
            $versionDocument = InductionDocument::where(function($q) use ($rootDocumentId) {
                    $q->where('id', $rootDocumentId)
                      ->orWhere('parent_document_id', $rootDocumentId);
                })
                ->where('version', $version)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->first();

            if (!$versionDocument) {
                return $this->error('Version not found', 404);
            }

            // Get all document IDs in the chain to check for existing signatures
            $allDocumentIdsInChain = InductionDocument::where(function($q) use ($rootDocumentId) {
                    $q->where('id', $rootDocumentId)
                      ->orWhere('parent_document_id', $rootDocumentId);
                })
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->pluck('id')
                ->toArray();
            
            if (empty($allDocumentIdsInChain)) {
                $allDocumentIdsInChain = [$versionDocument->id];
            }
            
            // Check if already signed this version (check across entire document chain)
            $existingSignature = InductionDocumentSignature::whereIn('induction_document_id', $allDocumentIdsInChain)
                ->where('document_version', $version)
                ->where('employee_id', $employee->id)
                ->where('is_valid', true)
                ->where('del', '0')
                ->first();

            if ($existingSignature) {
                return $this->error('You have already signed this document version', 422);
            }

            DB::beginTransaction();
            try {
                // Upload signature image
                $signaturePath = null;
                if ($request->hasFile('signature')) {
                    $signatureFile = $request->file('signature');
                    $directory = 'InductionSignatures';
                    $uploadPath = public_path($directory);
                    if (!File::exists($uploadPath)) {
                        File::makeDirectory($uploadPath, 0755, true);
                    }
                    $extension = $signatureFile->getClientOriginalExtension();
                    $imageName = $employee->id . '_' . $versionDocument->id . '_' . time() . '.' . $extension;
                    $signatureFile->move($uploadPath, $imageName);
                    $signaturePath = "$directory/$imageName";
                }

                // Upload signed PDF file (document with signature embedded)
                $signedFilePath = null;
                if ($request->hasFile('signed_file')) {
                    // Create a temporary request with only signed_file to avoid processing other files
                    $signedFileRequest = new Request();
                    $signedFileRequest->files->set('signed_file', $request->file('signed_file'));
                    $uploadResult = $this->handleFileImageUpload($signedFileRequest, 'InductionSignedFiles');
                    $signedFilePath = $uploadResult['path'] ?? null;
                }

                // Create signature record
                $signature = InductionDocumentSignature::create([
                    'induction_document_id' => $versionDocument->id,
                    'document_version' => $version,
                    'employee_id' => $employee->id,
                    'user_id' => null,
                    'signature_type' => 'subcontractor_employee',
                    'signature_path' => $signaturePath,
                    'signed_file' => $signedFilePath,
                    'signed_at' => now(),
                    'is_valid' => true,
                    'notes' => $request->notes,
                    'customer_id' => $customerId,
                    'workspace_id' => $workspaceId,
                    'del' => '0',
                ]);

                DB::commit();

                // Update induction status
                $this->calculateAndUpdateInductionStatus($employee->id, $customerId, $workspaceId);

                return $this->success([
                    'signature_id' => $signature->id,
                    'document_id' => $versionDocument->id,
                    'version' => $version,
                    'signed_at' => $signature->signed_at,
                    'signature_path' => $signature->signature_path,
                    'signature_url' => $signature->signature_path ? url($signature->signature_path) : null,
                    'signed_file' => $signature->signed_file,
                    'signed_file_url' => $signature->signed_file ? url($signature->signed_file) : null,
                ], 'Document signed successfully', 200);
            } catch (\Exception $e) {
                DB::rollBack();
                return $this->error('An error occurred while uploading signature: ' . $e->getMessage(), 500);
            }
        } catch (\Exception $e) {
            return $this->error('An error occurred while signing document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Calculate and update induction status for an employee
     * 
     * @param int $employeeId
     * @param int $customerId
     * @param int $workspaceId
     * @return void
     */
    private function calculateAndUpdateInductionStatus($employeeId, $customerId, $workspaceId)
    {
        // Get all active induction documents for this customer/workspace
        $inductionDocuments = InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->get();
            
        // Filter documents relevant to subcontractor employees
        $relevantDocs = $inductionDocuments->filter(function($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) 
                || in_array('subcontractor_employee', $roleTypes)
                || in_array('external', $roleTypes);
        });
        
        if ($relevantDocs->isEmpty()) {
            SubcontractorEmployeeInvitation::updateEmployeeInductionStatus(
                $employeeId, 
                $customerId, 
                $workspaceId, 
                SubcontractorEmployeeInvitation::INDUCTION_STATUS_COMPLETE
            );
            return;
        }

        // Check signatures for each relevant document
        $signedCount = 0;
        foreach ($relevantDocs as $doc) {
            // Check if signed current version
            $hasSignedCurrent = InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('employee_id', $employeeId)
                ->where('is_valid', true)
                ->where('del', '0')
                ->exists();
                
            if ($hasSignedCurrent) {
                $signedCount++;
                continue;
            }
            
            // If not signed current, check if minor update and signed previous
            if ($doc->update_type === 'minor') {
                // Find root document id
                $rootDocumentId = $doc->id;
                $currentDoc = $doc;
                while ($currentDoc && $currentDoc->parent_document_id) {
                    $rootDocumentId = $currentDoc->parent_document_id;
                    $currentDoc = InductionDocument::find($rootDocumentId);
                }
                
                // Get all IDs in chain
                $documentIdsInChain = InductionDocument::where(function($q) use ($rootDocumentId) {
                        $q->where('id', $rootDocumentId)
                          ->orWhere('parent_document_id', $rootDocumentId);
                    })
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->pluck('id')
                    ->toArray();
                    
                // Check if signed any version in chain
                $hasSignedAny = InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
                    ->where('employee_id', $employeeId)
                    ->where('is_valid', true)
                    ->where('del', '0')
                    ->exists();
                    
                if ($hasSignedAny) {
                    $signedCount++;
                }
            }
            // If major update and not signed current, it counts as not signed
        }

        $status = SubcontractorEmployeeInvitation::INDUCTION_STATUS_NOT_SIGNED;
        
        if ($signedCount === 0) {
            $status = SubcontractorEmployeeInvitation::INDUCTION_STATUS_NOT_SIGNED;
        } elseif ($signedCount >= $relevantDocs->count()) {
            $status = SubcontractorEmployeeInvitation::INDUCTION_STATUS_COMPLETE;
        } else {
            $status = SubcontractorEmployeeInvitation::INDUCTION_STATUS_PARTIAL;
        }

        SubcontractorEmployeeInvitation::updateEmployeeInductionStatus(
            $employeeId, 
            $customerId, 
            $workspaceId, 
            $status
        );
    }

    /**
     * Get document and induction statuses for a specific project and subcontractor
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDocumentStatuses(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'project_id' => 'required|integer|exists:projects,id',
            'subcontractor_id' => 'required|integer|exists:users,id',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            // Verify the employee is associated with this subcontractor and project
            $association = DB::table('employees_subcontractors_metas')
                ->where('emp_id', $employee->id)
                ->where('subcontractor_id', $request->subcontractor_id)
                ->first();

            if (!$association) {
                return $this->error('You are not associated with this subcontractor', 404);
            }

            // Check if project is in the association's project_ids
            $projectIds = [];
            if ($association->project_ids) {
                $projectIds = is_string($association->project_ids) 
                    ? json_decode($association->project_ids, true) 
                    : $association->project_ids;
            }

            if (!in_array($request->project_id, $projectIds)) {
                return $this->error('You are not assigned to this project', 404);
            }

            // Verify project exists
            $project = DB::table('projects')
                ->where('id', $request->project_id)
                ->where('is_deleted', '0')
                ->first();

            if (!$project) {
                return $this->error('Project not found', 404);
            }

            // Get invitation record for this specific project
            $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('subcontractor_id', $request->subcontractor_id)
                ->where('project_id', $request->project_id)
                ->first();

            // If no invitation record exists, return default statuses
            if (!$invitation) {
                return $this->success([
                    'project_id' => $request->project_id,
                    'subcontractor_id' => $request->subcontractor_id,
                    'required_docs_status' => 0,
                    'required_docs_status_label' => 'not_uploaded',
                    'induction_status' => 0,
                    'induction_status_label' => 'not_signed',
                    'invitation_status' => 'pending',
                ], 'Document statuses retrieved successfully', 200);
            }

            // Return statuses from invitation record
            return $this->success([
                'project_id' => $request->project_id,
                'subcontractor_id' => $request->subcontractor_id,
                'required_docs_status' => $invitation->required_docs_status ?? 0,
                'required_docs_status_label' => $invitation->required_docs_status_label ?? 'not_uploaded',
                'induction_status' => $invitation->induction_status ?? 0,
                'induction_status_label' => $invitation->induction_status_label ?? 'not_signed',
                'invitation_status' => $invitation->invitation_status ?? 'pending',
                'customer_id' => $invitation->customer_id,
                'workspace_id' => $invitation->workspace_id,
            ], 'Document statuses retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while retrieving document statuses: ' . $e->getMessage(), 500);
        }
    }
}
