<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Overtime;
use App\Models\Sites;
use App\Models\EmpTeamsMember;
use App\Models\EmpPersonalDetails;
use App\Models\EmployeeSubcontractor;
use App\Models\RosterAssign;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Pagination\LengthAwarePaginator;

class OvertimeController extends Controller
{


    public function index(Request $request)
    {

        $validator = $this->overtimeFilterValidationRequest($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $overtimesQuery = Overtime::with(['sites', 'emppersonaldetails', 'subcontractorEmployee', 'empcompanydetails'])->latest('id');
        
        // Apply filter to include both regular and subcontractor employees
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->error('Unauthorized access', 403);
        }
        
        $overtimesQuery->where(function ($q) use ($ids) {
            // Regular employees (where empcompanydetails exists)
            $q->whereHas('empcompanydetails', function ($subQ) use ($ids) {
                $subQ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id']);
            })
            // Subcontractor employees (where subcontractor_id is not null)
            ->orWhere(function ($subQ) use ($ids) {
                $subQ->whereNotNull('subcontractor_id')
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id']);
            });
        });
        
        // Apply search filter
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $overtimesQuery->where(function ($q) use ($searchTerm) {
                $q->where('description', 'like', '%' . $searchTerm . '%')
                    // Apply name search for both regular and subcontractor employees
                    ->orWhere(function ($nameQuery) use ($searchTerm) {
                        $nameQuery->whereHas('emppersonaldetails', function ($subquery) use ($searchTerm) {
                            $subquery->where(function ($query) use ($searchTerm) {
                                $query->where('first_name', 'like', '%' . $searchTerm . '%')
                                    ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                                    ->orWhere('last_name', 'like', '%' . $searchTerm . '%');
                            });
                        })
                        ->orWhereHas('subcontractorEmployee', function ($subquery) use ($searchTerm) {
                            $subquery->where(function ($query) use ($searchTerm) {
                                $query->where('first_name', 'like', '%' . $searchTerm . '%')
                                    ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                                    ->orWhere('last_name', 'like', '%' . $searchTerm . '%')
                                    ->orWhere('email', 'like', '%' . $searchTerm . '%');
                            });
                        });
                    })
                    ->orWhereHas('sites', function ($subquery) use ($searchTerm) {
                        $subquery->where('title', 'like', '%' . $searchTerm . '%');
                    });
            });
        }
        // Apply customer and workspace filters
        $result = $this->withCount($overtimesQuery, 'Overtime records retrieved successfully');
        
        // Transform the result to map subcontractor employees
        return $this->transformOvertimeCollection($result);
    }
    
    /**
     * Transform overtime collection to map subcontractor employee details
     */
    private function transformOvertimeCollection($result)
    {
        // If result is a JSON response, decode it first
        if ($result instanceof \Illuminate\Http\JsonResponse) {
            $content = json_decode($result->getContent(), true);
            if (isset($content['data']) && is_array($content['data'])) {
                $content['data'] = array_map(function ($overtime) {
                    return $this->transformOvertimeForSubcontractor($overtime);
                }, $content['data']);
            }
            return response()->json($content, $result->getStatusCode());
        }
        
        // Check if it's an array first (arrays can't have methods)
        if (is_array($result)) {
            // Array - transform each item
            return array_map(function ($overtime) {
                return $this->transformOvertimeForSubcontractor($overtime);
            }, $result);
        }
        
        // Check if it's an object before calling method_exists
        if (is_object($result)) {
            if (method_exists($result, 'getCollection')) {
                // Paginated result - transform the collection items
                $result->getCollection()->transform(function ($overtime) {
                    return $this->transformOvertimeForSubcontractor($overtime);
                });
                return $result;
            } elseif ($result instanceof LengthAwarePaginator) {
                // Another pagination type
                $items = $result->items();
                foreach ($items as $key => $overtime) {
                    $items[$key] = $this->transformOvertimeForSubcontractor($overtime);
                }
                $result->setCollection(collect($items));
                return $result;
            } elseif (method_exists($result, 'map')) {
                // Collection - transform each item
                return $result->map(function ($overtime) {
                    return $this->transformOvertimeForSubcontractor($overtime);
                });
            }
        }
        
        // Fallback: return as-is if we can't transform
        return $result;
    }
    
    /**
     * Transform a single overtime record to use subcontractor employee details if applicable
     */
    private function transformOvertimeForSubcontractor($overtime)
    {
        // Convert to array if it's a model instance
        $overtimeArray = $overtime instanceof \Illuminate\Database\Eloquent\Model 
            ? $overtime->toArray() 
            : (array)$overtime;
        
        // Check for subcontractor_id (can be in snake_case or camelCase)
        $subcontractorId = $overtimeArray['subcontractor_id'] ?? $overtimeArray['subcontractorId'] ?? null;
        
        // Get subcontractor employee data (check both naming conventions)
        $subcontractorEmp = $overtimeArray['subcontractor_employee'] ?? $overtimeArray['subcontractorEmployee'] ?? null;
        
        // Get empcompanydetails for regular employees
        $empCompanyDetails = $overtimeArray['empcompanydetails'] ?? $overtimeArray['emp_company_details'] ?? null;
        
        // If subcontractor_id is not null and subcontractor employee exists, use subcontractor employee details
        if ($subcontractorId !== null && $subcontractorEmp && !empty($subcontractorEmp)) {
            // Map subcontractor employee details to match regular employee structure
            $subcontractorEmpArray = is_array($subcontractorEmp) ? $subcontractorEmp : (array)$subcontractorEmp;
            
            // Create a mapped structure similar to empPersonalDetails
            $mappedEmployee = [
                'id' => $subcontractorEmpArray['id'] ?? null,
                'emp_id' => $subcontractorEmpArray['id'] ?? null,
                'first_name' => $subcontractorEmpArray['first_name'] ?? $subcontractorEmpArray['firstName'] ?? '',
                'middle_name' => $subcontractorEmpArray['middle_name'] ?? $subcontractorEmpArray['middleName'] ?? '',
                'last_name' => $subcontractorEmpArray['last_name'] ?? $subcontractorEmpArray['lastName'] ?? '',
                'email' => $subcontractorEmpArray['email'] ?? '',
                'subcontractors' => $subcontractorEmpArray['subcontractors'] ?? '',
                'mobile_number' => $subcontractorEmpArray['mobile_number'] ?? $subcontractorEmpArray['mobileNumber'] ?? '',
                'full_name' => trim(($subcontractorEmpArray['first_name'] ?? $subcontractorEmpArray['firstName'] ?? '') . ' ' . 
                    ($subcontractorEmpArray['middle_name'] ?? $subcontractorEmpArray['middleName'] ?? '') . ' ' . 
                    ($subcontractorEmpArray['last_name'] ?? $subcontractorEmpArray['lastName'] ?? '')),
            ];
            
            // Replace empPersonalDetails with mapped subcontractor employee
            $overtimeArray['emppersonaldetails'] = $mappedEmployee;
            // $overtimeArray['emp_personal_details'] = $mappedEmployee;
            
            // Set user_type for subcontractor employees (1 = external/subcontractor employee)
            $overtimeArray['user_type'] = '1';
        } else {
            // For regular employees, get user_type from empcompanydetails
            if ($empCompanyDetails) {
                $empCompanyDetailsArray = is_array($empCompanyDetails) ? $empCompanyDetails : (array)$empCompanyDetails;
                $overtimeArray['user_type'] = (string)($empCompanyDetailsArray['user_type'] ?? '0');
            } else {
                // Fallback: try to get from the model relationship if available
                if ($overtime instanceof \Illuminate\Database\Eloquent\Model && $overtime->relationLoaded('empcompanydetails')) {
                    $overtimeArray['user_type'] = (string)($overtime->empcompanydetails->user_type ?? '0');
                } else {
                    // Default to 0 (internal employee) if we can't determine
                    $overtimeArray['user_type'] = '0';
                }
            }
        }
        
        return $overtimeArray;
    }

    /**
     * Store a new overtime record
     */
    public function store(Request $request)
    {
        try {
            $validator = $this->overtimeStoreValidationRequest($request);
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
            $ids = $this->getCustomerAndWorkspaceIds();
            $validatedData = $validator->validated();
            
            // Check if this is a subcontractor employee (user_type == 1)
            $isSubcontractorEmployee = isset($validatedData['user_type']) && $validatedData['user_type'] == 1;
            $subcontractorId = null;
            
            // Get subcontractor_id from roster_assigns if it's a subcontractor employee
            if ($isSubcontractorEmployee) {
                $dateForRosterCheck = is_string($validatedData['date']) ? $validatedData['date'] : Carbon::parse($validatedData['date'])->format('Y-m-d');
                $rosterRecord = DB::table('roster_assigns')
                    ->where('assign_to', $validatedData['employee_id'])
                    ->where('schedule_date', $dateForRosterCheck)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->first();
                
                if ($rosterRecord) {
                    $subcontractorId = $rosterRecord->subcontractor_id ?? null;
                }
            }
            
            // Check for existing overlapping overtime (pass subcontractor_id for proper filtering)
            if ($this->hasOverlappingOvertime($validatedData, null, $subcontractorId)) {
                return $this->error('An overtime record with overlapping times already exists for this employee', 400);
            }
            // Calculate working hours
            $workingHours = $this->calculateWorkingHours($validatedData['check_in'], $validatedData['check_out']);
            $overtime = Overtime::create([
                'employee_id' => $validatedData['employee_id'],
                'site_id' => $validatedData['site_id'],
                'check_in' => $validatedData['check_in'],
                'check_out' => $validatedData['check_out'],
                'date' => $validatedData['date'],
                'working_hours' => $workingHours,
                'description' => $validatedData['description'],
                'status' => $validatedData['status'] ?? config('constants.overtime_status.pending'),
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
                'subcontractor_id' => $subcontractorId,
            ]);

            // Send notifications for overtime application if status is pending
            if (($validatedData['status'] ?? config('constants.overtime_status.pending')) === config('constants.overtime_status.pending')) {
                $this->sendOvertimeNotifications($overtime, $ids);
            }

            return $this->success($overtime, 'Overtime record created successfully');
        } catch (\Exception $e) {
            Log::error('Overtime store error: ' . $e->getMessage());
            return $this->error('Failed to create overtime record', 500);
        }
    }

    /**
     * Show a specific overtime record
     */
    public function show($id)
    {
        try {
            $overtime = Overtime::find($id);
            if (!$overtime) {
                return $this->error('Overtime record not found', 404);
            }
            // Check customer/workspace access
            $ids = $this->getCustomerAndWorkspaceIds();
            if ($overtime->customer_id !== $ids['customer_id'] || $overtime->workspace_id !== $ids['workspace_id']) {
                return $this->error('You do not have access to this overtime record', 403);
            }
            return $this->success($overtime, 'Overtime record retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Overtime show error: ' . $e->getMessage());
            return $this->error('Failed to retrieve overtime record', 500);
        }
    }

    /**
     * Update an existing overtime record
     */
    public function update(Request $request)
    {
        try {
            $validator = $this->overtimeUpdateValidationRequest($request);
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
            $overtime = Overtime::find($request->id);
            if (!$overtime) {
                return $this->error('Overtime record not found', 404);
            }
            // Check customer/workspace access
            $ids = $this->getCustomerAndWorkspaceIds();
            if ($overtime->customer_id !== $ids['customer_id'] || $overtime->workspace_id !== $ids['workspace_id']) {
                return $this->error('You do not have access to this overtime record', 403);
            }
            $validatedData = $validator->validated();
            
            // Check if this is a subcontractor employee (user_type == 1)
            $isSubcontractorEmployee = isset($validatedData['user_type']) && $validatedData['user_type'] == 1;
            $subcontractorId = $overtime->subcontractor_id; // Preserve existing subcontractor_id by default
            
            // Get subcontractor_id from roster_assigns if it's a subcontractor employee
            if ($isSubcontractorEmployee) {
                $dateForRosterCheck = is_string($validatedData['date']) ? $validatedData['date'] : Carbon::parse($validatedData['date'])->format('Y-m-d');
                $rosterRecord = DB::table('roster_assigns')
                    ->where('assign_to', $validatedData['employee_id'])
                    ->where('schedule_date', $dateForRosterCheck)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->first();
                
                if ($rosterRecord) {
                    $subcontractorId = $rosterRecord->subcontractor_id ?? $overtime->subcontractor_id ?? null;
                }
            } else {
                // For regular employees, ensure subcontractor_id is null
                $subcontractorId = null;
            }
            
            // Check for existing overlapping overtime (excluding current record)
            // Pass subcontractor_id to ensure we only check overlaps for the same subcontractor
            if ($this->hasOverlappingOvertime($validatedData, $request->id, $subcontractorId)) {
                return $this->error('An overtime record with overlapping times already exists for this employee', 400);
            }
            // Calculate working hours
            $workingHours = $this->calculateWorkingHours($validatedData['check_in'], $validatedData['check_out']);
            $validatedData['working_hours'] = $workingHours;
            $validatedData['subcontractor_id'] = $subcontractorId;
            $overtime->update($validatedData);
            return $this->success($overtime, 'Overtime record updated successfully');
        } catch (\Exception $e) {
            Log::error('Overtime update error: ' . $e->getMessage());
            return $this->error('Failed to update overtime record', 500);
        }
    }

    /**
     * Update overtime status
     */
    public function updateStatus(Request $request)
    {
        try {
            $validator = $this->overtimeStatusValidationRequest($request);
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
            $overtime = Overtime::find($request->id);
            if (!$overtime) {
                return $this->error('Overtime record not found', 404);
            }
            // Check customer/workspace access
            $ids = $this->getCustomerAndWorkspaceIds();
            if ($overtime->customer_id !== $ids['customer_id'] || $overtime->workspace_id !== $ids['workspace_id']) {
                return $this->error('You do not have access to this overtime record', 403);
            }
            $oldStatus = $overtime->status;
            $overtime->update(['status' => $request->status]);

            // Send notification for status change
            if ($oldStatus !== $request->status) {
                $this->sendOvertimeStatusChangeNotification($overtime, $oldStatus, $request->status, $ids);
            }

            return $this->success($overtime, 'Overtime status updated successfully');
        } catch (\Exception $e) {
            Log::error('Overtime status update error: ' . $e->getMessage());
            return $this->error('Failed to update overtime status', 500);
        }
    }

    /**
     * Delete an overtime record
     */
    public function destroy($id)
    {
        try {
            $overtime = Overtime::query();
            $overtime = $this->applyCustomerWorkspaceFilter($overtime);
            $overtime = $overtime->find($id);
            if (!$overtime) {
                return $this->error('Overtime record not found', 404);
            }
            $overtime->delete();
            return $this->message('Overtime record deleted successfully');
        } catch (\Exception $e) {
            Log::error('Overtime delete error: ' . $e->getMessage());
            return $this->error('Failed to delete overtime record', 500);
        }
    }

    /**
     * Apply for overtime (employee self-service)
     */
    public function applyOvertime(Request $request)
    {
        try {
            $validator = $this->overtimeApplicationValidationRequest($request);
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
            $ids = $this->getCustomerAndWorkspaceIds();
            if (!$ids) {
                return $this->error('Unauthorized access', 403);
            }
            $validatedData = $validator->validated();
            $employeeId = Auth::id();
            $validatedData['employee_id'] = $employeeId;
            
            // Check if authenticated user is a subcontractor employee
            $isSubcontractorEmployee = EmployeeSubcontractor::where('id', $employeeId)->exists();
            $subcontractorId = null;
            
            // Get subcontractor_id from roster_assigns if it's a subcontractor employee
            if ($isSubcontractorEmployee) {
                $dateForRosterCheck = is_string($validatedData['date']) ? $validatedData['date'] : Carbon::parse($validatedData['date'])->format('Y-m-d');
                $rosterRecord = DB::table('roster_assigns')
                    ->where('assign_to', $employeeId)
                    ->where('schedule_date', $dateForRosterCheck)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->first();
                
                if ($rosterRecord) {
                    $subcontractorId = $rosterRecord->subcontractor_id ?? null;
                }
            }
            
            // Check for existing overlapping overtime (pass subcontractor_id for proper filtering)
            if ($this->hasOverlappingOvertime($validatedData, null, $subcontractorId)) {
                return $this->error('An overtime record with overlapping times already exists', 400);
            }
            // Calculate working hours
            $workingHours = $this->calculateWorkingHours($validatedData['check_in'], $validatedData['check_out']);
            $overtime = Overtime::create([
                'employee_id' => $validatedData['employee_id'],
                'site_id' => $validatedData['site_id'],
                'check_in' => $validatedData['check_in'],
                'check_out' => $validatedData['check_out'],
                'date' => $validatedData['date'],
                'working_hours' => $workingHours,
                'description' => $validatedData['description'],
                'status' => config('constants.overtime_status.pending'),
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
                'subcontractor_id' => $subcontractorId,
            ]);

            // Send notifications for overtime application
            $this->sendOvertimeNotifications($overtime, $ids);

            return $this->success($overtime, 'Overtime application submitted successfully');
        } catch (\Exception $e) {
            Log::error('Overtime application error: ' . $e->getMessage());
            return $this->error('Failed to submit overtime application', 500);
        }
    }

    /**
     * Bulk delete overtime records
     */
    public function bulkDelete(Request $request)
    {
        try {
            $validator = $this->overtimeBulkDeleteValidationRequest($request);
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
            $deletedCount = Overtime::query();
            $deletedCount = $this->applyCustomerWorkspaceFilter($deletedCount);
            $deletedCount = $deletedCount->whereIn('id', $request->ids)->delete();
            return $this->success(['deleted_count' => $deletedCount], 'Overtime records deleted successfully');
        } catch (\Exception $e) {
            Log::error('Overtime bulk delete error: ' . $e->getMessage());
            return $this->error('Failed to delete overtime records', 500);
        }
    }

    /**
     * Check for overlapping overtime records
     * 
     * @param array $data The overtime data to check
     * @param int|null $excludeId ID to exclude from the check (for updates)
     * @param int|null $subcontractorId Subcontractor ID to filter by (for subcontractor employees)
     * @return bool
     */
    private function hasOverlappingOvertime($data, $excludeId = null, $subcontractorId = null)
    {
        $query = Overtime::where('employee_id', $data['employee_id'])
            ->where(function ($query) use ($data) {
                $query->whereBetween('check_in', [$data['check_in'], $data['check_out']])
                    ->orWhereBetween('check_out', [$data['check_in'], $data['check_out']])
                    ->orWhere(function ($query) use ($data) {
                        $query->where('check_in', '<', $data['check_in'])
                            ->where('check_out', '>', $data['check_out']);
                    });
            });
        
        // For subcontractor employees, also filter by subcontractor_id
        if ($subcontractorId !== null) {
            $query->where('subcontractor_id', $subcontractorId);
        } else {
            // For regular employees, ensure subcontractor_id is null
            $query->whereNull('subcontractor_id');
        }
        
        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }
        
        return $query->exists();
    }

    /**
     * Calculate working hours in minutes
     */
    private function calculateWorkingHours($checkIn, $checkOut)
    {
        return Carbon::parse($checkIn)->diffInMinutes(Carbon::parse($checkOut));
    }

    /**
     * Apply name filter to query
     */
    private function applyNameFilter($query, $name)
    {
        $nameComponents = explode(' ', $name);

        $query->whereHas('emppersonaldetails', function ($subquery) use ($nameComponents) {
            $subquery->where(function ($q) use ($nameComponents) {
                foreach ($nameComponents as $component) {
                    $q->orWhere('first_name', 'like', '%' . $component . '%')
                        ->orWhere('middle_name', 'like', '%' . $component . '%')
                        ->orWhere('last_name', 'like', '%' . $component . '%');
                }
            });
        });
    }

    /**
     * Send notifications for overtime application
     */
    private function sendOvertimeNotifications($overtime, $ids)
    {

        try {
            // Check if this is a subcontractor employee
            $isSubcontractorEmployee = !is_null($overtime->subcontractor_id);
            $employeeName = '';
            
            if ($isSubcontractorEmployee) {
                // Get subcontractor employee details
                $subcontractorEmployee = EmployeeSubcontractor::find($overtime->employee_id);
                if (!$subcontractorEmployee) {
                    Log::warning('Subcontractor employee not found for overtime notification: ' . $overtime->employee_id);
                    return;
                }
                $employeeName = trim($subcontractorEmployee->first_name . ' ' . ($subcontractorEmployee->middle_name ?? '') . ' ' . $subcontractorEmployee->last_name);
            } else {
                // Get regular employee details
                $employee = EmpPersonalDetails::where('emp_id', $overtime->employee_id)->first();
                if (!$employee) {
                    Log::warning('Employee not found for overtime notification: ' . $overtime->employee_id);
                    return;
                }
                $employeeName = trim($employee->first_name . ' ' . $employee->middle_name . ' ' . $employee->last_name);
            }

            // Get site details
            $site = Sites::find($overtime->site_id);
            if (!$site) {
                Log::warning('Site not found for overtime notification: ' . $overtime->site_id);
                return;
            }
            // Prepare notification message
            $date = Carbon::parse($overtime->date)->format('d/m/Y');
            // Get raw database values to avoid accessor formatting issues
            $rawCheckIn = $overtime->getRawOriginal('check_in');
            $rawCheckOut = $overtime->getRawOriginal('check_out');

            // Debug: Log the raw values to see what's in the database
            Log::info('Overtime notification - Raw check_in from DB: ' . $rawCheckIn);
            Log::info('Overtime notification - Raw check_out from DB: ' . $rawCheckOut);

            // Extract time from datetime values
            $checkIn = $this->getTimeFromDatetimeHelper($rawCheckIn);
            $checkOut = $this->getTimeFromDatetimeHelper($rawCheckOut);

            // Debug: Log the extracted time values
            Log::info('Overtime notification - Extracted check_in time: ' . $checkIn);
            Log::info('Overtime notification - Extracted check_out time: ' . $checkOut);
            $workingHours = round($overtime->working_hours / 60, 2);
            $notificationTitle = 'Overtime Application Submitted';
            $notificationMessage = "Employee {$employeeName} has submitted an overtime application for {$date} from {$checkIn} to {$checkOut} ({$workingHours} hours) at {$site->title}.";

            // Send notification to customer (customer_id represents the customer who owns the workspace)
            // The customer will receive notification about overtime applications from their employees
            $this->save_notifications(
                $notificationTitle,
                $notificationMessage,
                config('constants.employee_types.employee'),
                $overtime->employee_id, // Sender is the employee
                config('constants.employee_types.customer'),
                $ids['customer_id'],    // Receiver is the customer
                config('constants.notification_types.overtime_applied'),
                $ids['customer_id'],
                $ids['workspace_id']
            );
        } catch (\Exception $e) {
            Log::error('Error sending overtime notifications: ' . $e->getMessage());
        }
    }

    function getTimeFromDatetimeHelper($datetime)
    {
        if (!$datetime) {
            return '00:00:00';
        }

        try {
            // Try using Carbon for more robust parsing
            $carbon = Carbon::parse($datetime);
            return $carbon->format('H:i:s');
        } catch (\Exception $e) {
            // Fallback to strtotime if Carbon fails
            try {
                $timestamp = strtotime($datetime);
                if ($timestamp !== false) {
                    return date('H:i:s', $timestamp);
                }
            } catch (\Exception $e2) {
                Log::warning("Failed to parse datetime in getTimeFromDatetimeHelper: " . $e2->getMessage());
            }

            // If all parsing fails, return default
            return '00:00:00';
        }
    }

    /**
     * Send notification when overtime status changes
     */
    private function sendOvertimeStatusChangeNotification($overtime, $oldStatus, $newStatus, $ids)
    {
        try {
            // Check if this is a subcontractor employee
            $isSubcontractorEmployee = !is_null($overtime->subcontractor_id);
            
            if ($isSubcontractorEmployee) {
                // Get subcontractor employee details
                $subcontractorEmployee = EmployeeSubcontractor::find($overtime->employee_id);
                if (!$subcontractorEmployee) {
                    return;
                }
            } else {
                // Get regular employee details
                $employee = EmpPersonalDetails::where('emp_id', $overtime->employee_id)->first();
                if (!$employee) {
                    return;
                }
            }

            $date = Carbon::parse($overtime->date)->format('d/m/Y');

            $statusText = ucfirst($newStatus);
            $notificationTitle = "Overtime {$statusText}";
            $notificationMessage = "Your overtime application for {$date} has been {$newStatus}.";

            // Send notification to employee
            $this->save_notifications(
                $notificationTitle,
                $notificationMessage,
                config('constants.employee_types.customer'),
                Auth::id(), // Sender is the approver/rejector
                config('constants.employee_types.employee'),
                $overtime->employee_id,
                config('constants.notification_types.overtime_' . $newStatus),
                $ids['customer_id'],
                $ids['workspace_id']
            );
        } catch (\Exception $e) {
            Log::error('Error sending overtime status change notification: ' . $e->getMessage());
        }
    }
}
