<?php

namespace App\Http\Controllers;

use App\Models\EmpCompanyDetails;
use App\Models\EmployeeAttendance;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\RosterAssign;
use App\Models\Project;
use App\Models\ProjectSite;
use App\Models\XeroToken;
use App\Models\XeroPayslipHistory;
use App\Services\XeroService;
use App\Http\Controllers\Traits\TimesheetTrait;
use App\Http\Controllers\Traits\HelperTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class TimesheetController extends Controller
{
    use TimesheetTrait, HelperTrait;
    /**
     * Get all internal employees with their attendance records
     * Shows daily attendance with project and site information
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    /**
     * Get user's preferred date format (from BaseModel)
     * Uses EmployeeAttendance model to access BaseModel's getUserDateFormat method
     */
    private function getUserDateFormat()
    {
        $attendanceModel = new EmployeeAttendance();
        return $attendanceModel->getUserDateFormat();
    }

    public function getTimesheets(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        
        // Chunk 1: Validate and parse dates
        $dateData = $this->validateAndParseTimesheetDates($request);
        if (isset($dateData['error'])) {
            return $this->handleValidationFailure($dateData['error']);
        }
        $fromDate = $dateData['fromDate'];
        $toDate = $dateData['toDate'];
        $userDateFormat = $dateData['userDateFormat'];
        // Get date array for subcontractor employees
        $datesArray = [];
        $start = Carbon::parse($fromDate);
        $end = Carbon::parse($toDate);
        while ($start->lte($end)) {
            $datesArray[] = $start->toDateString();
            $start->addDay();
        }
        // Chunk 2: Fetch internal employees data (only if subcontractor_id is not provided)
        $internalEmployees = collect();
        $internalAttendances = collect();
        $internalProjects = collect();
        $internalRosterAssigns = collect();
        $internalProjectSitesMap = [];
        // Only fetch internal employees if subcontractor_id is NOT provided
        if (!$request->filled('subcontractor_id')) {
            $employeesData = $this->fetchInternalEmployeesData($request, $ids, $fromDate, $toDate);
            $internalEmployees = $employeesData ? $employeesData['employees'] : collect();
            // Ensure internal employees have user_type and subcontractor_id set
            $internalEmployees = $internalEmployees->map(function($employee) {
                // Add user_type and subcontractor_id if not already set
                if (!isset($employee->user_type)) {
                    $employee->user_type = 0; // Internal employee
                }
                if (!isset($employee->subcontractor_id)) {
                    $employee->subcontractor_id = null; // Internal employees don't have subcontractor_id
                }
                return $employee;
            });
            $internalAttendances = $employeesData ? $employeesData['allAttendances'] : collect();
            $internalProjects = $employeesData ? $employeesData['projects'] : collect();
            $internalRosterAssigns = $employeesData ? $employeesData['rosterAssigns'] : collect();
            $internalProjectSitesMap = $employeesData ? $employeesData['projectSitesMap'] : [];
            if (isset($employeesData['empty'])) {
                return $this->success([], 'No attendance records found for the selected site');
            }
        }
        // Chunk 3: First, get all subcontractor employee attendances in the date range
        // This ensures we get all employees with attendance, even if they're not in getSubcontractorEmployeesForRoster
        $subAttendanceQuery = \App\Models\EmployeeAttendance::whereNotNull('subcontractor_id')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->whereBetween('date', [$fromDate, $toDate]);
        // Filter by subcontractor_id if provided
        if ($request->filled('subcontractor_id')) {
            $subAttendanceQuery->where('subcontractor_id', $request->subcontractor_id);
        }
        if ($request->filled('site_id')) {
            $subAttendanceQuery->where('site_id', $request->site_id);
        }
        $subAttendances = $subAttendanceQuery->with([
            'sites' => function($query) {
                $query->select('id', 'title', 'project_id');
            },
            'sites.project' => function($query) use ($ids) {
                $query->where('customer_id', $ids['customer_id'])
                      ->where('workspace_id', $ids['workspace_id'])
                      ->select('id', 'title');
            },
            'breaks' => function($query) {
                $query->select('id', 'emp_attendance_id', 'break_in', 'break_out', 'date')
                      ->orderBy('break_in', 'asc');
            }
        ])
        ->orderBy('date', 'desc')
        ->get();
        // Get unique employee IDs from attendances
        $attendanceEmployeeIds = $subAttendances->pluck('employee_id')->unique()->toArray();
        // Get subcontractor employees using getSubcontractorEmployeesForRoster
        $subcontractorEmployeesData = $this->getSubcontractorEmployeesForRoster(
            $ids['customer_id'],
            $ids['workspace_id'],
            $datesArray,
            $request->site_id ?? null,
            $request->project_id ?? null
        );
        // Process subcontractor employees and get their attendance data
        $subcontractorEmployees = collect();
        $subcontractorEmployeeIds = [];
        $subcontractorIds = []; // Collect all subcontractor IDs
        // Process employees from getSubcontractorEmployeesForRoster
        if (!empty($subcontractorEmployeesData)) {
            foreach ($subcontractorEmployeesData as $subEmpData) {
                $subEmpId = $subEmpData['id'];
                // Filter by subcontractor_id if provided
                if ($request->filled('subcontractor_id')) {
                    $requestedSubcontractorId = (int)$request->subcontractor_id;
                    $employeeSubcontractorIds = [];
                    // Get subcontractor IDs from the employee's subcontractors array
                    if (!empty($subEmpData['subcontractors'])) {
                        foreach ($subEmpData['subcontractors'] as $subcontractor) {
                            if (isset($subcontractor['id'])) {
                                $employeeSubcontractorIds[] = (int)$subcontractor['id'];
                            }
                        }
                    }   
                    // Skip this employee if they don't belong to the requested subcontractor
                    if (!in_array($requestedSubcontractorId, $employeeSubcontractorIds)) {
                        continue;
                    }
                }
                $subcontractorEmployeeIds[] = $subEmpId;
                // Collect subcontractor IDs from the employee's subcontractors array
                if (!empty($subEmpData['subcontractors'])) {
                    foreach ($subEmpData['subcontractors'] as $subcontractor) {
                        if (isset($subcontractor['id'])) {
                            $subcontractorIds[] = $subcontractor['id'];
                        }
                    }
                }
                // Get actual EmployeeSubcontractor to get email and xero_emp_id
                $subEmp = \App\Models\EmployeeSubcontractor::find($subEmpId);
                // Create a mock employee object similar to EmpCompanyDetails structure
                $mockEmployee = (object)[
                    'id' => $subEmpId,
                    'employee_email' => $subEmp ? $subEmp->email : null,
                    'xero_emp_id' => $subEmp ? $subEmp->xero_emp_id : null,
                    'user_type' => 1, // External/subcontractor employee
                    'empPersonalDetails' => (object)[
                        'first_name' => $subEmpData['personal_details']['first_name'] ?? '',
                        'middle_name' => $subEmpData['personal_details']['middle_name'] ?? '',
                        'last_name' => $subEmpData['personal_details']['last_name'] ?? '',
                        'image' => $subEmpData['personal_details']['image'] ?? null,
                    ],
                    'subcontractor_id' => !empty($subEmpData['subcontractors']) ? $subEmpData['subcontractors'][0]['id'] ?? null : null, // Get first subcontractor ID
                    'subcontractors' => $subEmpData['subcontractors'] ?? [], // Include subcontractors array
                ];
                $subcontractorEmployees->push($mockEmployee);
            }
        }
        // Find employees with attendance that aren't in the initial list from getSubcontractorEmployeesForRoster
        $missingEmployeeIds = array_diff($attendanceEmployeeIds, $subcontractorEmployeeIds);
            // If there are employees with attendance but not in our initial list, add them
            if (!empty($missingEmployeeIds)) {
                $missingEmployees = \App\Models\EmployeeSubcontractor::whereIn('id', $missingEmployeeIds)->get();
                // Get subcontractor IDs from attendances for these employees
                $employeeSubcontractorMap = [];
                foreach ($subAttendances as $attendance) {
                    if (in_array($attendance->employee_id, $missingEmployeeIds)) {
                        if (!isset($employeeSubcontractorMap[$attendance->employee_id])) {
                            $employeeSubcontractorMap[$attendance->employee_id] = [];
                        }
                        if ($attendance->subcontractor_id) {
                            $employeeSubcontractorMap[$attendance->employee_id][] = $attendance->subcontractor_id;
                        }
                    }
                }
                foreach ($missingEmployees as $missingEmp) {
                    // Get subcontractor ID from attendance or from meta
                    $subId = null;
                    if (isset($employeeSubcontractorMap[$missingEmp->id]) && !empty($employeeSubcontractorMap[$missingEmp->id])) {
                        $subId = $employeeSubcontractorMap[$missingEmp->id][0];
                    } else {
                        // Fallback: get from meta
                        $empSubcontractorIds = \App\Models\EmployeeSubcontractorMeta::where('emp_id', $missingEmp->id)
                            ->where('active', 1)
                            ->pluck('subcontractor_id')
                            ->toArray();
                        
                        $validSubIds = \App\Models\SubcontractorCompany::whereIn('user_id', $empSubcontractorIds)
                            ->where('customer_id', $ids['customer_id'])
                            ->where('workspace_id', $ids['workspace_id'])
                            ->where('del', '0')
                            ->pluck('user_id')
                            ->toArray();       
                        $subId = !empty($validSubIds) ? $validSubIds[0] : null;
                    }
                    // Filter by subcontractor_id if provided
                    if ($request->filled('subcontractor_id')) {
                        $requestedSubcontractorId = (int)$request->subcontractor_id;
                        if ($subId != $requestedSubcontractorId) {
                            continue; // Skip this employee if they don't belong to the requested subcontractor
                        }
                    }
                    // Filter by subcontractor_id if provided
                    if ($request->filled('subcontractor_id')) {
                        $requestedSubcontractorId = (int)$request->subcontractor_id;
                        if ($subId != $requestedSubcontractorId) {
                            continue; // Skip this employee if they don't belong to the requested subcontractor
                        }
                    }
                    // Only add if subcontractor belongs to this customer/workspace
                    if ($subId) {
                        $subExists = \App\Models\SubcontractorCompany::where('user_id', $subId)
                            ->where('customer_id', $ids['customer_id'])
                            ->where('workspace_id', $ids['workspace_id'])
                            ->where('del', '0')
                            ->exists();
                        if ($subExists) {
                            // Get subcontractors array from the EmployeeSubcontractor model (accessor)
                            // The model has $appends = ['subcontractors'] which automatically includes it
                            $subcontractorsArray = $missingEmp->subcontractors ?? [];
                            
                            $mockEmployee = (object)[
                                'id' => $missingEmp->id,
                                'employee_email' => $missingEmp->email ?? null,
                                'xero_emp_id' => $missingEmp->xero_emp_id ?? null,
                                'user_type' => 1, // External/subcontractor employee
                                'empPersonalDetails' => (object)[
                                    'first_name' => $missingEmp->first_name ?? '',
                                    'middle_name' => $missingEmp->middle_name ?? '',
                                    'last_name' => $missingEmp->last_name ?? '',
                                    'image' => $missingEmp->profile_image ? url($missingEmp->profile_image) : 'assets/img/default.png',
                                ],
                                'subcontractor_id' => $subId,
                                'subcontractors' => $subcontractorsArray, // Include subcontractors array
                            ];
                            $subcontractorEmployees->push($mockEmployee);
                            $subcontractorEmployeeIds[] = $missingEmp->id;
                        }
                    }
                }
            }
        // Group attendances by employee_id - ensure keys are integers
        $subcontractorAttendancesGrouped = $subAttendances->groupBy(function($item) {
            return (int)$item->employee_id;
        });
        // Convert to array to avoid collection merge issues
        $subcontractorAttendancesArray = [];
        foreach ($subcontractorAttendancesGrouped as $employeeId => $attendances) {
            $subcontractorAttendancesArray[(int)$employeeId] = $attendances;
        }
        // Check if we should filter by employee type or subcontractor_id
        // If subcontractor_id is provided, only return subcontractor employees
        // Otherwise, check employee_type filter
        $shouldFilterByType = $request->filled('employee_type');
        $requestedEmployeeType = $request->input('employee_type'); // 'internal' or 'external'
        $hasSubcontractorFilter = $request->filled('subcontractor_id');
        // Merge internal and subcontractor employees based on filter
        $allEmployees = collect();
        // If subcontractor_id is provided, only return subcontractor employees
        if ($hasSubcontractorFilter) {
            $allEmployees = $allEmployees->concat($subcontractorEmployees);
        } else {
            // Otherwise, use employee_type filter if provided
            if (!$shouldFilterByType || $requestedEmployeeType === 'internal') {
                $allEmployees = $allEmployees->concat($internalEmployees);
            }
            if (!$shouldFilterByType || $requestedEmployeeType === 'external') {
                $allEmployees = $allEmployees->concat($subcontractorEmployees);
            }
        }
        // Merge attendances - convert both to arrays first, then merge
        $allAttendances = collect();
        // If subcontractor_id is provided, only add subcontractor attendances
        if ($hasSubcontractorFilter) {
            foreach ($subcontractorAttendancesArray as $employeeId => $attendances) {
                $allAttendances[$employeeId] = $attendances;
            }
        } else {
            // Otherwise, use employee_type filter if provided
            // Add internal attendances if not filtering or if filtering for internal
            if (!$shouldFilterByType || $requestedEmployeeType === 'internal') {
                foreach ($internalAttendances as $employeeId => $attendances) {
                    $allAttendances[$employeeId] = $attendances;
                }
            }
            // Add subcontractor attendances if not filtering or if filtering for external
            if (!$shouldFilterByType || $requestedEmployeeType === 'external') {
                foreach ($subcontractorAttendancesArray as $employeeId => $attendances) {
                    if ($allAttendances->has($employeeId)) {
                        // If employee already exists, merge the attendances
                        $existingAttendances = $allAttendances[$employeeId];
                        if ($existingAttendances instanceof \Illuminate\Support\Collection) {
                            $allAttendances[$employeeId] = $existingAttendances->concat($attendances);
                        } else {
                            $allAttendances[$employeeId] = collect($existingAttendances)->concat($attendances);
                        }
                    } else {
                        // If employee doesn't exist, add new entry
                        $allAttendances[$employeeId] = $attendances;
                    }
                }
            }
        }
        // Fetch roster assigns for subcontractor employees
        $subcontractorRosterAssigns = collect();
        if (!empty($subcontractorEmployeeIds)) {
            $subcontractorRosterQuery = RosterAssign::whereIn('assign_to', $subcontractorEmployeeIds)
                ->whereNotNull('subcontractor_id') // Only get roster assigns for subcontractor employees
                ->whereBetween('schedule_date', [$fromDate, $toDate])
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);   
            // Filter by subcontractor_id if provided
            if ($request->filled('subcontractor_id')) {
                $subcontractorRosterQuery->where('subcontractor_id', $request->subcontractor_id);
            }
            if ($request->filled('site_id')) {
                $subcontractorRosterQuery->where('site_id', $request->site_id);
            }
            if ($request->filled('project_id')) {
                $subcontractorRosterQuery->where('project_id', $request->project_id);
            }
            $subcontractorRosterAssigns = $subcontractorRosterQuery->get()
                ->groupBy(function($item) {
                    $scheduleDate = $item->schedule_date instanceof Carbon 
                        ? $item->schedule_date->format('Y-m-d') 
                        : Carbon::parse($item->schedule_date)->format('Y-m-d');
                    return $item->assign_to . '_' . $scheduleDate;
                });
        }
        
        // Merge projects (subcontractor employees use same projects)
        // If subcontractor_id is provided, only use subcontractor projects (empty for now, will be populated from attendances)
        $hasSubcontractorFilter = $request->filled('subcontractor_id');
        $allProjects = $hasSubcontractorFilter ? collect() : $internalProjects;
        
        // Merge roster assigns - manually merge grouped collections
        // If subcontractor_id is provided, only use subcontractor roster assigns
        $allRosterAssigns = $hasSubcontractorFilter ? collect() : $internalRosterAssigns;
        foreach ($subcontractorRosterAssigns as $key => $rosterAssigns) {
            if ($allRosterAssigns->has($key)) {
                // If key already exists, merge the roster assigns
                $allRosterAssigns[$key] = $allRosterAssigns[$key]->concat($rosterAssigns);
            } else {
                // If key doesn't exist, add new entry
                $allRosterAssigns[$key] = $rosterAssigns;
            }
        }
        
        // Merge project sites map
        // If subcontractor_id is provided, only use subcontractor project sites map
        $allProjectSitesMap = $hasSubcontractorFilter ? [] : $internalProjectSitesMap;
        
        if ($allEmployees->isEmpty()) {
            return $this->success([], 'No employees found');
        }
        
        // Chunk 4: Build timesheet data for all employees
        $timesheets = $this->buildTimesheetData(
            $request,
            $ids,
            $allEmployees,
            $allAttendances,
            $allProjects,
            $allRosterAssigns,
            $allProjectSitesMap,
            $userDateFormat
        );
        
        return $this->success($timesheets, 'Timesheets retrieved successfully');
    }

    /**
     * Sync timesheets to Xero for selected employees
     * Independent API endpoint for Xero timesheet synchronization
     * Accepts array of employees with id and employee_type (subcontractor_employee or regular)
     */
    public function syncTimesheetsToXero(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employees' => 'required|array|min:1',
            'employees.*.id' => 'required|integer',
            'employees.*.employee_type' => 'required|string|in:subcontractor_employee,regular',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $employeesData = $request->employees; // Array of {id, employee_type}
        $ids = $this->getCustomerAndWorkspaceIds();
        // Check if Xero token exists
        $xeroToken = XeroToken::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->first();
        if (!$xeroToken) {
            return $this->message('Please connect your Xero account first from settings', 400);
        }
        
        // Verify token is not expired (or can be refreshed)
        if ($xeroToken->isExpired()) {
            Log::info('Xero token is expired, attempting to refresh', [
                'token_id' => $xeroToken->id,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ]);
        }
        
        // Get XeroService with customer/workspace credentials
        $xeroService = new XeroService($ids['customer_id'], $ids['workspace_id']);
        
        // Verify XeroService has valid credentials
        try {
            $testToken = $xeroService->getValidAccessToken($ids['customer_id'], $ids['workspace_id']);
            if (!$testToken) {
                return $this->message('Xero authentication failed. Please reconnect your Xero account.', 400);
            }
        } catch (\Exception $e) {
            Log::error('Xero token validation failed', [
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
                'error' => $e->getMessage(),
            ]);
            return $this->message('Xero authentication error: ' . $e->getMessage() . '. Please reconnect your Xero account.', 400);
        }
        
        // Separate employees by type
        $subcontractorEmployeeIds = [];
        $regularEmployeeIds = [];
        
        foreach ($employeesData as $emp) {
            $empId = $emp['id'];
            
            if ($emp['employee_type'] === 'subcontractor_employee') {
                $subcontractorEmployeeIds[] = $empId;
            } else {
                $regularEmployeeIds[] = $empId;
            }
        }
        
        // Get employees with xero_emp_id from appropriate tables
        $employees = collect();
        
        // Fetch regular employees (internal)
        if (!empty($regularEmployeeIds)) {
            $regularEmployees = EmpCompanyDetails::whereIn('id', $regularEmployeeIds)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->whereNotNull('xero_emp_id')
                ->where('xero_emp_id', '!=', '')
                ->get();
            
            // Normalize regular employees - they already have employee_email
            foreach ($regularEmployees as $emp) {
                $employees->push($emp);
            }
        }
        
        // Fetch subcontractor employees (external)
        if (!empty($subcontractorEmployeeIds)) {
            $subcontractorEmployees = EmployeeSubcontractor::whereIn('id', $subcontractorEmployeeIds)
                ->whereNotNull('xero_emp_id')
                ->where('xero_emp_id', '!=', '')
                ->get();
            
            // Normalize subcontractor employees - map email to employee_email for consistency
            foreach ($subcontractorEmployees as $emp) {
                // Add employee_email property to the model instance for consistency
                $emp->employee_email = $emp->email;
                $employees->push($emp);
            }
        }
        
        // Key by id for easy lookup
        $employees = $employees->keyBy('id');
        
        if ($employees->isEmpty()) {
            return $this->message('No employees found with Xero Employee ID. Please sync employees to Xero first.', 404);
        }
        
        $syncedTimesheets = [];
        $failedTimesheets = [];
        // Process each employee
        foreach ($employeesData as $employeeData) {
            $empId = $employeeData['id'];
            // Check if employee exists and has xero_emp_id
            if (!isset($employees[$empId])) {
                $failedTimesheets[] = [
                    'employee_id' => $empId,
                    'employee_type' => $employeeData['employee_type'],
                    'error' => 'Employee not found or does not have Xero Employee ID',
                ];
                continue;
            }
            $employee = $employees[$empId];
            try {
                // Step 1-4: Validate and fetch Xero employee data, PayTemplate, and PayrollCalendar
                $xeroData = $this->validateAndFetchXeroEmployeeData($xeroService, $employee, $failedTimesheets, $ids['customer_id'], $ids['workspace_id']);
                if (!$xeroData) {
                    continue;
                }
                $xeroEmployee = $xeroData['xeroEmployee'];
                $payrollCalendar = $xeroData['payrollCalendar'];
                $payrollCalendarId = $xeroData['payrollCalendarId'];
                // Step 5: Calculate pay period dates
                $periodData = $this->calculatePayPeriodDates($xeroService, $employee, $payrollCalendar, $payrollCalendarId, $failedTimesheets);
                if (!$periodData) {
                    continue;
                }
                $startDate = $periodData['startDate'];
                $endDate = $periodData['endDate'];
                $startDateCarbon = $periodData['startDateCarbon'];
                $endDateCarbon = $periodData['endDateCarbon'];
                $useXeroDateStrings = $periodData['useXeroDateStrings'];
                $periodDates = $periodData['periodDates'];
                // Step 6-9: Prepare timesheet data
                $timesheetData = $this->prepareTimesheetData($xeroService, $employee, $xeroEmployee, $startDateCarbon, $endDateCarbon, $periodDates, $useXeroDateStrings, $ids, $failedTimesheets);
                if (!$timesheetData) {
                    continue;
                }
                $earningsRateId = $timesheetData['earningsRateId'];
                $dailyHours = $timesheetData['dailyHours'];
                $totalHours = $timesheetData['totalHours'];
                $startDateFormatted = $timesheetData['startDateFormatted'];
                $endDateFormatted = $timesheetData['endDateFormatted'];
                $startDateFormattedDb = $timesheetData['startDateFormattedDb'];
                $endDateFormattedDb = $timesheetData['endDateFormattedDb'];
                $existingTimesheet = $timesheetData['existingTimesheet'];
                $isUpdate = $timesheetData['isUpdate'];
                $xeroTimesheetId = $timesheetData['xeroTimesheetId'];
                // Step 10-11: Prepare payload and sync timesheet to Xero with retry logic
                $timesheetPayload = [
                    'EmployeeID' => $employee->xero_emp_id,
                    'StartDate' => $startDateFormatted,
                    'EndDate' => $endDateFormatted,
                    'Status' => 'DRAFT',
                    'TimesheetLines' => [
                        [
                            'EarningsRateID' => $earningsRateId,
                            'NumberOfUnits' => array_values($dailyHours), // Array of daily hours
                        ]
                    ],
                ];
                $syncResult = $this->syncTimesheetToXeroWithRetry($xeroService, $employee, $timesheetPayload, $isUpdate, $xeroTimesheetId, $existingTimesheet, $startDateFormattedDb, $endDateFormattedDb, $payrollCalendarId, $ids);
                $response = $syncResult['response'];
                $xeroTimesheetId = $syncResult['xeroTimesheetId'];
                $isUpdate = $syncResult['isUpdate'];
                
                // Determine emp_type and subcontractor_id
                $empType = $employeeData['employee_type'] === 'subcontractor_employee' ? 'external' : 'internal';
                $subcontractorId = null;
                if ($empType === 'external') {
                    // Get subcontractor_id from employees_subcontractors_metas
                    $employeeMeta = EmployeeSubcontractorMeta::where('emp_id', $employee->id)
                        ->where('active', 1)
                        ->first();
                    if ($employeeMeta) {
                        $subcontractorId = $employeeMeta->subcontractor_id;
                    }
                }
                
                // Step 12: Save or update xero_payslip_history table
                $this->saveTimesheetHistory($isUpdate, $existingTimesheet, $totalHours, $response, $xeroTimesheetId, $ids, $employee, $startDateFormattedDb, $endDateFormattedDb, $empType, $subcontractorId);
                $syncedTimesheets[] = [
                    'employee_id' => $employee->id,
                    'employee_type' => $employeeData['employee_type'],
                    'employee_email' => $employee->employee_email,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'start_date' => $startDateFormattedDb,
                    'end_date' => $endDateFormattedDb,
                    'daily_hours' => $dailyHours,
                    'total_hours' => $totalHours,
                    'xero_timesheet_id' => $xeroTimesheetId,
                    'xero_response' => $response,
                ];
                Log::info('Timesheet synced to Xero successfully and saved to history', [
                    'employee_id' => $employee->id,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'xero_timesheet_id' => $xeroTimesheetId,
                    'total_hours' => $totalHours,
                ]);
            } catch (\Exception $e) {
                $errorMessage = $e->getMessage();
                // Enhance error message for pay period validation errors
                if (strpos($errorMessage, "doesn't correspond with a pay period") !== false || 
                    strpos($errorMessage, "pay period") !== false) {
                    $errorMessage = "The current pay period doesn't align with Xero's requirements. " .
                                   "Please ensure the employee's Payroll Calendar is properly configured in Xero. " .
                                   "Original error: " . $errorMessage;
                }
                $failedTimesheets[] = [
                    'employee_id' => $employee->id,
                    'employee_type' => $employeeData['employee_type'],
                    'employee_email' => $employee->employee_email,
                    'xero_emp_id' => $employee->xero_emp_id ?? null,
                    'error' => $errorMessage,
                ];
                Log::error('Failed to sync timesheet to Xero', [
                    'employee_id' => $employee->id,
                    'xero_emp_id' => $employee->xero_emp_id ?? null,
                    'error' => $errorMessage,
                    'trace' => $e->getTraceAsString(),
                ]);
            }
        }
        // Prepare response data
        $responseData = [
            'synced_count' => count($syncedTimesheets),
            'failed_count' => count($failedTimesheets),
            'synced_timesheets' => $syncedTimesheets,
            'failed_timesheets' => $failedTimesheets,
        ];
        // Determine response based on results
        if (count($failedTimesheets) > 0 && count($syncedTimesheets) === 0) {
            // All timesheets failed - show the actual error message from the first failed timesheet
            $failedCount = count($failedTimesheets);
            $firstError = $failedTimesheets[0]['error'] ?? 'Unknown error occurred';
            $employeeEmail = $failedTimesheets[0]['employee_email'] ?? 'employee';
            // If there's only one failure, show the specific error
            if ($failedCount === 1) {
                $errorMessage = "Failed to sync timesheet for {$employeeEmail}. {$firstError}";
            } else {
                // If multiple failures, show the first error and mention there are more
                $errorMessage = "Failed to sync timesheets for {$failedCount} employees. First error: {$firstError}";
            }
            return $this->error($errorMessage, 400);
        } elseif (count($failedTimesheets) > 0) {
            // Partial success - show the actual error from first failed timesheet
            $syncedCount = count($syncedTimesheets);
            $failedCount = count($failedTimesheets);
            $firstError = $failedTimesheets[0]['error'] ?? 'Unknown error occurred';
            $employeeEmail = $failedTimesheets[0]['employee_email'] ?? 'employee';
            $errorMessage = "Timesheet sync partially completed. {$syncedCount} employee" . ($syncedCount > 1 ? 's' : '') . " synced successfully, but {$failedCount} employee" . ($failedCount > 1 ? 's' : '') . " failed. First error: {$employeeEmail} - {$firstError}";
            return $this->error($errorMessage, 207);
        } elseif (count($syncedTimesheets) > 0) {
            // All succeeded - use success() with data
            $syncedCount = count($syncedTimesheets);
            $successMessage = "Timesheets synced successfully to Xero. {$syncedCount} employee" . ($syncedCount > 1 ? 's' : '') . " processed.";
            return $this->success($responseData, $successMessage, 200);
        } else {
            // No employees processed (shouldn't happen, but handle it)
            return $this->error('No employees were selected for timesheet synchronization. Please select at least one employee and try again.', 400);
        }
    }

    public function getXeroTimesheetHistory(Request $request)
    {
        $xeroPayslipHistory = XeroPayslipHistory::query();
        $xeroPayslipHistory = $this->applyCustomerWorkspaceFilter($xeroPayslipHistory);
        $xeroPayslipHistory = $xeroPayslipHistory->latest()->get();
        // Load employee details based on emp_type
        $xeroPayslipHistory->each(function ($history) {
            if ($history->emp_type === 'external') {
                // Load subcontractor employee relationship (but hide from response)
                $history->load('subcontractorEmployee', 'subcontractor');
                // Add employee details to response - standardized structure
                if ($history->subcontractorEmployee) {
                    $history->employee_details = [
                        'id' => $history->subcontractorEmployee->id,
                        'first_name' => $history->subcontractorEmployee->first_name ?? null,
                        'middle_name' => $history->subcontractorEmployee->middle_name ?? null,
                        'last_name' => $history->subcontractorEmployee->last_name ?? null,
                        'email' => $history->subcontractorEmployee->email ?? null,
                        'phone' => $history->subcontractorEmployee->phone ?? null,
                        'profile_image' => $history->subcontractorEmployee->profile_image ? url($history->subcontractorEmployee->profile_image) : null,
                        'xero_emp_id' => $history->xero_emp_id ?? null,
                    ];
                } else {
                    // Set default structure even if employee not found
                    $history->employee_details = [
                        'id' => $history->employee_id,
                        'first_name' => null,
                        'middle_name' => null,
                        'last_name' => null,
                        'email' => null,
                        'phone' => null,
                        'profile_image' => null,
                        'xero_emp_id' => $history->xero_emp_id ?? null,
                    ];
                }
                // Add subcontractor details
                if ($history->subcontractor) {
                    $history->subcontractor_details = [
                        'id' => $history->subcontractor->id,
                        'name' => $history->subcontractor->company_name ?: $history->subcontractor->name,
                        'email' => $history->subcontractor->email,
                    ];
                }
                // Hide the full relationship objects from JSON response to avoid duplication
                $history->makeHidden(['subcontractorEmployee', 'subcontractor']);
            } else {
                // Load internal employee relationship (but hide from response)
                $history->load('employee');
                // Add employee details to response - standardized structure (same as external)
                if ($history->employee) {
                    // Load personal details
                    $history->employee->load('empPersonalDetails');
                    
                    $history->employee_details = [
                        'id' => $history->employee->id,
                        'first_name' => $history->employee->empPersonalDetails->first_name ?? null,
                        'middle_name' => $history->employee->empPersonalDetails->middle_name ?? null,
                        'last_name' => $history->employee->empPersonalDetails->last_name ?? null,
                        'email' => $history->employee->employee_email ?? null,
                        'phone' => $history->employee->empPersonalDetails->mobile ?? null,
                        'profile_image' => $history->employee->empPersonalDetails->image ? url($history->employee->empPersonalDetails->image) : null,
                        'xero_emp_id' => $history->employee->xero_emp_id ?? null,
                    ];
                } else {
                    // Set default structure even if employee not found
                    $history->employee_details = [
                        'id' => $history->employee_id,
                        'first_name' => null,
                        'middle_name' => null,
                        'last_name' => null,
                        'email' => null,
                        'phone' => null,
                        'profile_image' => null,
                        'xero_emp_id' => $history->xero_emp_id ?? null,
                    ];
                }
                // Hide the full relationship object from JSON response to avoid duplication
                $history->makeHidden(['employee']);
            }
        });
        
        return $this->success($xeroPayslipHistory);
    }
}
