<?php

namespace App\Http\Controllers\Traits;

use App\Models\Defect;
use App\Models\DefectImage;
use App\Models\DefectHistory;
use App\Models\EmpCompanyDetails;
use App\Models\SubcontractorCompany;
use App\Models\Adminsettings;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

trait DefectTrait
{
    // ---------- Email helpers ----------
    protected function sendDefectEmailToAssignee($defect, $subjectPrefix = 'Defect Update')
    {
        try {
            $ids = $this->getCustomerAndWorkspaceIds();
            // Collect recipients based on assignment type
            $recipients = [];
            if ($defect->assignment_type === 'internal_employee' && $defect->assigned_emp_id) {
                $emp = \App\Models\EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->find($defect->assigned_emp_id);
                if ($emp && !empty($emp->employee_email)) {
                    $nameModel = \App\Models\EmpPersonalDetails::where('emp_id', $defect->assigned_emp_id)->first();
                    $name = $nameModel ? trim(($nameModel->first_name ?? '') . ' ' . ($nameModel->last_name ?? '')) : null;
                    $recipients[] = ['email' => $emp->employee_email, 'name' => $name];
                }
            } elseif ($defect->assignment_type === 'subcontractor' && $defect->subcontractor_id) {
                // subcontractor_id directly references users.id (subcontractor user)
                $sub = \App\Models\User::find($defect->subcontractor_id);
                if ($sub && !empty($sub->email)) {
                    $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
                }
            } elseif ($defect->assignment_type === 'subcontractor_employee') {
                // Subcontractor
                if ($defect->subcontractor_id) {
                    // subcontractor_id directly references users.id (subcontractor user)
                    $sub = \App\Models\User::find($defect->subcontractor_id);
                    if ($sub && !empty($sub->email)) {
                        $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
                    }
                }
                // Assigned subcontractor employee
                if ($defect->assigned_subcontractor_emp_id) {
                    $emp = \App\Models\EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->find($defect->assigned_subcontractor_emp_id);
                    if ($emp && !empty($emp->employee_email)) {
                        $nameModel = \App\Models\EmpPersonalDetails::where('emp_id', $defect->assigned_subcontractor_emp_id)->first();
                        $name = $nameModel ? trim(($nameModel->first_name ?? '') . ' ' . ($nameModel->last_name ?? '')) : null;
                        $recipients[] = ['email' => $emp->employee_email, 'name' => $name];
                    }
                }
            }
            if (empty($recipients)) {
                return; // no email to send
            }
            $statusMap = config('constants.defect_status');
            $statusLabel = collect($statusMap)->firstWhere('key', $defect->status)['value'] ?? 'Unknown';
            $subject = $subjectPrefix . ' | ' . ' - ' . $statusLabel . ' | ' . env('APP_NAME');
            // Prepare email data based on status
            $emailData = [
                'subject' => $subjectPrefix,
                'defect_title' => $defect->title,
                'defect_description' => $defect->description,
                'defect_priority' => $defect->priority,
                'defect_status_label' => $statusLabel,
                'defect_due_date' => $defect->due_date ? \Carbon\Carbon::parse($defect->due_date)->format('d M Y') : null,
                'action_url' => env('FRONTEND_URL') ? rtrim(env('FRONTEND_URL')) : null,
                'customer_id' => $ids['customer_id'] ?? null,
                'reject_reason' => $defect->reject_reason ?? null,
            ];
            // Add status-specific data
            if ($defect->status == config('constants.defect_status.rejected.key')) {
                // When rejected, show reject reason to assignee
                $emailData['reject_reason'] = $defect->reject_reason ?? null;
            }
            foreach ($recipients as $recipient) {
                $emailData['assignee_name'] = $recipient['name'] ?? null;
                $params = [
                    'to' => $recipient['email'],
                    'subject' => $subject,
                            'msg' => view('Emails.defect-assignee', $emailData + ['customer_id' => $defect->customer_id ?? null])->render(),
                ];
                $this->SendInstantEmail($params);
            }
        } catch (\Throwable $e) {
            Log::error('Failed to send defect email to assignee: ' . $e->getMessage());
        }
    }

    protected function sendDefectEmailToCreator($defect, $subjectPrefix = 'Defect Update')
    {
        try {
            $ids = $this->getCustomerAndWorkspaceIds();
            $creatorEmail = null;
            $creatorName = null;
            if ($defect->created_by_type === 'employee') {
                $emp = \App\Models\EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->find($defect->created_by);
                $creatorEmail = $emp->employee_email ?? null;
                $nameModel = \App\Models\EmpPersonalDetails::where('emp_id', $defect->created_by)->first();
                $creatorName = $nameModel ? trim(($nameModel->first_name ?? '') . ' ' . ($nameModel->last_name ?? '')) : null;
            } elseif ($defect->created_by_type === 'subcontractor' || $defect->created_by_type === 'customer') {
                // Both subcontractors and customers are User models
                $user = \App\Models\User::find($defect->created_by);
                $creatorEmail = $user->email ?? null;
                $creatorName = $user->name ?? null;
            }
            if (!$creatorEmail) {
                return;
            }
            $statusMap = config('constants.defect_status');
            $statusLabel = collect($statusMap)->firstWhere('key', $defect->status)['value'] ?? 'Unknown';
            $subject = $subjectPrefix . ' | ' . ' - ' . $statusLabel . ' | ' . env('APP_NAME');
            // Prepare email data based on status
            $emailData = [
                'subject' => $subjectPrefix,
                'creator_name' => $creatorName,
                'defect_title' => $defect->title,
                'defect_description' => $defect->description,
                'defect_priority' => $defect->priority,
                'defect_status_label' => $statusLabel,
                'defect_due_date' => $defect->due_date ? \Carbon\Carbon::parse($defect->due_date)->format('d M Y') : null,
                'action_url' => env('FRONTEND_URL') ? rtrim(env('FRONTEND_URL')) : null,
                'customer_id' => $ids['customer_id'] ?? null,
            ];
            // Add status-specific data
            if ($defect->status == config('constants.defect_status.submitted.key')) {
                // When submitted, show completion description to creator
                $emailData['completion_description'] = $defect->completion_description ?? null;
            } elseif ($defect->status == config('constants.defect_status.rejected.key')) {
                // When rejected, show reject reason to creator
                $emailData['reject_reason'] = $defect->reject_reason ?? null;
            }
            $params = [
                'to' => $creatorEmail,
                'subject' => $subject,
                'msg' => view('Emails.defect-creator', $emailData + ['customer_id' => $defect->customer_id ?? null])->render(),
            ];
            $this->SendInstantEmail($params);
        } catch (\Throwable $e) {
            Log::error('Failed to send defect email to creator: ' . $e->getMessage());
        }
    }
    /**
     * Send email to customer/admin who owns the defect
     * This is called when subcontractors update defects to notify the customer/admin
     */
    protected function sendDefectEmailToCustomer($defect, $subjectPrefix = 'Defect Update')
    {
        try {
            // Get customer/admin who owns the defect (customer_id in defects table)
            if (!$defect->customer_id) {
                return; // No customer to notify
            }
            
            $customer = \App\Models\User::find($defect->customer_id);
            if (!$customer || empty($customer->email)) {
                Log::warning('Customer not found or no email for defect customer notification', [
                    'defect_id' => $defect->id,
                    'customer_id' => $defect->customer_id
                ]);
                return;
            }
            
            $statusMap = config('constants.defect_status');
            $statusLabel = collect($statusMap)->firstWhere('key', $defect->status)['value'] ?? 'Unknown';
            $subject = $subjectPrefix . ' | ' . ' - ' . $statusLabel . ' | ' . env('APP_NAME');
            
            // Prepare email data (using creator_name as template expects it)
            $emailData = [
                'subject' => $subjectPrefix,
                'creator_name' => $customer->name ?? 'there',
                'defect_title' => $defect->title,
                'defect_description' => $defect->description,
                'defect_priority' => $defect->priority,
                'defect_status_label' => $statusLabel,
                'defect_due_date' => $defect->due_date ? \Carbon\Carbon::parse($defect->due_date)->format('d M Y') : null,
                'action_url' => env('FRONTEND_URL') ? rtrim(env('FRONTEND_URL')) : null,
                'customer_id' => $defect->customer_id,
            ];
            
            // Add status-specific data
            if ($defect->status == config('constants.defect_status.submitted.key')) {
                // When submitted, show completion description
                $emailData['completion_description'] = $defect->completion_description ?? null;
            } elseif ($defect->status == config('constants.defect_status.rejected.key')) {
                // When rejected, show reject reason
                $emailData['reject_reason'] = $defect->reject_reason ?? null;
            }
            
            $params = [
                'to' => $customer->email,
                'subject' => $subject,
                'msg' => view('Emails.defect-creator', $emailData)->render(),
            ];
            
            $this->SendInstantEmail($params);
            Log::info('Defect email sent to customer/admin', [
                'defect_id' => $defect->id,
                'customer_email' => $customer->email,
                'subject_prefix' => $subjectPrefix
            ]);
        } catch (\Throwable $e) {
            Log::error('Failed to send defect email to customer/admin: ' . $e->getMessage(), [
                'defect_id' => $defect->id ?? null,
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    protected function defectValidation(Request $request, $isUpdate = false)
    {
        $rules = [
            'project_id' => 'required|integer|exists:projects,id',
            'site_id' => 'required|integer|exists:sites,id',
            'assignment_type' => 'required|in:internal_employee,subcontractor,subcontractor_employee',
            'assigned_emp_id' => 'nullable|integer|exists:emp_company_details,id|required_if:assignment_type,internal_employee',
            'subcontractor_id' => 'nullable|integer|exists:users,id|required_if:assignment_type,subcontractor',
            'assigned_subcontractor_emp_id' => 'nullable|integer|required_if:assignment_type,subcontractor_employee',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'priority' => 'required|in:low,medium,high,critical',
            'due_date' => 'nullable|date|after_or_equal:today',
            'images.*' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
        ];
        if ($isUpdate) {
            $rules['id'] = 'required|integer|exists:defects,id';
        }
        // Add conditional validation based on assignment type
        $validator = Validator::make($request->all(), $rules);
        $validator->after(function ($validator) use ($request) {
            $assignmentType = $request->assignment_type;
            switch ($assignmentType) {
                case 'internal_employee':
                    if (!$request->assigned_emp_id) {
                        $validator->errors()->add('assigned_emp_id', 'Employee ID is required for internal employee assignment.');
                    }
                    break;
                case 'subcontractor':
                    if (!$request->subcontractor_id) {
                        $validator->errors()->add('subcontractor_id', 'Subcontractor ID is required for subcontractor assignment.');
                    }
                    break;
                case 'subcontractor_employee':
                    if (!$request->subcontractor_id) {
                        $validator->errors()->add('subcontractor_id', 'Subcontractor ID is required for subcontractor employee assignment.');
                    }
                    if (!$request->assigned_subcontractor_emp_id) {
                        $validator->errors()->add('assigned_subcontractor_emp_id', 'Subcontractor employee ID is required for subcontractor employee assignment.');
                    }
                    break;
            }
        });
        return $validator;
    }

    protected function getDefects(Request $request)
    {
        $filters = $request->only(['project_id', 'site_id', 'assigned_emp_id', 'priority', 'status', 'title']);
        $ids = $this->getCustomerAndWorkspaceIds();
        $query = Defect::where('del', '0');
        $user = auth()->user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        
        // For subcontractors: Don't apply customer/workspace filter initially (will filter by assigned defects)
        // For others: Apply customer/workspace filter
        if (!$isSubcontractor) {
            $query = $this->applyCustomerWorkspaceFilter($query);
        }
        
        $query = $query->with([
            'project',
            'site',
            'subcontractor',
        ]);
        
        // Check if user is customer, employee, or subcontractor
        if ($isSubcontractor) {
            // For subcontractor users: Show defects they created AND defects assigned to them
            // Get all company associations for this subcontractor (across all companies)
            $subcontractorCompanies = SubcontractorCompany::where('user_id', $user->id)
                ->where('del', '0')
                ->get();
            
            // Collect all defect_ids from all company associations
            $allDefectIds = [];
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                $allDefectIds = array_merge($allDefectIds, $defectIds);
            }
            
            // Remove duplicates
            $allDefectIds = array_unique($allDefectIds);
            
            $query->where(function ($q) use ($user, $allDefectIds) {
                // Defects created by this subcontractor
                $q->where(function ($subQ) use ($user) {
                    $subQ->where('created_by', $user->id)
                        ->where('created_by_type', 'subcontractor');
                });
                
                // OR defects assigned to this subcontractor (from defect_ids array across all companies)
                if (!empty($allDefectIds)) {
                    $q->orWhereIn('id', $allDefectIds);
                }
            });
        } elseif ($userTable === 'customer') {
            // For regular customer users: Show all defects under their customer_id (primary key id)
            $query->where('customer_id', $user->id);
        } else {
            // For employee users
            $query->where(function ($q) use ($user) {
                $q->where('created_by', $user->id)
                    ->where('created_by_type', 'employee')
                    ->orWhere('assigned_emp_id', $user->id)
                    ->orWhere('assigned_subcontractor_emp_id', $user->id);
            });
        }
        // Apply additional filters
        $query = $query
    
            ->when($filters['project_id'] ?? null, function ($q, $projectId) {
                $q->where('project_id', $projectId);
            })
            ->when($filters['site_id'] ?? null, function ($q, $siteId) {
                $q->where('site_id', $siteId);
            })
            // ->when($filters['assigned_emp_id'] ?? null, function ($q, $empId) {
            //     $q->where('assigned_emp_id', $empId);
            // })
            ->when($filters['priority'] ?? null, function ($q, $priority) {
                $q->where('priority', $priority);
            })
            ->when(isset($filters['status']), function ($q) use ($filters) {
                $q->where('status', $filters['status']);
            })
            ->when($filters['title'] ?? null, function ($q, $title) {
                $q->where('title', 'like', '%' . $title . '%');
            })
            ->orderBy('created_at', 'desc');
        $data = $query->get();
        
        // Get tier permissions for the logged-in user
        $tierPermissions = [];
        
        // Check if user is employee
        $isEmployee = $ids['flag'] === 'emp' || $this->getUserTable() === 'emp';
        
        if ($isEmployee) {
            $employeeId = Auth::id();
            $employee = \App\Models\EmpCompanyDetails::find($employeeId);
            
            if ($employee) {
                // Get "Site" module permissions using the global function
                $siteModulePermission = $this->getTierPermissionsByModule($employee, 'Site Defects');
                if ($siteModulePermission) {
                    $tierPermissions = [$siteModulePermission];
                }
            }
            
            // If employee and no data found, return success with empty data and null tierPermissions
            if ($data->isEmpty()) {
                $responseData = [
                    'tier_permissions' => $tierPermissions,
                ];
                return $this->success($responseData, 'Defects Not Found');
            }
        }
        
        // Get the original response structure
        $response = $this->success($data, 'Defects retrieved successfully');
        
        // Add tier_permissions to the response data
        $responseData = $response->getData(true);
        $responseData['tier_permissions'] = $tierPermissions;
        
        return response()->json($responseData, $response->getStatusCode());
    }

    protected function createDefect(Request $request)
    {
        $validator = $this->defectValidation($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $data = $validator->validated();
        $data['customer_id'] = $ids['customer_id'];
        $data['workspace_id'] = $ids['workspace_id'];
        
        // Fix creator information based on user type
        $user = Auth::user();
        $userTable = $this->getUserTable();
        
        if ($ids['flag'] === 'emp') {
            $data['created_by'] = Auth::id();
            $data['created_by_type'] = 'employee';
        } elseif ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
            // For subcontractor users
            $data['created_by'] = Auth::id();
            $data['created_by_type'] = 'subcontractor';
        } else {
            // For regular customer users
            $data['created_by'] = Auth::id();
            $data['created_by_type'] = 'customer';
        }
        $defect = Defect::create($data);
        // Handle image uploads if provided during creation
        if ($request->hasFile('images')) {
            // Determine image type based on creator type
            if ($ids['flag'] === 'emp') {
                $type = 'created_by_employee';
            } elseif ($data['created_by_type'] === 'subcontractor') {
                $type = 'created_by_subcontractor';
            } else {
                $type = 'created_by_customer';
            }
            $uploadedBy = Auth::id();
            foreach ($request->file('images') as $imageFile) {
                if ($imageFile->isValid()) {
                    // Generate unique filename
                    $extension = strtolower($imageFile->getClientOriginalExtension());
                    $originalName = pathinfo($imageFile->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = preg_replace('/[^a-zA-Z0-9_-]/', '', $originalName);
                    $sanitizedFileName = substr($sanitizedFileName, 0, 100);
                    $uniqueFileName = time() . '_' . uniqid() . '_' . $sanitizedFileName . '.' . $extension;
                    // Create directory if it doesn't exist
                    $directory = public_path('DefectsManagement');
                    if (!file_exists($directory)) {
                        mkdir($directory, 0777, true);
                    }
                    // Move uploaded file
                    $imageFile->move($directory, $uniqueFileName);
                    $imagePath = 'DefectsManagement/' . $uniqueFileName;
                    // Save to database
                    DefectImage::create([
                        'defect_id' => $defect->id,
                        'uploaded_by' => $uploadedBy,
                        'image' => $imagePath,
                        'type' => $type,
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                }
            }
        }

        $this->logDefectHistory($defect->id, 'created', 'Defect created', null, $defect->toArray());
        // Sync defect assignment to subcontractor_companies if assigned to subcontractor
        $this->syncDefectToSubcontractorCompany($defect);
        // Email notifications: on create send to assignee
        $this->sendDefectEmailToAssignee($defect, 'New Defect Assigned');
        return $this->success($defect, 'Defect created successfully');
    }

    protected function getDefectById($id, $ids)
    {
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        
        $query = Defect::with([
            'project',
            'site',
            'employeePersonalDetails',
            'images',
            'histories',
            'subcontractor',
            'assignedSubcontractorEmployeePersonalDetails'
        ])
            ->where('id', $id);
        
        // For subcontractors: Don't filter by customer/workspace initially (will check access after fetching)
        // For others: Apply customer/workspace filter
        if (!$isSubcontractor) {
            $query->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);
        }
        
        $defect = $query->first();
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        
        // Authorization check: Subcontractors can only view defects they created or are assigned to
        if ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            // If not created by subcontractor, check if defect is assigned to them
            if (!$isCreator) {
                // Use defect's customer_id and workspace_id (not current user's context)
                $subcontractorCompany = SubcontractorCompany::where('user_id', $user->id)
                    ->where('customer_id', $defect->customer_id)
                    ->where('workspace_id', $defect->workspace_id)
                    ->where('del', '0')
                    ->first();
                
                $defectIds = $subcontractorCompany ? ($subcontractorCompany->defect_ids ?? []) : [];
                $isAssigned = in_array($defect->id, $defectIds);
                
                if (!$isAssigned) {
                    return $this->error('You do not have permission to view this defect', 403);
                }
            }
        }
        // Add creator information and clean up duplicates
        if ($defect->created_by_type === 'employee') {
            $defect->creator_info = $defect->createdByEmployee;
            $defect->creator_type = 'employee';
            // Remove the duplicate relation data
            unset($defect->created_by_employee);
        } elseif ($defect->created_by_type === 'subcontractor') {
            // Subcontractors are User models, so use createdByCustomer relationship
            $defect->creator_info = $defect->createdByCustomer;
            $defect->creator_type = 'subcontractor';
            // Remove the duplicate relation data
            unset($defect->created_by_customer);
        } else {
            $defect->creator_info = $defect->createdByCustomer;
            $defect->creator_type = 'customer';
            // Remove the duplicate relation data
            unset($defect->created_by_customer);
        }
        
        // Process histories to update creator_info in old_value and new_value based on changed_by_type
        if ($defect->relationLoaded('histories') && $defect->histories) {
            $defect->histories = $defect->histories->map(function ($historyItem) {
                // Update creator_info in old_value and new_value based on who changed it
                if ($historyItem->changed_by_type === 'subcontractor' && $historyItem->changed_by) {
                    // Get subcontractor info
                    $subcontractor = \App\Models\User::where('id', $historyItem->changed_by)
                        ->select('id', 'name', 'email', 'company_name')
                        ->first();
                    
                    if ($subcontractor) {
                        $changerInfo = [
                            'id' => $subcontractor->id,
                            'name' => $subcontractor->name ?? $subcontractor->company_name ?? null,
                            'email' => $subcontractor->email ?? null,
                        ];
                        
                        // Update old_value if it exists - need to get array, modify, then set back
                        if ($historyItem->old_value && is_array($historyItem->old_value)) {
                            $oldValue = $historyItem->old_value;
                            $oldValue['creator_info'] = $changerInfo;
                            $oldValue['creator_type'] = 'subcontractor';
                            $historyItem->old_value = $oldValue;
                        }
                        
                        // Update new_value if it exists - need to get array, modify, then set back
                        if ($historyItem->new_value && is_array($historyItem->new_value)) {
                            $newValue = $historyItem->new_value;
                            $newValue['creator_info'] = $changerInfo;
                            $newValue['creator_type'] = 'subcontractor';
                            $historyItem->new_value = $newValue;
                        }
                    }
                } elseif ($historyItem->changed_by_type === 'employee' && $historyItem->changed_by) {
                    // Get employee info
                    $employee = \App\Models\EmpPersonalDetails::where('emp_id', $historyItem->changed_by)
                        ->select('emp_id', 'first_name', 'middle_name', 'last_name')
                        ->first();
                    
                    if ($employee) {
                        $fullName = trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? ''));
                        $changerInfo = [
                            'id' => $employee->emp_id,
                            'name' => $fullName ?: null,
                            'first_name' => $employee->first_name ?? null,
                            'middle_name' => $employee->middle_name ?? null,
                            'last_name' => $employee->last_name ?? null,
                        ];
                        
                        // Update old_value if it exists
                        if ($historyItem->old_value && is_array($historyItem->old_value)) {
                            $oldValue = $historyItem->old_value;
                            $oldValue['creator_info'] = $changerInfo;
                            $oldValue['creator_type'] = 'employee';
                            $historyItem->old_value = $oldValue;
                        }
                        
                        // Update new_value if it exists
                        if ($historyItem->new_value && is_array($historyItem->new_value)) {
                            $newValue = $historyItem->new_value;
                            $newValue['creator_info'] = $changerInfo;
                            $newValue['creator_type'] = 'employee';
                            $historyItem->new_value = $newValue;
                        }
                    }
                } elseif ($historyItem->changed_by_type === 'customer' && $historyItem->changed_by) {
                    // Get customer info
                    $customer = \App\Models\User::where('id', $historyItem->changed_by)
                        ->select('id', 'name', 'email', 'company_name')
                        ->first();
                    
                    if ($customer) {
                        $changerInfo = [
                            'id' => $customer->id,
                            'name' => $customer->name ?? $customer->company_name ?? null,
                            'email' => $customer->email ?? null,
                        ];
                        
                        // Update old_value if it exists
                        if ($historyItem->old_value && is_array($historyItem->old_value)) {
                            $oldValue = $historyItem->old_value;
                            $oldValue['creator_info'] = $changerInfo;
                            $oldValue['creator_type'] = 'customer';
                            $historyItem->old_value = $oldValue;
                        }
                        
                        // Update new_value if it exists
                        if ($historyItem->new_value && is_array($historyItem->new_value)) {
                            $newValue = $historyItem->new_value;
                            $newValue['creator_info'] = $changerInfo;
                            $newValue['creator_type'] = 'customer';
                            $historyItem->new_value = $newValue;
                        }
                    }
                }
                
                return $historyItem;
            });
        }
        
        return $this->success($defect, 'Defect retrieved successfully');
    }

    protected function updateDefect(Request $request)
    {
        $validator = $this->defectValidation($request, true);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $data = $validator->validated();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        
        $query = Defect::where('id', $data['id']);
        
        // For subcontractors: Don't filter by customer/workspace initially (will check access after fetching)
        // For others: Apply customer/workspace filter
        if (!$isSubcontractor) {
            $query->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);
        }
        
        $defect = $query->first();
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        
        // Authorization check: Subcontractors can only update defects they created or defects assigned to them
        if ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            // If not created by subcontractor, check if defect is assigned to them
            if (!$isCreator) {
                // Use defect's customer_id and workspace_id (not current user's context)
                $subcontractorCompany = SubcontractorCompany::where('user_id', $user->id)
                    ->where('customer_id', $defect->customer_id)
                    ->where('workspace_id', $defect->workspace_id)
                    ->where('del', '0')
                    ->first();
                
                $defectIds = $subcontractorCompany ? ($subcontractorCompany->defect_ids ?? []) : [];
                $isAssigned = in_array($defect->id, $defectIds);
                
                if (!$isAssigned) {
                    return $this->error('You do not have permission to update this defect', 403);
                }
            }
        }
        
        $old = $defect->toArray();
        $oldSubcontractorId = $defect->subcontractor_id;
        $oldAssignmentType = $defect->assignment_type;

        // Don't update created_by and created_by_type on update - keep original values
        unset($data['created_by'], $data['created_by_type']);

        $defect->update($data);
        
        // Refresh defect to get updated values
        $defect->refresh();
        
        // Check if assignment changed from/to subcontractor
        $assignmentChanged = ($oldAssignmentType !== $defect->assignment_type) || 
                            ($oldSubcontractorId !== $defect->subcontractor_id);
        
        if ($assignmentChanged) {
            // If old assignment was to subcontractor, remove from old subcontractor
            if (in_array($oldAssignmentType, ['subcontractor', 'subcontractor_employee']) && $oldSubcontractorId) {
                $this->removeDefectFromSubcontractorCompany($defect->id, $oldSubcontractorId, $defect->customer_id, $defect->workspace_id);
            }
            
            // If new assignment is to subcontractor, add to new subcontractor
            if (in_array($defect->assignment_type, ['subcontractor', 'subcontractor_employee'])) {
                $this->syncDefectToSubcontractorCompany($defect);
            }
        } else {
            // Even if assignment didn't change, ensure defect is synced (handles cases where defect was created before this feature)
            if (in_array($defect->assignment_type, ['subcontractor', 'subcontractor_employee'])) {
                $this->syncDefectToSubcontractorCompany($defect);
            }
        }

        // Handle image uploads if provided during update
        if ($request->hasFile('images')) {
            // Delete existing images (both from database and physical files)
            $this->deleteExistingDefectImages($defect->id, $ids);
            
            // Determine image type based on current user type
            $user = Auth::user();
            $userTable = $this->getUserTable();
            if ($ids['flag'] === 'emp') {
                $type = 'created_by_employee';
            } elseif ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
                $type = 'created_by_subcontractor';
            } else {
                $type = 'created_by_customer';
            }
            $uploadedBy = Auth::id();

            foreach ($request->file('images') as $imageFile) {
                if ($imageFile->isValid()) {
                    // Generate unique filename
                    $extension = strtolower($imageFile->getClientOriginalExtension());
                    $originalName = pathinfo($imageFile->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = preg_replace('/[^a-zA-Z0-9_-]/', '', $originalName);
                    $sanitizedFileName = substr($sanitizedFileName, 0, 100);
                    $uniqueFileName = time() . '_' . uniqid() . '_' . $sanitizedFileName . '.' . $extension;

                    // Create directory if it doesn't exist
                    $directory = public_path('DefectsManagement');
                    if (!file_exists($directory)) {
                        mkdir($directory, 0777, true);
                    }

                    // Move uploaded file
                    $imageFile->move($directory, $uniqueFileName);
                    $imagePath = 'DefectsManagement/' . $uniqueFileName;

                    // Save to database
                    DefectImage::create([
                        'defect_id' => $defect->id,
                        'uploaded_by' => $uploadedBy,
                        'image' => $imagePath,
                        'type' => $type,
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                }
            }
        }

        $this->logDefectHistory($defect->id, 'updated', 'Defect updated', $old, $defect->toArray());
        // Notify creator about update
        $this->sendDefectEmailToCreator($defect, 'Defect Updated');
        return $this->success($defect, 'Defect updated successfully');
    }

    protected function deleteDefect($id, $ids)
    {
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        
        $query = Defect::where('id', $id)
            ->where('del', '0');
        
        // For subcontractors: Don't filter by customer/workspace initially (will check access after fetching)
        // For others: Apply customer/workspace filter
        if (!$isSubcontractor) {
            $query->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);
        }
        
        $defect = $query->first();
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        
        // Authorization check: Subcontractors can only delete defects they created
        if ($isSubcontractor) {
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            if (!$isCreator) {
                return $this->error('You can only delete defects you created', 403);
            }
        }
        
        // Remove defect from subcontractor_companies if assigned to subcontractor
        // Use defect's customer_id and workspace_id (not current user's context)
        if (in_array($defect->assignment_type, ['subcontractor', 'subcontractor_employee']) && $defect->subcontractor_id) {
            $this->removeDefectFromSubcontractorCompany($defect->id, $defect->subcontractor_id, $defect->customer_id, $defect->workspace_id);
        }
        
        $defect->del = 1;
        $defect->save();
        $this->logDefectHistory($defect->id, 'deleted', 'Defect deleted', $defect->toArray(), null);
        return $this->message('Defect deleted successfully');
    }

  

    protected function deleteDefectImages($id, $ids)
    {
        $defect = DefectImage::where('id', $id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();
        if (!$defect) {
            return $this->error('Defect image not found', 404);
        }
        if ($defect->defect->status == config('constants.defect_status.submitted.key')) {
            return $this->error('Defect is submitted, cannot delete images', 400);
        }
        if ($defect->defect->status == config('constants.defect_status.approved.key')) {
            return $this->error('Defect is approved, cannot delete images', 400);
        }
        if ($defect->defect->status == config('constants.defect_status.rejected.key')) {
            return $this->error('Defect is rejected, cannot delete images', 400);
        }
        // if ($defect->defect()->status == config('constants.defect_status.completed.key')) {
        //     return $this->error('Defect is completed, cannot delete images', 400);
        // }
        
        // Delete physical file before deleting database record
        $this->deletePhysicalImageFile($defect->image);
        
        $defect->delete();
        return $this->message('Defect image deleted successfully');
    }

    /**
     * Delete all existing images for a defect (both database records and physical files)
     */
    protected function deleteExistingDefectImages($defectId, $ids)
    {
        $existingImages = DefectImage::where('defect_id', $defectId)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->get();

        foreach ($existingImages as $image) {
            // Delete physical file
            $this->deletePhysicalImageFile($image->image);
            
            // Delete database record
            $image->delete();
        }
    }

    /**
     * Delete physical image file from storage
     */
    protected function deletePhysicalImageFile($imagePath)
    {
        if ($imagePath && file_exists(public_path($imagePath))) {
            unlink(public_path($imagePath));
        }
    }

    protected function submitCompletionDefect(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:defects,id',
            'completion_description' => 'nullable|string',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        
        // First, fetch the defect to get its customer_id and workspace_id
        $defect = Defect::where('id', $request->id)
            ->where('status', config('constants.defect_status.in_progress.key')) // in_progress
            ->first();
        
        if (!$defect) {
            return $this->error('Defect not found or not in in-progress status', 404);
        }
        
        // Check access based on user type
        if ($userTable === 'customer' && $user->user_type == config('constants.user_types.subcontractor')) {
            // For subcontractor users: Check if defect is assigned to them OR created by them
            // First, check if defect was created by this subcontractor
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            // If not created by subcontractor, check if defect is assigned to them
            if (!$isCreator) {
                // Use defect's customer_id and workspace_id (not current user's context)
                $subcontractorCompany = SubcontractorCompany::where('user_id', $user->id)
                    ->where('customer_id', $defect->customer_id)
                    ->where('workspace_id', $defect->workspace_id)
                    ->where('del', '0')
                    ->first();
                
                if (!$subcontractorCompany) {
                    return $this->error('Subcontractor company record not found', 404);
                }
                
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                if (empty($defectIds) || !in_array($request->id, $defectIds)) {
                    return $this->error('Defect not assigned to this subcontractor and you did not create it', 403);
                }
            }
            
            // Defect is either created by subcontractor or assigned to them, continue (defect already fetched)
        } else {
            // For employees or regular customers: verify defect belongs to their customer/workspace
            if ($defect->customer_id != $ids['customer_id'] || $defect->workspace_id != $ids['workspace_id']) {
                return $this->error('Defect not found', 404);
            }
            
            // Check if user has access (assigned to them or created by them)
            $hasAccess = ($defect->assigned_emp_id == Auth::id()) ||
                        ($defect->assigned_subcontractor_emp_id == Auth::id()) ||
                        ($defect->created_by == Auth::id());
            
            if (!$hasAccess) {
                return $this->error('You do not have permission to submit completion for this defect', 403);
            }
        }
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        $old = $defect->toArray();
        $defect->completion_description = $request->completion_description;
        $defect->status = config('constants.defect_status.submitted.key'); // submitted
        $defect->completed_at = now();
        $defect->save();
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $imageFile) {
                if ($imageFile->isValid()) {
                    // Generate unique filename
                    $extension = strtolower($imageFile->getClientOriginalExtension());
                    $originalName = pathinfo($imageFile->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = preg_replace('/[^a-zA-Z0-9_-]/', '', $originalName);
                    $sanitizedFileName = substr($sanitizedFileName, 0, 100);
                    $uniqueFileName = time() . '_' . uniqid() . '_' . $sanitizedFileName . '.' . $extension;

                    // Create directory if it doesn't exist
                    $directory = public_path('DefectsManagement');
                    if (!file_exists($directory)) {
                        mkdir($directory, 0777, true);
                    }

                    // Move uploaded file
                    $imageFile->move($directory, $uniqueFileName);
                    $imagePath = 'DefectsManagement/' . $uniqueFileName;

                    DefectImage::create([
                        'defect_id' => $defect->id,
                        'uploaded_by' => Auth::id(),
                        'image' => $imagePath,
                        'type' => 'assigned_employee',
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                }
            }
        }
        $this->logDefectHistory($defect->id, 'submitted', 'Task submitted by employee', $old, $defect->toArray());
        
        // Email: notify creator on submitted
        $this->sendDefectEmailToCreator($defect, 'Defect Submitted');
        
        // If submitted by subcontractor and assigned to subcontractor, also notify customer/admin
        if ($userTable === 'customer' && $user->user_type == config('constants.user_types.subcontractor') && 
            in_array($defect->assignment_type, ['subcontractor', 'subcontractor_employee'])) {
            $this->sendDefectEmailToCustomer($defect, 'Defect Submitted');
        }

        return $this->success($defect, 'Task submitted successfully');
    }

    protected function approveDefect(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:defects,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        
        // First, fetch the defect to get its customer_id and workspace_id
        $defect = Defect::where('id', $request->id)
            ->where('del', '0')
            ->first();
        
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        
        // Check access based on user type
        if ($isSubcontractor) {
            // For subcontractor users: Check if defect is assigned to them OR created by them
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            // If not created by subcontractor, check if defect is assigned to them
            if (!$isCreator) {
                // Use defect's customer_id and workspace_id (not current user's context)
                $subcontractorCompany = SubcontractorCompany::where('user_id', $user->id)
                    ->where('customer_id', $defect->customer_id)
                    ->where('workspace_id', $defect->workspace_id)
                    ->where('del', '0')
                    ->first();
                
                if (!$subcontractorCompany) {
                    return $this->error('Subcontractor company record not found', 404);
                }
                
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                if (empty($defectIds) || !in_array($request->id, $defectIds)) {
                    return $this->error('Defect not assigned to this subcontractor and you did not create it', 403);
                }
            }
        } else {
            // For employees or regular customers: verify defect belongs to their customer/workspace
            if ($defect->customer_id != $ids['customer_id'] || $defect->workspace_id != $ids['workspace_id']) {
                return $this->error('Defect not found', 404);
            }
        }
        
        $old = $defect->toArray();
        $defect->status = config('constants.defect_status.completed.key'); // approved/completed
        $defect->reject_reason = null;
        $defect->save();
        $this->logDefectHistory($defect->id, 'completed', 'Defect completed', $old, $defect->toArray());
        
        // Email: notify assignee on approved
        $this->sendDefectEmailToAssignee($defect, 'Defect Approved');
        
        // If completed by subcontractor and defect was assigned to subcontractor, notify customer/admin
        if ($isSubcontractor && in_array($defect->assignment_type, ['subcontractor', 'subcontractor_employee'])) {
            $this->sendDefectEmailToCustomer($defect, 'Defect Completed');
        }
        
        return $this->success($defect, 'Defect approved successfully');
    }

    protected function rejectDefect(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:defects,id',
            'reject_reason' => 'required|string',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        
        // First, fetch the defect to get its customer_id and workspace_id
        $defect = Defect::where('id', $request->id)
            ->where('del', '0')
            ->first();
        
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        
        // Check access based on user type
        if ($isSubcontractor) {
            // For subcontractor users: Check if defect is assigned to them OR created by them
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            // If not created by subcontractor, check if defect is assigned to them
            if (!$isCreator) {
                // Use defect's customer_id and workspace_id (not current user's context)
                $subcontractorCompany = SubcontractorCompany::where('user_id', $user->id)
                    ->where('customer_id', $defect->customer_id)
                    ->where('workspace_id', $defect->workspace_id)
                    ->where('del', '0')
                    ->first();
                
                if (!$subcontractorCompany) {
                    return $this->error('Subcontractor company record not found', 404);
                }
                
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                if (empty($defectIds) || !in_array($request->id, $defectIds)) {
                    return $this->error('Defect not assigned to this subcontractor and you did not create it', 403);
                }
            }
        } else {
            // For employees or regular customers: verify defect belongs to their customer/workspace
            if ($defect->customer_id != $ids['customer_id'] || $defect->workspace_id != $ids['workspace_id']) {
                return $this->error('Defect not found', 404);
            }
        }
        
        $old = $defect->toArray();
        $defect->status = config('constants.defect_status.in_progress.key');
        $defect->reject_reason = $request->reject_reason;
        
        // Determine image type based on defect's created_by_type (not current user's context)
        $type = ($defect->created_by_type === 'employee') ? 'created_by_employee' : 'created_by_customer';
        
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $imageFile) {
                if ($imageFile->isValid()) {
                    // Generate unique filename
                    $extension = strtolower($imageFile->getClientOriginalExtension());
                    $originalName = pathinfo($imageFile->getClientOriginalName(), PATHINFO_FILENAME);
                    $sanitizedFileName = preg_replace('/[^a-zA-Z0-9_-]/', '', $originalName);
                    $sanitizedFileName = substr($sanitizedFileName, 0, 100);
                    $uniqueFileName = time() . '_' . uniqid() . '_' . $sanitizedFileName . '.' . $extension;
                    // Create directory if it doesn't exist
                    $directory = public_path('DefectsManagement');
                    if (!file_exists($directory)) {
                        mkdir($directory, 0777, true);
                    }
                    // Move uploaded file
                    $imageFile->move($directory, $uniqueFileName);
                    $imagePath = 'DefectsManagement/' . $uniqueFileName;
                    // Use defect's customer_id and workspace_id (not current user's context)
                    DefectImage::create([
                        'defect_id' => $defect->id,
                        'uploaded_by' => Auth::id(),
                        'image' => $imagePath,
                        'type' => $type,
                        'customer_id' => $defect->customer_id,
                        'workspace_id' => $defect->workspace_id,
                    ]);
                }
            }
        }
        $defect->save();
        $this->logDefectHistory($defect->id, 'rejected', 'Defect rejected', $old, $defect->toArray());
        // Email: notify creator on rejected
        $this->sendDefectEmailToAssignee($defect, 'Defect Rejected');
        return $this->success($defect, 'Defect rejected with reason');
    }

    


    protected function getDefectHistory($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        
        $query = Defect::where('id', $id);
        
        // For subcontractors: Don't filter by customer/workspace initially (will check access after fetching)
        // For others: Apply customer/workspace filter
        if (!$isSubcontractor) {
            $query->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);
        }
        
        $defect = $query->first();
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        
        // Authorization check: Subcontractors can only view history of defects they created or are assigned to
        if ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            // If not created by subcontractor, check if defect is assigned to them
            if (!$isCreator) {
                // Use defect's customer_id and workspace_id (not current user's context)
                $subcontractorCompany = SubcontractorCompany::where('user_id', $user->id)
                    ->where('customer_id', $defect->customer_id)
                    ->where('workspace_id', $defect->workspace_id)
                    ->where('del', '0')
                    ->first();
                
                $defectIds = $subcontractorCompany ? ($subcontractorCompany->defect_ids ?? []) : [];
                $isAssigned = in_array($defect->id, $defectIds);
                
                if (!$isAssigned) {
                    return $this->error('You do not have permission to view this defect history', 403);
                }
            }
        }
        
        $history = DefectHistory::where('defect_id', $defect->id)
            ->orderBy('created_at', 'desc')
            ->get();
        
        return $this->success($history, 'History retrieved successfully');
    }

    protected function logDefectHistory($defectId, $action, $description = null, $old = null, $new = null)
    {
        try {
            $ids = $this->getCustomerAndWorkspaceIds();
            $userTable = $this->getUserTable();
            $user = Auth::user();
            $changedByType = 'employee';
            if ($userTable === 'customer') {
                if ($user->user_type == config('constants.user_types.subcontractor')) {
                    $changedByType = 'subcontractor';
                } else {
                    $changedByType = 'customer';
                }
            }
            
            DefectHistory::create([
                'defect_id' => $defectId,
                'action' => $action,
                'description' => $description,
                'changed_by' => Auth::id(),
                'changed_by_type' => $changedByType,
                'old_value' => $old,
                'new_value' => $new,
                'customer_id' => $ids['customer_id'] ?? null,
                'workspace_id' => $ids['workspace_id'] ?? null,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to log defect history: ' . $e->getMessage());
        }
    }

    protected function getAssignedDefectsForEmployee()
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $defects = Defect::with(['project', 'site', 'employeePersonalDetails', 'images', 'histories'])
            ->where('assigned_emp_id', Auth::id())
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->get();
        $defects->transform(function ($defect) {
            if ($defect->created_by_type === 'employee') {
                $defect->creator_info = $defect->createdByEmployee;
                $defect->creator_type = 'employee';
                // Remove the duplicate relation data
                unset($defect->created_by_employee);
            } else {
                $defect->creator_info = $defect->createdByCustomer;
                $defect->creator_type = 'customer';
                // Remove the duplicate relation data
                unset($defect->created_by_customer);
            }
            return $defect;
        });
        return $this->success($defects, 'Assigned defects retrieved successfully');
    }

    protected function getAssignedDefectsForSubcontractor(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        
        // Get ALL SubcontractorCompany records for this subcontractor user (across all companies)
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $user->id)
            ->where('del', '0')
            ->get();
        
        // Collect all defect_ids from all company associations
        $allDefectIds = [];
        if (!$subcontractorCompanies->isEmpty()) {
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                $allDefectIds = array_merge($allDefectIds, $defectIds);
            }
        }
        
        // Also get defects where subcontractor_id directly matches this subcontractor's user ID
        // This covers defects assigned directly via the subcontractor_id column in defects table
        $directDefectIds = Defect::where('subcontractor_id', $user->id)
            ->where('del', '0')
            ->whereIn('assignment_type', ['subcontractor', 'subcontractor_employee'])
            ->pluck('id')
            ->toArray();
        
        // Merge both sources of defect IDs
        $allDefectIds = array_merge($allDefectIds, $directDefectIds);
        
        // Collect all unique customer_id and workspace_id pairs to fetch company_name from settings in bulk
        $customerWorkspacePairs = [];
        
        // From subcontractor companies
        if (!$subcontractorCompanies->isEmpty()) {
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                if ($subcontractorCompany->customer_id && $subcontractorCompany->workspace_id) {
                    $key = $subcontractorCompany->customer_id . '_' . $subcontractorCompany->workspace_id;
                    if (!isset($customerWorkspacePairs[$key])) {
                        $customerWorkspacePairs[$key] = [
                            'customer_id' => $subcontractorCompany->customer_id,
                            'workspace_id' => $subcontractorCompany->workspace_id
                        ];
                    }
                }
            }
        }
        
        // Also collect customer/workspace pairs from direct defects
        $directDefects = Defect::where('subcontractor_id', $user->id)
            ->where('del', '0')
            ->whereIn('assignment_type', ['subcontractor', 'subcontractor_employee'])
            ->select('customer_id', 'workspace_id')
            ->distinct()
            ->get();
        
        foreach ($directDefects as $defect) {
            if ($defect->customer_id && $defect->workspace_id) {
                $key = $defect->customer_id . '_' . $defect->workspace_id;
                if (!isset($customerWorkspacePairs[$key])) {
                    $customerWorkspacePairs[$key] = [
                        'customer_id' => $defect->customer_id,
                        'workspace_id' => $defect->workspace_id
                    ];
                }
            }
        }
        
        // Fetch company_name from adminsettings for all customer/workspace pairs in bulk
        $companyNamesMap = [];
        if (!empty($customerWorkspacePairs)) {
            foreach ($customerWorkspacePairs as $key => $pair) {
                $companyName = DB::table('adminsettings')
                    ->where('customer_id', $pair['customer_id'])
                    ->where('workspace', $pair['workspace_id'])
                    ->where('key', 'company_company_name')
                    ->value('value');
                
                // If not found with workspace, try without workspace filter
                if (empty($companyName)) {
                    $companyName = DB::table('adminsettings')
                        ->where('customer_id', $pair['customer_id'])
                        ->where('key', 'company_company_name')
                        ->value('value');
                }
                
                // Fallback to User model company_name or name if not found in settings
                if (empty($companyName)) {
                    $customer = User::find($pair['customer_id']);
                    if ($customer) {
                        $companyName = $customer->company_name ?? $customer->name ?? null;
                    }
                }
                
                $companyNamesMap[$key] = $companyName;
            }
        }
        
        // Create a map: defect_id => company_name
        // This maps each defect to its company name based on which SubcontractorCompany has it in defect_ids
        $defectCompanyMap = [];
        if (!$subcontractorCompanies->isEmpty()) {
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                $companyName = null;
                
                if ($subcontractorCompany->customer_id && $subcontractorCompany->workspace_id) {
                    $key = $subcontractorCompany->customer_id . '_' . $subcontractorCompany->workspace_id;
                    $companyName = $companyNamesMap[$key] ?? null;
                }
                
                // Map each defect_id to its company name
                foreach ($defectIds as $defectId) {
                    // Only set if not already set (use first company association found for each defect)
                    if (!isset($defectCompanyMap[$defectId])) {
                        $defectCompanyMap[$defectId] = $companyName;
                    }
                }
            }
        }
        
        // Also map company names for direct defects (where subcontractor_id matches)
        // Get defects with their customer_id and workspace_id to map company names
        $directDefectsForMapping = Defect::where('subcontractor_id', $user->id)
            ->where('del', '0')
            ->whereIn('assignment_type', ['subcontractor', 'subcontractor_employee'])
            ->select('id', 'customer_id', 'workspace_id')
            ->get();
        
        foreach ($directDefectsForMapping as $defect) {
            // Only set if not already set (defect_ids array takes precedence)
            if (!isset($defectCompanyMap[$defect->id]) && $defect->customer_id && $defect->workspace_id) {
                $key = $defect->customer_id . '_' . $defect->workspace_id;
                $defectCompanyMap[$defect->id] = $companyNamesMap[$key] ?? null;
            }
        }
        
        // Remove duplicates
        $allDefectIds = array_unique($allDefectIds);
        
        // Get defects where ID is in the defect_ids array OR subcontractor_id directly matches
        // Also include defects where subcontractor_id directly matches (even if not in defect_ids array)
        $query = Defect::with([
            'project',
            'site',
            'subcontractor',
            'assignedSubcontractorEmployeePersonalDetails',
            'images',
            'histories'
        ])
            ->where(function ($q) use ($allDefectIds, $user) {
                // Defects from defect_ids array (if any)
                if (!empty($allDefectIds)) {
                    $q->whereIn('id', $allDefectIds);
                }
                // OR defects where subcontractor_id directly matches
                if (empty($allDefectIds)) {
                    // If no defect_ids, only get direct matches
                    $q->where('subcontractor_id', $user->id)
                        ->whereIn('assignment_type', ['subcontractor', 'subcontractor_employee']);
                } else {
                    // If defect_ids exist, use OR condition
                    $q->orWhere(function ($subQ) use ($user) {
                        $subQ->where('subcontractor_id', $user->id)
                            ->whereIn('assignment_type', ['subcontractor', 'subcontractor_employee']);
                    });
                }
            })
            ->where('del', '0'); // Ensure defect is not deleted
        
        // Note: We don't filter by customer_id/workspace_id here because subcontractor might have defects across multiple companies
        // The filtering by customer/workspace can be done via request filters if needed
        
        // Check if we have any defects before applying filters
        $defectsCount = $query->count();
        if ($defectsCount === 0) {
            return $this->message('No defects assigned to this subcontractor');
        }
        
        // Apply additional filters
        $query = $query
            ->when($request->filled('project_id'), function ($q) use ($request) {
                $q->where('project_id', $request->project_id);
            })
            ->when($request->filled('site_id'), function ($q) use ($request) {
                $q->where('site_id', $request->site_id);
            })
            ->when($request->filled('priority'), function ($q) use ($request) {
                $q->where('priority', $request->priority);
            })
            ->when($request->filled('status'), function ($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->when($request->filled('title'), function ($q) use ($request) {
                $q->where('title', 'like', '%' . $request->title . '%');
            })
            ->orderBy('created_at', 'desc');
        
        $defects = $query->get();
        
        $defects->transform(function ($defect) use ($defectCompanyMap) {
            if ($defect->created_by_type === 'employee') {
                $defect->creator_info = $defect->createdByEmployee;
                $defect->creator_type = 'employee';
                unset($defect->created_by_employee);
            } elseif ($defect->created_by_type === 'subcontractor') {
                $defect->creator_info = $defect->createdByCustomer;
                $defect->creator_type = 'subcontractor';
                unset($defect->created_by_customer);
            } else {
                $defect->creator_info = $defect->createdByCustomer;
                $defect->creator_type = 'customer';
                unset($defect->created_by_customer);
            }
            
            // Add company_name to each defect
            $defect->company_name = $defectCompanyMap[$defect->id] ?? null;
            
            return $defect;
        });
        
        return $this->success($defects, 'Assigned defects retrieved successfully');
    }

    protected function markInProgressDefect(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:defects,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        
        // First, fetch the defect to get its customer_id and workspace_id
        $defect = Defect::where('id', $request->id)
            ->where('status', config('constants.defect_status.open.key'))
            ->first();
        
        if (!$defect) {
            return $this->error('Defect not found or not in open status', 404);
        }
        
        // Check access based on user type
        if ($userTable === 'customer' && $user->user_type == config('constants.user_types.subcontractor')) {
            // For subcontractor users: Check if defect is assigned to them OR created by them
            // First, check if defect was created by this subcontractor
            $isCreator = ($defect->created_by == $user->id && $defect->created_by_type == 'subcontractor');
            
            // If not created by subcontractor, check if defect is assigned to them
            if (!$isCreator) {
                // Use defect's customer_id and workspace_id (not current user's context)
                $subcontractorCompany = SubcontractorCompany::where('user_id', $user->id)
                    ->where('customer_id', $defect->customer_id)
                    ->where('workspace_id', $defect->workspace_id)
                    ->where('del', '0')
                    ->first();
                
                if (!$subcontractorCompany) {
                    return $this->error('Subcontractor company record not found', 404);
                }
                
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                if (empty($defectIds) || !in_array($request->id, $defectIds)) {
                    return $this->error('Defect not assigned to this subcontractor and you did not create it', 403);
                }
            }
            
            // Defect is either created by subcontractor or assigned to them, continue (defect already fetched)
        } else {
            // For employees or regular customers: verify defect belongs to their customer/workspace
            if ($defect->customer_id != $ids['customer_id'] || $defect->workspace_id != $ids['workspace_id']) {
                return $this->error('Defect not found', 404);
            }
            
            // Check if user has access (assigned to them or created by them)
            $hasAccess = ($defect->assigned_emp_id == Auth::id()) ||
                        ($defect->assigned_subcontractor_emp_id == Auth::id()) ||
                        ($defect->created_by == Auth::id());
            
            if (!$hasAccess) {
                return $this->error('You do not have permission to mark this defect in progress', 403);
            }
        }
        
        $old = $defect->toArray();
        $defect->status = config('constants.defect_status.in_progress.key'); // in progress
        $defect->save();
        $this->logDefectHistory($defect->id, 'in_progress', 'Defect marked in progress', $old, $defect->toArray());
        
        // Email: notify creator when in progress
        $this->sendDefectEmailToCreator($defect, 'Defect In Progress');
        
        // If updated by subcontractor and assigned to subcontractor, also notify customer/admin
        $isSubcontractorUser = ($userTable === 'customer' && $user->user_type == config('constants.user_types.subcontractor'));
        if ($isSubcontractorUser && in_array($defect->assignment_type, ['subcontractor', 'subcontractor_employee'])) {
            $this->sendDefectEmailToCustomer($defect, 'Defect In Progress');
        }
        
        return $this->success($defect, 'Defect marked in progress successfully');
    }

    // Helper method to get assignee information for logging
    protected function getAssigneeInfo($assignmentType, $request)
    {
        switch ($assignmentType) {
            case 'internal_employee':
                return 'Internal Employee ID: ' . $request->assigned_emp_id;
            case 'subcontractor':
                return 'Subcontractor ID: ' . $request->subcontractor_id;
            case 'subcontractor_employee':
                return 'Subcontractor ID: ' . $request->subcontractor_id . ', Employee ID: ' . $request->assigned_subcontractor_emp_id;
            default:
                return 'Unknown assignment type';
        }
    }

    // Get assignable internal employees
    protected function getAssignableInternalEmployees()
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $employees = EmpCompanyDetails::with('employeePersonalDetails')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->get();
        return $this->success($employees, 'Internal employees retrieved successfully');
    }

    // Get assignable subcontractors
    // Returns User records (subcontractors from users table) via SubcontractorCompany
    protected function getAssignableSubcontractors()
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        // Get subcontractors from subcontractor_companies table which links to users table
        $subcontractorCompanies = SubcontractorCompany::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->with(['user' => function ($query) {
                $query->select('id', 'name', 'email')
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0');
            }])
            ->get();
        
        // Extract User records and filter out any nulls
        $subcontractors = $subcontractorCompanies->map(function ($sc) {
            return $sc->user;
        })->filter()->values();
        
        return $this->success($subcontractors, 'Subcontractors retrieved successfully');
    }

    // Get assignable subcontractor employees
    protected function getAssignableSubcontractorEmployees($subcontractorId = null)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $query = EmpCompanyDetails::with('employeePersonalDetails')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0');
        if ($subcontractorId) {
            $query->where('company_id', $subcontractorId);
        }
        $employees = $query->get();
        return $this->success($employees, 'Subcontractor employees retrieved successfully');
    }

    // Get all assignable entities (for dropdown/selection purposes)
    protected function getAllAssignableEntities()
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $internalEmployees = EmpCompanyDetails::with('employeePersonalDetails')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->get();
        
        // Get subcontractors from subcontractor_companies table which links to users table
        $subcontractorCompanies = SubcontractorCompany::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->with(['user' => function ($query) {
                $query->select('id', 'name', 'email')
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0');
            }])
            ->get();
        
        // Extract User records and filter out any nulls
        $subcontractors = $subcontractorCompanies->map(function ($sc) {
            return $sc->user;
        })->filter()->values();
        
        $data = [
            'internal_employees' => $internalEmployees,
            'subcontractors' => $subcontractors
        ];
        return $this->success($data, 'All assignable entities retrieved successfully');
    }

    /**
     * Sync defect assignment to subcontractor_companies table
     * Adds defect ID to defect_ids array when assigned to subcontractor
     * 
     * @param Defect $defect
     * @return void
     */
    protected function syncDefectToSubcontractorCompany($defect)
    {
        try {
            // Only sync if assignment type is subcontractor or subcontractor_employee
            if (!in_array($defect->assignment_type, ['subcontractor', 'subcontractor_employee'])) {
                return;
            }

            if (!$defect->subcontractor_id) {
                return;
            }

            // CRITICAL FIX: Use defect's customer_id and workspace_id, not the current user's context
            // This ensures we find the correct SubcontractorCompany record
            $customerId = $defect->customer_id;
            $workspaceId = $defect->workspace_id;

            if (!$customerId || !$workspaceId) {
                Log::warning("Defect missing customer_id or workspace_id. Defect ID: {$defect->id}");
                return;
            }
            
            // subcontractor_id directly references users.id (subcontractor user in users table)
            $userId = $defect->subcontractor_id;
            
            // Verify this is a valid subcontractor user
            $subcontractorUser = \App\Models\User::find($userId);
            if (!$subcontractorUser || $subcontractorUser->user_type != config('constants.user_types.subcontractor')) {
                Log::warning("Invalid subcontractor user. user_id: {$userId}, defect_id: {$defect->id}");
                return;
            }

            // Find SubcontractorCompany record using defect's customer_id and workspace_id
            // subcontractor_companies.user_id → users.id (the subcontractor user)
            $subcontractorCompany = SubcontractorCompany::where('user_id', $userId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->first();

            if (!$subcontractorCompany) {
                Log::warning("SubcontractorCompany record not found for user_id: {$userId}, customer_id: {$customerId}, workspace_id: {$workspaceId}, defect_id: {$defect->id}");
                return;
            }

            // Get current defect_ids array
            $defectIds = $subcontractorCompany->defect_ids ?? [];
            
            // Add defect ID if not already present
            if (!in_array($defect->id, $defectIds)) {
                $defectIds[] = $defect->id;
                $subcontractorCompany->defect_ids = array_values($defectIds); // array_values to reindex
                $subcontractorCompany->save();
                Log::info("Successfully added defect ID {$defect->id} to subcontractor_companies defect_ids for user_id: {$userId}");
            }
        } catch (\Throwable $e) {
            Log::error('Failed to sync defect to subcontractor company: ' . $e->getMessage(), [
                'defect_id' => $defect->id ?? null,
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Remove defect assignment from subcontractor_companies table
     * Removes defect ID from defect_ids array when unassigned from subcontractor
     * 
     * @param int $defectId
     * @param int|null $oldSubcontractorId
     * @param int $customerId Defect's customer_id
     * @param int $workspaceId Defect's workspace_id
     * @return void
     */
    protected function removeDefectFromSubcontractorCompany($defectId, $oldSubcontractorId, $customerId, $workspaceId)
    {
        try {
            if (!$oldSubcontractorId) {
                return;
            }

            if (!$customerId || !$workspaceId) {
                Log::warning("Cannot remove defect: missing customer_id or workspace_id. Defect ID: {$defectId}");
                return;
            }

            // oldSubcontractorId directly references users.id (subcontractor user in users table)
            $userId = $oldSubcontractorId;
            
            // Verify this is a valid subcontractor user
            $subcontractorUser = \App\Models\User::find($userId);
            if (!$subcontractorUser || $subcontractorUser->user_type != config('constants.user_types.subcontractor')) {
                Log::warning("Invalid subcontractor user when removing defect. user_id: {$userId}, defect_id: {$defectId}");
                return;
            }

            // Find SubcontractorCompany record using defect's customer_id and workspace_id
            // subcontractor_companies.user_id → users.id (the subcontractor user)
            $subcontractorCompany = SubcontractorCompany::where('user_id', $userId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->first();

            if (!$subcontractorCompany) {
                Log::warning("SubcontractorCompany record not found when removing defect. user_id: {$userId}, customer_id: {$customerId}, workspace_id: {$workspaceId}, defect_id: {$defectId}");
                return;
            }

            // Get current defect_ids array
            $defectIds = $subcontractorCompany->defect_ids ?? [];
            
            // Remove defect ID if present
            $defectIds = array_filter($defectIds, function($id) use ($defectId) {
                return $id != $defectId;
            });
            
            $subcontractorCompany->defect_ids = array_values($defectIds); // array_values to reindex
            $subcontractorCompany->save();
            Log::info("Successfully removed defect ID {$defectId} from subcontractor_companies defect_ids for user_id: {$userId}");
        } catch (\Throwable $e) {
            Log::error('Failed to remove defect from subcontractor company: ' . $e->getMessage(), [
                'defect_id' => $defectId,
                'trace' => $e->getTraceAsString()
            ]);
        }
    }


    protected function assignDefectToSubcontractorEmp($request)
    {
        $validator = Validator::make($request->all(), [
            'defect_id' => 'required|integer|exists:defects,id',
            'subcontractor_employee_id' => 'required|integer|exists:employees_subcontractors,id',
        ]);
        if ($validator->fails()) {
            return $this->error('Validation failed', 422, $validator->errors());
        }
        $defect = Defect::find($request->defect_id);
        if (!$defect) {
            return $this->error('Defect not found', 404);
        }
        $defect->assigned_subcontractor_emp_id = $request->subcontractor_employee_id;
        $defect->assignment_type = 'subcontractor_employee';
        $defect->assigned_emp_id = $request->subcontractor_employee_id;
        $defect->save();
        $this->syncDefectToSubcontractorCompany($defect);
        $this->logDefectHistory($defect->id, 'assigned', 'Defect assigned to subcontractor employee', $defect->toArray(), $defect->toArray());
        $this->sendDefectEmailToAssignee($defect, 'Defect Assigned');
        return $this->success($defect, 'Defect assigned to subcontractor employee successfully');
    }
}
