<?php

namespace App\Http\Controllers\Traits;

use App\Models\Adminsettings;
use App\Models\User;
use SendGrid;
use SendGrid\Mail\Mail;
use SendGrid\Mail\Attachment;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;


trait EmailTrait
{
    use HelperTrait;
    /**
     * Get email settings from database
     * 
     * @param int|null $customer_id
     * @param int|null $workspace_id
     * @return array
     */

     use HelperTrait;
    private function getEmailSettings($customer_id = null, $workspace_id = null)
    {
        try {
            $emailOptions = [
                'email_mail_from_address',
                'email_mail_encryption', 
                'email_api_key',
                'email_mail_from_name', 
                'email_mail_host', 
                'email_mail_port', 
                'email_mail_password'
            ];
            
            // If customer_id is provided, try to get customer-specific email settings
            if ($customer_id) {
                $query = Adminsettings::where('customer_id', $customer_id);
                if ($workspace_id) {
                    $query->where('workspace', $workspace_id);
                }
                $settings = $query->whereIn('key', $emailOptions)
                    ->pluck('value', 'key')
                    ->toArray();
                
                // If we got settings, return them
                if (!empty($settings)) {
                    return $settings;
                }
            }
            
            // Fallback to admin settings
            $admin = User::where('user_type', config('constants.user_types.admin'))->first();
            if ($admin) {
                $settings = Adminsettings::where('customer_id', $admin->id)
                    ->whereIn('key', $emailOptions)
                    ->pluck('value', 'key')
                    ->toArray();
                return $settings;
            }
    
            return [];
        } catch (\Exception $e) {
            Log::error("Failed to fetch email settings: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Send email using customer specific email configuration
     * 
     * @param array $params
     * @return bool
     */
    public function SendInstantEmail(array $params)
    {
        try {
            // Validate email address
            if (empty($params['to']) || !filter_var($params['to'], FILTER_VALIDATE_EMAIL)) {
                Log::error("Email sending failed: Invalid recipient email", [
                    'to' => $params['to'] ?? 'not provided'
                ]);
                return false;
            }
            // Get customer_id and workspace_id from multiple sources
            // Priority: 1. Params (explicit override), 2. HelperTrait::getCustomerAndWorkspaceIds() (primary source), 3. Auth user context (fallback)
            $customer_id = $params['customer_id'] ?? null;
            $workspace_id = $params['workspace_id'] ?? null;
            $source = 'none';
            
            // Primary source: Use HelperTrait::getCustomerAndWorkspaceIds() method to get from authenticated user context
            // This method is from App\Http\Controllers\Traits\HelperTrait
            // Works for customers, employees, and subcontractors (not super admin)
            // Since Controller base class uses both HelperTrait and EmailTrait, this method is available
                try {
                    $ids = $this->getCustomerAndWorkspaceIds();
                    if ($ids && is_array($ids) && isset($ids['customer_id']) && isset($ids['workspace_id'])) {
                        // Only use if not explicitly provided in params (params take precedence)
                        $customer_id = $customer_id ?? $ids['customer_id'] ?? null;
                        $workspace_id = $workspace_id ?? $ids['workspace_id'] ?? null;
                        $source = $customer_id && $workspace_id && !isset($params['customer_id']) ? 'helper_trait' : ($params['customer_id'] ?? null ? 'params' : 'helper_trait');
                    }
                } catch (\Exception $e) {
                    Log::warning('Failed to get customer/workspace IDs from HelperTrait::getCustomerAndWorkspaceIds()', [
                        'error' => $e->getMessage()
                    ]);
                }
            
            
            // Fallback: If still not found, try to get from authenticated user directly
            if ((!$customer_id || !$workspace_id) && Auth::check()) {
                $user = Auth::user();
                if ($user instanceof \App\Models\User) {
                    $customer_id = $customer_id ?? $user->id ?? null;
                    $workspace_id = $workspace_id ?? $user->current_workspace_id ?? null;
                    $source = $source === 'none' ? 'auth_user' : $source;
                } elseif ($user instanceof \App\Models\EmpCompanyDetails) {
                    $customer_id = $customer_id ?? $user->customer_id ?? null;
                    $workspace_id = $workspace_id ?? $user->workspace_id ?? null;
                    $source = $source === 'none' ? 'auth_user' : $source;
                }
            }
            
            Log::info('Email customer/workspace IDs determined', [
                'customer_id' => $customer_id,
                'workspace_id' => $workspace_id,
                'source' => $source
            ]);
            // Get email settings from database
            $emailSettings = $this->getEmailSettings($customer_id, $workspace_id);
            // Validate sender email
            if (empty($emailSettings['email_mail_from_address']) || 
                !filter_var($emailSettings['email_mail_from_address'], FILTER_VALIDATE_EMAIL)) {
                Log::error("Email sending failed: Invalid sender email", [
                    'from' => $emailSettings['email_mail_from_address'] ?? 'not provided'
                ]);
                return false;
            }
            // Get dynamic sender name
            // If customer_id is provided, prioritize brand title over email_mail_from_name
            // This ensures customer branding is used when sending emails for specific customers
            $senderName = null;
            if ($customer_id || $workspace_id) {
                // For customer-specific emails, use brand title as primary sender name
                $senderName = \App\Services\CompanyEmailDetails::getBrandTitle($customer_id, $workspace_id);
                // Only override with email_mail_from_name if it's specifically set for this customer
                // Check if customer has their own email_mail_from_name setting
                if ($customer_id) {
                    $customerEmailFromName = Adminsettings::where('customer_id', $customer_id)
                        ->where('key', 'email_mail_from_name')
                        ->when($workspace_id, function($q) use ($workspace_id) {
                            return $q->where('workspace', $workspace_id);
                        })
                        ->value('value');
                    
                    // If customer has their own email_mail_from_name, use it instead of brand title
                    if (!empty($customerEmailFromName)) {
                        $senderName = $customerEmailFromName;
                        Log::info('Using customer-specific email_mail_from_name', [
                            'customer_id' => $customer_id,
                            'email_mail_from_name' => $customerEmailFromName,
                        ]);
                    }
                }
            } else {
                // For non-customer emails (admin/system emails), use email_mail_from_name from settings
                $senderName = $emailSettings['email_mail_from_name'] ?? null;
                Log::info('Using admin email_mail_from_name', [
                    'email_mail_from_name' => $senderName,
                ]);
            }
            // Final fallback to default
            if (empty($senderName)) {
                $senderName = env('APP_NAME', 'WMS');
                Log::info('Using default sender name', ['sender_name' => $senderName]);
            }
            Log::info('Final email sender name', [
                'sender_name' => $senderName,
                'customer_id' => $customer_id,
                'workspace_id' => $workspace_id,
            ]);
            $email = new Mail();
            $email->setFrom(
                $emailSettings['email_mail_from_address'] , 
                $senderName
            );
            $email->setSubject($params['subject']);
            $email->addTo($params['to']);
            $email->addContent("text/plain", strip_tags($params['msg'])); // Plain text version
            $email->addContent("text/html", $params['msg']); // HTML version
            // Handle attachments if provided
            if (!empty($params['attachments']) && is_array($params['attachments'])) {
                foreach ($params['attachments'] as $attachmentData) {
                    $filePath = null;
                    $fileName = null;
                    // Check if file exists in public directory
                    if (isset($attachmentData['path']) && file_exists(public_path($attachmentData['path']))) {
                        $filePath = public_path($attachmentData['path']);
                        $fileName = $attachmentData['name'] ?? basename($attachmentData['path']);
                    } elseif (isset($attachmentData['path']) && file_exists($attachmentData['path'])) {
                        // Handle absolute paths
                        $filePath = $attachmentData['path'];
                        $fileName = $attachmentData['name'] ?? basename($filePath);
                    }
                    if ($filePath && $fileName && file_exists($filePath)) {
                        try {
                            $fileContent = file_get_contents($filePath);
                            if ($fileContent === false) {
                                Log::error("Failed to read attachment file", [
                                    'path' => $filePath
                                ]);
                                continue;
                            }
                            
                            $fileEncoded = base64_encode($fileContent);
                            
                            // Create SendGrid Attachment object
                            $attachment = new Attachment();
                            $attachment->setContent($fileEncoded);
                            $attachment->setType($attachmentData['type'] ?? 'application/pdf');
                            $attachment->setFilename($fileName);
                            $attachment->setDisposition('attachment');
                            
                            $email->addAttachment($attachment);
                            
                            Log::info("Attachment added to email", [
                                'filename' => $fileName,
                                'type' => $attachmentData['type'] ?? 'application/pdf',
                                'size' => strlen($fileContent)
                            ]);
                        } catch (\Exception $e) {
                            Log::error("Failed to add attachment", [
                                'path' => $filePath,
                                'error' => $e->getMessage()
                            ]);
                        }
                    } else {
                        Log::warning("Attachment file not found", [
                            'path' => $attachmentData['path'] ?? 'not provided',
                            'name' => $fileName ?? 'not provided',
                            'filePath' => $filePath ?? 'not determined',
                            'exists' => $filePath ? (file_exists($filePath) ? 'yes' : 'no') : 'N/A'
                        ]);
                    }
                }
            }
            
            // Use API key from settings table or fall back to .env
            $password = $emailSettings['email_mail_password'] ?? null;
            if (empty($password)) {
                Log::error("Email sending failed: No API key found");
                return false;
            }
            
            $sendgrid = new SendGrid($password);
            $response = $sendgrid->send($email);

            if ($response->statusCode() >= 200 && $response->statusCode() < 300) {
                return true;
            } else {
                Log::error("SendGrid email failed. Status Code: " . $response->statusCode());
                return false;
            }
        } catch (\Exception $e) {
            Log::error("Email sending failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send referral signup email to user
     * 
     * @param array $userData
     * @return bool
     */
    public function sendReferralSignupUserEmail(array $userData)
    {
        try {
            $emailData = [
                'email_type' => 'referral_signup_user',
                'name' => $userData['name'] ?? 'Valued Partner',
                'email' => $userData['email'] ?? '',
                'mobile_number' => $userData['mobile_number'] ?? '',
                'applied_at' => $userData['applied_at'] ?? \Carbon\Carbon::now(),
            ];

            $params = [
                'to' => $userData['email'],
                'subject' => 'Your Referral Request is Pending Approval | ' . env('APP_NAME'),
                'msg' => view('Emails.referral-notification', $emailData)->render(),
            ];

            return $this->SendInstantEmail($params);
        } catch (\Exception $e) {
            Log::error("Referral signup user email failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send referral signup notification email to admin
     * 
     * @param array $userData
     * @return bool
     */
    public function sendReferralSignupAdminEmail(array $userData)
    {
        try {
            // Get all admin users
            $adminUsers = User::where('user_type', 1)->get();
            
            if ($adminUsers->isEmpty()) {
                Log::warning("No admin users found for referral signup notification");
                return false;
            }

            $emailData = [
                'email_type' => 'referral_signup_admin',
                'name' => $userData['name'] ?? 'N/A',
                'email' => $userData['email'] ?? 'N/A',
                'mobile_number' => $userData['mobile_number'] ?? 'N/A',
                'applied_at' => $userData['applied_at'] ?? \Carbon\Carbon::now(),
            ];

            $successCount = 0;
            foreach ($adminUsers as $admin) {
                $params = [
                    'to' => $admin->email,
                    'subject' => 'New Referral Signup Request | ' . env('APP_NAME'),
                    'msg' => view('Emails.referral-notification', $emailData)->render(),
                ];

                if ($this->SendInstantEmail($params)) {
                    $successCount++;
                }
            }

            return $successCount > 0;
        } catch (\Exception $e) {
            Log::error("Referral signup admin email failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send referral approval email to user
     * 
     * @param array $userData
     * @return bool
     */
    public function sendReferralApprovalEmail(array $userData)
    {
        try {
            $emailData = [
                'email_type' => 'referral_approved',
                'name' => $userData['name'] ?? 'Valued Partner',
                'email' => $userData['email'] ?? '',
                'referral_code' => $userData['referral_code'] ?? '',
                'referral_link' => $userData['referral_link'] ?? '',
                'approved_at' => $userData['approved_at'] ?? \Carbon\Carbon::now(),
            ];

            $params = [
                'to' => $userData['email'],
                'subject' => 'Referral Partnership Approved! | ' . env('APP_NAME'),
                'msg' => view('Emails.referral-notification', $emailData)->render(),
            ];

            return $this->SendInstantEmail($params);
        } catch (\Exception $e) {
            Log::error("Referral approval email failed: " . $e->getMessage());
            return false;
        }
    }
}