<?php

namespace App\Http\Controllers\Traits;

use App\Models\Adminsettings;
use App\Models\EmpCompanyDetails;
use App\Models\EmpTeam;
use App\Models\Tier;
use App\Models\User;
use App\Models\SubcontractorCompany;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\RosterAssign;
use App\Models\PublicHoliday;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\RequiredDocument;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;
use App\Models\SubcontractorEmployeeDocument;
use App\Models\Project;
use App\Models\Role;
use App\Models\LeavePackage;
use App\Models\LeaveRequest;
use App\Models\BaseModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Models\EmployeeAttendance;



trait HelperTrait
{

    public function getLatestData($model, Request $request, $relations = [])
    {
        $latestData = $model->latest();
        if (!empty($relations)) {
            $relations = is_array($relations) ? $relations : [$relations];
            foreach ($relations as $relation) {
                $latestData = $latestData->with($relation);
            }
        }
        return $this->withCount($latestData);
    }

    private function createDefaultSubcontractorSettings($subcontractorId, $workspaceId)
    {
        $defaultSettings = [
            ['key' => 'system_menu_type', 'value' => 'topbar'],
            ['key' => 'brand_favicon', 'value' => 'favicon.png'],
            ['key' => 'brand_logo_dark', 'value' => 'logo.png'],
            ['key' => 'sidebar_brand_logo', 'value' => 'sidebarlogo.png'],
        ];

        $createdSettings = [];

        foreach ($defaultSettings as $setting) {
            $existing = Adminsettings::where('customer_id', $subcontractorId)
                ->where('workspace', $workspaceId)
                ->where('key', $setting['key'])
                ->first();

            if (!$existing) {
                $created = Adminsettings::create([
                    'key' => $setting['key'],
                    'value' => $setting['value'],
                    'workspace' => $workspaceId,
                    'customer_id' => $subcontractorId,
                ]);
                $createdSettings[] = $created;
            } else {
                $createdSettings[] = $existing;
            }
        }

        return $createdSettings;
    }
    //Attendance Helper Function for Subcontractor Employees
    protected function applyCustomerWorkspaceFilterWithSubcontractors($query, $customer_id = null, $workspace_id = null)
    {
        // Get customer_id and workspace_id if not provided
        if ($customer_id === null || $workspace_id === null) {
            $userTable = $this->getUserTable();
            if ($userTable === "customer") {
                $customer_id = $customer_id ?? auth()->id();
                $workspace_id = $workspace_id ?? auth()->user()->current_workspace_id;
            } elseif ($userTable === "emp") {
                $customer_id = $customer_id ?? auth()->user()->customer_id;
                $workspace_id = $workspace_id ?? auth()->user()->workspace_id;
            } else {
                // Fallback to getCustomerAndWorkspaceIds
                $ids = $this->getCustomerAndWorkspaceIds();
                if (!$ids) {
                    return $query->whereRaw('1 = 0'); // No results
                }
                $customer_id = $customer_id ?? $ids['customer_id'];
                $workspace_id = $workspace_id ?? $ids['workspace_id'];
            }
        }

        // Filter to include both regular employees and subcontractor employees
        return $query->where(function ($q) use ($customer_id, $workspace_id) {
            // Regular employees (where empcompanydetails exists)
            $q->whereHas('EmpCompanyDetails', function ($subQ) use ($customer_id, $workspace_id) {
                $subQ->where('customer_id', $customer_id)
                    ->where('workspace_id', $workspace_id);
            })
            // Subcontractor employees (where subcontractor_id is not null)
            ->orWhere(function ($subQ) use ($customer_id, $workspace_id) {
                $subQ->whereNotNull('subcontractor_id')
                    ->where('customer_id', $customer_id)
                    ->where('workspace_id', $workspace_id);
            });
        });
    }

    protected function applyNameSearchWithSubcontractors($query, $searchTerm)
    {
        // Handle both string and array input
        if (is_string($searchTerm)) {
            $filterValue = explode(" ", trim($searchTerm));
        } else {
            $filterValue = is_array($searchTerm) ? $searchTerm : [$searchTerm];
        }

        return $query->where(function ($q) use ($filterValue) {
            // Search in regular employees
            $q->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                if (count($filterValue) === 1) {
                    $subquery->where(function ($query) use ($filterValue) {
                        $query->where('first_name', 'like', '%' . $filterValue[0] . '%')
                            ->orWhere('middle_name', 'like', '%' . $filterValue[0] . '%')
                            ->orWhere('last_name', 'like', '%' . $filterValue[0] . '%');
                    });
                } elseif (count($filterValue) === 2) {
                    $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                        ->where(function ($query) use ($filterValue) {
                            $query->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                                ->orWhere('last_name', 'like', '%' . $filterValue[1] . '%');
                        });
                } elseif (count($filterValue) >= 3) {
                    $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                        ->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                        ->where('last_name', 'like', '%' . $filterValue[2] . '%');
                }
            })
            // Search in subcontractor employees
            ->orWhereHas('subcontractorEmployee', function ($subquery) use ($filterValue) {
                if (count($filterValue) === 1) {
                    $subquery->where(function ($query) use ($filterValue) {
                        $query->where('first_name', 'like', '%' . $filterValue[0] . '%')
                            ->orWhere('middle_name', 'like', '%' . $filterValue[0] . '%')
                            ->orWhere('last_name', 'like', '%' . $filterValue[0] . '%')
                            ->orWhere('email', 'like', '%' . $filterValue[0] . '%');
                    });
                } elseif (count($filterValue) === 2) {
                    $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                        ->where(function ($query) use ($filterValue) {
                            $query->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                                ->orWhere('last_name', 'like', '%' . $filterValue[1] . '%');
                        });
                } elseif (count($filterValue) >= 3) {
                    $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                        ->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                        ->where('last_name', 'like', '%' . $filterValue[2] . '%');
                }
            });
        });
    }

    protected function transformAttendanceCollection($result)
    {
        // Check if it's an array first (arrays can't have methods)
        if (is_array($result)) {
            // Array - transform each item
            return array_map(function ($attendance) {
                return $this->transformAttendanceForSubcontractor($attendance);
            }, $result);
        }
        
        // Check if it's an object before calling method_exists
        if (is_object($result)) {
            if (method_exists($result, 'getCollection')) {
                // Paginated result - transform the collection items
                $result->getCollection()->transform(function ($attendance) {
                    return $this->transformAttendanceForSubcontractor($attendance);
                });
                return $result;
            } elseif ($result instanceof LengthAwarePaginator) {
                // Another pagination type
                $items = $result->items();
                foreach ($items as $key => $attendance) {
                    $items[$key] = $this->transformAttendanceForSubcontractor($attendance);
                }
                $result->setCollection(collect($items));
                return $result;
            } elseif (method_exists($result, 'map')) {
                // Collection - transform each item
                return $result->map(function ($attendance) {
                    return $this->transformAttendanceForSubcontractor($attendance);
                });
            }
        }
        
        // Fallback: return as-is if we can't transform
        return $result;
    }
    
    protected function transformLeaveRequestForSubcontractor($leaveRequest)
    {
        // Convert to array if it's a model instance
        if ($leaveRequest instanceof \Illuminate\Database\Eloquent\Model) {
            $leaveArray = $leaveRequest->toArray();
            // Also try to get relationship data directly from model if not in array
            if (!isset($leaveArray['subcontractor_employee']) && $leaveRequest->relationLoaded('subcontractorEmployee')) {
                $subcontractorEmpModel = $leaveRequest->subcontractorEmployee;
                if ($subcontractorEmpModel) {
                    $leaveArray['subcontractor_employee'] = $subcontractorEmpModel->toArray();
                }
            }
        } else {
            $leaveArray = (array)$leaveRequest;
        }
        
        // Check for subcontractor_id
        $subcontractorId = $leaveArray['subcontractor_id'] ?? null;
        
        // Get subcontractor employee data (check both snake_case and camelCase)
        $subcontractorEmp = $leaveArray['subcontractor_employee'] ?? $leaveArray['subcontractorEmployee'] ?? null;
        
        // Set user_type: 0 for regular employees, 1 for subcontractor employees
        $leaveArray['user_type'] = $subcontractorId !== null ? 1 : 0;
        
        // If subcontractor_id is not null and subcontractor employee exists, map to emp_personal_details
        if ($subcontractorId !== null && $subcontractorEmp && !empty($subcontractorEmp) && is_array($subcontractorEmp)) {
            // Map subcontractor employee data to emp_personal_details structure
            $empPersonalData = [
                'id' => null,
                'emp_id' => (string)($subcontractorEmp['id'] ?? ''),
                'first_name' => $subcontractorEmp['first_name'] ?? null,
                'middle_name' => $subcontractorEmp['middle_name'] ?? null,
                'last_name' => $subcontractorEmp['last_name'] ?? null,
                'mobile' => $subcontractorEmp['mobile'] ?? null,
                'streat_address' => $subcontractorEmp['address'] ?? null,
                'suburb' => $subcontractorEmp['suburb'] ?? null,
                'state' => $subcontractorEmp['state'] ?? null,
                'postcode' => null,
                'image' => $subcontractorEmp['profile_image'] ?? null,
                'date_of_birth' => $subcontractorEmp['dob'] ?? null,
                'blood_group' => null,
                'bank_name' => null,
                'account_holder_name' => null,
                'ibn_number' => null,
                'account_number' => null,
                'created_by' => null,
                'del' => '0',
                'created_at' => $subcontractorEmp['created_at'] ?? null,
                'updated_at' => $subcontractorEmp['updated_at'] ?? null,
                'subcontractors' => $subcontractorEmp['subcontractors'] ?? null,
            ];
            
            // Set emp_personal_details (matching regular employee structure)
            $leaveArray['emp_personal_details'] = $empPersonalData;
        }
        
        // Always remove subcontractor_employee from response to keep it clean (for both regular and subcontractor employees)
        unset($leaveArray['subcontractor_employee']);
        unset($leaveArray['subcontractorEmployee']);
        
        // Return as array for consistency
        return $leaveArray;
    }
    
    protected function transformLeaveRequestCollection($result)
    {
        // Check if it's an array first (arrays can't have methods)
        if (is_array($result)) {
            // Array - transform each item
            return array_map(function ($leaveRequest) {
                return $this->transformLeaveRequestForSubcontractor($leaveRequest);
            }, $result);
        }
        
        // Check if it's an object before calling method_exists
        if (is_object($result)) {
            if (method_exists($result, 'getCollection')) {
                // Paginated result - transform the collection items
                $result->getCollection()->transform(function ($leaveRequest) {
                    return $this->transformLeaveRequestForSubcontractor($leaveRequest);
                });
                return $result;
            } elseif ($result instanceof LengthAwarePaginator) {
                // Another pagination type
                $items = $result->items();
                foreach ($items as $key => $leaveRequest) {
                    $items[$key] = $this->transformLeaveRequestForSubcontractor($leaveRequest);
                }
                $result->setCollection(collect($items));
                return $result;
            } elseif (method_exists($result, 'map')) {
                // Collection - transform each item
                return $result->map(function ($leaveRequest) {
                    return $this->transformLeaveRequestForSubcontractor($leaveRequest);
                });
            } elseif (method_exists($result, 'transform')) {
                // Collection with transform method
                $result->transform(function ($leaveRequest) {
                    return $this->transformLeaveRequestForSubcontractor($leaveRequest);
                });
                return $result;
            }
        }
        
        // Fallback: return as-is if we can't transform
        return $result;
    }

    protected function transformSubcontractorEmployeeToEmpDetails($subcontractorEmp, $customerId = null)
    {
        // Get default EMP role for subcontractor employees
        $defaultRole = Role::where('code', 'EMP')->where('del', '0')->first();
        
        // Map subcontractor employee to emp_details structure (similar to EmpCompanyDetails)
        $empDetails = (object)[
            'id' => $subcontractorEmp->id,
            'employee_email' => $subcontractorEmp->email ?? null,
            'employment_type' => null,
            'access_role' => $defaultRole ? $defaultRole->code : null,
            'work_permit_type' => null,
            'tax_file_no' => null,
            'tier_id' => null,
            'employment_start_date' => null,
            'employment_end_date' => null,
            'attendance_effective_from' => null,
            'status' => '1',
            'invited' => '1',
            'approved' => '1',
            'rejected' => '0',
            'approved_by' => null,
            'approved_status' => null,
            'approved_at' => null,
            'is_submitted' => '0',
            'compeleted' => '1',
            'credentials' => '0',
            'created_by' => $customerId ? (string)$customerId : null,
            'del' => '0',
            'created_at' => $subcontractorEmp->created_at,
            'updated_at' => $subcontractorEmp->updated_at,
            'two_factor' => 0,
            'force_reset' => $subcontractorEmp->force_reset ?? 0,
            'user_type' => '1', // External/subcontractor employee
            'job_title' => null,
            'trade_qualified' => $subcontractorEmp->trade_qualified ?? null,
            'trade_qualified_year' => $subcontractorEmp->trade_qualified_year ?? null,
            'trade_licensed' => $subcontractorEmp->trade_licensed ?? null,
            'trade_licensed_year' => $subcontractorEmp->trade_licensed_year ?? null,
            'work_experience' => $subcontractorEmp->work_experience ?? null,
            'temporary_student_visa' => null,
            'worker_type' => null,
            'year_commenced' => $subcontractorEmp->year_commenced ?? null,
            'classified_high_risk' => null,
            'citizenship_status' => $subcontractorEmp->citizenship_status ?? null,
            'citizenship_file' => null,
            'allergies' => $subcontractorEmp->allergies ?? null,
            'previous_injuries' => $subcontractorEmp->previous_injuries ?? null,
            'medical_condition' => $subcontractorEmp->medical_condition ?? null,
            'employer_name' => null,
            'legally_australia' => null,
            'link_key' => null,
            'xero_emp_id' => $subcontractorEmp->xero_emp_id ?? null,
            'signature' => $subcontractorEmp->signature ?? null,
            'details_allergies' => $subcontractorEmp->allergies ?? null,
            'details_previous_injuries' => $subcontractorEmp->previous_injuries ?? null,
            'emp_personal_details' => [
                'id' => null,
                'emp_id' => (string)$subcontractorEmp->id,
                'first_name' => $subcontractorEmp->first_name ?? null,
                'middle_name' => $subcontractorEmp->middle_name ?? null,
                'last_name' => $subcontractorEmp->last_name ?? null,
                'mobile' => $subcontractorEmp->mobile ?? null,
                'streat_address' => $subcontractorEmp->address ?? null,
                'suburb' => $subcontractorEmp->suburb ?? null,
                'state' => $subcontractorEmp->state ?? null,
                'postcode' => null,
                'image' => $subcontractorEmp->profile_image ?? null,
                'date_of_birth' => $subcontractorEmp->dob ?? null,
                'blood_group' => null,
                'bank_name' => null,
                'account_holder_name' => null,
                'ibn_number' => null,
                'account_number' => null,
                'created_by' => null,
                'del' => '0',
                'created_at' => $subcontractorEmp->created_at,
                'updated_at' => $subcontractorEmp->updated_at,
            ],
            'accessRole' => $defaultRole ? (object)[
                'id' => $defaultRole->id,
                'title' => $defaultRole->title,
                'code' => $defaultRole->code,
                'del' => (string)$defaultRole->del,
                'is_seeded' => $defaultRole->is_seeded,
                'created_at' => $defaultRole->created_at,
                'updated_at' => $defaultRole->updated_at,
            ] : null,
        ];
        
        return $empDetails;
    }

    protected function getLeavePackagesForEmployee($employeeId, $subcontractorId, $customerId, $workspaceId, $roleId = null)
    {
        // If subcontractor employee, use default EMP role
        if ($subcontractorId !== null) {
            $defaultRole = Role::where('code', 'EMP')->where('del', '0')->first();
            $roleId = $defaultRole ? $defaultRole->id : null;
        }
        
        if (!$roleId) {
            return collect();
        }
        
        $leavePackages = LeavePackage::whereRaw("FIND_IN_SET(?, employee_roles)", [$roleId])
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->get();
        
        // Calculate leave_count for each package
        return $leavePackages->map(function($package) use ($employeeId, $subcontractorId) {
            // Get all approved leave requests for this package
            $approvedLeavesQuery = LeaveRequest::where('employee_id', $employeeId)
                ->where('leave_package_id', $package->id)
                ->where('status', 1); // Approved leaves only
            
            // Filter by subcontractor_id if it's a subcontractor employee
            if ($subcontractorId !== null) {
                $approvedLeavesQuery->where('subcontractor_id', $subcontractorId);
            } else {
                $approvedLeavesQuery->whereNull('subcontractor_id');
            }
            
            $approved_leaves = $approvedLeavesQuery->get();
            
            // Calculate total leave days (not just count of requests)
            $total_leave_days = 0;
            foreach ($approved_leaves as $leave) {
                $fromDate = BaseModel::safeCarbonParse($leave->from, 'leave-calculations.from');
                $toDate = BaseModel::safeCarbonParse($leave->to, 'leave-calculations.to');
                
                if ($fromDate instanceof Carbon && $toDate instanceof Carbon) {
                    $total_leave_days += $fromDate->diffInDays($toDate) + 1; // +1 to include both start and End Dates
                }
            }
            
            $package->leave_count = $total_leave_days;
            return $package;
        });
    }
    
    protected function getEmployeeListWithSubcontractors($regular_employee_ids, $subcontractor_employee_ids, $dateRange, $site_id = null)
    {
        $regular_employees = collect([]);
        $subcontractor_employees = collect([]);

        // Query for regular employees (only if there are any)
        if (!empty($regular_employee_ids)) {
            $regular_employees_query = EmployeeAttendance::query()
                ->join('emp_personal_details', 'employee_attendances.employee_id', '=', 'emp_personal_details.emp_id')
                ->join('emp_company_details', 'employee_attendances.employee_id', '=', 'emp_company_details.id')
                ->leftJoin('roles', 'emp_company_details.access_role', '=', 'roles.code')
                ->whereIn('employee_attendances.employee_id', $regular_employee_ids)
                ->whereNull('employee_attendances.subcontractor_id')
                ->whereBetween('employee_attendances.date', [$dateRange['from'], $dateRange['to']]);

            if (!is_null($site_id)) {
                $regular_employees_query->where('employee_attendances.site_id', $site_id);
            }

            $regular_employees = $regular_employees_query
                ->groupBy('employee_attendances.employee_id')
                ->select(
                    'employee_attendances.employee_id as emp_id',
                    DB::raw('MAX(emp_personal_details.first_name) as first_name'),
                    DB::raw('MAX(emp_personal_details.middle_name) as middle_name'),
                    DB::raw('MAX(emp_personal_details.last_name) as last_name'),
                    DB::raw('MAX(emp_personal_details.image) as image'),
                    DB::raw('MAX(emp_company_details.user_type) as user_type'),
                    DB::raw('MAX(roles.title) as title')
                )
                ->get();
        }

        // Query for subcontractor employees (only if there are any)
        if (!empty($subcontractor_employee_ids)) {
            $subcontractor_employees_query = EmployeeAttendance::query()
                ->join('employees_subcontractors', 'employee_attendances.employee_id', '=', 'employees_subcontractors.id')
                ->whereIn('employee_attendances.employee_id', $subcontractor_employee_ids)
                ->whereNotNull('employee_attendances.subcontractor_id')
                ->whereBetween('employee_attendances.date', [$dateRange['from'], $dateRange['to']]);

            if (!is_null($site_id)) {
                $subcontractor_employees_query->where('employee_attendances.site_id', $site_id);
            }

            $subcontractor_employees = $subcontractor_employees_query
                ->groupBy('employee_attendances.employee_id')
                ->select(
                    'employee_attendances.employee_id as emp_id',
                    DB::raw('MAX(employees_subcontractors.first_name) as first_name'),
                    DB::raw('MAX(employees_subcontractors.middle_name) as middle_name'),
                    DB::raw('MAX(employees_subcontractors.last_name) as last_name'),
                    DB::raw('MAX(employees_subcontractors.profile_image) as image'),
                    DB::raw("'1' as user_type"), // Subcontractor employees have user_type = 1
                    DB::raw("NULL as title") // Subcontractor employees don't have roles
                )
                ->get();

            // Load full EmployeeSubcontractor models with associations to get subcontractors data
            $subcontractor_employee_models = EmployeeSubcontractor::whereIn('id', $subcontractor_employee_ids)
                ->with('associations.subcontractor:id,name,company_name')
                ->get()
                ->keyBy('id');

            // Map subcontractor employees to include subcontractors, empcompanydetails, and type
            $subcontractor_employees = $subcontractor_employees->map(function ($employee) use ($subcontractor_employee_models) {
                $employeeModel = $subcontractor_employee_models->get($employee->emp_id);
                
                // Get subcontractors from the model (using the accessor we created)
                $subcontractors = [];
                if ($employeeModel) {
                    $subcontractors = $employeeModel->subcontractors ?? [];
                }

                // Convert to array and add additional fields
                $employeeArray = $employee->toArray();
                $employeeArray['empcompanydetails'] = null; // Subcontractor employees don't have empcompanydetails
                $employeeArray['type'] = 'external'; // Subcontractor employees are external
                $employeeArray['subcontractors'] = $subcontractors;
                
                // Format image - use default if empty
                if (empty($employeeArray['image'])) {
                    $employeeArray['image'] = 'assets/img/default.png';
                }

                return $employeeArray;
            });
        }

        // Format regular employees to match the same structure
        $regular_employees = $regular_employees->map(function ($employee) {
            $employeeArray = $employee->toArray();
            $employeeArray['empcompanydetails'] = null; // Will be set elsewhere if needed
            $employeeArray['type'] = ($employee->user_type == 0) ? 'internal' : 'external';
            $employeeArray['subcontractors'] = []; // Regular employees don't have subcontractors
            
            // Format image - use default if empty
            if (empty($employeeArray['image'])) {
                $employeeArray['image'] = 'assets/img/default.png';
            }

            return $employeeArray;
        });

        // Combine both results
        return $regular_employees->concat($subcontractor_employees);
    }

    private function transformAttendanceForSubcontractor($attendance)
    {
        // Convert to array if it's a model instance
        $attendanceArray = $attendance instanceof \Illuminate\Database\Eloquent\Model 
            ? $attendance->toArray() 
            : (array)$attendance;
        
        // Check for subcontractor_id (can be in snake_case or camelCase)
        $subcontractorId = $attendanceArray['subcontractor_id'] ?? $attendanceArray['subcontractorId'] ?? null;
        
        // Get subcontractor employee data (check both naming conventions)
        $subcontractorEmp = $attendanceArray['subcontractor_employee'] ?? $attendanceArray['subcontractorEmployee'] ?? null;
        
        // If subcontractor_id is not null and subcontractor employee exists, use subcontractor employee details
        if ($subcontractorId !== null && $subcontractorEmp && !empty($subcontractorEmp)) {
            // Determine the correct key names based on what's already in the array
            $hasSnakeCase = isset($attendanceArray['emp_company_details']) || isset($attendanceArray['emp_personal_details']);
            $hasCamelCase = isset($attendanceArray['empCompanyDetails']) || isset($attendanceArray['empPersonalDetails']);
            
            // Set empcompanydetails (use the key that exists, or default to lowercase)
            $empCompanyKey = $hasSnakeCase ? 'emp_company_details' : ($hasCamelCase ? 'empCompanyDetails' : 'empcompanydetails');
            
            // Build empcompanydetails with structure matching regular employees
            $attendanceArray[$empCompanyKey] = [
                'id' => $subcontractorEmp['id'] ?? null,
                'employee_email' => $subcontractorEmp['email'] ?? null,
                'employment_type' => null,
                'access_role' => null,
                'work_permit_type' => null,
                'tax_file_no' => null,
                'tier_id' => null,
                'employment_start_date' => null,
                'employment_end_date' => null,
                'attendance_effective_from' => null,
                'status' => '1',
                'invited' => '1',
                'approved' => null,
                'rejected' => null,
                'approved_by' => null,
                'approved_status' => null,
                'approved_at' => null,
                'is_submitted' => null,
                'compeleted' => null,
                'credentials' => null,
                'created_by' => null,
                'del' => '0',
                'created_at' => null,
                'updated_at' => null,
                'two_factor' => null,
                'force_reset' => null,
                'user_type' => '1', // External/subcontractor employee
                'job_title' => null,
                'trade_qualified' => $subcontractorEmp['trade_qualified'] ?? null,
                'trade_qualified_year' => $subcontractorEmp['trade_qualified_year'] ?? null,
                'trade_licensed' => $subcontractorEmp['trade_licensed'] ?? null,
                'trade_licensed_year' => $subcontractorEmp['trade_licensed_year'] ?? null,
                'work_experience' => $subcontractorEmp['work_experience'] ?? null,
                'temporary_student_visa' => null,
                'worker_type' => null,
                'year_commenced' => $subcontractorEmp['year_commenced'] ?? null,
                'classified_high_risk' => null,
                'citizenship_status' => $subcontractorEmp['citizenship_status'] ?? null,
                'citizenship_file' => null,
                'allergies' => $subcontractorEmp['allergies'] ?? null,
                'previous_injuries' => $subcontractorEmp['previous_injuries'] ?? null,
                'medical_condition' => $subcontractorEmp['medical_condition'] ?? null,
                'employer_name' => null,
                'legally_australia' => null,
                'link_key' => null,
                'xero_emp_id' => $subcontractorEmp['xero_emp_id'] ?? null,
                'signature' => $subcontractorEmp['signature'] ?? null,
                'details_allergies' => null,
                'details_previous_injuries' => null,
            ];
            
            // Set emp_personal_details (use snake_case to match regular employee structure)
            $empPersonalData = [
                'id' => null,
                'emp_id' => (string)($subcontractorEmp['id'] ?? ''),
                'first_name' => $subcontractorEmp['first_name'] ?? null,
                'middle_name' => $subcontractorEmp['middle_name'] ?? null,
                'last_name' => $subcontractorEmp['last_name'] ?? null,
                'mobile' => $subcontractorEmp['mobile'] ?? null,
                'streat_address' => $subcontractorEmp['address'] ?? null,
                'suburb' => $subcontractorEmp['suburb'] ?? null,
                'state' => $subcontractorEmp['state'] ?? null,
                'postcode' => null,
                'image' => $subcontractorEmp['profile_image'] ?? null,
                'date_of_birth' => $subcontractorEmp['dob'] ?? null,
                'blood_group' => null,
                'bank_name' => null,
                'account_holder_name' => null,
                'ibn_number' => null,
                'account_number' => null,
                'created_by' => null,
                'del' => '0',
                'created_at' => null,
                'updated_at' => null,
                'subcontractors' => $subcontractorEmp['subcontractors'] ?? null,
            ];
            
            // Explicitly set type to 'external' for subcontractor employees
            $attendanceArray['type'] = 'external';
            
            // Set both snake_case and camelCase keys to ensure compatibility
            $attendanceArray['emp_personal_details'] = $empPersonalData;
            // $attendanceArray['empPersonalDetails'] = $empPersonalData;
            
            // Also set empcompanydetails in lowercase (matching regular employee response structure)
            // $attendanceArray['empcompanydetails'] = $attendanceArray[$empCompanyKey];
            
            // Also set EmpCompanyDetails (PascalCase) if it exists in the original array
            if (isset($attendanceArray['EmpCompanyDetails'])) {
                $attendanceArray['EmpCompanyDetails'] = $attendanceArray[$empCompanyKey];
            }
        unset($attendanceArray['subcontractor_employee']);

        } else {
            // For regular employees, ensure type is set correctly based on empcompanydetails
            // The accessor will handle this, but we can also set it explicitly if needed
            if (!isset($attendanceArray['type'])) {
                // Type will be set by the accessor when the model is converted to array
                // But if it's already an array, we need to check empcompanydetails
                $empCompanyDetails = $attendanceArray['empcompanydetails'] ?? $attendanceArray['empCompanyDetails'] ?? $attendanceArray['emp_company_details'] ?? null;
                if ($empCompanyDetails && isset($empCompanyDetails['user_type'])) {
                    $attendanceArray['type'] = ($empCompanyDetails['user_type'] == 0) ? 'internal' : 'external';
                }
            }
        }
        
        // Calculate working hours for all attendance records (both subcontractor and regular employees)
        // if check_in and check_out exist and working_hours is 0 or missing
        if (isset($attendanceArray['check_in']) && isset($attendanceArray['check_out']) && 
            $attendanceArray['check_in'] && $attendanceArray['check_out']) {
            
            try {
                $checkIn = \Carbon\Carbon::parse($attendanceArray['check_in']);
                $checkOut = \Carbon\Carbon::parse($attendanceArray['check_out']);
                $totalMinutes = $checkOut->diffInMinutes($checkIn);
                
                // Subtract break time if breaks exist
                $totalBreakMinutes = 0;
                $breaks = $attendanceArray['breaks'] ?? [];
                if (is_array($breaks)) {
                    foreach ($breaks as $break) {
                        if (isset($break['break_in']) && isset($break['break_out']) && 
                            $break['break_in'] && $break['break_out']) {
                            try {
                                $breakIn = \Carbon\Carbon::parse($break['break_in']);
                                $breakOut = \Carbon\Carbon::parse($break['break_out']);
                                $totalBreakMinutes += $breakOut->diffInMinutes($breakIn);
                            } catch (\Exception $e) {
                                // Skip invalid break times
                            }
                        }
                    }
                }
                
                // Calculate working hours in minutes (not converting to hours)
                $workingMinutes = $totalMinutes - $totalBreakMinutes;
                
                // Always recalculate and update working_hours to ensure accuracy
                // This handles cases where working_hours is 0, stored incorrectly, or needs recalculation
                // Store as minutes (not hours)
                $attendanceArray['working_hours'] = (string)$workingMinutes;
            } catch (\Exception $e) {
                // If parsing fails, keep original working_hours
            }
        }
        
        // Return as array for consistency
        return $attendanceArray;
    }

    public function handleFileImageUpload(Request $request, $directory = null, $currentFiles = null)
    {
        $directory = $directory ?? 'uploads'; // Default directory for uploads
        $uploadedPaths = [];
        $files = $request->allFiles(); // Get all uploaded files from the request
        foreach ($files as $key => $file) {
            if (!$file instanceof \Illuminate\Http\UploadedFile && !is_array($file)) {
                continue; // Skip non-file inputs
            }
            $fileItems = is_array($file) ? $file : [$file];
            foreach ($fileItems as $fileItem) {
                if (!$fileItem instanceof \Illuminate\Http\UploadedFile || !$fileItem->isValid()) {
                    continue;
                }
                // Get file extension and sanitize filename
                $extension = strtolower($fileItem->getClientOriginalExtension());
                $originalName = pathinfo($fileItem->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = preg_replace('/[^a-zA-Z0-9_-]/', '', $originalName);
                $sanitizedFileName = substr($sanitizedFileName, 0, 100); // Limit filename length
                $uniqueFileName = time() . '_' . $sanitizedFileName . '.' . $extension;
                // Validate file type
                $allowedImageExtensions = ['jpg', 'jpeg', 'jfif', 'png', 'gif', 'webp'];
                $allowedDocumentExtensions = ['pdf', 'csv', 'doc', 'docx', 'xlsx', 'txt'];
                if (!in_array($extension, array_merge($allowedImageExtensions, $allowedDocumentExtensions))) {
                    continue; // Skip invalid file types
                }
                // Move file to public directory
                $fileItem->move(public_path($directory), $uniqueFileName);
                $uploadedPaths[] = [
                    'key' => $key, // Dynamic key
                    'path' => "$directory/$uniqueFileName",
                    'type' => in_array($extension, $allowedImageExtensions) ? 'image' : 'document'
                ];
                // Delete old file if it exists
                if (is_string($currentFiles) && File::exists(public_path($currentFiles))) {
                    File::delete(public_path($currentFiles));
                }
                if (is_array($currentFiles) && isset($currentFiles[$key])) {
                    foreach ((array) $currentFiles[$key] as $oldFilePath) {
                        if (File::exists(public_path($oldFilePath))) {
                            File::delete(public_path($oldFilePath));
                        }
                    }
                }
            }
        }
        return count($uploadedPaths) === 1 ? $uploadedPaths[0] : $uploadedPaths;
    }

    function createSlug($title)
    {
        return strtolower(preg_replace('/[^A-Za-z0-9-]+/', '-', trim($title)));
    }

    function generateTicketId()
    {
        do {
            $ticketId = 'TK-' . strtoupper(substr(uniqid(), -6));
        } while (\App\Models\HelpdeskTicket::where('ticket_id', $ticketId)->exists());
        return $ticketId;
    }

    function generateCsvFromQuery($query)
    {
        if ($query->getModel() === null) {
            return $this->message('Invalid query. Ensure the query is a valid Eloquent query.');
        }
        $modelName = strtolower(class_basename($query->getModel()));
        $directory = 'csv';
        $directoryPath = public_path($directory);
        if (!File::exists($directoryPath)) {
            File::makeDirectory($directoryPath, 0755, true);
        }
        $fileName = "{$modelName}.csv";
        $filePath = "{$directory}/{$fileName}";
        $fullFilePath = public_path($filePath);
        if (File::exists($fullFilePath)) {
            File::delete($fullFilePath);
        }
        $file = fopen($fullFilePath, 'w');
        $records = $query->get();
        if ($records->isEmpty()) {
            fclose($file);
            return $this->message('The query returned no records to export.');
        }
        $headers = array_keys($records->first()->toArray());
        fputcsv($file, $headers);
        foreach ($records as $record) {
            $flattenedRecord = array_map(function ($value) {
                if (is_array($value) || is_object($value)) {
                    return json_encode($value); // Convert to JSON string
                }
                return $value;
            }, $record->toArray());
            fputcsv($file, $flattenedRecord);
        }
        fclose($file);
        return asset($filePath);
    }

    function generateOrderId()
    {
        do {

            $OrderId = 'ODR-' . strtoupper(substr(uniqid(), -6));
        } while (\App\Models\Order::where('order_id', $OrderId)->exists());
        return $OrderId;
    }

    public static function generateCode($prefix = null)
    {
        $code = Str::upper(Str::random(10));
        if ($prefix) {
            return $prefix . $code;
        }
        return $code;
    }

    function getUserTable($user = null)
    {
        if (!$user) {
            $user = Auth::user();
        }
        if ($user instanceof \App\Models\User) {
            return 'customer';
        }
        if ($user instanceof \App\Models\EmpCompanyDetails) {
            return 'emp';
        }
        return null; // Return null if the user does not match either table
    }

    protected function getCustomerAndWorkspaceIds()
    {
       
        $userTable = $this->getUserTable();
        if($userTable === 'customer' && Auth::user()->user_type == config('constants.user_types.subcontractor')) {
            return [
                'customer_id' => Auth::user()->current_company ?? auth()->user()->subcontractorCompanies->first()->customer_id,
                'workspace_id' => Auth::user()->current_company_workspace ?? auth()->user()->subcontractorCompanies->first()->workspace_id,
                'flag' => 'subcontractor',
            ];
        }
        elseif ($userTable === 'customer') {
            return [
                'customer_id' => Auth::id(),
                'workspace_id' => Auth::user()->current_workspace_id,
                'flag' => 'customer',
            ];
        } elseif ($userTable === 'emp') {
            return [
                'customer_id' => Auth::user()->customer_id,
                'workspace_id' => Auth::user()->workspace_id,
                'flag' => 'emp',
            ];
        }
        return null;
    }

    protected function applyCustomerWorkspaceFilter($query)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $query->whereRaw('1 = 0'); // No results
        }
        return $query->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id']);
    }

    private function getPaymentStatusConfig($paymentStatus)
    {
        $configs = [
            // Pending - 0
            config('constants.payment_statuses.pending') => [
                'subject' => 'New Pending Payment Request',
                'status_text' => 'Pending',
                'color' => '#F59E0B', // Amber
                'background' => 'linear-gradient(135deg, #fff 0%, #fff 100%)',
                'icon' => '⏳',
                'message' => 'A new payment request requires your attention and approval.',
                'action_required' => true,
                'action_message' => 'Please verify the payment and approve this request through the admin dashboard'
            ],
            // Completed - 1
            config('constants.payment_statuses.completed') => [
                'subject' => 'Payment Successfully Processed',
                'status_text' => 'Completed',
                'color' => '#22C55E', // Green
                'background' => 'linear-gradient(135deg, #fff 0%, #fff 100%)',
                'icon' => '✅',
                'message' => 'A payment has been successfully processed and confirmed.',
                'action_required' => false,
                'action_message' => ''
            ],
            // Failed - 2
            config('constants.payment_statuses.failed') => [
                'subject' => 'Payment Failed',
                'status_text' => 'Failed',
                'color' => '#EF4444', // Red
                'background' => 'linear-gradient(135deg, #fff 0%, #fff 100%)',
                'icon' => '❌',
                'message' => 'A payment attempt has failed and requires attention.',
                'action_required' => true,
                'action_message' => 'Please review the payment details and contact the customer if necessary'
            ],
            // Refunded - 3
            config('constants.payment_statuses.refunded') => [
                'subject' => 'Payment Refunded',
                'status_text' => 'Refunded',
                'color' => '#6366F1', // Indigo
                'background' => 'linear-gradient(135deg, #fff 0%, #fff 100%)',
                'icon' => '↩️',
                'message' => 'A payment has been refunded to the customer.',
                'action_required' => false,
                'action_message' => ''
            ],
            // Rejected - 4
            config('constants.payment_statuses.rejected') => [
                'subject' => 'Payment Rejected',
                'status_text' => 'Rejected',
                'color' => '#DC2626', // Dark Red
                'background' => 'linear-gradient(135deg, #fff 0%, #fff 100%)',
                'icon' => '⛔',
                'message' => 'A payment request has been rejected.',
                'action_required' => false,
                'action_message' => 'The payment has been reviewed and rejected'
            ]
        ];
        // Return the config for the requested status, or fallback to pending if status not found
        return $configs[$paymentStatus] ?? $configs[config('constants.payment_statuses.pending')];
    }


    private function getPaymentTypeName($paymentType, $paymentStatus)
    {
        $type = 'Other';
        // Determine basic payment type
        if ($paymentType == config('constants.payment_types.bank')) {
            $type = 'Bank Transfer';
        } else if ($paymentType == config('constants.payment_types.stripe')) {
            $type = 'Stripe';
        } else if ($paymentType == config('constants.payment_types.pay_pal')) {
            $type = 'PayPal';
        }
        // Add status context for specific cases
        if (
            $paymentType == config('constants.payment_types.bank') &&
            $paymentStatus == config('constants.payment_statuses.pending')
        ) {
            $type .= ' (Pending)';
        } else if ($paymentStatus == config('constants.payment_statuses.failed')) {
            $type .= ' (Failed)';
        } else if ($paymentStatus == config('constants.payment_statuses.refunded')) {
            $type .= ' (Refunded)';
        } else if ($paymentStatus == config('constants.payment_statuses.rejected')) {
            $type .= ' (Rejected)';
        }
        return $type;
    }

    public function getTierEmployees($tierKeys, $query = null)
    {
        if (!$query) {
            $query = EmpCompanyDetails::query();
        }
        $tierIds = Tier::whereIn('tier_key', $tierKeys)->pluck('id')->toArray();
        $query->whereIn('tier_id', $tierIds);
        return $query;
    }

    function createConstructionRoles($customer_id, $workspace_id, $created_by = null)
    {
        // Roles are now global and managed by super admin
        // This function now only creates default team for customer/workspace
        $created_by = $created_by ?? $customer_id;

        // Create a default team if it doesn't exist
        $this->createDefaultTeam($customer_id, $workspace_id, $created_by);

        // Return empty array as roles are no longer created per customer
        return [];
    }

    /**
     * Create a default team if it doesn't exist
     */
    private function createDefaultTeam($customer_id, $workspace_id, $created_by = null)
    {
        $created_by = $created_by ?? $customer_id;

        // Check if a default team already exists for this customer/workspace
        $existingTeam = EmpTeam::where('customer_id', $customer_id)
            ->where('workspace_id', $workspace_id)
            ->where('del', '0')
            ->first();

        if (!$existingTeam) {
            // Create a default team with null supervisor (since employees might not exist yet)
            EmpTeam::create([
                'title' => 'Default Team',
                'supervisor' => null, // Will be updated later when employees are created
                'description' => 'Default team created automatically during role setup',
                'customer_id' => $customer_id,
                'workspace_id' => $workspace_id,
                'created_by' => $created_by,
                'del' => '0',
            ]);
        }
    }

    private function getSuperAdminId()
    {
        $superAdmin = User::where('user_type', config('constants.user_types.admin'))->first();
        return $superAdmin->id;
    }

    public function formatSystemDate($dateInput)
    {
        if (empty($dateInput)) {
            return null;
        }

        try {
            $dateFormat = $this->getSystemDateFormat();

            if ($dateInput instanceof Carbon) {
                $carbonDate = $dateInput;
            } elseif (is_numeric($dateInput)) {
                $carbonDate = Carbon::createFromTimestamp($dateInput);
            } elseif (is_string($dateInput)) {
                $carbonDate = Carbon::parse($dateInput);
            } else {
                return null; // Invalid input type
            }

            $formattedDate = $carbonDate->format($dateFormat);

            $time = $carbonDate->format('H:i:s');
            if ($time !== '00:00:00') {
                return $formattedDate . ' ' . $time; // Return date with time
            }

            return $formattedDate; // Return date only

        } catch (\Exception $e) {
            // Log error and return null
            Log::error('Date formatting error: ' . $e->getMessage(), [
                'input' => $dateInput,
            ]);
            return null;
        }
    }

    /**
     * Apply intelligent name search to a query
     * Handles full names, combinations, and partial matches
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $searchTerm
     * @param string $relationName The relationship name (e.g., 'empPersonalDetails', 'empDetails')
     * @param array $nameFields Array of field names ['first_name', 'middle_name', 'last_name']
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function applyNameSearch($query, $searchTerm, $relationName = null, $nameFields = ['first_name', 'middle_name', 'last_name'])
    {
        if (!$relationName) {
            $relationName = 'empPersonalDetails';
        }
        return $query->whereHas($relationName, function ($subquery) use ($searchTerm, $nameFields) {
            $subquery->where(function ($nameQuery) use ($searchTerm, $nameFields) {
                // Split search term into words for full name search
                $nameParts = explode(' ', trim($searchTerm));

                if (count($nameParts) == 1) {
                    // Single word - search in any name field
                    $nameQuery->where($nameFields[0], 'like', '%' . $nameParts[0] . '%')
                        ->orWhere($nameFields[1], 'like', '%' . $nameParts[0] . '%')
                        ->orWhere($nameFields[2], 'like', '%' . $nameParts[0] . '%');
                } elseif (count($nameParts) == 2) {
                    // Two words - could be first+last or first+middle
                    $nameQuery->where(function ($twoWordQuery) use ($nameParts, $nameFields) {
                        $twoWordQuery->where($nameFields[0], 'like', '%' . $nameParts[0] . '%')
                            ->where($nameFields[2], 'like', '%' . $nameParts[1] . '%')
                            ->orWhere($nameFields[0], 'like', '%' . $nameParts[0] . '%')
                            ->where($nameFields[1], 'like', '%' . $nameParts[1] . '%');
                    });
                } elseif (count($nameParts) >= 3) {
                    // Three or more words - full name search
                    $nameQuery->where($nameFields[0], 'like', '%' . $nameParts[0] . '%')
                        ->where($nameFields[1], 'like', '%' . $nameParts[1] . '%')
                        ->where($nameFields[2], 'like', '%' . $nameParts[2] . '%');
                }

                // Also search for the full term in any field (for partial matches)
                $nameQuery->orWhere($nameFields[0], 'like', '%' . $searchTerm . '%')
                    ->orWhere($nameFields[1], 'like', '%' . $searchTerm . '%')
                    ->orWhere($nameFields[2], 'like', '%' . $searchTerm . '%');
            });
        });
    }

    /**
     * Get tier permissions for a specific module by module name
     * 
     * @param EmpCompanyDetails|User $employee The employee or user object
     * @param string $moduleName The name of the module (e.g., "Sites", "Roster", "Projects")
     * @return array|null Returns module permissions with sub_modules, or null if not found
     */
    protected function getTierPermissionsByModule($employee, $moduleName)
    {
        // Get tier_id from employee
        $tierId = null;
        if ($employee instanceof EmpCompanyDetails) {
            $tierId = $employee->tier_id;
        } elseif ($employee instanceof User) {
            // For customers, you might need to get tier differently
            // For now, return null for customers
            return null;
        }

        if (!$tierId) {
            return null;
        }

        // Get the module by name
        $module = DB::table('permissions_modules')
            ->where('title', $moduleName)
            ->where('del', 0)
            ->first(['id', 'title', 'image', 'link', 'priority']);

        if (!$module) {
            return null;
        }

        // Get module-level permission
        $modulePermission = DB::table('tier_permissions')
            ->where('tier_id', $tierId)
            ->where('module_id', $module->id)
            ->whereNull('sub_module_id')
            ->first(['view', 'maintain']);

        $modulePermissions = [
            'module_id' => $module->id,
            'module_name' => $module->title,
            'image' => $module->image,
            'link' => $module->link,
            'priority' => $module->priority,
            'view' => $modulePermission ? (int)$modulePermission->view : 0,
            'maintain' => $modulePermission ? (int)$modulePermission->maintain : 0,
            'sub_modules' => []
        ];

        // Get sub-modules if module has view or maintain permission
        if (!$modulePermission || $modulePermission->view || $modulePermission->maintain) {
            $subModuleIds = DB::table('permissions_sub_modules')
                ->where('module_id', $module->id)
                ->where('del', 0)
                ->pluck('id');

            $modulePermissions['sub_modules'] = DB::table('tier_permissions')
                ->where('tier_id', $tierId)
                ->whereIn('sub_module_id', $subModuleIds)
                ->where(function ($query) {
                    $query->where('view', 1)
                        ->orWhere('maintain', 1);
                })
                ->get()
                ->map(function ($permission) {
                    $subModule = DB::table('permissions_sub_modules')
                        ->where('id', $permission->sub_module_id)
                        ->first();

                    if (!$subModule) {
                        return null;
                    }

                    return [
                        'sub_module_id' => $subModule->id,
                        'sub_module_name' => $subModule->title,
                        'view' => (int)$permission->view,
                        'maintain' => (int)$permission->maintain,
                        'image' => $subModule->image,
                        'link' => $subModule->link,
                    ];
                })
                ->filter() // Remove null values
                ->values()
                ->toArray();
        }

        // Only return if module has permissions or has sub-modules
        if (($modulePermission && ($modulePermission->view || $modulePermission->maintain)) ||
            !empty($modulePermissions['sub_modules'])
        ) {
            return $modulePermissions;
        }

        return null;
    }

    /**
     * Get employee or subcontractor induction documents with signed and not signed status
     * 
     * @param int|null $employeeId The employee ID or subcontractor ID (if null, uses Auth::user() if employee)
     * @return array|null Returns array with 'signed' and 'not_signed' documents, or null if not an employee or subcontractor
     */
    public function getEmployeeInductionDocuments($employeeId = null)
    {
        $isSubcontractor = false;
        $employee = null;
        $subcontractor = null;
        $customerId = null;
        $workspaceId = null;
        $roleType = null;

        // If no employee ID provided, try to get from authenticated user
        if ($employeeId === null) {
            $userTable = $this->getUserTable();
            if ($userTable !== 'emp') {
                return null; // Not an employee
            }
            $employeeId = Auth::id();
        }

        // First, try to get employee details
        $employee = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
            ->where('id', $employeeId)
            ->first();

        if ($employee) {
            $customerId = $employee->customer_id;
            $workspaceId = $employee->workspace_id;
            $roleType = $employee->user_type == 0 ? 'internal' : 'external';
        } else {
            // If not an employee, check if it's a subcontractor
            $subcontractor = \App\Models\LinkManagement::where('id', $employeeId)->first();
            if ($subcontractor) {
                $isSubcontractor = true;
                $customerId = $subcontractor->customer_id;
                $workspaceId = $subcontractor->workspace_id;
                $roleType = 'subcontractor';
            } else {
                return null; // Not an employee or subcontractor
            }
        }

        // Get all active induction documents for this customer/workspace
        $allDocuments = \App\Models\InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->orderBy('title')
            ->orderBy('version', 'desc')
            ->get();

        // Group by title and get only the latest version for each document
        $documents = [];
        foreach ($allDocuments as $doc) {
            $title = $doc->title;
            if (!isset($documents[$title])) {
                $documents[$title] = $doc;
            }
        }
        $documents = array_values($documents); // Convert to indexed array

        $signedDocuments = [];
        $notSignedDocuments = [];
        $validSignatures = [];
        $invalidSignatures = [];

        foreach ($documents as $doc) {
            $roleTypes = $doc->role_types ?? ['all'];

            // Check if employee/subcontractor matches role types
            if (!in_array('all', $roleTypes) && !in_array($roleType, $roleTypes)) {
                continue; // Skip if doesn't match role types
            }

            // Check for valid signature for this specific version
            $signatureQuery = \App\Models\InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('del', '0')
                ->where('is_valid', true);

            if ($isSubcontractor) {
                // Subcontractors use user_id field in signatures
                $signatureQuery->where('user_id', $employeeId);
            } else {
                // Employees use employee_id field in signatures
                $signatureQuery->where('employee_id', $employeeId);
            }

            // Get valid signature for current version
            $validSignature = $signatureQuery->first();
            
            // If not signed current version BUT it's a minor update, check chain
            if (!$validSignature && ($doc->update_type === 'minor')) {
                // Find root document
                $rootDocumentId = $doc->id;
                $currentDoc = $doc;
                while ($currentDoc && $currentDoc->parent_document_id) {
                    $rootDocumentId = $currentDoc->parent_document_id;
                    $currentDoc = \App\Models\InductionDocument::find($rootDocumentId);
                }
                
                // Get all IDs in chain
                $documentIdsInChain = \App\Models\InductionDocument::where(function($q) use ($rootDocumentId) {
                        $q->where('id', $rootDocumentId)
                          ->orWhere('parent_document_id', $rootDocumentId);
                    })
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->pluck('id')
                    ->toArray();
                    
                // Check for signature on any version
                $chainQuery = \App\Models\InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
                    ->where('del', '0')
                    ->where('is_valid', true)
                    ->orderBy('signed_at', 'desc');
                    
                if ($isSubcontractor) {
                    $chainQuery->where('user_id', $employeeId);
                } else {
                    $chainQuery->where('employee_id', $employeeId);
                }
                
                $validSignature = $chainQuery->first();
            }

            // Get all signatures (valid and invalid) for current version for detail view
            $allSignaturesQuery = \App\Models\InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('del', '0');
                
            if ($isSubcontractor) {
                $allSignaturesQuery->where('user_id', $employeeId);
            } else {
                $allSignaturesQuery->where('employee_id', $employeeId);
            }
            $allSignatures = $allSignaturesQuery->get();

            // Get invalid signatures
            $invalidSignature = $allSignatures->where('is_valid', false)->first();

            $documentData = [
                'id' => $doc->id,
                'title' => $doc->title,
                'document_type' => $doc->document_type,
                'version' => $doc->version,
                'file_path' => $doc->file_path,
                'description' => $doc->description,
                'role_types' => $doc->role_types ?? [],
                'created_at' => $doc->created_at,
            ];

            if ($validSignature) {
                // Add signature details
                $documentData['signature'] = [
                    'id' => $validSignature->id,
                    'signature_path' => $validSignature->signature_path,
                    'signed_at' => $validSignature->signed_at,
                    'notes' => $validSignature->notes,
                    'is_valid' => $validSignature->is_valid,
                ];
                $signedDocuments[] = $documentData;
                $validSignatures[] = $documentData;
            } elseif ($invalidSignature) {
                // Document has invalid signature
                $documentData['signature'] = [
                    'id' => $invalidSignature->id,
                    'signature_path' => $invalidSignature->signature_path,
                    'signed_at' => $invalidSignature->signed_at,
                    'notes' => $invalidSignature->notes,
                    'is_valid' => $invalidSignature->is_valid,
                ];
                $invalidSignatures[] = $documentData;
                // Also add to not_signed since it's not valid
                $notSignedDocuments[] = $documentData;
            } else {
                // No signature at all
                $notSignedDocuments[] = $documentData;
            }
        }

        $totalValidSignatures = count($validSignatures);
        $totalInvalidSignatures = count($invalidSignatures);
        $totalSigned = count($signedDocuments);
        $totalNotSigned = count($notSignedDocuments);
        $totalDocuments = $totalSigned + $totalNotSigned;

        return [
            'signed' => $signedDocuments,
            'not_signed' => $notSignedDocuments,
            'valid_signatures' => $validSignatures,
            'invalid_signatures' => $invalidSignatures,
            'total_signed' => $totalSigned,
            'total_not_signed' => $totalNotSigned,
            'total_documents' => $totalDocuments,
            'total_valid_signatures' => $totalValidSignatures,
            'total_invalid_signatures' => $totalInvalidSignatures,
            'all_signatures_valid' => $totalInvalidSignatures === 0 && $totalValidSignatures > 0,
            'has_invalid_signatures' => $totalInvalidSignatures > 0,
            'all_documents_signed' => $totalNotSigned === 0,
        ];
    }

    protected function getSubcontractorsForCustomer()
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        return SubcontractorCompany::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->whereHas('user', function ($query) {
                $query->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0');
            })
            ->with(['user' => function ($query) {
                $query->select('id', 'name', 'email');
            }])
            ->get()
            ->map(function ($subcontractorCompany) {
                return $subcontractorCompany->user;
            })
            ->filter()
            ->unique('id')
            ->values();
    }


    /**
     * Get subcontractor employees
     * 
     * @param int $subcontractorId The subcontractor ID (user ID where user_type = 5)
     * @param bool $attendance Flag for future attendance calculation (currently not used)
     * @param int|null $companyId Optional company_id filter
     * @param bool $activeOnly Only return active employees (default: true)
     * @param array|null $employeeIds Optional array of specific employee IDs to filter
     * @return \Illuminate\Database\Eloquent\Collection Collection of EmployeeSubcontractor models
     */
    protected function subcontractorEmployeesGetter($subcontractorId, $attendance = false, $companyId = null, $activeOnly = true, $employeeIds = null)
    {
        // Get employee IDs from employees_subcontractors_metas table
        $associationQuery = EmployeeSubcontractorMeta::where('subcontractor_id', $subcontractorId);
        
        // Filter by active status if requested
        if ($activeOnly) {
            $associationQuery->where('active', true);
        }
        
        // Filter by specific employee IDs if provided
        if ($employeeIds !== null && !empty($employeeIds)) {
            $associationQuery->whereIn('emp_id', $employeeIds);
        }
        
        $metaEmployeeIds = $associationQuery->pluck('emp_id')->toArray();
        
        if (empty($metaEmployeeIds)) {
            return collect([]);
        }
        
        // Get all employee data from employees_subcontractors table
        $employees = EmployeeSubcontractor::whereIn('id', $metaEmployeeIds)->get();
        
        return $employees;
    }

    /**
     * Get subcontractor employees for roster
     * This function retrieves subcontractor employees with their roster assignments
     * 
     * @param int $customerId
     * @param int $workspaceId
     * @param array $datesArray
     * @return array
     */
    protected function getSubcontractorEmployeesForRoster($customerId, $workspaceId, $datesArray, $siteId = null, 
            $projectId = null ,$documentationCheck = false)
    {
        // Get all project IDs for this customer/workspace
        $projectIds = DB::table('projects')
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('is_deleted', '0')
            ->pluck('id')
            ->toArray();

        if (empty($projectIds)) {
            return [];
        }

        // Get subcontractor metas where project_ids intersect with customer's projects
        // AND employee has accepted invitation for this customer
        $subcontractorMetas = DB::table('employees_subcontractors_metas')
            ->join('subcontractor_employee_invitations', function ($join) use ($customerId, $workspaceId) {
                $join->on('employees_subcontractors_metas.emp_id', '=', 'subcontractor_employee_invitations.employee_id')
                    ->where('subcontractor_employee_invitations.customer_id', $customerId)
                    ->where('subcontractor_employee_invitations.workspace_id', $workspaceId)
                    ->where('subcontractor_employee_invitations.invitation_status', 'accepted');
            })
            ->where(function ($query) use ($projectIds) {
                foreach ($projectIds as $projectId) {
                    $query->orWhere('employees_subcontractors_metas.project_ids', 'like', '%' . $projectId . '%');
                }
            })
            ->where('employees_subcontractors_metas.active', 1)
            ->select('employees_subcontractors_metas.emp_id')
            ->distinct()
            ->get();

        $employeeIds = $subcontractorMetas->pluck('emp_id')->unique()->toArray();

        if (empty($employeeIds)) {
            return [];
        }

        // Get employees with their associations
        $employees = EmployeeSubcontractor::whereIn('id', $employeeIds)
            ->with(['associations.subcontractor'])
            ->get();

        $subcontractorEmployees = [];

        foreach ($employees as $employee) {
            // Double check: Employee must have accepted invitation for this customer
            $query = SubcontractorEmployeeInvitation::query()
                ->where('employee_id', $employee->id)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('invitation_status', 'accepted');

            if ($documentationCheck) {
                $query->where('required_docs_status', 1)
                    ->where('induction_status', 1);
            }

            $invitation = $query->first();

            if (!$invitation) {
                continue;
            }


            // Check induction completion (Status 1 = Complete)
            if ($invitation->induction_status != 1) {
                continue;
            }

            // Check document approval (Status 1 = Complete)
            if ($invitation->required_docs_status != 1) {
                continue;
            }

            // Get associated subcontractors and projects - ONLY with accepted invitations
            $associatedData = $this->getEmployeeAssociations($employee, $projectIds, $customerId, $workspaceId);

            // Get subcontractor IDs for this employee that belong to this customer/workspace
            $subcontractorIds = EmployeeSubcontractorMeta::where('emp_id', $employee->id)
                ->where('active', 1)
                ->pluck('subcontractor_id')
                ->toArray();
            
            // Get subcontractor companies for these subcontractors that match customer/workspace
            $validSubcontractorIds = SubcontractorCompany::whereIn('user_id', $subcontractorIds)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->pluck('user_id')
                ->toArray();

            // Build shifts array with actual roster assignments
            $userShifts = [];
            foreach ($datesArray as $date) {
                // Get roster assignments for this employee on this date with matching subcontractor_id
                $shiftsOfThisBox = RosterAssign::with('rosterTemplate', 'site')
                    ->where('assign_to', $employee->id)
                    ->where('schedule_date', $date)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId);
                
                // Filter by subcontractor_id if we have valid subcontractors
                if (!empty($validSubcontractorIds)) {
                    $shiftsOfThisBox->whereIn('subcontractor_id', $validSubcontractorIds);
                } else {
                    $shiftsOfThisBox->whereNull('subcontractor_id');
                }
                
                // Apply site_id filter if provided
                if ($siteId !== null) {
                    $shiftsOfThisBox->where('site_id', $siteId);
                }
                
                // Apply project_id filter if provided
                if ($projectId !== null) {
                    $shiftsOfThisBox->where('project_id', $projectId);
                }
                
                $shiftsOfThisBox = $shiftsOfThisBox->get();

                $holiday = PublicHoliday::where('from', '<=', $date)
                    ->where('to', '>=', $date)
                    ->first();

                // Subcontractor employees don't have leave requests in this system
                $leave = null;

                $shifts = [];
                foreach ($shiftsOfThisBox as $shift) {
                    if ($shift->rosterTemplate) {
                        // Check for conflict using hasShiftConflict if method exists
                        // Convert to array format expected by hasShiftConflict
                        $conflict = false;
                        if (method_exists($this, 'hasShiftConflict')) {
                            // Convert Eloquent models to array format for hasShiftConflict
                            $shiftsArray = $shiftsOfThisBox->map(function($s) {
                                if ($s->rosterTemplate) {
                                    return [
                                        'roster_template' => [
                                            'start_time' => $s->rosterTemplate->start_time,
                                            'end_time' => $s->rosterTemplate->end_time,
                                        ]
                                    ];
                                }
                                return null;
                            })->filter()->values()->toArray();
                            
                            $shiftArray = [
                                'roster_template' => [
                                    'start_time' => $shift->rosterTemplate->start_time,
                                    'end_time' => $shift->rosterTemplate->end_time,
                                ]
                            ];
                            
                            $conflict = $this->hasShiftConflict($shiftsArray, $shiftArray);
                        }
                        
                        $shifts[] = [
                            'start_time' => $shift->rosterTemplate->start_time,
                            'end_time' => $shift->rosterTemplate->end_time,
                            'color_code' => $shift->rosterTemplate->color_code,
                            'template_id' => $shift->roster_template_id,
                            'conflict' => $conflict,
                            'site_id' => $shift->site_id,
                            'site_title' => $shift->site ? $shift->site->title : null,
                            'project_id' => $shift->project_id,
                        ];
                    }
                }

                $userShifts[] = [
                    'date' => $date,
                    'holiday' => $holiday ? $holiday->title : null,
                    'leave' => $leave,
                    'shifts' => $shifts
                ];
            }

            // Format personal details similar to regular employees
            $personalDetails = [
                'emp_id' => (string)$employee->id,
                'first_name' => $employee->first_name,
                'middle_name' => $employee->middle_name,
                'last_name' => $employee->last_name,
                'image' => $employee->profile_image ? url($employee->profile_image) : 'assets/img/default.png'
            ];

            $subcontractorEmployees[] = [
                'id' => $employee->id,
                'user_type' => 1,
                'personal_details' => $personalDetails,
                'shifts' => $userShifts,
                'subcontractors' => $associatedData['subcontractors']
            ];
        }

        return $subcontractorEmployees;
    }

    /**
     * Get associated subcontractors and projects for employee
     * Only returns associations where invitation has been accepted for this specific subcontractor
     */
    protected function getEmployeeAssociations($employee, $customerProjectIds, $customerId, $workspaceId)
    {
        // Filter associations that:
        // 1. Have project intersection with customer's projects
        // 2. Have accepted invitations for this employee-subcontractor pair
        $associations = $employee->associations->filter(function ($association) use ($customerProjectIds, $employee, $customerId, $workspaceId) {
            // Check project intersection
            $projectIds = is_array($association->project_ids) ? $association->project_ids : json_decode($association->project_ids, true);
            if (empty(array_intersect($projectIds, $customerProjectIds))) {
                return false;
            }

            // Check if employee has accepted invitation specifically for this subcontractor
            // Get the subcontractor ID from the association
            $subcontractorId = $association->subcontractor_id ?? null;
            
            if (!$subcontractorId) {
                return false;
            }

            // Verify invitation exists for this specific subcontractor
            $hasAcceptedInvitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('subcontractor_id', $subcontractorId)
                ->where('invitation_status', 'accepted')
                ->exists();

            return $hasAcceptedInvitation;
        });

        $subcontractors = [];
        foreach ($associations as $association) {
            if ($association->subcontractor) {
                $subcontractorName = $association->subcontractor->company_name ?: $association->subcontractor->name;
                $subcontractorId = $association->subcontractor->id;

                $projectIds = is_array($association->project_ids) ? $association->project_ids : json_decode($association->project_ids, true);
                $validProjectIds = array_intersect($projectIds, $customerProjectIds);

                if (empty($validProjectIds)) {
                    continue;
                }

                // Use the reusable helper method to get projects with sites
                $projectDetails = $this->getProjectsWithSites($validProjectIds, $customerId, $workspaceId)
                    ->map(function ($project) {
                        return [
                            'id' => $project['id'],
                            'title' => $project['title'],
                            'sites' => collect($project['sites'])->map(function ($site) {
                                return [
                                    'id' => $site['id'],
                                    'title' => $site['title']
                                ];
                            })->toArray()
                        ];
                    });
                
                if ($projectDetails->isNotEmpty()) {
                    $subcontractors[] = [
                        'id' => $subcontractorId,
                        'name' => $subcontractorName,
                        'projects' => $projectDetails->toArray()
                    ];
                }
            }
        }

        return [
            'subcontractors' => $subcontractors
        ];
    }

    /**
     * Get projects with their sites for given project IDs
     * Reusable method to avoid code duplication
     * 
     * @param array $projectIds
     * @param int $customerId
     * @param int $workspaceId
     * @return \Illuminate\Support\Collection
     */
    protected function getProjectsWithSites($projectIds, $customerId, $workspaceId)
    {
        if (empty($projectIds)) {
            return collect([]);
        }

        return Project::whereIn('id', $projectIds)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('is_deleted', '0')
            ->with(['site' => function ($query) {
                $query->whereHas('siteDetails', function ($siteQuery) {
                    $siteQuery->where('active', '1')->where('del', '0');
                });
            }, 'site.siteDetails' => function ($query) {
                $query->where('active', '1')->where('del', '0');
            }])
            ->get()
            ->map(function ($project) {
                $sites = $project->site->map(function ($projectSite) {
                    if ($projectSite->siteDetails) {
                        return [
                            'id' => $projectSite->siteDetails->id,
                            'title' => $projectSite->siteDetails->title,
                            'address' => $projectSite->siteDetails->address ?? null,
                            'state' => $projectSite->siteDetails->state ?? null,
                            'suburb' => $projectSite->siteDetails->suburb ?? null,
                        ];
                    }
                    return null;
                })->filter()->values();

                return [
                    'id' => $project->id,
                    'title' => $project->title,
                    'description' => $project->description ?? null,
                    'address' => $project->address ?? null,
                    'start_date' => $project->start_date ?? null,
                    'end_date' => $project->end_date ?? null,
                    'sites' => $sites
                ];
            });
    }

    /**
     * Get only sites array from projects (flattened, unique sites)
     * Returns array of sites with id and title only
     * 
     * @param array $projectIds
     * @param int $customerId
     * @param int $workspaceId
     * @return \Illuminate\Support\Collection
     */
    protected function getSitesFromProjects($projectIds, $customerId, $workspaceId)
    {
        if (empty($projectIds)) {
            return collect([]);
        }

        $sites = collect([]);

        Project::whereIn('id', $projectIds)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('is_deleted', '0')
            ->with(['site' => function ($query) {
                $query->whereHas('siteDetails', function ($siteQuery) {
                    $siteQuery->where('active', '1')->where('del', '0');
                });
            }, 'site.siteDetails' => function ($query) {
                $query->where('active', '1')->where('del', '0');
            }])
            ->get()
            ->each(function ($project) use (&$sites) {
                $project->site->each(function ($projectSite) use (&$sites) {
                    if ($projectSite->siteDetails) {
                        $sites->push([
                            'id' => $projectSite->siteDetails->id,
                            'title' => $projectSite->siteDetails->title,
                        ]);
                    }
                });
            });

        // Remove duplicates based on site id and return unique sites
        return $sites->unique('id')->values();
    }

    /**
     * Get company name from adminsettings for a single customer/workspace pair
     * 
     * @param int $customerId
     * @param int|null $workspaceId
     * @return string|null
     */
    protected function getCompanyName($customerId, $workspaceId = null)
    {
        if (!$customerId) {
            return null;
        }

        $companyName = null;

        // First try: with workspace filter
        if ($workspaceId) {
            $companyName = DB::table('adminsettings')
                ->where('customer_id', $customerId)
                ->where('workspace', $workspaceId)
                ->where('key', 'company_company_name')
                ->value('value');
        }

        // Fallback: try without workspace filter
        if (empty($companyName)) {
            $companyName = DB::table('adminsettings')
                ->where('customer_id', $customerId)
                ->where('key', 'company_company_name')
                ->value('value');
        }

        // Fallback: try to get from User table
        if (empty($companyName)) {
            $customer = User::find($customerId);
            if ($customer) {
                $companyName = $customer->company_name ?? $customer->name ?? null;
            }
        }

        return $companyName;
    }

    /**
     * Get company names in bulk for multiple customer/workspace pairs
     * Returns an associative array with keys like "customer_id_workspace_id" => "company_name"
     * 
     * @param array $customerWorkspacePairs Array of arrays with 'customer_id' and 'workspace_id' keys
     * @return array Associative array mapping "customer_id_workspace_id" to company name
     */
    protected function getCompanyNamesBulk($customerWorkspacePairs)
    {
        $companyNamesMap = [];

        if (empty($customerWorkspacePairs)) {
            return $companyNamesMap;
        }

        foreach ($customerWorkspacePairs as $key => $pair) {
            $customerId = $pair['customer_id'] ?? null;
            $workspaceId = $pair['workspace_id'] ?? null;

            if (!$customerId) {
                continue;
            }

            $companyName = $this->getCompanyName($customerId, $workspaceId);
            $companyNamesMap[$key] = $companyName;
        }

        return $companyNamesMap;
    }
}
