<?php

namespace App\Http\Controllers\Traits;

use App\Models\ProjectManagement;
use App\Models\ProjectManagementHistory;
use App\Models\ProjectManagementDocument;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

trait ProjectTaskManagementTrait
{
    /**
     * Send email to assigned person for project management task
     */
    protected function sendProjectManagementEmailToAssignee($task, $subjectPrefix = 'Task Assignment')
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        // Collect recipients based on assignment type
        $recipients = [];
        if ($task->assignment_type === 'internal_employee' && $task->assigned_emp_id) {
            $emp = \App\Models\EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->find($task->assigned_emp_id);
            if ($emp && !empty($emp->employee_email)) {
                $nameModel = \App\Models\EmpPersonalDetails::where('emp_id', $task->assigned_emp_id)->first();
                $name = $nameModel ? trim(($nameModel->first_name ?? '') . ' ' . ($nameModel->last_name ?? '')) : null;
                $recipients[] = ['email' => $emp->employee_email, 'name' => $name];
            }
        } elseif ($task->assignment_type === 'subcontractor' && $task->subcontractor_id) {
            $sub = \App\Models\User::find($task->subcontractor_id);
            if ($sub && !empty($sub->email)) {
                $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
            }
        } elseif ($task->assignment_type === 'subcontractor_employee') {
            // Subcontractor
            if ($task->subcontractor_id) {
                $sub = \App\Models\User::find($task->subcontractor_id);
                if ($sub && !empty($sub->email)) {
                    $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
                }
            }
            // Assigned subcontractor employee (using assigned_emp_id)
            if ($task->assigned_emp_id) {
                $emp = \App\Models\EmployeeSubcontractor::find($task->assigned_emp_id);
                if ($emp && !empty($emp->email)) {
                    $name = trim(($emp->first_name ?? '') . ' ' . ($emp->last_name ?? ''));
                    $recipients[] = ['email' => $emp->email, 'name' => $name];
                }
            }
        }
        if (empty($recipients)) {
            return;
        }
        $brandTitle = getBrandTitle($ids['customer_id'] ?? null, $ids['workspace_id'] ?? null);
        $statusLabel = $task->getStatusLabel();
        $subject = $subjectPrefix . ' | ' . $statusLabel . ' | ' . $brandTitle;
        // Load relationships if not already loaded
        if (!$task->relationLoaded('project')) {
            $task->load('project');
        }
        if (!$task->relationLoaded('site')) {
            $task->load('site');
        }
        // Prepare email data
        $emailData = [
            'subject' => $subject,
            'task_title' => $task->title,
            'task_description' => $task->description,
            'task_priority' => $task->priority ?? 'Not Set',
            'task_status_label' => $statusLabel,
            'task_due_date' => $task->due_date ? \Carbon\Carbon::parse($task->due_date)->format('d M Y') : 'Not Set',
            'project_name' => $task->project->title ?? null,
            'site_name' => $task->site->title ?? null,
            'action_url' => env('FRONTEND_URL') ? rtrim(env('FRONTEND_URL')) : null,
            'customer_id' => $ids['customer_id'] ?? null,
            'reject_reason' => $task->reject_reason ?? null,
        ];
        // Send email to each recipient
        foreach ($recipients as $recipient) {
            $emailData['assignee_name'] = $recipient['name'] ?? null;
            $params = [
                'to' => $recipient['email'],
                'subject' => $subject,
                'msg' => view('Emails.project-management-assignee', $emailData + ['customer_id' => $task->customer_id ?? null])->render(),
            ];
            $this->SendInstantEmail($params);
        }
    }

    /**
     * Send status update email to assigned person for project management task
     */
    protected function sendProjectManagementStatusUpdateEmail($task, $oldStatusLabel, $newStatusLabel)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        // Collect recipients based on assignment type
        $recipients = [];
        if ($task->assignment_type === 'internal_employee' && $task->assigned_emp_id) {
            $emp = \App\Models\EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->find($task->assigned_emp_id);
            if ($emp && !empty($emp->employee_email)) {
                $nameModel = \App\Models\EmpPersonalDetails::where('emp_id', $task->assigned_emp_id)->first();
                $name = $nameModel ? trim(($nameModel->first_name ?? '') . ' ' . ($nameModel->last_name ?? '')) : null;
                $recipients[] = ['email' => $emp->employee_email, 'name' => $name];
            }
        } elseif ($task->assignment_type === 'subcontractor' && $task->subcontractor_id) {
            $sub = \App\Models\User::find($task->subcontractor_id);
            if ($sub && !empty($sub->email)) {
                $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
            }
        } elseif ($task->assignment_type === 'subcontractor_employee') {
            // Subcontractor
            if ($task->subcontractor_id) {
                $sub = \App\Models\User::find($task->subcontractor_id);
                if ($sub && !empty($sub->email)) {
                    $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
                }
            }
            // Assigned subcontractor employee (using assigned_emp_id)
            if ($task->assigned_emp_id) {
                $emp = \App\Models\EmployeeSubcontractor::find($task->assigned_emp_id);
                if ($emp && !empty($emp->email)) {
                    $name = trim(($emp->first_name ?? '') . ' ' . ($emp->last_name ?? ''));
                    $recipients[] = ['email' => $emp->email, 'name' => $name];
                }
            }
        }
        if (empty($recipients)) {
            return;
        }
        $brandTitle = getBrandTitle($ids['customer_id'] ?? null, $ids['workspace_id'] ?? null);
        $subject = 'Task Status Updated | ' . $newStatusLabel . ' | ' . $brandTitle;
        // Load relationships if not already loaded
        if (!$task->relationLoaded('project')) {
            $task->load('project');
        }
        if (!$task->relationLoaded('site')) {
            $task->load('site');
        }
        // Prepare email data
        $emailData = [
            'subject' => $subject,
            'task_title' => $task->title,
            'task_description' => $task->description,
            'task_priority' => $task->priority ?? 'Not Set',
            'old_status_label' => $oldStatusLabel,
            'new_status_label' => $newStatusLabel,
            'task_due_date' => $task->due_date ? \Carbon\Carbon::parse($task->due_date)->format('d M Y') : 'Not Set',
            'project_name' => $task->project->title ?? null,
            'site_name' => $task->site->title ?? null,
            'action_url' => env('FRONTEND_URL') ? rtrim(env('FRONTEND_URL')) : null,
            'customer_id' => $ids['customer_id'] ?? null,
            'completion_description' => $task->completion_description ?? null,
            'reject_reason' => $task->reject_reason ?? null,
        ];
        // Send email to each recipient
        foreach ($recipients as $recipient) {
            $emailData['assignee_name'] = $recipient['name'] ?? null;
            $params = [
                'to' => $recipient['email'],
                'subject' => $subject,
                'msg' => view('Emails.project-management-status-update', $emailData + ['customer_id' => $task->customer_id ?? null])->render(),
            ];
            $this->SendInstantEmail($params);
        }
    }

    /**
     * Validation rules for creating/updating project management task
     */
    protected function projectManagementValidation(Request $request, $isUpdate = false)
    {
        $rules = [
            'project_id' => 'required|integer|exists:projects,id',
            'site_id' => 'nullable|integer|exists:sites,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'priority' => 'nullable|in:low,medium,high,critical',
            'status' => 'nullable|integer|in:0,1,2,3,4',
            'assigned_emp_type' => 'required|in:internal_employee,subcontractor,subcontractor_employee',
            'due_date' => 'nullable|date',
            'prerequisite_task' => 'nullable|integer|exists:project_management,id',
            'documents.*' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240',
        ];
        // Conditional rules based on assigned_emp_type
        if ($request->assigned_emp_type === 'internal_employee') {
            $rules['assigned_emp_id'] = 'required|integer|exists:emp_company_details,id';
        } elseif ($request->assigned_emp_type === 'subcontractor') {
            $rules['subcontractor_id'] = 'required|integer|exists:users,id';
        } elseif ($request->assigned_emp_type === 'subcontractor_employee') {
            $rules['subcontractor_id'] = 'required|integer|exists:users,id';
            $rules['assigned_emp_id'] = 'required|integer|exists:employees_subcontractors,id';
        }
        return Validator::make($request->all(), $rules);
    }

    /**
     * Load creatorEntity for a task based on created_by_type
     * This function handles loading the correct creator entity (customer, employee, or subcontractor)
     * and sets it as a relation on the task model with creation timestamp
     * 
     * @param ProjectManagement $task The task model instance
     * @return void
     */
    protected function loadCreatorEntityForTask($task)
    {
        if (!$task->created_by || !$task->created_by_type) {
            return;
        }
        $creator = null;
        if ($task->created_by_type === 'employee' && $task->created_by) {
            $creator = \App\Models\EmpPersonalDetails::where('emp_id', $task->created_by)
                ->select('emp_id', 'first_name', 'middle_name', 'last_name')
                ->first();
            if ($creator) {
                $creator->type = 'employee';
            }
        } elseif ($task->created_by_type === 'subcontractor' && $task->created_by) {
            $creator = \App\Models\User::where('id', $task->created_by)
                ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                ->first();
            if ($creator) {
                $creator->type = 'subcontractor';
            }
        } elseif ($task->created_by_type === 'customer' && $task->created_by) {
            $creator = \App\Models\User::where('id', $task->created_by)
                ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                ->first();
            if ($creator) {
                $creator->type = 'customer';
            }
        }
        if ($creator) {
            // Add creation timestamp
            $creator->created_at = $task->created_at;
        }
        $task->setRelation('creatorEntity', $creator);
    }

    /**
     * Load assignedEntity for a task based on assignment_type
     * This function handles loading the correct entity (internal_employee, subcontractor, or subcontractor_employee)
     * and sets it as a relation on the task model
     * 
     * @param ProjectManagement $task The task model instance
     * @return void
     */
    protected function loadAssignedEntityForTask($task)
    {
        if ($task->assignment_type === 'internal_employee' && $task->assigned_emp_id) {
            $entity = \App\Models\EmpPersonalDetails::where('emp_id', $task->assigned_emp_id)
                ->select('emp_id', 'first_name', 'middle_name', 'last_name')
                ->first();
            if ($entity) {
                $entity->type = 'internal_employee';
            }
            $task->setRelation('assignedEntity', $entity);
        } elseif ($task->assignment_type === 'subcontractor') {
            $subcontractorId = $task->assigned_emp_id ?? $task->subcontractor_id;
            if ($subcontractorId) {
                $entity = \App\Models\User::where('id', $subcontractorId)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                    ->first();
                if ($entity) {
                    $entity->type = 'subcontractor';
                }
                $task->setRelation('assignedEntity', $entity);
            }
        } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
            $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                ->first();
            if ($entity) {
                $entity->type = 'subcontractor_employee';
                // Filter subcontractors to only include the one matching task's subcontractor_id
                if ($task->subcontractor_id) {
                    // Get all subcontractors from the accessor
                    $allSubcontractors = $entity->subcontractors;
                    // Filter to only include the one matching the task's subcontractor_id
                    $filteredSubcontractors = array_filter($allSubcontractors, function ($sub) use ($task) {
                        return isset($sub['id']) && $sub['id'] == $task->subcontractor_id;
                    });
                    // Set the filtered array as a custom attribute (this will override the accessor when serialized)
                    $entity->setAttribute('subcontractors', array_values($filteredSubcontractors));
                }
            }
            $task->setRelation('assignedEntity', $entity);
        }
    }

    /**
     * Load documents with uploaded_by information
     */
    protected function loadDocumentsWithUploader($task)
    {
        if (!$task->relationLoaded('documents') || $task->documents->isEmpty()) {
            return;
        }
        $task->documents->each(function ($document) {
            $uploader = null;

            if ($document->uploaded_by_type === 'internal_employee' && $document->uploaded_by) {
                $uploader = \App\Models\EmpPersonalDetails::where('emp_id', $document->uploaded_by)
                    ->select('emp_id', 'first_name', 'middle_name', 'last_name')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'internal_employee';
                    $uploader->name = trim(($uploader->first_name ?? '') . ' ' . ($uploader->middle_name ?? '') . ' ' . ($uploader->last_name ?? ''));
                }
            } elseif ($document->uploaded_by_type === 'subcontractor' && $document->uploaded_by) {
                $uploader = \App\Models\User::where('id', $document->uploaded_by)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'subcontractor';
                }
            } elseif ($document->uploaded_by_type === 'subcontractor_employee' && $document->uploaded_by) {
                $uploader = \App\Models\EmployeeSubcontractor::where('id', $document->uploaded_by)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'subcontractor_employee';
                    $uploader->name = trim(($uploader->first_name ?? '') . ' ' . ($uploader->middle_name ?? '') . ' ' . ($uploader->last_name ?? ''));
                }
            } elseif ($document->uploaded_by_type === 'customer' && $document->uploaded_by) {
                $uploader = \App\Models\User::where('id', $document->uploaded_by)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'customer';
                }
            }
            $document->setRelation('uploadedBy', $uploader);
        });
    }

    /**
     * Load history with assigned employee information
     */
    protected function loadHistoryWithAssignedEmployee($task)
    {
        if (!$task->relationLoaded('histories') || $task->histories->isEmpty()) {
            return;
        }
        // Sort histories by created_at descending (latest first)
        $task->histories = $task->histories->sortByDesc('created_at')->values();
        $task->histories->each(function ($history) {
            $assignedEmployee = null;

            if (!$history->assigned_emp_id) {
                $history->setRelation('assignedEmployee', null);
                return;
            }

            if ($history->assigned_emp_type === 'internal_employee') {
                $assignedEmployee = \App\Models\EmpPersonalDetails::where('emp_id', $history->assigned_emp_id)
                    ->select('emp_id', 'first_name', 'middle_name', 'last_name')
                    ->first();
                if ($assignedEmployee) {
                    $assignedEmployee->type = 'internal_employee';
                    $assignedEmployee->name = trim(($assignedEmployee->first_name ?? '') . ' ' . ($assignedEmployee->middle_name ?? '') . ' ' . ($assignedEmployee->last_name ?? ''));
                }
            } elseif ($history->assigned_emp_type === 'subcontractor_employee') {
                $assignedEmployee = \App\Models\EmployeeSubcontractor::where('id', $history->assigned_emp_id)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($assignedEmployee) {
                    $assignedEmployee->type = 'subcontractor_employee';
                    $assignedEmployee->name = trim(($assignedEmployee->first_name ?? '') . ' ' . ($assignedEmployee->middle_name ?? '') . ' ' . ($assignedEmployee->last_name ?? ''));
                }
            } elseif ($history->assigned_emp_type === 'subcontractor') {
                // Check if it's a customer or subcontractor
                $user = \App\Models\User::where('id', $history->assigned_emp_id)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name', 'user_type')
                    ->first();
                if ($user) {
                    // Check if it's a customer (user_type = 2)
                    if ($user->user_type == config('constants.user_types.customer')) {
                        $user->type = 'customer';
                    } else {
                        $user->type = 'subcontractor';
                    }
                    $assignedEmployee = $user;
                }
            }
            $history->setRelation('assignedEmployee', $assignedEmployee);
        });
    }

    /**
     * Log project management history
     */
    protected function logProjectManagementHistory($projectManagementId, $description, $ids = null, $task = null, $data = null)
    {

        $historyData = [
            'project_management_id' => $projectManagementId,
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
            'description' => $description,
        ];

        // Always store who made the change (current authenticated user), not who the task is assigned to
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $historyData['assigned_emp_id'] = Auth::id();

        // Determine the type of user who made the change
        if ($ids['flag'] === 'emp') {
            $historyData['assigned_emp_type'] = 'internal_employee';
        } elseif ($userTable === 'customer') {
            if ($user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
                $historyData['assigned_emp_type'] = 'subcontractor';
            } else {
                // Customer user - store as subcontractor type (we'll handle customer loading in loadHistoryWithAssignedEmployee)
                // Since enum doesn't have 'customer', we'll use a type that can be identified
                // We'll check in loadHistoryWithAssignedEmployee if it's actually a customer
                $historyData['assigned_emp_type'] = 'subcontractor'; // Temporary, will be identified in loadHistoryWithAssignedEmployee
            }
        } else {
            $historyData['assigned_emp_type'] = 'internal_employee';
        }

        ProjectManagementHistory::create($historyData);
    }

    /**
     * Store a new project management task
     * Handles both simple tasks and prerequisite tasks
     */
    protected function createProjectManagementTask(Request $request)
    {
        $validator = $this->projectManagementValidation($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $data = $validator->validated();
        $data['customer_id'] = $ids['customer_id'];
        $data['workspace_id'] = $ids['workspace_id'];
        // Map assigned_emp_type to assignment_type for database
        if (isset($data['assigned_emp_type'])) {
            $data['assignment_type'] = $data['assigned_emp_type'];
            unset($data['assigned_emp_type']);
        }
        // When assignment_type is subcontractor, also save subcontractor_id to assigned_emp_id
        if (isset($data['assignment_type']) && $data['assignment_type'] === 'subcontractor' && isset($data['subcontractor_id'])) {
            $data['assigned_emp_id'] = $data['subcontractor_id'];
        }
        // Set default status if not provided
        if (!isset($data['status'])) {
            $data['status'] = ProjectManagement::STATUS_TODO;
        }
        // Fix creator information based on user type
        $user = Auth::user();
        $userTable = $this->getUserTable();
        if ($ids['flag'] === 'emp') {
            $data['created_by'] = Auth::id();
            $data['created_by_type'] = 'employee';
        } elseif ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
            // For subcontractor users
            $data['created_by'] = Auth::id();
            $data['created_by_type'] = 'subcontractor';
        } else {
            // For regular customer users
            $data['created_by'] = Auth::id();
            $data['created_by_type'] = 'customer';
        }
        DB::beginTransaction();
        // Check if this is a prerequisite task scenario
        if (isset($data['prerequisite_task']) && !empty($data['prerequisite_task'])) {
            // Prerequisite task scenario: Link to existing task
            $prerequisiteTaskId = $data['prerequisite_task'];
            // Verify the prerequisite task exists and belongs to same customer/workspace
            $prerequisiteTask = ProjectManagement::where('id', $prerequisiteTaskId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('del', '0')
                ->first();
            if (!$prerequisiteTask) {
                DB::rollBack();
                return $this->error('Prerequisite task not found or you do not have access to it.', 404);
            }
            if ($prerequisiteTask->status == ProjectManagement::STATUS_DONE) {
                $data['status'] = ProjectManagement::STATUS_TODO;
            }
            // Create the new task with prerequisite link
            $task = ProjectManagement::create($data);
            // Handle document uploads
            $this->saveProjectManagementDocuments($request, $task->id, $ids, $user, $userTable);
            // Log history for prerequisite task linking (pass data array to ensure we have assignment info)
            $this->logProjectManagementHistory(
                $task->id,
                'Task created and linked to prerequisite task #' . $prerequisiteTaskId,
                $ids,
                $task,
                $data
            );
            // Send email to assignee
            $this->sendProjectManagementEmailToAssignee($task->load(['project', 'site']), 'New Task Assigned');
            DB::commit();
            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'assignedInternalEmployee',
                'assignedSubcontractor',
                'assignedSubcontractorByEmpId',
                'assignedSubcontractorEmployee',
                'prerequisiteTask',
                'documents'
            ]);
            // Load assignedEntity for response
            $this->loadAssignedEntityForTask($task);
            return $this->success($task, 'Task created successfully with prerequisite link');
        } else {
            // Simple task scenario: Create new task without prerequisite
            $task = ProjectManagement::create($data);
            // Handle document uploads
            $this->saveProjectManagementDocuments($request, $task->id, $ids, $user, $userTable);
            // Log history for task creation (pass data array to ensure we have assignment info)
            $this->logProjectManagementHistory(
                $task->id,
                'Task created',
                $ids,
                $task,
                $data
            );
            // Send email to assignee
            $this->sendProjectManagementEmailToAssignee($task->load(['project', 'site']), 'New Task Assigned');
            DB::commit();
            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'documents'
            ]);
            // Load assignedEntity for response
            $this->loadAssignedEntityForTask($task);
            return $this->success($task, 'Task created successfully');
        }
    }

    /**
     * Get project management tasks with filters
     */
    protected function getProjectManagementTasks(Request $request)
    {
        // Validate date parameters if provided
        if ($request->filled('start_date') || $request->filled('end_date')) {
            $dateRules = [];
            if ($request->filled('start_date')) {
                $dateRules['start_date'] = 'required|date';
            }
            if ($request->filled('end_date')) {
                $dateRules['end_date'] = 'required|date';
                if ($request->filled('start_date')) {
                    $dateRules['end_date'] .= '|after_or_equal:start_date';
                }
            }

            $validator = Validator::make($request->all(), $dateRules, [
                'start_date.date' => 'Start date must be a valid date.',
                'end_date.date' => 'End date must be a valid date.',
                'end_date.after_or_equal' => 'End date must be after or equal to start date.',
            ]);

            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
        }

        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        $query = ProjectManagement::with([
            'prerequisiteTask',
            'project',
            'site',

        ])
            ->where('del', '0');
        // For subcontractors: Filter by their assignments
        if ($isSubcontractor) {
            $query->where(function ($q) use ($user) {
                $q->where(function ($subQ) use ($user) {
                    $subQ->where('assignment_type', 'subcontractor')
                        ->where('subcontractor_id', $user->id);
                })
                ->orWhere(function ($subQ) use ($user) {
                    $subQ->where('assignment_type', 'subcontractor_employee')
                        ->where('subcontractor_id', $user->id);
                });
            });
        } else {
            // For others: Apply customer/workspace filter
            $query->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);
        }
        // Filter by project_id
        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }
        // Filter by site_id
        if ($request->filled('site_id')) {
            $query->where('site_id', $request->site_id);
        }

        // Filter by start_date and end_date
        if ($request->filled('start_date') || $request->filled('end_date')) {
            // Parse dates - handle dd-mm-yyyy format explicitly
            $parseDate = function ($dateString) {
                // Try different date formats
                $formats = ['d-m-Y', 'Y-m-d', 'm-d-Y', 'd/m/Y', 'Y/m/d', 'm/d/Y'];

                foreach ($formats as $format) {
                    try {
                        return \Carbon\Carbon::createFromFormat($format, $dateString);
                    } catch (\Exception $e) {
                        continue;
                    }
        }

                // Fallback to Carbon::parse if all formats fail
                try {
                    return \Carbon\Carbon::parse($dateString);
                } catch (\Exception $e) {
                    throw new \InvalidArgumentException("Invalid date format: {$dateString}");
                }
            };

            if ($request->filled('start_date')) {
                $startDate = $parseDate($request->start_date)->startOfDay();

                if ($request->filled('end_date')) {
                    // Both start_date and end_date provided: filter between dates
                    $endDate = $parseDate($request->end_date)->endOfDay();
                    $query->whereBetween('created_at', [$startDate, $endDate]);
            } else {
                    // Only start_date provided: filter from start_date onward
                    $query->where('created_at', '>=', $startDate);
                }
            } elseif ($request->filled('end_date')) {
                // Only end_date provided: filter up to end_date
                $endDate = $parseDate($request->end_date)->endOfDay();
                $query->where('created_at', '<=', $endDate);
            }
        }

        // Filter by status
        // if ($request->filled('status')) {
        //     $query->where('status', $request->status);
        // }

        // Filter by priority
        // if ($request->filled('priority')) {
        //     $query->where('priority', $request->priority);
        // }

        // Filter by assignment_type
        // if ($request->filled('assignment_type')) {
        //     $query->where('assignment_type', $request->assignment_type);
        // }

        // Filter by assigned_emp_id (for internal employees)
        // if ($request->filled('assigned_emp_id')) {
        //     $query->where('assigned_emp_id', $request->assigned_emp_id);
        // }

        // // Filter by subcontractor_id
        // if ($request->filled('subcontractor_id')) {
        //     $query->where('subcontractor_id', $request->subcontractor_id);
        // }


        // Search by title or description
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                    ->orWhere('description', 'like', '%' . $search . '%');
            });
        }
        // Order by
        $orderBy = $request->get('order_by', 'created_at');
        $orderDir = $request->get('order_dir', 'desc');
        $query->orderBy($orderBy, $orderDir);
        $tasks = $query->get();
        // Load assignedEntity for each task
        $tasks->each(function ($task) {
            $this->loadAssignedEntityForTask($task);
        });
        return $this->withCount($tasks, 'Project management tasks retrieved successfully');
    }

    /**
     * Get all project management tasks by site_id
     * Independent API that only requires site_id
     */
    protected function getProjectManagementTasksBySiteId(Request $request)
    {
        // Validation: site_id is required
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'site_id' => 'required|integer|exists:sites,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));

        $query = ProjectManagement::with([
            'prerequisiteTask',
            'project',
            'site',
        ])
            ->where('del', '0')
            ->where('status', ProjectManagement::STATUS_TODO)
            ->where('site_id', $request->site_id);

        // For subcontractors: Filter by their assignments
        if ($isSubcontractor) {
            $query->where(function ($q) use ($user) {
                $q->where(function ($subQ) use ($user) {
                    $subQ->where('assignment_type', 'subcontractor')
                        ->where('subcontractor_id', $user->id);
                })
                    ->orWhere(function ($subQ) use ($user) {
                        $subQ->where('assignment_type', 'subcontractor_employee')
                            ->where('subcontractor_id', $user->id);
                    });
            });
        } else {
            // For others: Apply customer/workspace filter
            $query->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);
        }

        // Order by
        $orderBy = $request->get('order_by', 'created_at');
        $orderDir = $request->get('order_dir', 'desc');
        $query->orderBy($orderBy, $orderDir);

        $tasks = $query->get();

        // Load assignedEntity for each task
        $tasks->each(function ($task) {
            $this->loadAssignedEntityForTask($task);
        });

        return $this->withCount($tasks, 'Project management tasks retrieved successfully');
    }

    /**
     * Get a single project management task by ID
     * Supports optional filters: document, history
     * By default, does NOT include documents and history (for faster response)
     * Pass document=true to get documents, history=true to get history
     */
    protected function getProjectManagementTaskById(Request $request, $id, $ids = null)
    {
        if (!$ids) {
            $ids = $this->getCustomerAndWorkspaceIds();
        }
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        // Determine what to load based on query parameters (default: false for both)
        $includeDocuments = $request->has('comment') ? filter_var($request->input('comment'), FILTER_VALIDATE_BOOLEAN) : false;
        $includeHistory = $request->has('history') ? filter_var($request->input('history'), FILTER_VALIDATE_BOOLEAN) : false;
        $with = [];
        // Only load basic relationships if we need the full task object
        if (!$includeDocuments && !$includeHistory) {
            $with = ['project', 'site', 'prerequisiteTask'];
        }
        // Only add documents/history if explicitly requested
        if ($includeDocuments) {
            $with[] = 'documents';
        }
        if ($includeHistory) {
            $with[] = 'histories';
        }
        $query = ProjectManagement::with($with)
            ->where('id', $id)
            ->where('del', '0');
        // For subcontractors: Check access
        if ($isSubcontractor) {
            $query->where(function ($q) use ($user) {
                $q->where(function ($subQ) use ($user) {
                    $subQ->where('assignment_type', 'subcontractor')
                        ->where('subcontractor_id', $user->id);
                })
                ->orWhere(function ($subQ) use ($user) {
                    $subQ->where('assignment_type', 'subcontractor_employee')
                        ->where('subcontractor_id', $user->id);
                });
            });
        } else {
            // For others: Apply customer/workspace filter
            $query->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id']);
        }
        $task = $query->first();
        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }
        // If only documents are requested, return only documents array
        if ($includeDocuments && !$includeHistory) {
            $this->loadDocumentsWithUploader($task);
            return $this->success($task->documents, 'Documents retrieved successfully');
        }
        // If only history is requested, return only history array
        if ($includeHistory && !$includeDocuments) {
            $this->loadHistoryWithAssignedEmployee($task);
            return $this->success($task->histories, 'History retrieved successfully');
        }
        // If both are requested, return both arrays
        if ($includeDocuments && $includeHistory) {
            $this->loadDocumentsWithUploader($task);
            $this->loadHistoryWithAssignedEmployee($task);
            return $this->success([
                'documents' => $task->documents,
                'histories' => $task->histories
            ], 'Documents and history retrieved successfully');
        }
        // If neither is requested, return full task object
        $this->loadAssignedEntityForTask($task);
        $task->unsetRelation('documents');
        $task->unsetRelation('histories');
        return $this->success($task, 'Task retrieved successfully');
    }

    /**
     * Update project management task (status, documents, comment)
     */
    protected function updateProjectManagementTaskStatus(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $isSubcontractor = ($ids['flag'] === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        // Only customers (not subcontractors) can update tasks
        if ($isSubcontractor) {
            return $this->error('Subcontractors are not allowed to update tasks.', 403);
        }
        // Find the task
        $query = ProjectManagement::where('id', $request->task_id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id']);
        $task = $query->first();
        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }
        // Validation rules - handle both 'documents' and 'dacoment' (typo in request)
        $rules = [
            'task_id' => 'required|integer|exists:project_management,id',
            'status' => 'nullable|integer|in:0,1,2,3,4',
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        DB::beginTransaction();
        try {
            $updateData = [];
            $historyDescriptions = [];
            // Update status if provided
            if ($request->filled('status')) {
                $oldStatus = $task->status;
                $newStatus = $request->status;
                $updateData['status'] = $newStatus;
                $statusLabels = [
                    ProjectManagement::STATUS_BLOCKED => 'Blocked',
                    ProjectManagement::STATUS_TODO => 'Todo',
                    ProjectManagement::STATUS_IN_PROGRESS => 'In Progress',
                    ProjectManagement::STATUS_REVIEW => 'Review',
                    ProjectManagement::STATUS_DONE => 'Done',
                ];
                $oldStatusLabel = $statusLabels[$oldStatus] ?? 'Unknown';
                $newStatusLabel = $statusLabels[$newStatus] ?? 'Unknown';
                $historyDescriptions[] = "Status changed from {$oldStatusLabel} to {$newStatusLabel}";
                // If status is set to "Done", set completed_at
                if ($newStatus == ProjectManagement::STATUS_DONE) {
                    $updateData['completed_at'] = now();
                } elseif ($oldStatus == ProjectManagement::STATUS_DONE && $newStatus != ProjectManagement::STATUS_DONE) {
                    // If status was "Done" and is being changed to something else, clear completed_at
                    $updateData['completed_at'] = null;
                }
                if ($newStatus == ProjectManagement::STATUS_DONE || $newStatus == 4) {
                    // Find all child tasks that depend on this parent task
                    $dependentTasks = ProjectManagement::where('prerequisite_task', $task->id)
                        ->where('customer_id', $ids['customer_id'])
                        ->where('workspace_id', $ids['workspace_id'])
                        ->where('del', '0')
                        ->where('status', '!=', ProjectManagement::STATUS_DONE)
                        ->get();
                    foreach ($dependentTasks as $dependentTask) {
                        // Update child task to Todo status (1) - explicitly set to 1
                        $dependentTask->update([
                            'status' => ProjectManagement::STATUS_TODO
                        ]);
                        // Log history for dependent task
                        $this->logProjectManagementHistory(
                            $dependentTask->id,
                            "Prerequisite task #{$task->id} ({$task->title}) completed. Status updated to Todo.",
                            $ids,
                            $dependentTask,
                            ['assignment_type' => $dependentTask->assignment_type, 'assigned_emp_id' => $dependentTask->assigned_emp_id, 'subcontractor_id' => $dependentTask->subcontractor_id]
                        );
                    }
                }
            }
            // Update the task
            if (!empty($updateData)) {
                $task->update($updateData);
            }
            // Send status update email if status was changed
            if ($request->filled('status') && isset($oldStatus) && isset($newStatus) && $oldStatus != $newStatus) {
                // Reload task with relationships before sending email
                $task->load(['project', 'site']);
                $this->sendProjectManagementStatusUpdateEmail($task, $oldStatusLabel, $newStatusLabel);
            }
            DB::commit();
            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'prerequisiteTask',
                'histories',
            ]);
            // Load assignedEntity for response
            $this->loadAssignedEntityForTask($task);
            return $this->success($task, 'Task updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating project management task: ' . $e->getMessage());
            return $this->error('Failed to update task: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Update project management task (general update - can update any field)
     */
    protected function updateProjectManagementTask(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($ids['flag'] === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        // Only customers (not subcontractors) can update tasks
        if ($isSubcontractor) {
            return $this->error('Subcontractors are not allowed to update tasks.', 403);
        }
        // Validation rules for update (all fields optional except task_id)
        $rules = [
            'task_id' => 'required|integer|exists:project_management,id',
            'project_id' => 'nullable|integer|exists:projects,id',
            'site_id' => 'nullable|integer|exists:sites,id',
            'title' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'priority' => 'nullable|in:low,medium,high,critical',
            'status' => 'nullable|integer|in:0,1,2,3,4',
            'assigned_emp_type' => 'nullable|in:internal_employee,subcontractor,subcontractor_employee',
            'due_date' => 'nullable|date',
            'prerequisite_task' => 'nullable|integer|exists:project_management,id',
            'completion_description' => 'nullable|string',
            'reject_reason' => 'nullable|string',
            'documents.*' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240',
            'dacoment.*' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240', // Handle typo
            'comment' => 'nullable|string',
        ];
        // Conditional rules based on assigned_emp_type (only if assigned_emp_type is provided)
        if ($request->filled('assigned_emp_type')) {
            if ($request->assigned_emp_type === 'internal_employee') {
                $rules['assigned_emp_id'] = 'required|integer|exists:emp_company_details,id';
            } elseif ($request->assigned_emp_type === 'subcontractor') {
                $rules['subcontractor_id'] = 'required|integer|exists:users,id';
            } elseif ($request->assigned_emp_type === 'subcontractor_employee') {
                $rules['subcontractor_id'] = 'required|integer|exists:users,id';
                $rules['assigned_emp_id'] = 'required|integer|exists:employees_subcontractors,id';
            }
        }
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Find the task
        $query = ProjectManagement::where('id', $request->task_id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id']);
        $task = $query->first();
        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }
        DB::beginTransaction();
        try {
            $updateData = [];
            $historyDescriptions = [];
            $oldStatus = $task->status;
            $oldAssignmentType = $task->assignment_type;
            $oldAssignedEmpId = $task->assigned_emp_id;
            $oldSubcontractorId = $task->subcontractor_id;
            // Update basic fields
            if ($request->filled('project_id')) {
                $updateData['project_id'] = $request->project_id;
                $historyDescriptions[] = "Project updated";
            }
            if ($request->filled('site_id')) {
                $updateData['site_id'] = $request->site_id;
                $historyDescriptions[] = "Site updated";
            }
            if ($request->filled('title')) {
                $updateData['title'] = $request->title;
                $historyDescriptions[] = "Title updated";
            }
            if ($request->filled('description')) {
                $updateData['description'] = $request->description;
                $historyDescriptions[] = "Description updated";
            }
            if ($request->filled('priority')) {
                $updateData['priority'] = $request->priority;
                $historyDescriptions[] = "Priority updated to " . ucfirst($request->priority);
            }
            if ($request->filled('due_date')) {
                $updateData['due_date'] = $request->due_date;
                $historyDescriptions[] = "Due date updated";
            }
            if ($request->filled('prerequisite_task')) {
                $updateData['prerequisite_task'] = $request->prerequisite_task;
                $historyDescriptions[] = "Prerequisite task updated";
            }
            if ($request->filled('completion_description')) {
                $updateData['completion_description'] = $request->completion_description;
                $historyDescriptions[] = "Completion description updated";
            }
            if ($request->filled('reject_reason')) {
                $updateData['reject_reason'] = $request->reject_reason;
                $historyDescriptions[] = "Reject reason updated";
            }
            // Handle status update
            if ($request->filled('status')) {
                $newStatus = $request->status;
                $updateData['status'] = $newStatus;
                $statusLabels = [
                    ProjectManagement::STATUS_BLOCKED => 'Blocked',
                    ProjectManagement::STATUS_TODO => 'Todo',
                    ProjectManagement::STATUS_IN_PROGRESS => 'In Progress',
                    ProjectManagement::STATUS_REVIEW => 'Review',
                    ProjectManagement::STATUS_DONE => 'Done',
                ];
                $oldStatusLabel = $statusLabels[$oldStatus] ?? 'Unknown';
                $newStatusLabel = $statusLabels[$newStatus] ?? 'Unknown';
                $historyDescriptions[] = "Status changed from {$oldStatusLabel} to {$newStatusLabel}";
                // If status is set to "Done", set completed_at
                if ($newStatus == ProjectManagement::STATUS_DONE) {
                    $updateData['completed_at'] = now();
                } elseif ($oldStatus == ProjectManagement::STATUS_DONE && $newStatus != ProjectManagement::STATUS_DONE) {
                    // If status was "Done" and is being changed to something else, clear completed_at
                    $updateData['completed_at'] = null;
                }
                // If status is set to "Done", update dependent tasks
                if ($newStatus == ProjectManagement::STATUS_DONE || $newStatus == 4) {
                    $dependentTasks = ProjectManagement::where('prerequisite_task', $task->id)
                        ->where('customer_id', $ids['customer_id'])
                        ->where('workspace_id', $ids['workspace_id'])
                        ->where('del', '0')
                        ->where('status', '!=', ProjectManagement::STATUS_DONE)
                        ->get();
                    foreach ($dependentTasks as $dependentTask) {
                        $dependentTask->update(['status' => ProjectManagement::STATUS_TODO]);
                        $this->logProjectManagementHistory(
                            $dependentTask->id,
                            "Prerequisite task #{$task->id} ({$task->title}) completed. Status updated to Todo.",
                            $ids,
                            $dependentTask,
                            ['assignment_type' => $dependentTask->assignment_type, 'assigned_emp_id' => $dependentTask->assigned_emp_id, 'subcontractor_id' => $dependentTask->subcontractor_id]
                        );
                    }
                }
            }
            // Handle assignment changes
            if ($request->filled('assigned_emp_type')) {
                $updateData['assignment_type'] = $request->assigned_emp_type;
                if ($request->assigned_emp_type === 'internal_employee' && $request->filled('assigned_emp_id')) {
                    $updateData['assigned_emp_id'] = $request->assigned_emp_id;
                    $updateData['subcontractor_id'] = null;
                    if ($oldAssignmentType != $request->assigned_emp_type || $oldAssignedEmpId != $request->assigned_emp_id) {
                        $historyDescriptions[] = "Assignment changed to internal employee";
                    }
                } elseif ($request->assigned_emp_type === 'subcontractor' && $request->filled('subcontractor_id')) {
                    $updateData['subcontractor_id'] = $request->subcontractor_id;
                    $updateData['assigned_emp_id'] = $request->subcontractor_id; // Save subcontractor_id to assigned_emp_id
                    if ($oldAssignmentType != $request->assigned_emp_type || $oldSubcontractorId != $request->subcontractor_id) {
                        $historyDescriptions[] = "Assignment changed to subcontractor";
                    }
                } elseif ($request->assigned_emp_type === 'subcontractor_employee') {
                    if ($request->filled('subcontractor_id')) {
                        $updateData['subcontractor_id'] = $request->subcontractor_id;
                    }
                    if ($request->filled('assigned_emp_id')) {
                        $updateData['assigned_emp_id'] = $request->assigned_emp_id;
                    }
                    if ($oldAssignmentType != $request->assigned_emp_type || $oldAssignedEmpId != $request->assigned_emp_id || $oldSubcontractorId != $request->subcontractor_id) {
                        $historyDescriptions[] = "Assignment changed to subcontractor employee";
                    }
                }
            }
            // Update the task
            if (!empty($updateData)) {
                $task->update($updateData);
            }
            // Handle document uploads
            if ($request->hasFile('documents') || $request->hasFile('dacoment')) {
                $this->saveProjectManagementDocuments($request, $task->id, $ids, $user, $userTable);
                $historyDescriptions[] = "Documents uploaded";
            }
            // Add comment to history if provided
            if ($request->filled('comment')) {
                $historyDescriptions[] = "Comment added: " . $request->comment;
            }
            // Send email notifications
            $task->refresh(); // Refresh to get updated values
            // Send status update email if status changed
            if ($request->filled('status') && $oldStatus != $request->status) {
                $statusLabels = [
                    ProjectManagement::STATUS_BLOCKED => 'Blocked',
                    ProjectManagement::STATUS_TODO => 'Todo',
                    ProjectManagement::STATUS_IN_PROGRESS => 'In Progress',
                    ProjectManagement::STATUS_REVIEW => 'Review',
                    ProjectManagement::STATUS_DONE => 'Done',
                ];
                $oldStatusLabel = $statusLabels[$oldStatus] ?? 'Unknown';
                $newStatusLabel = $statusLabels[$request->status] ?? 'Unknown';
                $task->load(['project', 'site']);
                $this->sendProjectManagementStatusUpdateEmail($task, $oldStatusLabel, $newStatusLabel);
            }
            // Send assignment email if assignment changed
            if (
                $request->filled('assigned_emp_type') &&
                ($oldAssignmentType != $request->assigned_emp_type ||
                    $oldAssignedEmpId != $task->assigned_emp_id ||
                    $oldSubcontractorId != $task->subcontractor_id)
            ) {
                $task->load(['project', 'site']);
                $this->sendProjectManagementEmailToAssignee($task, 'Task Reassigned');
            }
            // Log history
            if (!empty($historyDescriptions)) {
                $description = implode('. ', $historyDescriptions);
                $this->logProjectManagementHistory(
                    $task->id,
                    $description,
                    $ids,
                    $task,
                    ['assignment_type' => $task->assignment_type, 'assigned_emp_id' => $task->assigned_emp_id, 'subcontractor_id' => $task->subcontractor_id]
                );
            }
            DB::commit();
            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'prerequisiteTask',
                'histories',
                'documents'
            ]);
            // Load assignedEntity for response
            $this->loadAssignedEntityForTask($task);
            return $this->success($task, 'Task updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating project management task: ' . $e->getMessage());
            return $this->error('Failed to update task: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Upload documents and add comment to project management task
     */
    protected function uploadProjectManagementDocuments(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($ids['flag'] === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor'));
        // Only customers (not subcontractors) can upload documents
        if ($isSubcontractor) {
            return $this->error('Subcontractors are not allowed to upload documents.', 403);
        }
        // Validation rules - single document only
        $rules = [
            'task_id' => 'required|integer|exists:project_management,id',
            'document' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240',
            'comment' => 'nullable|string',
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Find the task
        $query = ProjectManagement::where('id', $request->task_id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id']);
        $task = $query->first();
        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }
        // Check if document or comment are provided
        $hasDocument = $request->hasFile('document');
        $hasComment = $request->filled('comment');
        if (!$hasDocument && !$hasComment) {
            return $this->error('Please provide at least one document or a comment.', 400);
        }
        DB::beginTransaction();
        try {
            $historyDescriptions = [];

            // Determine uploaded_by_type
            $uploadedByType = 'customer';
            if ($ids['flag'] === 'emp') {
                $uploadedByType = 'internal_employee';
            } elseif ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
                $uploadedByType = 'subcontractor';
            }

            // Upload single document if provided
            if ($hasDocument) {
                $document = $request->file('document');
                if ($document && $document->isValid()) {
                    // Create a temporary request with the document
                    $tempRequest = new Request();
                    $tempRequest->files->set('documents', $document);
                    $uploadResult = $this->handleFileImageUpload($tempRequest, 'ProjectManagementDocuments');
                    $documentPath = null;
                    if (is_array($uploadResult)) {
                        if (isset($uploadResult['path'])) {
                            $documentPath = $uploadResult['path'];
                        } elseif (!empty($uploadResult) && isset($uploadResult[0]['path'])) {
                            $documentPath = $uploadResult[0]['path'];
                        }
                    }
                    if ($documentPath) {
                        ProjectManagementDocument::create([
                            'project_management_id' => $task->id,
                            'document_name' => $document->getClientOriginalName(),
                            'document_path' => $documentPath,
                            'comment' => $request->input('comment', null),
                            'uploaded_by' => Auth::id(),
                            'uploaded_by_type' => $uploadedByType,
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);
                        $historyDescriptions[] = "Document uploaded";
                    }
                }
            }

            // Store comment if provided (always store, regardless of document)
            if ($hasComment) {
                // If document was uploaded, comment is already saved with the document
                // But if no document, create a separate record for the comment
                if (!$hasDocument) {
                    ProjectManagementDocument::create([
                        'project_management_id' => $task->id,
                        'document_name' => null,
                        'document_path' => null,
                        'comment' => $request->comment,
                        'uploaded_by' => Auth::id(),
                        'uploaded_by_type' => $uploadedByType,
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                }
                $historyDescriptions[] = "Comment added: " . $request->comment;
            }
            // Log history
            if (!empty($historyDescriptions)) {
                $description = implode('. ', $historyDescriptions);
                $this->logProjectManagementHistory(
                    $task->id,
                    $description,
                    $ids,
                    $task,
                    ['assignment_type' => $task->assignment_type, 'assigned_emp_id' => $task->assigned_emp_id, 'subcontractor_id' => $task->subcontractor_id]
                );
            }
            DB::commit();
            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'prerequisiteTask',
                'histories',
                'documents'
            ]);
            // Load assignedEntity for response
            $this->loadAssignedEntityForTask($task);
            return $this->success($task, 'Document and comment uploaded successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error uploading project management documents: ' . $e->getMessage());
            return $this->error('Failed to upload documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Save project management documents
     */
    protected function saveProjectManagementDocuments($request, $taskId, $ids, $user, $userTable)
    {
        // Handle both 'documents' and 'dacoment' (typo in request)
        $files = null;
        if ($request->hasFile('documents')) {
            $files = $request->file('documents');
        } elseif ($request->hasFile('dacoment')) {
            $files = $request->file('dacoment');
        }
        if (!$files) {
            return;
        }
        // Normalize to array if single file
        if (!is_array($files)) {
            $files = [$files];
        }
        // Determine uploaded_by_type
        $uploadedByType = 'customer';
        if ($ids['flag'] === 'emp') {
            $uploadedByType = 'internal_employee';
        } elseif ($userTable === 'customer' && $user instanceof \App\Models\User && $user->user_type == config('constants.user_types.subcontractor')) {
            $uploadedByType = 'subcontractor';
        }
        $uploadedBy = Auth::id();
        $comment = $request->input('comment', null);
        // Handle each document
        foreach ($files as $document) {
            if (!$document->isValid()) {
                continue;
            }
            // Create a temporary request with only this document
            $tempRequest = new Request();
            $tempRequest->files->set('documents', $document);
            $uploadResult = $this->handleFileImageUpload($tempRequest, 'ProjectManagementDocuments');
            // handleFileImageUpload returns single array if one file, or array of arrays if multiple
            $documentPath = null;
            if (is_array($uploadResult)) {
                // Check if it's a single result (has 'path' key directly)
                if (isset($uploadResult['path'])) {
                    $documentPath = $uploadResult['path'];
                } 
                // Check if it's an array of results (first element has 'path')
                elseif (!empty($uploadResult) && isset($uploadResult[0]['path'])) {
                    $documentPath = $uploadResult[0]['path'];
                }
            }
            if (!$documentPath) {
                continue;
            }
            // Save document to database with comment
            ProjectManagementDocument::create([
                'project_management_id' => $taskId,
                'document_name' => $document->getClientOriginalName(),
                'document_path' => $documentPath,
                'comment' => $comment, // Use the comment from request
                'uploaded_by' => $uploadedBy,
                'uploaded_by_type' => $uploadedByType,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ]);
        }
    }

    /**
     * Update project management document (comment or document or both)
     */
    protected function updateProjectManagementDocument(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        // Validation rules
        $rules = [
            'document_id' => 'required|integer|exists:project_management_dacuments,id',
            'document' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240',
            'comment' => 'nullable|string',
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Find the document
        $document = ProjectManagementDocument::where('id', $request->document_id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();
        if (!$document) {
            return $this->error('Document not found or you do not have access to it.', 404);
        }
        // Check if at least one field is being updated
        $hasNewDocument = $request->hasFile('document');
        $hasNewComment = $request->filled('comment');
        if (!$hasNewDocument && !$hasNewComment) {
            return $this->error('Please provide at least a new document or a comment to update.', 400);
        }
        DB::beginTransaction();
        try {
            $historyDescriptions = [];
            $oldDocumentPath = $document->document_path;
            $oldComment = $document->comment;
            // Update document if provided
            if ($hasNewDocument) {
                $file = $request->file('document');
                if ($file && $file->isValid()) {
                    // Delete old file if it exists
                    if ($oldDocumentPath && file_exists(public_path($oldDocumentPath))) {
                        unlink(public_path($oldDocumentPath));
                    }
                    // Upload new document
                    $tempRequest = new Request();
                    $tempRequest->files->set('documents', $file);
                    $uploadResult = $this->handleFileImageUpload($tempRequest, 'ProjectManagementDocuments');
                    $documentPath = null;
                    if (is_array($uploadResult)) {
                        if (isset($uploadResult['path'])) {
                            $documentPath = $uploadResult['path'];
                        } elseif (!empty($uploadResult) && isset($uploadResult[0]['path'])) {
                            $documentPath = $uploadResult[0]['path'];
                        }
                    }
                    if ($documentPath) {
                        $document->document_name = $file->getClientOriginalName();
                        $document->document_path = $documentPath;
                        $historyDescriptions[] = "Document updated";
                    }
                }
            }
            // Update comment if provided
            if ($hasNewComment) {
                $document->comment = $request->comment;
                if ($oldComment != $request->comment) {
                    $historyDescriptions[] = "Comment updated: " . $request->comment;
        }
            }
            // Save the document
            $document->save();
            // Log history
            if (!empty($historyDescriptions)) {
                $description = implode('. ', $historyDescriptions);
                $this->logProjectManagementHistory(
                    $document->project_management_id,
                    $description,
                    $ids,
                    null,
                    ['document_id' => $document->id]
                );
            }
            DB::commit();
            // Reload task with all relationships
            $task = ProjectManagement::where('id', $document->project_management_id)
                ->where('del', '0')
                ->with([
                    'project',
                    'site',
                    'prerequisiteTask',
                    'histories',
                    'documents'
                ])
                ->first();
            if ($task) {
                // Load assignedEntity for response
                $this->loadAssignedEntityForTask($task);
            }
            return $this->success($task ?? $document, 'Document updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating project management document: ' . $e->getMessage());
            return $this->error('Failed to update document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Delete project management document
     */
    protected function deleteProjectManagementDocument($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();


        // Find the document
        $document = ProjectManagementDocument::where('id', $id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();

        if (!$document) {
            return $this->error('Document not found or you do not have access to it.', 404);
        }

        DB::beginTransaction();
        try {
            $taskId = $document->project_management_id;
            $documentPath = $document->document_path;
            $hadDocument = !empty($document->document_path);
            $hadComment = !empty($document->comment);

            // Delete the file if it exists
            if ($documentPath && file_exists(public_path($documentPath))) {
                unlink(public_path($documentPath));
            }

            // Soft delete the document record
            $document->del = '1';
            $document->save();

            // Log history
            $historyDescriptions = [];
            if ($hadDocument && $hadComment) {
                $historyDescriptions[] = "Document and comment deleted";
            } elseif ($hadDocument) {
                $historyDescriptions[] = "Document deleted";
            } elseif ($hadComment) {
                $historyDescriptions[] = "Comment deleted";
            }

            if (!empty($historyDescriptions)) {
                $description = implode('. ', $historyDescriptions);
                $this->logProjectManagementHistory(
                    $taskId,
                    $description,
                    $ids,
                    null,
                    ['document_id' => $document->id]
                );
            }

            DB::commit();

            // Reload task with all relationships
            $task = ProjectManagement::where('id', $taskId)
                ->where('del', '0')
                ->with([
                    'project',
                    'site',
                    'prerequisiteTask',
                    'histories',
                    'documents'
                ])
                ->first();

            if ($task) {
                // Load assignedEntity for response
                $this->loadAssignedEntityForTask($task);
            }

            return $this->success($task ?? ['message' => 'Document deleted successfully'], 'Document deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting project management document: ' . $e->getMessage());
            return $this->error('Failed to delete document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Delete project management task
     * Rules:
     * - Cannot delete if task is IN_PROGRESS, REVIEW, or DONE
     * - If task has prerequisite_task and task is in TODO, parent can be deleted
     * - If task is deleted and it was a prerequisite for blocked tasks, update those to TODO
     */
    protected function deleteProjectManagementTask($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        
        // Find the task
        $task = ProjectManagement::where('id', $id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();

        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }

        // Check if task can be deleted (cannot delete if IN_PROGRESS, REVIEW, or DONE)
        $nonDeletableStatuses = [
            ProjectManagement::STATUS_IN_PROGRESS,
            ProjectManagement::STATUS_REVIEW,
            ProjectManagement::STATUS_DONE
        ];

        if (in_array($task->status, $nonDeletableStatuses)) {
            $statusLabels = [
                ProjectManagement::STATUS_IN_PROGRESS => 'In Progress',
                ProjectManagement::STATUS_REVIEW => 'Review',
                ProjectManagement::STATUS_DONE => 'Done',
            ];
            $statusLabel = $statusLabels[$task->status] ?? 'Unknown';
            return $this->error("Cannot delete task with status '{$statusLabel}'. Only tasks with status 'Blocked' or 'Todo' can be deleted.", 400);
        }

        DB::beginTransaction();
        try {
            // Find all tasks that have this task as their prerequisite_task (blocked tasks)
            $dependentTasks = ProjectManagement::where('prerequisite_task', $task->id)
                ->where('del', '0')
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->get();

            // Update blocked tasks to TODO status
            foreach ($dependentTasks as $dependentTask) {
                if ($dependentTask->status == ProjectManagement::STATUS_BLOCKED) {
                    $dependentTask->status = ProjectManagement::STATUS_TODO;
                    $dependentTask->save();

                    // Log history for dependent task
                    $this->logProjectManagementHistory(
                        $dependentTask->id,
                        "Prerequisite task #{$task->id} ({$task->title}) deleted. Status updated to Todo.",
                        $ids,
                        $dependentTask,
                        ['assignment_type' => $dependentTask->assignment_type, 'assigned_emp_id' => $dependentTask->assigned_emp_id, 'subcontractor_id' => $dependentTask->subcontractor_id]
                    );
                }
            }

            // Soft delete the task
            $task->del = '1';
            $task->save();

            // Log history for deleted task
            $this->logProjectManagementHistory(
                $task->id,
                "Task deleted",
                $ids,
                $task,
                ['assignment_type' => $task->assignment_type, 'assigned_emp_id' => $task->assigned_emp_id, 'subcontractor_id' => $task->subcontractor_id]
            );

            DB::commit();

            return $this->success(['message' => 'Task deleted successfully'], 'Task deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting project management task: ' . $e->getMessage());
            return $this->error('Failed to delete task: ' . $e->getMessage(), 500);
        }
    }
}
