<?php

namespace App\Http\Controllers\Traits;

use App\Models\ProjectManagement;
use App\Models\ProjectManagementHistory;
use App\Models\ProjectManagementDocument;
use App\Models\EmployeeSubcontractor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

trait SubcontractorProjectTaskManagementTrait
{
    /**
     * Validation rules for creating/updating project management task (subcontractor specific)
     */
    protected function subcontractorProjectTaskValidation(Request $request, $isUpdate = false)
    {
        $rules = [
            'site_id' => 'nullable|integer|exists:sites,id',
            'description' => 'nullable|string',
            'priority' => 'nullable|in:low,medium,high,critical',
            'status' => 'nullable|integer|in:0,1,2,3,4',
            'due_date' => 'nullable|date',
            'prerequisite_task' => 'nullable|integer|exists:project_management,id',
            'document' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240',
        ];

        // For subcontractors, they can only assign to their employees or themselves
        if (!$isUpdate) {
            // On create, these fields are required
            $rules['project_id'] = 'required|integer|exists:projects,id';
            $rules['title'] = 'required|string|max:255';
            $rules['assigned_emp_type'] = 'required|in:subcontractor,subcontractor_employee';
            
            if ($request->assigned_emp_type === 'subcontractor_employee') {
                $rules['assigned_emp_id'] = 'required|integer|exists:employees_subcontractors,id';
            }
        } else {
            // For updates, title and project_id are optional
            $rules['project_id'] = 'nullable|integer|exists:projects,id';
            $rules['title'] = 'nullable|string|max:255';
            // For updates, allow updating status, completion_description, etc.
            $rules['completion_description'] = 'nullable|string';
            $rules['reject_reason'] = 'nullable|string';
            // Allow updating assignment type and assigned employee
            $rules['assigned_emp_type'] = 'nullable|in:subcontractor,subcontractor_employee';
            $rules['assigned_emp_id'] = 'nullable|integer';
            $rules['subcontractor_id'] = 'nullable|integer';
        }

        $validator = Validator::make($request->all(), $rules);
        
        // Add custom validation for update: if assigned_emp_type is subcontractor_employee, assigned_emp_id is required
        if ($isUpdate) {
            $validator->after(function ($validator) use ($request) {
                if ($request->has('assigned_emp_type') && $request->assigned_emp_type === 'subcontractor_employee') {
                    if (!$request->has('assigned_emp_id') || $request->assigned_emp_id === null) {
                        $validator->errors()->add('assigned_emp_id', 'The assigned employee ID is required when assignment type is subcontractor_employee.');
                    }
                }
            });
        }

        return $validator;
    }

    protected function getSubcontractorProjectTasks(Request $request)
    {
        $user = Auth::user();
        if ($request->filled('start_date') || $request->filled('end_date')) {
            $dateRules = [];
            if ($request->filled('start_date')) {
                $dateRules['start_date'] = 'required|date';
            }
            if ($request->filled('end_date')) {
                $dateRules['end_date'] = 'required|date';
                if ($request->filled('start_date')) {
                    $dateRules['end_date'] .= '|after_or_equal:start_date';
                }
            }
            $validator = Validator::make($request->all(), $dateRules, [
                'start_date.date' => 'Start date must be a valid date.',
                'end_date.date' => 'End date must be a valid date.',
                'end_date.after_or_equal' => 'End date must be after or equal to start date.',
            ]);
            
            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }
        }
        $subcontractorId = $user->id;

        // Get all employee IDs for this subcontractor
        $employeeIds = EmployeeSubcontractor::whereHas('associations', function ($q) use ($subcontractorId) {
            $q->where('subcontractor_id', $subcontractorId)
              ->where('active', '1');
        })->pluck('id')->toArray();

        $query = ProjectManagement::with([
            'project',
            'site',
            'prerequisiteTask',
        ])
            ->where('del', '0')
            ->where(function ($q) use ($subcontractorId, $employeeIds) {
                // Tasks assigned to subcontractor directly
                $q->where(function ($subQ) use ($subcontractorId) {
                    $subQ->where('assignment_type', 'subcontractor')
                        ->where(function ($sq) use ($subcontractorId) {
                            $sq->where('subcontractor_id', $subcontractorId)
                               ->orWhere('assigned_emp_id', $subcontractorId);
                        });
                })
                // Tasks assigned to subcontractor employees
                ->orWhere(function ($subQ) use ($subcontractorId, $employeeIds) {
                    $subQ->where('assignment_type', 'subcontractor_employee')
                        ->where('subcontractor_id', $subcontractorId)
                        ->whereIn('assigned_emp_id', $employeeIds);
                });
            });

        // Filter by project_id (for PROJECT dropdown)
        if ($request->filled('project_id') && $request->project_id != 'all' && $request->project_id != '') {
            $query->where('project_id', $request->project_id);
        }

        // Filter by site_id (for SITE LOCATION dropdown)
        if ($request->filled('site_id') && $request->site_id != 'all' && $request->site_id != '') {
            $query->where('site_id', $request->site_id);
        }

        if ($request->filled('start_date') || $request->filled('end_date')) {
            // Parse dates - handle dd-mm-yyyy format explicitly
            $parseDate = function($dateString) {
                // Try different date formats
                $formats = ['d-m-Y', 'Y-m-d', 'm-d-Y', 'd/m/Y', 'Y/m/d', 'm/d/Y'];
                
                foreach ($formats as $format) {
                    try {
                        return \Carbon\Carbon::createFromFormat($format, $dateString);
                    } catch (\Exception $e) {
                        continue;
                    }
                }
                
                // Fallback to Carbon::parse if all formats fail
                try {
                    return \Carbon\Carbon::parse($dateString);
                } catch (\Exception $e) {
                    throw new \InvalidArgumentException("Invalid date format: {$dateString}");
                }
            };
            
            if ($request->filled('start_date')) {
                $startDate = $parseDate($request->start_date)->startOfDay();
                
                if ($request->filled('end_date')) {
                    // Both start_date and end_date provided: filter between dates
                    $endDate = $parseDate($request->end_date)->endOfDay();
                    $query->whereBetween('created_at', [$startDate, $endDate]);
                } else {
                    // Only start_date provided: filter from start_date onward
                    $query->where('created_at', '>=', $startDate);
                }
            } elseif ($request->filled('end_date')) {
                // Only end_date provided: filter up to end_date
                $endDate = $parseDate($request->end_date)->endOfDay();
                $query->where('created_at', '<=', $endDate);
            }
        }

        // Search by title or description (for SEARCH TASKS input)
        if ($request->filled('search') && trim($request->search) != '') {
            $search = trim($request->search);
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                    ->orWhere('description', 'like', '%' . $search . '%');
            });
        }

        // Order by
        $orderBy = $request->get('order_by', 'created_at');
        $orderDir = $request->get('order_dir', 'desc');
        $query->orderBy($orderBy, $orderDir);

        $tasks = $query->get();

        // Make customer_id and workspace_id visible temporarily for processing
        $tasks->makeVisible(['customer_id', 'workspace_id']);

        // Collect all unique customer_id and workspace_id pairs from the tasks
        $customerWorkspacePairs = [];
        foreach ($tasks as $task) {
            $customerId = $task->customer_id;
            $workspaceId = $task->workspace_id;
            if ($customerId && $workspaceId) {
                $key = $customerId . '_' . $workspaceId;
                if (!isset($customerWorkspacePairs[$key])) {
                    $customerWorkspacePairs[$key] = [
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId
                    ];
                }
            }
        }

        // Fetch company_name from adminsettings for all customer/workspace pairs in bulk
        $companyNamesMap = $this->getCompanyNamesBulk($customerWorkspacePairs);

        // Manually load assignedEntity for each task and add company name
        $tasks->each(function ($task) use ($companyNamesMap) {
            if ($task->assignment_type === 'subcontractor') {
                $subcontractorId = $task->assigned_emp_id ?? $task->subcontractor_id;
                if ($subcontractorId) {
                    $entity = \App\Models\User::where('id', $subcontractorId)
                        ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor';
                    }
                    $task->setRelation('assignedEntity', $entity);
                }
            } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
                $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($entity) {
                    $entity->type = 'subcontractor_employee';
                }
                $task->setRelation('assignedEntity', $entity);
            }

            // Add company name to task
            $companyName = null;
            if ($task->customer_id && $task->workspace_id) {
                $key = $task->customer_id . '_' . $task->workspace_id;
                $companyName = $companyNamesMap[$key] ?? null;
            }
            $task->company_name = $companyName;
        });

        // Get response from withCount (it will handle pagination)
        $response = $this->withCount($tasks, 'Project tasks retrieved successfully');

        // Transform the paginated response to add company names
        if ($response->getStatusCode() === 200) {
            $responseData = json_decode($response->getContent(), true);
            $tasksData = $responseData['data'] ?? [];

            if (!empty($tasksData)) {
                // Create a map of task IDs to company names from our transformed collection
                $taskCompanyMap = [];
                foreach ($tasks as $task) {
                    $taskCompanyMap[$task->id] = $task->company_name ?? null;
                }

                // Add company name to each task in the paginated response
                foreach ($tasksData as &$task) {
                    $taskId = $task['id'] ?? null;
                    if ($taskId && isset($taskCompanyMap[$taskId])) {
                        $task['company_name'] = $taskCompanyMap[$taskId];
                    } else {
                        $task['company_name'] = null;
                    }
                }
                unset($task); // Break reference

                $responseData['data'] = $tasksData;
                return response()->json($responseData, 200);
            }
        }

        return $response;
    }

    protected function loadDocumentsWithUploader($task)
    {
        if (!$task->relationLoaded('documents') || $task->documents->isEmpty()) {
            return;
        }
        $task->documents->each(function ($document) {
            $uploader = null;
            
            if ($document->uploaded_by_type === 'internal_employee' && $document->uploaded_by) {
                $uploader = \App\Models\EmpPersonalDetails::where('emp_id', $document->uploaded_by)
                    ->select('emp_id', 'first_name', 'middle_name', 'last_name')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'internal_employee';
                    $uploader->name = trim(($uploader->first_name ?? '') . ' ' . ($uploader->middle_name ?? '') . ' ' . ($uploader->last_name ?? ''));
                }
            } elseif ($document->uploaded_by_type === 'subcontractor' && $document->uploaded_by) {
                $uploader = \App\Models\User::where('id', $document->uploaded_by)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'subcontractor';
                }
            } elseif ($document->uploaded_by_type === 'subcontractor_employee' && $document->uploaded_by) {
                $uploader = \App\Models\EmployeeSubcontractor::where('id', $document->uploaded_by)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'subcontractor_employee';
                    $uploader->name = trim(($uploader->first_name ?? '') . ' ' . ($uploader->middle_name ?? '') . ' ' . ($uploader->last_name ?? ''));
                }
            } elseif ($document->uploaded_by_type === 'customer' && $document->uploaded_by) {
                $uploader = \App\Models\User::where('id', $document->uploaded_by)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                    ->first();
                if ($uploader) {
                    $uploader->type = 'customer';
                }
            }
            $document->setRelation('uploadedBy', $uploader);
        });
    }

    protected function loadHistoryWithAssignedEmployee($task)
    {
        if (!$task->relationLoaded('histories') || $task->histories->isEmpty()) {
            return;
        }
        // Sort histories by created_at descending (latest first)
        $task->histories = $task->histories->sortByDesc('created_at')->values();
        $task->histories->each(function ($history) {
            $assignedEmployee = null;
            
            if (!$history->assigned_emp_id) {
                $history->setRelation('assignedEmployee', null);
                return;
            }
            
            if ($history->assigned_emp_type === 'internal_employee') {
                $assignedEmployee = \App\Models\EmpPersonalDetails::where('emp_id', $history->assigned_emp_id)
                    ->select('emp_id', 'first_name', 'middle_name', 'last_name')
                    ->first();
                if ($assignedEmployee) {
                    $assignedEmployee->type = 'internal_employee';
                    $assignedEmployee->name = trim(($assignedEmployee->first_name ?? '') . ' ' . ($assignedEmployee->middle_name ?? '') . ' ' . ($assignedEmployee->last_name ?? ''));
                }
            } elseif ($history->assigned_emp_type === 'subcontractor_employee') {
                $assignedEmployee = \App\Models\EmployeeSubcontractor::where('id', $history->assigned_emp_id)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($assignedEmployee) {
                    $assignedEmployee->type = 'subcontractor_employee';
                    $assignedEmployee->name = trim(($assignedEmployee->first_name ?? '') . ' ' . ($assignedEmployee->middle_name ?? '') . ' ' . ($assignedEmployee->last_name ?? ''));
                }
            } elseif ($history->assigned_emp_type === 'subcontractor') {
                // Check if it's a customer or subcontractor
                $user = \App\Models\User::where('id', $history->assigned_emp_id)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name', 'user_type')
                    ->first();
                if ($user) {
                    // Check if it's a customer (user_type = 2)
                    if ($user->user_type == config('constants.user_types.customer')) {
                        $user->type = 'customer';
                    } else {
                        $user->type = 'subcontractor';
                    }
                    $assignedEmployee = $user;
                }
            }
            $history->setRelation('assignedEmployee', $assignedEmployee);
        });
    }

    protected function getSubcontractorProjectTaskById(Request $request, $id)
    {
        $user = Auth::user();
        // Verify user is a subcontractor
        if (!($user instanceof \App\Models\User) || $user->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized: Only subcontractors can access this endpoint', 403);
        }
        $subcontractorId = $user->id;
        // Get all employee IDs for this subcontractor
        $employeeIds = EmployeeSubcontractor::whereHas('associations', function ($q) use ($subcontractorId) {
            $q->where('subcontractor_id', $subcontractorId)
              ->where('active', '1');
        })->pluck('id')->toArray();

        // Determine what to load based on query parameters (default: false for both)
        $includeDocuments = $request->has('comment') ? filter_var($request->input('comment'), FILTER_VALIDATE_BOOLEAN) : false;
        $includeHistory = $request->has('history') ? filter_var($request->input('history'), FILTER_VALIDATE_BOOLEAN) : false;
        
        $with = [];
        // Only load basic relationships if we need the full task object
        if (!$includeDocuments && !$includeHistory) {
            $with = ['project', 'site', 'prerequisiteTask'];
        }
        // Only add documents/history if explicitly requested
        if ($includeDocuments) {
            $with[] = 'documents';
        }
        if ($includeHistory) {
            $with[] = 'histories';
        }

        $query = ProjectManagement::with($with)
            ->where('id', $id)
            ->where('del', '0')
            ->where(function ($q) use ($subcontractorId, $employeeIds) {
                // Tasks assigned to subcontractor directly
                $q->where(function ($subQ) use ($subcontractorId) {
                    $subQ->where('assignment_type', 'subcontractor')
                        ->where(function ($sq) use ($subcontractorId) {
                            $sq->where('subcontractor_id', $subcontractorId)
                               ->orWhere('assigned_emp_id', $subcontractorId);
                        });
                })
                // Tasks assigned to subcontractor employees
                ->orWhere(function ($subQ) use ($subcontractorId, $employeeIds) {
                    $subQ->where('assignment_type', 'subcontractor_employee')
                        ->where('subcontractor_id', $subcontractorId)
                        ->whereIn('assigned_emp_id', $employeeIds);
                });
            });

        $task = $query->first();
        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }

        // If only documents are requested, return only documents array
        if ($includeDocuments && !$includeHistory) {
            $this->loadDocumentsWithUploader($task);
            return $this->success($task->documents, 'Documents retrieved successfully');
        }

        // If only history is requested, return only history array
        if ($includeHistory && !$includeDocuments) {
            $this->loadHistoryWithAssignedEmployee($task);
            return $this->success($task->histories, 'History retrieved successfully');
        }

        // If both are requested, return both arrays
        if ($includeDocuments && $includeHistory) {
            $this->loadDocumentsWithUploader($task);
            $this->loadHistoryWithAssignedEmployee($task);
            return $this->success([
                'documents' => $task->documents,
                'histories' => $task->histories
            ], 'Documents and history retrieved successfully');
        }

        // If neither is requested, return full task object
        // Manually load assignedEntity with type
        if ($task->assignment_type === 'subcontractor') {
            $subcontractorIdForTask = $task->assigned_emp_id ?? $task->subcontractor_id;
            if ($subcontractorIdForTask) {
                $entity = \App\Models\User::where('id', $subcontractorIdForTask)
                    ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                    ->first();
                if ($entity) {
                    $entity->type = 'subcontractor';
                }
                $task->setRelation('assignedEntity', $entity);
            }
        } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
            $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                ->first();
            if ($entity) {
                $entity->type = 'subcontractor_employee';
            }
            $task->setRelation('assignedEntity', $entity);
        }
        $task->unsetRelation('documents');
        $task->unsetRelation('histories');
        return $this->success($task, 'Task retrieved successfully');
    }

    protected function createSubcontractorProjectTask(Request $request)
    {
        $user = Auth::user();
        
        // Verify user is a subcontractor
        if (!($user instanceof \App\Models\User) || $user->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized: Only subcontractors can create tasks', 403);
        }

        $validator = $this->subcontractorProjectTaskValidation($request, false);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->error('Unable to determine customer and workspace', 400);
        }

        $data = $validator->validated();
        $data['customer_id'] = $ids['customer_id'];
        $data['workspace_id'] = $ids['workspace_id'];
        
        // Map assigned_emp_type to assignment_type
        if (isset($data['assigned_emp_type'])) {
            $data['assignment_type'] = $data['assigned_emp_type'];
            unset($data['assigned_emp_type']);
        }

        // Set subcontractor_id
        $data['subcontractor_id'] = $user->id;

        // When assignment_type is subcontractor, also save subcontractor_id to assigned_emp_id
        if (isset($data['assignment_type']) && $data['assignment_type'] === 'subcontractor') {
            $data['assigned_emp_id'] = $user->id;
        } elseif (isset($data['assignment_type']) && $data['assignment_type'] === 'subcontractor_employee') {
            // Verify the employee belongs to this subcontractor
            $employee = EmployeeSubcontractor::whereHas('associations', function ($q) use ($user) {
                $q->where('subcontractor_id', $user->id)
                  ->where('active', '1');
            })->where('id', $data['assigned_emp_id'])->first();

            if (!$employee) {
                return $this->error('Employee not found or does not belong to this subcontractor', 404);
            }
        }

        // Set default status if not provided
        if (!isset($data['status'])) {
            $data['status'] = ProjectManagement::STATUS_TODO;
        }

        // Set creator information
        $data['created_by'] = $user->id;
        $data['created_by_type'] = 'subcontractor';

        DB::beginTransaction();
        
        try {
            // Check if this is a prerequisite task scenario
            if (isset($data['prerequisite_task']) && !empty($data['prerequisite_task'])) {
                $prerequisiteTaskId = $data['prerequisite_task'];
                
                // Verify the prerequisite task exists and is accessible
                $prerequisiteTask = ProjectManagement::where('id', $prerequisiteTaskId)
                    ->where('del', '0')
                    ->where(function ($q) use ($user) {
                        $q->where(function ($subQ) use ($user) {
                            $subQ->where('assignment_type', 'subcontractor')
                                ->where(function ($sq) use ($user) {
                                    $sq->where('subcontractor_id', $user->id)
                                       ->orWhere('assigned_emp_id', $user->id);
                                });
                        })
                        ->orWhere(function ($subQ) use ($user) {
                            $employeeIds = EmployeeSubcontractor::whereHas('associations', function ($q) use ($user) {
                                $q->where('subcontractor_id', $user->id)->where('active', '1');
                            })->pluck('id')->toArray();
                            
                            $subQ->where('assignment_type', 'subcontractor_employee')
                                ->where('subcontractor_id', $user->id)
                                ->whereIn('assigned_emp_id', $employeeIds);
                        });
                    })
                    ->first();

                if (!$prerequisiteTask) {
                    DB::rollBack();
                    return $this->error('Prerequisite task not found or you do not have access to it.', 404);
                }

                $task = ProjectManagement::create($data);
                $this->saveSubcontractorProjectTaskDocuments($request, $task->id, $ids, $user);
                $this->logSubcontractorProjectTaskHistory(
                    $task->id,
                    'Task created and linked to prerequisite task #' . $prerequisiteTaskId,
                    $ids,
                    $task,
                    $data
                );
                $this->sendProjectManagementEmailToAssignee($task->load(['project', 'site']), 'New Task Assigned');
            } else {
                $task = ProjectManagement::create($data);
                $this->saveSubcontractorProjectTaskDocuments($request, $task->id, $ids, $user);
                $this->logSubcontractorProjectTaskHistory(
                    $task->id,
                    'Task created',
                    $ids,
                    $task,
                    $data
                );
                $this->sendProjectManagementEmailToAssignee($task->load(['project', 'site']), 'New Task Assigned');
            }

            DB::commit();
            
            // Reload task with relationships
            $task = $task->fresh(['project', 'site', 'prerequisiteTask', 'documents']);
            if ($task->assignment_type) {
                if ($task->assignment_type === 'subcontractor') {
                    $entity = \App\Models\User::where('id', $task->assigned_emp_id ?? $task->subcontractor_id)
                        ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor';
                    }
                    $task->setRelation('assignedEntity', $entity);
                } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
                    $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                        ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor_employee';
                    }
                    $task->setRelation('assignedEntity', $entity);
                }
            }

            $message = isset($data['prerequisite_task']) && !empty($data['prerequisite_task']) 
                ? 'Task created successfully with prerequisite link' 
                : 'Task created successfully';
            
            return $this->success($task, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating subcontractor project task: ' . $e->getMessage());
            return $this->error('Failed to create task: ' . $e->getMessage(), 500);
        }
    }

    protected function updateSubcontractorProjectTask(Request $request, $id)
    {
        $user = Auth::user();
        
        // Verify user is a subcontractor
        if (!($user instanceof \App\Models\User) || $user->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized: Only subcontractors can update tasks', 403);
        }

        $subcontractorId = $user->id;

        // Optimize: Get all employee IDs for this subcontractor using join instead of whereHas
        $employeeIds = DB::table('employees_subcontractors')
            ->join('employees_subcontractors_metas', 'employees_subcontractors.id', '=', 'employees_subcontractors_metas.emp_id')
            ->where('employees_subcontractors_metas.subcontractor_id', $subcontractorId)
            ->where('employees_subcontractors_metas.active', '1')
            ->pluck('employees_subcontractors.id')
            ->toArray();

        // Optimize: Find the task with prerequisite task eager loaded
        $task = ProjectManagement::with('prerequisiteTask:id,title,status,del')
            ->where('id', $id)
            ->where('del', '0')
            ->where(function ($q) use ($subcontractorId, $employeeIds) {
                $q->where(function ($subQ) use ($subcontractorId) {
                    $subQ->where('assignment_type', 'subcontractor')
                        ->where(function ($sq) use ($subcontractorId) {
                            $sq->where('subcontractor_id', $subcontractorId)
                               ->orWhere('assigned_emp_id', $subcontractorId);
                        });
                })
                ->orWhere(function ($subQ) use ($subcontractorId, $employeeIds) {
                    $subQ->where('assignment_type', 'subcontractor_employee')
                        ->where('subcontractor_id', $subcontractorId)
                        ->whereIn('assigned_emp_id', $employeeIds);
                });
            })
            ->first();

        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }

        // Check if task has a prerequisite task and verify it's completed (status 4 = DONE)
        // Optimize: Use eager loaded relationship instead of separate query
        if ($task->prerequisite_task) {
            $prerequisiteTask = $task->prerequisiteTask;
            
            if (!$prerequisiteTask || $prerequisiteTask->del != '0') {
                return $this->error('Prerequisite task not found.', 404);
            }
            
            // Only allow update if prerequisite task is completed (status 4 = DONE)
            if ($prerequisiteTask->status != ProjectManagement::STATUS_DONE) {
                $statusLabel = $prerequisiteTask->getStatusLabel();
                return $this->error("Cannot update this task. The prerequisite task ({$prerequisiteTask->title}) must be completed (Done) first. Current status: {$statusLabel}", 422);
            }
        }

        // Validation for update
        $validator = $this->subcontractorProjectTaskValidation($request, true);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->error('Unable to determine customer and workspace', 400);
        }

        $data = $validator->validated();
        $oldData = $task->toArray();

        DB::beginTransaction();
        
        try {
            // Map assigned_emp_type to assignment_type (if provided)
            if (isset($data['assigned_emp_type'])) {
                $data['assignment_type'] = $data['assigned_emp_type'];
                unset($data['assigned_emp_type']);
            }

            // Validate subcontractor_id if provided (must match authenticated subcontractor)
            if (isset($data['subcontractor_id']) && $data['subcontractor_id'] != $subcontractorId) {
                DB::rollBack();
                return $this->error('You can only assign tasks to yourself or your employees', 403);
            }

            // Handle assignment type changes
            $assignmentTypeChanged = false;
            if (isset($data['assignment_type']) && $data['assignment_type'] !== $task->assignment_type) {
                $assignmentTypeChanged = true;
                
                // Validate the new assignment
                if ($data['assignment_type'] === 'subcontractor_employee') {
                    // Verify employee belongs to this subcontractor - use cached employeeIds
                    if (isset($data['assigned_emp_id'])) {
                        if (!in_array($data['assigned_emp_id'], $employeeIds)) {
                            DB::rollBack();
                            return $this->error('Employee not found or does not belong to this subcontractor', 404);
                        }
                        
                        // Set subcontractor_id and assigned_emp_id for subcontractor_employee
                        // Always use authenticated subcontractor's ID for security
                        $task->subcontractor_id = $subcontractorId;
                        $task->assigned_emp_id = $data['assigned_emp_id'];
                    } else {
                        DB::rollBack();
                        return $this->error('assigned_emp_id is required when assignment_type is subcontractor_employee', 422);
                    }
                } elseif ($data['assignment_type'] === 'subcontractor') {
                    // When assignment_type is subcontractor, set assigned_emp_id to subcontractor_id
                    // Always use authenticated subcontractor's ID for security
                    $task->subcontractor_id = $subcontractorId;
                    $task->assigned_emp_id = $subcontractorId;
                }
                
                $task->assignment_type = $data['assignment_type'];
            } elseif (isset($data['assigned_emp_id']) && $data['assigned_emp_id'] !== $task->assigned_emp_id) {
                // If only assigned_emp_id is changing (not assignment_type)
                if ($task->assignment_type === 'subcontractor_employee') {
                    // Verify the new employee belongs to this subcontractor - use cached employeeIds
                    if (!in_array($data['assigned_emp_id'], $employeeIds)) {
                        DB::rollBack();
                        return $this->error('Employee not found or does not belong to this subcontractor', 404);
                    }
                    
                    $task->assigned_emp_id = $data['assigned_emp_id'];
                } elseif ($task->assignment_type === 'subcontractor') {
                    // For subcontractor type, assigned_emp_id should be the subcontractor_id
                    if ($data['assigned_emp_id'] != $subcontractorId) {
                        DB::rollBack();
                        return $this->error('For subcontractor assignment type, assigned_emp_id must be the subcontractor ID', 422);
                    }
                    $task->assigned_emp_id = $data['assigned_emp_id'];
                }
            }
            
            // Handle subcontractor_id update (if provided and different)
            if (isset($data['subcontractor_id']) && $data['subcontractor_id'] !== $task->subcontractor_id) {
                // Verify it's the same subcontractor (security check)
                if ($data['subcontractor_id'] != $subcontractorId) {
                    DB::rollBack();
                    return $this->error('You can only assign tasks to yourself or your employees', 403);
                }
                $task->subcontractor_id = $data['subcontractor_id'];
            }

            // Update allowed fields
            $allowedFields = [
                'title',
                'description',
                'priority',
                'status',
                'due_date',
                'completion_description',
                'reject_reason',
                'site_id',
            ];

            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $task->$field = $data[$field];
                }
            }

            // Handle status changes
            if (isset($data['status'])) {
                if ($data['status'] == ProjectManagement::STATUS_DONE) {
                    $task->completed_at = now();
                } elseif ($data['status'] != ProjectManagement::STATUS_DONE && $task->completed_at) {
                    $task->completed_at = null;
                }
            }

            $task->save();

            // Handle document uploads
            $this->saveSubcontractorProjectTaskDocuments($request, $task->id, $ids, $user);

            // Log history
            $description = 'Task updated';
            if (isset($data['status']) && $data['status'] != $oldData['status']) {
                $statusLabels = ProjectManagement::getStatusOptions();
                $oldStatus = $statusLabels[$oldData['status']] ?? 'Unknown';
                $newStatus = $statusLabels[$data['status']] ?? 'Unknown';
                $description = "Status changed from {$oldStatus} to {$newStatus}";
            }
            
            // Log assignment type change if it occurred
            if ($assignmentTypeChanged) {
                $oldAssignmentType = $oldData['assignment_type'] ?? 'Unknown';
                $newAssignmentType = $task->assignment_type ?? 'Unknown';
                $description .= ($description !== 'Task updated' ? '; ' : '') . "Assignment type changed from {$oldAssignmentType} to {$newAssignmentType}";
            }

            $this->logSubcontractorProjectTaskHistory(
                $task->id,
                $description,
                $ids,
                $task,
                $data
            );

            // Send email if status changed
            if (isset($data['status']) && $data['status'] != $oldData['status']) {
                // Optimize: Load relationships before sending email
                $task->load(['project', 'site']);
                $this->sendProjectManagementEmailToAssignee($task, 'Task Status Updated');
            }

            DB::commit();

            // Optimize: Reload task with relationships in a single query
            $task = ProjectManagement::with([
                'project:id,title',
                'site:id,title',
                'prerequisiteTask:id,title,status',
                'histories' => function($q) {
                    $q->latest()->limit(10);
                },
                'documents' => function($q) {
                    $q->latest()->limit(20);
                }
            ])->find($task->id);
            
            // Optimize: Load assignedEntity using single query based on assignment type
            if ($task->assignment_type === 'subcontractor') {
                $subcontractorIdForTask = $task->assigned_emp_id ?? $task->subcontractor_id;
                if ($subcontractorIdForTask) {
                    $entity = \App\Models\User::where('id', $subcontractorIdForTask)
                        ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor';
                    }
                    $task->setRelation('assignedEntity', $entity);
                }
            } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
                // Optimize: Only query if employee ID is in our cached list
                if (in_array($task->assigned_emp_id, $employeeIds)) {
                    $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                        ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor_employee';
                    }
                    $task->setRelation('assignedEntity', $entity);
                }
            }

            return $this->success($task, 'Task updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating subcontractor project task: ' . $e->getMessage());
            return $this->error('Failed to update task: ' . $e->getMessage(), 500);
        }
    }

    protected function saveSubcontractorProjectTaskDocuments($request, $taskId, $ids, $user, $comment = null)
    {
        // Handle single document
        if (!$request->hasFile('document')) {
            return;
        }
        $document = $request->file('document');
        if (!$document->isValid()) {
            return;
        }
        $uploadedBy = $user->id;
        $uploadedByType = 'subcontractor';
        // Get comment from parameter or request
        if ($comment === null) {
            $comment = $request->input('comment', null);
        }
        // Create a temporary request with the document
        $tempRequest = new Request();
        $tempRequest->files->set('documents', $document);
        $uploadResult = $this->handleFileImageUpload($tempRequest, 'ProjectManagementDocuments');
        $documentPath = null;
        if (is_array($uploadResult)) {
            if (isset($uploadResult['path'])) {
                $documentPath = $uploadResult['path'];
            } elseif (!empty($uploadResult) && isset($uploadResult[0]['path'])) {
                $documentPath = $uploadResult[0]['path'];
            }
        }
        if (!$documentPath) {
            return;
        }
        ProjectManagementDocument::create([
            'project_management_id' => $taskId,
            'document_name' => $document->getClientOriginalName(),
            'document_path' => $documentPath,
            'comment' => $comment,
            'uploaded_by' => $uploadedBy,
            'uploaded_by_type' => $uploadedByType,
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
        ]);
    }

    protected function uploadSubcontractorProjectTaskDocuments(Request $request)
    {
        $user = Auth::user();
        // Verify user is a subcontractor
        if (!($user instanceof \App\Models\User) || $user->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized: Only subcontractors can upload documents', 403);
        }
        $subcontractorId = $user->id;
        // Get all employee IDs for this subcontractor
        $employeeIds = EmployeeSubcontractor::whereHas('associations', function ($q) use ($subcontractorId) {
            $q->where('subcontractor_id', $subcontractorId)
              ->where('active', '1');
        })->pluck('id')->toArray();
        // Validation rules - single file only
        $rules = [
            'task_id' => 'required|integer|exists:project_management,id',
            'document' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240',
            'comment' => 'nullable|string',
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->error('Unable to determine customer and workspace', 400);
        }
        // Find the task and verify access
        $task = ProjectManagement::where('id', $request->task_id)
            ->where('del', '0')
            ->where(function ($q) use ($subcontractorId, $employeeIds) {
                // Tasks assigned to subcontractor directly
                $q->where(function ($subQ) use ($subcontractorId) {
                    $subQ->where('assignment_type', 'subcontractor')
                        ->where(function ($sq) use ($subcontractorId) {
                            $sq->where('subcontractor_id', $subcontractorId)
                               ->orWhere('assigned_emp_id', $subcontractorId);
                        });
                })
                // Tasks assigned to subcontractor employees
                ->orWhere(function ($subQ) use ($subcontractorId, $employeeIds) {
                    $subQ->where('assignment_type', 'subcontractor_employee')
                        ->where('subcontractor_id', $subcontractorId)
                        ->whereIn('assigned_emp_id', $employeeIds);
                });
            })
            ->first();

        if (!$task) {
            return $this->error('Task not found or you do not have access to it.', 404);
        }
        // Check if document or comment are provided
        $hasDocument = $request->hasFile('document');
        $hasComment = $request->filled('comment');
        if (!$hasDocument && !$hasComment) {
            return $this->error('Please provide at least one document or a comment.', 400);
        }
        DB::beginTransaction();
        try {
            $historyDescriptions = [];
            // Upload single document if provided
            if ($hasDocument) {
                // Get the single file
                $document = $request->file('document');
                if ($document && $document->isValid()) {
                    // Create a temporary request with the document
                    $tempRequest = new Request();
                    $tempRequest->files->set('documents', $document);
                    $uploadResult = $this->handleFileImageUpload($tempRequest, 'ProjectManagementDocuments');
                    $documentPath = null;
                    if (is_array($uploadResult)) {
                        if (isset($uploadResult['path'])) {
                            $documentPath = $uploadResult['path'];
                        } elseif (!empty($uploadResult) && isset($uploadResult[0]['path'])) {
                            $documentPath = $uploadResult[0]['path'];
                        }
                    }
                    if ($documentPath) {
                        ProjectManagementDocument::create([
                            'project_management_id' => $task->id,
                            'document_name' => $document->getClientOriginalName(),
                            'document_path' => $documentPath,
                            'comment' => $request->input('comment', null),
                            'uploaded_by' => $user->id,
                            'uploaded_by_type' => 'subcontractor',
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);   
                        $historyDescriptions[] = "Document uploaded";
                    }
                }
            }
            // Store comment if provided (always store, regardless of document)
            if ($hasComment) {
                // If document was uploaded, comment is already saved with the document
                // But if no document, create a separate record for the comment
                if (!$hasDocument) {
                    ProjectManagementDocument::create([
                        'project_management_id' => $task->id,
                        'document_name' => null,
                        'document_path' => null,
                        'comment' => $request->comment,
                        'uploaded_by' => $user->id,
                        'uploaded_by_type' => 'subcontractor',
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                }
                $historyDescriptions[] = "Comment added: " . $request->comment;
            }
            // Log history
            if (!empty($historyDescriptions)) {
                $description = implode('. ', $historyDescriptions);
                $this->logSubcontractorProjectTaskHistory(
                    $task->id,
                    $description,
                    $ids,
                    $task,
                    ['assignment_type' => $task->assignment_type, 'assigned_emp_id' => $task->assigned_emp_id, 'subcontractor_id' => $task->subcontractor_id]
                );
            }
            DB::commit();
            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'prerequisiteTask',
                'histories',
                'documents'
            ]);
            // Load assignedEntity for response
            if ($task->assignment_type === 'subcontractor') {
                $subcontractorIdForTask = $task->assigned_emp_id ?? $task->subcontractor_id;
                if ($subcontractorIdForTask) {
                    $entity = \App\Models\User::where('id', $subcontractorIdForTask)
                        ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor';
                    }
                    $task->setRelation('assignedEntity', $entity);
                }
            } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
                $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($entity) {
                    $entity->type = 'subcontractor_employee';
                }
                $task->setRelation('assignedEntity', $entity);
            }
            return $this->success($task, 'Document and comment uploaded successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error uploading subcontractor project task documents: ' . $e->getMessage());
            return $this->error('Failed to upload documents: ' . $e->getMessage(), 500);
        }
    }

    protected function logSubcontractorProjectTaskHistory($projectManagementId, $description, $ids = null, $task = null, $data = null)
    {
        if (!$ids) {
            $ids = $this->getCustomerAndWorkspaceIds();
        }

        $user = Auth::user();
        
        $historyData = [
            'project_management_id' => $projectManagementId,
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
            'description' => $description,
            'assigned_emp_id' => $user->id,
            'assigned_emp_type' => 'subcontractor',
        ];

        // Determine assignment info from data or task
        $assignmentType = null;
        $assignedEmpId = null;
        $subcontractorId = null;

        if ($data && isset($data['assignment_type'])) {
            $assignmentType = $data['assignment_type'];
            $assignedEmpId = $data['assigned_emp_id'] ?? null;
            $subcontractorId = $data['subcontractor_id'] ?? null;
        } elseif ($task) {
            $assignmentType = $task->assignment_type ?? null;
            $assignedEmpId = $task->assigned_emp_id ?? null;
            $subcontractorId = $task->subcontractor_id ?? null;
        }

        // Update history with assignment info if available
        if ($assignmentType === 'subcontractor' && $subcontractorId) {
            $historyData['assigned_emp_id'] = $subcontractorId;
            $historyData['assigned_emp_type'] = 'subcontractor';
        } elseif ($assignmentType === 'subcontractor_employee' && $assignedEmpId) {
            $historyData['assigned_emp_id'] = $assignedEmpId;
            $historyData['assigned_emp_type'] = 'subcontractor_employee';
        }

        ProjectManagementHistory::create($historyData);
    }

    protected function sendProjectManagementEmailToAssignee($task, $subjectPrefix = 'Task Assignment')
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        
        // Collect recipients - for subcontractor side, send to both customer/admin and assignee
        $recipients = [];
        
        // 1. Always send email to the customer/admin who owns the project/task
        if ($task->customer_id) {
            $customer = \App\Models\User::find($task->customer_id);
            if ($customer && !empty($customer->email)) {
                $recipients[] = ['email' => $customer->email, 'name' => $customer->name ?? $customer->company_name ?? 'Customer'];
            }
        }
        
        // 2. Also send email to the assignee (subcontractor or subcontractor employee)
        if ($task->assignment_type === 'subcontractor' && $task->subcontractor_id) {
            $sub = \App\Models\User::find($task->subcontractor_id);
            if ($sub && !empty($sub->email)) {
                // Avoid duplicate if customer and subcontractor are the same
                $exists = false;
                foreach ($recipients as $recipient) {
                    if ($recipient['email'] === $sub->email) {
                        $exists = true;
                        break;
                    }
                }
                if (!$exists) {
                    $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
                }
            }
        } elseif ($task->assignment_type === 'subcontractor_employee') {
            // Subcontractor (if different from customer)
            if ($task->subcontractor_id) {
                $sub = \App\Models\User::find($task->subcontractor_id);
                if ($sub && !empty($sub->email)) {
                    // Avoid duplicate if customer and subcontractor are the same
                    $exists = false;
                    foreach ($recipients as $recipient) {
                        if ($recipient['email'] === $sub->email) {
                            $exists = true;
                            break;
                        }
                    }
                    if (!$exists) {
                        $recipients[] = ['email' => $sub->email, 'name' => $sub->name];
                    }
                }
            }
            // Assigned subcontractor employee
            if ($task->assigned_emp_id) {
                $emp = \App\Models\EmployeeSubcontractor::find($task->assigned_emp_id);
                if ($emp && !empty($emp->email)) {
                    $name = trim(($emp->first_name ?? '') . ' ' . ($emp->last_name ?? ''));
                    // Avoid duplicate emails
                    $exists = false;
                    foreach ($recipients as $recipient) {
                        if ($recipient['email'] === $emp->email) {
                            $exists = true;
                            break;
                        }
                    }
                    if (!$exists) {
                        $recipients[] = ['email' => $emp->email, 'name' => $name];
                    }
                }
            }
        }
        
        if (empty($recipients)) {
            return;
        }
        
        $brandTitle = getBrandTitle($ids['customer_id'] ?? null, $ids['workspace_id'] ?? null);
        $statusLabel = $task->getStatusLabel();
        $subject = $subjectPrefix . ' | ' . $statusLabel . ' | ' . $brandTitle;
        
        // Load relationships if not already loaded
        if (!$task->relationLoaded('project')) {
            $task->load('project');
        }
        if (!$task->relationLoaded('site')) {
            $task->load('site');
        }
        
        // Prepare email data
        // Format due_date using BaseModel's date formatting (respects system settings)
        $dueDateFormatted = 'Not Set';
        if ($task->due_date) {
            try {
                // Get the raw date value to avoid any timezone conversion issues
                $rawDueDate = $task->getRawOriginal('due_date');
                if ($rawDueDate) {
                    // Use BaseModel's getUserDateFormat method to format according to system settings
                    $userDateFormat = $task->getUserDateFormat();
                    $dueDateFormatted = \Carbon\Carbon::parse($rawDueDate)->format($userDateFormat);
                } else {
                    // Fallback: use the already formatted value from model (BaseModel should have formatted it)
                    $dueDateFormatted = $task->due_date;
                }
            } catch (\Exception $e) {
                // Fallback: use the already formatted value from model
                $dueDateFormatted = $task->due_date ?? 'Not Set';
            }
        }
        
        $emailData = [
            'subject' => $subject,
            'task_title' => $task->title,
            'task_description' => $task->description,
            'task_priority' => $task->priority ?? 'Not Set',
            'task_status_label' => $statusLabel,
            'task_due_date' => $dueDateFormatted,
            'project_name' => $task->project->title ?? null,
            'site_name' => $task->site->title ?? null,
            'action_url' => env('FRONTEND_URL') ? rtrim(env('FRONTEND_URL')) : null,
            'customer_id' => $ids['customer_id'] ?? null,
            'reject_reason' => $task->reject_reason ?? null,
        ];
        
        // Send email to each recipient
        foreach ($recipients as $recipient) {
            $emailData['assignee_name'] = $recipient['name'] ?? null;
            $params = [
                'to' => $recipient['email'],
                'subject' => $subject,
                'msg' => view('Emails.project-management-assignee', $emailData + ['customer_id' => $task->customer_id ?? null])->render(),
            ];
            $this->SendInstantEmail($params);
        }
    }

    /**
     * Update subcontractor project task document (comment or document or both)
     */
    protected function updateSubcontractorProjectTaskDocument(Request $request)
    {
        $user = Auth::user();
        // Verify user is a subcontractor
        if (!($user instanceof \App\Models\User) || $user->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized: Only subcontractors can update documents', 403);
        }

        $subcontractorId = $user->id;
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->error('Unable to determine customer and workspace', 400);
        }

        // Validation rules
        $rules = [
            'document_id' => 'required|integer|exists:project_management_dacuments,id',
            'document' => 'nullable|file|mimes:pdf,jpeg,png,jpg,gif,webp|max:10240',
            'comment' => 'nullable|string',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        // Find the document and verify it belongs to this subcontractor
        $document = ProjectManagementDocument::where('id', $request->document_id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('uploaded_by', $subcontractorId)
            ->where('uploaded_by_type', 'subcontractor')
            ->first();

        if (!$document) {
            return $this->error('Document not found or you do not have permission to update it.', 404);
        }

        // Verify access to the task
        $task = ProjectManagement::where('id', $document->project_management_id)
            ->where('del', '0')
            ->first();

        if (!$task) {
            return $this->error('Task not found.', 404);
        }

        // Get all employee IDs for this subcontractor
        $employeeIds = EmployeeSubcontractor::whereHas('associations', function ($q) use ($subcontractorId) {
            $q->where('subcontractor_id', $subcontractorId)
              ->where('active', '1');
        })->pluck('id')->toArray();

        // Verify subcontractor has access to this task
        $hasAccess = false;
        if ($task->assignment_type === 'subcontractor') {
            $hasAccess = ($task->subcontractor_id == $subcontractorId || $task->assigned_emp_id == $subcontractorId);
        } elseif ($task->assignment_type === 'subcontractor_employee') {
            $hasAccess = ($task->subcontractor_id == $subcontractorId && in_array($task->assigned_emp_id, $employeeIds));
        }

        if (!$hasAccess) {
            return $this->error('You do not have access to this task.', 403);
        }

        // Check if at least one field is being updated
        $hasNewDocument = $request->hasFile('document');
        $hasNewComment = $request->filled('comment');

        if (!$hasNewDocument && !$hasNewComment) {
            return $this->error('Please provide at least a new document or a comment to update.', 400);
        }

        DB::beginTransaction();
        try {
            $historyDescriptions = [];
            $oldDocumentPath = $document->document_path;
            $oldComment = $document->comment;

            // Update document if provided
            if ($hasNewDocument) {
                $file = $request->file('document');
                if ($file && $file->isValid()) {
                    // Delete old file if it exists
                    if ($oldDocumentPath && file_exists(public_path($oldDocumentPath))) {
                        unlink(public_path($oldDocumentPath));
                    }

                    // Upload new document
                    $tempRequest = new Request();
                    $tempRequest->files->set('documents', $file);
                    $uploadResult = $this->handleFileImageUpload($tempRequest, 'ProjectManagementDocuments');
                    
                    $documentPath = null;
                    if (is_array($uploadResult)) {
                        if (isset($uploadResult['path'])) {
                            $documentPath = $uploadResult['path'];
                        } elseif (!empty($uploadResult) && isset($uploadResult[0]['path'])) {
                            $documentPath = $uploadResult[0]['path'];
                        }
                    }

                    if ($documentPath) {
                        $document->document_name = $file->getClientOriginalName();
                        $document->document_path = $documentPath;
                        $historyDescriptions[] = "Document updated";
                    }
                }
            }

            // Update comment if provided
            if ($hasNewComment) {
                $document->comment = $request->comment;
                if ($oldComment != $request->comment) {
                    $historyDescriptions[] = "Comment updated: " . $request->comment;
                }
            }

            // Save the document
            $document->save();

            // Log history
            if (!empty($historyDescriptions)) {
                $description = implode('. ', $historyDescriptions);
                $this->logSubcontractorProjectTaskHistory(
                    $document->project_management_id,
                    $description,
                    $ids,
                    $task,
                    ['assignment_type' => $task->assignment_type, 'assigned_emp_id' => $task->assigned_emp_id, 'subcontractor_id' => $task->subcontractor_id]
                );
            }

            DB::commit();

            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'prerequisiteTask',
                'histories',
                'documents'
            ]);

            // Load assignedEntity for response
            if ($task->assignment_type === 'subcontractor') {
                $subcontractorIdForTask = $task->assigned_emp_id ?? $task->subcontractor_id;
                if ($subcontractorIdForTask) {
                    $entity = \App\Models\User::where('id', $subcontractorIdForTask)
                        ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor';
                    }
                    $task->setRelation('assignedEntity', $entity);
                }
            } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
                $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($entity) {
                    $entity->type = 'subcontractor_employee';
                }
                $task->setRelation('assignedEntity', $entity);
            }

            return $this->success($task, 'Document updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating subcontractor project task document: ' . $e->getMessage());
            return $this->error('Failed to update document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Delete subcontractor project task document
     */
    protected function deleteSubcontractorProjectTaskDocument($id)
    {
        $user = Auth::user();
        // Verify user is a subcontractor
        if (!($user instanceof \App\Models\User) || $user->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized: Only subcontractors can delete documents', 403);
        }

        $subcontractorId = $user->id;
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->error('Unable to determine customer and workspace', 400);
        }

        // Find the document and verify it belongs to this subcontractor
        $document = ProjectManagementDocument::where('id', $id)
            ->where('del', '0')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('uploaded_by', $subcontractorId)
            ->where('uploaded_by_type', 'subcontractor')
            ->first();

        if (!$document) {
            return $this->error('Document not found or you do not have permission to delete it.', 404);
        }

        // Verify access to the task
        $task = ProjectManagement::where('id', $document->project_management_id)
            ->where('del', '0')
            ->first();

        if (!$task) {
            return $this->error('Task not found.', 404);
        }

        // Get all employee IDs for this subcontractor
        $employeeIds = EmployeeSubcontractor::whereHas('associations', function ($q) use ($subcontractorId) {
            $q->where('subcontractor_id', $subcontractorId)
              ->where('active', '1');
        })->pluck('id')->toArray();

        // Verify subcontractor has access to this task
        $hasAccess = false;
        if ($task->assignment_type === 'subcontractor') {
            $hasAccess = ($task->subcontractor_id == $subcontractorId || $task->assigned_emp_id == $subcontractorId);
        } elseif ($task->assignment_type === 'subcontractor_employee') {
            $hasAccess = ($task->subcontractor_id == $subcontractorId && in_array($task->assigned_emp_id, $employeeIds));
        }

        if (!$hasAccess) {
            return $this->error('You do not have access to this task.', 403);
        }

        DB::beginTransaction();
        try {
            $taskId = $document->project_management_id;
            $documentPath = $document->document_path;
            $hadDocument = !empty($document->document_path);
            $hadComment = !empty($document->comment);

            // Delete the file if it exists
            if ($documentPath && file_exists(public_path($documentPath))) {
                unlink(public_path($documentPath));
            }

            // Soft delete the document record
            $document->del = '1';
            $document->save();

            // Log history
            $historyDescriptions = [];
            if ($hadDocument && $hadComment) {
                $historyDescriptions[] = "Document and comment deleted";
            } elseif ($hadDocument) {
                $historyDescriptions[] = "Document deleted";
            } elseif ($hadComment) {
                $historyDescriptions[] = "Comment deleted";
            }

            if (!empty($historyDescriptions)) {
                $description = implode('. ', $historyDescriptions);
                $this->logSubcontractorProjectTaskHistory(
                    $taskId,
                    $description,
                    $ids,
                    $task,
                    ['assignment_type' => $task->assignment_type, 'assigned_emp_id' => $task->assigned_emp_id, 'subcontractor_id' => $task->subcontractor_id]
                );
            }

            DB::commit();

            // Reload task with all relationships
            $task = $task->fresh([
                'project',
                'site',
                'prerequisiteTask',
                'histories',
                'documents'
            ]);

            // Load assignedEntity for response
            if ($task->assignment_type === 'subcontractor') {
                $subcontractorIdForTask = $task->assigned_emp_id ?? $task->subcontractor_id;
                if ($subcontractorIdForTask) {
                    $entity = \App\Models\User::where('id', $subcontractorIdForTask)
                        ->select('id', 'name', 'email', 'mobile_number', 'company_name')
                        ->first();
                    if ($entity) {
                        $entity->type = 'subcontractor';
                    }
                    $task->setRelation('assignedEntity', $entity);
                }
            } elseif ($task->assignment_type === 'subcontractor_employee' && $task->assigned_emp_id) {
                $entity = \App\Models\EmployeeSubcontractor::where('id', $task->assigned_emp_id)
                    ->select('id', 'first_name', 'middle_name', 'last_name', 'email')
                    ->first();
                if ($entity) {
                    $entity->type = 'subcontractor_employee';
                }
                $task->setRelation('assignedEntity', $entity);
            }

            return $this->success($task, 'Document deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting subcontractor project task document: ' . $e->getMessage());
            return $this->error('Failed to delete document: ' . $e->getMessage(), 500);
        }
    }
}

