<?php

namespace App\Http\Controllers\Traits;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

trait ValidationTrait
{

    protected function adminLoginValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'email' => 'nullable|email', // For User table
            'password' => 'required|string',
        ]);
    }

    protected function subscriptionPlanValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'name' => 'nullable|string|max:255',
            'number_of_user' => 'nullable|integer',
            'custom_plan' => 'nullable|string',
            'active' => 'nullable|integer',
            'is_free_plan' => 'nullable|integer',
            'status' => 'nullable|integer',
            'package_price_monthly' => 'nullable|integer',
            'package_price_yearly' => 'nullable|integer',
            'price_per_user_monthly' => 'nullable|integer',
            'price_per_user_yearly' => 'nullable|integer',
            'price_per_workspace_monthly' => 'nullable|integer',
            'price_per_workspace_yearly' => 'nullable|integer',
            'module_id' => 'nullable|string',
            'number_of_workspace' => 'nullable|integer',
        ]);
    }

    protected function couponValidationRequest(Request $request, $isUpdate = false)
    {
        $rules = [
            'name' => 'nullable|string|max:255',
            'code' => 'nullable|string|max:255', // No need for unique check when updating
            'discount' => 'nullable|integer|min:0|max:100',
            'limit' => 'nullable|integer|min:0',
            'type' => 'nullable|string|in:percentage,flat,fixed_plan_wise',
            'minimum_spend' => 'nullable|integer|min:0',
            'maximum_spend' => 'nullable|integer|min:0',
            'limit_per_user' => 'nullable|integer|min:1',
            'expiry_date' => 'nullable|date|after:today',
            'included_plan' => 'nullable|array',
            'included_plan.*' => 'nullable|integer|exists:plans,id',
            'excluded_plan' => 'nullable|array',
            'description' => 'nullable|string|max:500',
            'is_active' => 'nullable|boolean',
        ];

        if (!$isUpdate) {
            // Apply unique validation for 'code' only when creating a coupon
            $rules['code'] = 'nullable|string|unique:coupons,code|max:255';
        }

        return Validator::make($request->all(), $rules);
    }




    protected function moduleValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'status' => 'nullable|integer',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
    }

    protected function subModuleValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'module_id' => 'required|integer',
            'status' => 'nullable|integer',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
    }
    private function customerSignupValidation(Request $request)
    {
        return Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'workspace_name' => 'required|string|max:255',
            'mobile_number' => 'nullable|string',
            'terms_accepted' => 'required|boolean|accepted',
        ]);
    }
    protected function customerValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'workspace_name' => 'required|string|max:255',
            'enable_domain' => 'nullable|string|max:255',
            'domain_type' => 'nullable|string|max:255',
            'domain' => 'nullable|string|max:255',
            'subdomain' => 'nullable|string|max:255',
            'status' => 'nullable|integer',
            'slug' => 'nullable|string|max:255',
            'is_disable' => 'nullable|integer',

            'customer_name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('emp_company_details', 'employee_email')->ignore($request->emp_id, 'id'), Rule::unique('users', 'email')->ignore($request->emp_id, 'id'),],
            'mobile_number' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'is_enable_login' => 'nullable|boolean',
            'user_type' => 'nullable|integer',
            'active_status' => 'nullable|integer',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'active_plan_id' => 'nullable|integer',
            'plan_expire_date' => 'nullable|date',
        ]);
    }

    protected function adminResetPasswordValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'password' => 'required|string|min:6',
            'password_confirmation' => 'required|same:password',
        ]);
    }

    protected function loginToggleValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'customer_id' => 'required|integer',
            'is_enable_login' => 'required|boolean',
        ]);
    }
    protected function companyLoginValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'customer_id' => 'required|integer',
        ]);
    }

    protected function AssignPlanValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'customer_id' => 'required|exists:users,id',
            'plan_id' => 'required|exists:plans,id',
            'price' => 'required|numeric',
            'expiration_date' => 'nullable|date|after:today',
        ]);
    }


    protected function helpdeskCategoryValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'color' => 'required|string|max:255',
        ]);
    }

    protected function documentTypeValidationRequest(Request $request, $isUpdate = false)
    {
        $rules = [
            'title' => 'required|string|max:255',
            'del' => 'nullable|string|in:0,1',
        ];

        if ($isUpdate) {
            $rules['id'] = 'required|integer|exists:document_types,id';
            // Make title optional for update (only validate if provided)
            $rules['title'] = 'nullable|string|max:255';
        }

        return Validator::make($request->all(), $rules);
    }

    protected function roleValidationRequest(Request $request, $isUpdate = false)
    {
        $rules = [
            'title' => 'required|string|max:255',
            'code' => 'required|string|max:50',
            'del' => 'nullable|string|in:0,1',
        ];

        if ($isUpdate) {
            $rules['id'] = 'required|integer|exists:roles,id';
            // Make fields optional for update (only validate if provided)
            $rules['title'] = 'nullable|string|max:255';
            $rules['code'] = 'nullable|string|max:50';
        }

        return Validator::make($request->all(), $rules);
    }

    protected function adminCustomerEditValidationRequest(Request $request)
    {

        return Validator::make($request->all(), [
            'name' => 'nullable|string|min:3',
            'email' => 'nullable|email',  // Add email validation

        ]);
    }

    protected function customDomainValidation(Request $request)
    {
        return Validator::make($request->all(), [
            'domain' => 'required',
        ]);
    }


    protected function helpdeskValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'user_id' => 'required|integer',
            'category_id' => 'required|integer',
            'email' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'descriptions' => 'required|string|max:255',
            'status' => 'nullable|string|max:255', // Accepts null, default handled below
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);
    }



    protected function handleValidationFailure($validator)
    {
        $errors = $validator->errors()->toArray();
        $message = reset($errors)[0];
        return $this->error($message, 422);
    }

    protected function customDomainStatusValidation(Request $request)
    {
        return Validator::make($request->all(), [
            'id' => 'required|integer|exists:custom_domain_requests,id',
            'status' => 'required|integer|in:0,1,2', // Adjust the 'in' values based on your status codes
        ]);
    }

    protected function adminGeneralSettingValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'settings' => 'required|array',
            'settings.*.key' => 'required|string',
            'settings.*.value' => 'nullable', // We'll validate the type in a custom way later
            'workspace' => 'nullable|integer',
            'created_by' => 'nullable|integer',
        ]);
    }


    protected function replyValidationRequest(Request $request)
    {

        return Validator::make($request->all(), [
            'ticket_id' => 'required|integer',
            'description' => 'required',
            'attachments' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);
    }

    protected function customerSubscriptionRequestValidationRequest(Request $request)
    {

        return Validator::make($request->all(), [
            'plan_id' => 'required|string|max:255',
            'name' => 'nullable|string|max:255',
            'email' => 'nullable|string|max:255',
            'months' => 'nullable|string|max:255',
            'plan_name' => 'nullable|string|max:255',
            'price' => 'nullable|string|max:255',
            'price_curency' => 'nullable|string|max:255',
            'payment_status' => 'nullable|string|max:255',
            'txn_id' => 'nullable|string|max:255',
            'receipt' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'user_id' => 'nullable|string|max:255',
            'is_refund' => 'nullable|string|max:255',
            'payment_type' => 'nullable|integer',
            'price' => 'nullable|string|max:255',
            'price_currency' => 'nullable|string|max:255',
            'type' => 'nullable|string|max:255',
            'bank_account_id' => 'nullable|string|max:255',
            'request' => 'nullable|string|max:255',
            'is_recurring' => 'nullable',
            'attachments' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'interval' => 'nullable|integer',
        ]);
    }

    protected function BankTransferRequestStatusValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'bank_id' => 'required|integer',
            'status' => 'required|integer',
            'customer_id' => 'required|integer',
        ]);
    }

    protected function BankTransferRequestDeleteValidationRequest(Request $request)
    {

        return Validator::make($request->all(), [
            'id' => 'required|integer',


        ]);
    }

    protected function companyStoreValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'address' => 'nullable|string|max:500',
            'abn_number' => 'nullable|string|max:100',
            'contact_persons' => 'array',
            'contact_persons.*.name' => 'nullable|string|max:255',
            'contact_persons.*.email' => 'nullable|email|max:255',
            'contact_persons.*.phone' => 'nullable|string|max:20',
            'contact_persons.*.role' => 'nullable|string|max:100',
            'logo_img' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp'
        ]);
    }

    protected function bulkScheduleRoasterCreateValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'start_date' => 'required',
            'end_date' => 'required|after_or_equal:start_date',
            'start_time' => 'required',
            'end_time' => 'required|after:start_time',
            'color_code' => 'required',
            'break_minutes' => 'required',
            'shift_notes' => 'required',
            'users_ids' => ['required', 'array', 'min:1'],
            'working_days_hidden' => 'required'
        ], [

            'end_time.after' => 'End time must be after start time',
            'users_ids.required' => 'Please select atleast one employee',
            'working_days_hidden.required' => 'Please select atleast one day'
        ]);
    }


    protected function customerProfileValidationRequest(Request $request)
    {
        $rules = [
            'name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'mobile_number' => 'nullable|string|max:20',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ];

        // Add password validation rules if password fields are present
        if ($request->filled('old_password') || $request->filled('new_password') || $request->filled('new_password_confirmation')) {
            $rules = array_merge($rules, [
                'old_password' => 'required|string|min:6',
                'new_password' => 'required|string|min:6|confirmed',
                'new_password_confirmation' => 'required|string|min:6',
            ]);
        }

        return Validator::make($request->all(), $rules);
    }

    protected function subcontractorProfileValidationRequest(Request $request)
    {
        $rules = [
            'name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'mobile_number' => 'nullable|string|max:20',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ];

        // Add password validation rules if password fields are present
        if ($request->filled('old_password') || $request->filled('new_password') || $request->filled('new_password_confirmation')) {
            $rules = array_merge($rules, [
                'old_password' => 'required|string|min:6',
                'new_password' => 'required|string|min:6|confirmed',
                'new_password_confirmation' => 'required|string|min:6',
            ]);
        }

        return Validator::make($request->all(), $rules);
    }


    // =================================
    // Validate payroll filter request
    // =================================



    protected function payrollFilterValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee' => 'nullable|integer|exists:emp_company_details,id',
            'pay_year' => 'nullable|integer|min:2000|max:2100',
            'pay_month' => 'nullable|integer|min:1|max:12',
            'status' => 'nullable|integer|in:0,1',
        ]);
    }

    /**
     * Validate fine filter request
     */
    protected function fineFilterValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee' => 'nullable|integer|exists:emp_company_details,id',
            'fine_amount' => 'nullable|numeric|min:0',
            'type' => 'nullable|integer|in:0,1',
            'date' => 'nullable|date',
        ]);
    }

    /**
     * Validate salary filter request
     */
    protected function salaryFilterValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee' => 'nullable|integer|exists:emp_company_details,id',
            'from' => 'nullable',
            'to' => 'nullable',
            'working_hours' => 'nullable|numeric',
            'basic_salary' => 'nullable|numeric|min:0',
        ]);
    }

    /**
     * Validate payroll store request
     */
    protected function payrollStoreValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'pay_year' => 'required|integer|min:2000|max:2100',
            'pay_month' => 'required|integer|min:1|max:12',
            'basic_salary' => 'required|numeric|min:0',
            'working_hours' => 'nullable|numeric',
            'hours_spent' => 'nullable|numeric|min:0',
            'calculated_salary' => 'required|numeric|min:0',
        ]);
    }

    /**
     * Validate fine store request
     */
    protected function fineStoreValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'fine_date' => 'required|date',
            'fine_amount' => 'required|numeric|min:0',
            'fine_reason' => 'required|string|max:500',
            'type' => 'required|integer|in:0,1',
        ]);
    }

    /**
     * Validate salary store request
     */
    protected function salaryStoreValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'from' => 'required',
            'to' => 'required|after_or_equal:from',
            'basic_salary' => 'required|numeric|min:0',
            'working_hours' => 'nullable|numeric',
        ]);
    }

    /**
     * Validate payroll update request
     */
    protected function payrollUpdateValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'pay_year' => 'required|integer|min:2000|max:2100',
            'pay_month' => 'required|integer|min:1|max:12',
            'basic_salary' => 'required|numeric|min:0',
            'working_hours' => 'nullable|numeric',
            'hours_spent' => 'nullable|numeric',
            'calculated_salary' => 'nullable|numeric',
        ]);
    }

    /**
     * Validate fine update request
     */
    protected function fineUpdateValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'fine_date' => 'required|date',
            'fine_amount' => 'required|numeric|min:0',
            'fine_reason' => 'required|string|max:500',
            'type' => 'required|integer|in:0,1',
        ]);
    }

    /**
     * Validate salary update request
     */
    protected function salaryUpdateValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'from' => 'required',
            'to' => 'required|after_or_equal:from',
            'basic_salary' => 'required|numeric|min:0',
            'working_hours' => 'nullable|numeric',
        ]);
    }

    /**
     * Validate employee fines request
     */
    protected function employeeFinesValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'from' => 'required|date',
            'to' => 'required|date|after_or_equal:from',
        ]);
    }

    /**
     * Validate select employees request
     */
    protected function selectEmployeesValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'month' => 'nullable|string|regex:/^\d{4}-\d{2}$/',
            'employee_search' => 'nullable|string|max:255',
        ]);
    }

    /**
     * Validate payroll generate request
     */
    protected function payrollGenerateValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'required|integer|exists:emp_company_details,id',
            'month' => 'required',
        ]);
    }

    /**
     * Validate bulk delete request
     */
    protected function bulkDeleteValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'required|integer|exists:emp_company_details,id',
            'month' => 'required',
        ]);
    }

    /**
     * Validate roster working hours request
     */
    protected function rosterWorkingHoursValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'year' => 'required|integer|min:2000|max:2100',
            'month' => 'required|integer|min:1|max:12',
        ]);
    }

    /**
     * Validate save payroll PDF request
     */
    protected function savePayrollPdfValidationRequest($request)
    {
        return Validator::make($request->all(), [
            'pdf' => 'required|file|mimes:pdf|max:10240', // 10MB max
        ]);
    }

    // =================================
    // Validate overtime request
    // =================================

    protected function overtimeFilterValidationRequest(Request $request)
    {
        $statuses = implode(',', config('constants.overtime_status'));
        return Validator::make($request->all(), [
            'employee_id' => 'nullable|integer|exists:emp_company_details,id',
            'site_id' => 'nullable|integer|exists:sites,id',
            'status' => "nullable|string|in:$statuses",
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date|after_or_equal:date_from',
            'first_name' => 'nullable|string|max:255',
            'teams' => 'nullable|integer|exists:emp_teams,id',
        ]);
    }

    /**
     * Validate overtime store request
     */
    protected function overtimeStoreValidationRequest(Request $request)
    {
        $statuses = implode(',', config('constants.overtime_status'));
        
        // Determine employee validation rule based on user_type
        $employeeValidation = 'required|integer';
        if (isset($request->user_type) && $request->user_type == 1) {
            // Subcontractor employee - check in employees_subcontractors table
            $employeeValidation .= '|exists:employees_subcontractors,id';
        } else {
            // Regular employee - check in emp_company_details table
            $employeeValidation .= '|exists:emp_company_details,id';
        }
        
        return Validator::make($request->all(), [
            'employee_id' => $employeeValidation,
            'site_id' => 'required|integer|exists:sites,id',
            'check_in' => 'required|date',
            'check_out' => 'required|date|after:check_in',
            'date' => 'required|date',
            'description' => 'nullable|string|max:1000',
            'status' => "nullable|string|in:$statuses",
            'user_type' => 'nullable|in:0,1', // 0 = regular employee, 1 = subcontractor employee
        ], [
            'employee_id.required' => 'Employee is required',
            'employee_id.exists' => 'Selected employee does not exist',
            'site_id.required' => 'Site is required',
            'site_id.exists' => 'Selected site does not exist',
            'check_in.required' => 'Check-in time is required',
            'check_out.required' => 'Check-out time is required',
            'check_out.after' => 'Check-out time must be after check-in time',
            'date.required' => 'Date is required',
            'description.required' => 'Description is required',
            'description.max' => 'Description cannot exceed 1000 characters',
        ]);
    }

    /**
     * Validate overtime update request
     */
    protected function overtimeUpdateValidationRequest(Request $request)
    {
        $statuses = implode(',', config('constants.overtime_status'));
        
        // Determine employee validation rule based on user_type
        $employeeValidation = 'required|integer';
        if (isset($request->user_type) && $request->user_type == 1) {
            // Subcontractor employee - check in employees_subcontractors table
            $employeeValidation .= '|exists:employees_subcontractors,id';
        } else {
            // Regular employee - check in emp_company_details table
            $employeeValidation .= '|exists:emp_company_details,id';
        }
        
        return Validator::make($request->all(), [
            'id' => 'required|integer|exists:overtimes,id',
            'employee_id' => $employeeValidation,
            'site_id' => 'required|integer|exists:sites,id',
            'check_in' => 'required|date',
            'check_out' => 'required|date|after:check_in',
            'date' => 'required|date',
            'description' => 'required|string|max:1000',
            'status' => "nullable|string|in:$statuses",
            'user_type' => 'nullable|in:0,1', // 0 = regular employee, 1 = subcontractor employee
        ], [
            'id.required' => 'Overtime ID is required',
            'id.exists' => 'Overtime record not found',
            'employee_id.required' => 'Employee is required',
            'employee_id.exists' => 'Selected employee does not exist',
            'site_id.required' => 'Site is required',
            'site_id.exists' => 'Selected site does not exist',
            'check_in.required' => 'Check-in time is required',
            'check_out.required' => 'Check-out time is required',
            'check_out.after' => 'Check-out time must be after check-in time',
            'date.required' => 'Date is required',
            'description.required' => 'Description is required',
            'description.max' => 'Description cannot exceed 1000 characters',
        ]);
    }

    /**
     * Validate overtime status update request
     */
    protected function overtimeStatusValidationRequest(Request $request)
    {
        $statuses = implode(',', config('constants.overtime_status'));
        $statusList = implode(', ', config('constants.overtime_status'));
        return Validator::make($request->all(), [
            'id' => 'required|integer|exists:overtimes,id',
            'status' => "required|string|in:$statuses",
        ], [
            'id.required' => 'Overtime ID is required',
            'id.exists' => 'Overtime record not found',
            'status.required' => 'Status is required',
            'status.in' => "Status must be $statusList",
        ]);
    }

    /**
     * Validate overtime application request (for employees)
     */
    protected function overtimeApplicationValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'site_id' => 'required|integer|exists:sites,id',
            'check_in' => 'required|date',
            'check_out' => 'required|date|after:check_in',
            'date' => 'required|date',
            'description' => 'required|string|max:1000',
        ], [
            'site_id.required' => 'Site is required',
            'site_id.exists' => 'Selected site does not exist',
            'check_in.required' => 'Check-in time is required',
            'check_out.required' => 'Check-out time is required',
            'check_out.after' => 'Check-out time must be after check-in time',
            'date.required' => 'Date is required',
            'description.required' => 'Description is required',
            'description.max' => 'Description cannot exceed 1000 characters',
        ]);
    }

    /**
     * Validate bulk delete request
     */
    protected function overtimeBulkDeleteValidationRequest(Request $request)
    {
        return Validator::make($request->all(), [
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:overtimes,id',
        ], [
            'ids.required' => 'At least one overtime record must be selected',
            'ids.array' => 'Invalid data format',
            'ids.min' => 'At least one overtime record must be selected',
            'ids.*.integer' => 'Invalid overtime ID format',
            'ids.*.exists' => 'One or more overtime records not found',
        ]);
    }

    protected function whsqStepContentValidationRequest(Request $request)
    {

        return Validator::make($request->all(), [
            'report_id' => 'required|integer',
            'step_id' => 'required|integer',
            'content_data' => 'required|array',
            'content_data.*.parent_id' => 'nullable|integer',
            'content_data.*.child_id' => 'nullable|integer',
            'content_data.*.grandchild_id' => 'nullable|integer',
            'content_data.*.sort_order' => 'nullable|integer',

            // Parent content validation
            'content_data.*.parent_content' => 'nullable|array',
            'content_data.*.parent_content.content' => 'nullable|string',
            'content_data.*.parent_content.start_new' => 'nullable|boolean',
            'content_data.*.parent_content.options' => 'nullable|array',
            'content_data.*.parent_content.options.*.option_id' => 'nullable|integer',
            'content_data.*.parent_content.options.*.option_value' => 'nullable|in:checked,unchecked,selected,unselected',
            'content_data.*.parent_content.options.*.option_type' => 'nullable|in:checkbox,radio,select',
            'content_data.*.parent_content.options.*.option_label' => 'nullable|string',
            'content_data.*.parent_content.options.*.main_heading' => 'nullable|string',
            'content_data.*.parent_content.options.*.group_order' => 'nullable|integer',
            'content_data.*.parent_content.dates' => 'nullable|array',
            'content_data.*.parent_content.dates.*.date_id' => 'nullable|integer',
            'content_data.*.parent_content.dates.*.date_value' => 'nullable|date',
            'content_data.*.parent_content.dates.*.date_label' => 'nullable|string',
            'content_data.*.parent_content.dates.*.date_type' => 'nullable|string',

            // Child content validation
            'content_data.*.child_content' => 'nullable|array',
            'content_data.*.child_content.content' => 'nullable|string',
            'content_data.*.child_content.start_new' => 'nullable|boolean',
            'content_data.*.child_content.options' => 'nullable|array',
            'content_data.*.child_content.options.*.option_id' => 'nullable|integer',
            'content_data.*.child_content.options.*.option_value' => 'nullable|in:checked,unchecked,selected,unselected',
            'content_data.*.child_content.options.*.option_type' => 'nullable|in:checkbox,radio,select',
            'content_data.*.child_content.options.*.option_label' => 'nullable|string',
            'content_data.*.child_content.options.*.main_heading' => 'nullable|string',
            'content_data.*.child_content.options.*.group_order' => 'nullable|integer',
            'content_data.*.child_content.dates' => 'nullable|array',
            'content_data.*.child_content.dates.*.date_id' => 'nullable|integer',
            'content_data.*.child_content.dates.*.date_value' => 'nullable|date',
            'content_data.*.child_content.dates.*.date_label' => 'nullable|string',
            'content_data.*.child_content.dates.*.date_type' => 'nullable|string',

            // Grandchild content validation
            'content_data.*.grandchild_content' => 'nullable|array',
            'content_data.*.grandchild_content.content' => 'nullable|string',
            'content_data.*.grandchild_content.start_new' => 'nullable|boolean',
            'content_data.*.grandchild_content.options' => 'nullable|array',
            'content_data.*.grandchild_content.options.*.option_id' => 'nullable|integer',
            'content_data.*.grandchild_content.options.*.option_value' => 'nullable|in:checked,unchecked,selected,unselected',
            'content_data.*.grandchild_content.options.*.option_type' => 'nullable|in:checkbox,radio,select',
            'content_data.*.grandchild_content.options.*.option_label' => 'nullable|string',
            'content_data.*.grandchild_content.options.*.main_heading' => 'nullable|string',
            'content_data.*.grandchild_content.options.*.group_order' => 'nullable|integer',
            'content_data.*.grandchild_content.dates' => 'nullable|array',
            'content_data.*.grandchild_content.dates.*.date_id' => 'nullable|integer',
            'content_data.*.grandchild_content.dates.*.date_value' => 'nullable|date',
            'content_data.*.grandchild_content.dates.*.date_label' => 'nullable|string',
            'content_data.*.grandchild_content.dates.*.date_type' => 'nullable|string'
        ]);
    }


    protected function defectAddImageValidation(Request $request)
    {
        return Validator::make($request->all(), [
            'defect_id' => 'required|integer|exists:defects,id',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
        ]);
    }

    protected function defectStatusValidation(Request $request)
    {
        return Validator::make($request->all(), [
            'id' => 'required|integer|exists:defects,id',
            'status' => 'required|integer|in:1,2,3,4,5,6',
            'reject_reason' => 'nullable|string',
            'assigned_emp_id' => 'nullable|integer|exists:emp_company_details,id',
        ]);
    }

    protected function siteDiaryValidation(Request $request, $isUpdate = false)
    {
        $rules = [
            'site_id' => 'required|integer|exists:sites,id',
            'weather' => 'nullable|string|max:255',
            'work_log' => 'nullable|string',
            'notes' => 'nullable|string',
            'equipment' => 'nullable|array',
            'equipment.*.id' => 'nullable|integer|exists:site_diary_equipment,id',
            'equipment.*.equipment_name' => 'required_with:equipment|string|max:255',
            'equipment.*.quantity' => 'required_with:equipment|integer|min:1',
            'tasks' => 'nullable|array',
            'tasks.*.id' => 'nullable|integer|exists:site_diary_tasks,id',
            'tasks.*.title' => 'required_with:tasks|string|max:255',
            'tasks.*.is_completed' => 'nullable|boolean',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
            'summary' => 'nullable|string',
            'diary_date' => 'nullable|date',
            'project_management_task_ids' => 'nullable|array',
            'project_management_task_ids.*' => 'integer|exists:project_management,id',
        ];

        if ($isUpdate) {
            $rules['id'] = 'required|integer|exists:site_diaries,id';
        }

        return Validator::make($request->all(), $rules);
    }
}
