<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\XeroService;
use App\Models\XeroToken;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;


class XeroOAuthController extends Controller
{
    protected $xeroService;

    public function __construct()
    {
        // XeroService will be instantiated per request with customer/workspace IDs
    }

    /**
     * Get or create XeroService instance with customer/workspace credentials
     */
    private function getXeroService($customerId = null, $workspaceId = null)
    {
        if (!$customerId || !$workspaceId) {
            // Try to get from authenticated user
            $ids = $this->getCustomerAndWorkspaceIds();
            if ($ids) {
                $customerId = $ids['customer_id'];
                $workspaceId = $ids['workspace_id'];
            }
        }

        return new XeroService($customerId, $workspaceId);
    }

    /**
     * Unified endpoint: Generate authorization URL
     * After visiting the URL and authorizing, the callback will automatically exchange the code for tokens
     * Then use /api/xero/token-status to check if token was stored successfully
     */
    public function authorizeXero(Request $request)
    {
        try {
            $user = Auth::user();
            if (!$user) {
                return $this->message('Unauthorized', 401);
            }
            // Get customer and workspace IDs to store with token
            $ids = $this->getCustomerAndWorkspaceIds();
            if (!$ids) {
                return $this->message('Unable to determine customer or workspace', 400);
            }
            // Generate state for CSRF protection and encode customer/workspace info
            // This way we can store customer_id and workspace_id in the token record
            $randomState = bin2hex(random_bytes(16));
            $stateData = [
                'state' => $randomState,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ];
            // Encode state data as base64 JSON (Xero will return this in state parameter)
            $state = base64_encode(json_encode($stateData));

            // Get XeroService with customer/workspace credentials
            $xeroService = $this->getXeroService($ids['customer_id'], $ids['workspace_id']);

            // Generate authorization URL
            $authUrl = $xeroService->getAuthorizationUrl($state);

            return $this->success([
                'authorization_url' => $authUrl,
                'state' => $state,
                'redirect_uri' => config('services.xero.redirect_uri'),
                'scopes' => config('services.xero.scopes'),
                'note' => 'The authorization code will be automatically exchanged for tokens when Xero redirects to /api/redirect. Access token expires in 30 minutes, use refresh_token to get a new one.',
            ], 'Authorization URL generated successfully');
        } catch (\Exception $e) {
            Log::error('Xero authorization error: ' . $e->getMessage());
            return $this->message('Failed to generate authorization URL: ' . $e->getMessage(), 500);
        }
    }

    public function redirect(Request $request)
    {
        try {
            $code = $request->get('code');
            $error = $request->get('error');
            $requestState = $request->get('state');
            $frontendUrl = env('FRONTEND_URL', 'https://staging.workforcems.com.au');
            // Handle errors from Xero
            if ($error) {
                $errorDescription = $request->get('error_description') ?? $error;
                Log::error('Xero callback error from Xero', [
                    'error' => $error,
                    'error_description' => $errorDescription,
                ]);
                $redirectUrl = $frontendUrl . '/settings?xero=error&error=' . urlencode($error) . '&error_description=' . urlencode($errorDescription);
                return redirect($redirectUrl);
            }
            if (!$code) {
                Log::error('Xero callback - no code provided', ['request_params' => $request->all()]);
                $redirectUrl = $frontendUrl . '/settings?xero=error&error=no_code&error_description=' . urlencode('Authorization code not provided');
                return redirect($redirectUrl);
            }
            // Decode state to get customer_id and workspace_id for storing in token record
            $customerId = null;
            $workspaceId = null;
            if ($requestState) {
                try {
                    $stateData = json_decode(base64_decode($requestState), true);
                    if ($stateData && isset($stateData['customer_id']) && isset($stateData['workspace_id'])) {
                        $customerId = $stateData['customer_id'];
                        $workspaceId = $stateData['workspace_id'];

                        Log::info('Xero callback - decoded state for token storage', [
                            'customer_id' => $customerId,
                            'workspace_id' => $workspaceId,
                        ]);
                    }
                } catch (\Exception $e) {
                    Log::warning('Xero callback - failed to decode state', [
                        'state' => $requestState,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            // Get XeroService with customer/workspace credentials
            if (!$customerId || !$workspaceId) {
                Log::error('Xero callback - missing customer/workspace IDs for service initialization');
                $redirectUrl = $frontendUrl . '/settings?xero=error&error=missing_info&error_description=' . urlencode('Customer or workspace information not found');
                return redirect($redirectUrl);
            }

            $xeroService = $this->getXeroService($customerId, $workspaceId);

            // Automatically exchange code for tokens (pass customer_id and workspace_id to save in table)
            $xeroToken = $xeroService->exchangeCodeForTokens($code, $customerId, $workspaceId);

            // Redirect to frontend settings page with success message
            $redirectUrl = $frontendUrl . '/settings?xero=success&tenant_id=' . urlencode($xeroToken->tenant_id) . '&tenant_name=' . urlencode($xeroToken->tenant_name ?? '');

            Log::info('Xero OAuth successful, redirecting to frontend', [
                'redirect_url' => $redirectUrl,
                'tenant_id' => $xeroToken->tenant_id,
            ]);

            return redirect($redirectUrl);
        } catch (\Exception $e) {
            Log::error('Xero OAuth callback error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request_params' => $request->all(),
            ]);
            $frontendUrl = env('FRONTEND_URL', 'https://staging.workforcems.com.au');
            $redirectUrl = $frontendUrl . '/settings?xero=error&error=exchange_failed&error_description=' . urlencode($e->getMessage());
            return redirect($redirectUrl);
        }
    }

    /**
     * Handle OAuth callback and exchange code for tokens automatically (legacy endpoint)
     * This endpoint is called by Xero after user authorizes the app
     * Xero redirects to: /api/xero/callback?code=AUTHORIZATION_CODE&state=STATE_VALUE
     * Redirects to frontend after successful token exchange
     */
    public function callback(Request $request)
    {
        return $this->redirect($request);
    }





    /**
     * Refresh token manually (usually done automatically)
     */
    public function refreshToken(Request $request)
    {

        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }
        $xeroToken = XeroToken::getActiveToken();

        if (!$xeroToken) {
            return $this->message('No Xero token found. Please authenticate first.', 404);
        }
        // Get XeroService with customer/workspace credentials from token
        $xeroService = $this->getXeroService($xeroToken->customer_id, $xeroToken->workspace_id);
        // Refresh token
        $xeroToken = $xeroService->refreshToken($xeroToken);
        return $this->success([
            'token_id' => $xeroToken->id,
            'expires_at' => $xeroToken->expires_at,
        ], 'Token refreshed successfully');
    }
}
