<?php

namespace App\Models;

class Defect extends BaseModel
{
    protected $fillable = [
        'project_id',
        'site_id',
        'assigned_emp_id',
        'assignment_type',
        'subcontractor_id',
        'assigned_subcontractor_emp_id',
        'customer_id',
        'workspace_id',
        'title',
        'description',
        'priority',
        'status',
        'due_date',
        'completion_description',
        'completed_at',
        'reject_reason',
        'created_by',
        'created_by_type',
        'del',
    ];

    protected $casts = [
        'due_date' => 'date',
        'completed_at' => 'datetime',
    
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
        // Hide the individual relations to prevent duplication
        'createdByEmployee',
        'createdByCustomer',
    ];

    // Add these to appends to automatically include in JSON
    protected $appends = [
        'creator_info',
        'creator_type',
        'assigned_employee'
    ];

    // Existing relations remain the same...
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id')->select('id', 'title', 'description');
    }

    public function site()
    {
        return $this->belongsTo(Sites::class, 'site_id')->select('id', 'title', 'description', 'longitude', 'latitude');
    }

    public function employee()
    {
        return $this->belongsTo(EmpCompanyDetails::class, 'assigned_emp_id')->select('id', 'employee_email');
    }

    public function employeePersonalDetails()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'assigned_emp_id', 'emp_id')
            ->select('emp_id', 'first_name', 'middle_name', 'last_name');
    }

    public function assignedSubcontractorEmployeePersonalDetails()
    {
        return $this->belongsTo(EmployeeSubcontractor::class, 'assigned_subcontractor_emp_id', 'id')
            ->select('id', 'first_name', 'middle_name', 'last_name', 'email');
    }

    public function images()
    {
        return $this->hasMany(DefectImage::class, 'defect_id');
    }

    public function histories()
    {
        return $this->hasMany(DefectHistory::class, 'defect_id');
    }

    public function createdByCustomer()
    {
        return $this->belongsTo(User::class, 'created_by')->select('id', 'name' , 'email');
    }

    public function createdByEmployee()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'created_by', 'emp_id')
            ->select('emp_id', 'first_name', 'middle_name', 'last_name');
    }

    public function subcontractor()
    {
        // subcontractor_id directly references users.id (subcontractor user in users table)
        return $this->belongsTo(User::class, 'subcontractor_id')
            ->select('id', 'name', 'email','mobile_number', 'company_name');
    }

    public function assignedSubcontractorEmployee()
    {
        return $this->belongsTo(EmployeeSubcontractor::class, 'assigned_subcontractor_emp_id', 'id')->select('id', 'email');
    }

    // NEW ACCESSORS - These will automatically be included in JSON output
    public function getCreatorInfoAttribute()
    {
        if ($this->created_by_type === 'employee') {
            return $this->createdByEmployee;
        } elseif ($this->created_by_type === 'subcontractor') {
            // Subcontractors are User models, so use createdByCustomer relationship
            return $this->createdByCustomer;
        } else {
            return $this->createdByCustomer;
        }
    }

    public function getCreatorTypeAttribute()
    {
        return $this->created_by_type;
    }

    public function getAssignedEmployeeAttribute()
    {
        // Check assignment_type first to determine which relationship to use
        // This ensures reliability even if both assigned_emp_id and assigned_subcontractor_emp_id are set
        
        // For regular employees (internal_employee)
        if ($this->assignment_type === 'internal_employee' && $this->assigned_emp_id) {
            if ($this->employeePersonalDetails) {
                $emp = $this->employeePersonalDetails;
                return [
                    'id' => (int)$emp->emp_id, // Convert emp_id (string) to integer id
                    'emp_id' => $emp->emp_id,
                    'first_name' => $emp->first_name,
                    'middle_name' => $emp->middle_name,
                    'last_name' => $emp->last_name,
                ];
            }
        }
        
        // For subcontractor employees
        if ($this->assignment_type === 'subcontractor_employee' && $this->assigned_subcontractor_emp_id) {
            if ($this->assignedSubcontractorEmployeePersonalDetails) {
                $emp = $this->assignedSubcontractorEmployeePersonalDetails;
                return [
                    'id' => (int)$emp->id, // Ensure id is integer
                    'emp_id' => (string)$emp->id, // Also include emp_id for consistency
                    'first_name' => $emp->first_name,
                    'middle_name' => $emp->middle_name,
                    'last_name' => $emp->last_name,
                    'email' => $emp->email ?? null,
                ];
            }
        }
        
        // For assignment_type = 'subcontractor', there's no employee assigned, return null
        // Also return null if assignment_type doesn't match or relationships don't exist
        return null;
    }

    // Keep your existing helper methods...
    public function getAssignedEntity()
    {
        switch ($this->assignment_type) {
            case 'internal_employee':
                return [
                    'type' => 'internal_employee',
                    'employee' => $this->employee,
                    'employee_details' => $this->employeePersonalDetails
                ];
            case 'subcontractor':
                return [
                    'type' => 'subcontractor',
                    'subcontractor' => $this->subcontractor
                ];
            case 'subcontractor_employee':
                return [
                    'type' => 'subcontractor_employee',
                    'subcontractor' => $this->subcontractor,
                    'employee' => $this->assignedSubcontractorEmployee,
                    'employee_details' => $this->assignedSubcontractorEmployeePersonalDetails
                ];
            default:
                return null;
        }
    }

    public function getAssigneeName()
    {
        $assigned = $this->getAssignedEntity();
        if (!$assigned) return 'Not Assigned';

        switch ($assigned['type']) {
            case 'internal_employee':
                if ($assigned['employee_details']) {
                    return $assigned['employee_details']->first_name . ' ' . $assigned['employee_details']->last_name;
                }
                return 'Internal Employee';
            case 'subcontractor':
                return $assigned['subcontractor'] ? $assigned['subcontractor']->name : 'Subcontractor';
            case 'subcontractor_employee':
                $name = $assigned['subcontractor'] ? $assigned['subcontractor']->name : 'Subcontractor';
                if ($assigned['employee_details']) {
                    $name .= ' - ' . $assigned['employee_details']->first_name . ' ' . $assigned['employee_details']->last_name;
                }
                return $name;
            default:
                return 'Unknown';
        }
    }
}
