<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class InductionDocumentSignature extends BaseModel
{
    use HasFactory;

    protected $table = 'induction_document_signatures';

    protected $fillable = [
        'induction_document_id',
        'document_version',
        'employee_id',
        'user_id',
        'signature_type',
        'signature_path',
        'signed_file',
        'signed_at',
        'is_valid',
        'notes',
        'customer_id',
        'workspace_id',
        'del'
    ];

    protected $casts = [
        'is_valid' => 'boolean',
        'signed_at' => 'datetime'
    ];

    /**
     * Get the induction document
     */
    public function inductionDocument()
    {
        return $this->belongsTo(InductionDocument::class, 'induction_document_id', 'id');
    }

    /**
     * Get the employee who signed (if employee)
     */
    public function employee()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'employee_id', 'emp_id');
    }

    /**
     * Get the user who signed (if customer user)
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'id');
    }

   
  

    /**
     * Scope to get only valid signatures
     */
    public function scopeValid($query)
    {
        return $query->where('is_valid', true)->where('del', 0);
    }

    /**
     * Scope to get signatures for a specific version
     */
    public function scopeForVersion($query, $version)
    {
        return $query->where('document_version', $version);
    }

    /**
     * Scope to get signatures for a specific document
     */
    public function scopeForDocument($query, $documentId)
    {
        return $query->where('induction_document_id', $documentId);
    }

    /**
     * Scope to get signatures by employee
     */
    public function scopeByEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * Scope to get signatures by user
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Check if a user/employee has signed a specific version
     */
    public static function hasSigned($documentId, $version, $employeeId = null, $userId = null)
    {
        $query = static::where('induction_document_id', $documentId)
            ->where('document_version', $version)
            ->where('is_valid', true)
            ->where('del', 0);
        
        if ($employeeId) {
            $query->where('employee_id', $employeeId);
        }
        
        if ($userId) {
            $query->where('user_id', $userId);
        }
        
        return $query->exists();
    }

    /**
     * Get the latest signature for a user/employee for any version of a document
     */
    public static function getLatestSignature($documentId, $employeeId = null, $userId = null)
    {
        $query = static::where('induction_document_id', $documentId)
            ->where('is_valid', true)
            ->where('del', 0);
        
        if ($employeeId) {
            $query->where('employee_id', $employeeId);
        }
        
        if ($userId) {
            $query->where('user_id', $userId);
        }
        
        return $query->orderBy('signed_at', 'desc')->first();
    }

    /**
     * Invalidate all signatures for a document (used in major updates)
     */
    public static function invalidateSignatures($documentId, $version = null)
    {
        $query = static::where('induction_document_id', $documentId)
            ->where('is_valid', true);
        
        if ($version) {
            $query->where('document_version', $version);
        }
        
        return $query->update(['is_valid' => false]);
    }
}

