<?php

namespace App\Models;

class ProjectManagement extends BaseModel
{
    // Status constants
    const STATUS_BLOCKED = 0;
    const STATUS_TODO = 1;
    const STATUS_IN_PROGRESS = 2;
    const STATUS_REVIEW = 3;
    const STATUS_DONE = 4;

    protected $table = 'project_management';

    protected $fillable = [
        'project_id',
        'site_id',
        'assigned_emp_id',
        'customer_id',
        'workspace_id',
        'title',
        'prerequisite_task',
        'description',
        'priority',
        'status',
        'assignment_type',
        'subcontractor_id',
        'due_date',
        'completion_description',
        'completed_at',
        'reject_reason',
        'created_by',
        'created_by_type',
        'del',
    ];

    protected $casts = [
        'due_date' => 'date',
        'completed_at' => 'datetime',
        'status' => 'integer',
        'prerequisite_task' => 'integer',
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];

    // Relationships
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id')->select('id', 'title', 'description');
    }

    public function site()
    {
        return $this->belongsTo(Sites::class, 'site_id')->select('id', 'title', 'description', 'longitude', 'latitude');
    }

    // Separate relationships for each assignment type
    public function assignedInternalEmployee()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'assigned_emp_id', 'emp_id')
            ->select('emp_id', 'first_name', 'middle_name', 'last_name');
    }

    public function assignedSubcontractor()
    {
        // For subcontractor, check both subcontractor_id and assigned_emp_id
        // Use COALESCE to prefer assigned_emp_id if it exists, otherwise use subcontractor_id
        return $this->belongsTo(User::class, 'subcontractor_id')
            ->select('id', 'name', 'email', 'mobile_number', 'company_name');
    }

    public function assignedSubcontractorByEmpId()
    {
        // Alternative relationship for when assigned_emp_id is used for subcontractor
        return $this->belongsTo(User::class, 'assigned_emp_id')
            ->select('id', 'name', 'email', 'mobile_number', 'company_name');
    }

    public function assignedSubcontractorEmployee()
    {
        return $this->belongsTo(EmployeeSubcontractor::class, 'assigned_emp_id', 'id')
            ->select('id', 'first_name', 'middle_name', 'last_name', 'email');
    }

    // Accessor to get assigned entity with type
    // This will use manually set relation if available, otherwise fall back to eager loaded relationships
    public function getAssignedEntityAttribute()
    {
        // First, check if assignedEntity was manually set via setRelation()
        if ($this->relationLoaded('assignedEntity')) {
            return $this->getRelation('assignedEntity');
        }

        // Fall back to eager loaded relationships if not manually set
        $entity = null;
        $type = $this->assignment_type;

        if ($type === 'internal_employee' && $this->relationLoaded('assignedInternalEmployee')) {
            $entity = $this->assignedInternalEmployee;
        } elseif ($type === 'subcontractor') {
            // For subcontractor, check both relationships
            if ($this->relationLoaded('assignedSubcontractorByEmpId') && $this->assigned_emp_id) {
                $entity = $this->assignedSubcontractorByEmpId;
            } elseif ($this->relationLoaded('assignedSubcontractor')) {
                $entity = $this->assignedSubcontractor;
            }
        } elseif ($type === 'subcontractor_employee' && $this->relationLoaded('assignedSubcontractorEmployee')) {
            $entity = $this->assignedSubcontractorEmployee;
        }

        if ($entity) {
            $entity->type = $type;
        }

        return $entity;
    }

    public function createdByCustomer()
    {
        return $this->belongsTo(User::class, 'created_by')->select('id', 'name', 'email');
    }

    public function createdByEmployee()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'created_by', 'emp_id')
            ->select('emp_id', 'first_name', 'middle_name', 'last_name');
    }

    public function prerequisiteTask()
    {
        return $this->belongsTo(ProjectManagement::class, 'prerequisite_task', 'id');
    }

    public function histories()
    {
        return $this->hasMany(ProjectManagementHistory::class, 'project_management_id')
            ->orderBy('created_at', 'desc');
    }

    public function documents()
    {
        return $this->hasMany(ProjectManagementDocument::class, 'project_management_id')
            ->where('del', '0');
    }

    // Helper methods
    public function getAssignedEntity()
    {
        $assignedEntity = $this->assignedEntity;
        if (!$assignedEntity) {
            return null;
        }

        return [
            'type' => $this->assignment_type,
            'entity' => $assignedEntity
        ];
    }

    public function getAssigneeName()
    {
        $assignedEntity = $this->assignedEntity;
        if (!$assignedEntity) {
            return 'Not Assigned';
        }

        switch ($this->assignment_type) {
            case 'internal_employee':
                if ($assignedEntity && isset($assignedEntity->first_name)) {
                    return $assignedEntity->first_name . ' ' . $assignedEntity->last_name;
                }
                return 'Internal Employee';
            case 'subcontractor':
                return $assignedEntity && isset($assignedEntity->name) ? $assignedEntity->name : 'Subcontractor';
            case 'subcontractor_employee':
                if ($assignedEntity && isset($assignedEntity->first_name)) {
                    return $assignedEntity->first_name . ' ' . $assignedEntity->last_name;
                }
                return 'Subcontractor Employee';
            default:
                return 'Unknown';
        }
    }

    public function getCreatorInfo()
    {
        if ($this->created_by_type === 'employee') {
            return $this->createdByEmployee;
        } elseif ($this->created_by_type === 'subcontractor') {
            return $this->createdByCustomer;
        } else {
            return $this->createdByCustomer;
        }
    }

    /**
     * Get status label
     */
    public function getStatusLabel()
    {
        $statusLabels = [
            self::STATUS_BLOCKED => 'Blocked',
            self::STATUS_TODO => 'Todo',
            self::STATUS_IN_PROGRESS => 'In Progress',
            self::STATUS_REVIEW => 'Review',
            self::STATUS_DONE => 'Done',
        ];

        return $statusLabels[$this->status] ?? 'Unknown';
    }

    /**
     * Get all status options
     */
    public static function getStatusOptions()
    {
        return [
            self::STATUS_BLOCKED => 'Blocked',
            self::STATUS_TODO => 'Todo',
            self::STATUS_IN_PROGRESS => 'In Progress',
            self::STATUS_REVIEW => 'Review',
            self::STATUS_DONE => 'Done',
        ];
    }

    /**
     * Check if status is completed
     */
    public function isCompleted()
    {
        return $this->status === self::STATUS_DONE;
    }

    /**
     * Check if status is in progress
     */
    public function isInProgress()
    {
        return $this->status === self::STATUS_IN_PROGRESS;
    }

    /**
     * Check if status is blocked
     */
    public function isBlocked()
    {
        return $this->status === self::STATUS_BLOCKED;
    }
}
