<?php

namespace App\Models;

class ProjectManagementHistory extends BaseModel
{
    protected $table = 'project_management_histories';

    protected $fillable = [
        'project_management_id',
        'assigned_emp_id',
        'assigned_emp_type',
        'customer_id',
        'workspace_id',
        'description',
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];

    // Relationships
    public function projectManagement()
    {
        return $this->belongsTo(ProjectManagement::class, 'project_management_id');
    }

    public function assignedEmployee()
    {
        if ($this->assigned_emp_type === 'internal_employee') {
            return $this->belongsTo(EmpPersonalDetails::class, 'assigned_emp_id', 'emp_id')
                ->select('emp_id', 'first_name', 'middle_name', 'last_name');
        } elseif ($this->assigned_emp_type === 'subcontractor') {
            return $this->belongsTo(User::class, 'assigned_emp_id')
                ->select('id', 'name', 'email', 'mobile_number', 'company_name');
        } elseif ($this->assigned_emp_type === 'subcontractor_employee') {
            return $this->belongsTo(EmployeeSubcontractor::class, 'assigned_emp_id', 'id')
                ->select('id', 'first_name', 'middle_name', 'last_name', 'email');
        }
        return null;
    }

    /**
     * Override toArray to format created_at and updated_at with time
     * This ensures timestamps include time component when serialized to JSON
     */
    public function toArray()
    {
        $array = parent::toArray();
        
        // Get user's preferred date and time format settings
        $userDateFormat = $this->getUserDateFormat();
        $userTimeFormat = $this->getUserTimeFormat();
        $userDateTimeFormat = $userDateFormat . ' ' . $userTimeFormat;
        
        // Format created_at and updated_at with time using raw database values
        if (isset($array['created_at']) && $array['created_at']) {
            try {
                $rawTimestamp = $this->getRawOriginal('created_at');
                if ($rawTimestamp) {
                    $carbonDate = \Carbon\Carbon::parse($rawTimestamp);
                    $array['created_at'] = $carbonDate->format($userDateTimeFormat);
                }
            } catch (\Exception $e) {
                // Keep original if parsing fails
            }
        }
        
        if (isset($array['updated_at']) && $array['updated_at']) {
            try {
                $rawTimestamp = $this->getRawOriginal('updated_at');
                if ($rawTimestamp) {
                    $carbonDate = \Carbon\Carbon::parse($rawTimestamp);
                    $array['updated_at'] = $carbonDate->format($userDateTimeFormat);
                }
            } catch (\Exception $e) {
                // Keep original if parsing fails
            }
        }
        
        return $array;
    }
}
