<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class RequiredDocument extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'title',
        'identity_text',
        'identity_file',
        'for_internal',
        'for_external',
        'for_subcontractor',
        'customer_id',
        'workspace_id',
        'priority',
        'del'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'for_internal' => 'boolean',
        'for_external' => 'boolean',
        'for_subcontractor' => 'boolean',
    ];

    public function requiredDocumentField()
    {
        return $this->hasMany(RequiredDocumentField::class, 'doc_id', 'id')->orderBy('priority');
    }

    public function dashboardRequiredDocumentField()
    {
        return $this->hasOne(RequiredDocumentField::class, 'doc_id', 'id')->orderBy('priority')
            ->where(['field_required' => 1, 'field_name' => 'Expiry Date'])->select('id', 'doc_id', 'field_name');
    }

    public function requiredDocumentActiveField()
    {
        return $this->hasMany(RequiredDocumentField::class, 'doc_id', 'id')->where('status', '1')->orderBy('priority');
    }

    /**
     * Scope to filter documents for internal employees
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForInternal($query)
    {
        return $query->where('for_internal', true);
    }

    /**
     * Scope to filter documents for external employees
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForExternal($query)
    {
        return $query->where('for_external', true);
    }

    /**
     * Scope to filter documents for subcontractors
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForSubcontractor($query)
    {
        return $query->where('for_subcontractor', true);
    }

    /**
     * Scope to filter documents by audience type
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $audienceType - 'internal', 'external', or 'subcontractor'
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForAudience($query, string $audienceType)
    {
        switch ($audienceType) {
            case 'internal':
                return $query->where('for_internal', true);
            case 'external':
                return $query->where('for_external', true);
            case 'subcontractor':
                return $query->where('for_subcontractor', true);
            default:
                return $query;
        }
    }

    /**
     * Parse audience input from frontend and set the appropriate flags
     * 
     * Accepts:
     * - Single value: 'internal', 'external', 'subcontractor', 'all'
     * - Comma-separated: 'internal,external', 'internal,subcontractor', etc.
     * - Array: ['internal', 'external']
     * 
     * @param string|array $audiences
     * @return array ['for_internal' => bool, 'for_external' => bool, 'for_subcontractor' => bool]
     */
    public static function parseAudiences($audiences): array
    {
        $result = [
            'for_internal' => false,
            'for_external' => false,
            'for_subcontractor' => false,
        ];

        // Handle null or empty
        if (empty($audiences)) {
            return $result;
        }

        // Convert to array if string
        if (is_string($audiences)) {
            // Handle 'all' special case
            if (strtolower($audiences) === 'all') {
                return [
                    'for_internal' => true,
                    'for_external' => true,
                    'for_subcontractor' => true,
                ];
            }
            
            // Handle legacy 'both' value (internal + external only, no subcontractor)
            if (strtolower($audiences) === 'both') {
                return [
                    'for_internal' => true,
                    'for_external' => true,
                    'for_subcontractor' => false,
                ];
            }

            // Split comma-separated values
            $audiences = array_map('trim', explode(',', $audiences));
        }

        // Process each audience type
        foreach ($audiences as $audience) {
            $audience = strtolower(trim($audience));
            switch ($audience) {
                case 'internal':
                    $result['for_internal'] = true;
                    break;
                case 'external':
                    $result['for_external'] = true;
                    break;
                case 'subcontractor':
                case 'subcontractors':
                    $result['for_subcontractor'] = true;
                    break;
                case 'all':
                    $result['for_internal'] = true;
                    $result['for_external'] = true;
                    $result['for_subcontractor'] = true;
                    break;
            }
        }

        return $result;
    }

    /**
     * Get audiences as array of strings
     * 
     * @return array
     */
    public function getAudiencesAttribute(): array
    {
        $audiences = [];
        
        if ($this->for_internal) {
            $audiences[] = 'internal';
        }
        if ($this->for_external) {
            $audiences[] = 'external';
        }
        if ($this->for_subcontractor) {
            $audiences[] = 'subcontractor';
        }
        
        return $audiences;
    }

    /**
     * Get audiences as comma-separated string
     * 
     * @return string
     */
    public function getAudiencesStringAttribute(): string
    {
        $audiences = $this->audiences;
        
        // If all three are selected, return 'all'
        if (count($audiences) === 3) {
            return 'all';
        }
        
        return implode(',', $audiences);
    }

    /**
     * Check if document is applicable for a given audience type
     * 
     * @param string $audienceType
     * @return bool
     */
    public function isForAudience(string $audienceType): bool
    {
        switch (strtolower($audienceType)) {
            case 'internal':
                return (bool) $this->for_internal;
            case 'external':
                return (bool) $this->for_external;
            case 'subcontractor':
            case 'subcontractors':
                return (bool) $this->for_subcontractor;
            default:
                return false;
        }
    }
}
