<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;

class SubcontractorCompany extends BaseModel
{
    use HasFactory;

    protected $table = 'subcontractor_companies';

    /**
     * Required document status constants
     * 
     * STATUS_NOT_UPLOADED (0): Required documents NOT uploaded
     * STATUS_COMPLETE (1): Required documents uploaded and approved by admin
     * STATUS_NEW_DOCS_ADDED (2): Documents were uploaded, but company added NEW documents after that
     * STATUS_REJECTED (3): Required documents uploaded but rejected by admin
     * STATUS_PENDING (4): Required documents uploaded and pending admin approval
     */
    const STATUS_NOT_UPLOADED = 0;
    const STATUS_COMPLETE = 1;
    const STATUS_NEW_DOCS_ADDED = 2;
    const STATUS_REJECTED = 3;
    const STATUS_PENDING = 4;

    /**
     * Induction status constants
     * 
     * INDUCTION_NOT_SIGNED (0): Induction documents NOT signed
     * INDUCTION_COMPLETE (1): All induction documents signed
     * INDUCTION_NEW_DOCS_ADDED (2): Induction documents were signed, but company added NEW documents (major update)
     * 
     * Note: Minor updates do NOT require re-signing, only major updates reset the status to 2.
     */
    const INDUCTION_NOT_SIGNED = 0;
    const INDUCTION_COMPLETE = 1;
    const INDUCTION_NEW_DOCS_ADDED = 2;

    protected $fillable = [
        'user_id',
        'customer_id',
        'workspace_id',
        'trade_id',
        'project_ids',
        'defect_ids',
        'required_docs_status',
        'induction_status',
        'rejection_reason',
        'del',
    ];

    protected $casts = [
        'project_ids' => 'array',
        'defect_ids' => 'array',
        'required_docs_status' => 'integer',
        'induction_status' => 'integer',
    ];

    /**
     * Get the user (subcontractor) that owns this company association.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the trade associated with this subcontractor company.
     */
    public function trade()
    {
        return $this->belongsTo(Trade::class, 'trade_id');
    }

    /**
     * Get the customer/company associated with this subcontractor.
     */
    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    /**
     * Get the required documents uploaded by this subcontractor for this company.
     */
    public function uploadedDocuments()
    {
        return $this->hasMany(SubcontractorRequiredDocument::class, 'subcontractor_id', 'user_id')
            ->where('customer_id', $this->customer_id)
            ->where('workspace_id', $this->workspace_id);
    }

    /**
     * Get readable status label
     * 
     * @return string
     */
    public function getRequiredDocsStatusLabelAttribute()
    {
        switch ($this->required_docs_status) {
            case self::STATUS_NOT_UPLOADED:
                return 'not_uploaded';
            case self::STATUS_COMPLETE:
                return 'complete';
            case self::STATUS_NEW_DOCS_ADDED:
                return 'new_docs_added';
            case self::STATUS_REJECTED:
                return 'rejected';
            case self::STATUS_PENDING:
                return 'pending';
            default:
                return 'unknown';
        }
    }

    /**
     * Check if all required documents are uploaded
     * 
     * @return bool
     */
    public function areAllDocumentsUploaded()
    {
        return SubcontractorRequiredDocument::areAllDocumentsComplete(
            $this->user_id,
            $this->customer_id,
            $this->workspace_id
        );
    }

    /**
     * Update the required_docs_status based on current document upload state
     * 
     * @return void
     */
    public function updateRequiredDocsStatus()
    {
        if ($this->areAllDocumentsUploaded()) {
            $this->update(['required_docs_status' => self::STATUS_COMPLETE]);
        }
    }

    /**
     * Mark all subcontractors as needing to upload new documents
     * Called when a company adds a new required document for subcontractors
     * Only affects subcontractors who have already been approved (status = 1)
     * 
     * @param int $customerId
     * @param int $workspaceId
     * @return int Number of affected subcontractors
     */
    public static function markNewDocsAddedForCompany($customer_id, $workspace_id)
    {
        // Define the statuses that should be changed
        $statusesToUpdate = [
            self::STATUS_PENDING,    // 4
            self::STATUS_COMPLETE    // 1
        ];
        
        // Update only subcontractors with STATUS_PENDING or STATUS_COMPLETE
        $affectedCount = self::where('customer_id', $customer_id)
            ->where('workspace_id', $workspace_id)
            ->whereIn('required_docs_status', $statusesToUpdate)
            ->update([
                'required_docs_status' => self::STATUS_NEW_DOCS_ADDED  // Change to 2
            ]);
        
        return $affectedCount;
    }

    /**
     * Get readable induction status label
     * 
     * @return string
     */
    public function getInductionStatusLabelAttribute()
    {
        switch ($this->induction_status) {
            case self::INDUCTION_NOT_SIGNED:
                return 'not_signed';
            case self::INDUCTION_COMPLETE:
                return 'complete';
            case self::INDUCTION_NEW_DOCS_ADDED:
                return 'new_docs_added';
            default:
                return 'unknown';
        }
    }

    /**
     * Mark all subcontractors as needing to sign new induction documents
     * Called when a company adds a new induction document OR creates a MAJOR update
     * Only affects subcontractors who have already signed (status = 1)
     * 
     * Note: Minor updates do NOT require re-signing
     * 
     * @param int $customerId
     * @param int $workspaceId
     * @return int Number of affected subcontractors
     */
    public static function markNewInductionDocsAddedForCompany($customer_id, $workspace_id)
    {
        // Only update subcontractors with INDUCTION_COMPLETE status
        $affectedCount = self::where('customer_id', $customer_id)
            ->where('workspace_id', $workspace_id)
            ->where('induction_status', self::INDUCTION_COMPLETE)
            ->where('del', '0')
            ->update([
                'induction_status' => self::INDUCTION_NEW_DOCS_ADDED  // Change to 2
            ]);
        
        return $affectedCount;
    }

    /**
     * Calculate and update induction status based on current signature state
     * 
     * @param int $subcontractorId The user_id of the subcontractor
     * @param int $customerId
     * @param int $workspaceId
     * @return bool Returns true if all induction documents are signed (complete)
     */
    public static function calculateAndUpdateInductionStatus($subcontractorId, $customerId, $workspaceId)
    {
        $subcontractorCompany = self::where('user_id', $subcontractorId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if (!$subcontractorCompany) {
            return false;
        }

        // Get all active induction documents for this customer/workspace
        $inductionDocuments = InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->get();

        // Filter documents that include 'subcontractor' in role_types or 'all'
        $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });

        // If no induction documents exist, consider as complete (induction_status = 1)
        if ($filteredInductionDocuments->isEmpty()) {
            $subcontractorCompany->update(['induction_status' => self::INDUCTION_COMPLETE]);
            return true;
        }

        // Check if all documents are signed with valid signatures
        $allSigned = true;
        foreach ($filteredInductionDocuments as $doc) {
            // For major updates, must sign the current version
            // For minor updates, any previous version signature is valid
            $hasSigned = InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('user_id', $subcontractorId)
                ->where('is_valid', true)
                ->where('del', '0')
                ->exists();

            // If not signed current version and it's a minor update, check previous versions
            if (!$hasSigned && $doc->update_type === 'minor') {
                $hasSigned = self::hasSignedAnyVersionInChain($doc, $subcontractorId, $customerId, $workspaceId);
            }

            if (!$hasSigned) {
                $allSigned = false;
                break;
            }
        }

        // Update induction status
        if ($allSigned) {
            $subcontractorCompany->update(['induction_status' => self::INDUCTION_COMPLETE]);
            return true;
        } else {
            // Don't change to NOT_SIGNED if currently NEW_DOCS_ADDED
            // Only set to NOT_SIGNED if currently 0 or undefined
            if ($subcontractorCompany->induction_status == self::INDUCTION_COMPLETE) {
                // This shouldn't happen in normal flow, but handle it
                $subcontractorCompany->update(['induction_status' => self::INDUCTION_NOT_SIGNED]);
            }
            return false;
        }
    }

    /**
     * Check if subcontractor has signed any version in the document chain (for minor updates)
     * 
     * @param InductionDocument $doc
     * @param int $subcontractorId
     * @param int $customerId
     * @param int $workspaceId
     * @return bool
     */
    public static function hasSignedAnyVersionInChain($doc, $subcontractorId, $customerId, $workspaceId)
    {
        // Find root document
        $rootDocumentId = $doc->id;
        $currentDoc = $doc;
        while ($currentDoc && $currentDoc->parent_document_id) {
            $rootDocumentId = $currentDoc->parent_document_id;
            $currentDoc = InductionDocument::where('id', $rootDocumentId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->first();
            if (!$currentDoc) {
                break;
            }
        }

        // Get all document IDs in the chain
        $documentIdsInChain = InductionDocument::where(function($q) use ($rootDocumentId) {
                $q->where('id', $rootDocumentId)
                  ->orWhere('parent_document_id', $rootDocumentId);
            })
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->pluck('id')
            ->toArray();

        // Check if subcontractor signed any version
        return InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
            ->where('user_id', $subcontractorId)
            ->where('is_valid', true)
            ->where('del', '0')
            ->exists();
    }

    /**
     * Check if both required docs and induction are complete
     * 
     * @return bool
     */
    public function isFullyCompliant()
    {
        return $this->required_docs_status == self::STATUS_COMPLETE 
            && $this->induction_status == self::INDUCTION_COMPLETE;
    }
}

