<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class SubcontractorEmployeeDocument extends BaseModel
{
    use HasFactory;

    protected $table = 'subcontractor_employee_documents';

    protected $fillable = [
        'employee_id',
        'subcontractor_id',
        'project_id',
        'customer_id',
        'workspace_id',
        'required_document_id',
        'required_document_field_id',
        'value',
        'approval_status',
        'rejection_reason',
        'approved_by',
        'approved_at',
        'del',
    ];

    protected $casts = [
        'approved_at' => 'datetime',
    ];

    /**
     * Approval status constants
     */
    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    /**
     * Get the employee that owns this document.
     */
    public function employee()
    {
        return $this->belongsTo(EmployeeSubcontractor::class, 'employee_id');
    }

    /**
     * Get the subcontractor (user) associated with this document.
     */
    public function subcontractor()
    {
        return $this->belongsTo(User::class, 'subcontractor_id');
    }

    /**
     * Get the project for this document.
     */
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    /**
     * Get the customer (company) for this document.
     */
    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    /**
     * Get the required document template.
     */
    public function requiredDocument()
    {
        return $this->belongsTo(RequiredDocument::class, 'required_document_id');
    }

    /**
     * Get the required document field.
     */
    public function requiredDocumentField()
    {
        return $this->belongsTo(RequiredDocumentField::class, 'required_document_field_id');
    }

    /**
     * Get the user who approved/rejected this document.
     */
    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Scope to filter by employee
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * Scope to filter by project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->where('project_id', $projectId);
    }

    /**
     * Scope to filter by customer
     */
    public function scopeForCustomer($query, $customerId, $workspaceId = null)
    {
        $query->where('customer_id', $customerId);
        if ($workspaceId !== null) {
            $query->where('workspace_id', $workspaceId);
        }
        return $query;
    }

    /**
     * Scope to filter by approval status
     */
    public function scopeWithApprovalStatus($query, $status)
    {
        return $query->where('approval_status', $status);
    }

    /**
     * Scope to filter pending documents
     */
    public function scopePending($query)
    {
        return $query->where('approval_status', self::STATUS_PENDING);
    }

    /**
     * Scope to filter approved documents
     */
    public function scopeApproved($query)
    {
        return $query->where('approval_status', self::STATUS_APPROVED);
    }

    /**
     * Scope to filter rejected documents
     */
    public function scopeRejected($query)
    {
        return $query->where('approval_status', self::STATUS_REJECTED);
    }

    /**
     * Scope to filter active (not deleted) documents
     */
    public function scopeActive($query)
    {
        return $query->where('del', '0');
    }
}

