<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class SubcontractorEmployeeInvitation extends BaseModel
{
    use HasFactory;

    protected $table = 'subcontractor_employee_invitations';

    /**
     * Required document status constants (same as SubcontractorCompany)
     * 
     * DOCS_STATUS_NOT_UPLOADED (0): Required documents NOT uploaded
     * DOCS_STATUS_COMPLETE (1): Required documents uploaded and approved by admin
     * DOCS_STATUS_NEW_DOCS_ADDED (2): Documents were approved, but company added NEW documents after that
     * DOCS_STATUS_REJECTED (3): Required documents uploaded but rejected by admin
     * DOCS_STATUS_PENDING (4): Required documents uploaded and pending admin approval
     */
    const DOCS_STATUS_NOT_UPLOADED = 0;
    const DOCS_STATUS_COMPLETE = 1;
    const DOCS_STATUS_NEW_DOCS_ADDED = 2;
    const DOCS_STATUS_REJECTED = 3;
    const DOCS_STATUS_PENDING = 4;

    /**
     * Induction status constants
     * 
     * INDUCTION_STATUS_NOT_SIGNED (0): No induction documents signed
     * INDUCTION_STATUS_COMPLETE (1): All required induction documents signed
     * INDUCTION_STATUS_NEW_DOCS_ADDED (2): Company added NEW induction documents after prior completion
     * INDUCTION_STATUS_PARTIAL (3): Some induction documents signed, others still pending
     */
    const INDUCTION_STATUS_NOT_SIGNED = 0;
    const INDUCTION_STATUS_COMPLETE = 1;
    const INDUCTION_STATUS_NEW_DOCS_ADDED = 2;
    const INDUCTION_STATUS_PARTIAL = 3;

    protected $fillable = [
        'employee_id',
        'subcontractor_id',
        'project_id',
        'customer_id',
        'workspace_id',
        'invitation_status',
        'status', // 1=active, 0=inactive - each company can manage independently
        'required_docs_status',
        'induction_status',
        'rejection_reason',
        'responded_at',
    ];

    protected $casts = [
        'responded_at' => 'datetime',
        'status' => 'integer',
        'required_docs_status' => 'integer',
        'induction_status' => 'integer',
    ];

    protected $appends = [
        'required_docs_status_label',
        'induction_status_label',
    ];

    /**
     * Invitation status constants
     */
    const STATUS_PENDING = 'pending';
    const STATUS_ACCEPTED = 'accepted';
    const STATUS_REJECTED = 'rejected';

    /**
     * Employee active status constants (for company-specific active/inactive)
     */
    const ACTIVE = 1;
    const INACTIVE = 0;

    /**
     * Get the employee that owns this invitation.
     */
    public function employee()
    {
        return $this->belongsTo(EmployeeSubcontractor::class, 'employee_id');
    }

    /**
     * Get the subcontractor (user) that sent this invitation.
     */
    public function subcontractor()
    {
        return $this->belongsTo(User::class, 'subcontractor_id');
    }

    /**
     * Get the project for this invitation.
     */
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    /**
     * Get the customer (company) for this invitation.
     */
    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    /**
     * Scope to filter by employee
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * Scope to filter by subcontractor
     */
    public function scopeForSubcontractor($query, $subcontractorId)
    {
        return $query->where('subcontractor_id', $subcontractorId);
    }

    /**
     * Scope to filter by project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->where('project_id', $projectId);
    }

    /**
     * Scope to filter by status
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('invitation_status', $status);
    }

    /**
     * Scope to filter pending invitations
     */
    public function scopePending($query)
    {
        return $query->where('invitation_status', self::STATUS_PENDING);
    }

    /**
     * Scope to filter accepted invitations
     */
    public function scopeAccepted($query)
    {
        return $query->where('invitation_status', self::STATUS_ACCEPTED);
    }

    /**
     * Scope to filter rejected invitations
     */
    public function scopeRejected($query)
    {
        return $query->where('invitation_status', self::STATUS_REJECTED);
    }

    /**
     * Scope to filter active employees (status = 1)
     */
    public function scopeActive($query)
    {
        return $query->where('status', self::ACTIVE);
    }

    /**
     * Scope to filter inactive employees (status = 0)
     */
    public function scopeInactive($query)
    {
        return $query->where('status', self::INACTIVE);
    }

    /**
     * Scope to filter by customer (for per-customer status management)
     */
    public function scopeForCustomer($query, $customerId, $workspaceId = null)
    {
        $query->where('customer_id', $customerId);
        if ($workspaceId !== null) {
            $query->where('workspace_id', $workspaceId);
        }
        return $query;
    }

    /**
     * Get readable required_docs_status label
     * 
     * @return string
     */
    public function getRequiredDocsStatusLabelAttribute()
    {
        switch ($this->required_docs_status) {
            case self::DOCS_STATUS_NOT_UPLOADED:
                return 'not_uploaded';
            case self::DOCS_STATUS_COMPLETE:
                return 'complete';
            case self::DOCS_STATUS_NEW_DOCS_ADDED:
                return 'new_docs_added';
            case self::DOCS_STATUS_REJECTED:
                return 'rejected';
            case self::DOCS_STATUS_PENDING:
                return 'pending';
            default:
                return 'unknown';
        }
    }

    /**
     * Get readable induction_status label
     * 
     * @return string
     */
    public function getInductionStatusLabelAttribute()
    {
        switch ($this->induction_status) {
            case self::INDUCTION_STATUS_NOT_SIGNED:
                return 'not_signed';
            case self::INDUCTION_STATUS_COMPLETE:
                return 'complete';
            case self::INDUCTION_STATUS_NEW_DOCS_ADDED:
                return 'new_docs_added';
            case self::INDUCTION_STATUS_PARTIAL:
                return 'partial';
            default:
                return 'unknown';
        }
    }

    /**
     * Mark all accepted employee invitations as needing to upload new documents
     * Called when a company adds a new required document for external employees
     * Only affects employees who have already been approved (status = 1) or are pending (status = 4)
     * 
     * @param int $customerId
     * @param int $workspaceId
     * @return int Number of affected invitations
     */
    public static function markNewDocsAddedForCustomer($customerId, $workspaceId)
    {
        // Define the statuses that should be changed
        $statusesToUpdate = [
            self::DOCS_STATUS_PENDING,    // 4
            self::DOCS_STATUS_COMPLETE    // 1
        ];
        
        // Update only accepted invitations with STATUS_PENDING or STATUS_COMPLETE
        $affectedCount = self::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('invitation_status', self::STATUS_ACCEPTED)
            ->whereIn('required_docs_status', $statusesToUpdate)
            ->update([
                'required_docs_status' => self::DOCS_STATUS_NEW_DOCS_ADDED  // Change to 2
            ]);
        
        return $affectedCount;
    }

    /**
     * Mark all accepted employee invitations as needing to sign new induction documents
     * Called when a company adds a new induction document that applies to external employees
     * Only affects employees who have already completed induction
     * 
     * @param int $customerId
     * @param int $workspaceId
     * @return int Number of affected invitations
     */
    public static function markNewInductionDocsAddedForCustomer($customerId, $workspaceId)
    {
        // Define the statuses that should be changed
        $statusesToUpdate = [
            self::INDUCTION_STATUS_PARTIAL,    // 3
            self::INDUCTION_STATUS_COMPLETE    // 1
        ];
        
        // Update only accepted invitations with STATUS_PARTIAL or STATUS_COMPLETE
        $affectedCount = self::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('invitation_status', self::STATUS_ACCEPTED)
            ->whereIn('induction_status', $statusesToUpdate)
            ->update([
                'induction_status' => self::INDUCTION_STATUS_NEW_DOCS_ADDED  // Change to 2
            ]);
        
        return $affectedCount;
    }

    /**
     * Update document status for a specific employee across all their invitations for a customer
     * 
     * @param int $employeeId
     * @param int $customerId
     * @param int $workspaceId
     * @param int $status
     * @return int Number of affected invitations
     */
    public static function updateEmployeeDocsStatus($employeeId, $customerId, $workspaceId, $status)
    {
        return self::where('employee_id', $employeeId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('invitation_status', self::STATUS_ACCEPTED)
            ->update(['required_docs_status' => $status]);
    }

    /**
     * Update induction status for a specific employee across all their invitations for a customer
     * 
     * @param int $employeeId
     * @param int $customerId
     * @param int $workspaceId
     * @param int $status
     * @return int Number of affected invitations
     */
    public static function updateEmployeeInductionStatus($employeeId, $customerId, $workspaceId, $status)
    {
        return self::where('employee_id', $employeeId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('invitation_status', self::STATUS_ACCEPTED)
            ->update(['induction_status' => $status]);
    }
}

