<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class SubcontractorRequiredDocument extends BaseModel
{
    use HasFactory;

    protected $table = 'subcontractor_required_documents';

    protected $fillable = [
        'subcontractor_id',
        'customer_id',
        'workspace_id',
        'required_document_id',
        'required_document_field_id',
        'value',
        'approval_status',
        'rejection_reason',
        'approved_by',
        'approved_at',
        'del',
    ];

    protected $casts = [
        'approved_at' => 'datetime',
    ];

    /**
     * Approval status constants
     */
    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    /**
     * Get the subcontractor (user) that uploaded this document.
     */
    public function subcontractor()
    {
        return $this->belongsTo(User::class, 'subcontractor_id');
    }

    /**
     * Get the customer/company this document belongs to.
     */
    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    /**
     * Get the required document template.
     */
    public function requiredDocument()
    {
        return $this->belongsTo(RequiredDocument::class, 'required_document_id');
    }

    /**
     * Get the required document field.
     */
    public function requiredDocumentField()
    {
        return $this->belongsTo(RequiredDocumentField::class, 'required_document_field_id');
    }

    /**
     * Get the user who approved/rejected this document.
     */
    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Scope to filter by subcontractor
     */
    public function scopeForSubcontractor($query, $subcontractorId)
    {
        return $query->where('subcontractor_id', $subcontractorId);
    }

    /**
     * Scope to filter by approval status
     */
    public function scopeWithApprovalStatus($query, $status)
    {
        return $query->where('approval_status', $status);
    }

    /**
     * Scope to filter pending documents
     */
    public function scopePending($query)
    {
        return $query->where('approval_status', self::STATUS_PENDING);
    }

    /**
     * Scope to filter approved documents
     */
    public function scopeApproved($query)
    {
        return $query->where('approval_status', self::STATUS_APPROVED);
    }

    /**
     * Scope to filter rejected documents
     */
    public function scopeRejected($query)
    {
        return $query->where('approval_status', self::STATUS_REJECTED);
    }

    /**
     * Scope to filter by company (customer)
     */
    public function scopeForCompany($query, $customerId, $workspaceId = null)
    {
        $query->where('customer_id', $customerId);
        if ($workspaceId !== null) {
            $query->where('workspace_id', $workspaceId);
        }
        return $query;
    }

    /**
     * Check if a specific required document has all its required fields uploaded
     * 
     * @param int $subcontractorId
     * @param int $customerId
     * @param int $workspaceId
     * @param int $requiredDocumentId
     * @return bool
     */
    public static function isDocumentComplete($subcontractorId, $customerId, $workspaceId, $requiredDocumentId)
    {
        // Get all required fields for this document
        $requiredFields = RequiredDocumentField::where('doc_id', $requiredDocumentId)
            ->where('field_required', 1)
            ->where('status', 1)
            ->pluck('id')
            ->toArray();

        if (empty($requiredFields)) {
            // No required fields, document is considered complete if any field is uploaded
            return self::where('subcontractor_id', $subcontractorId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('required_document_id', $requiredDocumentId)
                ->where('del', '0')
                ->whereNotNull('value')
                ->where('value', '!=', '')
                ->exists();
        }

        // Get uploaded fields for this document
        $uploadedFields = self::where('subcontractor_id', $subcontractorId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('required_document_id', $requiredDocumentId)
            ->where('del', '0')
            ->whereNotNull('value')
            ->where('value', '!=', '')
            ->pluck('required_document_field_id')
            ->toArray();

        // Check if all required fields are uploaded
        return empty(array_diff($requiredFields, $uploadedFields));
    }

    /**
     * Check if all required documents for a company are uploaded by a subcontractor
     * 
     * @param int $subcontractorId
     * @param int $customerId
     * @param int $workspaceId
     * @return bool
     */
    public static function areAllDocumentsComplete($subcontractorId, $customerId, $workspaceId)
    {
        // Get all required documents for subcontractors from this company
        $requiredDocuments = RequiredDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->get();

        if ($requiredDocuments->isEmpty()) {
            // No required documents defined, consider as complete
            return true;
        }

        foreach ($requiredDocuments as $requiredDocument) {
            if (!self::isDocumentComplete($subcontractorId, $customerId, $workspaceId, $requiredDocument->id)) {
                return false;
            }
        }

        return true;
    }
}

