<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class WHSQEContentDate extends BaseModel
{
    use HasFactory;

    protected $table = 'whsqe_content_dates';

    protected $fillable = [
        'content_meta_id',
        'date_value',
        'date_label',
        'date_type',
        'level',
        'level_id',
        'customer_id',
        'workspace_id'
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];

    protected $casts = [
        'date_value' => 'date',
        'level_id' => 'integer'
    ];

    /**
     * Get the content meta this date belongs to
     */
    public function contentMeta()
    {
        return $this->belongsTo(WHSQETableOfContentMeta::class, 'content_meta_id');
    }

    /**
     * Scope to filter by customer and workspace
     */
    public function scopeByCustomerAndWorkspace($query, $customerId, $workspaceId)
    {
        return $query->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId);
    }

    /**
     * Scope to filter by date type
     */
    public function scopeByType($query, $type)
    {
        return $query->where('date_type', $type);
    }

    /**
     * Scope to filter by level
     */
    public function scopeByLevel($query, $level)
    {
        return $query->where('level', $level);
    }

    /**
     * Scope to filter by level and level ID
     */
    public function scopeByLevelAndId($query, $level, $levelId)
    {
        return $query->where('level', $level)->where('level_id', $levelId);
    }

    /**
     * Scope to filter by date range
     */
    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('date_value', [$startDate, $endDate]);
    }



    /**
     * Get dates by level
     */
    public static function getDatesByLevel($contentMetaId, $level)
    {
        return self::where('content_meta_id', $contentMetaId)
            ->where('level', $level)
            ->orderBy('date_value')
            ->get();
    }

    /**
     * Get dates by level and level ID
     */
    public static function getDatesByLevelAndId($contentMetaId, $level, $levelId)
    {
        return self::where('content_meta_id', $contentMetaId)
            ->where('level', $level)
            ->where('level_id', $levelId)
            ->orderBy('date_value')
            ->get();
    }
} 