<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Carbon\Carbon;

class XeroToken extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'customer_id',
        'workspace_id',
        'access_token',
        'refresh_token',
        'id_token',
        'expires_in',
        'expires_at',
        'token_type',
        'tenant_id',
        'tenant_name',
        'tenant_type',
        'is_active',
    ];

    protected $hidden = [
        'access_token',
        'refresh_token',
        'id_token',
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'is_active' => 'boolean',
    ];

    /**
     * Check if token is expired
     */
    public function isExpired()
    {
        if (!$this->expires_at) {
            return true;
        }
        
        // Ensure we have a Carbon instance
        $expiresAt = $this->expires_at;
        if (is_string($expiresAt)) {
            // Use safeCarbonParse to handle various date formats
            $expiresAt = BaseModel::safeCarbonParse($expiresAt, 'XeroToken isExpired expires_at');
            if (!($expiresAt instanceof \Carbon\Carbon)) {
                // If parsing fails, assume expired
                return true;
            }
        } elseif (!($expiresAt instanceof \Carbon\Carbon)) {
            // If not a string and not Carbon, assume expired
            return true;
        }
        
        // Add 5 minute buffer before actual expiration
        return $expiresAt->copy()->subMinutes(5)->isPast();
    }

    /**
     * Get active token (global, not per customer/workspace)
     */
    public static function getActiveToken()
    {
        return static::where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->first();
    }
}
