<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class AddAudienceFlagsToRequiredDocumentsTable extends Migration
{
    /**
     * Run the migrations.
     * 
     * This migration adds boolean flags for each audience type to support
     * multi-select audience targeting for required documents.
     * 
     * Previous approach: Single 'for_who' column with values: 'internal', 'external', 'both'
     * New approach: Three boolean flags that can be combined in any way
     *
     * @return void
     */
    public function up()
    {
        Schema::table('required_documents', function (Blueprint $table) {
            $table->boolean('for_internal')->default(false)->after('for_who');
            $table->boolean('for_external')->default(false)->after('for_internal');
            $table->boolean('for_subcontractor')->default(false)->after('for_external');
        });

        // Migrate existing data from for_who to new boolean flags
        // 'internal' -> for_internal = true
        // 'external' -> for_external = true
        // 'both' -> for_internal = true, for_external = true
        DB::statement("UPDATE required_documents SET for_internal = 1 WHERE for_who = 'internal'");
        DB::statement("UPDATE required_documents SET for_external = 1 WHERE for_who = 'external'");
        DB::statement("UPDATE required_documents SET for_internal = 1, for_external = 1 WHERE for_who = 'both'");
        
        // Handle any NULL or empty for_who values - default to internal
        DB::statement("UPDATE required_documents SET for_internal = 1 WHERE for_who IS NULL OR for_who = ''");

        // Drop the for_who column as it's no longer needed
        Schema::table('required_documents', function (Blueprint $table) {
            $table->dropColumn('for_who');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        // Recreate for_who column first
        Schema::table('required_documents', function (Blueprint $table) {
            $table->string('for_who')->nullable()->after('identity_file');
        });

        // Migrate data back to for_who column before dropping flags
        // Priority: if both internal and external are true -> 'both'
        // Otherwise, use the single flag that's true
        // Note: subcontractor flag cannot be represented in old for_who, so we'll use 'both' as fallback
        DB::statement("UPDATE required_documents SET for_who = 'both' WHERE for_internal = 1 AND for_external = 1");
        DB::statement("UPDATE required_documents SET for_who = 'internal' WHERE for_internal = 1 AND for_external = 0 AND for_subcontractor = 0");
        DB::statement("UPDATE required_documents SET for_who = 'external' WHERE for_internal = 0 AND for_external = 1 AND for_subcontractor = 0");
        // If subcontractor is involved, default to 'both' (closest match)
        DB::statement("UPDATE required_documents SET for_who = 'both' WHERE for_subcontractor = 1");

        Schema::table('required_documents', function (Blueprint $table) {
            $table->dropColumn(['for_internal', 'for_external', 'for_subcontractor']);
        });
    }
}
