<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Traits\PermissionTrait;
use App\Models\Permission;
use App\Models\PermissionsModules;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AdminPermissionController extends Controller
{
    use PermissionTrait;

    /**
     * Generate a unique permission code
     *
     * @return string
     */
    protected function generateUniquePermissionCode()
    {
        do {
            $code = 'PERM-' . strtoupper(substr(uniqid(), -8));
        } while (Permission::where('permission_code', $code)->exists());

        return $code;
    }

    /**
     * Display a listing of permissions.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $modulesQuery = PermissionsModules::where('del', '0');
        if ($request->filled('module_id')) {
            $modulesQuery->where('id', $request->input('module_id'));
        }
        $modulesQuery->orderByRaw('COALESCE(priority, 999999) ASC');
        $modulesQuery->with([
            'permissionsSubModule' => function ($query) {
                $query->where('del', '0')
                    ->orderByRaw('COALESCE(priority, 999999) ASC');
            },
            'permissions' => function ($query) use ($request) {
                $query->where('del', '0');
                if ($request->filled('status')) {
                    $query->where('status', $request->input('status'));
                }
                if ($request->filled('search')) {
                    $search = $request->input('search');
                    $query->where('title', 'like', '%' . $search . '%');
                }
                $query->orderBy('title', 'asc');
            }
        ]);
        $modules = $modulesQuery->get();
        $result = [];
        foreach ($modules as $module) {
            $modulePermissions = $module->permissions;
            $subModules = $module->permissionsSubModule;
            $subModulesData = [];
            foreach ($subModules as $subModule) {
                $subModulePermissionsQuery = $subModule->permissions();
                if ($request->filled('status')) {
                    $subModulePermissionsQuery->where('status', $request->input('status'));
                }
                if ($request->filled('search')) {
                    $search = $request->input('search');
                    $subModulePermissionsQuery->where('title', 'like', '%' . $search . '%');
                }
                $subModulePermissions = $subModulePermissionsQuery->orderBy('title', 'asc')->get();
                if ($subModulePermissions->isNotEmpty() || (!$request->filled('status') && !$request->filled('search'))) {
                    $subModulesData[] = [
                        'id' => $subModule->id,
                        'title' => $subModule->title,
                        'module_id' => $subModule->module_id,
                        'image' => $subModule->image,
                        'priority' => $subModule->priority,
                        'link' => $subModule->link,
                        'permissions' => $subModulePermissions->toArray(),
                    ];
                }
            }
            if ($modulePermissions->isNotEmpty() || !empty($subModulesData) || (!$request->filled('status') && !$request->filled('search'))) {
                $result[] = [
                    'id' => $module->id,
                    'title' => $module->title,
                    'customer_id' => $module->customer_id,
                    'workspace_id' => $module->workspace_id,
                    'priority' => $module->priority,
                    'image' => $module->image,
                    'link' => $module->link,
                    'permissions' => $modulePermissions->toArray(),
                    'sub_modules' => $subModulesData,
                ];
            }
        }
        return $this->success($result, 'Permissions retrieved successfully');
    }

    /**
     * Store a newly created permission.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = $this->permissionValidationRequest($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $validatedData = $validator->validated();
        $existingPermissionQuery = Permission::where('title', $validatedData['title'])
            ->where('del', '0');
        if (isset($validatedData['module_id']) && $validatedData['module_id']) {
            $existingPermissionQuery->where('module_id', $validatedData['module_id']);
            if (isset($validatedData['module_type']) && $validatedData['module_type']) {
                $existingPermissionQuery->where('module_type', $validatedData['module_type']);
            }
        } else {
            $existingPermissionQuery->whereNull('module_id');
        }
        $existingPermission = $existingPermissionQuery->first();

        if ($existingPermission) {
            return $this->error('Permission with this title already exists for this module', 422);
        }
        $moduleValidation = $this->validateModuleAndType(
            $validatedData['module_id'] ?? null,
            $validatedData['module_type'] ?? null
        );
        if (!$moduleValidation['valid']) {
            return $this->error($moduleValidation['message'], 422);
        }
        $validatedData['status'] = $validatedData['status'] ?? 'active';
        $validatedData['del'] = '0';
        // Generate unique permission code
        $validatedData['permission_code'] = $this->generateUniquePermissionCode();
        $permission = Permission::create($validatedData);
        return $this->success($permission, 'Permission created successfully');
    }

    /**
     * Display the specified permission.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $permission = $this->getPermissionWithModule($id);
        if (!$permission || $permission->del !== '0') {
            return $this->notFound('Permission not found', 404);
        }
        return $this->success($permission, 'Permission retrieved successfully');
    }

    /**
     * Update the specified permission.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        $validator = $this->permissionValidationRequest($request, true);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $validatedData = $validator->validated();
        $id = $request->input('id');
        $permission = Permission::where('id', $id)
            ->where('del', '0')
            ->first();
        if (!$permission) {
            return $this->notFound('Permission not found', 404);
        }
        $newModuleId = $validatedData['module_id'] ?? $permission->module_id;
        $newModuleType = $validatedData['module_type'] ?? $permission->module_type;
        $newTitle = $validatedData['title'] ?? $permission->title;
        $moduleChanged = ($newModuleId != $permission->module_id || $newModuleType != $permission->module_type);
        if ($moduleChanged) {
            $existingPermissionQuery = Permission::where('title', $newTitle)
                ->where('del', '0');
            if ($newModuleId) {
                $existingPermissionQuery->where('module_id', $newModuleId);

                if ($newModuleType) {
                    $existingPermissionQuery->where('module_type', $newModuleType);
                }
            } else {
                $existingPermissionQuery->whereNull('module_id');
            }
            $existingPermission = $existingPermissionQuery->first();
            if ($existingPermission) {
                return $this->error('Permission with this title already exists for this module', 422);
            }
            $moduleValidation = $this->validateModuleAndType($newModuleId, $newModuleType);
            if (!$moduleValidation['valid']) {
                return $this->error($moduleValidation['message'], 422);
            }
            $newPermission = Permission::create([
                'title' => $newTitle,
                'module_id' => $newModuleId,
                'module_type' => $newModuleType,
                'status' => $validatedData['status'] ?? $permission->status,
                'permission_code' => $this->generateUniquePermissionCode(),
                'del' => '0',
            ]);
            return $this->success($newPermission, 'Permission created successfully for new module');
        }
        if (isset($validatedData['title']) && $validatedData['title'] !== $permission->title) {
            $existingPermissionQuery = Permission::where('title', $validatedData['title'])
                ->where('id', '!=', $id)
                ->where('del', '0');
            if ($newModuleId) {
                $existingPermissionQuery->where('module_id', $newModuleId);

                if ($newModuleType) {
                    $existingPermissionQuery->where('module_type', $newModuleType);
                }
            } else {
                $existingPermissionQuery->whereNull('module_id');
            }
            $existingPermission = $existingPermissionQuery->first();
            if ($existingPermission) {
                return $this->error('Permission with this title already exists for this module', 422);
            }
        }
        $moduleValidation = $this->validateModuleAndType($newModuleId, $newModuleType);
        if (!$moduleValidation['valid']) {
            return $this->error($moduleValidation['message'], 422);
        }
        // Remove permission_code from validated data to prevent editing
        unset($validatedData['permission_code']);
        $permission->update($validatedData);
        $permission->refresh();
        return $this->success($permission, 'Permission updated successfully');
    }

    /**
     * Remove the specified permission (soft delete).
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $permission = Permission::where('id', $id)
            ->where('del', '0')
            ->first();
        if (!$permission) {
            return $this->notFound('Permission not found', 404);
        }
        $permission->update(['del' => '1']);
        return $this->message('Permission deleted successfully');
    }

    /**
     * Change status of the specified permission.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:permissions,id',
            'status' => 'required|string|in:active,inactive',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $permission = Permission::where('id', $request->id)
            ->where('del', '0')
            ->first();
        if (!$permission) {
            return $this->notFound('Permission not found', 404);
        }
        $permission->update([
            'status' => $request->status
        ]);
        $permission->refresh();
        return $this->message('Permission status updated successfully');
    }
}
