<?php

namespace App\Http\Controllers;

use App\Models\CustomerPermission;
use App\Models\Permission;
use App\Models\PermissionsModules;
use App\Models\PermissionsSubModules;
use Illuminate\Http\Request;

class CustomerPermissionController extends Controller
{
    public function index(Request $request)
    {
        // Get unique permission IDs from customer permissions
        $permissionIds = CustomerPermission::where('del', '0')
            ->whereNotNull('permission_id')
            ->distinct()
            ->pluck('permission_id')
            ->toArray();
        
        if (empty($permissionIds)) {
            return $this->success([], 'Permissions retrieved successfully');
        }
        
        // Get permissions with their module/subModule relationships
        $permissions = Permission::whereIn('id', $permissionIds)
            ->where('del', '0')
            ->with(['module', 'subModule'])
            ->get();
        
        // Group permissions by module using collections
        $modulePermissions = $permissions->where('module_type', 'module')->groupBy('module_id');
        $subModulePermissions = $permissions->where('module_type', 'sub_module')->groupBy('module_id');
        
        // Get all unique module IDs
        $moduleIds = $permissions->map(function ($permission) {
            if ($permission->module_type === 'module') {
                return $permission->module_id;
            } elseif ($permission->module_type === 'sub_module' && $permission->subModule) {
                return $permission->subModule->module_id;
            }
            return null;
        })->filter()->unique()->toArray();
        
        if (empty($moduleIds)) {
            return $this->success([], 'Permissions retrieved successfully');
        }
        
        // Get modules
        $modules = PermissionsModules::whereIn('id', $moduleIds)
            ->where('del', '0')
            ->orderByRaw('COALESCE(priority, 999999) ASC')
            ->get();
        
        // Build response
        $result = $modules->map(function ($module) use ($modulePermissions, $subModulePermissions) {
            $moduleData = [
                'module_id' => $module->id,
                'module_name' => $module->title,
                'image' => $module->image,
                'link' => $module->link,
                'priority' => $module->priority,
                'permissions' => [],
                'sub_modules' => []
            ];
            
            // Add module-level permissions
            if ($modulePermissions->has($module->id)) {
                $moduleData['permissions'] = $modulePermissions->get($module->id)->map(function ($permission) {
                    return [
                        'id' => $permission->id,
                        'title' => $permission->title,
                        'module_id' => $permission->module_id,
                        'module_type' => $permission->module_type,
                        'status' => $permission->status,
                        'permission_code' => $permission->permission_code,
                        'created_at' => $permission->created_at,
                        'updated_at' => $permission->updated_at,
                    ];
                })->values()->toArray();
            }
            
            // Add sub_modules with their permissions (only submodules that have permissions)
            $subModuleIds = $subModulePermissions->keys()->toArray();
            if (!empty($subModuleIds)) {
                $subModules = PermissionsSubModules::whereIn('id', $subModuleIds)
                    ->where('module_id', $module->id)
                    ->where('del', '0')
                    ->orderByRaw('COALESCE(priority, 999999) ASC')
                    ->get();
                
                $moduleData['sub_modules'] = $subModules->map(function ($subModule) use ($subModulePermissions) {
                    $subModuleData = [
                        'sub_module_id' => $subModule->id,
                        'sub_module_name' => $subModule->title,
                        'image' => $subModule->image,
                        'link' => $subModule->link,
                        'priority' => $subModule->priority,
                        'permissions' => []
                    ];
                    
                    // Get permissions for this submodule
                    if ($subModulePermissions->has($subModule->id)) {
                        $subModuleData['permissions'] = $subModulePermissions->get($subModule->id)->map(function ($permission) {
                            return [
                                'id' => $permission->id,
                                'title' => $permission->title,
                                'module_id' => $permission->module_id,
                                'module_type' => $permission->module_type,
                                'status' => $permission->status,
                                'permission_code' => $permission->permission_code,
                                'created_at' => $permission->created_at,
                                'updated_at' => $permission->updated_at,
                            ];
                        })->values()->toArray();
                    }
                    
                    return $subModuleData;
                })->values()->toArray();
            }
            
            return $moduleData;
        })->toArray();
        
        return $this->success($result, 'Permissions retrieved successfully');
    }
}
