<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\Fund;
use App\Models\Role;
use App\Models\Tier;
use App\Models\User;
use App\Models\EmpTeam;
use App\Models\EmpType;
use App\Models\EmpAccess;
use App\Models\EmpPermit;
use App\Models\FundsMeta;
use App\General\MetaClass;
use App\Models\EmpDocuments;
use App\Models\RosterAssign;
use Illuminate\Http\Request;
use App\Models\EmpWorkerType;
use App\Models\RosterHistory;
use App\Models\Adminsettings;
use App\Models\EmpTeamsMember;
use App\Models\LinkManagement;
use App\Models\RosterTemplate;
use Illuminate\Validation\Rule;
use App\Models\RequiredDocument;
use App\Models\EmpCompanyDetails;
use App\Models\EmpWorkExperience;
use App\Models\EmpPersonalDetails;
use App\Models\PorfileSubmittedLog;
use App\Traits\RosterTemplateTrait;
use App\Models\EmpEmergencyContacts;
use App\Models\EmployeeSubcontractor;
use App\Models\EmpSubcontractor;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Intervention\Image\Facades\Image;
use App\Models\RequiredExternalDocument;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\EmpTeamController;


class ExternalEmployeeOnboardingController extends Controller
{
    use RosterTemplateTrait;
    public function externalEmployeeOnboardingUpdate(Request $request)
    {
        // Start database transaction
        DB::beginTransaction();
        
        try {
            $validator = Validator::make($request->all(), [
                'emp_id' => 'required|exists:employees_subcontractors,id',
                'team_id' => 'nullable',
                'first_name' => 'required',
                'middle_name' => 'nullable',
                'last_name' => 'required',
                'image' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
                'temporary_student_visa' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
                'signature' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
                'medical_attach_files.*' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
                'employee_email' => ['required', 'email', Rule::unique('employees_subcontractors', 'email')->ignore($request->emp_id, 'id'), Rule::unique('emp_company_details', 'employee_email'), Rule::unique('users', 'email')],
                'date_of_birth' => 'required|date|before:' . Carbon::now()->subYears(15)->format('Y-m-d'),
                'mobile' => 'required|numeric',
                'suburb' => 'required|regex:/^[a-zA-Z\s]*$/',
                'state' => 'required|regex:/^[a-zA-Z\s]*$/',
                'streat_address' => 'required',
                'job_title' => 'nullable',
                'employer_name' => 'nullable',
                'worker_type' => 'nullable',
                'trade_qualified' => 'required',
                'trade_qualified_year' => 'required_if:trade_qualified,=,1',
                'trade_licensed' => 'required',
                'trade_licensed_year' => 'required_if:trade_licensed,=,1',
                'work_experience' => 'required',
                'year_commenced' => 'required',
                'legally_australia' => 'required',
                'citizenship_status' => 'required',
                'classified_high_risk' => 'nullable',
                'contact_name_first' => 'required_without:emergency_contacts',
                'contact_relation_first' => 'required_with:contact_name_first',
                'contact_phone_first' => 'required_without:emergency_contacts',
                'contact_email_first' => 'nullable|email',
                'contact_address_first' => 'nullable',
                'emergency_contacts' => 'nullable|array',
                'emergency_contacts.*.name' => 'required_with:emergency_contacts',
                'emergency_contacts.*.phone' => 'required_with:emergency_contacts',
                'emergency_contacts.*.relation' => 'nullable',
                'emergency_contacts.*.email' => 'nullable|email',
                'emergency_contacts.*.address' => 'nullable',
                'allergies' => 'nullable',
                'details_allergies' => 'nullable',
                'previous_injuries' => 'nullable',
                'details_previous_injuries' => 'nullable',
                'medical_condition' => 'nullable',
                'project_ids' => 'nullable|json',
                'subcontractor_id' => 'nullable|exists:users,id',
            ], [
                'emp_id.required' => 'Employee ID is required.',
                'emp_id.exists' => 'Employee not found.',
                'image.mimes' => 'The image must be a file of type: jpeg, png, jpg, gif.',
                'image.max' => 'The image may not be greater than 1MB.',
                'suburb.regex' => 'The suburb field must contain only alphabetic characters.',
                'state.regex' => 'The state field contain only alphabetic characters.',
                'date_of_birth.before' => 'You must be at least 15 years old to register.',
                'trade_qualified_year.required_if' => 'The trade qualified year field is required',
                'trade_licensed_year.required_if' => 'The trade licensed year field is required',
                'contact_name_first.required_without' => 'The contact name field is required when emergency_contacts is not provided',
                'contact_relation_first.required_with' => 'The contact relation field is required when contact name is provided',
                'contact_phone_first.required_without' => 'The contact phone field is required when emergency_contacts is not provided',
                'emergency_contacts.array' => 'Emergency contacts must be an array',
                'emergency_contacts.*.name.required_with' => 'Contact name is required for each emergency contact',
                'emergency_contacts.*.phone.required_with' => 'Contact phone is required for each emergency contact',
            ]);

            if ($validator->fails()) {
                $error = $validator->errors()->first();
                return $this->error($error, 422);
            }

            $userTable = $this->getUserTable();
            $auth_id = 0;
            $workspace_id = 0;
            
            if ($userTable === "customer") {
                $auth_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            } elseif ($userTable === "emp") {
                $auth_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
            } else {
                return $this->error('Unauthorized access.', 403);
            }

            // Verify employee exists
            $employee = EmployeeSubcontractor::find($request->emp_id);
            if (!$employee) {
                return $this->error('Employee not found.', 404);
            }

            // Get subcontractor_id if provided, otherwise get from existing association
            $subcontractorId = $request->subcontractor_id;
            if (!$subcontractorId) {
                $existingAssociation = EmpSubcontractor::where('emp_id', $request->emp_id)->first();
                if ($existingAssociation) {
                    $subcontractorId = $existingAssociation->subcontractor_id;
                } else {
                    return $this->error('Subcontractor association not found. Please provide subcontractor_id.', 422);
                }
            }

            // Verify subcontractor exists and is valid
            $subcontractor = User::where('id', $subcontractorId)
                ->where('user_type', 5)
                ->where('del', '0')
                ->first();

            if (!$subcontractor) {
                return $this->error('Subcontractor not found or invalid.', 422);
            }

            // Get customer_id and workspace_id from request or authenticated user context
            // For customer editing, use auth context
            if ($userTable === "customer") {
                $customerId = $auth_id;
                $workspaceId = $workspace_id;
            } else {
                // For subcontractor editing, get from first associated company or request
                // This allows subcontractors to work independently across all companies
                $subcontractorCompany = \App\Models\SubcontractorCompany::where('user_id', $subcontractorId)
                    ->where('del', '0')
                    ->first();
                
                if ($subcontractorCompany) {
                    $customerId = $subcontractorCompany->customer_id;
                    $workspaceId = $subcontractorCompany->workspace_id;
                } else {
                    return $this->error('Subcontractor is not associated with any customer workspace. Please contact the administrator.', 422);
                }
            }

            // Update employee data (isNewEmployee = false for update)
            $msg = $this->storeEmployeeData($request, false);
            
            // Update emergency contacts
            $this->storeEmergencyContacts($request);
            
            // Get project_ids from request if provided
            $projectIds = $request->filled('project_ids') ? json_decode($request->project_ids, true) : null;
            
            // Update association in emp_subcontractors table
            $existingAssociation = EmpSubcontractor::where('emp_id', $request->emp_id)
                ->where('subcontractor_id', $subcontractorId)
                ->first();

            if ($existingAssociation) {
                // Update existing association
                $updateData = [];
                if ($projectIds !== null) {
                    $updateData['project_ids'] = $projectIds;
                }
                if (!empty($updateData)) {
                    $existingAssociation->update($updateData);
                }
            } else {
                // Create new association if it doesn't exist
                EmpSubcontractor::create([
                    'emp_id' => $request->emp_id,
                    'subcontractor_id' => $subcontractorId,
                    'project_ids' => $projectIds,
                    'active' => true,
                ]);
            }

            // Commit transaction if everything succeeds
            DB::commit();

            $data['emp_id'] = $request->emp_id;
            return $this->success($data, 'Employee updated successfully.');
            
        } catch (\Exception $e) {
            // Rollback transaction on any error
            DB::rollBack();
            Log::error('externalEmployeeOnboardingUpdate error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request' => $request->all()
            ]);
            return $this->error('An error occurred while updating the employee. Please try again.', 500);
        }
    }

    public function externalEmployeeOnboardingEdit(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'emp_id' => 'required'
        ], messages: [
            'emp_id.required' => 'Employee ID is required.'
        ]);

        if ($validator->fails()) {

            $error = $validator->errors()->first();

            return response()->json([
                'message' => $error

            ], 422);
        } else {

            $validatedData =  $validator->validated();
            $userTable = $this->getUserTable();
            $auth_id = 0;
            $workspace_id = 0;
            if ($userTable === "customer") {
                $auth_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
                $authPersonalDetails = User::where('id', Auth::user()->id)->first();
            }
            if ($userTable === "emp") {
                $auth_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
                $authPersonalDetails = EmpPersonalDetails::where('emp_id', Auth::user()->id)->first();
            }
            $emp_company_details =  EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->with(['empPersonalDetails', 'empEmergencyContacts', 'empAccess', 'empworkExperience', 'workerType', 'empTeamsMembers'])->where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->where('id', $validatedData['emp_id'])->first();
            if ($emp_company_details) unset($emp_company_details['password']);

            $data['emp_company_details'] = $emp_company_details;

            $emp_permits =  EmpPermit::where('del', '0')->where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->select('id', 'title')->get();

            $data['emp_permits'] = $emp_permits;

            $emp_types =  EmpType::where('del', '0')->where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->select('id', 'title')->get();

            $data['emp_types'] = $emp_types;

            $pkg_tiers =  Tier::where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->select('id', 'title')->get();

            $data['pkg_tiers'] = $pkg_tiers;

            $roles = Role::where('del', '0')->get();

            $data['emp_roles'] = $roles;

            $emp_work_experiences =  EmpWorkExperience::where('del', '0')->where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->select('id', 'title')->get();

            $data['emp_work_experiences'] = $emp_work_experiences;

            $emp_worker_types =  EmpWorkerType::where('del', '0')->where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->get();

            $data['emp_worker_types'] = $emp_worker_types;


            $funds = Fund::where('del', '0')->where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->get();
            $funds_meta = FundsMeta::where('emp_id', $validatedData['emp_id'])->with('fund')->get();

            $c = 0;

            foreach ($funds as $f) {

                foreach ($funds_meta as $fm) {
                    if ($fm->fund_id == $f->id) {
                        $funds[$c]['usi_number'] = $fm->usi_number;
                    }
                }

                $c++;
            }

            $data['funds'] = $funds;
            $data['funds_meta'] = $funds_meta;

            $user_meta =  DB::table('user_meta')->where('emp_id', $validatedData['emp_id'])->get();

            $account_title = MetaClass::getOptionValue('account_title', $user_meta);
            $account_number = MetaClass::getOptionValue('account_number', $user_meta);
            $bank_name = MetaClass::getOptionValue('bank_name', $user_meta);

            $data['meta']['account_title'] = $account_title;
            $data['meta']['account_number'] = $account_number;
            $data['meta']['bank_name'] = $bank_name;


            $data['medicalAttachFiles'] = DB::table('medical_attach_files')->where('emp_id', $validatedData['emp_id'])->get();




            // For external employees, return documents with for_external = true
            $data['required_documents'] = RequiredDocument::with('requiredDocumentField')
                ->where('customer_id', $auth_id)
                ->where('workspace_id', $workspace_id)
                ->where('del', '0')
                ->where(function ($q) {
                    $q->where('for_external', true)
                        ->orWhere('for_subcontractor', true);
                })
                ->get();
            
            // Documents are still stored with emp_id reference
            $data['emp_documents'] = DB::table('emp_documents')->where('emp_id', $validatedData['emp_id'])->where('del', '0')->get();
            $data['emp_emergency_contacts']  = EmpEmergencyContacts::where('emp_id', $validatedData['emp_id'])->get();
            $data['teams'] = EmpTeam::where('customer_id', $auth_id)->where('workspace_id', $workspace_id)->get();
            $data['employee_team_members'] = isset($emp_company_details->empTeamsMembers) ? $emp_company_details->empTeamsMembers : [];

            return $this->success($data, 'External employee Get successfully');
        }
    }

    public function externalEmployeeOnboardingsStore(Request $request)
    {
        if ($request->emp_id == 0) {
            $validator = Validator::make($request->all(), [
                'image' => 'required|',
            ], [
                'image.required' => 'The Profile Image field is required.',
            ]);
    
            if ($validator->fails()) {
                $error = $validator->errors()->first();
                return $this->error($error, 422);
            }
        }
    
        $validator = Validator::make($request->all(), [
            'emp_id' => 'required',
            'team_id' => 'nullable',
            'first_name' => 'required',
            'middle_name' => 'nullable',
            'last_name' => 'required',
            'image' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'temporary_student_visa' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'signature' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'medical_attach_files.*' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'employee_email' => ['required', 'email', Rule::unique('emp_company_details', 'employee_email')->ignore($request->emp_id, 'id'), Rule::unique('users', 'email')->ignore($request->emp_id, 'id'), Rule::unique('employees_subcontractors', 'email')->ignore($request->emp_id, 'id')],
            'date_of_birth' => 'required|date|before:' . Carbon::now()->subYears(15)->format('Y-m-d'),
            'mobile' => 'required|numeric',
            'suburb' => 'required|regex:/^[a-zA-Z\s]*$/',
            'state' => 'required|regex:/^[a-zA-Z\s]*$/',
            'streat_address' => 'required',
            'job_title' => 'nullable',
            'employer_name' => 'nullable',
            'worker_type' => 'nullable',
            'trade_qualified' => 'required',
            'trade_qualified_year' => 'required_if:trade_qualified,=,1',
            'trade_licensed' => 'required',
            'trade_licensed_year' => 'required_if:trade_licensed,=,1',
            'work_experience' => 'required',
            'year_commenced' => 'required',
            'legally_australia' => 'required',
            'citizenship_status' => 'required',
            'classified_high_risk' => 'nullable',
            'contact_name_first' => 'required',
            'contact_relation_first' => 'required',
            'contact_phone_first' => 'required',
            'allergies' => 'nullable',
            'details_allergies' => 'nullable',
            'previous_injuries' => 'nullable',
            'details_previous_injuries' => 'nullable',
            'medical_condition' => 'nullable'
        ], [
            'image.mimes' => 'The image must be a file of type: jpeg, png, jpg, gif.',
            'image.max' => 'The image may not be greater than 1MB.',
            'suburb.regex' => 'The suburb field must contain only alphabetic characters.',
            'state.regex' => 'The state field contain only alphabetic characters.',
            'date_of_birth.before' => 'You must be at least 15 years old to register.',
            'trade_qualified_year.required_if' => 'The trade qualified year field is required',
            'trade_licensed_year.required_if' => 'The trade licensed year field is required',
            'contact_name_first.required' => 'The contact name field is required',
            'contact_relation_first.required' => 'The contact relation field is required',
            'contact_phone_first.required' => 'The contact phone field is required',
            'bank_name.required' => 'The BSB field is required',
            'account_title.required' => 'The Account Name field is required',
            'bank_name.size' => 'The BSB field must be exactly 6 digits long',
            'employer_name.required' => 'The employer company name field is required',
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return $this->error($error, 422);
        } else {
    
            $link_detail = LinkManagement::where('secret_key', $request->key)->first();
    
            // Validate LinkManagement record exists
            if (!$link_detail) {
                return $this->error('Invalid or expired onboarding link', 422);
            }
    
            // Validate customer_id exists
            if (!$link_detail->customer_id) {
                return $this->error('Customer information not found for this onboarding link', 422);
            }
    
            // required documents code commented because it is not needed for external employees
    
            // $required_documents = RequiredDocument::with('requiredDocumentField')
            //     ->where('del', '0')
            //     ->where('customer_id', $link_detail->customer_id)
            //     ->where('workspace_id', $link_detail->workspace_id)
            //     ->where(function ($q) {
            //         $q->where('for_external', true)
            //             ->orWhere('for_subcontractor', true);
            //     })
            //     ->get();
            // $count = EmpDocuments::where('emp_id', $request->emp_id)->count();
    
            // $i = 0;
    
            // foreach ($required_documents as $required_document) {
            //     foreach ($required_document->requiredDocumentField as $required_document_field) {
            //         if ($required_document_field->field_required == "1" && !$request->filled('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id) && !$request->hasFile('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id)) {
            //             if ($count == 0) {
            //                 return $this->error($required_document_field->field_name . ' field is required', 422);
            //             } else {
            //                 if ($required_document_field->field_type != 'image') {
            //                     return $this->error($required_document_field->field_name . ' field is required', 422);
            //                 }
            //             }
            //         }
            //         if ($required_document_field->field_type == "image" && $request->hasFile('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id)) {
            //             $allowedExtensions = ['jpg', 'jpeg', 'png', 'docx', 'pdf'];
            //             $document_file = $request->file('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id);
            //             $extension = $document_file->getClientOriginalExtension();
            //             $mimeType = $document_file->getMimeType();
    
            //             // Validate extension and MIME type
            //             if (!in_array($extension, $allowedExtensions) || !in_array($mimeType, ['image/jpeg', 'image/png', 'application/pdf', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'])) {
            //                 return $this->error($required_document_field->field_name . 'file must be a file of type: jpg, jpeg, png, docx, pdf.', 422);
            //             }
            //         }
            //     }
            //     $i++;
            // }
    
            // STEP 1: Call companyDetailStore FIRST to create emp_id
            $msg = $this->companyDetailStore($request);
            
            // STEP 2: Determine if this is a new employee
            $isNewEmployee = str_contains($msg, 'Register');
            
            // STEP 3: Call storeEmployeeData with correct parameter
            $msg2 = $this->storeEmployeeData($request, $isNewEmployee);
            
            // STEP 4: Store other details
            $this->personalDetailsStore($request);
            $this->documentsStore($request);
            $this->emergencyContactsStore($request);
            $this->accessStore($request);
            
            // STEP 5: Handle team assignments
            $team_ids = explode(',', $request->team_id);
            if ($team_ids) {
                EmpTeamsMember::where([
                    'emp_id' => $request->emp_id,
                ])->delete();
                
                foreach ($team_ids as $team_id) {
                    EmpTeamsMember::insert([
                        'team_id' => $team_id,
                        'emp_id' => $request->emp_id,
                        'applied' => 1
                    ]);
                }
            }
    
            if ($isNewEmployee) {
                // Verify employee was created in employees_subcontractors table
                $employeeExists = EmployeeSubcontractor::where('id', $request->emp_id)->exists();
                
                if (!$employeeExists) {
                    Log::error('Employee not found in employees_subcontractors', [
                        'emp_id' => $request->emp_id,
                        'link_detail_id' => $link_detail->id
                    ]);
                    return $this->error('Failed to create employee record. Please try again.', 500);
                }
               
                // Create subcontractor association
                if ($link_detail->subcontractor_id) {
                    try {
                        EmpSubcontractor::updateOrCreate(
                            [
                                'emp_id' => $request->emp_id,
                                'subcontractor_id' => $link_detail->subcontractor_id
                            ],
                            [
                                'active' => 1,
                                'created_at' => now(),
                                'updated_at' => now()
                            ]
                        );
                    } catch (\Exception $e) {
                        Log::error('Failed to create subcontractor association', [
                            'emp_id' => $request->emp_id,
                            'subcontractor_id' => $link_detail->subcontractor_id,
                            'error' => $e->getMessage()
                        ]);
                        // Don't fail the entire process, just log the error
                        // The employee was created successfully
                    }
                }
            
                $this->assignShift($request->key, $request->emp_id);
                $this->sendExternalOnboardingEmail($request->emp_id, $link_detail->customer_id);
            }
    
            $data['emp_id'] = $request->emp_id;
            return $this->success($data, $msg);
        }
    }
    public function companyDetailStore($request)
{
    $validatedData = $request->only([
        'employee_email',
        'job_title',
        'employer_name',
        'worker_type',
        'trade_qualified',
        'trade_qualified_year',
        'trade_licensed',
        'trade_licensed_year',
        'work_experience',
        'year_commenced',
        'citizenship_status',
        'classified_high_risk',
        'allergies',
        'details_allergies',
        'previous_injuries',
        'details_previous_injuries',
        'medical_condition',
        'legally_australia'
    ]);

    $link_detail = LinkManagement::where('secret_key', $request->key)->first();
    
    // Check if employee exists and create/update WITHOUT files first
    if (EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->where('id', $request->emp_id)->count() == 0) {
        $validatedData['status'] = "1";
        $validatedData['credentials'] = "1";
        $validatedData['user_type'] = "1";
        $validatedData['created_by'] = "1";
        $validatedData['access_role'] = "ext-emp";
        $validatedData['tier_id'] = "3";
        $validatedData['force_reset'] = "0";
        $validatedData['two_factor'] = "1";
        $validatedData['compeleted'] = "1";
        $validatedData['approved'] = '1';
        $validatedData['approved_by'] = '1';
        $validatedData['customer_id'] = $link_detail->customer_id;
        $validatedData['workspace_id'] = $link_detail->workspace_id;

        // Create employee FIRST to get ID
        $request->emp_id = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->insertGetId($validatedData);
        $msg = 'Registered Successfully.Verify your email address.';
    } else {
        $msg = 'Updated Successfully.';
    }

    // NOW handle file uploads with valid emp_id
    $fileData = [];

    if ($request->hasFile('temporary_student_visa')) {
        $temporary_student_visa = $request->file('temporary_student_visa');
        $imageName = time() . '.' . $temporary_student_visa->getClientOriginalExtension();
        $temporary_student_visa->move(public_path('upload/temporary_student_visa'), $imageName);
        $fileData['temporary_student_visa'] = 'upload/temporary_student_visa/' . $imageName;
    }

    if ($request->hasFile('signature')) {
        $signature = $request->file('signature');
        $imageName = time() . '2.' . $signature->getClientOriginalExtension();
        $signature->move(public_path('upload/signature'), $imageName);
        $fileData['signature'] = 'upload/signature/' . $imageName;
    }

    // Handle medical attached files - Delete existing files first if new files are being uploaded
    if ($request->hasFile('medical_attach_files')) {
        // Get existing medical files from database
        $existingFiles = DB::table('medical_attach_files')
            ->where('emp_id', $request->emp_id)
            ->get();

        // Delete existing files from public folder and database
        foreach ($existingFiles as $existingFile) {
            $filePath = public_path($existingFile->file);
            if (file_exists($filePath)) {
                unlink($filePath);
            }
        }

        // Delete records from database
        DB::table('medical_attach_files')
            ->where('emp_id', $request->emp_id)
            ->delete();

        // Upload new medical files
        $medical_attach_files = $request->file('medical_attach_files');
        $i = 0;
        foreach ($medical_attach_files as $medical_attach_file) {
            $i++;
            $imageName = time() . $i . '.' . $medical_attach_file->getClientOriginalExtension();
            $medical_attach_file->move(public_path('upload/medical_attach_files'), $imageName);
            DB::table('medical_attach_files')->insert([
                'emp_id' => $request->emp_id,
                'file' => 'upload/medical_attach_files/' . $imageName
            ]);
        }
    }

    // Update employee with file data if any
    if (!empty($fileData)) {
        EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
            ->where('id', $request->emp_id)
            ->update($fileData);
    }

    return $msg;
}

    public function personalDetailsStore($request)
    {
        $validatedData = $request->only([
            'emp_id',
            'first_name',
            'middle_name',
            'last_name',
            'date_of_birth',
            'mobile',
            'suburb',
            'state',
            'streat_address'
        ]);

        // if ($request->hasFile('image')) {

        //     $image = $request->file('image');
        //     $imageName = time() . '.' . $image->getClientOriginalExtension();
        //     $image->move(public_path('upload/images'), $imageName);
        //     $validatedData['image'] = 'upload/images/' . $imageName;
        // }

        $validatedData['emp_id'] = $request->emp_id;
        if (EmpPersonalDetails::where('emp_id', $request->emp_id)->count() == 0) {
            $validatedData['created_by']   =  '0';
            EmpPersonalDetails::insertGetId($validatedData);
        } else {
            unset($validatedData['emp_id']);
            EmpPersonalDetails::where('emp_id', $request->emp_id)->update($validatedData);
        }
        return 0;
    }
    public function emergencyContactsStore($request)
    {
        EmpEmergencyContacts::where('emp_id', $request->emp_id)->delete();

        EmpEmergencyContacts::insert([
            'emp_id' => $request->emp_id,
            'name' => $request->contact_name_first,
            'relation' => $request->contact_relation_first,
            'phone' => $request->contact_phone_first,
            'created_by' => '0',
        ]);

        return 0;
    }
    /**
     * Store employee data directly in employees_subcontractors table
     */
    public function storeEmployeeData($request, $isNewEmployee = true)
{
    // Prepare data for employees_subcontractors table
    $employeeData = [
        'first_name' => $request->first_name,
        'middle_name' => $request->middle_name ?? null,
        'last_name' => $request->last_name,
        'email' => $request->employee_email,
        'phone' => $request->mobile,
        'address' => $request->streat_address,
        'state' => $request->state,
        'suburb' => $request->suburb,
        'dob' => $request->date_of_birth,
        'work_experience' => $request->work_experience,
        'trade_qualified' => $request->trade_qualified,
        'trade_qualified_year' => $request->trade_qualified_year ?? null,
        'trade_licensed' => $request->trade_licensed,
        'trade_licensed_year' => $request->trade_licensed_year ?? null,
        'year_commenced' => $request->year_commenced,
        'citizenship_status' => $request->citizenship_status,
        'allergies' => $request->details_allergies ?? $request->allergies ?? null,
        'previous_injuries' => $request->details_previous_injuries ?? $request->previous_injuries ?? null,
        'medical_condition' => $request->medical_condition ?? null,
    ];

    // Check if employee exists in employees_subcontractors table
    $existingEmployee = EmployeeSubcontractor::where('id', $request->emp_id)->first();
    
    if (!$existingEmployee) {
        // Employee doesn't exist in employees_subcontractors yet
        // Use DB::insert to manually set the ID
        try {
            $employeeData['id'] = $request->emp_id;
            $employeeData['created_at'] = now();
            $employeeData['updated_at'] = now();
            
            // Use raw insert with the specific ID
            $inserted = DB::table('employees_subcontractors')->insert($employeeData);
            
            if (!$inserted) {
                Log::error('Failed to insert employee into employees_subcontractors', [
                    'emp_id' => $request->emp_id,
                    'data' => $employeeData
                ]);
                throw new \Exception('Failed to insert employee record');
            }
            
            $msg = 'Registered Successfully!';
            $isUpdate = false;
        } catch (\Exception $e) {
            Log::error('Error creating employee in employees_subcontractors', [
                'emp_id' => $request->emp_id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    } else {
        // Employee exists, update it
        $msg = 'Updated Successfully.';
        $isUpdate = true;
    }

    // NOW handle file uploads with valid emp_id
    $fileData = [];

    // Handle visa attachment (only if not already handled by companyDetailStore)
    if ($request->hasFile('temporary_student_visa')) {
        $companyDetail = EmpCompanyDetails::find($request->emp_id);
        if (!$companyDetail || !$companyDetail->temporary_student_visa) {
            $temporary_student_visa = $request->file('temporary_student_visa');
            $imageName = time() . '_visa.' . $temporary_student_visa->getClientOriginalExtension();
            $temporary_student_visa->move(public_path('upload/temporary_student_visa'), $imageName);
            $fileData['visa_attachement'] = 'upload/temporary_student_visa/' . $imageName;
        }
    }

    // Handle profile image
    if ($request->hasFile('image')) {
        // Delete existing profile image if updating
        if ($isUpdate) {
            $existingEmployee = EmployeeSubcontractor::find($request->emp_id);
            if ($existingEmployee && $existingEmployee->profile_image) {
                $oldImagePath = public_path($existingEmployee->profile_image);
                if (file_exists($oldImagePath)) {
                    unlink($oldImagePath);
                }
            }
        }
        
        $image = $request->file('image');
        $imageName = time() . '_profile.' . $image->getClientOriginalExtension();
        $image->move(public_path('upload/images'), $imageName);
        $fileData['profile_image'] = 'upload/images/' . $imageName;
    }

    // Handle signature (only if not already handled by companyDetailStore)
    if ($request->hasFile('signature')) {
        $companyDetail = EmpCompanyDetails::find($request->emp_id);
        if (!$companyDetail || !$companyDetail->signature) {
            $signature = $request->file('signature');
            $imageName = time() . '_sig.' . $signature->getClientOriginalExtension();
            $signature->move(public_path('upload/signature'), $imageName);
            $fileData['signature'] = 'upload/signature/' . $imageName;
        }
    }

    // Medical files are already handled by companyDetailStore, so we'll get them from there
    $medicalFiles = DB::table('medical_attach_files')
        ->where('emp_id', $request->emp_id)
        ->pluck('file')
        ->toArray();
    
    if (!empty($medicalFiles)) {
        $fileData['medical_attachement_files'] = implode(',', $medicalFiles);
    }

    // Update employee with file data if any files were uploaded
    if (!empty($fileData)) {
        DB::table('employees_subcontractors')
            ->where('id', $request->emp_id)
            ->update($fileData);
    }

    return $msg;
}

    public function documentsStore($request)
    {
        $link_detail = LinkManagement::where('secret_key', $request->key)->first();

        // For external onboarding, get documents that are for external employees or subcontractors
        $required_documents = RequiredDocument::with('requiredDocumentField')
            ->where('del', '0')
            ->where('customer_id', $link_detail->customer_id)
            ->where('workspace_id', $link_detail->workspace_id)
            ->where(function ($q) {
                $q->where('for_external', true)
                    ->orWhere('for_subcontractor', true);
            })
            ->get();
        $count = EmpDocuments::where('emp_id', $request->emp_id)->count();

        // Ensure upload directory exists
        $uploadDirectory = public_path('upload/documents');
        if (!file_exists($uploadDirectory)) {
            mkdir($uploadDirectory, 0755, true);
        }

        $i = 0;

        foreach ($required_documents as $required_document) {
            foreach ($required_document->requiredDocumentField as $required_document_field) {

                $EmpDocumentsData = [
                    'emp_id' => $request->emp_id,
                    'required_document' => $required_document_field->doc_id,
                    'required_document_field' =>  $required_document_field->id,
                ];
                if ($required_document_field->field_type == "image" && $request->hasFile('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id)) {
                    $document_file = $request->file('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id);
                    
                    // Validate file object
                    if (!$document_file || !$document_file->isValid()) {
                        Log::error('Invalid file uploaded', [
                            'emp_id' => $request->emp_id,
                            'doc_id' => $required_document_field->doc_id,
                            'field_id' => $required_document_field->id
                        ]);
                        continue;
                    }
                    
                    $documentName = $i . time() . '.' . $document_file->getClientOriginalExtension();
                    
                    // Move file and check if successful
                    if (!$document_file->move($uploadDirectory, $documentName)) {
                        Log::error('Failed to move uploaded file', [
                            'emp_id' => $request->emp_id,
                            'doc_id' => $required_document_field->doc_id,
                            'field_id' => $required_document_field->id,
                            'temp_path' => $document_file->getRealPath(),
                            'destination' => $uploadDirectory . '/' . $documentName
                        ]);
                        continue;
                    }
                    
                    $EmpDocumentsData['value'] = 'upload/documents/' . $documentName;
                    $EmpDocumentsData['created_at'] = Carbon::now();
                    $EmpDocumentsData['updated_at'] = Carbon::now();
                    EmpDocuments::where(['emp_id' => $request->emp_id, 'required_document' => $required_document_field->doc_id, 'required_document_field' => $required_document_field->id])->delete();
                    EmpDocuments::insert($EmpDocumentsData);
                }

                if ($required_document_field->field_type != "image") {
                    if (isset($request->document_array[$required_document_field->doc_id][$required_document_field->id])) {
                        $EmpDocumentsData['value'] =  $request->document_array[$required_document_field->doc_id][$required_document_field->id];
                        $EmpDocumentsData['created_at'] = Carbon::now();
                        $EmpDocumentsData['updated_at'] = Carbon::now();
                        EmpDocuments::where(['emp_id' => $request->emp_id, 'required_document' => $required_document_field->doc_id, 'required_document_field' => $required_document_field->id])->delete();
                        EmpDocuments::insert($EmpDocumentsData);
                    }
                }
            }

            $i++;
        }


        return 0;
    }

    /**
     * Store multiple emergency contacts for external employee
     */
    public function storeEmergencyContacts($request)
    {
        // Delete existing emergency contacts for this employee
        EmpEmergencyContacts::where('emp_id', $request->emp_id)->where('del', '0')->delete();

        // Check if multiple contacts are provided as array
        if ($request->filled('emergency_contacts') && is_array($request->emergency_contacts)) {
            // Multiple contacts provided as array
            foreach ($request->emergency_contacts as $contact) {
                if (!empty($contact['name']) && !empty($contact['phone'])) {
                    EmpEmergencyContacts::insert([
                        'emp_id' => $request->emp_id,
                        'name' => $contact['name'],
                        'relation' => $contact['relation'] ?? 'Emergency Contact',
                        'phone' => $contact['phone'],
                        'email' => $contact['email'] ?? null,
                        'address' => $contact['address'] ?? null,
                        'created_by' => '0',
                        'del' => '0',
                    ]);
                }
            }
        } else {
            // Single contact provided (backward compatibility)
            if ($request->filled('contact_name_first') && $request->filled('contact_phone_first')) {
                EmpEmergencyContacts::insert([
                    'emp_id' => $request->emp_id,
                    'name' => $request->contact_name_first,
                    'relation' => $request->contact_relation_first ?? 'Emergency Contact',
                    'phone' => $request->contact_phone_first,
                    'email' => $request->contact_email_first ?? null,
                    'address' => $request->contact_address_first ?? null,
                    'created_by' => '0',
                    'del' => '0',
                ]);
            }
        }

        return 0;
    }

    public function accessStore(Request $request)
    {
        $validatedData = [
            'emp_id' => $request->emp_id,
            'web_app' => '0',
            'mobile_app' => '1',
            'attendance' => '1',
            'timesheet_reminders' => '1',
            'email' => '1',
            'timesheet' => '1'
        ];

        EmpAccess::where('emp_id', $request->emp_id)->delete();
        EmpAccess::insertGetId($validatedData);
    }

    public function deleteMedicalAttachment(Request $request)
    {
        DB::table('medical_attach_files')->where(['emp_id' => $request->emp_id, 'id' => $request->id])->delete();
        return 0;
    }

    public function sendOnExternalOnboarding($id)
    {
        $user = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->where('id', $id)->exists();

        if ($user) {

            $user_details = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->where('id', $id)->first();

            $digits = 6;
            $password =  rand(pow(10, $digits - 1), pow(10, $digits) - 1);
            EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->where('id', $id)->update([
                'password' => Hash::make($password)
            ]);

            $params = [
                'subject' => 'Employee Registration | ' . env("APP_NAME"),
                'to' => $user_details->employee_email,
                'blade' => view('Emails/external_employee_registration', ['email' => $user_details->employee_email, 'password' => $password, 'subject' => 'Employee Registration'])->render()
            ];

            $emailSent = $this->SendInstantEmail($params);

            if ($emailSent) {
                return 'Email has been sent, Please check your Email.';
            } else {
                return 'SMTP is not Working, Try Later!';
            }
        } else {
            return 'Account with the entered email does not exsit.';
        }
    }

    public function assignShift($key, $id)
    {
        $link_management =  DB::table('link_management')->where(['secret_key' => $key])->first();

        // Validate link_management exists
        if (!$link_management) {
            Log::error('assignShift: Link management not found', ['key' => $key, 'id' => $id]);
            throw new \Exception('Link management not found for the provided key');
        }

        $role = DB::table('roles')->where('id', $link_management->role_id)->first();

        // Validate role exists
        if (!$role) {
            Log::error('assignShift: Role not found', ['role_id' => $link_management->role_id, 'key' => $key, 'id' => $id]);
            throw new \Exception('Role not found for the provided role_id: ' . $link_management->role_id);
        }

        // Validate role has code property
        if (!isset($role->code) || $role->code === null) {
            Log::error('assignShift: Role code is missing', ['role_id' => $link_management->role_id, 'key' => $key, 'id' => $id]);
            throw new \Exception('Role code is missing for role_id: ' . $link_management->role_id);
        }

        EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->where('id', $id)->update(['access_role' => $role->code, 'tier_id' => $link_management->tier_id, 'link_key' => $key]);
        $days = [];
        // Validate working_days exists and is not empty
        if (!empty($link_management->working_days)) {
            $working_days = explode(',', $link_management->working_days);
            foreach ($working_days as $day) {
                // Convert to proper case (MONDAY -> Monday) before looking up in constants
                $proper_case_day = ucfirst(strtolower(trim($day)));
                $mapped_day = config('constants.weekdays_name.' . $proper_case_day);
                if ($mapped_day) {
                    $days[] = $mapped_day;
                }
            }
        }
        
        $arr = [
            'start_date' => Carbon::today()->format('Y-m-d'),
            'end_date' =>  Carbon::today()->addDays($link_management->number_of_days ?? 1)->format('Y-m-d'),
            'start_time' => $link_management->start_time ?? '09:00:00',
            'end_time' => $link_management->end_time ?? '17:00:00',
            'working_hours' => $link_management->required_hours ?? 8,
            'color_code' => $link_management->color_code ?? null,
            'break_minutes' => $link_management->paid_break ?? 0,
            'shift_notes' => $link_management->description ?? null,
            'users_ids' => [$id],
            'working_days_hidden' => implode(',', $days),
            'site_id' => $link_management->site_id ?? null
        ];

        return  $this->externalbulkScheduleRoasterCreate($arr, $arr, $key); //Trait
    }

    public function storeHistory($arr)
    {
        RosterHistory::create([
            'roster_template_id' => $arr['roster_template_id'],
            'updated_by' => '1',
            'description' => $arr['description']
        ]);
    }

    function required_external_data(Request $request)
    {
        $secritkey = $request->key;
        $user = User::where('secret_key', $secritkey)->first();
        $data = [];
        $data['name'] = $user->name;
        $data['company_name'] = $user->company_name;
        $data['email'] = $user->email;
        $data['mobile_number'] = $user->mobile_number;
        return $this->success($data, 'Get data successfully');
    }

    /**
     * Send external employee onboarding email with login credentials
     * 
     * @param int $empId
     * @return void
     */
    private function sendExternalOnboardingEmail($empId, $customer_id)
    {
        // Validate customer_id
        if (!$customer_id) {
            Log::error('sendExternalOnboardingEmail: customer_id is missing', ['empId' => $empId]);
            return;
        }

        // Get employee data
        $employeeCompany = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
            ->find($empId);
        $employee = EmpPersonalDetails::where('emp_id', $empId)->first();

        if (!$employeeCompany || !$employee) {
            Log::error('sendExternalOnboardingEmail: Employee data not found', ['empId' => $empId, 'customer_id' => $customer_id]);
            return;
        }



        $teamMember = EmpTeamsMember::where('emp_id', $empId)->first();
        $teamName = 'N/A';
        
        if ($teamMember && isset($teamMember->team_id)) {
            $team = EmpTeam::where('id', $teamMember->team_id)->first();
            $teamName = $team->title ?? 'N/A';
        }

        // Generate random password
        $password = $this->generateRandomPassword();

        // Hash and save password to employee
        $employeeCompany->password = Hash::make($password);
        $employeeCompany->save();

        // Get worker type
        $workerType = EmpWorkerType::find($employeeCompany->worker_type);

        // Prepare login URL
        $loginUrl = env('FRONTEND_URL', url('/'));

        // Get roster assignments and schedule details for this employee
        $rosterAssignments = RosterAssign::where('assign_to', $empId)
            ->with('rosterTemplate')
            ->orderBy('schedule_date', 'asc')
            ->get();

        $rosterDetails = [
            'individual_shifts' => [],
            'bulk_schedule' => null
        ];

        // Get link management details to show bulk schedule info
        $employeeDetails = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)->find($empId);
        $linkManagement = null;
        if ($employeeDetails && $employeeDetails->link_key) {
            $linkManagement = LinkManagement::where('secret_key', $employeeDetails->link_key)->first();
        }

        // If we have link management, show bulk schedule details
        if ($linkManagement) {
            $working_days = explode(',', $linkManagement->working_days);
            $working_day_names = [];
            foreach ($working_days as $day) {
                // Convert to proper case (MONDAY -> Monday) before looking up in constants
                $proper_case_day = ucfirst(strtolower(trim($day)));
                $mapped_day = config('constants.weekdays_name.' . $proper_case_day, $proper_case_day);
                $working_day_names[] = $mapped_day;
            }

            $startDate = Carbon::today();
            $endDate = Carbon::today()->addDays($linkManagement->number_of_days);

            $rosterDetails['bulk_schedule'] = [
                'schedule_period_start' => $startDate->format('d-m-Y'),
                'schedule_period_end' => $endDate->format('d-m-Y'),
                'total_days' => $linkManagement->number_of_days,
                'working_days' => implode(', ', $working_day_names),
                'daily_start_time' => date('h:i A', strtotime($linkManagement->start_time)),
                'daily_end_time' => date('h:i A', strtotime($linkManagement->end_time)),
                'daily_break_minutes' => $linkManagement->paid_break ?? 0,
                'shift_notes' => $linkManagement->description ?? '',
                'total_working_days' => count($working_days)
            ];
        }

        // Get some example individual shifts (next 7 days)
        $exampleShifts = $rosterAssignments->take(7);
        foreach ($exampleShifts as $assignment) {
            if ($assignment->rosterTemplate) {
                $rosterDetails['individual_shifts'][] = [
                    'schedule_date' => date('d-m-Y', strtotime($assignment->schedule_date)),
                    'day_name' => date('l', strtotime($assignment->schedule_date)),
                    'start_time' => date('h:i A', strtotime($assignment->rosterTemplate->start_time)),
                    'end_time' => date('h:i A', strtotime($assignment->rosterTemplate->end_time)),
                    'break_minutes' => $assignment->rosterTemplate->break_minutes ?? 0,
                    'shift_notes' => $assignment->rosterTemplate->shift_notes ?? '',
                    'working_hours' => $assignment->rosterTemplate->working_hours ?? 0
                ];
            }
        }

        // Send email with login credentials and roster details
        $emailParams = [
            'to' => $employeeCompany->employee_email,
            'subject' => 'Welcome to ' . env('APP_NAME', 'WMS') . ' - Your Login Credentials & Schedule',
            'msg' => view('Emails.external_employee_onboarding', [
                'subject' => 'Welcome to ' . env('APP_NAME', 'WMS') . ' - Your Login Credentials & Schedule',
                'employee' => $employee,
                'employeeCompany' => $employeeCompany,
                'workerType' => $workerType,
                'password' => $password,
                'login_url' => $loginUrl,
                'rosterDetails' => $rosterDetails,
                'teamName' => $teamName,
                'customer_id' => $customer_id
            ])->render()
        ];

        $this->SendInstantEmail($emailParams);
    }

    /**
     * Generate random password
     * 
     * @return string
     */
    private function generateRandomPassword()
    {
        $length = 8;
        $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $password = '';

        for ($i = 0; $i < $length; $i++) {
            $password .= $characters[rand(0, strlen($characters) - 1)];
        }

        return $password;
    }
    
}
