<?php

namespace App\Http\Controllers\Mobilev2;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Meeting;
use App\Models\MeetingUser;
use App\Models\MeetingDocument;
use App\Models\MeetingDocumentNote;
use App\Models\MeetingNoteSignature;
use App\Models\EmployeeSignedDocument;
use App\Models\MeetingAttendance;
use App\Models\EmployeeAttendance;
use App\Models\Sites;
use App\Models\EmployeeSubcontractor;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\EmpCompanyDetails;
use App\Models\EmpPersonalDetails;
use App\General\GeoLoacation;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class MeetingController extends Controller
{
    public $current_date;

    public function __construct()
    {
        $this->current_date = date('Y-m-d');
    }

    /**
     * Get employee meetings for subcontractor employees
     * Similar to regular employee meetings but handles subcontractor employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getEmployeeMeetings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'site_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $validatedData = $validator->validated();

        // Get employee's active invitation to get customer_id, workspace_id, and subcontractor_id
        $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
            ->where('invitation_status', 'accepted')
            ->where('status', 1)
            ->first();

        if (!$invitation) {
            return $this->error('No active invitation found for this employee.', 404);
        }

        // Verify site belongs to customer and workspace
        $site = Sites::where('id', $validatedData['site_id'])
            ->where('customer_id', $invitation->customer_id)
            ->where('workspace_id', $invitation->workspace_id)
            ->where('active', 1)
            ->where('del', 0)
            ->first();

        if (!$site) {
            return $this->error('Site not found or unauthorized.', 404);
        }

        // Fetch meetings for the employee and site
        $data = $this->getAllSubcontractorEmployeeMeetings($employee->id, $validatedData['site_id']);

        if (count($data) == 0) {
            return $this->success($data, 'There is no meeting for today.');
        }

        return $this->success($data, 'Employee meetings fetched successfully.');
    }

    /**
     * Get employee meetings before check-in for subcontractor employees
     * Similar to regular employee but handles subcontractor employees with geofencing
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getEmployeeMeetingBeforeCheckIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'longitude' => 'required|numeric',
            'latitude' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        // Get employee's active invitation to get customer_id, workspace_id, and subcontractor_id
        $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
            ->where('invitation_status', 'accepted')
            ->where('status', 1)
            ->first();

        if (!$invitation) {
            return $this->error('No active invitation found for this employee.', 404);
        }

        $customerId = $invitation->customer_id;
        $workspaceId = $invitation->workspace_id;
        $subcontractorId = $invitation->subcontractor_id;

        // Validate location (geofencing)
        $point = [
            'longitude' => (float) $request->longitude,
            'latitude' => (float) $request->latitude,
        ];

        $sites = Sites::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('active', 1)
            ->where('del', 0)
            ->get();

        $foundSite = null;
        foreach ($sites as $site) {
            $center = [
                'longitude' => (float) $site->longitude,
                'latitude' => (float) $site->latitude,
            ];
            $radius = (float) $site->area_radius;
            if (GeoLoacation::isPointWithinRadius($point, $center, $radius)) {
                $foundSite = $site;
                break;
            }
        }

        if (!$foundSite) {
            $message = 'The provided location is outside the permitted radius of any site.';
            return $this->error($message, 422);
        }

        // // Check if employee has an active attendance record and verify site match
        // $attendance = EmployeeAttendance::where('employee_id', $employee->id)
        //     ->where('subcontractor_id', $subcontractorId)
        //     ->whereNull('check_out')
        //     ->latest()
        //     ->first();

        // if ($attendance && $attendance->site_id != $foundSite->id) {
        //     $attendanceSite = Sites::find($attendance->site_id);
        //     $attendanceSiteName = $attendanceSite ? $attendanceSite->title : 'Unknown Site';
        //     $message = 'You are trying to get meetings from the site: ' . $foundSite->title . '. Your assigned site is: ' . $attendanceSiteName . '.';
        //     return $this->error($message, 422);
        // }

        // Get meetings for this site (is_before_meeting = 1)
        $meetings = $this->getAllSubcontractorEmployeeMeetings($employee->id, $foundSite->id, "1");

        if (count($meetings) > 0) {
            $message = 'Employee meetings retrieved successfully.';
            return $this->success($meetings, $message);
        }

        $message = "There are no meetings for today.";
        return $this->success($meetings, $message);
    }

    /**
     * Get all employee meetings for subcontractor employees
     * Similar to getAllEmployeeMeetings but handles subcontractor employees
     * 
     * @param int $employee_id Subcontractor employee ID
     * @param int $site_id Site ID
     * @param string $is_before_meeting "1" for before meeting, "" for regular meetings
     * @return array
     */
    private function getAllSubcontractorEmployeeMeetings($employee_id, $site_id, $is_before_meeting = "")
    {
        $currentTime = now()->format('H:i:s');
        $dayNumber = Carbon::parse($this->current_date)->format('N');

        $data = [];
        $allowed_clock_in = 1;

        // Get meetings for the site
        $meetings = Meeting::with('today_meeting_documents', 'today_meeting_document_notes', 'meeting_users')
            ->where('meetings.site_id', $site_id)
            ->where('meetings.is_deleted', 0)
            ->where('meetings.start_date', "<=", $this->current_date)
            ->where('meetings.end_date', ">=", $this->current_date)
            ->where('meetings.start_time', "<=", $currentTime)
            ->where('meetings.end_time', ">=", $currentTime)
            ->where('meetings.is_before_meeting', $is_before_meeting)
            ->select('meetings.*')
            ->get();
        foreach ($meetings as $key => $meeting) {
            $meeting_users = $meeting->meeting_users->toArray();
            unset($meetings[$key]['meeting_users']);

            // Check if this meeting is for specific members only
            if ($meeting->allow_meeting_members == 1) {
                // This meeting is for specific members only
                // Filter meeting_users to get employee_ids where user_type matches
                $employee_ids = [];
                foreach ($meeting_users as $mu) {
                    if (isset($mu['employee_id'])) {
                        $employee_ids[] = $mu['employee_id'];
                    }
                }

                // Check if the current subcontractor employee is in the meeting members list
                // Need to check both user_type = 0 (regular) and user_type = 1 (subcontractor)
                $isEmployeeInMeeting = false;
                foreach ($meeting_users as $mu) {
                    if (isset($mu['employee_id']) && $mu['employee_id'] == $employee_id) {
                        // Check if it's a subcontractor employee (user_type = 1)
                        if (isset($mu['user_type']) && $mu['user_type'] == 1) {
                            $isEmployeeInMeeting = true;
                            break;
                        }
                    }
                }

                if (!$isEmployeeInMeeting) {
                    // Employee is not in the specific members list, skip this meeting
                    continue;
                }

                // Get meeting employees for specific members (both regular and subcontractor)
                $meeting_employees = [];
                
                // Get regular employees (user_type = 0)
                $regularEmployeeIds = [];
                foreach ($meeting_users as $mu) {
                    if (isset($mu['user_type']) && $mu['user_type'] == 0 && isset($mu['employee_id'])) {
                        $regularEmployeeIds[] = $mu['employee_id'];
                    }
                }
                
                if (!empty($regularEmployeeIds)) {
                    $regularEmployees = DB::table('emp_company_details')
                        ->join('emp_personal_details', 'emp_company_details.id', '=', 'emp_personal_details.emp_id')
                        ->where('emp_company_details.status', 1)
                        ->where('emp_company_details.approved_by', 1)
                        ->where('emp_company_details.del', '!=', 1)
                        ->where(function ($query) {
                            $query->where('emp_company_details.invited', 1)
                                ->orWhere(function ($q) {
                                    $q->where('emp_company_details.invited', 0)
                                        ->where('emp_company_details.approved', 1)
                                        ->where('emp_company_details.compeleted', 1)
                                        ->where('emp_company_details.status', 1);
                                });
                        })
                        ->whereIn('emp_company_details.id', $regularEmployeeIds)
                        ->select([
                            'emp_personal_details.id',
                            'emp_personal_details.emp_id',
                            'emp_personal_details.first_name',
                            'emp_personal_details.middle_name',
                            'emp_personal_details.last_name',
                            'emp_personal_details.image'
                        ])
                        ->get()
                        ->map(function ($emp) {
                            return [
                                'id' => $emp->id,
                                'emp_id' => (string)$emp->emp_id,
                                'first_name' => $emp->first_name,
                                'middle_name' => $emp->middle_name,
                                'last_name' => $emp->last_name,
                                'image' => $emp->image,
                                'employee_type' => 'regular',
                            ];
                        });
                    $meeting_employees = array_merge($meeting_employees, $regularEmployees->toArray());
                }

                // Get subcontractor employees (user_type = 1)
                $subcontractorEmployeeIds = [];
                foreach ($meeting_users as $mu) {
                    if (isset($mu['user_type']) && $mu['user_type'] == 1 && isset($mu['employee_id'])) {
                        $subcontractorEmployeeIds[] = $mu['employee_id'];
                    }
                }
                
                if (!empty($subcontractorEmployeeIds)) {
                    $subcontractorEmployees = EmployeeSubcontractor::whereIn('id', $subcontractorEmployeeIds)
                        ->select('id', 'first_name', 'middle_name', 'last_name', 'email', 'profile_image')
                        ->get()
                        ->map(function ($emp) {
                            return [
                                'id' => $emp->id,
                                'emp_id' => (string)$emp->id,
                                'first_name' => $emp->first_name,
                                'middle_name' => $emp->middle_name,
                                'last_name' => $emp->last_name,
                                'image' => $emp->profile_image,
                                'email' => $emp->email,
                                'employee_type' => 'subcontractor_employee',
                            ];
                        });
                    $meeting_employees = array_merge($meeting_employees, $subcontractorEmployees->toArray());
                }

                $meetings[$key]['meeting_employees'] = $meeting_employees;
            } else {
                // Meeting is for all employees - get all regular employees for the site
                $meetings[$key]['meeting_employees'] = DB::table('emp_company_details')
                    ->join('emp_personal_details', 'emp_company_details.id', '=', 'emp_personal_details.emp_id')
                    ->where('emp_company_details.status', 1)
                    ->where('emp_company_details.approved_by', 1)
                    ->where('emp_company_details.del', '!=', 1)
                    ->where(function ($query) {
                        $query->where('emp_company_details.invited', 1)
                            ->orWhere(function ($q) {
                                $q->where('emp_company_details.invited', 0)
                                    ->where('emp_company_details.approved', 1)
                                    ->where('emp_company_details.compeleted', 1)
                                    ->where('emp_company_details.status', 1);
                            });
                    })
                    ->select([
                        'emp_personal_details.id',
                        'emp_personal_details.emp_id',
                        'emp_personal_details.first_name',
                        'emp_personal_details.middle_name',
                        'emp_personal_details.last_name',
                        'emp_personal_details.image'
                    ])
                    ->get()
                    ->map(function ($emp) {
                        return [
                            'id' => $emp->id,
                            'emp_id' => (string)$emp->emp_id,
                            'first_name' => $emp->first_name,
                            'middle_name' => $emp->middle_name,
                            'last_name' => $emp->last_name,
                            'image' => $emp->image,
                            'employee_type' => 'regular',
                        ];
                    })
                    ->toArray();
            }

            // Loop through the meeting documents and check if any document is signed
            foreach ($meeting->today_meeting_documents as $doc_key => $value) {
                // Check if employee has signed this document today
                $is_document_signed_today = EmployeeSignedDocument::where('meeting_id', $value->meeting_id)
                    ->where('employee_id', $employee_id)
                    ->where('document_id', $value->id)
                    ->where('signed', 1)
                    ->where('date', $this->current_date)
                    ->exists();

                // Set is_signed based on signature status for today
                // is_signed = 1 means NOT signed (employee needs to sign)
                // is_signed = 0 means signed (employee has already signed)
                if ($is_document_signed_today) {
                    $meetings[$key]['today_meeting_documents'][$doc_key]['is_signed'] = 0;
                }

                // Check if document requires signature
                if ($value->is_signed) {
                    // For daily documents, check if signed today
                    // For specific date documents, check if signed on that date
                    if ($value->is_daily == 1) {
                        // Daily document - must be signed every day
                        if (!$is_document_signed_today) {
                            $allowed_clock_in = 0;
                        }
                    } else {
                        // Specific date document - check if signed on the specific date
                        $is_sign_document = EmployeeSignedDocument::where('meeting_id', $value->meeting_id)
                            ->where('employee_id', $employee_id)
                            ->where('document_id', $value->id)
                            ->where('signed', 1)
                            ->where('date', $value->meeting_date ?? $this->current_date)
                            ->exists();

                        if (!$is_sign_document) {
                            $allowed_clock_in = 0;
                        }
                    }
                }
            }

            foreach ($meeting->today_meeting_documents as $value) {
                if (!is_null($value->meeting_note_documents_pdf)) {
                    $meeting->today_meeting_documents[] = $value->meeting_note_documents_pdf;
                }
            }

            foreach ($meeting->today_meeting_document_notes as $doc_key => $value) {
                // Check if employee has signed this meeting document note today
                $is_note_signed_today = DB::table('meeting_note_signatures')
                    ->where('employee_note_id', $value->id)
                    ->where('employee_id', $employee_id)
                    ->whereNotNull('signature')
                    ->where('date', $this->current_date)
                    ->exists();

                // Set is_signed based on signature status for today
                // is_signed = 1 means NOT signed (employee needs to sign)
                // is_signed = 0 means signed (employee has already signed)
                $is_signed_value = $is_note_signed_today ? 0 : 1;
                $meetings[$key]['today_meeting_document_notes'][$doc_key]['is_signed'] = $is_signed_value;

                // Also set is_signed for the related PDF document if it exists
                if (isset($meetings[$key]['today_meeting_document_notes'][$doc_key]['meeting_note_documents_pdf'])) {
                    $meetings[$key]['today_meeting_document_notes'][$doc_key]['meeting_note_documents_pdf']['is_signed'] = $is_signed_value;
                }

                // If document note is not signed today, don't allow clock in
                if (!$is_note_signed_today) {
                    $allowed_clock_in = 0;
                }
            }

            // Process meeting document notes PDFs
            $meeting_document_notes_with_pdfs = [];
            foreach ($meeting->today_meeting_document_notes as $value) {
                if (!is_null($value->meeting_note_documents_pdf)) {
                    // Check signature status for this document note today
                    $is_note_signed_today = DB::table('meeting_note_signatures')
                        ->where('employee_note_id', $value->id)
                        ->where('employee_id', $employee_id)
                        ->whereNotNull('signature')
                        ->where('date', $this->current_date)
                        ->exists();

                    $pdf_document = $value->meeting_note_documents_pdf->toArray();
                    $pdf_document['is_signed'] = $is_note_signed_today ? 0 : 1;

                    // Transform the document structure: move meeting_note_id to id and remove meeting_note_id
                    if (isset($pdf_document['meeting_note_id'])) {
                        $pdf_document['id'] = $pdf_document['meeting_note_id'];
                        unset($pdf_document['meeting_note_id']);
                    }

                    // Add base URL to document path
                    if (isset($pdf_document['document_path']) && !empty($pdf_document['document_path'])) {
                        $pdf_document['document_path'] = url($pdf_document['document_path']);
                    }

                    $meeting_document_notes_with_pdfs[] = $pdf_document;
                }
            }

            // Add base URL to today_meeting_documents paths
            $today_meeting_documents = $meeting->today_meeting_documents->toArray();
            foreach ($today_meeting_documents as &$document) {
                if (isset($document['document_path']) && !empty($document['document_path'])) {
                    $document['document_path'] = url($document['document_path']);
                }
            }

            // Add the PDFs to the meeting data
            $meetings[$key]['meeting_documents'] = array_merge($today_meeting_documents, $meeting_document_notes_with_pdfs);

            // Remove the original fields to avoid duplication
            unset($meetings[$key]['today_meeting_documents']);
            unset($meetings[$key]['today_meeting_document_notes']);

            // Check if employee has joined the meeting
            $employee_attendance = MeetingAttendance::where('meeting_id', $meeting->id)
                ->where('employee_id', $employee_id)
                ->where('date', $this->current_date)
                ->exists();

            if ($employee_attendance) {
                $meetings[$key]['is_joined'] = 1;
            } else {
                $meetings[$key]['is_joined'] = 0;
            }

            $meetings[$key]['allowed_clock_in'] = $allowed_clock_in;

            // Add meeting to data array
            $data[] = $meeting;
        }

        return $data;
    }

    /**
     * Add meeting attendance (join/leave) for subcontractor employees
     * Similar to regular employee but handles subcontractor employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addMeetingAttendance(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'meeting_id' => 'required|exists:meetings,id',
            'attendance_type' => 'required|in:in,out'
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $validatedData = $validator->validated();

        // Get employee's active invitation to get customer_id, workspace_id, and subcontractor_id
        $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
            ->where('invitation_status', 'accepted')
            ->where('status', 1)
            ->first();

        if (!$invitation) {
            return $this->error('No active invitation found for this employee.', 404);
        }

        $customerId = $invitation->customer_id;
        $workspaceId = $invitation->workspace_id;

        // Verify meeting exists and belongs to customer/workspace
        $meeting = Meeting::where('id', $validatedData['meeting_id'])
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('is_deleted', 0)
            ->first();

        if (!$meeting) {
            return $this->error('Meeting not found or unauthorized.', 404);
        }

        // Verify employee is part of this meeting
        $isEmployeeInMeeting = MeetingUser::where('meeting_id', $validatedData['meeting_id'])
            ->where('employee_id', $employee->id)
            ->where('user_type', 1) // Subcontractor employee
            ->exists();

        // If meeting is for specific members, check if employee is in the list
        if ($meeting->allow_meeting_members == 1 && !$isEmployeeInMeeting) {
            return $this->error('You are not authorized to attend this meeting.', 403);
        }

        // Handle check-in
        if ($validatedData['attendance_type'] == "in") {
            // Check if already checked in today
            $existingAttendance = MeetingAttendance::where('meeting_id', $validatedData['meeting_id'])
                ->where('employee_id', $employee->id)
                ->where('date', $this->current_date)
                ->first();

            if ($existingAttendance) {
                return $this->success($existingAttendance, 'You have already punched in for the meeting.');
            }

            // Create new attendance record
            $attendance = new MeetingAttendance();
            $attendance->meeting_id = $validatedData['meeting_id'];
            $attendance->employee_id = $employee->id;
            $attendance->date = $this->current_date;
            $attendance->check_in = now()->format('H:i:s');
            $attendance->save();

            $responseData = $attendance->toArray();
            
            // Format check_in time
            if (isset($responseData['check_in']) && $responseData['check_in']) {
                $rawCheckIn = $attendance->getRawOriginal('check_in');
                $responseData['check_in'] = $rawCheckIn ? Carbon::parse($rawCheckIn)->format('H:i:s') : $responseData['check_in'];
            }

            return $this->success($responseData, 'You punch-in the meeting successfully.');
        }

        // Handle check-out
        $attendance = MeetingAttendance::where('meeting_id', $validatedData['meeting_id'])
            ->where('employee_id', $employee->id)
            ->where('date', $this->current_date)
            ->first();

        if (!$attendance) {
            return $this->error('Please punch in before punching out of the meeting.', 422);
        }

        if ($attendance->check_out) {
            return $this->success($attendance, 'You have already punched out of the meeting.');
        }

        $attendance->check_out = now()->format('H:i:s');
        $attendance->save();

        // Format response data
        $responseData = $attendance->toArray();
        if (isset($responseData['check_in']) && $responseData['check_in']) {
            $responseData['check_in'] = Carbon::parse($responseData['check_in'])->format('H:i:s');
        }
        if (isset($responseData['check_out']) && $responseData['check_out']) {
            $responseData['check_out'] = Carbon::parse($responseData['check_out'])->format('H:i:s');
        }

        return $this->success($responseData, 'You punch-out the meeting successfully.');
    }

    /**
     * Sign meeting document for subcontractor employees
     * Similar to regular employee but handles subcontractor employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function signDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'document_id' => 'required',
            'meeting_id' => 'required|exists:meetings,id',
            'signature' => 'required|file|mimes:jpeg,png,jpg,pdf|max:7168',
            'type_of_setting' => 'nullable' // For custom note documents
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        $validatedData = $validator->validated();

        // Get employee's active invitation to get customer_id, workspace_id, and subcontractor_id
        $invitation = SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
            ->where('invitation_status', 'accepted')
            ->where('status', 1)
            ->first();

        if (!$invitation) {
            return $this->error('No active invitation found for this employee.', 404);
        }

        $customerId = $invitation->customer_id;
        $workspaceId = $invitation->workspace_id;

        // Verify meeting exists and belongs to customer/workspace
        $meeting = Meeting::where('id', $validatedData['meeting_id'])
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('is_deleted', 0)
            ->first();

        if (!$meeting) {
            return $this->error('Meeting not found or unauthorized.', 404);
        }

        // Verify employee is part of this meeting
        $isEmployeeInMeeting = MeetingUser::where('meeting_id', $validatedData['meeting_id'])
            ->where('employee_id', $employee->id)
            ->where('user_type', 1) // Subcontractor employee
            ->exists();

        // If meeting is for specific members, check if employee is in the list
        if ($meeting->allow_meeting_members == 1 && !$isEmployeeInMeeting) {
            return $this->error('You are not authorized to sign documents for this meeting.', 403);
        }

        $data = [];

        // Check if it's a custom (note-based) document
        $isCustomNoteDocument = $request->filled('type_of_setting');

        if ($isCustomNoteDocument) {
            // Save in meeting_note_signatures
            $existing = MeetingNoteSignature::where('employee_note_id', $validatedData['document_id'])
                ->where('employee_id', $employee->id)
                ->first();

            $insertData = [
                'meeting_id' => $validatedData['meeting_id'],
                'employee_note_id' => $validatedData['document_id'],
                'employee_id' => $employee->id,
                'date' => $this->current_date,
                'updated_at' => now()
            ];

            // Handle file upload
            if ($request->hasFile('signature')) {
                $uploadResult = $this->handleFileImageUpload($request, 'MeetingDocuments');
                // handleFileImageUpload returns single array if one file, or array of arrays if multiple
                $uploadPath = null;
                if (is_array($uploadResult)) {
                    // Check if it's a single file result (has 'path' key directly)
                    if (isset($uploadResult['path'])) {
                        $uploadPath = $uploadResult['path'];
                    } elseif (!empty($uploadResult) && is_array($uploadResult[0]) && isset($uploadResult[0]['path'])) {
                        // Multiple files - get first one
                        $uploadPath = $uploadResult[0]['path'];
                    }
                }
                
                if (!$uploadPath) {
                    return $this->error('Failed to upload document.', 422);
                }
                $insertData['signature'] = $uploadPath;
            } else {
                return $this->error('Please upload the signed custom document.', 422);
            }

            if ($existing) {
                // Update existing record
                $existing->update($insertData);
            } else {
                // Create new record
                $insertData['created_at'] = now();
                MeetingNoteSignature::create($insertData);
            }

        } else {
            // Save in employee_signed_documents
            $meeting_document = MeetingDocument::where('id', $validatedData['document_id'])
                ->where('meeting_id', $validatedData['meeting_id'])
                ->first();

            if (!$meeting_document) {
                return $this->error('No record found for this document.', 404);
            }

            // Check if a signed document record already exists
            $signed_document = EmployeeSignedDocument::where('meeting_id', $validatedData['meeting_id'])
                ->where('employee_id', $employee->id)
                ->where('date', $this->current_date)
                ->where('document_id', $validatedData['document_id'])
                ->first();

            // Handle file upload
            if (!$request->hasFile('signature')) {
                return $this->error('Please upload the signed document.', 422);
            }

            $uploadResult = $this->handleFileImageUpload($request, 'MeetingDocuments');
            // handleFileImageUpload returns single array if one file, or array of arrays if multiple
            $uploadPath = null;
            if (is_array($uploadResult)) {
                // Check if it's a single file result (has 'path' key directly)
                if (isset($uploadResult['path'])) {
                    $uploadPath = $uploadResult['path'];
                } elseif (!empty($uploadResult) && is_array($uploadResult[0]) && isset($uploadResult[0]['path'])) {
                    // Multiple files - get first one
                    $uploadPath = $uploadResult[0]['path'];
                }
            }
            
            if (!$uploadPath) {
                return $this->error('Failed to upload document.', 422);
            }

            if ($signed_document) {
                // Update existing record
                $signed_document->signed = 1;
                $signed_document->date = $this->current_date;
                $signed_document->document = $uploadPath;
                $signed_document->save();
            } else {
                // Create new record
                $signed_document = new EmployeeSignedDocument();
                $signed_document->meeting_id = $validatedData['meeting_id'];
                $signed_document->employee_id = $employee->id;
                $signed_document->document_id = $validatedData['document_id'];
                $signed_document->signed = 1;
                $signed_document->date = $this->current_date;
                $signed_document->document = $uploadPath;
                $signed_document->save();
            }
        }

        // Get remaining unsigned documents
        $remaining_documents = $this->getUnsignedDocuments($validatedData['meeting_id'], $employee->id);
        $data['documents'] = $remaining_documents;
        $data['allowed_clock_in'] = count($remaining_documents) === 0 ? 1 : 0;

        return $this->success($data, 'Document signed successfully.');
    }

    /**
     * Get unsigned documents for a meeting and employee
     * Similar to unSignedDocuments but for subcontractor employees
     * 
     * @param int $meeting_id
     * @param int $employee_id Subcontractor employee ID
     * @return \Illuminate\Support\Collection
     */
    private function getUnsignedDocuments($meeting_id, $employee_id)
    {
        // Get all meeting documents that require signature
        $allDocuments = MeetingDocument::where('meeting_id', $meeting_id)
            ->where('is_signed', 1)
            ->get();

        // Get all signed documents for this employee today
        $signedDocuments = EmployeeSignedDocument::where('meeting_id', $meeting_id)
            ->where('employee_id', $employee_id)
            ->where('date', $this->current_date)
            ->where('signed', 1)
            ->pluck('document_id')
            ->toArray();

        // Filter out signed documents
        $unsignedDocuments = $allDocuments->filter(function ($document) use ($signedDocuments) {
            return !in_array($document->id, $signedDocuments);
        });

        return $unsignedDocuments->values();
    }
}
