<?php

namespace App\Http\Controllers\MobileV2;

use App\Http\Controllers\Controller;
use App\Models\EmpCompanyDetails;
use App\Models\EmployeeSubcontractor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class MobileAuthController extends Controller
{
    /**
     * Mobile API V2 Login
     * 
     * Handles login for both regular employees (EmpCompanyDetails) 
     * and subcontractor employees (EmployeeSubcontractor)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed', 422);
        }

        $email = $request->email;
        $password = $request->password;
        $masterPassword = env('MASTER_LOGIN_PASSWORD', 'gtechmaster@1122!');

        // Try to find user in EmpCompanyDetails (regular employees)
        $regularEmployee = EmpCompanyDetails::where('employee_email', $email)->first();
        
        // Try to find user in EmployeeSubcontractor (subcontractor employees)
        $subcontractorEmployee = EmployeeSubcontractor::where('email', $email)->first();

        $user = null;
        $userType = null;
        $scope = null;
        $guard = null;

        // Check regular employee first
        if ($regularEmployee) {
            // Check if account is active and not deleted
            if ($regularEmployee->status == '0') {
                return $this->error('The account is currently inactive', 400);
            }

            if ($regularEmployee->del == '1') {
                return $this->error('The account has been deleted', 400);
            }

            // Verify password or master password
            if (Hash::check($password, $regularEmployee->password) || $password === $masterPassword) {
                $user = $regularEmployee;
                $userType = 'employee';
                $scope = ['api'];
                $guard = 'api';
            }
        }

        // If not found in regular employees, check subcontractor employees
        if (!$user && $subcontractorEmployee) {
            // Verify password or master password
            if (Hash::check($password, $subcontractorEmployee->password) || $password === $masterPassword) {
                $user = $subcontractorEmployee;
                $userType = 'subcontractor_employee';
                $scope = ['subcontractor_employee'];
                $guard = 'subcontractor_employee';
            }
        }

        // If no user found or password incorrect
        if (!$user) {
            return $this->error('Invalid email or password', 401);
        }

        // Delete existing tokens if SINGLE_SESSION is enabled
        if (env('SINGLE_SESSION', true)) {
            $user->tokens()->delete();
        }

        // Get token expiration time from env or use default of 30 days
        $tokenExpiration = env('TOKEN_EXPIRATION_DAYS', 30);

        // Create token with appropriate scope
        $tokenResult = $user->createToken('Mobile API V2 Token', $scope, now()->addDays($tokenExpiration));
        $token = $tokenResult->accessToken;

        // Check if password reset is required
        $isResetRequired = 0;
        if ($userType === 'employee') {
            // For regular employees, check force_reset column
            $isResetRequired = ($user->force_reset == 1) ? 1 : 0;
        } else {
            // For subcontractor employees, check force_reset column
            $isResetRequired = (isset($user->force_reset) && $user->force_reset == 1) ? 1 : 0;
        }

        // Prepare user data based on type
        if ($userType === 'employee') {
            // Load relationships for regular employee
            $user->load(['EmpPersonalDetails', 'accessRole', 'accessTier']);
            
            $personalDetails = $user->EmpPersonalDetails;
            $fullName = $personalDetails 
                ? trim(($personalDetails->first_name ?? '') . ' ' . 
                      ($personalDetails->middle_name ?? '') . ' ' . 
                      ($personalDetails->last_name ?? ''))
                : null;
            
            $userData = [
                'id' => $user->id,
                'email' => $user->employee_email,
                'name' => $fullName,
                'first_name' => $personalDetails->first_name ?? null,
                'middle_name' => $personalDetails->middle_name ?? null,
                'last_name' => $personalDetails->last_name ?? null,
                'phone' => $personalDetails->mobile ?? null,
                'profile_image' => $personalDetails && $personalDetails->image ? url($personalDetails->image) : null,
                'user_type' => 'employee',
                'is_reset_required' => $isResetRequired,
            ];
            
        } else {
            // Subcontractor employee data
            $userData = [
                'id' => $user->id,
                'email' => $user->email,
                'name' => trim(($user->first_name ?? '') . ' ' . 
                             ($user->middle_name ?? '') . ' ' . 
                             ($user->last_name ?? '')),
                'first_name' => $user->first_name,
                'middle_name' => $user->middle_name,
                'last_name' => $user->last_name,
                'phone' => $user->phone,
                'profile_image' => $user->profile_image ? url($user->profile_image) : null,
                'user_type' => 'subcontractor_employee',
                'is_reset_required' => $isResetRequired,
            ];
        }

        // Pass user data directly to withToken - it will be placed in 'data' field
        return $this->withToken($userData, $token, 'Login successful', 200);
    }

    /**
     * Logout endpoint
     * 
     * Handles logout for both regular employees and subcontractor employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->error('Unauthorized access', 401);
        }
        
        // Delete all tokens for the user
        $user->tokens()->delete();

        return $this->message('Logged out successfully', 200);
    }

    /**
     * Reset password for both regular employees and subcontractor employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'new_password' => 'required|string|min:6',
            'confirm_password' => 'required|string|same:new_password',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed', 422);
        }

        // Get the authenticated user - the middleware uses auth:api,subcontractor_employee
        // We need to check the token scope to ensure we're using the correct guard/model
        $user = $request->user();
        
        if (!$user) {
            return $this->error('Unauthorized access', 401);
        }
        
        // Get token scopes to determine which guard/model should be used
        try {
            $token = $user->token();
            $scopes = $token ? ($token->scopes ?? []) : [];
            
            // If token has subcontractor_employee scope, ensure we're using the subcontractor_employee guard
            if (in_array('subcontractor_employee', $scopes)) {
                $user = Auth::guard('subcontractor_employee')->user();
                // Verify it's the correct model type
                if (!$user || !($user instanceof EmployeeSubcontractor)) {
                    return $this->error('Unauthorized: Invalid user type', 403);
                }
            } 
            // If token has api scope, ensure we're using the api guard
            elseif (in_array('api', $scopes)) {
                $user = Auth::guard('api')->user();
                // Verify it's the correct model type
                if (!$user || !($user instanceof EmpCompanyDetails)) {
                    return $this->error('Unauthorized: Invalid user type', 403);
                }
            }
        } catch (\Exception $e) {
            // If we can't check token, continue with the user from request
            // This should work in most cases, but might fail if IDs collide
        }

        // try {
            // Check if this is a force reset (first login) or profile password change
            $isForceReset = false;
            $userEmail = null;
            
            if ($user instanceof EmpCompanyDetails) {
                $isForceReset = ($user->force_reset == 1);
                $userEmail = $user->employee_email;
            } elseif ($user instanceof EmployeeSubcontractor) {
                $isForceReset = (isset($user->force_reset) && $user->force_reset == 1);
                $userEmail = $user->email;
            }

            // Check user type and update accordingly
            if ($user instanceof EmpCompanyDetails) {
                // Regular employee
                EmpCompanyDetails::where('id', $user->id)->update([
                    'password' => Hash::make($request->new_password),
                    'force_reset' => 0,
                ]);
            } elseif ($user instanceof EmployeeSubcontractor) {
                // Subcontractor employee
                EmployeeSubcontractor::where('id', $user->id)->update([
                    'password' => Hash::make($request->new_password),
                    'force_reset' => 0,
                ]);
            } else {
                return $this->error('Invalid user type', 400);
            }

            // Prepare response data
            $responseData = [
                'email' => $userEmail,
                'password_updated_at' => now()->toDateTimeString(),
            ];

            // If this was a force reset (first login), logout the user
            if ($isForceReset) {
                // Delete all tokens to force re-login with new password
                $user->tokens()->delete();
                
                $responseData['logout_required'] = true;
                return $this->success(
                    $responseData,
                    'Password reset successfully. Please login again with your new password.',
                    200
                );
            }

            // If not force reset, just update password without logging out
            return $this->success($responseData, 'Password reset successfully', 200);
        // } catch (\Exception $e) {
        //     return $this->error('An error occurred while resetting password', 500);
        // }
    }
}

