<?php

namespace App\Http\Controllers\MobileV2;

use App\Http\Controllers\Controller;
use App\Models\EmployeeSubcontractor;
use App\Models\EmpEmergencyContacts;
use App\Models\EmployeeAttendance;
use App\Models\LeaveRequest;
use App\Models\FundsMeta;
use App\Models\Fund;
use App\Models\Role;
use App\Models\EmpDocuments;
use App\Models\RequiredDocument;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;
use App\General\MetaClass;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ProfileController extends Controller
{

    public function getPersonalDetailsAndEmergencyContacts(Request $request)
    {
        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            $empId = $employee->id;

            // Get employee personal details
            $emp_personal_details = EmployeeSubcontractor::where('id', $empId)->first();

            if (!$emp_personal_details) {
                return $this->error('Employee not found', 404);
            }

            // Format personal details to match regular employee structure
            $personalDetails = [
                'id' => $emp_personal_details->id,
                'emp_id' => $emp_personal_details->id,
                'email' => $emp_personal_details->email,
                'first_name' => $emp_personal_details->first_name,
                'middle_name' => $emp_personal_details->middle_name,
                'last_name' => $emp_personal_details->last_name,
                'mobile' => $emp_personal_details->phone,
                'streat_address' => $emp_personal_details->address,
                'suburb' => $emp_personal_details->suburb,
                'state' => $emp_personal_details->state,
                'date_of_birth' => $emp_personal_details->dob,
                'medical_condition' => $emp_personal_details->medical_condition,
                'allergies' => $emp_personal_details->allergies,
                'previous_injuries' => $emp_personal_details->previous_injuries,
                'medical_attach_files' => $emp_personal_details->medical_attachement_files ? url($emp_personal_details->medical_attachement_files) : null,
                'profile_image' => $emp_personal_details->profile_image ? url($emp_personal_details->profile_image) : null,
                'signature' => $emp_personal_details->signature ? url($emp_personal_details->signature) : null,
            ];

            $data['emp_personal_details'] = $personalDetails;

            // Get emergency contacts from emp_emergency_contacts table
            $emp_emergency_contacts = EmpEmergencyContacts::where('emp_id', $empId)
                ->where('del', '0')
                ->get();

            $data['emp_emergency_contacts'] = $emp_emergency_contacts;

            // Get induction document counts and status
            $inductionDocs = $this->getEmployeeInductionDocument($empId);
            
            if ($inductionDocs !== null) {
                $data['induction_documents'] = [
                    'total_documents' => $inductionDocs['total_documents'] ?? 0,
                    'signed_documents' => $inductionDocs['total_signed'] ?? 0,
                    'pending_documents' => $inductionDocs['total_not_signed'] ?? 0,
                    'total_valid_signatures' => $inductionDocs['total_valid_signatures'] ?? 0,
                    'total_invalid_signatures' => $inductionDocs['total_invalid_signatures'] ?? 0,
                    'all_documents_signed' => $inductionDocs['all_documents_signed'] ?? false,
                    'all_signatures_valid' => $inductionDocs['all_signatures_valid'] ?? false,
                    'has_invalid_signatures' => $inductionDocs['has_invalid_signatures'] ?? false,
                ];
            } else {
                $data['induction_documents'] = [
                    'total_documents' => 0,
                    'signed_documents' => 0,
                    'pending_documents' => 0,
                    'total_valid_signatures' => 0,
                    'total_invalid_signatures' => 0,
                    'all_documents_signed' => true,
                    'all_signatures_valid' => true,
                    'has_invalid_signatures' => false,
                ];
            }

            return $this->success($data, 'Personal Details and Emergency Contacts Retrieved Successfully');
        } catch (\Exception $e) {
            Log::error('Error getting personal details and emergency contacts: ' . $e->getMessage());
            return $this->error('An error occurred while retrieving personal details and emergency contacts: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Store personal details and emergency contacts combined
     * Combines empPersonalDetailsStore and empEmergencyContactsStoreMulti
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function storePersonalDetailsAndEmergencyContacts(Request $request)
    {
        // Validate personal details
        $personalDetailsValidator = Validator::make($request->all(), [
            'first_name' => 'required',
            'middle_name' => 'nullable',
            'last_name' => 'required',
            'phone' => 'required|numeric',
            'address' => 'required',
            'suburb' => 'required|regex:/^[a-zA-Z\s]*$/',
            'state' => 'required|regex:/^[a-zA-Z\s]*$/',
            'dob' => 'required|date|before_or_equal:' . Carbon::now()->subYears(14)->format('Y-m-d'),
            'medical_condition' => 'nullable',
            'allergies' => 'nullable',
            'previous_injuries' => 'nullable',
        ], [
            'first_name.required' => 'First name is required.',
            'last_name.required' => 'Last name is required.',
            'phone.required' => 'Phone number is required.',
            'phone.numeric' => 'Phone must be numeric.',
            'address.required' => 'Address is required.',
            'suburb.required' => 'Suburb is required.',
            'suburb.regex' => 'Suburb must contain only alphabetic characters.',
            'state.required' => 'State is required.',
            'state.regex' => 'State must contain only alphabetic characters.',
            'dob.required' => 'Date of birth is required.',
            'dob.date' => 'Invalid date format for date of birth.',
            'dob.before' => 'You must be at least 14 years old to register.',
        ]);

        // Validate emergency contacts if provided
        if ($request->has('contacts')) {
            $emergencyContactsValidator = Validator::make($request->all(), [
                'contacts' => 'required|array',
                'contacts.*.name' => 'required',
                'contacts.*.relation' => 'required',
                'contacts.*.phone' => 'required',
                'contacts.*.email' => 'nullable|email',
                'contacts.*.address' => 'nullable'
            ], [
                'contacts.required' => 'Emergency contacts data is required.',
                'contacts.array' => 'Emergency contacts must be an array.',
                'contacts.*.name.required' => 'Name is required for each contact.',
                'contacts.*.email.email' => 'Invalid email format.',
                'contacts.*.phone.required' => 'Phone number is required for each contact.',
                'contacts.*.relation.required' => 'Relation is required for each contact.'
            ]);

            if ($emergencyContactsValidator->fails()) {
                $error = $emergencyContactsValidator->errors()->first();
                return $this->error($error, 422);
            }
        }

        if ($request->hasFile('profile_image')) {
            $personalDetailsValidator->sometimes('profile_image', 'image|mimes:jpeg,png,jpg,gif|max:2048', function ($input) {
                return $input->hasFile('profile_image');
            });
        }

        if ($request->hasFile('signature')) {
            $personalDetailsValidator->sometimes('signature', 'image|mimes:jpeg,png,jpg,gif|max:2048', function ($input) {
                return $input->hasFile('signature');
            });
        }

        if ($request->hasFile('medical_attach_files')) {
            $personalDetailsValidator->sometimes('medical_attach_files', 'array', function ($input) {
                return $input->hasFile('medical_attach_files');
            });
        }

        if ($personalDetailsValidator->fails()) {
            $error = $personalDetailsValidator->errors()->first();
            return $this->error($error, 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            $validatedData = $personalDetailsValidator->validated();
            $empId = $employee->id;

            // Use database transaction to ensure data integrity
            DB::beginTransaction();


            // Handle image upload
            if ($request->hasFile('profile_image')) {
                $image = $request->file('profile_image');
                $imageName = time() . '_' . $empId . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('upload/images'), $imageName);
                $validatedData['profile_image'] = 'upload/images/' . $imageName;
            }

            // Update employee personal details in employees_subcontractors table
            EmployeeSubcontractor::where('id', $empId)->update($validatedData);


            // // Update bank account data in user_meta table
            // $meta_ = $request->only([
            //     'account_holder_name',
            //     'account_number',
            //     'bank_name',
            //     'ibn_number'
            // ]);

            // $MetaArray = [];
            // foreach ($meta_ as $option => $value) {
            //     if ($value !== null && $value !== '') {
            //         $MetaArray[] = [
            //             'emp_id' => $empId,
            //             'option' => $option,
            //             'value' => $value,
            //             'created_at' => now(),
            //             'updated_at' => now()
            //         ];
            //         DB::table('user_meta')->where('emp_id', $empId)->where('option', $option)->delete();
            //     }
            // }

            // if (!empty($MetaArray)) {
            //     try {
            //         DB::table('user_meta')->insert($MetaArray);
            //     } catch (\Exception $e) {
            //         Log::error('Error saving bank account data: ' . $e->getMessage());
            //     }
            // }

            // Handle emergency contacts if provided
            $emergencyContactsData = [
                'emp_id' => $empId,
                'deleted_count' => 0,
                'success_count' => 0,
                'error_count' => 0
            ];

            if ($request->has('contacts') && is_array($request->contacts) && !empty($request->contacts)) {
                // Delete all existing emergency contacts for this employee
                $deletedCount = EmpEmergencyContacts::where('emp_id', $empId)->delete();
                Log::info("Deleted {$deletedCount} existing emergency contacts for employee {$empId}");
                $emergencyContactsData['deleted_count'] = $deletedCount;

                $successCount = 0;
                $errorCount = 0;

                // Insert new emergency contacts
                foreach ($request->contacts as $contact) {
                    try {
                        $contactData = [
                            'emp_id' => $empId,
                            'name' => $contact['name'],
                            'relation' => $contact['relation'],
                            'phone' => $contact['phone'],
                            'email' => $contact['email'] ?? null,
                            'address' => $contact['address'] ?? null,
                            'created_by' => $empId,
                            'created_at' => now(),
                            'updated_at' => now()
                        ];

                        EmpEmergencyContacts::insert($contactData);
                        $successCount++;
                    } catch (\Exception $e) {
                        $errorCount++;
                        Log::error('Error storing emergency contact: ' . $e->getMessage(), [
                            'contact' => $contact,
                            'error' => $e->getMessage()
                        ]);
                    }
                }

                $emergencyContactsData['success_count'] = $successCount;
                $emergencyContactsData['error_count'] = $errorCount;
            }

            DB::commit();

            $data = [
                'emp_id' => $empId,
                'emergency_contacts' => $emergencyContactsData
            ];

            $message = 'Personal Details and Emergency Contacts Saved Successfully';
            if ($emergencyContactsData['error_count'] > 0) {
                $message .= " but {$emergencyContactsData['error_count']} contact(s) failed to save";
            }

            return $this->success($data, $message);
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error storing personal details and emergency contacts: ' . $e->getMessage());
            return $this->error('An error occurred while storing personal details and emergency contacts: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get attendance statistics for subcontractor employee
     * Similar to get-attendance-stats for regular employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAttendanceStats(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'month' => 'required|date_format:Y-m',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            $validatedData = $validator->validated();
            $startDate = Carbon::parse($validatedData['month'])->startOfMonth();
            $endDate = Carbon::parse($validatedData['month'])->endOfMonth();

            // Get employee's active invitation to get subcontractor_id
            $invitation = \App\Models\SubcontractorEmployeeInvitation::where('employee_id', $employee->id)
                ->where('invitation_status', 'accepted')
                ->where('status', 1)
                ->first();

            if (!$invitation) {
                return $this->error('No active invitation found for this employee.', 404);
            }

            $subcontractorId = $invitation->subcontractor_id;

            $totalWorkingDays = $this->getWorkingDays($startDate, $endDate);

            // Get attendance records for subcontractor employee
            $attendanceRecords = EmployeeAttendance::where('employee_id', $employee->id)
                ->where('subcontractor_id', $subcontractorId)
                ->whereBetween('date', [$startDate, $endDate])
                ->get();

            // Get leave requests for subcontractor employee
            $leaveRequests = LeaveRequest::where('employee_id', $employee->id)
                ->where('subcontractor_id', $subcontractorId)
                ->where('status', 1) // 1 for approved leaves
                ->where(function($query) use ($startDate, $endDate) {
                    $query->whereBetween('from', [$startDate, $endDate])
                          ->orWhereBetween('to', [$startDate, $endDate])
                          ->orWhere(function($q) use ($startDate, $endDate) {
                              $q->where('from', '<=', $startDate)
                                ->where('to', '>=', $endDate);
                          });
                })
                ->get();

            $stats = [
                'total_days' => $totalWorkingDays,
                'clocked_in' => 0,
                'on_leave' => 0,
                'late' => 0,
                'absent' => 0
            ];

            // Track unique dates that have been clocked in
            $clockedInDates = [];
            // Track the earliest check-in time for each date to determine if it was late
            $earliestCheckInByDate = [];

            foreach ($attendanceRecords as $record) {
                if ($record->status == 1) {
                    try {
                        $parsedDate = \App\Models\BaseModel::safeCarbonParse($record->date, 'getAttendanceStats date');
                        if ($parsedDate instanceof Carbon) {
                            $dateKey = $parsedDate->format('Y-m-d');
                        } else {
                            $dateKey = $this->parseDateString($record->date);
                        }
                    } catch (\Exception $e) {
                        Log::warning("Failed to parse date for attendance record {$record->id}: " . $e->getMessage());
                        continue;
                    }
                   
                    if (!isset($clockedInDates[$dateKey])) {
                        $clockedInDates[$dateKey] = true;
                        $stats['clocked_in']++;
                    }

                    try {
                        $checkInTime = \App\Models\BaseModel::safeCarbonParse($record->check_in, 'check_in');
                        if ($checkInTime) {
                            if (!isset($earliestCheckInByDate[$dateKey]) || 
                                $checkInTime->lt($earliestCheckInByDate[$dateKey])) {
                                $earliestCheckInByDate[$dateKey] = $checkInTime;
                            }
                        }
                    } catch (\Exception $e) {
                        Log::warning("Failed to parse check_in time for attendance record {$record->id}: " . $e->getMessage());
                    }
                }
            }

            // Count late arrivals
            foreach ($earliestCheckInByDate as $dateKey => $checkInTime) {
                if ($checkInTime->format('H:i') > '09:00') {
                    $stats['late']++;
                }
            }

            // Calculate leave days
            $leaveDays = 0;
            foreach ($leaveRequests as $leave) {
                try {
                    $leaveStart = \App\Models\BaseModel::safeCarbonParse($leave->from, 'leave_from');
                    $leaveEnd = \App\Models\BaseModel::safeCarbonParse($leave->to, 'leave_to');
                    
                    if (!$leaveStart || !$leaveEnd) {
                        continue;
                    }
                    
                    if ($leaveStart->lt($startDate)) {
                        $leaveStart = $startDate;
                    }
                    if ($leaveEnd->gt($endDate)) {
                        $leaveEnd = $endDate;
                    }

                    $currentDate = $leaveStart->copy();
                    while ($currentDate <= $leaveEnd) {
                        if ($currentDate->dayOfWeek !== 0 && $currentDate->dayOfWeek !== 6) {
                            $leaveDays++;
                        }
                        $currentDate->addDay();
                    }
                } catch (\Exception $e) {
                    Log::warning("Failed to parse leave dates for leave request {$leave->id}: " . $e->getMessage());
                }
            }
            $stats['on_leave'] = $leaveDays;

            $stats['absent'] = $totalWorkingDays - ($stats['clocked_in'] + $stats['on_leave']);

            $stats['clocked_in_percentage'] = round(($stats['clocked_in'] / $totalWorkingDays) * 100, 2);
            $stats['on_leave_percentage'] = round(($stats['on_leave'] / $totalWorkingDays) * 100, 2);
            $stats['late_percentage'] = round(($stats['late'] / $totalWorkingDays) * 100, 2);
            $stats['absent_percentage'] = round(($stats['absent'] / $totalWorkingDays) * 100, 2);

            return $this->success($stats, 'Attendance statistics retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Error getting attendance stats: ' . $e->getMessage());
            return $this->error('An error occurred while retrieving attendance statistics: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get full profile for subcontractor employee
     * Similar to profile endpoint for regular employees
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function profile(Request $request)
    {
        $employee = $request->user();

        if (!$employee || !($employee instanceof EmployeeSubcontractor)) {
            return $this->error('Unauthorized access', 401);
        }

        try {
            $empId = $employee->id;

            // Get employee data
            $employeeData = EmployeeSubcontractor::where('id', $empId)->first();

            if (!$employeeData) {
                return $this->error('Employee not found', 404);
            }

            // Get active invitation to get customer_id and workspace_id
            $invitation = \App\Models\SubcontractorEmployeeInvitation::where('employee_id', $empId)
                ->where('invitation_status', 'accepted')
                ->where('status', 1)
                ->first();

            // Build profile matching regular employee structure
            $profile = [
                'id' => $employeeData->id,
                'employee_email' => $employeeData->email,
                'customer_id' => $invitation ? $invitation->customer_id : null,
                'workspace_id' => $invitation ? $invitation->workspace_id : null,
            ];

            // Add personal details
            $profile['empPersonalDetails'] = [
                'id' => $employeeData->id,
                'emp_id' => $employeeData->id,
                'first_name' => $employeeData->first_name,
                'middle_name' => $employeeData->middle_name,
                'last_name' => $employeeData->last_name,
                'mobile' => $employeeData->phone,
                'streat_address' => $employeeData->address,
                'suburb' => $employeeData->suburb,
                'state' => $employeeData->state,
                'postcode' => null,
                'date_of_birth' => $employeeData->dob,
                'blood_group' => null,
                'image' => $employeeData->profile_image,
            ];

            // Add emergency contacts
            $profile['empEmergencyContacts'] = EmpEmergencyContacts::where('emp_id', $empId)
                ->where('del', '0')
                ->get();

            // Add empAccess (null for subcontractor employees)
            $profile['empAccess'] = null;

            // Add empworkExperience (null for subcontractor employees)
            $profile['empworkExperience'] = null;

            // Add workerType (null for subcontractor employees)
            $profile['workerType'] = null;

            // Add empType (null for subcontractor employees)
            $profile['empType'] = null;

            // Add empWorkPermit (null for subcontractor employees)
            $profile['empWorkPermit'] = null;

            // Add empAccessRole (default EMP role)
            $defaultRole = Role::where('code', 'EMP')->where('del', '0')->first();
            $profile['empAccessRole'] = $defaultRole ? [
                'id' => $defaultRole->id,
                'title' => $defaultRole->title,
                'code' => $defaultRole->code,
            ] : null;

            $data['profile'] = $profile;

            // Funds meta
            $data['funds_meta'] = FundsMeta::with('fund')
                ->where('emp_id', $empId)
                ->get();

            // Medical files
            $data['medicalAttachFiles'] = DB::table('medical_attach_files')
                ->where('emp_id', $empId)
                ->get();

            // Employee documents
            $data['emp_documents'] = EmpDocuments::where('emp_id', $empId)
                ->where('del', '0')
                ->get();

            // Required documents - filter by external employee type
            if ($invitation) {
                $requiredDocsQuery = RequiredDocument::with('requiredDocumentField')
                    ->where('customer_id', $invitation->customer_id)
                    ->where('workspace_id', $invitation->workspace_id)
                    ->where('del', '0')
                    ->where('for_external', true); // Subcontractor employees are external
                
                $data['required_documents'] = $requiredDocsQuery->get();
            } else {
                $data['required_documents'] = [];
            }

            // Meta
            $user_meta = DB::table('user_meta')
                ->where('emp_id', $empId)
                ->get();

            $employee_payroll_or_external_id = MetaClass::getOptionValue('employee_payroll_or_external_id', $user_meta);
            $data['meta']['employee_payroll_or_external_id'] = $employee_payroll_or_external_id;

            // Get induction documents
            $data['induction_documents'] = $this->getEmployeeInductionDocument($empId);

            return $this->success($data, 'Get Profile Successfully');
        } catch (\Exception $e) {
            Log::error('Error getting profile: ' . $e->getMessage());
            return $this->error('An error occurred while retrieving profile: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get working days in a date range (excluding weekends)
     * 
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @return int
     */
    private function getWorkingDays($startDate, $endDate)
    {
        $workingDays = 0;
        $currentDate = $startDate->copy();

        while ($currentDate <= $endDate) {
            if ($currentDate->dayOfWeek !== 0 && $currentDate->dayOfWeek !== 6) {
                $workingDays++;
            }
            $currentDate->addDay();
        }

        return $workingDays;
    }

    /**
     * Parse date string with fallback
     * 
     * @param string $dateString
     * @return string
     */
    private function parseDateString($dateString)
    {
        try {
            return Carbon::parse($dateString)->format('Y-m-d');
        } catch (\Exception $e) {
            return $dateString;
        }
    }

    /**
     * Get employee induction documents with signature status
     * 
     * @param int $empId
     * @return array|null
     */
    private function getEmployeeInductionDocument($empId)
    {
        try {
            // Get active invitation to get customer_id and workspace_id
            $invitation = \App\Models\SubcontractorEmployeeInvitation::where('employee_id', $empId)
                ->where('invitation_status', 'accepted')
                ->where('status', 1)
                ->first();

            if (!$invitation) {
                return null;
            }

            // Get all induction documents for this customer/workspace
            $documents = InductionDocument::where('customer_id', $invitation->customer_id)
                ->where('workspace_id', $invitation->workspace_id)
                ->where('del', '0')
                ->get();

            if ($documents->isEmpty()) {
                return null;
            }

            $totalDocuments = $documents->count();
            $signedDocuments = 0;
            $notSignedDocuments = 0;
            $totalValidSignatures = 0;
            $totalInvalidSignatures = 0;

            foreach ($documents as $document) {
                $signature = InductionDocumentSignature::where('induction_document_id', $document->id)
                    ->where('employee_id', $empId)
                    ->first();

                if ($signature) {
                    $signedDocuments++;
                    if ($signature->is_valid) {
                        $totalValidSignatures++;
                    } else {
                        $totalInvalidSignatures++;
                    }
                } else {
                    $notSignedDocuments++;
                }
            }

            return [
                'total_documents' => $totalDocuments,
                'total_signed' => $signedDocuments,
                'total_not_signed' => $notSignedDocuments,
                'total_valid_signatures' => $totalValidSignatures,
                'total_invalid_signatures' => $totalInvalidSignatures,
                'all_documents_signed' => $notSignedDocuments === 0,
                'all_signatures_valid' => $totalInvalidSignatures === 0,
                'has_invalid_signatures' => $totalInvalidSignatures > 0,
            ];
        } catch (\Exception $e) {
            Log::error('Error getting induction documents: ' . $e->getMessage());
            return null;
        }
    }
}
