<?php

namespace App\Http\Controllers\MobileV2;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Traits\EmailTrait;
use App\Models\SubcontractorEmployeeDocument;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\SubcontractorCompany;
use App\Models\User;
use App\Models\RequiredDocument;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

/**
 * Controller for customer-side operations on subcontractor employee documents
 */
class SubcontractorEmployeeDocumentController extends Controller
{
    use EmailTrait;
    /**
     * Get all submitted documents for a subcontractor employee on a project
     * For customer use
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSubmittedDocuments(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:employees_subcontractors,id',
            // 'project_id' => 'required|integer|exists:projects,id',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        try {
            $user = $request->user();
            
            if (!$user) {
                return $this->error('Unauthorized access', 401);
            }

            // Get the customer_id and workspace_id
            $customerId = $user->id;
            $workspaceId = $user->current_workspace_id ?? null;

            // Get employee details
            $employee = EmployeeSubcontractor::find($request->employee_id);
            if (!$employee) {
                return $this->error('Employee not found', 404);
            }

            // Get subcontractor details
            $metas = EmployeeSubcontractorMeta::where('emp_id', $employee->id)
                ->where('active', true)
                ->with(['subcontractor'])
                ->get();
            
            $subcontractors = [];
            foreach ($metas as $meta) {
                $subcontractor = $meta->subcontractor;
                if ($subcontractor) {
                    $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
                        ->where('del', '0')
                        ->with('trade')
                        ->first();
                    
                    $subcontractors[] = [
                        'id' => $subcontractor->id,
                        'name' => $subcontractor->name,
                        'company_name' => $subcontractor->company_name,
                        'email' => $subcontractor->email,
                        'mobile_number' => $subcontractor->mobile_number,
                        'abn' => $subcontractor->abn,
                        'company_logo' => $subcontractor->company_logo ? url($subcontractor->company_logo) : null,
                        'trade_name' => $subcontractorCompany && $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null,
                    ];
                }
            }

            // Get submitted documents for this employee and customer (any project)
            // Documents are shared across all projects for the same customer
            $submittedDocuments = SubcontractorEmployeeDocument::where('employee_id', $request->employee_id)
                ->where('customer_id', $customerId)
                ->where(function ($query) use ($workspaceId) {
                    if ($workspaceId) {
                        $query->where('workspace_id', $workspaceId);
                    } else {
                        $query->whereNull('workspace_id');
                    }
                })
                ->where('del', '0')
                ->with(['requiredDocument', 'requiredDocumentField', 'approver'])
                ->get();

            // Get required documents for comparison
            $requiredDocuments = RequiredDocument::with(['requiredDocumentField' => function ($query) {
                    $query->where('status', 1)->where('del', 0);
                }])
                ->where('del', '0')
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where(function ($q) {
                    $q->where('for_external', true)
                        ->orWhere('for_subcontractor', true);
                })
                ->get();

            // Group submitted documents by required_document_id
            $submittedByDocId = $submittedDocuments->groupBy('required_document_id');

            $data = [
                'employee' => [
                    'id' => $employee->id,
                    'name' => trim($employee->first_name . ' ' . ($employee->middle_name ?? '') . ' ' . $employee->last_name),
                    'email' => $employee->email,
                    'subcontractors' => $subcontractors,
                ],
                'documents' => $requiredDocuments->map(function ($document) use ($submittedByDocId) {
                    $submissions = $submittedByDocId->get($document->id, collect());

                    // Calculate document-level approval status
                    $docApprovalStatus = null;
                    $docRejectionReason = null;
                    $docApprovedBy = null;
                    $docApprovedAt = null;
                    
                    if ($submissions->isNotEmpty()) {
                        $allApproved = $submissions->every(fn($s) => $s->approval_status === 'approved');
                        $anyRejected = $submissions->contains(fn($s) => $s->approval_status === 'rejected');
                        
                        if ($allApproved) {
                            $docApprovalStatus = 'approved';
                            $firstApproved = $submissions->first();
                            $docApprovedBy = $firstApproved->approver ? $firstApproved->approver->name : null;
                            // Use getRawOriginal to get raw timestamp from database
                            $docApprovedAt = $firstApproved->getRawOriginal('approved_at');
                        } elseif ($anyRejected) {
                            $docApprovalStatus = 'rejected';
                            $rejectedDoc = $submissions->firstWhere('approval_status', 'rejected');
                            $docRejectionReason = $rejectedDoc->rejection_reason;
                            $docApprovedBy = $rejectedDoc->approver ? $rejectedDoc->approver->name : null;
                            // Use getRawOriginal to get raw timestamp from database
                            $docApprovedAt = $rejectedDoc->getRawOriginal('approved_at');
                        } else {
                            $docApprovalStatus = 'pending';
                        }
                    }

                    // Check if document is complete
                    $requiredFields = $document->requiredDocumentField->where('field_required', '1');
                    
                    if ($requiredFields->count() > 0) {
                        // If there are required fields, all of them must be submitted
                        $submittedRequiredCount = $submissions->filter(function($s) use ($requiredFields) {
                            return $requiredFields->contains('id', $s->required_document_field_id) && !empty($s->value);
                        })->count();
                        $isComplete = $submittedRequiredCount >= $requiredFields->count();
                    } else {
                        // If there are NO required fields, it is complete if at least one field is submitted with a value
                        $isComplete = $submissions->contains(function($s) {
                            return !empty($s->value);
                        });
                    }

                    // $isComplete = true;
                    return [
                        'id' => $document->id,
                        'title' => $document->title,
                        'approval_status' => $docApprovalStatus,
                        'rejection_reason' => $docRejectionReason,
                        'approved_by' => $docApprovedBy,
                        'approved_at' => $docApprovedAt,
                        'is_complete' => $isComplete,
                        'fields' => $document->requiredDocumentField->map(function ($field) use ($submissions) {
                            $submission = $submissions->firstWhere('required_document_field_id', $field->id);
                            
                            // Handle file paths vs text/date values
                            $submittedValue = null;
                            $submittedFileUrl = null;
                            if ($submission && $submission->value) {
                                // Check if value is a file path
                                if (strpos($submission->value, 'upload/') === 0 || 
                                    strpos($submission->value, 'SubcontractorEmployeeDocuments/') === 0) {
                                    $submittedValue = $submission->value;
                                    $submittedFileUrl = url($submission->value);
                                } else {
                                    // Return text/date values as-is
                                    $submittedValue = $submission->value;
                                }
                            }
                            
                            return [
                                'id' => $field->id,
                                'field_name' => $field->field_name,
                                'field_type' => $field->field_type,
                                'field_required' => $field->field_required,
                                'priority' => $field->priority ?? 0,
                                'submission_id' => $submission ? $submission->id : null,
                                'submitted_value' => $submittedValue,
                                'submitted_file_url' => $submittedFileUrl,
                                // Return raw timestamp from database
                                'last_updated' => $submission ? $submission->getRawOriginal('updated_at') : null,
                            ];
                        }),
                    ];
                }),
            ];

            return $this->success($data, 'Submitted documents retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while retrieving documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Unified API to approve or reject documents (single or multiple)
     * Works with document template IDs (required_documents table) - approves/rejects ALL fields of documents at once
     * For customer use
     * 
     * Single document: { employee_id, document_id, action, rejection_reason }
     * Multiple documents: { employee_id, documents: [{ document_id, action, rejection_reason }, ...] }
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateDocumentStatus(Request $request)
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->error('Unauthorized access', 401);
        }

        // Check if it's a bulk operation (documents array) or single operation
        if ($request->has('documents') && is_array($request->documents)) {
            // Bulk operation - multiple documents
            return $this->bulkApproveRejectDocuments($request, $user);
        } else {
            // Single document operation
            return $this->singleApproveRejectDocument($request, $user);
        }
    }

    /**
     * Handle single document approval/rejection
     * 
     * @param Request $request
     * @param User $user
     * @return \Illuminate\Http\JsonResponse
     */
    private function singleApproveRejectDocument(Request $request, $user)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:employees_subcontractors,id',
            'document_id' => 'required|integer|exists:required_documents,id',
            'action' => 'required|in:approve,reject',
            'rejection_reason' => 'required_if:action,reject|nullable|string|max:500',
        ], [
            'employee_id.required' => 'Employee ID is required.',
            'employee_id.exists' => 'Employee not found.',
            'document_id.required' => 'Document ID is required.',
            'document_id.exists' => 'Document not found.',
            'action.required' => 'Action is required (approve or reject).',
            'action.in' => 'Action must be either approve or reject.',
            'rejection_reason.required_if' => 'Rejection reason is required when rejecting.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        try {
            $customerId = $user->id;
            $workspaceId = $user->current_workspace_id ?? null;
            $employeeId = $request->employee_id;
            $documentId = $request->document_id;
            $action = $request->action;
            $rejectionReason = $request->rejection_reason;

            // Verify the document template exists and belongs to this customer
            $documentTemplate = RequiredDocument::where('id', $documentId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where(function ($q) {
                    $q->where('for_external', true)
                        ->orWhere('for_subcontractor', true);
                })
                ->where('del', '0')
                ->first();

            if (!$documentTemplate) {
                return $this->error('Document template not found or not available for employees.', 404);
            }

            // Check if any documents are uploaded for this template
            $hasUploads = SubcontractorEmployeeDocument::where('employee_id', $employeeId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('required_document_id', $documentId)
                ->where('del', '0')
                ->exists();

            if (!$hasUploads) {
                return $this->error('No uploaded documents found for this template. The employee has not uploaded any documents yet.', 404);
            }

            // Determine the new status
            $newStatus = $action === 'approve' ? SubcontractorEmployeeDocument::STATUS_APPROVED : SubcontractorEmployeeDocument::STATUS_REJECTED;

            // Update ALL fields of this document template at once
            $updatedCount = SubcontractorEmployeeDocument::where('employee_id', $employeeId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('required_document_id', $documentId)
                ->where('del', '0')
                ->update([
                    'approval_status' => $newStatus,
                    'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                    'approved_by' => $user->id,
                    'approved_at' => now(),
                ]);

            $message = $action === 'approve' 
                ? "Document '{$documentTemplate->title}' approved successfully." 
                : "Document '{$documentTemplate->title}' rejected successfully.";

            // Update the employee invitation document status based on all documents
            $this->updateEmployeeInvitationDocumentStatus($employeeId, $customerId, $workspaceId);

            // Send email notification to employee
            try {
                $employee = EmployeeSubcontractor::find($employeeId);
                if ($employee && $employee->email) {
                    $brandTitle = \App\Services\CompanyEmailDetails::getBrandTitle($customerId, $workspaceId);
                    $subject = $action === 'approve' 
                        ? "Document Approved - {$documentTemplate->title} | {$brandTitle}"
                        : "Document Rejected - {$documentTemplate->title} | {$brandTitle}";
                    
                    $employeeName = trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? ''));
                    
                    \App\Jobs\SendNotificationEmailJob::dispatch([
                        'to' => $employee->email,
                        'subject' => $subject,
                        'template' => 'Emails.subcontractor-employee-document-status',
                        'template_data' => [
                            'subject' => $subject,
                            'employee_name' => $employeeName ?: 'there',
                            'document_title' => $documentTemplate->title,
                            'status' => $action === 'approve' ? 'approved' : 'rejected',
                            'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                            'customer_id' => $customerId,
                            'workspace_id' => $workspaceId,
                        ],
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId,
                    ]);
                    \Illuminate\Support\Facades\Log::info('Document status email sent to employee', [
                        'employee_id' => $employeeId,
                        'employee_email' => $employee->email,
                        'document_id' => $documentId,
                        'action' => $action,
                    ]);
                }
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error('Error sending document status email to employee: ' . $e->getMessage(), [
                    'employee_id' => $employeeId,
                    'document_id' => $documentId,
                    'action' => $action,
                ]);
                // Don't fail the request if email fails
            }

            return $this->success([
                'employee_id' => $employeeId,
                'updated_count' => 1,
                'updated_documents' => [
                    [
                        'document_id' => $documentId,
                        'document_title' => $documentTemplate->title,
                        'action' => $action,
                        'approval_status' => $newStatus,
                        'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                        'fields_updated' => $updatedCount,
                    ]
                ],
                'approved_by' => $user->id,
                'approved_at' => now()->toDateTimeString(),
                'errors' => [],
            ], $message, 200);
        } catch (\Exception $e) {
            return $this->error('An error occurred while updating document status: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Handle bulk document approval/rejection
     * 
     * @param Request $request
     * @param User $user
     * @return \Illuminate\Http\JsonResponse
     */
    private function bulkApproveRejectDocuments(Request $request, $user)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:employees_subcontractors,id',
            'documents' => 'required|array|min:1',
            'documents.*.document_id' => 'required|integer|exists:required_documents,id',
            'documents.*.action' => 'required|in:approve,reject',
            'documents.*.rejection_reason' => 'nullable|string|max:500',
        ], [
            'employee_id.required' => 'Employee ID is required.',
            'employee_id.exists' => 'Employee not found.',
            'documents.required' => 'Documents array is required.',
            'documents.min' => 'At least one document must be provided.',
            'documents.*.document_id.required' => 'Document ID is required for each document.',
            'documents.*.document_id.exists' => 'One or more documents not found.',
            'documents.*.action.required' => 'Action is required for each document.',
            'documents.*.action.in' => 'Action must be either approve or reject.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        try {
            $customerId = $user->id;
            $workspaceId = $user->current_workspace_id ?? null;
            $employeeId = $request->employee_id;
            $updatedDocuments = [];
            $errors = [];

            DB::beginTransaction();

            foreach ($request->documents as $docData) {
                $documentId = $docData['document_id'];
                $action = $docData['action'];
                $rejectionReason = $docData['rejection_reason'] ?? null;

                // Verify the document template exists and belongs to this customer
                $documentTemplate = RequiredDocument::where('id', $documentId)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where(function ($q) {
                        $q->where('for_external', true)
                            ->orWhere('for_subcontractor', true);
                    })
                    ->where('del', '0')
                    ->first();

                if (!$documentTemplate) {
                    $errors[] = "Document ID {$documentId} not found or not available for employees.";
                    continue;
                }

                // Check if any documents are uploaded for this template
                $hasUploads = SubcontractorEmployeeDocument::where('employee_id', $employeeId)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('required_document_id', $documentId)
                    ->where('del', '0')
                    ->exists();

                if (!$hasUploads) {
                    $errors[] = "No uploaded documents found for '{$documentTemplate->title}'.";
                    continue;
                }

                // Validate rejection reason for rejected status
                if ($action === 'reject' && empty($rejectionReason)) {
                    $errors[] = "Rejection reason required for document '{$documentTemplate->title}'.";
                    continue;
                }

                // Determine the new status
                $newStatus = $action === 'approve' ? SubcontractorEmployeeDocument::STATUS_APPROVED : SubcontractorEmployeeDocument::STATUS_REJECTED;

                // Update ALL fields of this document template at once
                $updatedCount = SubcontractorEmployeeDocument::where('employee_id', $employeeId)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('required_document_id', $documentId)
                    ->where('del', '0')
                    ->update([
                        'approval_status' => $newStatus,
                        'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                        'approved_by' => $user->id,
                        'approved_at' => now(),
                    ]);

                $updatedDocuments[] = [
                    'document_id' => $documentId,
                    'document_title' => $documentTemplate->title,
                    'action' => $action,
                    'approval_status' => $newStatus,
                    'fields_updated' => $updatedCount,
                ];
            }

            DB::commit();

            // Update the employee invitation document status based on all documents
            $this->updateEmployeeInvitationDocumentStatus($employeeId, $customerId, $workspaceId);

            // Send email notifications for each updated document
            try {
                $employee = EmployeeSubcontractor::find($employeeId);
                if ($employee && $employee->email && !empty($updatedDocuments)) {
                    $brandTitle = \App\Services\CompanyEmailDetails::getBrandTitle($customerId, $workspaceId);
                    $employeeName = trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? ''));
                    
                    // Group documents by action (approved/rejected)
                    $approvedDocs = collect($updatedDocuments)->where('action', 'approve');
                    $rejectedDocs = collect($updatedDocuments)->where('action', 'reject');
                    
                    // Send email for approved documents if any
                    if ($approvedDocs->isNotEmpty()) {
                        $subject = "Documents Approved | {$brandTitle}";
                        \App\Jobs\SendNotificationEmailJob::dispatch([
                            'to' => $employee->email,
                            'subject' => $subject,
                            'template' => 'Emails.subcontractor-employee-document-status',
                            'template_data' => [
                                'subject' => $subject,
                                'employee_name' => $employeeName ?: 'there',
                                'document_title' => $approvedDocs->pluck('document_title')->join(', '),
                                'status' => 'approved',
                                'rejection_reason' => null,
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                            ],
                            'customer_id' => $customerId,
                            'workspace_id' => $workspaceId,
                        ]);
                    }
                    
                    // Send email for rejected documents if any
                    if ($rejectedDocs->isNotEmpty()) {
                        $subject = "Documents Rejected | {$brandTitle}";
                        \App\Jobs\SendNotificationEmailJob::dispatch([
                            'to' => $employee->email,
                            'subject' => $subject,
                            'template' => 'Emails.subcontractor-employee-document-status',
                            'template_data' => [
                                'subject' => $subject,
                                'employee_name' => $employeeName ?: 'there',
                                'document_title' => $rejectedDocs->pluck('document_title')->join(', '),
                                'status' => 'rejected',
                                'rejection_reason' => 'Please check your portal for details.',
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                            ],
                            'customer_id' => $customerId,
                            'workspace_id' => $workspaceId,
                        ]);
                    }
                }
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error('Error sending bulk document status emails to employee: ' . $e->getMessage(), [
                    'employee_id' => $employeeId,
                ]);
                // Don't fail the request if email fails
            }

            return $this->success([
                'employee_id' => $employeeId,
                'updated_count' => count($updatedDocuments),
                'updated_documents' => $updatedDocuments,
                'approved_by' => $user->id,
                'approved_at' => now()->toDateTimeString(),
                'errors' => $errors,
            ], 'Documents updated successfully', 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->error('An error occurred while updating documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Update the overall employee invitation document status based on individual document approvals
     * Similar to updateSubcontractorCompanyDocumentStatus in SubContractorController
     * 
     * @param int $employeeId
     * @param int $customerId
     * @param int $workspaceId
     * @return void
     */
    private function updateEmployeeInvitationDocumentStatus($employeeId, $customerId, $workspaceId)
    {
        // Get all required documents for external/subcontractor employees
        $requiredDocuments = \App\Models\RequiredDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where(function ($q) {
                $q->where('for_external', true)
                    ->orWhere('for_subcontractor', true);
            })
            ->where('del', '0')
            ->pluck('id')
            ->toArray();

        if (empty($requiredDocuments)) {
            // No required documents for this customer, mark as complete
            SubcontractorEmployeeInvitation::updateEmployeeDocsStatus(
                $employeeId,
                $customerId,
                $workspaceId,
                SubcontractorEmployeeInvitation::DOCS_STATUS_COMPLETE
            );
            return;
        }

        // Get all submitted documents for this employee
        $submittedDocuments = SubcontractorEmployeeDocument::where('employee_id', $employeeId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->get();

        // Group by required_document_id to check per-document status
        $documentsByDocId = $submittedDocuments->groupBy('required_document_id');

        // Check how many required documents have been submitted
        $submittedDocIds = $documentsByDocId->keys()->toArray();
        $allDocsSubmitted = count(array_intersect($requiredDocuments, $submittedDocIds)) >= count($requiredDocuments);

        if (!$allDocsSubmitted) {
            // Not all documents submitted yet
            if ($submittedDocuments->isNotEmpty()) {
                // Some documents submitted
                SubcontractorEmployeeInvitation::updateEmployeeDocsStatus(
                    $employeeId,
                    $customerId,
                    $workspaceId,
                    SubcontractorEmployeeInvitation::DOCS_STATUS_PENDING
                );
            } else {
                // No documents submitted
                SubcontractorEmployeeInvitation::updateEmployeeDocsStatus(
                    $employeeId,
                    $customerId,
                    $workspaceId,
                    SubcontractorEmployeeInvitation::DOCS_STATUS_NOT_UPLOADED
                );
            }
            return;
        }

        // Check approval status of all submitted documents
        $allApproved = $submittedDocuments->every(function ($doc) {
            return $doc->approval_status === SubcontractorEmployeeDocument::STATUS_APPROVED;
        });

        $hasRejected = $submittedDocuments->contains(function ($doc) {
            return $doc->approval_status === SubcontractorEmployeeDocument::STATUS_REJECTED;
        });

        $hasPending = $submittedDocuments->contains(function ($doc) {
            return $doc->approval_status === SubcontractorEmployeeDocument::STATUS_PENDING;
        });

        // Update status based on document approvals
        if ($allApproved) {
            // All documents approved
            SubcontractorEmployeeInvitation::updateEmployeeDocsStatus(
                $employeeId,
                $customerId,
                $workspaceId,
                SubcontractorEmployeeInvitation::DOCS_STATUS_COMPLETE
            );
        } elseif ($hasRejected) {
            // At least one document rejected
            SubcontractorEmployeeInvitation::updateEmployeeDocsStatus(
                $employeeId,
                $customerId,
                $workspaceId,
                SubcontractorEmployeeInvitation::DOCS_STATUS_REJECTED
            );
        } elseif ($hasPending) {
            // Some documents still pending
            SubcontractorEmployeeInvitation::updateEmployeeDocsStatus(
                $employeeId,
                $customerId,
                $workspaceId,
                SubcontractorEmployeeInvitation::DOCS_STATUS_PENDING
            );
        }
    }


}

