<?php

namespace App\Http\Controllers;

use App\Models\PermissionTemplate;
use App\Models\PermissionTemplateItem;
use App\Models\Permission;
use App\Models\CustomerPermission;
use App\Models\User;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class PermissionTemplateController extends Controller
{
    /**
     * Display a listing of permission templates.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $query = PermissionTemplate::where('del', '0');
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where('title', 'like', '%' . $search . '%');
        }
        $query->withCount('permissions');
        $query->orderBy('created_at', 'desc');
        $templates = $query->get();
        return $this->success($templates, 'Permission templates retrieved successfully');
    }

    /**
     * Store a newly created permission template.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'nullable|string|in:active,inactive',
            'permission_ids' => 'required|array|min:1',
            'permission_ids.*' => 'required|integer|exists:permissions,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        DB::beginTransaction();
        try {
            $template = PermissionTemplate::create([
                'title' => $request->title,
                'description' => $request->description,
                'status' => $request->status ?? 'active',
                'created_by' => Auth::id(),
                'del' => '0',
            ]);
            $permissionIds = array_unique($request->permission_ids);
            $existingPermissions = Permission::whereIn('id', $permissionIds)
                ->where('del', '0')
                ->pluck('id')
                ->toArray();
            if (count($existingPermissions) !== count($permissionIds)) {
                DB::rollBack();
                return $this->error('One or more permission IDs are invalid or deleted', 422);
            }
            foreach ($permissionIds as $permissionId) {
                PermissionTemplateItem::create([
                    'template_id' => $template->id,
                    'permission_id' => $permissionId,
                ]);                              
            }
            $template->load(['permissions', 'creator:id,name,email']);
            DB::commit();
            return $this->success($template, 'Permission template created successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->error('Failed to create permission template: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Display the specified permission template.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $template = PermissionTemplate::where('id', $id)
            ->where('del', '0')
            ->with([
                'permissions' => function($query) {
                    $query->where('del', '0');
                },
                'permissions.module' => function($query) {
                    $query->where('del', '0');
                },
                'permissions.subModule' => function($query) {
                    $query->where('del', '0');
                },
                'creator:id,name,email'
            ])
            ->first();
        if (!$template) {
            return $this->notFound('Permission template not found', 404);
        }
        
        // Manually attach module_or_sub_module for each permission
        if ($template->permissions) {
            foreach ($template->permissions as $permission) {
                if ($permission->module_type === 'sub_module') {
                    $permission->module_or_sub_module = $permission->subModule;
                } else {
                    $permission->module_or_sub_module = $permission->module;
                }
            }
        }
        
        return $this->success($template, 'Permission template retrieved successfully');
    }

    /**
     * Update the specified permission template.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:permission_templates,id',
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'sometimes|required|string|in:active,inactive',
            'permission_ids' => 'sometimes|required|array|min:1',
            'permission_ids.*' => 'required|integer|exists:permissions,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $template = PermissionTemplate::where('id', $request->id)
            ->where('del', '0')
            ->first();
        if (!$template) {
            return $this->notFound('Permission template not found', 404);
        }
        
        // Check if template is attached to any plan that is assigned to customers
        $customerIds = CustomerPermission::where('template_id', $template->id)->where('del', '0')->pluck('customer_id')->toArray();
        if (!empty($customerIds)) {
            // Get all users with these plans
            $usersWithPermission = User::whereIn('id', $customerIds)
                ->where('del', '0')
                ->get();
                
            // Check each user's plan expiration and subscription status
            foreach ($usersWithPermission as $user) {
                $planExpired = false;
                
                // Check if plan is expired
                if ($user->plan_expire_date) {
                    $planExpired = Carbon::parse($user->plan_expire_date)->isPast();
                }
                
                // If plan is not expired, block update
                if (!$planExpired) {
                    return $this->error('This template is already assigned to users with active plans. Please create a new template.', 422);
                }
                
                // If plan is expired, check for active subscription
                if ($planExpired) {
                    $activeSubscription = Subscription::where('user_id', $user->id)
                        ->whereIn('stripe_status', ['active', 'trialing'])
                        ->where(function($query) {
                            $query->whereNull('cancel_at_period_end')
                                  ->orWhere('cancel_at_period_end', false);
                        })
                        ->first();
                    
                    // If user has active subscription even though plan is expired, block update
                    if ($activeSubscription) {
                        return $this->error('This template is already assigned to users with active subscriptions. Please create a new template.', 422);
                    }
                }
            }
            
            // Also check for any active subscriptions that might not have users with active_plan_id set
            $activeSubscriptions = Subscription::whereIn('user_id', $customerIds)
                ->whereIn('stripe_status', ['active', 'trialing'])
                ->where(function($query) {
                    $query->whereNull('cancel_at_period_end')
                          ->orWhere('cancel_at_period_end', false);
                })
                ->count();
            
            if ($activeSubscriptions > 0) {
                return $this->error('This template is already assigned to users with active subscriptions. Please create a new template.', 422);
            }
        }
        
        DB::beginTransaction();
        try {
            $updateData = [];
            if ($request->filled('title')) {
                $updateData['title'] = $request->title;
            }
            if ($request->has('description')) {
                $updateData['description'] = $request->description;
            }
            if ($request->filled('status')) {
                $updateData['status'] = $request->status;
            }
            if (!empty($updateData)) {
                $template->update($updateData);
            }
            if ($request->filled('permission_ids')) {
                PermissionTemplateItem::where('template_id', $template->id)->delete();
                $permissionIds = array_unique($request->permission_ids);
                $existingPermissions = Permission::whereIn('id', $permissionIds)
                    ->where('del', '0')
                    ->pluck('id')
                    ->toArray();
                if (count($existingPermissions) !== count($permissionIds)) {
                    DB::rollBack();
                    return $this->error('One or more permission IDs are invalid or deleted', 422);
                }
                foreach ($permissionIds as $permissionId) {
                    PermissionTemplateItem::create([
                        'template_id' => $template->id,
                        'permission_id' => $permissionId,
                    ]);
                }
            }
            $template->refresh();
            $template->load(['permissions', 'creator:id,name,email']);
            DB::commit();
            return $this->success($template, 'Permission template updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->error('Failed to update permission template: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Remove the specified permission template (soft delete).
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $template = PermissionTemplate::where('id', $id)
            ->where('del', '0')
            ->first();
        if (!$template) {
            return $this->notFound('Permission template not found', 404);
        }
        $assignments = CustomerPermission::where('template_id', $id)
            ->where('del', '0')
            ->count();
        if ($assignments > 0) {
            return $this->error('Cannot delete template. It is assigned to ' . $assignments . ' customers.', 422);
        }
        $template->update(['del' => '1']);
        // // Delete template items
        // PermissionTemplateItem::where('template_id', $id)->delete();
        return $this->message('Permission template deleted successfully');
    }

    /**
     * Change status of the specified permission template.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:permission_templates,id',
            'status' => 'required|string|in:active,inactive',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $template = PermissionTemplate::where('id', $request->id)
            ->where('del', '0')
            ->first();
        if (!$template) {
            return $this->notFound('Permission template not found', 404);
        }
        $template->update(['status' => $request->status]);
        $template->refresh();
        return $this->success($template, 'Permission template status updated successfully');
    }

    /**
     * Assign permission template to customer(s).
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function assignToCustomer(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'template_id' => 'required|integer|exists:permission_templates,id',
            'customer_ids' => 'required|array|min:1',
            'customer_ids.*' => 'required|integer|exists:users,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $template = PermissionTemplate::where('id', $request->template_id)
            ->where('del', '0')
            ->where('status', 'active')
            ->with('permissions')
            ->first();
        if (!$template) {
            return $this->notFound('Permission template not found or inactive', 404);
        }
        if ($template->permissions->isEmpty()) {
            return $this->error('Template has no permissions', 422);
        }
        DB::beginTransaction();
        try {
            $assigned = [];
            $skipped = [];
            foreach ($request->customer_ids as $customerId) {
                // Get user and their workspace_id
                $user = User::where('id', $customerId)->where('del', '0')->first();
                if (!$user) {
                    $skipped[] = $customerId;
                    continue;
                }
                $workspaceId = $user->current_workspace_id ?? $user->workspace_id;
                if (!$workspaceId) {
                    $skipped[] = $customerId;
                    continue;
                }
                
                // Check if customer already has this exact template assigned
                $hasSameTemplate = CustomerPermission::where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('template_id', $request->template_id)
                    ->where('del', '0')
                    ->exists();
                
                if ($hasSameTemplate) {
                    // Same template already assigned - just update existing permissions
                    // Get existing permission IDs for this template
                    $existingPermissionIds = CustomerPermission::where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('template_id', $request->template_id)
                        ->where('del', '0')
                        ->pluck('permission_id')
                        ->toArray();
                    
                    // Get new template permission IDs
                    $newPermissionIds = $template->permissions->pluck('id')->toArray();
                    
                    // Remove permissions that are no longer in the template
                    $permissionsToRemove = array_diff($existingPermissionIds, $newPermissionIds);
                    if (!empty($permissionsToRemove)) {
                        CustomerPermission::where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('template_id', $request->template_id)
                            ->whereIn('permission_id', $permissionsToRemove)
                            ->where('del', '0')
                            ->update(['del' => '1']);
                    }
                    
                    // Add new permissions that don't exist yet
                    $permissionsToAdd = array_diff($newPermissionIds, $existingPermissionIds);
                    foreach ($template->permissions as $permission) {
                        if (in_array($permission->id, $permissionsToAdd)) {
                            CustomerPermission::create([
                                'permission_id' => $permission->id,
                                'template_id' => $request->template_id,
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                                'module_id' => $permission->module_id,
                                'module_type' => $permission->module_type,
                                'status' => 'active',
                                'del' => '0',
                            ]);
                        }
                    }
                } else {
                    // Different template or no template - remove old template permissions first
                    $existingTemplate = CustomerPermission::where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('del', '0')
                        ->whereNotNull('template_id')
                        ->where('template_id', '!=', $request->template_id)
                        ->value('template_id');
                    
                    // If customer has a different template, remove all permissions from that template
                    if ($existingTemplate) {
                        CustomerPermission::where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('template_id', $existingTemplate)
                            ->where('del', '0')
                            ->update(['del' => '1']);
                    }
                    
                    // Assign new template permissions to customer
                    foreach ($template->permissions as $permission) {
                        // Check if customer permission already exists (without template_id or with different template_id)
                        $existingPermission = CustomerPermission::where('permission_id', $permission->id)
                            ->where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('del', '0')
                            ->first();
                        
                        if (!$existingPermission) {
                            CustomerPermission::create([
                                'permission_id' => $permission->id,
                                'template_id' => $request->template_id,
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                                'module_id' => $permission->module_id,
                                'module_type' => $permission->module_type,
                                'status' => 'active',
                                'del' => '0',
                            ]);
                        } else {
                            // Update existing permission to use new template_id
                            $existingPermission->update([
                                'template_id' => $request->template_id,
                                'module_id' => $permission->module_id,
                                'module_type' => $permission->module_type,
                            ]);
                        }
                    }
                }
                $assigned[] = $customerId;
            }
            DB::commit();
            return $this->success($assigned, 'Template assignment completed');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->error('Failed to assign template: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get customers assigned to a template.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAssignedCustomers($id)
    {
        $template = PermissionTemplate::where('id', $id)
            ->where('del', '0')
            ->first();
        if (!$template) {
            return $this->notFound('Permission template not found', 404);
        }
        // Get unique customers who have permissions from this template
        $customerIds = CustomerPermission::where('template_id', $id)
            ->where('del', '0')
            ->distinct()
            ->pluck('customer_id', 'workspace_id')
            ->toArray();
        // Group by customer_id and workspace_id
        $assignments = CustomerPermission::where('template_id', $id)
            ->where('del', '0')
            ->with(['customer:id,name,email', 'workspace:id,name'])
            ->get()
            ->groupBy(['customer_id', 'workspace_id'])
            ->map(function ($group) {
                $first = $group->first();
                return [
                    'customer_id' => $first->customer_id,
                    'workspace_id' => $first->workspace_id,
                    'customer' => $first->customer,
                    'workspace' => $first->workspace,
                    'permissions_count' => $group->count(),
                    'assigned_at' => $first->created_at,
                ];
            })
            ->values();
        return $this->success($assignments, 'Assigned customers retrieved successfully');
    }

}
