<?php

namespace App\Http\Controllers;

use App\Models\EmpCompanyDetails;
use DB;
use App\Models\User;
use Illuminate\Http\Request;
use App\Models\RequiredDocument;
use App\Models\EmpPersonalDetails;
use App\Models\SubcontractorCompany;
use App\Models\SubcontractorRequiredDocument;
use App\Models\SubcontractorEmployeeDocument;
use App\Models\SubcontractorEmployeeInvitation;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\RequiredDocumentField;
use Illuminate\Support\Facades\Validator;

class RequiredDocumentController extends Controller
{
    public function index(Request $request)
    {
        $userTable = $this->getUserTable();
        $query = RequiredDocument::query()->with('requiredDocumentField')->latest('id'); // Eager load the relationship

        if ($userTable === "customer") {
            $query->where('del', '0')
                ->where('customer_id', auth()->id())
                ->where('workspace_id', auth()->user()->current_workspace_id);
        }

        if ($userTable === "emp") {
            $query->where('del', '0')
                ->where('customer_id', auth()->user()->customer_id) // Assuming customer_id links employee to projects
                ->where('workspace_id', auth()->user()->workspace_id);
        }

        // Handle audience filtering based on emp_id parameter
        if ($request->has('emp_id')) {
            // Check if the provided emp_id is internal or external
            $employee = EmpCompanyDetails::withoutGlobalScopes()->find($request->emp_id);

            if (!$employee) {
                return $this->error('Employee not found', 404);
            }

            // Check if the employee belongs to the current user's workspace
            if ($userTable === "customer") {
                if ($employee->customer_id != auth()->id() || $employee->workspace_id != auth()->user()->current_workspace_id) {
                    return $this->error('You do not have access to this employee', 403);
                }
            } elseif ($userTable === "emp") {
                if ($employee->customer_id != auth()->user()->customer_id || $employee->workspace_id != auth()->user()->workspace_id) {
                    return $this->error('You do not have access to this employee', 403);
                }
            }

            $isInternal = ($employee->user_type === '0' || $employee->user_type === 0); // 0 = Internal, 1 = External
            $isExternal = ($employee->user_type === '1' || $employee->user_type === 1); // 1 = External

            if ($isInternal) {
                // For internal employees, return documents with for_internal = true
                $query->where('for_internal', true);
            } elseif ($isExternal) {
                // For external employees, return documents with for_external = true
                $query->where('for_external', true);
            } else {
                // If no conditions match (unknown user_type), return documents that apply to all
                $query->where(function ($q) {
                    $q->where('for_internal', true)
                        ->orWhere('for_external', true);
                });
            }
        }

        // Handle subcontractor filtering
        if ($request->has('subcontractor_id') || $request->get('audience_type') === 'subcontractor') {
            $query->where('for_subcontractor', true);
        }

        // Apply search filter
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', '%' . $searchTerm . '%')
                    ->orWhereHas('requiredDocumentField', function ($subquery) use ($searchTerm) {
                        $subquery->where('field_name', 'like', '%' . $searchTerm . '%');
                    });
            });
        }

        $required_documents = $query->get();

        return $this->success($required_documents, 'Required Documents Get Successfully');
    }

    public function store(Request $request)
    {

        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'title'      => 'required',
            'for_who'    => 'required', // Accepts: 'internal', 'external', 'subcontractor', 'all', or comma-separated like 'internal,external'
            'field_name' => 'required|array',
            'field_type' => 'required|array',
            'field_required' => 'required|array',
            'priority' => 'required|array',
        ]);

        if ($validator->fails()) {

            $error = $validator->errors()->first();

            return $this->error($error, 422);
        } else {

            $validatedData =  $validator->validated();

            // Parse audience flags from for_who input
            $audienceFlags = RequiredDocument::parseAudiences($validatedData['for_who']);

            $RequiredDocumentData = [
                'title'   =>  $validatedData['title'],
                'for_internal' => $audienceFlags['for_internal'],
                'for_external' => $audienceFlags['for_external'],
                'for_subcontractor' => $audienceFlags['for_subcontractor'],
            ];

            $userTable = $this->getUserTable();
            $auth_id = 0;
            $workspace_id = 0;
            if ($userTable === "customer") {
                $auth_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
                $authPersonalDetails = User::where('id', Auth::user()->id)->first();
            }
            if ($userTable === "emp") {
                $auth_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
                $authPersonalDetails = EmpPersonalDetails::where('emp_id', Auth::user()->id)->first();
            }
            $RequiredDocumentData['customer_id'] = $auth_id;
            $RequiredDocumentData['workspace_id'] = $workspace_id;

            $insertGetId =  RequiredDocument::insertGetId($RequiredDocumentData);

            $RequiredDocumentDataArr = [];

            $i = 0;
            foreach ($validatedData['field_name'] as $key => $value) {
                $RequiredDocumentDataArr[$i] = [
                    'field_type' => $validatedData['field_type'][$key],
                    'field_name' => $validatedData['field_name'][$key],
                    'doc_id'     => $insertGetId,
                    'field_required' => $validatedData['field_required'][$key],
                    'priority' => $validatedData['priority'][$key]
                ];

                $i++;
            }

            RequiredDocumentField::insert($RequiredDocumentDataArr);

            // If this document is for subcontractors, mark all subcontractors who have completed
            // their document uploads as needing to upload the new document
            if ($audienceFlags['for_subcontractor']) {
                $affectedCount = SubcontractorCompany::markNewDocsAddedForCompany($auth_id, $workspace_id);
                Log::info("Marked {$affectedCount} subcontractors as needing to upload new required document", [
                    'customer_id' => $auth_id,
                    'workspace_id' => $workspace_id,
                    'document_id' => $insertGetId
                ]);
                
                // Send email notifications to ALL subcontractors (similar to induction notifications)
                // This ensures all subcontractors are notified, regardless of their current status
                try {
                    $document = RequiredDocument::find($insertGetId);
                    $allSubcontractors = SubcontractorCompany::where('customer_id', $auth_id)
                        ->where('workspace_id', $workspace_id)
                        ->where('del', '0')
                        ->with('user')
                        ->get();
                    
                    $brandTitle = \App\Services\CompanyEmailDetails::getBrandTitle($auth_id, $workspace_id);
                    $subject = "New Required Document - {$document->title} | {$brandTitle}";
                    
                    foreach ($allSubcontractors as $subcontractorCompany) {
                        $subcontractor = $subcontractorCompany->user;
                        if ($subcontractor && $subcontractor->email) {
                            \App\Jobs\SendNotificationEmailJob::dispatch([
                                'to' => $subcontractor->email,
                                'subject' => $subject,
                                'template' => 'Emails.subcontractor-new-document-added',
                                'template_data' => [
                                    'subject' => $subject,
                                    'subcontractor_name' => $subcontractor->name ?? $subcontractor->company_name ?? 'there',
                                    'document_title' => $document->title,
                                    'document_description' => $document->identity_text ?? null,
                                    'customer_id' => $auth_id,
                                    'workspace_id' => $workspace_id,
                                ],
                                'customer_id' => $auth_id,
                                'workspace_id' => $workspace_id,
                            ]);
                        }
                    }
                    
                    Log::info("Sent new document notification emails to {$allSubcontractors->count()} subcontractors", [
                        'customer_id' => $auth_id,
                        'workspace_id' => $workspace_id,
                        'document_id' => $insertGetId
                    ]);
                } catch (\Exception $e) {
                    Log::error('Error sending new document notification emails to subcontractors: ' . $e->getMessage(), [
                        'customer_id' => $auth_id,
                        'workspace_id' => $workspace_id,
                        'document_id' => $insertGetId
                    ]);
                    // Don't fail the request if email fails
                }
            }

            // If this document is for external employees (subcontractor employees), mark all invitations
            if ($audienceFlags['for_external']) {
                $affectedInvites = \App\Models\SubcontractorEmployeeInvitation::markNewDocsAddedForCustomer($auth_id, $workspace_id);
                Log::info("Marked {$affectedInvites} subcontractor employees as needing to upload new required document", [
                    'customer_id' => $auth_id,
                    'workspace_id' => $workspace_id,
                    'document_id' => $insertGetId
                ]);
                
                // Send email notifications to ALL accepted employees (similar to induction notifications)
                // This ensures all employees are notified, regardless of their current status
                try {
                    $document = RequiredDocument::find($insertGetId);
                    $allInvitations = \App\Models\SubcontractorEmployeeInvitation::where('customer_id', $auth_id)
                        ->where('workspace_id', $workspace_id)
                        ->where('invitation_status', \App\Models\SubcontractorEmployeeInvitation::STATUS_ACCEPTED)
                        ->with('employee')
                        ->get();
                    
                    $brandTitle = \App\Services\CompanyEmailDetails::getBrandTitle($auth_id, $workspace_id);
                    $subject = "New Required Document - {$document->title} | {$brandTitle}";
                    
                    foreach ($allInvitations as $invitation) {
                        $employee = $invitation->employee;
                        if ($employee && $employee->email) {
                            $employeeName = trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? ''));
                            
                            \App\Jobs\SendNotificationEmailJob::dispatch([
                                'to' => $employee->email,
                                'subject' => $subject,
                                'template' => 'Emails.subcontractor-employee-new-document-added',
                                'template_data' => [
                                    'subject' => $subject,
                                    'employee_name' => $employeeName ?: 'there',
                                    'document_title' => $document->title,
                                    'document_description' => $document->identity_text ?? null,
                                    'customer_id' => $auth_id,
                                    'workspace_id' => $workspace_id,
                                ],
                                'customer_id' => $auth_id,
                                'workspace_id' => $workspace_id,
                            ]);
                        }
                    }
                    
                    Log::info("Sent new document notification emails to {$allInvitations->count()} employees", [
                        'customer_id' => $auth_id,
                        'workspace_id' => $workspace_id,
                        'document_id' => $insertGetId
                    ]);
                } catch (\Exception $e) {
                    Log::error('Error sending new document notification emails to employees: ' . $e->getMessage(), [
                        'customer_id' => $auth_id,
                        'workspace_id' => $workspace_id,
                        'document_id' => $insertGetId
                    ]);
                    // Don't fail the request if email fails
                }
            }

            $required_documents =  RequiredDocument::where('customer_id', $auth_id)
                ->where('workspace_id', $workspace_id)
                ->where('del', '0')->get();

            $data['required_documents'] = $required_documents;
            return $this->success($data, 'Required Documents Store Successfully');
        }
    }

    public function document_edit(Request $request)
    {
        $id = $request->id;
        $userTable = $this->getUserTable();
        $required_documents = RequiredDocument::find($id);
        if (
            $userTable == "customer" && ($required_documents->workspace_id != auth()->user()->current_workspace_id || $required_documents->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Employee Requird Document', 403);
        }

        if ($userTable == "emp" && ($required_documents->customer_id != auth()->user()->customer_id || $required_documents->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Employee Requird Document', 403);
        }
        return $this->success($required_documents, 'Get Employee Requird Document Details Successfully');
    }

    public function update(Request $request)
    {
        $user = auth()->user();
        $validator = Validator::make($request->all(), [
            'title'      => 'required',
            'for_who'    => 'required', // Accepts: 'internal', 'external', 'subcontractor', 'all', or comma-separated like 'internal,external'
            'document_id'    => 'required',
            'field_name' => 'required|array',
            'field_type' => 'required|array',
            'field_required' => 'required|array',
            'status' => 'required|array',
            'has_data' => 'required|array',
            'id_list' => 'required|array',
            'priority' => 'required|array',
        ]);

        if ($validator->fails()) {

            $errors = $validator->errors()->first();
            return $this->error($errors, 422);
        } else {

            $validatedData =  $validator->validated();

            // Check if all arrays have the same length as field_name
            $fieldCount = count($validatedData['field_name']);
            $arrayFields = ['field_type', 'field_required', 'status', 'has_data', 'id_list', 'priority'];

            foreach ($arrayFields as $field) {
                if (count($validatedData[$field]) !== $fieldCount) {
                    return $this->error("Array length mismatch: {$field} must have the same number of elements as field_name", 422);
                }
            }
            $document = RequiredDocument::where('id', $validatedData['document_id'])->first();
            $userTable = $this->getUserTable();

            if (
                $userTable == "customer" && ($document->workspace_id != auth()->user()->current_workspace_id || $document->customer_id != auth()->user()->id)
            ) {
                return $this->message('You do not have access to this document', 403);
            }

            if ($userTable == "emp" && ($document->customer_id != auth()->user()->customer_id || $document->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('You do not have access to this document', 403);
            }
            if (!$document) {
                return $this->notFound('document Not Found', 404);
            }

            // Parse audience flags from for_who input
            $audienceFlags = RequiredDocument::parseAudiences($validatedData['for_who']);

            // Check if subcontractor flag is being newly added (wasn't there before)
            $wasForSubcontractor = $document->for_subcontractor;
            
            RequiredDocument::where('id', $validatedData['document_id'])
                ->update([
                    'title' => $validatedData['title'],
                    'for_internal' => $audienceFlags['for_internal'],
                    'for_external' => $audienceFlags['for_external'],
                    'for_subcontractor' => $audienceFlags['for_subcontractor'],
                ]);

            // If subcontractor flag is newly added, mark subcontractors as needing to upload
            if ($audienceFlags['for_subcontractor'] && !$wasForSubcontractor) {
                $affectedCount = SubcontractorCompany::markNewDocsAddedForCompany($document->customer_id, $document->workspace_id);
                if ($affectedCount > 0) {
                    Log::info("Marked {$affectedCount} subcontractors as needing to upload updated required document", [
                        'customer_id' => $document->customer_id,
                        'workspace_id' => $document->workspace_id,
                        'document_id' => $document->id
                    ]);
                }
            }

            // If this document is for external employees (subcontractor employees)
            // We should mark invitations if:
            // 1. The document is NOW for external (and wasn't before) OR
            // 2. The document IS for external (and was before) - any edit might require re-upload/review
            // Ideally, we should check if meaningful changes occurred, but for now we'll trigger on any update if it targets external
            if ($audienceFlags['for_external']) {
                $affectedInvites = \App\Models\SubcontractorEmployeeInvitation::markNewDocsAddedForCustomer($document->customer_id, $document->workspace_id);
                if ($affectedInvites > 0) {
                    Log::info("Marked {$affectedInvites} subcontractor employees as needing to upload updated required document", [
                        'customer_id' => $document->customer_id,
                        'workspace_id' => $document->workspace_id,
                        'document_id' => $document->id
                    ]);
                }
            }
            $RequiredDocumentDataArr = [];
            
            // Track if any new required fields were added OR existing fields changed from non-required to required
            $hasNewRequiredField = false;
            $hasFieldChangedToRequired = false;

            // Fetch existing fields before update to compare field_required values
            $existingFields = RequiredDocumentField::where('doc_id', $validatedData['document_id'])
                ->whereIn('id', array_filter($validatedData['id_list'], function($id) {
                    return $id !== null && $id != 0;
                }))
                ->pluck('field_required', 'id')
                ->toArray();

            $i = 0;
            foreach ($validatedData['field_name'] as $key => $value) {
                // Safety check: ensure all required array keys exist (allow null for id_list)
                if (
                    !array_key_exists($key, $validatedData['id_list']) ||
                    !isset($validatedData['has_data'][$key]) ||
                    !isset($validatedData['field_type'][$key]) ||
                    !isset($validatedData['field_required'][$key]) ||
                    !isset($validatedData['status'][$key]) ||
                    !isset($validatedData['priority'][$key])
                ) {
                    continue; // Skip this iteration if any required data is missing
                }

                // Check if this is a new field (null or 0 in id_list)
                if ($validatedData['id_list'][$key] === null || $validatedData['id_list'][$key] == 0) {
                    // Check if this new field is required
                    if ($validatedData['field_required'][$key] == 1 || $validatedData['field_required'][$key] === '1') {
                        $hasNewRequiredField = true;
                    }
                    
                    $RequiredDocumentDataArr = [
                        'field_type' => $validatedData['field_type'][$key],
                        'field_name' => $validatedData['field_name'][$key],
                        'doc_id'     => $validatedData['document_id'],
                        'field_required' => $validatedData['field_required'][$key],
                        'status' => $validatedData['status'][$key],
                        'priority' => $validatedData['priority'][$key]
                    ];

                    // Log new field creation for debugging
                    Log::info('Creating new required document field', [
                        'field_name' => $validatedData['field_name'][$key],
                        'field_type' => $validatedData['field_type'][$key],
                        'field_required' => $validatedData['field_required'][$key],
                        'doc_id' => $validatedData['document_id']
                    ]);

                    RequiredDocumentField::create($RequiredDocumentDataArr);
                } else {
                    // Update existing field - check if field_required changed from non-required to required
                    $fieldId = $validatedData['id_list'][$key];
                    $oldFieldRequired = isset($existingFields[$fieldId]) ? (int)$existingFields[$fieldId] : 0;
                    $newFieldRequired = (int)$validatedData['field_required'][$key];
                    
                    // Check if field changed from non-required (0) to required (1)
                    if ($oldFieldRequired == 0 && $newFieldRequired == 1) {
                        $hasFieldChangedToRequired = true;
                        
                        Log::info('Existing field changed from non-required to required', [
                            'field_id' => $fieldId,
                            'field_name' => $validatedData['field_name'][$key],
                            'old_field_required' => $oldFieldRequired,
                            'new_field_required' => $newFieldRequired,
                            'doc_id' => $validatedData['document_id']
                        ]);
                    }
                    
                    $RequiredDocumentDataArr = [
                        'field_type' => $validatedData['field_type'][$key],
                        'field_name' => $validatedData['field_name'][$key],
                        'doc_id'     => $validatedData['document_id'],
                        'field_required' => $validatedData['field_required'][$key],
                        'status' => $validatedData['status'][$key],
                        'priority' => $validatedData['priority'][$key]
                    ];
                    RequiredDocumentField::where(['id' => $validatedData['id_list'][$key]])->update($RequiredDocumentDataArr);
                    RequiredDocumentField::where('doc_id', $validatedData['document_id'])->whereNotIn('id', $validatedData['id_list'])->delete();
                }
                $i++;
            }

            // If a new required field was added OR an existing field changed from non-required to required,
            // update statuses for submitted/approved documents
            if ($hasNewRequiredField || $hasFieldChangedToRequired) {
                $documentId = $validatedData['document_id'];
                $customerId = $document->customer_id;
                $workspaceId = $document->workspace_id;

                // Handle subcontractor documents if this document is for subcontractors
                if ($audienceFlags['for_subcontractor']) {
                    // Find all subcontractors who have uploaded documents for this required document
                    $subcontractorDocs = SubcontractorRequiredDocument::where('required_document_id', $documentId)
                        ->where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('del', '0')
                        ->whereIn('approval_status', [
                            SubcontractorRequiredDocument::STATUS_APPROVED,
                            SubcontractorRequiredDocument::STATUS_PENDING
                        ])
                        ->get()
                        ->groupBy('subcontractor_id');

                    foreach ($subcontractorDocs as $subcontractorId => $docs) {
                        // Update approval_status from 'approved' to 'pending' for approved documents
                        $approvedDocs = $docs->where('approval_status', SubcontractorRequiredDocument::STATUS_APPROVED);
                        if ($approvedDocs->isNotEmpty()) {
                            SubcontractorRequiredDocument::whereIn('id', $approvedDocs->pluck('id'))
                                ->update([
                                    'approval_status' => SubcontractorRequiredDocument::STATUS_PENDING,
                                    'approved_by' => null,
                                    'approved_at' => null,
                                    'rejection_reason' => null
                                ]);
                        }

                        // Update SubcontractorCompany status to "new docs added" if it was complete or pending
                        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
                            ->where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('del', '0')
                            ->first();

                        if ($subcontractorCompany && in_array($subcontractorCompany->required_docs_status, [
                            SubcontractorCompany::STATUS_COMPLETE,
                            SubcontractorCompany::STATUS_PENDING
                        ])) {
                            $previousStatus = $subcontractorCompany->required_docs_status;
                            $subcontractorCompany->update([
                                'required_docs_status' => SubcontractorCompany::STATUS_NEW_DOCS_ADDED
                            ]);

                            Log::info('Updated subcontractor company status to new docs added', [
                                'subcontractor_id' => $subcontractorId,
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                                'document_id' => $documentId,
                                'previous_status' => $previousStatus
                            ]);
                        }
                    }

                    $triggerReason = $hasNewRequiredField && $hasFieldChangedToRequired 
                        ? 'new required field added and existing field changed to required'
                        : ($hasNewRequiredField ? 'new required field added' : 'existing field changed to required');
                    
                    Log::info('Processed subcontractor documents for required field change', [
                        'document_id' => $documentId,
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId,
                        'affected_subcontractors' => $subcontractorDocs->count(),
                        'trigger_reason' => $triggerReason
                    ]);
                }

                // Handle external employee documents if this document is for external employees
                if ($audienceFlags['for_external']) {
                    // Find all employees who have uploaded documents for this required document
                    $employeeDocs = SubcontractorEmployeeDocument::where('required_document_id', $documentId)
                        ->where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('del', '0')
                        ->whereIn('approval_status', [
                            SubcontractorEmployeeDocument::STATUS_APPROVED,
                            SubcontractorEmployeeDocument::STATUS_PENDING
                        ])
                        ->get()
                        ->groupBy('employee_id');

                    foreach ($employeeDocs as $employeeId => $docs) {
                        // Update approval_status from 'approved' to 'pending' for approved documents
                        $approvedDocs = $docs->where('approval_status', SubcontractorEmployeeDocument::STATUS_APPROVED);
                        if ($approvedDocs->isNotEmpty()) {
                            SubcontractorEmployeeDocument::whereIn('id', $approvedDocs->pluck('id'))
                                ->update([
                                    'approval_status' => SubcontractorEmployeeDocument::STATUS_PENDING,
                                    'approved_by' => null,
                                    'approved_at' => null,
                                    'rejection_reason' => null
                                ]);
                        }

                        // Update SubcontractorEmployeeInvitation status to "new docs added" if it was complete or pending
                        $invitations = SubcontractorEmployeeInvitation::where('employee_id', $employeeId)
                            ->where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('invitation_status', SubcontractorEmployeeInvitation::STATUS_ACCEPTED)
                            ->whereIn('required_docs_status', [
                                SubcontractorEmployeeInvitation::DOCS_STATUS_COMPLETE,
                                SubcontractorEmployeeInvitation::DOCS_STATUS_PENDING
                            ])
                            ->get();

                        foreach ($invitations as $invitation) {
                            $previousStatus = $invitation->required_docs_status;
                            $invitation->update([
                                'required_docs_status' => SubcontractorEmployeeInvitation::DOCS_STATUS_NEW_DOCS_ADDED
                            ]);

                            Log::info('Updated employee invitation status to new docs added', [
                                'employee_id' => $employeeId,
                                'invitation_id' => $invitation->id,
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                                'document_id' => $documentId,
                                'previous_status' => $previousStatus
                            ]);
                        }
                    }

                    $triggerReason = $hasNewRequiredField && $hasFieldChangedToRequired 
                        ? 'new required field added and existing field changed to required'
                        : ($hasNewRequiredField ? 'new required field added' : 'existing field changed to required');
                    
                    Log::info('Processed employee documents for required field change', [
                        'document_id' => $documentId,
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId,
                        'affected_employees' => $employeeDocs->count(),
                        'trigger_reason' => $triggerReason
                    ]);
                }
            }

            $userTable = $this->getUserTable();
            $auth_id = 0;
            $workspace_id = 0;
            if ($userTable === "customer") {
                $auth_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
                $authPersonalDetails = User::where('id', Auth::user()->id)->first();
            }
            if ($userTable === "emp") {
                $auth_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
                $authPersonalDetails = EmpPersonalDetails::where('emp_id', Auth::user()->id)->first();
            }


            $required_documents =  RequiredDocument::where('customer_id', $auth_id)
                ->where('workspace_id', $workspace_id)
                ->where('del', '0')->get();

            $data['required_documents'] = $required_documents;

            return $this->success($data, 'Required Documents Updated Successfully');
        }
    }

    public function document_delete(Request $request)
    {
        $id = $request->id;
        $userTable = $this->getUserTable();
        $document = RequiredDocument::find($id);

        if ($userTable == "customer" && ($document->workspace_id != auth()->user()->current_workspace_id || $document->customer_id != auth()->user()->id)) {
            return $this->message('You do not have access to this Document', 403);
        }

        if ($userTable == "emp" && ($document->customer_id != auth()->user()->customer_id || $document->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Document', 403);
        }
        $document->del = '1';
        $document->save();

        $user = auth()->user();
        if ($userTable == "customer") {
            $data['document'] = RequiredDocument::where('customer_id', $user->id)
                ->where('workspace_id', $user->current_workspace_id)
                ->where('del', '0')
                ->get();
        } elseif ($userTable == "emp") {
            $data['document'] = RequiredDocument::where('customer_id', auth()->user()->customer_id)
                ->where('workspace_id', auth()->user()->workspace_id)
                ->where('del', '0')
                ->get();
        }

        return $this->success($data, 'Required Documents Deleted Successfully');
    }


    public function delete_field(Request $request)
    {
        RequiredDocumentField::where('id', $request->id)->delete();

        return $this->message('Field Deleted Successfully');
    }
}
