<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sites;
use App\Models\SiteDocument;
use App\Models\SiteDocumentSignature;
use App\Models\DocumentType;
use App\Models\LinkManagement;
use App\Models\Company;
use App\Models\Project;
use App\Models\EmpCompanyDetails;
use App\Models\SiteHistory;
use App\Models\EmpPersonalDetails;
use App\Models\User;
use App\Models\Role;
use App\Models\EmployeeAttendance;
use App\Models\RosterAssign;
use App\Models\SubcontractorCompany;
use App\Models\ProjectSite;
use App\Models\EmployeeSubcontractor;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;


class SiteController extends Controller
{
    public function __construct()
    {
        $this->middleware('Permissions:Sites Maintain')->only(['create', 'edit']);
    }
    public function index(Request $request)
    {
        $user = Auth::user();
        $userTable = $this->getUserTable();
        $isSubcontractor = ($userTable === 'customer' && $user instanceof User && $user->user_type == config('constants.user_types.subcontractor'));
        
        $query = Sites::where('del', 0);

        // When active=0 is passed, return ALL sites (both active and inactive) without filtering by active status
        // When active is not provided or active=1, return only active sites
        if($request->has('active') && (int)$request->active === 0){
            // Return all sites without active status filter
        }else{
            // Default: only return active sites
            $query->where('active', 1);
        }
        
        // For subcontractors: Filter by sites from assigned projects through project_sites table
        // For others: Apply customer/workspace filter as usual
        if ($isSubcontractor) {
            // Get all company associations for this subcontractor
            $subcontractorCompanies = SubcontractorCompany::where('user_id', $user->id)
                ->where('del', '0')
                ->get();
            
            if ($subcontractorCompanies->isEmpty()) {
                return $this->withCount(collect([]), 'No company associations found for this subcontractor.');
            }
            
            // Collect all project IDs from all company associations
            $allProjectIds = [];
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                $projectIds = $subcontractorCompany->project_ids ?? [];
                $allProjectIds = array_merge($allProjectIds, $projectIds);
            }
            
            // Remove duplicates
            $allProjectIds = array_unique($allProjectIds);
            
            if (empty($allProjectIds)) {
                return $this->withCount(collect([]), 'No projects assigned to this subcontractor.');
            }
            
            // Get all sites that belong to these projects through project_sites table
            $siteIds = ProjectSite::whereIn('project_id', $allProjectIds)
                ->whereHas('siteDetails', function($query) {
                    $query->where('del', '0');
                })
                ->pluck('site_id')
                ->unique()
                ->toArray();
            
            if (empty($siteIds)) {
                return $this->withCount(collect([]), 'No sites found for assigned projects.');
            }
            
            // Filter sites by those from assigned projects
            $query->whereIn('id', $siteIds);
        } else {
            // For regular customers/employees: Apply customer/workspace filter
            $query = $this->applyCustomerWorkspaceFilter($query);
        }
        
        $query->with('supervisor')->with('Company')->with('project');
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', '%' . $searchTerm . '%')
                ->orWhere('site_state', 'like', '%' . $searchTerm . '%')
                ->orWhere('street_address', 'like', '%' . $searchTerm . '%')
                ->orWhere(function ($nameQuery) use ($searchTerm) {
                    $this->applyNameSearch($nameQuery, $searchTerm, 'supervisor');
                })
                ->orWhereHas('company', function ($subquery) use ($searchTerm) {
                    $subquery->where('name', 'like', '%' . $searchTerm . '%');
                })
                ->orWhereHas('project', function ($subquery) use ($searchTerm) {
                    $subquery->where('title', 'like', '%' . $searchTerm . '%');
                });
            });
        }
        
        // Get sites collection (before pagination) to collect customer/workspace pairs
        $allSites = $query->get();
        
        // Make customer_id and workspace_id visible temporarily for processing
        $allSites->makeVisible(['customer_id', 'workspace_id']);
        
        // Collect all unique customer_id and workspace_id pairs from the sites
        $customerWorkspacePairs = [];
        foreach ($allSites as $site) {
            $customerId = $site->customer_id;
            $workspaceId = $site->workspace_id;
            if ($customerId && $workspaceId) {
                $key = $customerId . '_' . $workspaceId;
                if (!isset($customerWorkspacePairs[$key])) {
                    $customerWorkspacePairs[$key] = [
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId
                    ];
                }
            }
        }
        
        // Fetch company_name from adminsettings for all customer/workspace pairs in bulk
        $companyNamesMap = $this->getCompanyNamesBulk($customerWorkspacePairs);
        
        // Transform sites to add company name before pagination
        $allSites->transform(function ($site) use ($companyNamesMap) {
            $companyName = null;
            if ($site->customer_id && $site->workspace_id) {
                $key = $site->customer_id . '_' . $site->workspace_id;
                $companyName = $companyNamesMap[$key] ?? null;
            }
            $site->company_name = $companyName;
            return $site;
        });
        
        // Get response from withCount (it will handle pagination)
        $response = $this->withCount($query, 'Data get Successfully');
        
        // Transform the paginated response to add company names
        if ($response->getStatusCode() === 200) {
            $responseData = json_decode($response->getContent(), true);
            $sitesData = $responseData['data'] ?? [];
            
            if (!empty($sitesData)) {
                // Create a map of site IDs to company names from our transformed collection
                $siteCompanyMap = [];
                foreach ($allSites as $site) {
                    $siteCompanyMap[$site->id] = $site->company_name ?? null;
                }
                
                // Add company name to each site in the paginated response
                foreach ($sitesData as &$site) {
                    $siteId = $site['id'] ?? null;
                    if ($siteId && isset($siteCompanyMap[$siteId])) {
                        $site['company_name'] = $siteCompanyMap[$siteId];
                    } else {
                        $site['company_name'] = null;
                    }
                }
                unset($site); // Break reference
                
                $responseData['data'] = $sitesData;
                return response()->json($responseData, 200);
            }
        }
        
        return $response;
    }
    
 

    public function create(Request $request)
    {
        $get_employes = EmpCompanyDetails::with('EmpPersonalDetails')->where('del', 0)->where('compeleted', 1)->where('status', 1)->where('access_role', 'SPV')->get();
        $get_companies = Company::where('del', 0)->where('active', 1)->get();
        return view('Sites.create', compact('get_employes', 'get_companies'));
    }

    public function ssoList()
    {
        $userTable = $this->getUserTable();
        // Base query for employee list
        $query = EmpCompanyDetails::where('compeleted', '1')
            ->where('approved', '1')
            ->where('status', '1')
            ->where('del', '0')
            ->with([
                'empPersonalDetails' => function ($query) {
                    $query->select('emp_id', 'first_name', 'middle_name', 'last_name', 'customer_id', 'workspace_id');
                },
                'accessRole' => function ($query) {
                    $query->select('id', 'title', 'code');
                },
                'empTier' => function ($query) {
                    $query->select('id', 'title');
                },
                'accessTier' => function ($query) {
                    $query->select('id', 'title', 'tier_key');
                },
            ])
            ->select('id', 'access_role', 'tier_id');
        // Add Customer and Employee Access Checks
        if ($userTable === 'customer') {
            $query->whereHas('empPersonalDetails', function ($q) {
                $q->where('customer_id', auth()->id())
                    ->where('workspace_id', auth()->user()->current_workspace_id);
            });
        }
        if ($userTable === 'emp') {
            $query->whereHas('empPersonalDetails', function ($q) {
                $q->where('customer_id', auth()->user()->customer_id)
                    ->where('workspace_id', auth()->user()->workspace_id);
            });
        }
        // Execute query
        $empList = $query->get();
        // Filter only SSO roles
        $data = $empList->where('access_role', 'SSO')->values()->toArray();
        // Return response
        return $this->success($data, 'SSO List fetched successfully');
    }

    public function store(Request $request)
    {
       
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'state' => 'required',
            'external_id' => 'nullable',
            'supervisor' => 'nullable',
            'company_client' => 'nullable',
            'description' => 'nullable',
            'address' => 'required',
            'latitude' => 'required',
            'longitude' => 'required',
            'area_radius' => 'required',
            'project' => 'nullable',
            'forman_id' => 'nullable',
            'sso_id'=>'nullable'
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        try {
            $userTable = $this->getUserTable();
            $auth_id = 0;
            $workspace_id = 0;
            $created_by = 0;
            if ($userTable === "customer") {
                $auth_id = Auth::user()->id;
                $workspace_id = Auth::user()->current_workspace_id;
            }
            if ($userTable === "emp") {
                $auth_id = auth()->user()->customer_id;
                $workspace_id = auth()->user()->workspace_id;
                $created_by = auth()->user()->id;
            }
            $validatedData = $validator->validated();
            $Data = Sites::create([
                'title' => $validatedData['title'],
                'site_state' => $validatedData['state'],
                'external_id' => $validatedData['external_id'],
                'supervisor_id' => $validatedData['supervisor'],
                'company_client' => $validatedData['company_client'],
                'forman_id' => $validatedData['forman_id'],
                'sso_id' => $validatedData['sso_id'],   
                'description' => $validatedData['description'],
                'street_address' => $validatedData['address'],
                'latitude' => $validatedData['latitude'],
                'longitude' => $validatedData['longitude'],
                'area_radius' => $validatedData['area_radius'],
                'project_id' => $validatedData['project'],
                'customer_id' => $auth_id,
                'workspace_id' => $workspace_id,
                'created_by' => $created_by
            ]);
            return $this->success($Data, 'Site Created Successfully');
        } catch (QueryException $exception) {
            $errorMessage = $exception->getMessage();
            preg_match("/'([^']+)'/", $errorMessage, $matches);
            $errorField = count($matches) > 1 ? $matches[1] : 'unknown';
            return $this->message('The ' . $errorField . ' field is required.', 500);
        }
    }


    public function siteEdit($id)
    {
        $site = Sites::with(['emppersonaldetails', 'company','siteSafetyOfficer'])->where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not Found.', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        return $this->success($site, 'Site data retrieved successfully');
    }

    public function siteUpdate(Request $request, $id)
    {

        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'state' => 'required',
            'external_id' => 'nullable',
            'supervisor' => 'nullable',
            'company_client' => 'nullable',
            'description' => 'nullable',
            'address' => 'required',
            'latitude' => 'required',
            'longitude' => 'required',
            'area_radius' => 'required',
            'project' => 'nullable',
            'forman_id' => 'nullable',
            'sso_id'=>'nullable'
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        } else {

            $validatedData = $validator->validated();
            $site = Sites::where('id', $id)->where('del', '0')->first();
            if (!$site) {
                return $this->message('Site not found or already deleted', 404);
            }
            $updateData = [
                'title' => $validatedData['title'],
                'site_state' => $validatedData['state'],
                'external_id' => $validatedData['external_id'],
                'supervisor_id' => $validatedData['supervisor'],
                'company_client' => $validatedData['company_client'],
                'description' => $validatedData['description'],
                'street_address' => $validatedData['address'],
                'latitude' => $validatedData['latitude'],
                'longitude' => $validatedData['longitude'],
                'area_radius' => $validatedData['area_radius'],
                'forman_id' => $validatedData['forman_id'],
                'sso_id' => $validatedData['sso_id'],
                // 'project_id' => $validatedData['project']
            ];
            $userTable = $this->getUserTable();
            if (
                $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
            ) {
                return $this->message('You do not have access to this Site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('You do not have access to this Site', 403);
            }
            $site->update($updateData);
            $getuserdetail = DB::table('users')->where('id', auth()->user()->id)->first();
            $description = "<a href='" . url('/user-profile/' . Auth::id()) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'>" . e($getuserdetail->name ?? '') . "</a> changed the site detail: '" . e($site->title) . "'.";
            $history = [
                'site_id' => $site->id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);
            return $this->success($updateData, 'Site Updated Successfully');
        }
    }

    public function updateStatus($id)
    {
        $site = Sites::where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not found', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        // Toggle the active status
        $site->active = $site->active == 1 ? 0 : 1;
        $site->save();
        return $this->success($site, 'Status Updated Successfully');
    }

    public function destroy($id)
    {
        $site = Sites::where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not found or already deleted', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        Sites::where('id', $id)->update(['del' => '1']);
        $getuserdetail = DB::table('users')->where('id', auth()->user()->id)->first();
        if (!$getuserdetail) {
            return $this->message('User details not found');
        }
        $description = "<a href='" . url('/') . "/user-profile/" . Auth::user()->id . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$getuserdetail->name} </a>  Deleted that site: '{$site->title}'.";

        $history = [
            'site_id' => $site->id,
            'description' => $description,
            'updated_by' => auth()->user()->id,
        ];
        SiteHistory::create($history);
        return $this->message('User details not found', 200);
    }


    public function view($id)
    {
        $site = Sites::with('siteSafetyOfficer','supervisor','forman')->where('id', $id)->where('del', '0')->first();
        if (!$site) {
            return $this->message('Site not found or deleted', 404);
        }
        $userTable = $this->getUserTable();
        if (
            $userTable == "customer" && ($site->workspace_id != auth()->user()->current_workspace_id || $site->customer_id != auth()->user()->id)
        ) {
            return $this->message('You do not have access to this Site', 403);
        }

        if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('You do not have access to this Site', 403);
        }
        // $site = Sites::where('id', $id)->first();
        $sitehistories = SiteHistory::where('site_id', $id)->get();
        $company = Company::where('id', $site->company_client)->first();
        $project = Project::where('id', $site->project_id)->first();

        if ($site->created_by == 0) {
            $created_by = User::where('id', $site->customer_id)->select('name as name')->first();
        }else {
            $created_by = EmpPersonalDetails::where('emp_id', $site->created_by)->first();
        }
        
        $empsuper = EmpPersonalDetails::where('emp_id', $site->supervisor_id)->first();

        $site['company'] = $company;
        $site['project'] = $project;
        $site['created_by'] = $created_by;
        $site['sitehistories'] = $sitehistories;
        
        // Load site documents with roles
        $siteDocuments = SiteDocument::where('site_id', $id)
            ->orderBy('created_at', 'desc')
            ->get();
        
            // Load role details, document type, and subcontractor for each document
        foreach ($siteDocuments as $doc) {
            if ($doc->role_ids && is_array($doc->role_ids)) {
                $doc->roles = Role::whereIn('id', $doc->role_ids)->get();
            } else {
                $doc->roles = collect([]);
            }
            
            // Load document type relationship
            if ($doc->document_type) {
                $doc->documentTypeDetail = DocumentType::find($doc->document_type);
            }

            // Load subcontractor relationship
            if ($doc->subcontractors) {
                $subcontractorIds = is_array($doc->subcontractors) ? $doc->subcontractors : [$doc->subcontractors];
                
                $subcontractorUsers = User::whereIn('id', $subcontractorIds)
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0')
                    ->select('id', 'name', 'company_name', 'email', 'mobile_number')
                    ->get();
                
                $doc->subcontractorDetail = $subcontractorUsers->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'title' => $user->company_name ?? $user->name,
                        'name' => $user->name,
                        'company_name' => $user->company_name,
                        'email' => $user->email,
                        'mobile_number' => $user->mobile_number,
                    ];
                });
            }
        }
        
        $site['documents'] = $siteDocuments;

        return response()->json([
            'data' => $site
        ]);
        // return view('Sites.view', compact('site','company','emppersonal','empsuper','sitehistories'));
    }

    /**
     * Create/Store a site document
     */
    public function siteDocument(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255',
                'site_id' => 'required|integer|exists:sites,id',
                'document_type' => 'required|integer|exists:document_types,id',
                'roles' => 'nullable|array', // Array of role IDs
                'roles.*' => 'integer|exists:roles,id',
                'subcontractors' => 'nullable|array',
                'subcontractors.*' => 'integer|exists:users,id',
                'sign_requires' => 'nullable|integer|in:0,1',
                'signature_timing' => 'nullable|integer|in:0,1'
            ]);

            if ($validator->fails()) {
                return $this->message($validator->errors(), 422);
            }

            $userTable = $this->getUserTable();
            $site = Sites::find($request->site_id);
            
            if (!$site) {
                return $this->message('Site not found', 404);
            }

            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Validate subcontractors are actual subcontractors and belong to this customer/workspace
            if ($request->has('subcontractors') && is_array($request->subcontractors) && count($request->subcontractors) > 0) {
                $customerId = $site->customer_id;
                $workspaceId = $site->workspace_id;
                
                // Get valid subcontractor IDs for this customer/workspace
                $validSubcontractorIds = SubcontractorCompany::where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->pluck('user_id')
                    ->toArray();
                
                // Verify all provided subcontractor IDs are valid subcontractors
                $invalidSubcontractors = [];
                foreach ($request->subcontractors as $index => $subcontractorId) {
                    // Check if user exists and is a subcontractor
                    $user = User::where('id', $subcontractorId)
                        ->where('user_type', config('constants.user_types.subcontractor'))
                        ->where('del', '0')
                        ->first();
                    
                    if (!$user || !in_array($subcontractorId, $validSubcontractorIds)) {
                        $invalidSubcontractors[] = $index;
                    }
                }
                
                if (!empty($invalidSubcontractors)) {
                    $errors = [];
                    foreach ($invalidSubcontractors as $index) {
                        $errors["subcontractors.{$index}"] = ["The selected subcontractors.{$index} is invalid."];
                    }
                    return $this->message($errors, 422);
                }
            }

            // Handle file upload
            $document = null;
            if ($request->hasFile('file')) {
                $document = $this->handleFileImageUpload($request, 'SiteDocuments');
            }

            // Prepare data for mass assignment
            $documentData = [
                'title' => $request->title,
                'site_id' => $request->site_id,
                'document_type' => $request->document_type ?? null,
                'uploaded_by' => auth()->user()->id,
                'uploaded_by_type' => $userTable, // Set 'customer' or 'emp'
                'sign_requires' => $request->sign_requires ?? null,
                'signature_timing' => $request->signature_timing ?? null,
                'file' => $document['path'] ?? null,
                'role_ids' => $request->has('roles') && is_array($request->roles) ? $request->roles : null,
                'subcontractors' => $request->has('subcontractors') && is_array($request->subcontractors) ? $request->subcontractors : null,
            ];

            // Create site document using mass assignment (this will properly apply JSON casts)
            $siteDocument = SiteDocument::create($documentData);

            // Get uploaded by user info
            if ($userTable === 'emp') {
                $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                    ->select('first_name', 'middle_name', 'last_name')
                    ->first();
                $fullName = trim(($added_by->first_name ?? '') . " " . ($added_by->middle_name ?? '') . " " . ($added_by->last_name ?? ''));
            } else {
                $added_by = User::where('id', Auth::user()->id)->select('name')->first();
                $fullName = $added_by->name ?? '';
            }

            // Create site history entry
            $description = "<a href='" . url('/user-profile/' . Auth::user()->id) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$fullName} </a> uploaded a new document: '{$siteDocument->title}' to site: {$site->title}.";
            $history = [
                'site_id' => $siteDocument->site_id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);

            // Load role details, document type, and subcontractor for the document
            if ($siteDocument->role_ids && is_array($siteDocument->role_ids)) {
                $siteDocument->roles = Role::whereIn('id', $siteDocument->role_ids)->get();
            } else {
                $siteDocument->roles = collect([]);
            }
            
            // Load document type relationship
            if ($siteDocument->document_type) {
                $siteDocument->documentTypeDetail = DocumentType::find($siteDocument->document_type);
            }

            // Load subcontractor relationships
            if ($siteDocument->subcontractors && is_array($siteDocument->subcontractors)) {
                $subcontractorUsers = User::whereIn('id', $siteDocument->subcontractors)
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0')
                    ->select('id', 'name', 'company_name', 'email', 'mobile_number')
                    ->get();
                
                $siteDocument->subcontractorsDetail = $subcontractorUsers->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'title' => $user->company_name ?? $user->name,
                        'name' => $user->name,
                        'company_name' => $user->company_name,
                        'email' => $user->email,
                        'mobile_number' => $user->mobile_number,
                    ];
                });
            } else {
                $siteDocument->subcontractorsDetail = collect([]);
            }

            return $this->success($siteDocument, 'Site document saved successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error saving site document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get all documents for a site
     */
    public function siteDocuments(Request $request, $site_id)
    {
        try {
            $site = Sites::find($site_id);
            
            if (!$site) {
                return $this->message('Site not found', 404);
            }

            $userTable = $this->getUserTable();
            
            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Get documents with uploaded by info and roles
            $documents = SiteDocument::where('site_id', $site_id)
                ->orderBy('created_at', 'desc')
                ->get();

            // Add uploaded_by user information and roles
            foreach ($documents as $doc) {
                // Get user info based on uploaded_by_type
                if ($doc->uploaded_by_type == 'customer') {
                    $user = User::find($doc->uploaded_by);
                    if ($user) {
                        $doc->uploaded_by_name = $user->name;
                    }
                } else {
                    $emp = EmpPersonalDetails::where('emp_id', $doc->uploaded_by)->first();
                    if ($emp) {
                        $doc->uploaded_by_name = trim(($emp->first_name ?? '') . " " . ($emp->middle_name ?? '') . " " . ($emp->last_name ?? ''));
                    }
                }

                // Load roles from role_ids JSON array
                if ($doc->role_ids && is_array($doc->role_ids)) {
                    $doc->roles = Role::whereIn('id', $doc->role_ids)->get();
                } else {
                    $doc->roles = collect([]);
                }
                
                // Load document type relationship
                if ($doc->document_type) {
                    $doc->documentTypeDetail = DocumentType::find($doc->document_type);
                }

                // Load subcontractor relationships
                if ($doc->subcontractors && is_array($doc->subcontractors)) {
                    $subcontractorUsers = User::whereIn('id', $doc->subcontractors)
                        ->where('user_type', config('constants.user_types.subcontractor'))
                        ->where('del', '0')
                        ->select('id', 'name', 'company_name', 'email', 'mobile_number')
                        ->get();
                    
                    $doc->subcontractorsDetail = $subcontractorUsers->map(function ($user) {
                        return [
                            'id' => $user->id,
                            'title' => $user->company_name ?? $user->name,
                            'name' => $user->name,
                            'company_name' => $user->company_name,
                            'email' => $user->email,
                            'mobile_number' => $user->mobile_number,
                        ];
                    });
                } else {
                    $doc->subcontractorsDetail = collect([]);
                }
            }

            return $this->success($documents, 'Site documents retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error retrieving site documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Update a site document
     */
    public function updateSiteDocument(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'document_id' => 'required|exists:site_documents,id',
                'title' => 'required|string|max:255',
                'document_type' => 'required|integer|exists:document_types,id',
                'roles' => 'nullable|array',
                'roles.*' => 'integer|exists:roles,id',
                'subcontractors' => 'nullable|array',
                'subcontractors.*' => 'integer|exists:users,id'
            ]);

            if ($validator->fails()) {
                return $this->message($validator->errors(), 422);
            }

            $siteDocument = SiteDocument::find($request->document_id);
            
            if (!$siteDocument) {
                return $this->message('Site document not found', 404);
            }

            $site = Sites::find($siteDocument->site_id);
            
            if (!$site) {
                return $this->message('Site not found for the document', 404);
            }

            $userTable = $this->getUserTable();
            
            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Validate subcontractors are actual subcontractors and belong to this customer/workspace
            if ($request->has('subcontractors') && is_array($request->subcontractors) && count($request->subcontractors) > 0) {
                $customerId = $site->customer_id;
                $workspaceId = $site->workspace_id;
                
                // Get valid subcontractor IDs for this customer/workspace
                $validSubcontractorIds = SubcontractorCompany::where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->pluck('user_id')
                    ->toArray();
                
                // Verify all provided subcontractor IDs are valid subcontractors
                $invalidSubcontractors = [];
                foreach ($request->subcontractors as $index => $subcontractorId) {
                    // Check if user exists and is a subcontractor
                    $user = User::where('id', $subcontractorId)
                        ->where('user_type', config('constants.user_types.subcontractor'))
                        ->where('del', '0')
                        ->first();
                    
                    if (!$user || !in_array($subcontractorId, $validSubcontractorIds)) {
                        $invalidSubcontractors[] = $index;
                    }
                }
                
                if (!empty($invalidSubcontractors)) {
                    $errors = [];
                    foreach ($invalidSubcontractors as $index) {
                        $errors["subcontractors.{$index}"] = ["The selected subcontractors.{$index} is invalid."];
                    }
                    return $this->message($errors, 422);
                }
            }

            $oldTitle = $siteDocument->title;

            // Update document fields
            $siteDocument->title = $request->title;
            if ($request->has('document_type')) {
                $siteDocument->document_type = $request->document_type;
            }
            if ($request->has('sign_requires')) {
                $siteDocument->sign_requires = $request->sign_requires;
            }
            if ($request->has('signature_timing')) {
                $siteDocument->signature_timing = $request->signature_timing;
            }

            // Handle file upload if provided
            if ($request->hasFile('file')) {
                // Delete old file if exists
                if ($siteDocument->file && file_exists(public_path($siteDocument->file))) {
                    unlink(public_path($siteDocument->file));
                }
                $document = $this->handleFileImageUpload($request, 'SiteDocuments');
                $siteDocument->file = $document['path'] ?? null;
            }

            // Update role IDs as JSON array if provided
            if ($request->has('roles') && is_array($request->roles)) {
                $siteDocument->role_ids = $request->roles;
            }

            // Update subcontractor IDs as JSON array if provided
            if ($request->has('subcontractors') && is_array($request->subcontractors)) {
                $siteDocument->subcontractors = $request->subcontractors;
            }

            $siteDocument->save();

            // Get user info for history
            if ($userTable === 'emp') {
                $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                    ->select('first_name', 'middle_name', 'last_name')
                    ->first();
                $fullName = trim(($added_by->first_name ?? '') . " " . ($added_by->middle_name ?? '') . " " . ($added_by->last_name ?? ''));
            } else {
                $added_by = User::where('id', Auth::user()->id)->select('name')->first();
                $fullName = $added_by->name ?? '';
            }

            // Create site history entry
            $description = "<a href='" . url('/user-profile/' . Auth::user()->id) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$fullName} </a> updated the document title from: '{$oldTitle}' to: '{$request->title}'.";
            $history = [
                'site_id' => $site->id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);

            // Load role details, document type, and subcontractor for the document
            if ($siteDocument->role_ids && is_array($siteDocument->role_ids)) {
                $siteDocument->roles = Role::whereIn('id', $siteDocument->role_ids)->get();
            } else {
                $siteDocument->roles = collect([]);
            }
            
            // Load document type relationship
            if ($siteDocument->document_type) {
                $siteDocument->documentTypeDetail = DocumentType::find($siteDocument->document_type);
            }

            // Load subcontractor relationships
            if ($siteDocument->subcontractors && is_array($siteDocument->subcontractors)) {
                $subcontractorUsers = User::whereIn('id', $siteDocument->subcontractors)
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0')
                    ->select('id', 'name', 'company_name', 'email', 'mobile_number')
                    ->get();
                
                $siteDocument->subcontractorsDetail = $subcontractorUsers->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'title' => $user->company_name ?? $user->name,
                        'name' => $user->name,
                        'company_name' => $user->company_name,
                        'email' => $user->email,
                        'mobile_number' => $user->mobile_number,
                    ];
                });
            } else {
                $siteDocument->subcontractorsDetail = collect([]);
            }

            return $this->success($siteDocument, 'Site document updated successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error updating site document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Delete a site document
     */
    public function deleteSiteDocument(Request $request)
    {
        try {
            $siteDocument = SiteDocument::find($request->id);
            
            if (!$siteDocument) {
                return $this->message('Document not found', 404);
            }

            $site = Sites::find($siteDocument->site_id);
            
            if (!$site) {
                return $this->message('Site not found for the document', 404);
            }

            $userTable = $this->getUserTable();
            
            // Check access control
            if ($userTable == "customer" && ($site->customer_id != auth()->id() || $site->workspace_id != auth()->user()->current_workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }
            if ($userTable == "emp" && ($site->customer_id != auth()->user()->customer_id || $site->workspace_id != auth()->user()->workspace_id)) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Get user info for history
            if ($userTable === 'emp') {
                $added_by = EmpPersonalDetails::where('emp_id', Auth::user()->id)
                    ->select('first_name', 'middle_name', 'last_name')
                    ->first();
                $fullName = trim(($added_by->first_name ?? '') . " " . ($added_by->middle_name ?? '') . " " . ($added_by->last_name ?? ''));
            } else {
                $added_by = User::where('id', Auth::user()->id)->select('name')->first();
                $fullName = $added_by->name ?? '';
            }

            // Create site history entry
            $description = "<a href='" . url('/user-profile/' . Auth::user()->id) . "' style='text-transform: capitalize;' role='button' class='primary text-decoration-none'> {$fullName} </a> deleted the document: '{$siteDocument->title}' from site: {$site->title}.";
            $history = [
                'site_id' => $site->id,
                'description' => $description,
                'updated_by' => auth()->user()->id,
            ];
            SiteHistory::create($history);

            // Delete the file if it exists
            if ($siteDocument->file && file_exists(public_path($siteDocument->file))) {
                unlink(public_path($siteDocument->file));
            }

            // Delete the document (roles are stored in JSON, no need to detach)
            $siteDocument->delete();

            return $this->message('Document deleted successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error deleting site document: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get all document types for dropdown
     */
    public function getDocumentTypes()
    {
        try {
            $documentTypes = DocumentType::where('del', '0')
                ->orderBy('title', 'asc')
                ->select('id', 'title')
                ->get();

            return $this->success($documentTypes, 'Document types retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->message('Error retrieving document types: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Helper method to get employee's role ID
     * Uses withoutGlobalScopes to avoid filtering issues
     */
    private function getEmployeeRoleId($employeeId, $customerId, $workspaceId)
    {
        $employee = EmpCompanyDetails::withoutGlobalScopes()
            ->where('id', $employeeId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->first();
        
        if (!$employee || !$employee->access_role) {
            return null;
        }

        $role = Role::where('code', $employee->access_role)
            ->where('del', '0')
            ->first();

        return $role ? $role->id : null;
    }

    /**
     * Helper method to check if document applies to employee based on role_ids
     * Returns true if document should be shown to employee
     */
    private function documentAppliesToEmployee($document, $employeeRoleId)
    {
        // Document must have role_ids defined
        if (!$document->role_ids || !is_array($document->role_ids) || count($document->role_ids) == 0) {
            return false;
        }

        // Employee must have a role and it must be in document's role_ids
        if (!$employeeRoleId || !in_array($employeeRoleId, $document->role_ids)) {
            return false;
        }

        return true;
    }

    /**
     * Helper method to check if employee has signed a document
     */
    private function hasEmployeeSignedDocument($siteDocumentId, $employeeId, $siteId)
    {
        return SiteDocumentSignature::where('site_document_id', $siteDocumentId)
            ->where('employee_id', $employeeId)
            ->where('site_id', $siteId)
            ->exists();
    }

    /**
     * Helper method to build document response data
     */
    private function buildDocumentResponseData($document, $includeSignedStatus = false, $employeeId = null)
    {
        $documentData = [
            'id' => $document->id,
            'title' => $document->title,
            'file' => $document->file,
            'document_type' => $document->document_type,
            'site_id' => $document->site_id,
            'sign_requires' => $document->sign_requires,
        ];

        // Load document type detail
        if ($document->document_type) {
            $documentData['documentTypeDetail'] = DocumentType::find($document->document_type);
        }

        // Add signed status if requested
        if ($includeSignedStatus && $employeeId) {
            $signature = SiteDocumentSignature::where('site_document_id', $document->id)
                ->where('employee_id', $employeeId)
                ->where('site_id', $document->site_id)
                ->first();
            
            if ($signature) {
                $documentData['is_signed_required'] = 0; // Already signed
                $documentData['signed_at'] = $signature->signed_at;
            } else {
                $documentData['is_signed_required'] = 1; // Required to sign
            }
        }

        return $documentData;
    }

    /**
     * Helper method to check if employee needs to sign documents before check-in
     * Returns documents that require signature at check-in (signature_timing = 0)
     */
    public function getRequiredDocumentsForCheckIn($siteId, $employeeId, $customerId, $workspaceId)
    {
        $employeeRoleId = $this->getEmployeeRoleId($employeeId, $customerId, $workspaceId);

        // Get documents that require signature at check-in
        $documents = SiteDocument::where('site_id', $siteId)
            ->where('sign_requires', 1)
            ->where('signature_timing', 0)
            ->get();

        $requiredDocuments = [];

        foreach ($documents as $document) {
            // Check if document applies to this employee based on role
            if (!$this->documentAppliesToEmployee($document, $employeeRoleId)) {
                continue;
            }

            // Only return documents that haven't been signed yet
            if (!$this->hasEmployeeSignedDocument($document->id, $employeeId, $siteId)) {
                $documentData = $this->buildDocumentResponseData($document);
                $documentData['is_signed_required'] = 1; // Not signed yet - signature required
                $requiredDocuments[] = $documentData;
            }
        }

        return $requiredDocuments;
    }

    /**
     * API: Check if documents need signing before check-in
     * Called during check-in process to interrupt if signatures required
     */
    public function checkDocumentsBeforeCheckIn(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'site_id' => 'required|integer|exists:sites,id',
                'employee_id' => 'required|integer|exists:emp_company_details,id',
            ]);

            if ($validator->fails()) {
                return $this->message($validator->errors(), 422);
            }

            $userTable = $this->getUserTable();
            $customerId = $userTable == "customer" ? auth()->id() : auth()->user()->customer_id;
            $workspaceId = $userTable == "customer" ? auth()->user()->current_workspace_id : auth()->user()->workspace_id;

            // Get and validate site
            $site = Sites::find($request->site_id);
            if (!$site) {
                return $this->message('Site not found', 404);
            }

            // Check site access control
            if ($site->customer_id != $customerId || $site->workspace_id != $workspaceId) {
                return $this->message('Unauthorized access to this site', 403);
            }

            // Get and validate employee
            $employee = EmpCompanyDetails::withoutGlobalScopes()->find($request->employee_id);
            if (!$employee) {
                return $this->message('Employee not found', 404);
            }

            // Check employee access control
            if ($employee->customer_id != $customerId || $employee->workspace_id != $workspaceId) {
                return $this->message('Unauthorized access to this employee', 403);
            }

            // Get required documents for check-in
            $requiredDocuments = $this->getRequiredDocumentsForCheckIn(
                $request->site_id,
                $request->employee_id,
                $customerId,
                $workspaceId
            );

            $response = [
                'requires_signature' => count($requiredDocuments) > 0,
                'documents' => $requiredDocuments,
                'message' => count($requiredDocuments) > 0 
                    ? 'Please sign the required documents before checking in.' 
                    : 'No documents require signature before check-in.'
            ];

            return $this->success($response, 'Documents check completed', 200);
        } catch (\Exception $e) {
            return $this->message('Error checking documents: ' . $e->getMessage(), 500);
        }
    }

    /**
     * API: Submit document signature
     * Supports internal employees, external employees, and subcontractors
     */
    public function submitDocumentSignature(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'site_document_id' => 'required|integer|exists:site_documents,id',
            'site_id' => 'required|integer|exists:sites,id',
            'user_id' => 'required|integer',
            'user_type' => 'required|string|in:subcontractor,external,internal',
            'signature' => 'required|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
        ]);

        if ($validator->fails()) {
            return $this->message($validator->errors(), 422);
        }

        $validated = $validator->validated();
        $signerId = $validated['user_id'];
        $userType = $validated['user_type'];
        $siteId = $validated['site_id'];
        $siteDocumentId = $validated['site_document_id'];

        // Validate authenticated user matches payload
        if (Auth::id() != $signerId) {
            return $this->message('Unauthorized: user_id does not match authenticated user', 403);
        }

        // Get site and document
        $site = Sites::find($siteId);
        $siteDocument = SiteDocument::find($siteDocumentId);
        
        if (!$site) return $this->message('Site not found', 404);
        if (!$siteDocument) return $this->message('Site document not found', 404);
        if ($siteDocument->site_id != $siteId) {
            return $this->message('Document does not belong to this site', 422);
        }

        // Get customer/workspace based on user type
        $customerId = null;
        $workspaceId = null;
        $signerName = null;

        if ($userType === 'subcontractor') {
            $user = User::find($signerId);
            if (!$user || (int)$user->user_type !== (int)config('constants.user_types.subcontractor')) {
                return $this->message('Subcontractor user not found', 404);
            }
            
            $subcontractorCompany = SubcontractorCompany::where('user_id', $signerId)
                ->where('customer_id', $site->customer_id)
                ->where('workspace_id', $site->workspace_id)
                ->where('del', '0')
                ->first();
            
            if (!$subcontractorCompany) {
                return $this->message('You are not associated with this site\'s company', 403);
            }
            
            $customerId = $subcontractorCompany->customer_id;
            $workspaceId = $subcontractorCompany->workspace_id;
            $signerName = $user->company_name ?? $user->name ?? 'Subcontractor';
        } elseif ($userType === 'external') {
            $externalEmployee = \App\Models\EmployeeSubcontractor::find($signerId);
            if (!$externalEmployee) {
                return $this->message('External employee not found', 404);
            }
            
            $employeeMeta = \App\Models\EmployeeSubcontractorMeta::where('emp_id', $signerId)
                ->where('active', 1)
                ->first();
            
            if (!$employeeMeta) {
                return $this->message('External employee association not found', 404);
            }
            
            $subcontractorCompany = SubcontractorCompany::where('user_id', $employeeMeta->subcontractor_id)
                ->where('customer_id', $site->customer_id)
                ->where('workspace_id', $site->workspace_id)
                ->where('del', '0')
                ->first();
            
            if (!$subcontractorCompany) {
                return $this->message('External employee is not associated with this site\'s company', 403);
            }
            
            $customerId = $subcontractorCompany->customer_id;
            $workspaceId = $subcontractorCompany->workspace_id;
            $signerName = trim(($externalEmployee->first_name ?? '') . ' ' . 
                ($externalEmployee->middle_name ?? '') . ' ' . 
                ($externalEmployee->last_name ?? ''));
            $signerName = !empty(trim($signerName)) ? $signerName : ($externalEmployee->email ?? 'External Employee');
        } else {
            $employee = EmpCompanyDetails::withoutGlobalScopes()->find($signerId);
            if (!$employee) {
                return $this->message('Employee not found', 404);
            }
            
            $customerId = $employee->customer_id;
            $workspaceId = $employee->workspace_id;
            
            if ($site->customer_id != $customerId || $site->workspace_id != $workspaceId) {
                return $this->message('Unauthorized access to this site', 403);
            }
            
            $employeePersonal = EmpPersonalDetails::where('emp_id', $signerId)->first();
            if ($employeePersonal) {
                $signerName = trim(($employeePersonal->first_name ?? '') . ' ' . 
                    ($employeePersonal->middle_name ?? '') . ' ' . 
                    ($employeePersonal->last_name ?? ''));
            }
            $signerName = !empty(trim($signerName ?? '')) ? $signerName : ($employee->employee_email ?? 'Employee');
        }

        // Validate document applicability
        if ($userType === 'subcontractor') {
            $subcontractorIds = $siteDocument->subcontractors;
            if (!is_array($subcontractorIds) && $subcontractorIds) {
                $subcontractorIds = [$subcontractorIds];
            }
            if (!$subcontractorIds || !is_array($subcontractorIds) || !in_array($signerId, $subcontractorIds)) {
                return $this->message('This document is not assigned to you', 403);
            }
        } elseif ($userType === 'external') {
            $employeeRoleId = Role::where('code', 'EMP')->where('del', '0')->first();
            if (!$employeeRoleId || !$this->documentAppliesToEmployee($siteDocument, $employeeRoleId->id)) {
                return $this->message('This document is not applicable to your role', 403);
            }
        } else {
            if (!$this->documentAppliesToEmployee($siteDocument, $this->getEmployeeRoleId($signerId, $customerId, $workspaceId))) {
                return $this->message('This document is not applicable to your role', 403);
            }
        }

        // Handle signature upload
        if ($request->hasFile('signature')) {
            $uploadResult = $this->handleFileImageUpload($request, 'SiteDocumentSignatures');
            $signatureData = $uploadResult['path'] ?? null;
            if (!$signatureData) {
                return $this->message('Signature file upload failed', 422);
            }
        } else {
            $signatureData = $validated['signature'];
        }

        // Check if already signed
        if ($this->hasEmployeeSignedDocument($siteDocumentId, $signerId, $siteId)) {
            $signerType = $userType === 'subcontractor' ? 'subcontractor' : ($userType === 'external' ? 'external employee' : 'employee');
            return $this->message("Document already signed by this {$signerType}", 422);
        }

        // Create signature record
        $signature = SiteDocumentSignature::create([
            'site_document_id' => $siteDocumentId,
            'employee_id' => $signerId,
            'site_id' => $siteId,
            'user_type' => $userType,
            'customer_id' => $customerId,
            'workspace_id' => $workspaceId,
            'signature' => $signatureData,
            'signed_at' => now(),
        ]);

        // Send notification
        try {
            $signerTypeLabel = $userType === 'subcontractor' ? 'Subcontractor' : ($userType === 'external' ? 'External Employee' : 'Employee');
            $notificationMessage = "{$signerTypeLabel} {$signerName} has signed the document \"{$siteDocument->title}\" at " . now()->format('d-m-Y H:i:s');
            
            $this->save_notifications(
                "Site Document Signed",
                $notificationMessage,
                config('constants.employee_types.employee'),
                $signerId,
                config('constants.employee_types.customer'),
                $customerId,
                config('constants.notification_types.signature_added'),
                $customerId,
                $workspaceId
            );
        } catch (\Exception $e) {
            Log::error('SiteController:submitDocumentSignature: Error sending notification: ' . $e->getMessage());
        }

        // Prepare signature response
        $signatureResponse = [
            'signed_at' => $signature->signed_at,
            'user_type' => $signature->user_type,
        ];
        
        // If signature is a file path, return URL; otherwise return base64 string
        if ($request->hasFile('signature') && strpos($signature->signature, '/') !== false) {
            $signatureResponse['signature'] = $signature->signature;
            $signatureResponse['signature_url'] = asset('storage/' . $signature->signature);
        } else {
            $signatureResponse['signature'] = $signature->signature; // base64 string
        }

        return $this->success($signatureResponse, 'Saved Successfully');
    }

    /**
     * API: Get documents that can be signed later (after check-in)
     */
    public function getDocumentsToSignLater(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|integer|exists:emp_company_details,id',
            'site_id' => 'required|integer|exists:sites,id', // Optional: filter by site
        ]);
        
        if ($validator->fails()) {
            return $this->message($validator->errors(), 422);
        }
        
        $userTable = $this->getUserTable();
        
        // Get employee without global scopes to avoid filtering issues
        $employee = EmpCompanyDetails::withoutGlobalScopes()->find($request->employee_id);
        
        if (!$employee) {
            return $this->message('Employee not found', 404);
        }
        
        // Check access control
        if ($userTable == "customer" && ($employee->customer_id != auth()->id() || $employee->workspace_id != auth()->user()->current_workspace_id)) {
            return $this->message('Unauthorized access to this employee', 403);
        }
        if ($userTable == "emp" && ($employee->customer_id != auth()->user()->customer_id || $employee->workspace_id != auth()->user()->workspace_id)) {
            return $this->message('Unauthorized access to this employee', 403);
        }
        
        $customerId = $userTable == "customer" ? auth()->id() : auth()->user()->customer_id;
        $workspaceId = $userTable == "customer" ? auth()->user()->current_workspace_id : auth()->user()->workspace_id;
        $employeeRoleId = $this->getEmployeeRoleId($request->employee_id, $customerId, $workspaceId);

        // Get all site IDs that belong to this customer/workspace
        $siteIds = Sites::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->pluck('id')
            ->toArray();
        
        if (empty($siteIds)) {
            return $this->success([], 'No documents found', 200);
        }

        // Get documents that require signature and can be signed later
        // Filter by customer_id and workspace_id through sites relationship
        $documents = SiteDocument::whereIn('site_id', $siteIds)
            ->where('sign_requires', 1)
            ->where('signature_timing', 1)
            ->get();

        $documentsToSign = [];

        foreach ($documents as $document) {
            // Check if document applies to this employee based on role
            if (!$this->documentAppliesToEmployee($document, $employeeRoleId)) {
                continue;
            }

            // Build document data with signed status
            $documentData = $this->buildDocumentResponseData(
                $document,
                true, // Include signed status
                $request->employee_id
            );

            $documentsToSign[] = $documentData;
        }

        return $this->success($documentsToSign, 'Documents retrieved successfully', 200);
    }
    public function viewSiteDocumentSignatures($id)
    {
        try {
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|exists:site_documents,id',
            ]);

            if ($validator->fails()) {
                return $this->message($validator->errors()->first(), 422);
            }

            $userTable = $this->getUserTable();
            $customerId = $userTable == "customer" ? auth()->id() : auth()->user()->customer_id;
            $workspaceId = $userTable == "customer" ? auth()->user()->current_workspace_id : auth()->user()->workspace_id;

            // Get site document with relationships
            $siteDocument = SiteDocument::with(['site', 'documentType'])->find($id);
            if (!$siteDocument || !$siteDocument->site) {
                return $this->message('Site document not found', 404);
            }

            // Validate site access
            if ($siteDocument->site->customer_id != $customerId || $siteDocument->site->workspace_id != $workspaceId) {
                return $this->message('Unauthorized access to this document', 403);
            }

            // Build document data
            $documentData = $siteDocument->toArray();
            $documentData['uploaded_by_detail'] = $this->getUploadedByDetail($siteDocument);
            $documentData['roles'] = $this->getDocumentRoles($siteDocument);
            $documentData['subcontractors'] = $this->getDocumentSubcontractors($siteDocument);
            unset($documentData['role_ids']);

            // Get signatures and format them
            $signaturesData = SiteDocumentSignature::where('site_document_id', $id)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->orderBy('signed_at', 'desc')
                ->get();

            $userDateTimeFormat = (new SiteDocumentSignature())->getUserDateTimeFormat();
            $signatureCollection = $this->formatSignatures($signaturesData, $userDateTimeFormat);

            // Get widget counts and not signed employees
            $siteId = $siteDocument->site_id;
            $widgetCounts = $this->calculateDocumentSignatureWidgetCounts($siteDocument, $siteId, $customerId, $workspaceId);
            $notSignedEmployees = $this->getNotSignedEmployees($siteDocument, $siteId, $customerId, $workspaceId);

            return $this->success([
                'site_document' => $documentData,
                'signatures' => $signatureCollection,
                'widget_counts' => $widgetCounts,
                'not_signed_employees' => $notSignedEmployees,
            ], 'Site document signatures retrieved successfully', 200);
        } catch (\Exception $e) {
            return $this->message('An error occurred while retrieving signature data: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get uploaded by detail for document
     */
    private function getUploadedByDetail($siteDocument)
    {
        if (!$siteDocument->uploaded_by) {
            return null;
        }

        if ($siteDocument->uploaded_by_type == 'customer') {
            $user = User::find($siteDocument->uploaded_by);
            return $user ? ['id' => $user->id, 'name' => $user->name, 'type' => 'customer'] : null;
        }

        $emp = EmpPersonalDetails::where('emp_id', $siteDocument->uploaded_by)->first();
        if (!$emp) {
            return null;
        }

        $empName = trim(($emp->first_name ?? '') . ' ' . ($emp->middle_name ?? '') . ' ' . ($emp->last_name ?? ''));
        return ['id' => $siteDocument->uploaded_by, 'name' => $empName, 'type' => 'emp'];
    }

    /**
     * Get document roles details
     */
    private function getDocumentRoles($siteDocument)
    {
        if (!$siteDocument->role_ids || !is_array($siteDocument->role_ids) || empty($siteDocument->role_ids)) {
            return [];
        }

        return Role::whereIn('id', $siteDocument->role_ids)
            ->where('del', '0')
            ->select('id', 'title', 'code')
            ->get()
            ->map(fn($role) => ['id' => $role->id, 'title' => $role->title, 'code' => $role->code])
            ->toArray();
    }

    /**
     * Get document subcontractors details
     */
    private function getDocumentSubcontractors($siteDocument)
    {
        $subcontractorIds = $siteDocument->subcontractors;
        if (!is_array($subcontractorIds) && $subcontractorIds) {
            $subcontractorIds = [$subcontractorIds];
        }

        if (!$subcontractorIds || !is_array($subcontractorIds) || empty($subcontractorIds)) {
            return [];
        }

        return User::whereIn('id', $subcontractorIds)
            ->where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->select('id', 'name', 'company_name', 'email', 'mobile_number')
            ->get()
            ->map(fn($user) => [
                'id' => $user->id,
                'title' => $user->company_name ?? $user->name,
                'name' => $user->name,
                'company_name' => $user->company_name,
                'email' => $user->email,
                'mobile_number' => $user->mobile_number,
            ])
            ->toArray();
    }

    /**
     * Format signatures with signer details
     */
    private function formatSignatures($signaturesData, $userDateTimeFormat)
    {
        // Batch load signers to avoid N+1 queries
        $employeeIds = $signaturesData->pluck('employee_id')->unique()->toArray();
        $userTypes = $signaturesData->groupBy('user_type');
        
        $signers = [];
        
        // Load subcontractors
        if ($userTypes->has('subcontractor')) {
            $subIds = $signaturesData->where('user_type', 'subcontractor')->pluck('employee_id')->unique()->toArray();
            $subcontractors = User::whereIn('id', $subIds)
                ->where('user_type', config('constants.user_types.subcontractor'))
                ->where('del', '0')
                ->get()
                ->keyBy('id');
            $signers['subcontractor'] = $subcontractors;
        }
        
        // Load external employees
        if ($userTypes->has('external')) {
            $extIds = $signaturesData->where('user_type', 'external')->pluck('employee_id')->unique()->toArray();
            $externalEmployees = EmployeeSubcontractor::whereIn('id', $extIds)->get()->keyBy('id');
            $signers['external'] = $externalEmployees;
        }
        
        // Load internal employees
        if ($userTypes->has('internal') || $userTypes->has(null)) {
            $intIds = $signaturesData->whereIn('user_type', ['internal', null])->pluck('employee_id')->unique()->toArray();
            $empPersonal = EmpPersonalDetails::whereIn('emp_id', $intIds)->get()->keyBy('emp_id');
            $empCompany = EmpCompanyDetails::whereIn('id', $intIds)->get()->keyBy('id');
            $roleCodes = $empCompany->pluck('access_role')->filter()->unique()->toArray();
            $roles = Role::whereIn('code', $roleCodes)->where('del', '0')->get()->keyBy('code');
            
            $signers['internal'] = ['personal' => $empPersonal, 'company' => $empCompany, 'roles' => $roles];
        }

        return $signaturesData->map(function ($signature) use ($userDateTimeFormat, $signers) {
            $signatureData = $signature->toArray();
            $userType = $signature->user_type ?? 'internal';
            $employeeId = $signature->employee_id;

            // Format dates
            $this->formatSignatureDates($signatureData, $userDateTimeFormat);

            // Get signer details
            $signerDetails = $this->getSignerDetails($userType, $employeeId, $signers);
            
            return array_merge($signatureData, [
                'user_type' => $userType,
                'employee_name' => $signerDetails['name'] ?? '',
                'employee_first_name' => $signerDetails['first_name'] ?? null,
                'employee_middle_name' => $signerDetails['middle_name'] ?? null,
                'employee_last_name' => $signerDetails['last_name'] ?? null,
                'image' => $signerDetails['image'] ?? null,
                'role_code' => $signerDetails['role_code'] ?? null,
                'role_title' => $signerDetails['role_title'] ?? null,
            ]);
        })->values()->toArray();
    }

    /**
     * Format signature dates
     */
    private function formatSignatureDates(&$signatureData, $userDateTimeFormat)
    {
        foreach (['created_at', 'updated_at', 'signed_at'] as $field) {
            if (!empty($signatureData[$field])) {
                try {
                    $signatureData[$field] = \Carbon\Carbon::parse($signatureData[$field])->format($userDateTimeFormat);
                } catch (\Exception $e) {
                    Log::warning("Failed to format {$field} for signature: " . $e->getMessage());
                }
            }
        }
    }

    /**
     * Get signer details based on user type
     */
    private function getSignerDetails($userType, $employeeId, $signers)
    {
        if ($userType === 'subcontractor' && isset($signers['subcontractor'][$employeeId])) {
            $sub = $signers['subcontractor'][$employeeId];
            return [
                'name' => $sub->company_name ?? $sub->name ?? '',
                'first_name' => $sub->name ?? null,
                'last_name' => $sub->company_name ?? null,
                'image' => null,
                'role_title' => 'Subcontractor',
            ];
        }

        if ($userType === 'external' && isset($signers['external'][$employeeId])) {
            $ext = $signers['external'][$employeeId];
            $name = trim(($ext->first_name ?? '') . ' ' . ($ext->middle_name ?? '') . ' ' . ($ext->last_name ?? ''));
            return [
                'name' => $name,
                'first_name' => $ext->first_name ?? null,
                'middle_name' => $ext->middle_name ?? null,
                'last_name' => $ext->last_name ?? null,
                'image' => $ext->profile_image ? url($ext->profile_image) : null,
                'role_title' => 'External Employee',
            ];
        }

        // Internal employee
        $personal = $signers['internal']['personal'][$employeeId] ?? null;
        $company = $signers['internal']['company'][$employeeId] ?? null;
        $roles = $signers['internal']['roles'] ?? collect();

        $name = '';
        $roleTitle = null;
        $roleCode = null;

        if ($personal) {
            $name = trim(($personal->first_name ?? '') . ' ' . ($personal->middle_name ?? '') . ' ' . ($personal->last_name ?? ''));
        }

        if ($company && $company->access_role) {
            $roleCode = $company->access_role;
            $role = $roles[$roleCode] ?? null;
            $roleTitle = $role ? $role->title : null;
        }

        return [
            'name' => $name,
            'first_name' => $personal->first_name ?? null,
            'middle_name' => $personal->middle_name ?? null,
            'last_name' => $personal->last_name ?? null,
            'image' => $personal->image ? url($personal->image) : null,
            'role_code' => $roleCode,
            'role_title' => $roleTitle,
        ];
    }

    private function calculateDocumentSignatureWidgetCounts($siteDocument, $siteId, $customerId, $workspaceId)
    {
        try {
            if (!$siteDocument->sign_requires || $siteDocument->sign_requires != 1) {
                return ['total_required_signs' => 0, 'signed_signs' => 0, 'pending_signs' => 0];
            }

            $requiredEmployeeIds = [];

            // 1. Get internal employees with matching roles who have roster assignments
            $roleIds = $siteDocument->role_ids;
            $roleCodes = [];
            
            if (is_array($roleIds) && !empty($roleIds)) {
                $roleCodes = Role::whereIn('id', $roleIds)->where('del', '0')->pluck('code')->toArray();
                
                if (!empty($roleCodes)) {
                    $internalEmployeeIds = DB::table('emp_company_details')
                        ->join('roster_assigns', function ($join) use ($siteId, $customerId, $workspaceId) {
                            $join->on('emp_company_details.id', '=', 'roster_assigns.assign_to')
                                ->where('roster_assigns.site_id', $siteId)
                                ->where('roster_assigns.customer_id', $customerId)
                                ->where('roster_assigns.workspace_id', $workspaceId)
                                ->whereNull('roster_assigns.subcontractor_id'); // Internal employees have NULL subcontractor_id
                        })
                        ->whereIn('emp_company_details.access_role', $roleCodes)
                        ->where('emp_company_details.customer_id', $customerId)
                        ->where('emp_company_details.workspace_id', $workspaceId)
                        ->where('emp_company_details.del', '0')
                        ->distinct()
                        ->pluck('emp_company_details.id')
                        ->toArray();

                    $requiredEmployeeIds = array_merge($requiredEmployeeIds, $internalEmployeeIds);
                }
            }

            // 2. Get external employees (subcontractor employees) with roster assignments to this site
            // External employees typically have default 'EMP' role, so include them if EMP role is in document's role_ids
            if (!empty($roleCodes) && in_array('EMP', $roleCodes)) {
                $externalEmployeeIds = DB::table('employees_subcontractors')
                    ->join('roster_assigns', function ($join) use ($siteId, $customerId, $workspaceId) {
                        $join->on('employees_subcontractors.id', '=', 'roster_assigns.assign_to')
                            ->where('roster_assigns.site_id', $siteId)
                            ->where('roster_assigns.customer_id', $customerId)
                            ->where('roster_assigns.workspace_id', $workspaceId)
                            ->whereNotNull('roster_assigns.subcontractor_id'); // External employees have non-NULL subcontractor_id
                    })
                    ->distinct()
                    ->pluck('employees_subcontractors.id')
                    ->toArray();

                $requiredEmployeeIds = array_merge($requiredEmployeeIds, $externalEmployeeIds);
            }

            // 3. Get subcontractors (companies) assigned to this document
            $subcontractorIds = $siteDocument->subcontractors;
            if (!is_array($subcontractorIds) && $subcontractorIds) {
                $subcontractorIds = [$subcontractorIds];
            }
            
            if (!empty($subcontractorIds) && is_array($subcontractorIds)) {
                $validSubcontractorIds = User::whereIn('id', $subcontractorIds)
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0')
                    ->whereHas('subcontractorCompanies', function ($query) use ($customerId, $workspaceId) {
                        $query->where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('del', '0');
                    })
                    ->pluck('id')
                    ->toArray();

                $requiredEmployeeIds = array_merge($requiredEmployeeIds, $validSubcontractorIds);
            }

            $totalRequiredSigns = count(array_unique($requiredEmployeeIds));

            // Count signed signatures
            $signedSigns = SiteDocumentSignature::where('site_document_id', $siteDocument->id)
                ->where('site_id', $siteId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->distinct('employee_id')
                ->count('employee_id');

            return [
                'total_required_signs' => $totalRequiredSigns,
                'signed_signs' => $signedSigns,
                'pending_signs' => max(0, $totalRequiredSigns - $signedSigns)
            ];
        } catch (\Exception $e) {
            Log::error('SiteController:calculateDocumentSignatureWidgetCounts: Error: ' . $e->getMessage());
            return ['total_required_signs' => 0, 'signed_signs' => 0, 'pending_signs' => 0];
        }
    }
    private function getNotSignedEmployees($siteDocument, $siteId, $customerId, $workspaceId)
    {
        try {
            if (!$siteDocument->sign_requires || $siteDocument->sign_requires != 1) {
                return [];
            }

            // Get all signed IDs in one query
            $signedSignatures = SiteDocumentSignature::where('site_document_id', $siteDocument->id)
                ->where('site_id', $siteId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->select('employee_id', 'user_type')
                ->get();

            $signedEmployeeIds = $signedSignatures->pluck('employee_id')->unique()->toArray();
            $signedExternalEmployeeIds = $signedSignatures->where('user_type', 'external')->pluck('employee_id')->unique()->toArray();
            $signedSubcontractorIds = $signedSignatures->where('user_type', 'subcontractor')->pluck('employee_id')->unique()->toArray();

            $notSignedList = [];

            // 1. Get internal employees who haven't signed
            $roleIds = $siteDocument->role_ids;
            $roleCodes = [];
            
            if (is_array($roleIds) && !empty($roleIds)) {
                $roleCodes = Role::whereIn('id', $roleIds)->where('del', '0')->pluck('code')->toArray();
                
                if (!empty($roleCodes)) {
                    $requiredEmployeeIds = DB::table('emp_company_details')
                        ->join('roster_assigns', function ($join) use ($siteId, $customerId, $workspaceId) {
                            $join->on('emp_company_details.id', '=', 'roster_assigns.assign_to')
                                ->where('roster_assigns.site_id', $siteId)
                                ->where('roster_assigns.customer_id', $customerId)
                                ->where('roster_assigns.workspace_id', $workspaceId)
                                ->whereNull('roster_assigns.subcontractor_id'); // Internal employees have NULL subcontractor_id
                        })
                        ->whereIn('emp_company_details.access_role', $roleCodes)
                        ->where('emp_company_details.customer_id', $customerId)
                        ->where('emp_company_details.workspace_id', $workspaceId)
                        ->where('emp_company_details.del', '0')
                        ->distinct()
                        ->pluck('emp_company_details.id')
                        ->toArray();

                    $notSignedEmployeeIds = array_diff($requiredEmployeeIds, $signedEmployeeIds);

                    if (!empty($notSignedEmployeeIds)) {
                        $notSignedEmployees = DB::table('emp_company_details')
                            ->leftJoin('emp_personal_details', 'emp_company_details.id', '=', 'emp_personal_details.emp_id')
                            ->leftJoin('roles', function ($join) {
                                $join->on('emp_company_details.access_role', '=', 'roles.code')->where('roles.del', '0');
                            })
                            ->whereIn('emp_company_details.id', $notSignedEmployeeIds)
                            ->where('emp_company_details.customer_id', $customerId)
                            ->where('emp_company_details.workspace_id', $workspaceId)
                            ->where('emp_company_details.del', '0')
                            ->select(
                                'emp_company_details.id as employee_id',
                                'emp_company_details.employee_email',
                                'emp_personal_details.first_name',
                                'emp_personal_details.middle_name',
                                'emp_personal_details.last_name',
                                'emp_personal_details.image',
                                'emp_company_details.access_role as role_code',
                                'roles.title as role_title',
                                'roles.id as role_id'
                            )
                            ->get();

                        foreach ($notSignedEmployees as $employee) {
                            $fullName = trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? ''));
                            if (empty(trim($fullName))) {
                                $fullName = $employee->employee_email ?? 'Unknown Employee';
                            }

                            $notSignedList[] = [
                                'employee_id' => $employee->employee_id,
                                'employee_name' => $fullName,
                                'first_name' => $employee->first_name ?? null,
                                'middle_name' => $employee->middle_name ?? null,
                                'last_name' => $employee->last_name ?? null,
                                'employee_email' => $employee->employee_email,
                                'image' => $employee->image ?? null,
                                'user_type' => 'internal',
                                'role' => [
                                    'id' => $employee->role_id ?? null,
                                    'code' => $employee->role_code ?? null,
                                    'title' => $employee->role_title ?? null,
                                ]
                            ];
                        }
                    }
                }
            }

            // 2. Get external employees (subcontractor employees) who haven't signed
            if (!empty($roleCodes) && in_array('EMP', $roleCodes)) {
                $requiredExternalEmployeeIds = DB::table('employees_subcontractors')
                    ->join('roster_assigns', function ($join) use ($siteId, $customerId, $workspaceId) {
                        $join->on('employees_subcontractors.id', '=', 'roster_assigns.assign_to')
                            ->where('roster_assigns.site_id', $siteId)
                            ->where('roster_assigns.customer_id', $customerId)
                            ->where('roster_assigns.workspace_id', $workspaceId)
                            ->whereNotNull('roster_assigns.subcontractor_id'); // External employees have non-NULL subcontractor_id
                    })
                    ->distinct()
                    ->pluck('employees_subcontractors.id')
                    ->toArray();

                $notSignedExternalEmployeeIds = array_diff($requiredExternalEmployeeIds, $signedExternalEmployeeIds);

                if (!empty($notSignedExternalEmployeeIds)) {
                    $notSignedExternalEmployees = EmployeeSubcontractor::whereIn('id', $notSignedExternalEmployeeIds)
                        ->select('id', 'first_name', 'middle_name', 'last_name', 'email', 'profile_image')
                        ->get();

                    foreach ($notSignedExternalEmployees as $employee) {
                        $fullName = trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? ''));
                        if (empty(trim($fullName))) {
                            $fullName = $employee->email ?? 'Unknown Employee';
                        }

                        $notSignedList[] = [
                            'employee_id' => $employee->id,
                            'employee_name' => $fullName,
                            'first_name' => $employee->first_name ?? null,
                            'middle_name' => $employee->middle_name ?? null,
                            'last_name' => $employee->last_name ?? null,
                            'employee_email' => $employee->email ?? null,
                            'image' => $employee->profile_image ? url($employee->profile_image) : null,
                            'user_type' => 'external',
                            'role' => [
                                'id' => null,
                                'code' => 'EMP',
                                'title' => 'External Employee',
                            ]
                        ];
                    }
                }
            }

            // 3. Get subcontractors (companies) who haven't signed
            $subcontractorIds = $siteDocument->subcontractors;
            if (!is_array($subcontractorIds) && $subcontractorIds) {
                $subcontractorIds = [$subcontractorIds];
            }
            
            if (!empty($subcontractorIds) && is_array($subcontractorIds)) {
                $validSubcontractorIds = User::whereIn('id', $subcontractorIds)
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0')
                    ->whereHas('subcontractorCompanies', function ($query) use ($customerId, $workspaceId) {
                        $query->where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('del', '0');
                    })
                    ->pluck('id')
                    ->toArray();

                $notSignedSubcontractorIds = array_diff($validSubcontractorIds, $signedSubcontractorIds);

                if (!empty($notSignedSubcontractorIds)) {
                    $notSignedSubcontractors = User::whereIn('id', $notSignedSubcontractorIds)
                        ->where('user_type', config('constants.user_types.subcontractor'))
                        ->where('del', '0')
                        ->select('id', 'name', 'company_name', 'email', 'mobile_number', 'avatar')
                        ->get();

                    foreach ($notSignedSubcontractors as $subcontractor) {
                        $notSignedList[] = [
                            'employee_id' => $subcontractor->id,
                            'employee_name' => $subcontractor->company_name ?? $subcontractor->name ?? 'Subcontractor',
                            'first_name' => $subcontractor->name ?? null,
                            'middle_name' => null,
                            'last_name' => $subcontractor->company_name ?? null,
                            'employee_email' => $subcontractor->email ?? null,
                            'image' => $subcontractor->avatar ?? null,
                            'user_type' => 'subcontractor',
                            'role' => ['id' => null, 'code' => null, 'title' => 'Subcontractor']
                        ];
                    }
                }
            }

            return $notSignedList;
        } catch (\Exception $e) {
            Log::error('SiteController:getNotSignedEmployees: Error getting not signed employees: ' . $e->getMessage());
            return [];
        }
    }
}
