<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\SubcontractorContactPerson;
use App\Models\SubcontractorCompany;
use App\Models\SubcontractorRequiredDocument;
use App\Models\RequiredDocument;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\Project;
use App\Models\Workspace;
use App\Models\Defect;
use App\Models\Tender;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Http\Controllers\Traits\EmailTrait;
use App\Http\Controllers\Traits\SubcontractorTrait;

class SubContractorController extends Controller
{
    use EmailTrait, SubcontractorTrait;

    public function index(Request $request)
    {
        // Check if this is a request for dropdown filter (lightweight response)
        // Support filter={"dropdown":true} format
        if ($request->filled('filter')) {
            $filters = is_string($request->filter) ? json_decode($request->filter, true) : $request->filter;
            if (is_array($filters) && isset($filters['dropdown']) && ($filters['dropdown'] === true || $filters['dropdown'] === 'true' || $filters['dropdown'] === 1)) {
                return $this->getSubcontractorsForDropdown($request);
            }
        }

        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->withCount(collect([]), 'Subcontractors retrieved successfully');
        }
        // Join with subcontractor_companies to filter by customer and workspace
        $query = User::where('users.user_type', config('constants.user_types.subcontractor'))
            ->where('users.del', '0')
            ->join('subcontractor_companies', 'users.id', '=', 'subcontractor_companies.user_id')
            ->where('subcontractor_companies.customer_id', $ids['customer_id'])
            ->where('subcontractor_companies.workspace_id', $ids['workspace_id'])
            ->where('subcontractor_companies.del', '0')
            ->with('contactPersons')
            ->select('users.*')
            ->latest('users.id');
        // Apply search filters
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('users.name', 'like', '%' . $searchTerm . '%')
                    ->orWhere('users.company_name', 'like', '%' . $searchTerm . '%')
                    ->orWhere('users.email', 'like', '%' . $searchTerm . '%')
                    ->orWhere('users.mobile_number', 'like', '%' . $searchTerm . '%')
                    ->orWhere('users.abn', 'like', '%' . $searchTerm . '%')
                    ->orWhereHas('contactPersons', function ($subquery) use ($searchTerm) {
                        $subquery->where('name', 'like', '%' . $searchTerm . '%');
                    });
            });
        }
        // Filter by trade_id (from subcontractor_companies)
        if ($request->filled('trade_id')) {
            $query->where('subcontractor_companies.trade_id', $request->input('trade_id'));
        }
        if ($request->filled('filter')) {
            $filters = is_string($request->filter) ? json_decode($request->filter, true) : $request->filter;
            if (is_array($filters)) {
                // Remove dropdown from filters as it's handled separately
                unset($filters['dropdown']);
                if (!empty($filters)) {
                    $query = $this->applySubcontractorFilters($query, $filters, $ids);
                }
            }
        }
        $subcontractors = $query->get()->map(function ($subcontractor) use ($ids) {
            if ($subcontractor->company_logo) {
                $subcontractor->company_logo_url = url($subcontractor->company_logo);
            } else {
                $subcontractor->company_logo_url = null;
            }
            // Add project_ids, trade_id, and trade_title from subcontractor_companies
            $subcontractorCompany = SubcontractorCompany::with('trade')
                ->where('user_id', $subcontractor->id)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('del', '0')
                ->first();
            if ($subcontractorCompany) {
                $subcontractor->project_ids = $subcontractorCompany->project_ids ?? [];
                // Add required_docs_status and required_docs_status_label after project_ids
                $status = $subcontractorCompany->required_docs_status ?? 0;
                $subcontractor->required_docs_status = $status;
                // Map status to user-friendly labels
                $statusLabels = [
                    0 => 'not uploaded',
                    1 => 'approved',
                    2 => 'new docs added',
                    3 => 'rejected',
                    4 => 'pending approval'
                ];
                $subcontractor->required_docs_status_label = $statusLabels[$status] ?? 'unknown';
                $subcontractor->trade_id = $subcontractorCompany->trade_id;
                $subcontractor->trade_title = $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null;
            } else {
                $subcontractor->project_ids = [];
                $subcontractor->required_docs_status = 0;
                $subcontractor->required_docs_status_label = 'not uploaded';
                $subcontractor->trade_id = null;
                $subcontractor->trade_title = null;
            }
            return $subcontractor;
        });
        return $this->withCount($subcontractors, 'Subcontractors retrieved successfully');
    }

    protected function applySubcontractorFilters($query, $filters, $ids)
    {
        foreach ($filters as $filterName => $filterValue) {
            if ($filterValue === null || $filterValue === '') {
                continue;
            }

            switch ($filterName) {
                case 'company_name':
                    // Filter by company name
                    $query->where('users.company_name', 'like', '%' . $filterValue . '%');
                    break;

                case 'owner_name':
                case 'name':
                    // Filter by owner/contact person name
                    $query->where(function ($q) use ($filterValue) {
                        $q->where('users.name', 'like', '%' . $filterValue . '%')
                            ->orWhereHas('contactPersons', function ($subquery) use ($filterValue) {
                                $subquery->where('name', 'like', '%' . $filterValue . '%');
                            });
                    });
                    break;

                case 'contact_number':
                case 'mobile_number':
                    // Filter by contact/mobile number
                    $query->where('users.mobile_number', 'like', '%' . $filterValue . '%');
                    break;

                case 'email':
                    // Filter by email
                    $query->where('users.email', 'like', '%' . $filterValue . '%');
                    break;

                case 'abn':
                    // Filter by ABN
                    $query->where('users.abn', 'like', '%' . $filterValue . '%');
                    break;

                case 'trade_id':
                case 'trade':
                    // Filter by trade ID
                    $query->where('subcontractor_companies.trade_id', $filterValue);
                    break;

                case 'project_ids':
                case 'project_id':
                case 'projects':
                    // Filter by project IDs (array or single value)
                    // Since we're already joined with subcontractor_companies, filter directly
                    $projectIds = is_array($filterValue) ? $filterValue : [$filterValue];
                    $query->where(function ($q) use ($projectIds) {
                        foreach ($projectIds as $projectId) {
                            $q->orWhereJsonContains('subcontractor_companies.project_ids', (string)$projectId);
                        }
                    });
                    break;

                case 'status':
                case 'active_status':
                    // Filter by active status (0 = inactive, 1 = active)
                    if (is_numeric($filterValue) && in_array((int)$filterValue, [0, 1])) {
                        $query->where('users.active_status', (int)$filterValue);
                    }
                    break;

                case 'required_docs_status':
                    // Filter by required documents status
                    // Support both integer (0-4) and string labels
                    $statusValue = $filterValue;

                    // Map status labels to integers if string provided
                    if (is_string($filterValue)) {
                        $statusLabels = [
                            'not uploaded' => 0,
                            'approved' => 1,
                            'new docs added' => 2,
                            'rejected' => 3,
                            'pending approval' => 4,
                            'pending' => 4,
                            'complete' => 1,
                            'not_uploaded' => 0,
                            'new_docs_added' => 2,
                        ];
                        $statusValue = $statusLabels[strtolower(trim($filterValue))] ?? $filterValue;
                    }

                    // If still string or invalid, try to convert to integer
                    if (is_string($statusValue) && is_numeric($statusValue)) {
                        $statusValue = (int)$statusValue;
                    }

                    // Apply filter if valid status (0-4)
                    if (is_numeric($statusValue) && in_array((int)$statusValue, [0, 1, 2, 3, 4])) {
                        $query->where('subcontractor_companies.required_docs_status', (int)$statusValue);
                    }
                    break;
            }
        }

        return $query;
    }

    public function store(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();

        // Check if a user with the same email and user_type subcontractor already exists
        // This check is done before validation to allow same email for existing subcontractors
        $existingSubcontractor = null;
        if ($request->filled('email')) {
            $existingSubcontractor = User::where('email', $request->email)
                ->where('user_type', config('constants.user_types.subcontractor'))
                ->where('del', '0')
                ->first();
        }

        // Build validation rules - if email belongs to existing subcontractor, allow it
        $emailRules = ['nullable', 'email', 'max:255'];
        if (!$existingSubcontractor) {
            // Only check uniqueness if it's not an existing subcontractor
            $emailRules[] = Rule::unique('users', 'email');
            $emailRules[] = Rule::unique('emp_company_details', 'employee_email');
            $emailRules[] = Rule::unique('employees_subcontractors', 'email');
        } else {
            // If it's an existing subcontractor, check if already associated with this customer
            $existingCompany = SubcontractorCompany::where('user_id', $existingSubcontractor->id)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('del', '0')
                ->first();

            if ($existingCompany) {
                return $this->error('This subcontractor is already associated with your company', 422);
            }
        }

        $validator = Validator::make($request->all(), [
            'company_name' => 'required|string|max:255',
            'abn' => 'nullable|string|max:20',
            'company_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
            'trade_id' => 'nullable|integer',
            'company_number' => 'nullable|string|max:255',
            'name' => 'required|string|max:255',
            'mobile_number' => 'required|string|max:20',
            'email' => $emailRules,
            'project_ids' => 'nullable|array',
            'project_ids.*' => 'integer',
            'contact_persons' => 'nullable|array',
            'contact_persons.*.name' => 'required|string|max:255',
            'contact_persons.*.email' => 'nullable|email|max:255',
            'contact_persons.*.phone' => 'nullable|string|max:20',
            'contact_persons.*.role' => 'nullable|string|max:255',
        ], [
            'company_name.required' => 'Company name is required',
            'name.required' => 'Owner name is required',
            'mobile_number.required' => 'Contact number is required',
            'email.unique' => 'Email already exists',
            'company_logo.image' => 'Company logo must be an image',
            'company_logo.mimes' => 'Company logo must be jpeg, png, jpg, gif, svg or webp',
            'company_logo.max' => 'Company logo must not exceed 2MB',
            'contact_persons.*.name.required' => 'Name is required for all contact persons',
            'contact_persons.*.email.email' => 'Email must be a valid email address',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        // Initialize password variable (will be set for new subcontractors only)
        $password = null;

        // If user exists, use it; otherwise create a new user
        if ($existingSubcontractor) {
            $subcontractor = $existingSubcontractor;
            // If existing subcontractor doesn't have secret_key, generate one
            if (!$subcontractor->secret_key) {
                $subcontractor->secret_key = $this->generateSecretKey();
                $subcontractor->save();
            }
            // Note: Duplicate association check is already done before validation
        } else {
            // Handle company logo upload
            $companyLogoPath = null;
            if ($request->hasFile('company_logo')) {
                $companyLogoPath = $this->handleFileImageUpload($request, 'SubContractors')['path'] ?? null;
            }

            // Generate a random 8-digit password for new subcontractors
            $digits = 8;
            $password = rand(pow(10, $digits - 1), pow(10, $digits) - 1);

            $subcontractor = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'mobile_number' => $request->mobile_number,
                'company_name' => $request->company_name,
                'abn' => $request->abn,
                'company_logo' => $companyLogoPath,
                'company_number' => $request->company_number,
                'user_type' => config('constants.user_types.subcontractor'), // Subcontractor user type
                'active_status' => 1,
                'is_enable_login' => 1,
                'password' => Hash::make($password),
                'secret_key' => $this->generateSecretKey(),
                'del' => '0',
                'created_by' => Auth::id(),
            ]);

            if (!$subcontractor) {
                return $this->message('Failed to create subcontractor.', 500);
            }
        }

        // Create entry in subcontractor_companies table
        // If subcontractor already exists (same email), associate them with this new company
        $subcontractorCompany = SubcontractorCompany::create([
            'user_id' => $subcontractor->id,
            'customer_id' => $ids['customer_id'] ?? null,
            'workspace_id' => $ids['workspace_id'] ?? null,
            'trade_id' => $request->trade_id,
            'project_ids' => $request->project_ids ?? [],
            'del' => '0',
        ]);

        if (!$subcontractorCompany) {
            return $this->message('Failed to create subcontractor company association.', 500);
        }

        // Send welcome email to subcontractor (both new and existing) if email is provided
        if ($request->filled('email') && $subcontractor->email) {
            try {
                // For new subcontractors, include password; for existing, password will be null
                $this->sendSubcontractorWelcomeEmail($subcontractor, $password, $ids['customer_id'], $ids['workspace_id']);
            } catch (\Exception $e) {
                Log::error('Error sending welcome email to subcontractor: ' . $e->getMessage());
                // Don't fail the creation if email fails, just log it
            }
        }

        // Check if both required documents and induction are complete
        // If both are complete (or neither exists), auto-approve to activate projects
        $allComplete = $this->areAllDocumentsAndInductionComplete(
            $subcontractor->id,
            $ids['customer_id'],
            $ids['workspace_id'],
            $subcontractorCompany
        );

        if ($allComplete) {
            // Both required documents AND induction are complete (or neither exists), mark as approved so projects flow correctly
            $subcontractorCompany->update(['required_docs_status' => SubcontractorCompany::STATUS_COMPLETE]);
        }

        // Save contact persons if provided
        if (isset($request->contact_persons) && is_array($request->contact_persons)) {
            foreach ($request->contact_persons as $contactPerson) {
                SubcontractorContactPerson::create([
                    'subcontractor_id' => $subcontractor->id,
                    'name' => $contactPerson['name'],
                    'email' => $contactPerson['email'] ?? null,
                    'phone' => $contactPerson['phone'] ?? null,
                    'role' => $contactPerson['role'] ?? null,
                    'customer_id' => $ids['customer_id'] ?? null,
                    'workspace_id' => $ids['workspace_id'] ?? null,
                    'del' => '0',
                ]);
            }
        }

        // Add logo URL to response
        if ($subcontractor->company_logo) {
            $subcontractor->company_logo_url = url($subcontractor->company_logo);
        } else {
            $subcontractor->company_logo_url = null;
        }
        // Add project_ids to response
        $subcontractor->project_ids = $subcontractorCompany->project_ids ?? [];
        // Load contact persons for response
        $subcontractor->load('contactPersons');
        return $this->success($subcontractor, 'Subcontractor created successfully');
    }

    public function show($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        // Join with subcontractor_companies to filter by customer and workspace
        $subcontractor = User::where('users.user_type', config('constants.user_types.subcontractor'))
            ->where('users.del', '0')
            ->where('users.id', $id)
            ->join('subcontractor_companies', 'users.id', '=', 'subcontractor_companies.user_id')
            ->where('subcontractor_companies.customer_id', $ids['customer_id'])
            ->where('subcontractor_companies.workspace_id', $ids['workspace_id'])
            ->where('subcontractor_companies.del', '0')
            ->with('contactPersons')
            ->select('users.*')
            ->first();
        if (!$subcontractor) {
            return $this->message('Subcontractor not found', 404);
        }
        // Add logo URL
        if ($subcontractor->company_logo) {
            $subcontractor->company_logo_url = url($subcontractor->company_logo);
        } else {
            $subcontractor->company_logo_url = null;
        }
        // Add project_ids and trade_id from subcontractor_companies
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$subcontractorCompany) {
            return $this->message('Subcontractor not found', 404);
        }

        if ($subcontractorCompany) {
            $subcontractor->project_ids = $subcontractorCompany->project_ids ?? [];
            $subcontractor->trade_id = $subcontractorCompany->trade_id;
            $subcontractor->required_docs_status = $subcontractorCompany->required_docs_status;
            $subcontractor->required_docs_status_label = $subcontractorCompany->required_docs_status_label;
            $subcontractor->rejection_reason = $subcontractorCompany->rejection_reason;

            // Get all required documents for subcontractors from this company
            $requiredDocuments = RequiredDocument::where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('for_subcontractor', true)
                ->where('del', '0')
                ->with(['requiredDocumentActiveField'])
                ->get();

            // Get uploaded documents for this subcontractor
            $uploadedDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractor->id)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('del', '0')
                ->get()
                ->groupBy('required_document_id');

            // Merge uploaded data with required documents
            $documentsWithData = $requiredDocuments->map(function ($doc) use ($uploadedDocuments, $subcontractor, $ids) {
                $docId = $doc->id;
                $uploaded = $uploadedDocuments->get($docId) ?? collect();

                // Map uploaded values to fields
                $uploadedFieldValues = [];
                foreach ($uploaded as $uploadedDoc) {
                    $uploadedFieldValues[$uploadedDoc->required_document_field_id] = [
                        'id' => $uploadedDoc->id, // ID from subcontractor_required_documents table
                        'value' => $uploadedDoc->value,
                        'file_url' => ($uploadedDoc->value && (strpos($uploadedDoc->value, 'SubcontractorDocuments') !== false || strpos($uploadedDoc->value, 'upload') !== false)) ? url($uploadedDoc->value) : null,
                        'updated_at' => $uploadedDoc->updated_at,
                        'approval_status' => $uploadedDoc->approval_status ?? 'pending',
                        'rejection_reason' => $uploadedDoc->rejection_reason,
                        'approved_by' => $uploadedDoc->approved_by,
                        'approved_at' => $uploadedDoc->approved_at,
                    ];
                }

                // Check if document is complete
                $isComplete = SubcontractorRequiredDocument::isDocumentComplete(
                    $subcontractor->id,
                    $ids['customer_id'],
                    $ids['workspace_id'],
                    $docId
                );

                // Calculate document-level approval status
                $docApprovalStatus = null;
                $docRejectionReason = null;
                $docApprovedBy = null;
                $docApprovedAt = null;

                if ($uploaded->isNotEmpty()) {
                    $allApproved = $uploaded->every(fn($d) => $d->approval_status === 'approved');
                    $anyRejected = $uploaded->contains(fn($d) => $d->approval_status === 'rejected');

                    if ($allApproved) {
                        $docApprovalStatus = 'approved';
                        $firstApproved = $uploaded->first();
                        $docApprovedBy = $firstApproved->approved_by;
                        $docApprovedAt = $firstApproved->approved_at;
                    } elseif ($anyRejected) {
                        $docApprovalStatus = 'rejected';
                        $rejectedDoc = $uploaded->firstWhere('approval_status', 'rejected');
                        $docRejectionReason = $rejectedDoc->rejection_reason;
                        $docApprovedBy = $rejectedDoc->approved_by;
                        $docApprovedAt = $rejectedDoc->approved_at;
                    } else {
                        $docApprovalStatus = 'pending';
                    }
                }

                return [
                    'id' => $doc->id,
                    'title' => $doc->title,
                    'is_complete' => $isComplete,
                    'approval_status' => $docApprovalStatus,
                    'rejection_reason' => $docRejectionReason,
                    'approved_by' => $docApprovedBy,
                    'approved_at' => $docApprovedAt,
                    'fields' => $doc->requiredDocumentActiveField->map(function ($field) use ($uploadedFieldValues) {
                        $fieldId = $field->id;
                        $uploadedValue = $uploadedFieldValues[$fieldId] ?? null;

                        return [
                            'id' => $field->id,
                            'field_name' => $field->field_name,
                            'field_type' => $field->field_type,
                            'field_required' => $field->field_required,
                            'priority' => $field->priority,
                            'uploaded_value' => $uploadedValue ? $uploadedValue['value'] : null,
                            'uploaded_file_url' => $uploadedValue ? $uploadedValue['file_url'] : null,
                            'uploaded_id' => $uploadedValue ? $uploadedValue['id'] : null,
                            'last_updated' => $uploadedValue ? $uploadedValue['updated_at'] : null,
                        ];
                    }),
                ];
            });

            $subcontractor->required_documents = $documentsWithData;
        } else {
            $subcontractor->project_ids = [];
            $subcontractor->trade_id = null;
            $subcontractor->required_docs_status = 0;
            $subcontractor->required_docs_status_label = 'not_uploaded';
            $subcontractor->rejection_reason = null;
            $subcontractor->required_documents = [];
        }
        return $this->success($subcontractor, 'Subcontractor retrieved successfully');
    }

    public function getRequiredDocuments($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();

        // Verify subcontractor exists
        $subcontractor = User::where('users.user_type', config('constants.user_types.subcontractor'))
            ->where('users.del', '0')
            ->where('users.id', $id)
            ->join('subcontractor_companies', 'users.id', '=', 'subcontractor_companies.user_id')
            ->where('subcontractor_companies.customer_id', $ids['customer_id'])
            ->where('subcontractor_companies.workspace_id', $ids['workspace_id'])
            ->where('subcontractor_companies.del', '0')
            ->select('users.*')
            ->first();

        if (!$subcontractor) {
            return $this->message('Subcontractor not found', 404);
        }

        // Get all required documents for subcontractors from this company
        $requiredDocuments = RequiredDocument::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->with(['requiredDocumentActiveField'])
            ->get();

        // Get uploaded documents for this subcontractor
        $uploadedDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractor->id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->with('approver')
            ->get()
            ->groupBy('required_document_id');

        // Get system date format from BaseModel
        $baseModel = new SubcontractorRequiredDocument();
        $systemDateFormat = $baseModel->getUserDateFormat();

        // Merge uploaded data with required documents
        $documentsWithData = $requiredDocuments->map(function ($doc) use ($uploadedDocuments, $subcontractor, $ids, $systemDateFormat) {
            $docId = $doc->id;
            $uploaded = $uploadedDocuments->get($docId) ?? collect();

            // Map uploaded values to fields
            $uploadedFieldValues = [];
            foreach ($uploaded as $uploadedDoc) {
                $uploadedFieldValues[$uploadedDoc->required_document_field_id] = [
                    'id' => $uploadedDoc->id,
                    'value' => $uploadedDoc->value,
                    'file_url' => ($uploadedDoc->value && (strpos($uploadedDoc->value, 'SubcontractorDocuments') !== false || strpos($uploadedDoc->value, 'upload') !== false)) ? url($uploadedDoc->value) : null,
                    'updated_at' => $uploadedDoc->updated_at,
                    'approval_status' => $uploadedDoc->approval_status ?? 'pending',
                    'rejection_reason' => $uploadedDoc->rejection_reason,
                    'approved_by' => $uploadedDoc->approver ? $uploadedDoc->approver->name : null,
                    'approved_at' => $uploadedDoc->approved_at,
                ];
            }

            // Check if document is complete
            $isComplete = SubcontractorRequiredDocument::isDocumentComplete(
                $subcontractor->id,
                $ids['customer_id'],
                $ids['workspace_id'],
                $docId
            );

            // Calculate document-level approval status
            $docApprovalStatus = null;
            $docRejectionReason = null;
            $docApprovedBy = null;
            $docApprovedAt = null;

            if ($uploaded->isNotEmpty()) {
                $allApproved = $uploaded->every(fn($d) => $d->approval_status === 'approved');
                $anyRejected = $uploaded->contains(fn($d) => $d->approval_status === 'rejected');

                if ($allApproved) {
                    $docApprovalStatus = 'approved';
                    $firstApproved = $uploaded->first();
                    $docApprovedBy = $firstApproved->approver ? $firstApproved->approver->name : null;
                    $docApprovedAt = $firstApproved->approved_at;
                } elseif ($anyRejected) {
                    $docApprovalStatus = 'rejected';
                    $rejectedDoc = $uploaded->firstWhere('approval_status', 'rejected');
                    $docRejectionReason = $rejectedDoc->rejection_reason;
                    $docApprovedBy = $rejectedDoc->approver ? $rejectedDoc->approver->name : null;
                    $docApprovedAt = $rejectedDoc->approved_at;
                } else {
                    $docApprovalStatus = 'pending';
                }
            }

            return [
                'id' => $doc->id,
                'title' => $doc->title,
                'is_complete' => $isComplete,
                'approval_status' => $docApprovalStatus,
                'rejection_reason' => $docRejectionReason,
                'approved_by' => $docApprovedBy,
                'approved_at' => $docApprovedAt,
                'fields' => $doc->requiredDocumentActiveField->map(function ($field) use ($uploadedFieldValues, $systemDateFormat) {
                    $fieldId = $field->id;
                    $uploadedValue = $uploadedFieldValues[$fieldId] ?? null;

                    // Format date values according to system settings
                    $formattedValue = null;
                    if ($uploadedValue && $uploadedValue['value']) {
                        // If field type is date, format the stored Y-m-d date to system format
                        if ($field->field_type === 'date') {
                            try {
                                // Parse the stored date (Y-m-d format) and format it using system date format
                                $carbonDate = Carbon::createFromFormat('Y-m-d', $uploadedValue['value']);
                                $formattedValue = $carbonDate->format($systemDateFormat);
                            } catch (\Exception $e) {
                                // If parsing fails, try Carbon's parse method as fallback
                                try {
                                    $carbonDate = Carbon::parse($uploadedValue['value']);
                                    $formattedValue = $carbonDate->format($systemDateFormat);
                                } catch (\Exception $e2) {
                                    // If all parsing fails, return original value
                                    $formattedValue = $uploadedValue['value'];
                                }
                            }
                        } else {
                            // For non-date fields, return value as is
                            $formattedValue = $uploadedValue['value'];
                        }
                    }

                    return [
                        'id' => $field->id,
                        'field_name' => $field->field_name,
                        'field_type' => $field->field_type,
                        'field_required' => $field->field_required,
                        'priority' => $field->priority,
                        'uploaded_value' => $formattedValue,
                        'uploaded_file_url' => $uploadedValue ? $uploadedValue['file_url'] : null,
                        'uploaded_id' => $uploadedValue ? $uploadedValue['id'] : null,
                        'last_updated' => $uploadedValue ? $uploadedValue['updated_at'] : null,
                    ];
                }),
            ];
        });

        return $this->success($documentsWithData, 'Required documents retrieved successfully');
    }

    public function update(Request $request, $id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $subcontractor = User::where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->find($id);
        if (!$subcontractor) {
            return $this->message('Subcontractor not found', 404);
        }
        // Check authorization via subcontractor_companies
        $subcontractorCompany = SubcontractorCompany::where('user_id', $id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$subcontractorCompany) {
            return $this->message('You do not have access to this subcontractor', 403);
        }
        $validator = Validator::make($request->all(), [
            'company_name' => 'nullable|string|max:255',
            'abn' => 'nullable|string|max:20',
            'company_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
            'trade_id' => 'nullable|integer',
            'company_number' => 'nullable|string|max:255',
            'name' => 'required|string|max:255',
            'mobile_number' => 'required|string|max:20',
            'email' => ['nullable', 'email', 'max:255', Rule::unique('users', 'email')->ignore($id, 'id'), Rule::unique('emp_company_details', 'employee_email'), Rule::unique('employees_subcontractors', 'email')],
            'project_ids' => 'nullable|array',
            'project_ids.*' => 'integer',
            'contact_persons' => 'nullable|array',
            'contact_persons.*.name' => 'required|string|max:255',
            'contact_persons.*.email' => 'nullable|email|max:255',
            'contact_persons.*.phone' => 'nullable|string|max:20',
            'contact_persons.*.role' => 'nullable|string|max:255',
        ], [
            'company_name.required' => 'Company name is required',
            'name.required' => 'Owner name is required',
            'mobile_number.required' => 'Contact number is required',
            'email.unique' => 'Email already exists',
            'company_logo.image' => 'Company logo must be an image',
            'company_logo.mimes' => 'Company logo must be jpeg, png, jpg, gif, svg or webp',
            'company_logo.max' => 'Company logo must not exceed 2MB',
            'contact_persons.*.name.required' => 'Name is required for all contact persons',
            'contact_persons.*.email.email' => 'Email must be a valid email address',
        ]);
        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        // Handle company logo upload and retain the existing image if not updated
        $currentLogo = $subcontractor->company_logo;
        $companyLogoPath = $this->handleFileImageUpload($request, 'SubContractors', $currentLogo)['path'] ?? $currentLogo;
        $subcontractor->update([
            'name' => $request->name,
            'email' => $request->email,
            'mobile_number' => $request->mobile_number,
            'company_name' => $request->company_name,
            'abn' => $request->abn,
            'company_logo' => $companyLogoPath,
            'company_number' => $request->company_number,
        ]);
        // Update subcontractor_companies entry
        $subcontractorCompany->update([
            'trade_id' => $request->trade_id,
            'project_ids' => $request->project_ids ?? [],
        ]);
        // Update contact persons if provided
        if (isset($request->contact_persons) && is_array($request->contact_persons)) {
            // Delete existing contact persons (like CompanyController)
            $subcontractor->contactPersons()->delete();
            // Create new contact persons
            $ids = $this->getCustomerAndWorkspaceIds();
            foreach ($request->contact_persons as $contactPerson) {
                SubcontractorContactPerson::create([
                    'subcontractor_id' => $id,
                    'name' => $contactPerson['name'],
                    'email' => $contactPerson['email'] ?? null,
                    'phone' => $contactPerson['phone'] ?? null,
                    'role' => $contactPerson['role'] ?? null,
                    'customer_id' => $ids['customer_id'] ?? null,
                    'workspace_id' => $ids['workspace_id'] ?? null,
                    'del' => '0',
                ]);
            }
        }
        // Add logo URL to response
        if ($subcontractor->company_logo) {
            $subcontractor->company_logo_url = url($subcontractor->company_logo);
        } else {
            $subcontractor->company_logo_url = null;
        }
        // Add project_ids and trade_id to response
        $subcontractor->project_ids = $subcontractorCompany->project_ids ?? [];
        $subcontractor->trade_id = $subcontractorCompany->trade_id;
        // Load contact persons for response
        $subcontractor->load('contactPersons');
        // Check if there are any required documents for subcontractors
        // If no required documents exist, check induction status before auto-approving
        $requiredDocuments = RequiredDocument::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->get();

        if ($requiredDocuments->isEmpty()) {
            // No required documents defined, check if induction is also complete
            // Only auto-approve if there are no required documents AND no induction documents (or all signed)
            $allComplete = $this->areAllDocumentsAndInductionComplete(
                $subcontractor->id,
                $ids['customer_id'],
                $ids['workspace_id'],
                $subcontractorCompany
            );

            if ($allComplete) {
                // No required documents AND no induction (or all signed), mark as approved so projects flow correctly
                $subcontractorCompany->update(['required_docs_status' => SubcontractorCompany::STATUS_COMPLETE]);
            }
        }
        return $this->success($subcontractor, 'Subcontractor updated successfully');
    }

    /**
     * Remove the specified subcontractor from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $subcontractor = User::where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->find($id);
        if (!$subcontractor) {
            return $this->message('Subcontractor not found', 404);
        }
        // Check authorization via subcontractor_companies
        $subcontractorCompany = SubcontractorCompany::where('user_id', $id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$subcontractorCompany) {
            return $this->message('You do not have access to this subcontractor', 403);
        }
        // Soft delete the subcontractor_companies entry (not the user itself)
        $subcontractorCompany->update(['del' => '1']);
        return $this->message('Subcontractor deleted successfully');
    }

    public function toggleStatus($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $subcontractor = User::where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->find($id);
        if (!$subcontractor) {
            return $this->message('Subcontractor not found', 404);
        }
        $subcontractorCompany = SubcontractorCompany::where('user_id', $id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$subcontractorCompany) {
            return $this->message('You do not have access to this subcontractor', 403);
        }

        // Refresh the model to get latest values from database
        $subcontractor->refresh();

        // Get current values and ensure they are integers
        $currentActiveStatus = (int) $subcontractor->active_status;
        $currentLoginStatus = (int) $subcontractor->is_enable_login;

        // Toggle: if 1, set to 0; if 0 or null, set to 1
        $newActiveStatus = ($currentActiveStatus === 1) ? 0 : 1;
        $newLoginStatus = ($currentLoginStatus === 1) ? 0 : 1;

        $subcontractor->update([
            'active_status' => $newActiveStatus,
            'is_enable_login' => $newLoginStatus,
        ]);

        // Refresh again to get updated values
        $subcontractor->refresh();

        $statusText = ((int) $subcontractor->active_status === 1) ? 'activated' : 'deactivated';
        return $this->message('Subcontractor ' . $statusText . ' successfully');
    }

    public function sendCredentials($id)
    {

        $ids = $this->getCustomerAndWorkspaceIds();
        $subcontractor = User::where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->find($id);
        if (!$subcontractor) {
            return $this->error('Subcontractor not found', 404);
        }
        // Check authorization via subcontractor_companies
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$subcontractorCompany) {
            return $this->error('You do not have access to this subcontractor', 403);
        }
        if (empty($subcontractor->email)) {
            return $this->error('Subcontractor does not have an email address', 400);
        }
        // Generate a new 8-digit password
        $digits = 8;
        $password = rand(pow(10, $digits - 1), pow(10, $digits) - 1);
        // Update the password in the database
        $subcontractor->update([
            'password' => Hash::make($password)
        ]);
        // Send the email
        $this->sendSubcontractorCredentialsEmail($subcontractor, $password, $ids['customer_id'], $ids['workspace_id']);
        return $this->message('Login credentials sent successfully');
    }

    private function sendSubcontractorCredentialsEmail($subcontractor, $password, $customerId = null, $workspaceId = null)
    {
        try {
            $loginUrl = env('FRONTEND_URL');
            // Get company details using CompanyEmailDetails service
            $brandTitle = \App\Services\CompanyEmailDetails::getBrandTitle($customerId, $workspaceId);
            $emailParams = [
                'to' => $subcontractor->email,
                'subject' => 'Your Login Credentials - ' . $brandTitle,
                'msg' => view('Emails.employee-credentials', [
                    'subject' => 'Your Login Credentials',
                    'name' => $subcontractor->name ?? $subcontractor->company_name ?? 'Subcontractor',
                    'email' => $subcontractor->email,
                    'password' => $password,
                    'login_url' => $loginUrl ?? null,
                    'customer_id' => $customerId,
                    'workspace_id' => $workspaceId,
                ])->render(),
            ];
            return $this->SendInstantEmail($emailParams);
        } catch (\Exception $e) {
            Log::error('Error in sendSubcontractorCredentialsEmail: ' . $e->getMessage());
            return false;
        }
    }

    private function sendSubcontractorWelcomeEmail($subcontractor, $password = null, $customerId = null, $workspaceId = null)
    {
        try {
            // Get workspace name
            $workspaceName = '';
            if ($workspaceId) {
                $workspace = Workspace::find($workspaceId);
                if ($workspace) {
                    $workspaceName = $workspace->name ?? '';
                }
            }

            // Get customer name (the company that called/associated this subcontractor)
            $customerName = env('APP_NAME', 'WMS'); // Default fallback
            if ($customerId) {
                $customer = User::find($customerId);
                if ($customer) {
                    // Use company_name if available, otherwise use name
                    $customerName = $customer->company_name ?? $customer->name ?? $customerName;
                }
            }

            $subject = 'Welcome to ' . $customerName . '!';
            $emailParams = [
                'to' => $subcontractor->email,
                'subject' => $subject,
                'msg' => view('Emails.subcontractor-welcome', [
                    'subject' => $subject,
                    'name' => $subcontractor->name ?? $subcontractor->company_name ?? 'Subcontractor',
                    'company_name' => $subcontractor->company_name ?? '',
                    'email' => $subcontractor->email,
                    'password' => $password,
                    'workspace_id' => $workspaceId,
                    'workspace_name' => $workspaceName,
                    'customer_id' => $customerId,
                    'customer_name' => $customerName,
                ])->render()
            ];
            return $this->SendInstantEmail($emailParams);
        } catch (\Exception $e) {
            Log::error('Error in sendSubcontractorWelcomeEmail: ' . $e->getMessage(), [
                'subcontractor_id' => $subcontractor->id ?? null,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    private function generateSecretKey()
    {
        $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $length = 20;

        do {
            $secretKey = substr(str_shuffle(str_repeat($characters, ceil($length / strlen($characters)))), 0, $length);
        } while (User::where('secret_key', $secretKey)->exists());

        return $secretKey;
    }
    private function areAllDocumentsAndInductionComplete($subcontractorId, $customerId, $workspaceId, $subcontractorCompany = null)
    {
        // Get subcontractor company if not provided
        if (!$subcontractorCompany) {
            $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->first();
        }

        if (!$subcontractorCompany) {
            return false;
        }

        // Check required documents status
        // Get all required documents for subcontractors from this company
        $requiredDocuments = RequiredDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->get();

        $requiredDocsComplete = false;
        if ($requiredDocuments->isEmpty()) {
            // No required documents exist, consider as complete
            $requiredDocsComplete = true;
        } else {
            // Required documents exist, check if:
            // 1. All required documents are uploaded
            // 2. All required documents are approved (status = COMPLETE)
            $allDocumentsUploaded = SubcontractorRequiredDocument::areAllDocumentsComplete(
                $subcontractorId,
                $customerId,
                $workspaceId
            );

            $documentsApproved = $subcontractorCompany->required_docs_status == SubcontractorCompany::STATUS_COMPLETE;

            // Both upload and approval must be complete
            $requiredDocsComplete = $allDocumentsUploaded && $documentsApproved;
        }

        // Check induction status using the induction_status column directly
        // Get all active induction documents that are for subcontractors to determine if any exist
        $inductionDocuments = InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->get();

        // Filter documents that include 'subcontractor' in role_types or 'all'
        $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });

        $inductionComplete = false;
        if ($filteredInductionDocuments->isEmpty()) {
            // No induction documents exist for subcontractors, consider as complete
            $inductionComplete = true;
        } else {
            // Use the induction_status column directly for efficiency
            // Status 1 (INDUCTION_COMPLETE) means all documents are signed
            $inductionComplete = $subcontractorCompany->induction_status == SubcontractorCompany::INDUCTION_COMPLETE;
        }

        // Both required documents AND induction must be complete for projects to be active
        return $requiredDocsComplete && $inductionComplete;
    }

    public function checkSubcontractorByEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ], [
            'email.required' => 'Email is required',
            'email.email' => 'Email must be a valid email address',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $email = $request->email;

        // Search for subcontractor by email
        $subcontractor = User::where('email', $email)
            ->where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->first();

        if (!$subcontractor) {
            return $this->success([
                'exists' => false,
                'message' => 'No subcontractor found with this email address.',
            ], 'Subcontractor not found');
        }

        // Get all companies this subcontractor is associated with
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('del', '0')
            ->with('trade')
            ->get();

        // Get contact persons
        $contactPersons = SubcontractorContactPerson::where('subcontractor_id', $subcontractor->id)
            ->where('del', '0')
            ->get();

        // Prepare company associations data
        $associatedCompanies = [];
        foreach ($subcontractorCompanies as $subcontractorCompany) {
            $customer = User::find($subcontractorCompany->customer_id);
            if ($customer) {
                $associatedCompanies[] = [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'workspace_id' => $subcontractorCompany->workspace_id,
                    'trade_id' => $subcontractorCompany->trade_id,
                    'trade_title' => $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null,
                    'project_ids' => $subcontractorCompany->project_ids ?? [],
                ];
            }
        }

        // Prepare response data
        $responseData = [
            'exists' => true,
            'subcontractor' => [
                'id' => $subcontractor->id,
                'name' => $subcontractor->name,
                'email' => $subcontractor->email,
                'mobile_number' => $subcontractor->mobile_number,
                'company_name' => $subcontractor->company_name,
                'abn' => $subcontractor->abn,
                'company_number' => $subcontractor->company_number,
                'company_logo' => $subcontractor->company_logo,
                'company_logo_url' => $subcontractor->company_logo ? url($subcontractor->company_logo) : null,
                'active_status' => $subcontractor->active_status,
                'is_enable_login' => $subcontractor->is_enable_login,
                'associated_companies' => $associatedCompanies,
                'contact_persons' => $contactPersons->map(function ($contact) {
                    return [
                        'id' => $contact->id,
                        'name' => $contact->name,
                        'email' => $contact->email,
                        'phone' => $contact->phone,
                        'role' => $contact->role,
                    ];
                }),
            ],
        ];

        return $this->success($responseData, 'Subcontractor found successfully');
    }

    public function approveRejectRequiredDocuments(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $user = Auth::user();

        if (!$user) {
            return $this->error('Unauthorized access', 401);
        }

        // Check if it's a bulk operation (documents array) or single operation
        if ($request->has('documents') && is_array($request->documents)) {
            // Bulk operation - multiple documents
            return $this->bulkApproveRejectDocuments($request, $ids, $user);
        } else {
            // Single document operation
            return $this->singleApproveRejectDocument($request, $ids, $user);
        }
    }

    private function singleApproveRejectDocument(Request $request, $ids, $user)
    {
        $validator = Validator::make($request->all(), [
            'subcontractor_id' => 'required|integer|exists:users,id',
            'document_id' => 'required|integer|exists:required_documents,id',
            'action' => 'required|in:approve,reject',
            'rejection_reason' => 'required_if:action,reject|nullable|string|max:500',
        ], [
            'subcontractor_id.required' => 'Subcontractor ID is required.',
            'subcontractor_id.exists' => 'Subcontractor not found.',
            'document_id.required' => 'Document ID is required.',
            'document_id.exists' => 'Document not found.',
            'action.required' => 'Action is required (approve or reject).',
            'action.in' => 'Action must be either approve or reject.',
            'rejection_reason.required_if' => 'Rejection reason is required when rejecting.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        try {
            $subcontractorId = $request->subcontractor_id;
            $documentId = $request->document_id; // This is the required_documents template ID
            $action = $request->action;
            $rejectionReason = $request->rejection_reason;

            // Verify subcontractor exists and is associated with this company
            $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('del', '0')
                ->first();

            if (!$subcontractorCompany) {
                return $this->error('Subcontractor is not associated with your company.', 403);
            }

            // Verify the document template exists and belongs to this customer
            $documentTemplate = RequiredDocument::where('id', $documentId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('for_subcontractor', true)
                ->where('del', '0')
                ->first();

            if (!$documentTemplate) {
                return $this->error('Document template not found or not available for subcontractors.', 404);
            }

            // Get all uploaded documents (fields) for this document template
            $uploadedDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractorId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('required_document_id', $documentId)
                ->where('del', '0')
                ->get();

            if ($uploadedDocuments->isEmpty()) {
                return $this->error('No uploaded documents found for this template. The subcontractor has not uploaded any documents yet.', 404);
            }

            // Determine the new status
            $newStatus = $action === 'approve' ? SubcontractorRequiredDocument::STATUS_APPROVED : SubcontractorRequiredDocument::STATUS_REJECTED;

            // Update ALL fields of this document template at once
            $updatedCount = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractorId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('required_document_id', $documentId)
                ->where('del', '0')
                ->update([
                    'approval_status' => $newStatus,
                    'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                    'approved_by' => $user->id,
                    'approved_at' => now(),
                ]);

            // Update overall subcontractor company status based on all documents
            $this->updateSubcontractorCompanyDocumentStatus($subcontractorId, $ids['customer_id'], $ids['workspace_id']);

            // Send email notification to subcontractor
            try {
                $subcontractor = User::find($subcontractorId);
                if ($subcontractor && $subcontractor->email) {
                    $brandTitle = \App\Services\CompanyEmailDetails::getBrandTitle($ids['customer_id'], $ids['workspace_id']);
                    $subject = $action === 'approve' 
                        ? "Document Approved - {$documentTemplate->title} | {$brandTitle}"
                        : "Document Rejected - {$documentTemplate->title} | {$brandTitle}";
                    
                    \App\Jobs\SendNotificationEmailJob::dispatch([
                        'to' => $subcontractor->email,
                        'subject' => $subject,
                        'template' => 'Emails.subcontractor-document-status',
                        'template_data' => [
                            'subject' => $subject,
                            'subcontractor_name' => $subcontractor->name ?? $subcontractor->company_name ?? 'there',
                            'document_title' => $documentTemplate->title,
                            'status' => $action === 'approve' ? 'approved' : 'rejected',
                            'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ],
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                    Log::info('Document status email sent to subcontractor', [
                        'subcontractor_id' => $subcontractorId,
                        'subcontractor_email' => $subcontractor->email,
                        'document_id' => $documentId,
                        'action' => $action,
                    ]);
                }
            } catch (\Exception $e) {
                Log::error('Error sending document status email to subcontractor: ' . $e->getMessage(), [
                    'subcontractor_id' => $subcontractorId,
                    'document_id' => $documentId,
                    'action' => $action,
                ]);
                // Don't fail the request if email fails
            }

            // Log for audit trail
            Log::info('Subcontractor document ' . ($action === 'approve' ? 'approved' : 'rejected'), [
                'subcontractor_id' => $subcontractorId,
                'document_id' => $documentId,
                'document_title' => $documentTemplate->title,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
                'action' => $action,
                'rejection_reason' => $rejectionReason,
                'updated_by' => $user->id,
                'fields_updated' => $updatedCount,
            ]);

            $message = $action === 'approve'
                ? "Document '{$documentTemplate->title}' approved successfully."
                : "Document '{$documentTemplate->title}' rejected successfully.";

            return $this->success([
                'subcontractor_id' => $subcontractorId,
                'document_id' => $documentId,
                'document_title' => $documentTemplate->title,
                'action' => $action,
                'approval_status' => $newStatus,
                'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                'fields_updated' => $updatedCount,
                'approved_by' => $user->id,
                'approved_at' => now()->toDateTimeString(),
            ], $message, 200);
        } catch (\Exception $e) {
            Log::error('Error updating subcontractor document status: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'subcontractor_id' => $request->subcontractor_id ?? null,
                'document_id' => $request->document_id ?? null,
                'customer_id' => $ids['customer_id'] ?? null,
                'workspace_id' => $ids['workspace_id'] ?? null,
            ]);
            return $this->error('An error occurred while updating document status: ' . $e->getMessage(), 500);
        }
    }

    private function bulkApproveRejectDocuments(Request $request, $ids, $user)
    {
        $validator = Validator::make($request->all(), [
            'subcontractor_id' => 'required|integer|exists:users,id',
            'documents' => 'required|array|min:1',
            'documents.*.document_id' => 'required|integer|exists:required_documents,id',
            'documents.*.action' => 'required|in:approve,reject',
            'documents.*.rejection_reason' => 'nullable|string|max:500',
        ], [
            'subcontractor_id.required' => 'Subcontractor ID is required.',
            'subcontractor_id.exists' => 'Subcontractor not found.',
            'documents.required' => 'Documents array is required.',
            'documents.min' => 'At least one document must be provided.',
            'documents.*.document_id.required' => 'Document ID is required for each document.',
            'documents.*.document_id.exists' => 'One or more documents not found.',
            'documents.*.action.required' => 'Action is required for each document.',
            'documents.*.action.in' => 'Action must be either approve or reject.',
        ]);

        if ($validator->fails()) {
            return $this->error('Validation failed: ' . $validator->errors()->first(), 422);
        }

        try {
            $subcontractorId = $request->subcontractor_id;
            $updatedDocuments = [];
            $errors = [];

            // Verify subcontractor exists and is associated with this company
            $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('del', '0')
                ->first();

            if (!$subcontractorCompany) {
                return $this->error('Subcontractor is not associated with your company.', 403);
            }

            DB::beginTransaction();

            foreach ($request->documents as $docData) {
                $documentId = $docData['document_id'];
                $action = $docData['action'];
                $rejectionReason = $docData['rejection_reason'] ?? null;

                // Verify the document template exists and belongs to this customer
                $documentTemplate = RequiredDocument::where('id', $documentId)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->where('for_subcontractor', true)
                    ->where('del', '0')
                    ->first();

                if (!$documentTemplate) {
                    $errors[] = "Document ID {$documentId} not found or not available for subcontractors.";
                    continue;
                }

                // Check if any documents are uploaded for this template
                $hasUploads = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractorId)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->where('required_document_id', $documentId)
                    ->where('del', '0')
                    ->exists();

                if (!$hasUploads) {
                    $errors[] = "No uploaded documents found for '{$documentTemplate->title}'.";
                    continue;
                }

                // Validate rejection reason for rejected status
                if ($action === 'reject' && empty($rejectionReason)) {
                    $errors[] = "Rejection reason required for document '{$documentTemplate->title}'.";
                    continue;
                }

                // Determine the new status
                $newStatus = $action === 'approve' ? SubcontractorRequiredDocument::STATUS_APPROVED : SubcontractorRequiredDocument::STATUS_REJECTED;

                // Update ALL fields of this document template at once
                $updatedCount = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractorId)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->where('required_document_id', $documentId)
                    ->where('del', '0')
                    ->update([
                        'approval_status' => $newStatus,
                        'rejection_reason' => $action === 'reject' ? $rejectionReason : null,
                        'approved_by' => $user->id,
                        'approved_at' => now(),
                    ]);

                $updatedDocuments[] = [
                    'document_id' => $documentId,
                    'document_title' => $documentTemplate->title,
                    'action' => $action,
                    'approval_status' => $newStatus,
                    'fields_updated' => $updatedCount,
                ];
            }

            // Update overall subcontractor company status
            $this->updateSubcontractorCompanyDocumentStatus($subcontractorId, $ids['customer_id'], $ids['workspace_id']);

            // Send email notifications for each updated document
            try {
                $subcontractor = User::find($subcontractorId);
                if ($subcontractor && $subcontractor->email && !empty($updatedDocuments)) {
                    $brandTitle = \App\Services\CompanyEmailDetails::getBrandTitle($ids['customer_id'], $ids['workspace_id']);
                    
                    // Group documents by action (approved/rejected)
                    $approvedDocs = collect($updatedDocuments)->where('action', 'approve');
                    $rejectedDocs = collect($updatedDocuments)->where('action', 'reject');
                    
                    // Send email for approved documents if any
                    if ($approvedDocs->isNotEmpty()) {
                        $subject = "Documents Approved | {$brandTitle}";
                        \App\Jobs\SendNotificationEmailJob::dispatch([
                            'to' => $subcontractor->email,
                            'subject' => $subject,
                            'template' => 'Emails.subcontractor-document-status',
                            'template_data' => [
                                'subject' => $subject,
                                'subcontractor_name' => $subcontractor->name ?? $subcontractor->company_name ?? 'there',
                                'document_title' => $approvedDocs->pluck('document_title')->join(', '),
                                'status' => 'approved',
                                'rejection_reason' => null,
                                'customer_id' => $ids['customer_id'],
                                'workspace_id' => $ids['workspace_id'],
                            ],
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);
                    }
                    
                    // Send email for rejected documents if any
                    if ($rejectedDocs->isNotEmpty()) {
                        $subject = "Documents Rejected | {$brandTitle}";
                        \App\Jobs\SendNotificationEmailJob::dispatch([
                            'to' => $subcontractor->email,
                            'subject' => $subject,
                            'template' => 'Emails.subcontractor-document-status',
                            'template_data' => [
                                'subject' => $subject,
                                'subcontractor_name' => $subcontractor->name ?? $subcontractor->company_name ?? 'there',
                                'document_title' => $rejectedDocs->pluck('document_title')->join(', '),
                                'status' => 'rejected',
                                'rejection_reason' => 'Please check your portal for details.',
                                'customer_id' => $ids['customer_id'],
                                'workspace_id' => $ids['workspace_id'],
                            ],
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);
                    }
                }
            } catch (\Exception $e) {
                Log::error('Error sending bulk document status emails to subcontractor: ' . $e->getMessage(), [
                    'subcontractor_id' => $subcontractorId,
                ]);
                // Don't fail the request if email fails
            }

            DB::commit();

            return $this->success([
                'subcontractor_id' => $subcontractorId,
                'updated_count' => count($updatedDocuments),
                'updated_documents' => $updatedDocuments,
                'errors' => $errors,
            ], 'Documents updated successfully', 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error bulk updating subcontractor documents: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'subcontractor_id' => $request->subcontractor_id ?? null,
                'customer_id' => $ids['customer_id'] ?? null,
                'workspace_id' => $ids['workspace_id'] ?? null,
            ]);
            return $this->error('An error occurred while updating documents: ' . $e->getMessage(), 500);
        }
    }

    private function updateSubcontractorCompanyDocumentStatus($subcontractorId, $customerId, $workspaceId)
    {
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if (!$subcontractorCompany) {
            return;
        }

        // Check if all documents are uploaded
        $allUploaded = SubcontractorRequiredDocument::areAllDocumentsComplete($subcontractorId, $customerId, $workspaceId);

        if (!$allUploaded) {
            // Not all documents uploaded, set status to pending or not_uploaded
            $subcontractorCompany->update([
                'required_docs_status' => SubcontractorCompany::STATUS_PENDING,
            ]);
            return;
        }

        // Get all documents for this subcontractor
        $allDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractorId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->get();

        // Check approval status of all documents
        $allApproved = $allDocuments->every(function ($doc) {
            return $doc->approval_status === SubcontractorRequiredDocument::STATUS_APPROVED;
        });

        $hasRejected = $allDocuments->contains(function ($doc) {
            return $doc->approval_status === SubcontractorRequiredDocument::STATUS_REJECTED;
        });

        $hasPending = $allDocuments->contains(function ($doc) {
            return $doc->approval_status === SubcontractorRequiredDocument::STATUS_PENDING;
        });

        // Update status based on document approvals
        if ($allApproved) {
            // All documents approved
            $subcontractorCompany->update([
                'required_docs_status' => SubcontractorCompany::STATUS_COMPLETE,
                'rejection_reason' => null,
            ]);

            // Check if both required docs AND induction are complete
            $allComplete = $this->areAllDocumentsAndInductionComplete(
                $subcontractorId,
                $customerId,
                $workspaceId,
                $subcontractorCompany
            );

            if ($allComplete) {
                $subcontractorCompany->update(['required_docs_status' => SubcontractorCompany::STATUS_COMPLETE]);
            }
        } elseif ($hasRejected) {
            // At least one document rejected
            $rejectedDocs = $allDocuments->filter(function ($doc) {
                return $doc->approval_status === SubcontractorRequiredDocument::STATUS_REJECTED;
            });

            // Get rejection reasons (if any)
            $rejectionReasons = $rejectedDocs->pluck('rejection_reason')->filter()->unique()->implode('; ');

            $subcontractorCompany->update([
                'required_docs_status' => SubcontractorCompany::STATUS_REJECTED,
                'rejection_reason' => $rejectionReasons ?: 'One or more documents have been rejected.',
            ]);
        } elseif ($hasPending) {
            // Some documents still pending
            $subcontractorCompany->update([
                'required_docs_status' => SubcontractorCompany::STATUS_PENDING,
            ]);
        }
    }

    public function getCompaniesWithDocsStatus(Request $request)
    {
        $subcontractor = Auth::user();

        // Get all companies (customers) this subcontractor is associated with
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('del', '0')
            ->with('trade')
            ->get();

        $companies = [];
        foreach ($subcontractorCompanies as $subcontractorCompany) {
            $customer = User::find($subcontractorCompany->customer_id);
            if ($customer) {
                // Count required documents for this company
                $requiredDocsCount = RequiredDocument::where('customer_id', $subcontractorCompany->customer_id)
                    ->where('workspace_id', $subcontractorCompany->workspace_id)
                    ->where('for_subcontractor', true)
                    ->where('del', '0')
                    ->count();

                $companies[] = [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'current_workspace_id' => $customer->current_workspace_id,
                    'workspace_id' => $subcontractorCompany->workspace_id,
                    'trade_id' => $subcontractorCompany->trade_id,
                    'trade_title' => $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null,
                    'project_ids' => $subcontractorCompany->project_ids ?? [],
                    'required_docs_status' => $subcontractorCompany->required_docs_status,
                    'required_docs_status_label' => $subcontractorCompany->required_docs_status_label,
                    'required_docs_count' => $requiredDocsCount,
                ];
            }
        }

        return $this->success($companies, 'Companies with documents status retrieved successfully');
    }

    public function getCustomerAndAllSubcontractors(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();


        // Get pagination and search parameters
        $perPage = 12;
        $page = $request->input('page', 1);
        $search = $request->input('search', ''); // Search keyword

        // Get all subcontractors available in the system
        $allSubcontractorsQuery = User::where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->whereDoesntHave('subcontractorCompanies', function ($query) use ($ids) {
                $query->where('customer_id', $ids['customer_id'])
                    ->where('del', '0');
            })
            ->with('contactPersons');

        // Apply search filter if search keyword is provided
        if (!empty($search)) {
            $allSubcontractorsQuery->where(function ($query) use ($search) {
                $query->where('email', 'like', '%' . $search . '%');
            });
        }

        $allSubcontractorsQuery->latest('id');

        $allSubcontractorsPaginated = $allSubcontractorsQuery->paginate($perPage, ['*'], 'page', $page);

        $allSubcontractors = $allSubcontractorsPaginated->map(function ($subcontractor) {
            if ($subcontractor->company_logo) {
                $subcontractor->company_logo_url = url($subcontractor->company_logo);
            } else {
                $subcontractor->company_logo_url = null;
            }

            // Get all company associations for this subcontractor
            $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractor->id)
                ->where('del', '0')
                ->with('trade')
                ->get();

            $subcontractor->associated_companies = $subcontractorCompanies->map(function ($sc) {
                return [
                    'customer_id' => $sc->customer_id,
                    'workspace_id' => $sc->workspace_id,
                    'trade_id' => $sc->trade_id,
                    'trade_title' => $sc->trade ? $sc->trade->title : null,
                    // 'project_ids' => $sc->project_ids ?? [],
                ];
            });

            return $subcontractor;
        });

        // Explicitly return with HTTP 200 status to ensure correct status code
        return response()->json([
            'message' => 'Subcontractors retrieved successfully',
            'statusCode' => 200,
            'data' => [
                'all_subcontractors' => $allSubcontractors,
                'pagination' => [
                    'current_page' => $allSubcontractorsPaginated->currentPage(),
                    'last_page' => $allSubcontractorsPaginated->lastPage(),
                    'per_page' => $allSubcontractorsPaginated->perPage(),
                    'total' => $allSubcontractorsPaginated->total(),
                ],
            ],
        ], 200);
    }

    private function getSubcontractorsForDropdown(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->success([], 'Subcontractors retrieved successfully');
        }

        // Get subcontractors from subcontractor_companies table
        $subcontractorCompanies = SubcontractorCompany::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->with(['user' => function ($query) {
                $query->select('id', 'name', 'email', 'company_name')
                    ->where('user_type', config('constants.user_types.subcontractor'))
                    ->where('del', '0')
                    ->where('active_status', config('constants.user_status.active'));
            }])
            ->get();

        // Extract and format subcontractor data - only id, name, and email
        $subcontractors = $subcontractorCompanies->map(function ($sc) {
            if (!$sc->user) {
                return null;
            }
            return [
                'id' => $sc->user->id,
                'name' => $sc->user->name ?: $sc->user->company_name,
                'email' => $sc->user->email
            ];
        })->filter()->values();

        return $this->success($subcontractors, 'Subcontractors retrieved successfully');
    }

    public function getSubcontractorDetails($id)
    {

        $ids = $this->getCustomerAndWorkspaceIds();
        $subcontractorId = $id;
        // Validate subcontractor exists and belongs to this customer/workspace
        $subcontractor = User::where('id', $subcontractorId)
            ->where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->first();
        if (!$subcontractor) {
            return $this->error('Subcontractor not found or is not a valid subcontractor', 404);
        }
        // Check if subcontractor belongs to this customer/workspace
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$subcontractorCompany) {
            return $this->error('Subcontractor does not belong to this customer/workspace', 403);
        }
        // Get all project IDs assigned to this subcontractor from subcontractor_companies table
        $projectIds = $subcontractorCompany->project_ids ?? [];
        if (empty($projectIds)) {
            return $this->success([
                'subcontractor' => [
                    'id' => $subcontractor->id,
                    'name' => $subcontractor->company_name ?: $subcontractor->name,
                    'email' => $subcontractor->email,
                    'mobile_number' => $subcontractor->mobile_number ?? null,
                    'abn' => $subcontractor->abn ?? null
                ],
                'projects' => []
            ], 'Subcontractor details retrieved successfully. No projects assigned.');
        }
        // Get projects with their sites using the reusable helper method
        $projectsData = $this->getProjectsWithSites($projectIds, $ids['customer_id'], $ids['workspace_id']);
        // Get all employee IDs for this subcontractor
        $employeeMetas = EmployeeSubcontractorMeta::where('subcontractor_id', $subcontractorId)
            ->where('active', 1)
            ->pluck('emp_id')
            ->unique()
            ->toArray();
        // Get all invitations for this subcontractor's employees (for filtering employees per project)
        $invitations = collect();
        if (!empty($employeeMetas)) {
            $invitations = SubcontractorEmployeeInvitation::whereIn('employee_id', $employeeMetas)
                ->where('subcontractor_id', $subcontractorId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('invitation_status', 'accepted')
                ->where('induction_status', 1) // Completed induction
                ->get();
        }
        // Build projects array with employees for each project
        $projects = [];
        foreach ($projectsData as $project) {
            $projectId = $project['id'];
            // Get employees for this specific project who have accepted invitation and completed induction
            $projectInvitations = $invitations->where('project_id', $projectId);
            $employeeIds = $projectInvitations->pluck('employee_id')->unique()->toArray();
            $employees = [];
            if (!empty($employeeIds)) {
                $employeeRecords = EmployeeSubcontractor::whereIn('id', $employeeIds)->get();
                foreach ($employeeRecords as $employee) {
                    $employees[] = [
                        'id' => $employee->id,
                        'first_name' => $employee->first_name,
                        'middle_name' => $employee->middle_name ?? null,
                        'last_name' => $employee->last_name,
                        'email' => $employee->email,
                        'phone' => $employee->phone ?? null,
                        'profile_image' => $employee->profile_image ? url($employee->profile_image) : null,
                    ];
                }
            }
            // Add employees to project
            $project['employees'] = $employees;
            $projects[] = $project;
        }
        return $this->success([
            'subcontractor' => [
                'id' => $subcontractor->id,
                'name' => $subcontractor->company_name ?: $subcontractor->name,
                'email' => $subcontractor->email,
                'mobile_number' => $subcontractor->mobile_number ?? null,
                'abn' => $subcontractor->abn ?? null
            ],
            'projects' => $projects
        ], 'Subcontractor details retrieved successfully');
    }

    public function getSubcontractorManagementData($id)
    {
        $ids = $this->getCustomerAndWorkspaceIds();
        $subcontractorId = $id;
        $tabFilter = request()->get('tab', 'projects');
        $validTabs = ['projects', 'employees', 'defects', 'tenders', 'documents', 'induction'];
        if (!in_array($tabFilter, $validTabs)) {
            return $this->error('Invalid tab filter. Valid options: ' . implode(', ', $validTabs), 400);
        }
        $baseData = $this->prepareSubcontractorManagementBaseData($subcontractorId, $ids, $tabFilter);
        if ($baseData === false) {
            if (!User::where('id', $subcontractorId)->where('user_type', config('constants.user_types.subcontractor'))->where('del', '0')->exists()) {
                return $this->error('Subcontractor not found or is not a valid subcontractor', 404);
            }
            return $this->error('Subcontractor does not belong to this customer/workspace', 403);
        }
        switch ($tabFilter) {
            case 'projects':
                $responseData = $this->getSubcontractorManagementProjectsData($baseData, $subcontractorId, $ids);
                break;
            case 'employees':
                $responseData = $this->getSubcontractorManagementEmployeesData($baseData, $subcontractorId, $ids);
                break;
            case 'defects':
                $responseData = $this->getSubcontractorManagementDefectsData($baseData, $subcontractorId, $ids);
                break;
            case 'tenders':
                $responseData = $this->getSubcontractorManagementTendersData($baseData, $subcontractorId, $ids);
                break;
            case 'documents':
                $responseData = $this->getSubcontractorManagementDocumentsData($baseData, $subcontractorId, $ids);
                break;
            case 'induction':
                $responseData = $this->getSubcontractorManagementInductionData($baseData, $subcontractorId, $ids);
                break;
        }
        return $this->success($responseData, 'Subcontractor management data retrieved successfully');
    }

    public function assignProjectToSubcontractor(Request $request)
    {
        // Support both single project_id and multiple project_ids
        $validator = Validator::make($request->all(), [
            'subcontractor_id' => 'required|integer|exists:users,id',
            'project_id' => 'nullable|integer|exists:projects,id',
            'project_ids' => 'nullable|array',
            'project_ids.*' => 'integer|exists:projects,id',
        ], [
            'subcontractor_id.required' => 'Subcontractor ID is required.',
            'subcontractor_id.exists' => 'Subcontractor not found.',
            'project_id.integer' => 'Project ID must be an integer.',
            'project_id.exists' => 'Project not found.',
            'project_ids.array' => 'Project IDs must be an array.',
            'project_ids.*.integer' => 'Each project ID must be an integer.',
            'project_ids.*.exists' => 'One or more projects not found.',
        ]);

        // Require either project_id or project_ids
        if (!$request->has('project_id') && !$request->has('project_ids')) {
            return $this->error('Either project_id or project_ids is required.', 422);
        }

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $ids = $this->getCustomerAndWorkspaceIds();
        $subcontractorId = $request->subcontractor_id;

        // Get project IDs (support both single and multiple)
        $requestProjectIds = [];
        if ($request->has('project_ids') && is_array($request->project_ids)) {
            $requestProjectIds = $request->project_ids;
        } elseif ($request->has('project_id')) {
            $requestProjectIds = [$request->project_id];
        }

        // Normalize project IDs to integers
        $requestProjectIds = array_map('intval', array_filter($requestProjectIds, function($id) {
            return is_numeric($id) && $id > 0;
        }));

        if (empty($requestProjectIds)) {
            return $this->error('At least one valid project ID is required.', 422);
        }

        // Validate subcontractor
        $subcontractor = User::where('id', $subcontractorId)
            ->where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->first();
        if (!$subcontractor) {
            return $this->error('Subcontractor not found or is not a valid subcontractor', 404);
        }

        // Validate subcontractor belongs to this customer/workspace
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$subcontractorCompany) {
            return $this->error('Subcontractor does not belong to this customer/workspace', 403);
        }

        // Validate all projects exist and belong to this customer/workspace
        $validProjects = Project::whereIn('id', $requestProjectIds)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('is_deleted', 0)
            ->pluck('id')
            ->toArray();

        $invalidProjectIds = array_diff($requestProjectIds, $validProjects);
        if (!empty($invalidProjectIds)) {
            return $this->error('Some projects not found or do not belong to this customer/workspace: ' . implode(', ', $invalidProjectIds), 404);
        }

        // Get existing project IDs and normalize (for comparison only)
        $existingProjectIds = array_map('intval', array_filter($subcontractorCompany->project_ids ?? [], function($id) {
            return is_numeric($id) && $id > 0;
        }));

        // REPLACE the entire project list with the new assignments (not merge)
        // This ensures that only the projects sent in the request are assigned
        $allProjectIds = array_values(array_unique($validProjects));

        // Update subcontractor company
        $subcontractorCompany->update([
            'project_ids' => $allProjectIds,
        ]);

        // Refresh to ensure we have the latest data from database
        $subcontractorCompany->refresh();

        // Verify the update was saved correctly
        $savedProjectIds = array_map('intval', array_filter($subcontractorCompany->project_ids ?? [], function($id) {
            return is_numeric($id) && $id > 0;
        }));

        // Calculate which projects were newly assigned vs removed
        $newlyAssigned = array_diff($validProjects, $existingProjectIds);
        $removedProjects = array_diff($existingProjectIds, $validProjects);
        $keptProjects = array_intersect($validProjects, $existingProjectIds);

        $message = 'Projects assigned successfully.';
        if (!empty($removedProjects)) {
            $message .= ' Removed projects: ' . implode(', ', $removedProjects) . '.';
        }
        if (!empty($newlyAssigned)) {
            $message .= ' Newly assigned: ' . implode(', ', $newlyAssigned) . '.';
        }

        return $this->success([
            'subcontractor_id' => $subcontractorId,
            'assigned_projects' => $validProjects,
            'newly_assigned' => array_values($newlyAssigned),
            'kept_projects' => array_values($keptProjects),
            'removed_projects' => array_values($removedProjects),
            'total_assigned_projects' => count($savedProjectIds),
            'all_assigned_projects' => $savedProjectIds, // Return all assigned projects for verification
        ], $message);
    }
}
