<?php

namespace App\Http\Controllers\Subcontractor;


use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Roster;
use App\Models\RosterAssign;
use App\Models\RosterTemplate;
use App\Models\RosterHistory;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\SubcontractorCompany;
use App\Models\PublicHoliday;
use App\Models\Project;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use App\Traits\RosterTemplateTrait;

class RosterController extends Controller
{
    use RosterTemplateTrait;

    function hasShiftConflict($shifts, $currentShift)
    {
        // Ensure currentShift and its roster_template exist
        if (
            !isset($currentShift['roster_template']) ||
            empty($currentShift['roster_template']['start_time']) ||
            empty($currentShift['roster_template']['end_time'])
        ) {
            return false; // No conflict if data is invalid
        }
        // Helper function to parse time string (handles both H:i:s and H:i formats)
        $parseTime = function ($timeString) {
            if (empty($timeString)) {
                return null;
            }
            // Try H:i:s format first, then fall back to H:i
            try {
                return Carbon::createFromFormat('H:i:s', $timeString);
            } catch (\Exception $e) {
                try {
                    return Carbon::createFromFormat('H:i', $timeString);
                } catch (\Exception $e2) {
                    return null;
                }
            }
        };
        $currentStartDateTime = $parseTime($currentShift['roster_template']['start_time']);
        $currentEndDateTime = $parseTime($currentShift['roster_template']['end_time']);
        if (!$currentStartDateTime || !$currentEndDateTime) {
            return false; // Invalid time format
        }
        foreach ($shifts as $shift) {
            // Ensure shift and its roster_template exist
            if (
                !isset($shift['roster_template']) ||
                empty($shift['roster_template']['start_time']) ||
                empty($shift['roster_template']['end_time'])
            ) {
                continue; // Skip invalid shifts
            }
            // Avoid comparing the same shift
            if ($shift['roster_template'] === $currentShift['roster_template']) {
                continue;
            }
            $startDateTime = $parseTime($shift['roster_template']['start_time']);
            $endDateTime = $parseTime($shift['roster_template']['end_time']);
            if (!$startDateTime || !$endDateTime) {
                continue; // Skip invalid time formats
            }
            // Check for time overlap
            if ($currentStartDateTime < $endDateTime && $currentEndDateTime > $startDateTime) {
                return true; // Conflict found
            }
        }
        return false; // No conflicts found
    }

    function generateDatesWithDayNames($dates_array, $scheduler_mode)
    {
        $dates_with_info = [];
        foreach ($dates_array as $index => $date) {
            $carbonDate = Carbon::parse($date);
            $dayName = strtoupper($carbonDate->format('l')); // MONDAY, TUESDAY, etc.
            $dayNumber = $carbonDate->day;
            $dayOfWeek = $carbonDate->dayOfWeek; // 0 = Sunday, 1 = Monday, etc.   
            $dates_with_info[] = [
                'date' => $date,
                'day_name' => $dayName,
                'day_number' => $dayNumber,
                'day_of_week' => $dayOfWeek,
                'is_weekend' => in_array($dayOfWeek, [0, 6]), // Sunday and Saturday
                'formatted_date' => $carbonDate->format('Y-m-d'),
                'display_date' => $carbonDate->format('M d'),
            ];
        }
        return $dates_with_info;
    }

    public function index(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required',
            'end_date' => 'required',
            'scheduler_mode' => 'required',
            'selected_employee' => 'nullable',
            'site_id' => 'nullable|integer|exists:sites,id',
            'project_id' => 'nullable|integer|exists:projects,id'
        ]);

        if ($validator->fails()) {
            return $this->message($validator->errors()->first(), 404);
        }

        // Get authenticated subcontractor ID
        $subcontractorId = Auth::user()->id;

        // Get customer and workspace IDs from subcontractor company
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
            ->where('del', '0')
            ->first();

        if (!$subcontractorCompany) {
            return $this->message('Subcontractor company not found', 404);
        }

        $customerId = $subcontractorCompany->customer_id;
        $workspaceId = $subcontractorCompany->workspace_id;

        $validatedData = $validator->validated();
        $dates_array = $this->getDatesBetween($validatedData['start_date'], $validatedData['end_date']);
        // Generate dates with day names for proper frontend mapping
        $dates_with_info = $this->generateDatesWithDayNames($dates_array, $validatedData['scheduler_mode']);

        // Get employee IDs associated with this subcontractor
        $employeeQuery = EmployeeSubcontractorMeta::where('subcontractor_id', $subcontractorId)
            ->where('active', 1);

        // Filter by selected employees if provided
        $selected_employee = $request->selected_employee;
        if (!is_null($selected_employee) && count($selected_employee) > 0) {
            $employeeQuery->whereIn('emp_id', $selected_employee);
        }

        $employeeIds = $employeeQuery->pluck('emp_id')->toArray();

        if (empty($employeeIds)) {
            return response()->json([
                'message' => 'Roster fetched successfully',
                'data' => [
                    'dates' => $dates_array,
                    'dates_with_info' => $dates_with_info,
                    'scheduler_mode' => $validatedData['scheduler_mode'],
                    'users' => [],
                    'count' => 0,
                    'from' => 0,
                    'per_page' => 10
                ]
            ], 200);
        }

        // Get employees with their personal details
        $employees = EmployeeSubcontractor::whereIn('id', $employeeIds)->get();

        // Process shifts, holidays, and leaves for each employee
        $user_data = [];
        foreach ($employees as $employee) {
            $user_shifts = [];
            foreach ($dates_array as $date) {
                $shiftsQuery = RosterAssign::with('rosterTemplate', 'site')
                    ->where('assign_to', $employee->id)
                    ->where('schedule_date', $date)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('subcontractor_id', $subcontractorId);

                // Apply site_id filter if provided
                if ($request->filled('site_id')) {
                    $shiftsQuery->where('site_id', $request->site_id);
                }

                // Apply project_id filter if provided
                if ($request->filled('project_id')) {
                    $shiftsQuery->where('project_id', $request->project_id);
                }

                $shifts_of_this_box = $shiftsQuery->get();

                // Fetch all unique project IDs for this date's shifts
                $projectIds = $shifts_of_this_box->pluck('project_id')->filter()->unique()->toArray();
                $projects = [];
                if (!empty($projectIds)) {
                    $projects = Project::whereIn('id', $projectIds)
                        ->where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('is_deleted', '0')
                        ->pluck('title', 'id')
                        ->toArray();
                }

                $holiday = PublicHoliday::where('from', '<=', $date)
                    ->where('to', '>=', $date)
                    ->first();

                // Subcontractor employees don't have leave requests in this system
                $leave = null;

                $shifts = [];
                foreach ($shifts_of_this_box as $shift) {
                    if ($shift->rosterTemplate) {
                        // Convert to array format for hasShiftConflict
                        $shiftsArray = $shifts_of_this_box->map(function($s) {
                            if ($s->rosterTemplate) {
                                return [
                                    'roster_template' => [
                                        'start_time' => $s->rosterTemplate->start_time,
                                        'end_time' => $s->rosterTemplate->end_time,
                                    ]
                                ];
                            }
                            return null;
                        })->filter()->values()->toArray();

                        $shiftArray = [
                            'roster_template' => [
                                'start_time' => $shift->rosterTemplate->start_time,
                                'end_time' => $shift->rosterTemplate->end_time,
                            ]
                        ];

                        $conflict = $this->hasShiftConflict($shiftsArray, $shift);

                        // Get project title from pre-fetched projects
                        $projectTitle = isset($projects[$shift->project_id]) ? $projects[$shift->project_id] : null;

                        $shifts[] = [
                            'start_time' => $shift->rosterTemplate->start_time,
                            'end_time' => $shift->rosterTemplate->end_time,
                            'color_code' => $shift->rosterTemplate->color_code,
                            'template_id' => $shift->roster_template_id,
                            'conflict' => $conflict,
                            'site_id' => $shift->site_id,
                            'site_title' => $shift->site ? $shift->site->title : null,
                            'project_id' => $shift->project_id,
                            'project_title' => $projectTitle,
                        ];
                    }
                }

                $user_shifts[] = [
                    'date' => $date,
                    'holiday' => $holiday ? $holiday->title : null,
                    'leave' => $leave,
                    'shifts' => $shifts
                ];
            }

            // Format personal details similar to regular employees
            $personalDetails = [
                'emp_id' => (string)$employee->id,
                'first_name' => $employee->first_name,
                'middle_name' => $employee->middle_name,
                'last_name' => $employee->last_name,
                'image' => $employee->profile_image ? url($employee->profile_image) : 'assets/img/default.png'
            ];

            $user_data[] = [
                'id' => $employee->id,
                'user_type' => 1, // External/subcontractor employee
                'personal_details' => $personalDetails,
                'shifts' => $user_shifts
            ];
        }

        $count = count($user_data);
        $from = $request->input('from', 0);
        $per_page = $request->input('per_page', 10);

        $response_data = [
            'message' => 'Roster fetched successfully',
            'data' => [
                'dates' => $dates_array,
                'dates_with_info' => $dates_with_info,
                'scheduler_mode' => $validatedData['scheduler_mode'],
                'users' => $user_data,
                'count' => $count,
                'from' => $from,
                'per_page' => $per_page
            ]
        ];

        return response()->json($response_data, 200);
    }
}