<?php

namespace App\Http\Controllers\subcontractor;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\SubcontractorCompany;
use App\Models\SubcontractorRequiredDocument;
use App\Models\RequiredDocument;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;
use App\Models\EmployeeSubcontractor;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\Project;
use App\Models\EmployeeAttendance;
use App\Models\LeaveRequest;
use App\Models\Tender;
use App\Models\TenderLineItemsBid;
use App\Models\Defect;
use App\Models\SiteDocument;
use App\Models\SiteDocumentSignature;
use App\Models\Role;
use App\Models\DocumentType;
use App\Models\EmpPersonalDetails;
use App\Models\SubcontractorEmployeeDocument;
use App\Models\Adminsettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Traits\SubcontractorPortalTrait;
class SubcontractorPortalController extends Controller
{
    use SubcontractorPortalTrait;
    public function getAssociatedCompanies(Request $request)
    {
        $subcontractor = Auth::guard('admin')->user();
        if (!$subcontractor || $subcontractor->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized. Only subcontractors can access this endpoint.', 403);
        }
        // Get all companies (customers) this subcontractor is associated with
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('del', '0')
            ->get();
        $companies = [];
        foreach ($subcontractorCompanies as $subcontractorCompany) {
            $customer = User::find($subcontractorCompany->customer_id);
            if ($customer) {
                $companies[] = [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'current_workspace_id' => $customer->current_workspace_id,
                    'workspace_id' => $subcontractorCompany->workspace_id, // The workspace this subcontractor is associated with for this customer
                ];
            }
        }
        return $this->success($companies, 'Associated companies retrieved successfully');
    }

    public function getCompaniesProfile(Request $request)
    {
        $subcontractor = Auth::user();

        // Get all companies (customers) this subcontractor is associated with
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('del', '0')
            ->with('trade')
            ->get();
        $companies = [];
        $allProjectIds = [];
        foreach ($subcontractorCompanies as $subcontractorCompany) {
            $customer = User::find($subcontractorCompany->customer_id);
            if (!$customer) {
                continue;
            }

            $customerId = $subcontractorCompany->customer_id;
            $workspaceId = $subcontractorCompany->workspace_id;

            // Get required documents for this company
            $requiredDocuments = RequiredDocument::where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('for_subcontractor', true)
                ->where('del', '0')
                ->with(['requiredDocumentActiveField'])
                ->get();

            // Get uploaded documents for this subcontractor
            $uploadedDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractor->id)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->get()
                ->groupBy('required_document_id');

            // Merge uploaded data with required documents
            $documentsWithData = $requiredDocuments->map(function ($doc) use ($uploadedDocuments, $subcontractor, $customerId, $workspaceId) {
                $docId = $doc->id;
                $uploaded = $uploadedDocuments->get($docId) ?? collect();

                // Map uploaded values to fields
                $uploadedFieldValues = [];
                foreach ($uploaded as $uploadedDoc) {
                    $uploadedFieldValues[$uploadedDoc->required_document_field_id] = [
                        'id' => $uploadedDoc->id,
                        'value' => $uploadedDoc->value,
                        'file_url' => ($uploadedDoc->value && (strpos($uploadedDoc->value, 'SubcontractorDocuments') !== false || strpos($uploadedDoc->value, 'upload') !== false)) ? url($uploadedDoc->value) : null,
                        'updated_at' => $uploadedDoc->updated_at,
                    ];
                }

                // Check if document is complete
                $isComplete = SubcontractorRequiredDocument::isDocumentComplete(
                    $subcontractor->id,
                    $customerId,
                    $workspaceId,
                    $docId
                );

                // Calculate document-level approval status
                $docApprovalStatus = null;
                $docRejectionReason = null;
                $docApprovedBy = null;
                $docApprovedAt = null;

                if ($uploaded->isNotEmpty()) {
                    $allApproved = $uploaded->every(fn($d) => $d->approval_status === 'approved');
                    $anyRejected = $uploaded->contains(fn($d) => $d->approval_status === 'rejected');

                    if ($allApproved) {
                        $docApprovalStatus = 'approved';
                        $firstApproved = $uploaded->first();
                        $docApprovedBy = $firstApproved->approved_by;
                        $docApprovedAt = $firstApproved->approved_at;
                    } elseif ($anyRejected) {
                        $docApprovalStatus = 'rejected';
                        $rejectedDoc = $uploaded->firstWhere('approval_status', 'rejected');
                        $docRejectionReason = $rejectedDoc->rejection_reason;
                        $docApprovedBy = $rejectedDoc->approved_by;
                        $docApprovedAt = $rejectedDoc->approved_at;
                    } else {
                        $docApprovalStatus = 'pending';
                    }
                }

                return [
                    'id' => $doc->id,
                    'title' => $doc->title,
                    'is_complete' => $isComplete,
                    'approval_status' => $docApprovalStatus,
                    'rejection_reason' => $docRejectionReason,
                    'approved_by' => $docApprovedBy,
                    'approved_at' => $docApprovedAt,
                    'fields' => $doc->requiredDocumentActiveField->map(function ($field) use ($uploadedFieldValues) {
                        $fieldId = $field->id;
                        $uploadedValue = $uploadedFieldValues[$fieldId] ?? null;

                        return [
                            'id' => $field->id,
                            'field_name' => $field->field_name,
                            'field_type' => $field->field_type,
                            'field_required' => $field->field_required,
                            'priority' => $field->priority,
                            'uploaded_value' => $uploadedValue ? $uploadedValue['value'] : null,
                            'uploaded_file_url' => $uploadedValue ? $uploadedValue['file_url'] : null,
                            'uploaded_id' => $uploadedValue ? $uploadedValue['id'] : null,
                            'last_updated' => $uploadedValue ? $uploadedValue['updated_at'] : null,
                        ];
                    }),
                ];
            });

            // Get induction documents for this company
            $inductionDocuments = InductionDocument::where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->where('is_active', true)
                ->get();

            // Filter documents that include 'subcontractor' in role_types or 'all'
            $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
                $roleTypes = $doc->role_types ?? [];
                return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
            });

            // Get signature status for each induction document
            $inductionDocumentsWithStatus = $filteredInductionDocuments->map(function ($doc) use ($subcontractor) {
                // Check if subcontractor has signed this document version
                $hasSigned = InductionDocumentSignature::where('induction_document_id', $doc->id)
                    ->where('document_version', $doc->version)
                    ->where('user_id', $subcontractor->id)
                    ->where('is_valid', true)
                    ->where('del', '0')
                    ->exists();

                // Get signature details if signed
                $signature = null;
                if ($hasSigned) {
                    $signature = InductionDocumentSignature::where('induction_document_id', $doc->id)
                        ->where('document_version', $doc->version)
                        ->where('user_id', $subcontractor->id)
                        ->where('is_valid', true)
                        ->where('del', '0')
                        ->first();
                }

                return [
                    'id' => $doc->id,
                    'title' => $doc->title,
                    'document_type' => $doc->document_type,
                    'version' => $doc->version,
                    'file_path' => $doc->file_path,
                    'file_url' => $doc->file_path ? url($doc->file_path) : null,
                    'description' => $doc->description,
                    'role_types' => $doc->role_types ?? [],
                    'trades' => $doc->trades->map(function ($trade) {
                        return [
                            'id' => $trade->id,
                            'title' => $trade->title,
                        ];
                    })->toArray(),
                    'is_signed' => $hasSigned,
                    'signed_at' => $signature ? $signature->signed_at : null,
                    'signature_path' => $signature ? $signature->signature_path : null,
                    'signature_url' => $signature && $signature->signature_path ? url($signature->signature_path) : null,
                    'created_at' => $doc->created_at,
                ];
            })->values();

            // Calculate overall statistics
            $allRequiredDocsComplete = SubcontractorRequiredDocument::areAllDocumentsComplete(
                $subcontractor->id,
                $customerId,
                $workspaceId
            );

            $projectIds = $subcontractorCompany->project_ids ?? [];
            if (is_string($projectIds)) {
                $projectIds = json_decode($projectIds, true);
            }
            
            if (is_array($projectIds)) {
                $allProjectIds = array_merge($allProjectIds, $projectIds);
            }
            
            // Get full project details for the COMPANY specific array
            $projects = [];
            if (!empty($projectIds) && is_array($projectIds)) {
                $allProjects = Project::whereIn('id', $projectIds)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('is_deleted', '0')
                    ->where('active', 1)
                    ->pluck('id')
                    ->toArray();
                
                if (!empty($allProjects)) {
                    $projects = $this->getProjectsWithSites($allProjects, $customerId, $workspaceId)
                        ->map(function ($project) {
                            return [
                                'id' => $project['id'],
                                'title' => $project['title'],
                                'description' => $project['description'] ?? null,
                                'address' => $project['address'] ?? null,
                                'start_date' => $project['start_date'] ?? null,
                                'end_date' => $project['end_date'] ?? null,
                                'sites' => collect($project['sites'])->map(function ($site) {
                                    return [
                                        'id' => $site['id'],
                                        'title' => $site['title'],
                                        'address' => $site['address'] ?? null,
                                        'state' => $site['state'] ?? null,
                                        'suburb' => $site['suburb'] ?? null,
                                    ];
                                })->toArray()
                            ];
                        })
                        ->toArray();
                }
            }

            $companies[] = [
                'id' => $customer->id,
                'name' => $customer->name,
                'email' => $customer->email,
                'current_workspace_id' => $customer->current_workspace_id,
                'workspace_id' => $workspaceId,
                'trade_id' => $subcontractorCompany->trade_id,
                'trade_title' => $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null,
                'project_ids' => $subcontractorCompany->project_ids ?? [],
                'projects' => $projects,
                'required_docs_status' => $subcontractorCompany->required_docs_status,
                'required_docs_status_label' => $subcontractorCompany->required_docs_status_label,
                'rejection_reason' => $subcontractorCompany->rejection_reason,
                'required_documents' => [
                    'total_count' => $requiredDocuments->count(),
                    'completed_count' => $documentsWithData->where('is_complete', true)->count(),
                    'pending_count' => $documentsWithData->where('is_complete', false)->count(),
                    'all_complete' => $allRequiredDocsComplete,
                    'documents' => $documentsWithData,
                ],
                'induction_documents' => [
                    'total_count' => $inductionDocumentsWithStatus->count(),
                    'signed_count' => $inductionDocumentsWithStatus->where('is_signed', true)->count(),
                    'pending_count' => $inductionDocumentsWithStatus->where('is_signed', false)->count(),
                    'documents' => $inductionDocumentsWithStatus,
                ],
            ];
        }
        // Get authenticated user
        $user = Auth::user();
        // Check if user is subcontractor (user_type = 5)
        if ($user->user_type != config('constants.user_types.subcontractor')) {
            return $this->message('Unauthorized access', 403);
        }
        

        $uniqueProjectIds = array_unique($allProjectIds);

        $assignedProjects = Project::whereIn('id', $uniqueProjectIds)
            ->where('is_deleted', '0')
            ->select('id', 'title', 'active', 'start_date', 'end_date', 'address') 
            ->get()
            ->map(function ($project) {
                return [
                    'id' => $project->id,
                    'title' => $project->title,
                    'address' => $project->address,
                    'start_date' => $project->start_date,
                    'end_date' => $project->end_date,
                    // Return status as Active/Inactive string and boolean
                    'status' => $project->active == 1 ? 'Active' : 'Inactive',
                    'is_active' => (bool) $project->active
                ];
            });

        $userProfile = User::find($user->id);
        
        $userProfile->assigned_projects = $assignedProjects;

        $userProfile->projects_count = $assignedProjects->count();

        $data = [
            'profile' => $userProfile,
            'companies' => $companies,
        ];
        return $this->success($data, 'Profile and companies data retrieved successfully');
    }

    public function getInductionDocumentsForCompany(Request $request)
    {
        $subcontractor = Auth::user();

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer',
            'workspace_id' => 'required|integer',
        ], [
            'customer_id.required' => 'Customer ID is required.',
            'workspace_id.required' => 'Workspace ID is required.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $customerId = $request->customer_id;
        $workspaceId = $request->workspace_id;

        // Verify that the subcontractor is associated with this company
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if (!$subcontractorCompany) {
            return $this->error('You are not associated with this company.', 403);
        }

        // Get all active induction documents that are for subcontractors
        $documents = InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->get();

        // Filter documents that include 'subcontractor' in role_types or 'all'
        $filteredDocuments = $documents->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });

        // Get signature status for each document
        $documentsWithStatus = $filteredDocuments->map(function ($doc) use ($subcontractor, $customerId, $workspaceId) {
            // Check if subcontractor has signed this document version
            $hasSigned = InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('user_id', $subcontractor->id)
                ->where('is_valid', true)
                ->where('del', '0')
                ->exists();

            // For minor updates: if not signed current version, check if signed any previous version
            if (!$hasSigned && $doc->update_type === 'minor') {
                // Find root document
                $rootDocumentId = $doc->id;
                $currentDoc = $doc;
                while ($currentDoc && $currentDoc->parent_document_id) {
                    $rootDocumentId = $currentDoc->parent_document_id;
                    $currentDoc = InductionDocument::where('id', $rootDocumentId)
                        ->where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('del', '0')
                        ->first();
                    if (!$currentDoc) {
                        break;
                    }
                }
                
                // Get all document IDs in the chain (root and all children, excluding current version)
                $documentIdsInChain = InductionDocument::where(function($q) use ($rootDocumentId) {
                        $q->where('id', $rootDocumentId)
                          ->orWhere('parent_document_id', $rootDocumentId);
                    })
                    ->where('id', '!=', $doc->id) // Exclude current version
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->pluck('id')
                    ->toArray();
                
                // Check if subcontractor signed any previous version
                if (!empty($documentIdsInChain)) {
                    $hasSignedPreviousVersion = InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
                        ->where('user_id', $subcontractor->id)
                        ->where('is_valid', true)
                        ->where('del', '0')
                        ->exists();
                    
                    if ($hasSignedPreviousVersion) {
                        $hasSigned = true; // Mark as signed for minor updates if previous version was signed
                    }
                }
            }

            // Get signature details if signed
            $signature = null;
            if ($hasSigned) {
                // Try to get signature for current version first
                $signature = InductionDocumentSignature::where('induction_document_id', $doc->id)
                    ->where('document_version', $doc->version)
                    ->where('user_id', $subcontractor->id)
                    ->where('is_valid', true)
                    ->where('del', '0')
                    ->first();
                
                // If no signature for current version but it's a minor update, get any previous signature
                if (!$signature && $doc->update_type === 'minor') {
                    // Find root document again
                    $rootDocumentId = $doc->id;
                    $currentDoc = $doc;
                    while ($currentDoc && $currentDoc->parent_document_id) {
                        $rootDocumentId = $currentDoc->parent_document_id;
                        $currentDoc = InductionDocument::where('id', $rootDocumentId)
                            ->where('customer_id', $customerId)
                            ->where('workspace_id', $workspaceId)
                            ->where('del', '0')
                            ->first();
                        if (!$currentDoc) {
                            break;
                        }
                    }
                    
                    // Get any signature from previous versions
                    $documentIdsInChain = InductionDocument::where(function($q) use ($rootDocumentId) {
                            $q->where('id', $rootDocumentId)
                              ->orWhere('parent_document_id', $rootDocumentId);
                        })
                        ->where('id', '!=', $doc->id)
                        ->where('customer_id', $customerId)
                        ->where('workspace_id', $workspaceId)
                        ->where('del', '0')
                        ->pluck('id')
                        ->toArray();
                    
                    if (!empty($documentIdsInChain)) {
                        $signature = InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
                            ->where('user_id', $subcontractor->id)
                            ->where('is_valid', true)
                            ->where('del', '0')
                            ->orderBy('signed_at', 'desc')
                            ->first();
                    }
                }
            }

            return [
                'id' => $doc->id,
                'title' => $doc->title,
                'document_type' => $doc->document_type,
                'version' => $doc->version,
                'file_path' => $doc->file_path,
                'file_url' => $doc->file_path ? url($doc->file_path) : null,
                'description' => $doc->description,
                'role_types' => $doc->role_types ?? [],
                'trades' => $doc->trades->map(function ($trade) {
                    return [
                        'id' => $trade->id,
                        'title' => $trade->title,
                    ];
                })->toArray(),
                'is_signed' => $hasSigned,
                'signed_at' => $signature ? $signature->signed_at : null,
                'signed_file' => $signature ? $signature->signed_file : null,
                'signed_file_url' => $signature && $signature->signed_file ? url($signature->signed_file) : null,
                'signature_path' => $signature ? $signature->signature_path : null,
                'signature_url' => $signature && $signature->signature_path ? url($signature->signature_path) : null,
                'created_at' => $doc->created_at,
            ];
        })->values();

        return $this->success([
            'customer_id' => $customerId,
            'workspace_id' => $workspaceId,
            'documents' => $documentsWithStatus,
            'total_documents' => $documentsWithStatus->count(),
            'signed_count' => $documentsWithStatus->where('is_signed', true)->count(),
            'pending_count' => $documentsWithStatus->where('is_signed', false)->count(),
        ], 'Induction documents retrieved successfully');
    }

    public function signInductionDocument(Request $request)
    {
        $subcontractor = Auth::user();

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer',
            'workspace_id' => 'required|integer',
            'document_id' => 'required|integer|exists:induction_documents,id',
            'version' => 'nullable|string',
            'signature' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'signed_file' => 'nullable|file|mimes:pdf|max:20480', // Signed PDF with signature embedded (max 20MB)
            'notes' => 'nullable|string',
        ], [
            'customer_id.required' => 'Customer ID is required.',
            'workspace_id.required' => 'Workspace ID is required.',
            'document_id.required' => 'Document ID is required.',
            'signature.required' => 'Signature image is required.',
            'signature.image' => 'Signature must be an image file.',
            'signature.mimes' => 'Signature must be jpeg, png, jpg, gif, or svg.',
            'signature.max' => 'Signature image must not exceed 2MB.',
            'signed_file.file' => 'Signed file must be a valid file.',
            'signed_file.mimes' => 'Signed file must be a PDF.',
            'signed_file.max' => 'Signed file must not exceed 20MB.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $customerId = $request->customer_id;
        $workspaceId = $request->workspace_id;
        $documentId = $request->document_id;

        // Verify that the subcontractor is associated with this company
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if (!$subcontractorCompany) {
            return $this->error('You are not associated with this company.', 403);
        }

        // Get document
        $document = InductionDocument::where('id', $documentId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->first();

        if (!$document) {
            return $this->error('Document not found or not active', 404);
        }

        // Check if document is for subcontractors
        $roleTypes = $document->role_types ?? [];
        if (!in_array('all', $roleTypes) && !in_array('subcontractor', $roleTypes)) {
            return $this->error('You are not authorized to sign this document', 403);
        }

        // Use provided version or default to document's current version
        $version = $request->version ?? $document->version;

        // Verify version exists
        $versionDocument = InductionDocument::where('id', $documentId)
            ->where('version', $version)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if (!$versionDocument) {
            return $this->error('Version not found', 404);
        }

        // Check if already signed this version
        $existingSignature = InductionDocumentSignature::where('induction_document_id', $versionDocument->id)
            ->where('document_version', $version)
            ->where('user_id', $subcontractor->id)
            ->where('is_valid', true)
            ->where('del', '0')
            ->first();

        if ($existingSignature) {
            return $this->error('You have already signed this document version', 422);
        }

        DB::beginTransaction();
        try {
            // Upload signature image
            $signaturePath = null;
            if ($request->hasFile('signature')) {
                $signatureFile = $request->file('signature');
                $directory = 'InductionSignatures';
                $uploadPath = public_path($directory);
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }
                $extension = $signatureFile->getClientOriginalExtension();
                $imageName = $subcontractor->id . '_' . $versionDocument->id . '_' . time() . '.' . $extension;
                $signatureFile->move($uploadPath, $imageName);
                $signaturePath = "$directory/$imageName";
            }

            // Upload signed PDF file (document with signature embedded)
            $signedFilePath = null;
            if ($request->hasFile('signed_file')) {
                // Create a temporary request with only signed_file to avoid processing other files
                $signedFileRequest = new Request();
                $signedFileRequest->files->set('signed_file', $request->file('signed_file'));
                $uploadResult = $this->handleFileImageUpload($signedFileRequest, 'InductionSignedFiles');
                $signedFilePath = $uploadResult['path'] ?? null;
            }

            // Create signature record
            $signature = InductionDocumentSignature::create([
                'induction_document_id' => $versionDocument->id,
                'document_version' => $version,
                'employee_id' => null,
                'user_id' => $subcontractor->id,
                'signature_type' => 'subcontractor',
                'signature_path' => $signaturePath,
                'signed_file' => $signedFilePath,
                'signed_at' => now(),
                'is_valid' => true,
                'notes' => $request->notes,
                'customer_id' => $customerId,
                'workspace_id' => $workspaceId,
                'del' => '0',
            ]);

            // After signing, calculate and update induction status
            // This will set induction_status = 1 (INDUCTION_COMPLETE) if all documents are signed
            SubcontractorCompany::calculateAndUpdateInductionStatus(
                $subcontractor->id,
                $customerId,
                $workspaceId
            );

            DB::commit();

            return $this->success([
                'id' => $signature->id,
                'document_id' => $versionDocument->id,
                'document_title' => $versionDocument->title,
                'version' => $version,
                'signature_path' => $signaturePath,
                'signature_url' => $signaturePath ? url($signaturePath) : null,
                'signed_file' => $signedFilePath,
                'signed_file_url' => $signedFilePath ? url($signedFilePath) : null,
                'signed_at' => $signature->signed_at,
            ], 'Document signed successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error signing induction document: ' . $e->getMessage());
            return $this->error('An error occurred while uploading signature: ' . $e->getMessage(), 500);
        }
    }
    private function areAllDocumentsAndInductionComplete($subcontractorId, $customerId, $workspaceId, $subcontractorCompany = null)
    {
        // Get subcontractor company if not provided
        if (!$subcontractorCompany) {
            $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('del', '0')
                ->first();
        }

        if (!$subcontractorCompany) {
            return false;
        }

        // Check required documents status
        // Get all required documents for subcontractors from this company
        $requiredDocuments = RequiredDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->get();

        $requiredDocsComplete = false;
        if ($requiredDocuments->isEmpty()) {
            // No required documents exist, consider as complete
            $requiredDocsComplete = true;
        } else {
            // Required documents exist, check if:
            // 1. All required documents are uploaded
            // 2. All required documents are approved (status = COMPLETE)
            $allDocumentsUploaded = SubcontractorRequiredDocument::areAllDocumentsComplete(
                $subcontractorId,
                $customerId,
                $workspaceId
            );

            $documentsApproved = $subcontractorCompany->required_docs_status == SubcontractorCompany::STATUS_COMPLETE;

            // Both upload and approval must be complete
            $requiredDocsComplete = $allDocumentsUploaded && $documentsApproved;
        }

        // Check induction status using the induction_status column directly
        // Get all active induction documents that are for subcontractors to determine if any exist
        $inductionDocuments = InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->get();

        // Filter documents that include 'subcontractor' in role_types or 'all'
        $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });

        $inductionComplete = false;
        if ($filteredInductionDocuments->isEmpty()) {
            // No induction documents exist for subcontractors, consider as complete
            $inductionComplete = true;
        } else {
            // Use the induction_status column directly for efficiency
            // Status 1 (INDUCTION_COMPLETE) means all documents are signed
            $inductionComplete = $subcontractorCompany->induction_status == SubcontractorCompany::INDUCTION_COMPLETE;
        }

        // Both required documents AND induction must be complete for projects to be active
        return $requiredDocsComplete && $inductionComplete;
    }

    /**
     * Calculate induction status and label for a subcontractor-company relationship
     * 
     * @param int $subcontractorId
     * @param int $customerId
     * @param int $workspaceId
     * @return array ['induction_status' => int, 'induction_status_label' => string]
     */
    private function calculateInductionStatus($subcontractorId, $customerId, $workspaceId)
    {
        // Get all active induction documents that are for subcontractors
        $inductionDocuments = InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->get();

        // Filter documents that include 'subcontractor' in role_types or 'all'
        $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });

        if ($filteredInductionDocuments->isEmpty()) {
            // No induction documents exist, consider as complete
            return [
                'induction_status' => 1, // Complete
                'induction_status_label' => 'complete'
            ];
        }

        // Check if all documents are signed
        $allSigned = true;
        $signedCount = 0;
        foreach ($filteredInductionDocuments as $doc) {
            $hasSigned = InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('user_id', $subcontractorId)
                ->where('is_valid', true)
                ->where('del', '0')
                ->exists();

            if ($hasSigned) {
                $signedCount++;
            } else {
                $allSigned = false;
            }
        }

        // Determine status based on signing completion
        if ($allSigned) {
            return [
                'induction_status' => 1, // Complete
                'induction_status_label' => 'complete'
            ];
        } elseif ($signedCount > 0) {
            return [
                'induction_status' => 3, // Partial
                'induction_status_label' => 'partial'
            ];
        } else {
            return [
                'induction_status' => 0, // Not signed
                'induction_status_label' => 'not_signed'
            ];
        }
    }

    /**
     * Calculate overall status for required documents
     * Returns 1 (completed) only if ALL documents are complete AND approved
     * Returns 0 (pending) if any document is incomplete or pending approval
     * Returns 2 (rejected) if any document is rejected
     * 
     * @param int $subcontractorId
     * @param int $customerId
     * @param int $workspaceId
     * @return int 0 = pending, 1 = completed, 2 = rejected
     */
    private function calculateRequiredDocumentsOverallStatus($subcontractorId, $customerId, $workspaceId)
    {
        if (!$customerId || !$workspaceId) {
            return 0; // pending
        }

        // Get all required documents for this customer/workspace
        $requiredDocuments = RequiredDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->get();

        if ($requiredDocuments->isEmpty()) {
            // No required documents exist, consider as complete
            return 1; // completed
        }

        // Get all uploaded documents for this subcontractor
        $uploadedDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractorId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->get()
            ->groupBy('required_document_id');

        $allComplete = true;
        $allApproved = true;
        $hasRejected = false;

        // Check each required document
        foreach ($requiredDocuments as $doc) {
            $docId = $doc->id;
            $uploaded = $uploadedDocuments->get($docId) ?? collect();

            // Check if document is complete (all required fields uploaded)
            $isComplete = SubcontractorRequiredDocument::isDocumentComplete(
                $subcontractorId,
                $customerId,
                $workspaceId,
                $docId
            );

            if (!$isComplete) {
                // Document is not complete, overall status is pending
                $allComplete = false;
                $allApproved = false;
                break; // No need to check further
            }

            // Document is complete, check approval status
            if ($uploaded->isNotEmpty()) {
                // Check if all fields are approved
                $allFieldsApproved = $uploaded->every(function ($uploadedDoc) {
                    return $uploadedDoc->approval_status === SubcontractorRequiredDocument::STATUS_APPROVED;
                });

                $hasRejectedField = $uploaded->contains(function ($uploadedDoc) {
                    return $uploadedDoc->approval_status === SubcontractorRequiredDocument::STATUS_REJECTED;
                });

                $hasPendingField = $uploaded->contains(function ($uploadedDoc) {
                    return $uploadedDoc->approval_status === SubcontractorRequiredDocument::STATUS_PENDING ||
                           $uploadedDoc->approval_status === null;
                });

                if ($hasRejectedField) {
                    $hasRejected = true;
                    $allApproved = false;
                }

                if ($hasPendingField || !$allFieldsApproved) {
                    $allApproved = false;
                }
            } else {
                // Document has no uploaded fields (shouldn't happen if isComplete is true, but check anyway)
                $allComplete = false;
                $allApproved = false;
            }
        }

        // Determine overall status
        if (!$allComplete) {
            return 0; // pending
        }

        // If any document is rejected, return rejected
        if ($hasRejected) {
            return 2; // rejected
        }

        if ($allApproved) {
            return 1; // completed
        }

        // If any document is pending approval
        return 0; // pending
    }

    /**
     * Calculate overall status for induction documents
     * Returns 1 (completed) only if ALL documents are signed
     * Returns 0 (pending) if any document is not signed
     * 
     * @param int $subcontractorId
     * @param int $customerId
     * @param int $workspaceId
     * @return int 0 = pending, 1 = completed
     */
    private function calculateInductionOverallStatus($subcontractorId, $customerId, $workspaceId)
    {
        if (!$customerId || !$workspaceId) {
            return 0; // pending
        }

        // First, check the induction_status column from SubcontractorCompany
        // This is already properly calculated when signing documents
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if ($subcontractorCompany) {
            // Use the stored induction_status column if available
            // Status 1 = INDUCTION_COMPLETE
            if ($subcontractorCompany->induction_status == SubcontractorCompany::INDUCTION_COMPLETE) {
                return 1; // completed
            }
        }

        // Get all active induction documents that are for subcontractors
        // NOTE: Do NOT filter by update_type - we want ALL active documents including v1
        $inductionDocuments = InductionDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->where('is_active', true)
            ->get();

        // Filter documents that include 'subcontractor' in role_types or 'all'
        $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });

        if ($filteredInductionDocuments->isEmpty()) {
            // No induction documents exist, consider as complete
            return 1; // completed
        }

        // Check if all documents are signed (considering minor update logic)
        $allSigned = true;
        foreach ($filteredInductionDocuments as $doc) {
            // For major updates, must sign the current version
            // For minor updates, any previous version signature is valid
            $hasSigned = InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('user_id', $subcontractorId)
                ->where('is_valid', true)
                ->where('del', '0')
                ->exists();

            // If not signed current version and it's a minor update, check previous versions
            if (!$hasSigned && $doc->update_type === 'minor') {
                $hasSigned = SubcontractorCompany::hasSignedAnyVersionInChain($doc, $subcontractorId, $customerId, $workspaceId);
            }

            if (!$hasSigned) {
                $allSigned = false;
                break; // No need to check further
            }
        }

        if ($allSigned) {
            return 1; // completed
        } else {
            return 0; // pending
        }
    }

    public function updateCurrentCompany(Request $request)
    {

        return $this->success([
            'message' => 'Subcontractors now work independently and can access all associated companies.',
        ], 'This endpoint is deprecated. Subcontractors can access all their associated companies.');
    }

    public function publicGetSubcontractorEmployees(Request $request)
    {
        $secretKey = $request->secret_key;
        $check = User::where('secret_key', $secretKey)->first();
        if(!$check){
            return $this->error('Invalid secret key', 401);
        }else{
            return $this->getSubcontractorEmployees($request);
        }
    }
    public function getSubcontractorEmployees(Request $request)
    {
        if ($request->filled('search')) {
        $subcontractorId = auth()->id();
        $associationQuery = EmployeeSubcontractorMeta::query();
        }else{
        // Get subcontractor_id from request or use authenticated user's ID
        $subcontractorId = auth()->id();
        // Get employee IDs from employees_subcontractors_metas association table
        $associationQuery = EmployeeSubcontractorMeta::where('subcontractor_id', $subcontractorId);
        }
        // Filter by active status from association table
        // active=1 means true, active=0 means false
        if ($request->has('active')) {
            $activeValue = $request->active === '1' || $request->active === 1 || $request->active === 'true' || $request->active === true;
            $associationQuery->where('active', $activeValue);
        }
        $employeeIds = $associationQuery->pluck('emp_id')->toArray();
        // DEBUG: Log the employee IDs found
        Log::info('Employee IDs from employees_subcontractors_metas', [
            'subcontractor_id' => $subcontractorId,
            'employee_ids' => $employeeIds
        ]);
        if (empty($employeeIds)) {
            return $this->withCount(collect([]), 'No employees found for this subcontractor');
        }
        // Build query for employees from employees_subcontractors table
        $query = EmployeeSubcontractor::whereIn('id', $employeeIds);
        // DEBUG: Check if employees exist
        $foundEmployees = EmployeeSubcontractor::whereIn('id', $employeeIds)->pluck('id')->toArray();
        Log::info('Employees found in employees_subcontractors', [
            'looking_for' => $employeeIds,
            'found' => $foundEmployees,
            'missing' => array_diff($employeeIds, $foundEmployees)
        ]);
        // Apply search filter across multiple employee columns
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('email', 'like', '%' . $searchTerm . '%')
                    ->orWhere('first_name', 'like', '%' . $searchTerm . '%')
                    ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                    ->orWhere('last_name', 'like', '%' . $searchTerm . '%')
                    ->orWhere('phone', 'like', '%' . $searchTerm . '%')
                    ->orWhere('address', 'like', '%' . $searchTerm . '%')
                    ->orWhere('state', 'like', '%' . $searchTerm . '%')
                    ->orWhere('suburb', 'like', '%' . $searchTerm . '%');
            });
        }
        // Eager load associations
        $query->with(['associations' => function ($q) use ($subcontractorId) {
            $q->where('subcontractor_id', $subcontractorId);
        }]);
        $query->orderBy('id', 'desc');
        // Get filtered employees
        $users_collection = $query->get();
        // Filter by project_id if provided
        if ($request->filled('project_id')) {
            $projectId = (int) $request->project_id;
            
            // Get all employee IDs from the collection
            $allEmployeeIds = $users_collection->pluck('id')->toArray();
            
            // Fetch all invitations for this project in bulk
            $invitedEmployeeIds = [];
            if (!empty($allEmployeeIds)) {
                $invitedEmployeeIds = SubcontractorEmployeeInvitation::where('subcontractor_id', $subcontractorId)
                    ->whereIn('employee_id', $allEmployeeIds)
                    ->where('project_id', $projectId)
                    ->where('invitation_status', 'accepted')
                    ->pluck('employee_id')
                    ->toArray();
            }
            
            // Filter collection to only include employees with this project
            $users_collection = $users_collection->filter(function ($employee) use ($projectId, $invitedEmployeeIds) {
                // Check associations
                $hasProjectInAssociation = false;
                if ($employee->associations) {
                    foreach ($employee->associations as $association) {
                        if (!empty($association->project_ids) && is_array($association->project_ids)) {
                            if (in_array($projectId, $association->project_ids)) {
                                $hasProjectInAssociation = true;
                                break;
                            }
                        }
                    }
                }
                
                // Check invitations (from bulk fetch)
                $hasProjectInInvitation = in_array($employee->id, $invitedEmployeeIds);
                
                return $hasProjectInAssociation || $hasProjectInInvitation;
            })->values();
            
            if ($users_collection->isEmpty()) {
                return $this->withCount(collect([]), 'No employees found for this project');
            }
        }
        // Collect all unique project IDs from all associations
        $allProjectIds = [];
        foreach ($users_collection as $employee) {
            foreach ($employee->associations as $association) {
                if (!empty($association->project_ids) && is_array($association->project_ids)) {
                    $allProjectIds = array_merge($allProjectIds, $association->project_ids);
                }
            }
        }
        // Also collect project IDs from subcontractor_employee_invitations table
        $employeeIdsForInvitations = $users_collection->pluck('id')->toArray();
        if (!empty($employeeIdsForInvitations)) {
            $invitationProjectIds = SubcontractorEmployeeInvitation::where('subcontractor_id', $subcontractorId)
                ->whereIn('employee_id', $employeeIdsForInvitations)
                ->where('invitation_status', 'accepted')
                ->pluck('project_id')
                ->toArray();
            $allProjectIds = array_merge($allProjectIds, $invitationProjectIds);
        }
        $allProjectIds = array_unique($allProjectIds);
        // Fetch all projects in one query
        $projects = [];
        if (!empty($allProjectIds)) {
            $projects = Project::whereIn('id', $allProjectIds)
                ->select('id', 'title', 'customer_id', 'workspace_id')
                ->where('active', 1)
                ->where('is_deleted', '0')
                ->get()
                ->makeVisible(['customer_id', 'workspace_id']) // Make customer_id and workspace_id visible
                ->keyBy('id')
                ->toArray();
        }
        // Collect all unique customer_id and workspace_id pairs from projects and invitations
        $customerWorkspacePairs = [];
        // From projects
        foreach ($projects as $project) {
            $customerId = $project['customer_id'] ?? null;
            $workspaceId = $project['workspace_id'] ?? null;
            if ($customerId && $workspaceId) {
                $key = $customerId . '_' . $workspaceId;
                if (!isset($customerWorkspacePairs[$key])) {
                    $customerWorkspacePairs[$key] = [
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId
                    ];
                }
            }
        }
        // From invitations (for employees that might not have associations)
        if (!empty($employeeIdsForInvitations)) {
            $invitations = SubcontractorEmployeeInvitation::where('subcontractor_id', $subcontractorId)
                ->whereIn('employee_id', $employeeIdsForInvitations)
                ->where('invitation_status', 'accepted')
                ->select('customer_id', 'workspace_id')
                ->distinct()
                ->get();   
            foreach ($invitations as $invitation) {
                $customerId = $invitation->customer_id ?? null;
                $workspaceId = $invitation->workspace_id ?? null;
                if ($customerId && $workspaceId) {
                    $key = $customerId . '_' . $workspaceId;
                    if (!isset($customerWorkspacePairs[$key])) {
                        $customerWorkspacePairs[$key] = [
                            'customer_id' => $customerId,
                            'workspace_id' => $workspaceId
                        ];
                    }
                }
            }
        }
        // Fetch company_name from adminsettings for all customer/workspace pairs in bulk
        $companyNamesMap = $this->getCompanyNamesBulk($customerWorkspacePairs);
        // Transform associations to include project details and add company name to each employee
        $users_collection->transform(function ($employee) use ($projects, $subcontractorId, $companyNamesMap) {
            // Collect customer/workspace pairs for this employee
            $employeeCustomerWorkspacePairs = [];
            // From associations
            if ($employee->associations) {
                $employee->associations->transform(function ($association) use ($projects, &$employeeCustomerWorkspacePairs) {
                    if (!empty($association->project_ids) && is_array($association->project_ids)) {
                        // Transform project_ids array to include project details
                        $association->projects = array_map(function ($projectId) use ($projects, &$employeeCustomerWorkspacePairs) {
                            // Convert projectId to integer for proper key matching
                            $projectIdInt = (int) $projectId;
                            $project = $projects[$projectIdInt] ?? null;
                            if ($project) {
                                $customerId = $project['customer_id'] ?? null;
                                $workspaceId = $project['workspace_id'] ?? null;
                                if ($customerId && $workspaceId) {
                                    $key = $customerId . '_' . $workspaceId;
                                    if (!isset($employeeCustomerWorkspacePairs[$key])) {
                                        $employeeCustomerWorkspacePairs[$key] = [
                                            'customer_id' => $customerId,
                                            'workspace_id' => $workspaceId
                                        ];
                                    }
                                }
                            }
                            return [
                                'id' => $projectIdInt,
                                'name' => $project['title'] ?? null,
                                'customer_id' => $project['customer_id'] ?? null,
                                'workspace_id' => $project['workspace_id'] ?? null,
                            ];
                        }, $association->project_ids);
                    } else {
                        $association->projects = [];
                    }
                    return $association;
                });
            }
            // From invitations
            $invitations = SubcontractorEmployeeInvitation::where('subcontractor_id', $subcontractorId)
                ->where('employee_id', $employee->id)
                ->where('invitation_status', 'accepted')
                ->select('customer_id', 'workspace_id')
                ->distinct()
                ->get();
            foreach ($invitations as $invitation) {
                $customerId = $invitation->customer_id ?? null;
                $workspaceId = $invitation->workspace_id ?? null;
                if ($customerId && $workspaceId) {
                    $key = $customerId . '_' . $workspaceId;
                    if (!isset($employeeCustomerWorkspacePairs[$key])) {
                        $employeeCustomerWorkspacePairs[$key] = [
                            'customer_id' => $customerId,
                            'workspace_id' => $workspaceId
                        ];
                    }
                }
            }
            // Get company names for this employee's customer/workspace pairs
            $employeeCompanyNames = [];
            foreach ($employeeCustomerWorkspacePairs as $key => $pair) {
                $companyName = $companyNamesMap[$key] ?? null;
                if ($companyName) {
                    $employeeCompanyNames[] = $companyName;
                }
            }
            // Attach company names to employee (can be multiple if employee works with multiple companies)
            $employee->company_names = array_unique($employeeCompanyNames);
            // Also attach primary company name (first one)
            $employee->company_name = !empty($employeeCompanyNames) ? $employeeCompanyNames[0] : null; 
            return $employee;
        });
        // DEBUG: Log final result
        Log::info('Final employee collection', [
            'count' => $users_collection->count(),
            'ids' => $users_collection->pluck('id')->toArray()
        ]);

        return $this->withCount($users_collection, 'Employees retrieved successfully');
    }
    public function getAllSubcontractorEmployees(Request $request)
    {
        try {
            // Get the currently logged-in subcontractor ID
            $currentSubcontractorId = auth()->user()->id;

            // Get search parameter
            $search = $request->input('search', '');

            // Get all employees from other subcontractors with ONLY employees_subcontractors fields
            $employees = DB::table('employees_subcontractors_metas as es')
                ->join('employees_subcontractors as emp', 'es.emp_id', '=', 'emp.id')
                ->leftJoin('users as sub', 'es.subcontractor_id', '=', 'sub.id')
                ->where('es.subcontractor_id', '!=', $currentSubcontractorId)
                ->where('es.active', 1)
                ->when($search, function ($query) use ($search) {
                    return $query->where(function ($q) use ($search) {
                        $q->where('emp.email', 'like', '%' . $search . '%')
                            ->orWhere('emp.first_name', 'like', '%' . $search . '%')
                            ->orWhere('emp.last_name', 'like', '%' . $search . '%')
                            ->orWhere(DB::raw("CONCAT(emp.first_name, ' ', emp.last_name)"), 'like', '%' . $search . '%');
                    });
                })
                ->select(
                    'emp.id',
                    'emp.first_name',
                    'emp.middle_name',
                    'emp.last_name',
                    'emp.email',
                    'emp.phone',
                    'emp.profile_image',
                    'emp.address',
                    'emp.state',
                    'emp.suburb',
                    'emp.dob',
                    'emp.work_experience',
                    'emp.trade_qualified',
                    'emp.trade_qualified_year',
                    'emp.trade_licensed',
                    'emp.trade_licensed_year',
                    'emp.year_commenced',
                    'emp.citizenship_status',
                    'emp.allergies',
                    'emp.previous_injuries',
                    'emp.medical_condition',
                    'emp.visa_attachement',
                    'emp.signature',
                    'emp.medical_attachement_files',
                    'emp.created_at',
                    'emp.updated_at',
                    // Include subcontractor info for reference
                    'es.subcontractor_id',
                    'sub.name as subcontractor_name'
                )
                ->orderBy('sub.name', 'asc')
                ->orderBy('emp.first_name', 'asc')
                ->get();

            // Format the response with full URLs for files
            $formattedEmployees = $employees->map(function ($employee) {
                return [
                    'id' => $employee->id,
                    'first_name' => $employee->first_name,
                    'middle_name' => $employee->middle_name,
                    'last_name' => $employee->last_name,
                    'email' => $employee->email,
                    'phone' => $employee->phone,
                    'profile_image' => $employee->profile_image ? url($employee->profile_image) : null,
                    'address' => $employee->address,
                    'state' => $employee->state,
                    'suburb' => $employee->suburb,
                    'dob' => $employee->dob,
                    'work_experience' => $employee->work_experience,
                    'trade_qualified' => $employee->trade_qualified,
                    'trade_qualified_year' => $employee->trade_qualified_year,
                    'trade_licensed' => $employee->trade_licensed,
                    'trade_licensed_year' => $employee->trade_licensed_year,
                    'year_commenced' => $employee->year_commenced,
                    'citizenship_status' => $employee->citizenship_status,
                    'allergies' => $employee->allergies,
                    'previous_injuries' => $employee->previous_injuries,
                    'medical_condition' => $employee->medical_condition,
                    'visa_attachement' => $employee->visa_attachement ? url($employee->visa_attachement) : null,
                    'signature' => $employee->signature ? url($employee->signature) : null,
                    'medical_attachement_files' => $employee->medical_attachement_files ?
                        collect(explode(',', $employee->medical_attachement_files))->map(function ($file) {
                            return url(trim($file));
                        })->toArray() : null,
                    'created_at' => $employee->created_at,
                    'updated_at' => $employee->updated_at,
                    // Extra info for reference
                    'subcontractor_id' => $employee->subcontractor_id,
                    'subcontractor_name' => $employee->subcontractor_name,
                ];
            });

            $data = [
                'total_employees' => $formattedEmployees->count(),
                'search_query' => $search,
                'employees' => $formattedEmployees
            ];

            return $this->success($data, 'Employees retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Error fetching subcontractor employees', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return $this->error('Failed to retrieve employees', 500);
        }
    }

    public function getProjectSubcontractorEmployees(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'project_id' => 'required|integer|exists:projects,id',
            ]);

            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }

            $ids = $this->getCustomerAndWorkspaceIds();
            $customerId = $ids['customer_id'];
            $workspaceId = $ids['workspace_id'];
            $projectId = $request->project_id;

            // Verify project belongs to customer/workspace
            $project = Project::where('id', $projectId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('is_deleted', '0')
                ->first();

            if (!$project) {
                return $this->error('Project not found or you do not have access to it', 404);
            }

            // Get employees with accepted invitations, completed induction, and completed documents for this project
            $invitations = SubcontractorEmployeeInvitation::where('project_id', $projectId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('invitation_status', 'accepted')
                ->when(!$request->has('task'), function ($query) {
                    $query->where('induction_status', 1) // Complete
                          ->where('required_docs_status', 1); // Approved
                })
                ->with(['employee' => function ($query) {
                    $query->select('id', 'first_name', 'middle_name', 'last_name', 'email', 'phone', 'profile_image');
                }])
                ->get();

            if ($invitations->isEmpty()) {
                return $this->success([], 'No eligible employees found for this project');
            }

            // Format the response
            $employees = $invitations->map(function ($invitation) {
                $employee = $invitation->employee;
                if (!$employee) {
                    return null;
                }

                return [
                    'id' => $employee->id,
                    'first_name' => $employee->first_name,
                    'middle_name' => $employee->middle_name ?? null,
                    'last_name' => $employee->last_name,
                    'full_name' => trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? '')),
                    'email' => $employee->email,
                    'phone' => $employee->phone ?? null,
                    'profile_image' => $employee->profile_image ? url($employee->profile_image) : null,
                    'invitation_id' => $invitation->id,
                    'subcontractor_id' => $invitation->subcontractor_id,
                    'project_id' => $invitation->project_id,
                    'invitation_status' => $invitation->invitation_status,
                    'induction_status' => $invitation->induction_status,
                    'required_docs_status' => $invitation->required_docs_status,
                ];
            })->filter()->values();

            return $this->success($employees, 'Eligible subcontractor employees retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Error getting project subcontractor employees: ' . $e->getMessage(), [
                'project_id' => $request->project_id ?? null,
                'trace' => $e->getTraceAsString()
            ]);
            return $this->error('Failed to retrieve employees. Error: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get subcontractor employees assigned to a specific project
     * Returns employees from SubcontractorEmployeeInvitation for the given project_id
     * Only returns employees belonging to the authenticated subcontractor
     */
    public function getEmployeesAssignedToProject(Request $request)
    {
        try {
            // Get authenticated subcontractor
            $subcontractor = Auth::user();
            if (!$subcontractor || $subcontractor->user_type != config('constants.user_types.subcontractor')) {
                return $this->error('Unauthorized: Only subcontractors can access this endpoint', 403);
            }

            $subcontractorId = $subcontractor->id;

            // Validate request
            $validator = Validator::make($request->all(), [
                'project_id' => 'required|integer|exists:projects,id',
            ]);

            if ($validator->fails()) {
                return $this->handleValidationFailure($validator);
            }

            $projectId = $request->project_id;

            // Get customer and workspace IDs
            $ids = $this->getCustomerAndWorkspaceIds();
            if (!$ids) {
                return $this->error('Unable to determine customer and workspace', 400);
            }

            $customerId = $ids['customer_id'];
            $workspaceId = $ids['workspace_id'];

            // Verify project exists and is accessible
            $project = Project::where('id', $projectId)
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('is_deleted', '0')
                ->first();

            if (!$project) {
                return $this->error('Project not found or you do not have access to it', 404);
            }

            // Get employees with accepted invitations for this project
            // IMPORTANT: Filter by subcontractor_id to only get employees belonging to the authenticated subcontractor
            $invitations = SubcontractorEmployeeInvitation::where('project_id', $projectId)
                ->where('subcontractor_id', $subcontractorId) // Only get employees for this subcontractor
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('invitation_status', 'accepted')
                ->with(['employee' => function ($query) {
                    $query->select('id', 'first_name', 'middle_name', 'last_name', 'email', 'phone', 'profile_image');
                }])
                ->get();

            if ($invitations->isEmpty()) {
                return $this->success([], 'No employees assigned to this project');
            }

            // Format the response
            $employees = $invitations->map(function ($invitation) use ($projectId) {
                $employee = $invitation->employee;
                if (!$employee) {
                    return null;
                }

                return [
                    'id' => $employee->id,
                    'first_name' => $employee->first_name,
                    'middle_name' => $employee->middle_name ?? null,
                    'last_name' => $employee->last_name,
                    'full_name' => trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? '')),
                    'email' => $employee->email,
                    'phone' => $employee->phone ?? null,
                    'profile_image' => $employee->profile_image ? url($employee->profile_image) : null,
                    'invitation_id' => $invitation->id,
                    'subcontractor_id' => $invitation->subcontractor_id,
                    'project_id' => $projectId,
                    'invitation_status' => $invitation->invitation_status,
                    'induction_status' => $invitation->induction_status,
                    'required_docs_status' => $invitation->required_docs_status,
                ];
            })->filter()->values();

            return $this->success($employees, 'Subcontractor employees assigned to project retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Error getting employees assigned to project: ' . $e->getMessage(), [
                'project_id' => $request->project_id ?? null,
                'subcontractor_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);
            return $this->error('Failed to retrieve employees. Error: ' . $e->getMessage(), 500);
        }
    }

    public function getSubcontractorEmployeeAttendance(Request $request)
    {
        try {
            // Get authenticated subcontractor
            $subcontractor = Auth::guard('admin')->user();
            if (!$subcontractor || $subcontractor->user_type != config('constants.user_types.subcontractor')) {
                return $this->error('Unauthorized. Only subcontractors can access this endpoint.', 403);
            }

            $subcontractorId = $subcontractor->id;

            // Build query for attendance records where subcontractor_id matches
            $query = EmployeeAttendance::where('subcontractor_id', $subcontractorId);

            // Apply date range filter if provided
            if ($request->filled('from') && $request->filled('to')) {
                $query->whereBetween('date', [$request->from, $request->to]);
            } elseif ($request->filled('from')) {
                $query->where('date', '>=', $request->from);
            } elseif ($request->filled('to')) {
                $query->where('date', '<=', $request->to);
            }

            // Apply status filter if provided
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }

            // Apply search functionality
            if ($request->filled('search')) {
                $searchTerm = $request->search;
                $query->where(function ($q) use ($searchTerm) {
                    // Search in subcontractor employee's name
                    $q->whereHas('subcontractorEmployee', function ($subquery) use ($searchTerm) {
                        $subquery->where(function ($nameQuery) use ($searchTerm) {
                            $nameQuery->where('first_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('last_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('email', 'like', '%' . $searchTerm . '%');
                        });
                    })
                    // Search in added by name
                    ->orWhereHas('addedby', function ($subquery) use ($searchTerm) {
                        $subquery->where(function ($nameQuery) use ($searchTerm) {
                            $nameQuery->where('first_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('last_name', 'like', '%' . $searchTerm . '%');
                        });
                    })
                    // Search in site title
                    ->orWhereHas('sites', function ($subquery) use ($searchTerm) {
                        $subquery->where('title', 'like', '%' . $searchTerm . '%');
                    })
                    // Search in date
                    ->orWhere('date', 'like', '%' . $searchTerm . '%')
                    // Search in check_in or check_out
                    ->orWhere('check_in', 'like', '%' . $searchTerm . '%')
                    ->orWhere('check_out', 'like', '%' . $searchTerm . '%');
                });
            }

            // Load relationships
            $query->with([
                'subcontractorEmployee',
                'sites',
                'addedby',
                'breaks',
                'empTeamsMembers.empTeamsList'
            ]);

            $query->orderBy('id', 'desc');

            // Get results
            $query_result = $query->get();
            
            // Transform to simple format with only required fields
            $formattedData = $query_result->map(function ($attendance) {
                // Get employee name from subcontractor employee
                $employeeName = '';
                if ($attendance->subcontractorEmployee) {
                    $firstName = $attendance->subcontractorEmployee->first_name ?? '';
                    $middleName = $attendance->subcontractorEmployee->middle_name ?? '';
                    $lastName = $attendance->subcontractorEmployee->last_name ?? '';
                    $employeeName = trim($firstName . ' ' . $middleName . ' ' . $lastName);
                }
                
                // Get site title
                $siteTitle = $attendance->sites ? ($attendance->sites->title ?? '') : '';
                
                return [
                    'name' => $employeeName,
                    'site_title' => $siteTitle,
                    'start_time' => $attendance->check_in ?? '',
                    'end_time' => $attendance->check_out ?? '',
                    'working_hours' => $attendance->working_hours ?? '',
                    'date' => $attendance->date ?? '',
                    'status' => $attendance->status ?? 0,
                ];
            });
            
            // Use success() method which automatically handles pagination via prepareData() when page/per_page are present
            return $this->success($formattedData, 'Subcontractor employee attendance retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Error getting subcontractor employee attendance: ' . $e->getMessage());
            return $this->error('Failed to retrieve attendance list. Error: ' . $e->getMessage(), 500);
        }
    }
    
    public function getSubcontractorEmployeeLeaves(Request $request)
    {
        try {
            // Get authenticated subcontractor
            $subcontractor = Auth::guard('admin')->user();
            if (!$subcontractor || $subcontractor->user_type != config('constants.user_types.subcontractor')) {
                return $this->error('Unauthorized. Only subcontractors can access this endpoint.', 403);
            }

            $subcontractorId = $subcontractor->id;

            // Build query for leave requests where subcontractor_id matches
            $query = LeaveRequest::where('subcontractor_id', $subcontractorId);

            // Apply date range filter if provided
            // This filters leave requests that overlap with the specified date range
            if ($request->filled('from') && $request->filled('to')) {
                $fromDate = $request->from;
                $toDate = $request->to;
                $query->where(function ($q) use ($fromDate, $toDate) {
                    // Leave request overlaps with the date range if:
                    // - Leave starts before or on the end date AND
                    // - Leave ends on or after the start date
                    $q->where('from', '<=', $toDate)
                      ->where('to', '>=', $fromDate);
                });
            } elseif ($request->filled('from')) {
                // Filter leaves that end on or after the from date
                $query->where('to', '>=', $request->from);
            } elseif ($request->filled('to')) {
                // Filter leaves that start on or before the to date
                $query->where('from', '<=', $request->to);
            }

            // Apply status filter if provided
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }

            // Apply search functionality
            if ($request->filled('search')) {
                $searchTerm = $request->search;
                $query->where(function ($q) use ($searchTerm) {
                    // Search in subcontractor employee's name
                    $q->whereHas('subcontractorEmployee', function ($subquery) use ($searchTerm) {
                        $subquery->where(function ($nameQuery) use ($searchTerm) {
                            $nameQuery->where('first_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('middle_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('last_name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('email', 'like', '%' . $searchTerm . '%');
                        });
                    })
                    // Search in leave package title
                    ->orWhereHas('leavepackage', function ($subquery) use ($searchTerm) {
                        $subquery->where('title', 'like', '%' . $searchTerm . '%');
                    })
                    // Search in reason
                    ->orWhere('reason', 'like', '%' . $searchTerm . '%')
                    // Search in date range (from/to)
                    ->orWhere('from', 'like', '%' . $searchTerm . '%')
                    ->orWhere('to', 'like', '%' . $searchTerm . '%');
                });
            }

            // Load relationships
            $query->with([
                'subcontractorEmployee',
                'leavepackage'
            ]);

            $query->orderBy('id', 'desc');

            // Get results
            $query_result = $query->get();
            
            // Collect unique customer_id and workspace_id pairs from leave requests
            $customerWorkspacePairs = [];
            foreach ($query_result as $leave) {
                if ($leave->customer_id && $leave->workspace_id) {
                    $key = $leave->customer_id . '_' . $leave->workspace_id;
                    if (!isset($customerWorkspacePairs[$key])) {
                        $customerWorkspacePairs[$key] = [
                            'customer_id' => $leave->customer_id,
                            'workspace_id' => $leave->workspace_id,
                        ];
                    }
                }
            }
            
            // Fetch company names from adminsettings in bulk
            $companyNamesMap = [];
            if (!empty($customerWorkspacePairs)) {
                foreach ($customerWorkspacePairs as $key => $pair) {
                    $companyName = DB::table('adminsettings')
                        ->where('customer_id', $pair['customer_id'])
                        ->where('workspace', $pair['workspace_id'])
                        ->where('key', 'company_company_name')
                        ->value('value');
                    
                    // If not found with workspace, try without workspace filter
                    if (empty($companyName)) {
                        $companyName = DB::table('adminsettings')
                            ->where('customer_id', $pair['customer_id'])
                            ->where('key', 'company_company_name')
                            ->value('value');
                    }
                    
                    // Fallback to User model company_name or name if not found in settings
                    if (empty($companyName)) {
                        $customer = User::find($pair['customer_id']);
                        if ($customer) {
                            $companyName = $customer->company_name ?? $customer->name ?? null;
                        }
                    }
                    
                    $companyNamesMap[$key] = $companyName;
                }
            }
            
            // Transform to simple format with only required fields
            $formattedData = $query_result->map(function ($leave) use ($companyNamesMap) {
                // Get employee name from subcontractor employee
                $employeeName = '';
                if ($leave->subcontractorEmployee) {
                    $firstName = $leave->subcontractorEmployee->first_name ?? '';
                    $middleName = $leave->subcontractorEmployee->middle_name ?? '';
                    $lastName = $leave->subcontractorEmployee->last_name ?? '';
                    $employeeName = trim($firstName . ' ' . $middleName . ' ' . $lastName);
                }
                
                // Get leave package title
                $leavePackageTitle = $leave->leavepackage ? ($leave->leavepackage->title ?? '') : '';
                
                // Get company name
                $companyName = null;
                if ($leave->customer_id && $leave->workspace_id) {
                    $key = $leave->customer_id . '_' . $leave->workspace_id;
                    $companyName = $companyNamesMap[$key] ?? null;
                }
                
                return [
                    'id' => $leave->id ?? null,
                    'employee_name' => $employeeName,
                    'leave_package' => $leavePackageTitle,
                    'from_date' => $leave->from ?? '',
                    'to_date' => $leave->to ?? '',
                    'reason' => $leave->reason ?? '',
                    'status' => $leave->status ?? 0,
                    'full_day' => $leave->full_day ?? 0,
                    'start_time' => $leave->start_time ?? null,
                    'end_time' => $leave->end_time ?? null,
                    'applied_by' => $leave->applied_by ?? null,
                    'approved_by' => $leave->approved_by ?? null,
                    'company_name' => $companyName,
                ];
            });
            
            // Use success() method which automatically handles pagination via prepareData() when page/per_page are present
            return $this->success($formattedData, 'Subcontractor employee leave requests retrieved successfully');
        } catch (\Exception $e) {
            Log::error('Error getting subcontractor employee leave requests: ' . $e->getMessage());
            return $this->error('Failed to retrieve leave requests list. Error: ' . $e->getMessage(), 500);
        }
    }

    public function createEmployee(Request $request)
    {
        


        // Check if key is provided in the request
        if ($request->filled('key') && $request->key !== null) {
            // Find user by secret_key
            $userByKey = User::where('secret_key', $request->key)->first();

            if (!$userByKey) {
                return $this->error('Invalid key. No user found with the provided key.', 404);
            }

            // Use the user ID from the key lookup as subcontractor_id
            $subcontractorId = $userByKey->id;
        }else{
            $subcontractorId = Auth::user()->id;
        }
        // Validation rules (aligned with ExternalEmployeeOnboardingController for matching fields)
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email',
            'mobile' => 'required|numeric',
            'street_address' => 'required|string|max:500',
            'suburb' => 'required|regex:/^[a-zA-Z\s]*$/',
            'state' => 'required|regex:/^[a-zA-Z\s]*$/',
            'dob' => 'required|date|before:' . Carbon::now()->subYears(15)->format('Y-m-d'),
            'trade_qualified' => 'required|string|max:255',
            'trade_qualified_year' => 'required_if:trade_qualified,=,1',
            'trade_licensed' => 'required|string|max:255',
            'trade_licensed_year' => 'required_if:trade_licensed,=,1',
            'work_experience' => 'required|string|max:255',
            'year_commenced' => 'required|string|max:10',
            'legally_australia' => 'required|string|max:255',
            'citizenship_status' => 'required|integer',
            'allergies' => 'nullable|string',
            'previous_injuries' => 'nullable|string',
            'medical_condition' => 'nullable|string',
            'profile_image' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'visa_attachement' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'signature' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'medical_attachement_files' => 'nullable|array',
            'medical_attachement_files.*' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx,xls,xlsx,ppt,pptx|max:10240',
            // Emergency contact fields
            'relation' => 'nullable|string|max:255',
            'emergencyName' => 'nullable|string|max:255',
            // Meta data
            'company_id' => 'nullable|integer',
            'project_ids' => 'nullable|array',
            'project_ids.*' => 'integer',
            'active' => 'nullable|boolean',
        ], [
            'email.required' => 'Email is required',
            'email.email' => 'Email must be a valid email address',
            'first_name.required' => 'First name is required',
            'last_name.required' => 'Last name is required',
            'mobile.required' => 'Mobile number is required',
            'mobile.numeric' => 'Mobile number must be numeric',
            'street_address.required' => 'Street address is required',
            'suburb.required' => 'Suburb is required',
            'suburb.regex' => 'The suburb field must contain only alphabetic characters.',
            'state.required' => 'State is required',
            'state.regex' => 'The state field must contain only alphabetic characters.',
            'dob.required' => 'Date of birth is required',
            'dob.before' => 'You must be at least 15 years old to register.',
            'trade_qualified.required' => 'The trade qualified field is required',
            'trade_qualified_year.required_if' => 'The trade qualified year field is required',
            'trade_licensed.required' => 'The trade licensed field is required',
            'trade_licensed_year.required_if' => 'The trade licensed year field is required',
            'work_experience.required' => 'The work experience field is required',
            'year_commenced.required' => 'The year commenced field is required',
            'citizenship_status.required' => 'Citizenship status is required',
            'profile_image.mimes' => 'The image must be a file of type: jpeg, png, jpg, gif.',
            'profile_image.max' => 'The image may not be greater than 1MB.',
            'visa_attachement.mimes' => 'The visa attachment must be a file of type: jpeg, png, jpg, gif.',
            'visa_attachement.max' => 'The visa attachment may not be greater than 1MB.',
            'signature.mimes' => 'The signature must be a file of type: jpeg, png, jpg, gif.',
            'signature.max' => 'The signature may not be greater than 1MB.',
            'medical_attachement_files.*.mimes' => 'Medical attachment files must be of type: jpeg, png, jpg, gif.',
            'medical_attachement_files.*.max' => 'Each medical attachment file may not be greater than 1MB.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }


        // Check if employee already exists by email (if email is provided)
        $existingEmployee = null;
        $isNewEmployee = false;
        $plainPassword = null; // Initialize password variable

        if ($request->filled('email')) {
            $existingEmployee = EmployeeSubcontractor::where('email', $request->email)->first();
        }

        // If employee exists, check if already associated with this subcontractor
        if ($existingEmployee) {
            $existingMeta = EmployeeSubcontractorMeta::where('emp_id', $existingEmployee->id)
                ->where('subcontractor_id', $subcontractorId)
                ->first();

            if ($existingMeta) {
                DB::rollBack();
                return $this->error('This employee is already associated with your subcontractor account.', 422);
            }

            // Employee exists but not associated with this subcontractor - use existing employee
            $employee = $existingEmployee;
            $isNewEmployee = false;
        } else {
            // Employee doesn't exist - create new employee
            $isNewEmployee = true;

            // Generate password for new employee
            $digits = 8;
            $plainPassword = rand(pow(10, $digits - 1), pow(10, $digits) - 1);

            // Prepare employee data
            $employeeData = [
                'first_name' => $request->first_name,
                'middle_name' => $request->middle_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'password' => Hash::make($plainPassword),
                'phone' => $request->mobile,
                'address' => $request->street_address,
                'state' => $request->state,
                'suburb' => $request->suburb,
                'dob' => $request->dob,
                'work_experience' => $request->work_experience,
                'trade_qualified' => $request->trade_qualified,
                'trade_qualified_year' => $request->trade_qualified_year,
                'trade_licensed' => $request->trade_licensed,
                'trade_licensed_year' => $request->trade_licensed_year,
                'year_commenced' => $request->year_commenced,
                'citizenship_status' => $request->citizenship_status,
                'allergies' => $request->allergies,
                'previous_injuries' => $request->previous_injuries,
                'medical_condition' => $request->medical_condition,
                'relation' => $request->relation,
                'emergency_name' => $request->emergencyName,
                'mobile' => $request->mobile,
            ];

            // Handle profile image upload (only for new employees)
            if ($request->hasFile('profile_image')) {
                $profileImage = $request->file('profile_image');
                $imageName = time() . '_profile_' . $subcontractorId . '.' . $profileImage->getClientOriginalExtension();
                $uploadPath = public_path('upload/images');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }
                $profileImage->move($uploadPath, $imageName);
                $employeeData['profile_image'] = 'upload/images/' . $imageName;
            }

            // Handle signature upload (only for new employees)
            if ($request->hasFile('signature')) {
                $signature = $request->file('signature');
                $imageName = time() . '_signature_' . $subcontractorId . '.' . $signature->getClientOriginalExtension();
                $uploadPath = public_path('upload/signature');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }
                $signature->move($uploadPath, $imageName);
                $employeeData['signature'] = 'upload/signature/' . $imageName;
            }

            // Handle visa attachment upload (only for new employees)
            if ($request->hasFile('visa_attachement')) {
                $visaFile = $request->file('visa_attachement');
                $fileName = time() . '_visa_' . $subcontractorId . '.' . $visaFile->getClientOriginalExtension();
                $uploadPath = public_path('upload/visa_attachments');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }
                $visaFile->move($uploadPath, $fileName);
                $employeeData['visa_attachement'] = 'upload/visa_attachments/' . $fileName;
            }

            // Handle medical attachment files (multiple files) (only for new employees)
            if ($request->hasFile('medical_attachement_files')) {
                $medicalFiles = [];
                $uploadPath = public_path('upload/medical_attachments');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }

                foreach ($request->file('medical_attachement_files') as $medicalFile) {
                    $fileName = time() . '_medical_' . $subcontractorId . '_' . uniqid() . '.' . $medicalFile->getClientOriginalExtension();
                    $medicalFile->move($uploadPath, $fileName);
                    $medicalFiles[] = 'upload/medical_attachments/' . $fileName;
                }

                $employeeData['medical_attachement_files'] = implode(',', $medicalFiles);
            }

            // Create employee in employees_subcontractors table
            $employee = EmployeeSubcontractor::create($employeeData);

            if (!$employee) {
                DB::rollBack();
                return $this->error('Failed to create employee record.', 500);
            }
        }

        // Create meta record in employees_subcontractors_metas table
        $metaData = [
            'emp_id' => $employee->id,
            'subcontractor_id' => $subcontractorId,
            'company_id' => $request->company_id,
            'project_ids' => $request->project_ids ?? [],
            'active' => $request->active ?? true,
        ];

        $employeeMeta = EmployeeSubcontractorMeta::create($metaData);

        if (!$employeeMeta) {
            DB::rollBack();
            return $this->error('Failed to create employee meta record.', 500);
        }

        DB::commit();

        // Load relationships for response
        $employee->load('associations');

        // Add file URLs to response
        if ($employee->profile_image) {
            $employee->profile_image_url = url($employee->profile_image);
        }
        if ($employee->signature) {
            $employee->signature_url = url($employee->signature);
        }
        if ($employee->visa_attachement) {
            $employee->visa_attachement_url = url($employee->visa_attachement);
        }
        if ($employee->medical_attachement_files) {
            $employee->medical_attachement_files_urls = collect(explode(',', $employee->medical_attachement_files))
                ->map(function ($file) {
                    return url(trim($file));
                })->toArray();
        }

        // Send welcome email with credentials for new employees
        if ($isNewEmployee && !empty($employee->email)) {
            try {
                // Get customer_id from first company association or use null for default
                $customerId = null;
                if ($request->filled('company_id')) {
                    $subcontractorCompany = SubcontractorCompany::where('id', $request->company_id)
                        ->where('user_id', $subcontractorId)
                        ->first();
                    if ($subcontractorCompany) {
                        $customerId = $subcontractorCompany->customer_id;
                    }
                } else {
                    // Get from first company association
                    $firstCompany = SubcontractorCompany::where('user_id', $subcontractorId)
                        ->where('del', '0')
                        ->first();
                    if ($firstCompany) {
                        $customerId = $firstCompany->customer_id;
                    }
                }

                // Prepare employee full name
                $employeeName = trim(($employee->first_name ?? '') . ' ' . ($employee->middle_name ?? '') . ' ' . ($employee->last_name ?? ''));
                $employeeName = trim($employeeName) ?: $employee->email;
                $brandTitle = getBrandTitle($customerId ?? null, $workspaceId ?? null);
                // Send welcome email with credentials
                $emailParams = [
                    'to' => $employee->email,
                    'subject' => 'Welcome to ' . $brandTitle ?? env("APP_NAME") . ' - Your Account Credentials',
                    'msg' => view('Emails.subcontractor-employee-welcome', [
                        'subject' => 'Welcome - Your Account Credentials',
                        'name' => $employeeName,
                        'email' => $employee->email,
                        'password' => $plainPassword,
                        'customer_id' => $customerId,
                    ])->render()
                ];

                $this->SendInstantEmail($emailParams);
                Log::info('Welcome email sent to subcontractor employee: ' . $employee->email);
            } catch (\Exception $e) {
                // Log error but don't fail the employee creation
                Log::error('Error sending welcome email to subcontractor employee: ' . $e->getMessage(), [
                    'employee_id' => $employee->id,
                    'email' => $employee->email
                ]);
            }
        }

        $message = $isNewEmployee
            ? 'Employee created successfully.'
            : 'Employee associated with your subcontractor account successfully.';

        return $this->success([
            'employee' => $employee,
            'meta' => $employeeMeta,
            'is_new_employee' => $isNewEmployee,
        ], $message);
    }

    public function getEmployee(Request $request, $id)
    {

        $subcontractorId = Auth::user()->id;

        // Verify that the employee belongs to this subcontractor
        $employeeMeta = EmployeeSubcontractorMeta::where('emp_id', $id)
            ->where('subcontractor_id', $subcontractorId)
            ->first();

        if (!$employeeMeta) {
            return $this->error('Employee not found or you do not have permission to view it.', 404);
        }

        // Get employee details
        $employee = EmployeeSubcontractor::with(['associations' => function ($q) use ($subcontractorId) {
            $q->where('subcontractor_id', $subcontractorId);
        }])->find($id);

        if (!$employee) {
            return $this->error('Employee not found.', 404);
        }

        // Add file URLs to response
        if ($employee->profile_image) {
            $employee->profile_image_url = url($employee->profile_image);
        }
        if ($employee->signature) {
            $employee->signature_url = url($employee->signature);
        }
        if ($employee->visa_attachement) {
            $employee->visa_attachement_url = url($employee->visa_attachement);
        }
        if ($employee->medical_attachement_files) {
            $employee->medical_attachement_files_urls = collect(explode(',', $employee->medical_attachement_files))
                ->map(function ($file) {
                    return url(trim($file));
                })->toArray();
        }

        // Add meta information
        $employee->meta = $employeeMeta;

        return $this->success($employee, 'Employee retrieved successfully.');
    }
        /**
     * Update an employee
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateEmployee(Request $request, $id)
    {

        $subcontractorId = Auth::user()->id;

        // Verify that the employee belongs to this subcontractor
        $employeeMeta = EmployeeSubcontractorMeta::where('emp_id', $id)
            ->where('subcontractor_id', $subcontractorId)
            ->first();

        if (!$employeeMeta) {
            return $this->error('Employee not found or you do not have permission to update it.', 404);
        }

        // Get existing employee for email uniqueness check
        $existingEmployee = EmployeeSubcontractor::find($id);
        if (!$existingEmployee) {
            return $this->error('Employee not found.', 404);
        }

        // Build email validation rules with uniqueness checks
        $emailRules = ['required', 'email'];
        $emailRules[] = Rule::unique('employees_subcontractors', 'email')->ignore($id, 'id');
        $emailRules[] = Rule::unique('emp_company_details', 'employee_email')->ignore($id, 'id');
        $emailRules[] = Rule::unique('users', 'email')->ignore($id, 'id');

        // Validation rules (same as create, but email can be same for this employee)
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => $emailRules,
            'mobile' => 'required|numeric',
            'street_address' => 'required|string|max:500',
            'suburb' => 'required|regex:/^[a-zA-Z\s]*$/',
            'state' => 'required|regex:/^[a-zA-Z\s]*$/',
            'dob' => 'required|date|before:' . Carbon::now()->subYears(15)->format('Y-m-d'),
            'trade_qualified' => 'required|string|max:255',
            'trade_qualified_year' => 'required_if:trade_qualified,=,1',
            'trade_licensed' => 'required|string|max:255',
            'trade_licensed_year' => 'required_if:trade_licensed,=,1',
            'work_experience' => 'required|string|max:255',
            'year_commenced' => 'required|string|max:10',
            'legally_australia' => 'required|string|max:255',
            'citizenship_status' => 'required|integer',
            'allergies' => 'nullable|string',
            'previous_injuries' => 'nullable|string',
            'medical_condition' => 'nullable|string',
            'profile_image' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'visa_attachement' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'signature' => 'nullable|mimes:jpeg,png,jpg,gif|max:1024',
            'medical_attachement_files' => 'nullable|array',
            'medical_attachement_files.*' => 'nullable|mimes:jpeg,png,jpg,gif,pdf,doc,docx,xls,xlsx,ppt,pptx|max:10240',
            // Emergency contact fields
            'relation' => 'nullable|string|max:255',
            'emergencyName' => 'nullable|string|max:255',
            'company_id' => 'nullable|integer',
            'project_ids' => 'nullable|array',
            'project_ids.*' => 'integer',
            'active' => 'nullable|boolean',
        ], [
            'email.required' => 'Email is required',
            'email.email' => 'Email must be a valid email address',
            'first_name.required' => 'First name is required',
            'last_name.required' => 'Last name is required',
            'mobile.required' => 'Mobile number is required',
            'mobile.numeric' => 'Mobile number must be numeric',
            'street_address.required' => 'Street address is required',
            'suburb.required' => 'Suburb is required',
            'suburb.regex' => 'The suburb field must contain only alphabetic characters.',
            'state.required' => 'State is required',
            'state.regex' => 'The state field must contain only alphabetic characters.',
            'dob.required' => 'Date of birth is required',
            'dob.before' => 'You must be at least 15 years old to register.',
            'trade_qualified.required' => 'The trade qualified field is required',
            'trade_qualified_year.required_if' => 'The trade qualified year field is required',
            'trade_licensed.required' => 'The trade licensed field is required',
            'trade_licensed_year.required_if' => 'The trade licensed year field is required',
            'work_experience.required' => 'The work experience field is required',
            'year_commenced.required' => 'The year commenced field is required',
            'citizenship_status.required' => 'Citizenship status is required',
            'profile_image.mimes' => 'The image must be a file of type: jpeg, png, jpg, gif.',
            'profile_image.max' => 'The image may not be greater than 1MB.',
            'visa_attachement.mimes' => 'The visa attachment must be a file of type: jpeg, png, jpg, gif.',
            'visa_attachement.max' => 'The visa attachment may not be greater than 1MB.',
            'signature.mimes' => 'The signature must be a file of type: jpeg, png, jpg, gif.',
            'signature.max' => 'The signature may not be greater than 1MB.',
            'medical_attachement_files.*.mimes' => 'Medical attachment files must be of type: jpeg, png, jpg, gif.',
            'medical_attachement_files.*.max' => 'Each medical attachment file may not be greater than 1MB.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        DB::beginTransaction();
        try {
            // Prepare employee data
            $employeeData = [
                'first_name' => $request->first_name,
                'middle_name' => $request->middle_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'phone' => $request->mobile,
                'address' => $request->street_address,
                'state' => $request->state,
                'suburb' => $request->suburb,
                'dob' => $request->dob,
                'work_experience' => $request->work_experience,
                'trade_qualified' => $request->trade_qualified,
                'trade_qualified_year' => $request->trade_qualified_year,
                'trade_licensed' => $request->trade_licensed,
                'trade_licensed_year' => $request->trade_licensed_year,
                'year_commenced' => $request->year_commenced,
                'citizenship_status' => $request->citizenship_status,
                'allergies' => $request->allergies,
                'previous_injuries' => $request->previous_injuries,
                'medical_condition' => $request->medical_condition,
                'relation' => $request->relation,
                'emergency_name' => $request->emergencyName,
                'mobile' => $request->mobile,
            ];

            // Handle profile image upload
            if ($request->hasFile('profile_image')) {
                // Delete old profile image if exists
                if ($existingEmployee->profile_image && file_exists(public_path($existingEmployee->profile_image))) {
                    unlink(public_path($existingEmployee->profile_image));
                }

                $profileImage = $request->file('profile_image');
                $imageName = time() . '_profile_' . $subcontractorId . '.' . $profileImage->getClientOriginalExtension();
                $uploadPath = public_path('upload/images');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }
                $profileImage->move($uploadPath, $imageName);
                $employeeData['profile_image'] = 'upload/images/' . $imageName;
            }

            // Handle signature upload
            if ($request->hasFile('signature')) {
                // Delete old signature if exists
                if ($existingEmployee->signature && file_exists(public_path($existingEmployee->signature))) {
                    unlink(public_path($existingEmployee->signature));
                }

                $signature = $request->file('signature');
                $imageName = time() . '_signature_' . $subcontractorId . '.' . $signature->getClientOriginalExtension();
                $uploadPath = public_path('upload/signature');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }
                $signature->move($uploadPath, $imageName);
                $employeeData['signature'] = 'upload/signature/' . $imageName;
            }

            // Handle visa attachment upload
            if ($request->hasFile('visa_attachement')) {
                // Delete old visa attachment if exists
                if ($existingEmployee->visa_attachement && file_exists(public_path($existingEmployee->visa_attachement))) {
                    unlink(public_path($existingEmployee->visa_attachement));
                }

                $visaFile = $request->file('visa_attachement');
                $fileName = time() . '_visa_' . $subcontractorId . '.' . $visaFile->getClientOriginalExtension();
                $uploadPath = public_path('upload/visa_attachments');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }
                $visaFile->move($uploadPath, $fileName);
                $employeeData['visa_attachement'] = 'upload/visa_attachments/' . $fileName;
            }

            // Handle medical attachment files (multiple files)
            if ($request->hasFile('medical_attachement_files')) {
                // Delete old medical files if exists
                if ($existingEmployee->medical_attachement_files) {
                    $oldFiles = explode(',', $existingEmployee->medical_attachement_files);
                    foreach ($oldFiles as $oldFile) {
                        $filePath = public_path(trim($oldFile));
                        if (file_exists($filePath)) {
                            unlink($filePath);
                        }
                    }
                }

                $medicalFiles = [];
                $uploadPath = public_path('upload/medical_attachments');
                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }

                foreach ($request->file('medical_attachement_files') as $medicalFile) {
                    $fileName = time() . '_medical_' . $subcontractorId . '_' . uniqid() . '.' . $medicalFile->getClientOriginalExtension();
                    $medicalFile->move($uploadPath, $fileName);
                    $medicalFiles[] = 'upload/medical_attachments/' . $fileName;
                }

                $employeeData['medical_attachement_files'] = implode(',', $medicalFiles);
            }

            // Update employee
            $existingEmployee->update($employeeData);

            // Update meta record
            $metaData = [];
            if ($request->has('company_id')) {
                $metaData['company_id'] = $request->company_id;
            }
            if ($request->has('project_ids')) {
                $metaData['project_ids'] = $request->project_ids ?? [];
            }
            if ($request->has('active')) {
                $metaData['active'] = $request->active ?? true;
            }

            if (!empty($metaData)) {
                $employeeMeta->update($metaData);
            }

            DB::commit();

            // Reload employee with relationships
            $employee = EmployeeSubcontractor::with(['associations' => function ($q) use ($subcontractorId) {
                $q->where('subcontractor_id', $subcontractorId);
            }])->find($id);

            // Add file URLs to response
            if ($employee->profile_image) {
                $employee->profile_image_url = url($employee->profile_image);
            }
            if ($employee->signature) {
                $employee->signature_url = url($employee->signature);
            }
            if ($employee->visa_attachement) {
                $employee->visa_attachement_url = url($employee->visa_attachement);
            }
            if ($employee->medical_attachement_files) {
                $employee->medical_attachement_files_urls = collect(explode(',', $employee->medical_attachement_files))
                    ->map(function ($file) {
                        return url(trim($file));
                    })->toArray();
            }

            $employee->meta = $employeeMeta->fresh();

            return $this->success($employee, 'Employee updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating employee: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request' => $request->all(),
                'employee_id' => $id
            ]);
            return $this->error('An error occurred while updating the employee. Please try again.', 500);
        }
    }

    public function deleteEmployee(Request $request, $id)
    {

        $subcontractorId = Auth::user()->id;
        // Verify that the employee belongs to this subcontractor
        $employeeMeta = EmployeeSubcontractorMeta::where('emp_id', $id)
            ->where('subcontractor_id', $subcontractorId)
            ->first();
        if (!$employeeMeta) {
            return $this->error('Employee not found or you do not have permission to delete it.', 404);
        }

        DB::beginTransaction();
        try {
            // Delete the association (meta record)
            // Note: We don't delete the employee record itself as it might be associated with other subcontractors
            $employeeMeta->delete();

            DB::commit();

            return $this->message('Employee  removed successfully.', 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting employee : ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'employee_id' => $id,
                'subcontractor_id' => $subcontractorId
            ]);
            return $this->error('An error occurred while removing the employee . Please try again.', 500);
        }
    }

    public function toggleEmployeeStatus(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'emp_id' => 'required|integer|exists:employees_subcontractors_metas,emp_id',
        ], [
            'emp_id.required' => 'Employee ID is required.',
            'emp_id.integer' => 'Employee ID must be an integer.',
            'emp_id.exists' => 'Employee not found.',
        ]);

        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return $this->error($error, 422);
        }

        $subcontractorId = auth()->id();
        $empId = $request->emp_id;

        // Verify that the association belongs to the authenticated subcontractor
        $empSubcontractor = EmployeeSubcontractorMeta::where('emp_id', $empId)
            ->where('subcontractor_id', $subcontractorId)
            ->first();

        if (!$empSubcontractor) {
            return $this->error('Employee not found or you do not have permission to modify it.', 404);
        }

        // Toggle the active status
        $empSubcontractor->active = !$empSubcontractor->active;
        $empSubcontractor->save();

        $status = $empSubcontractor->active ? 'activated' : 'deactivated';

        return $this->success([
            'emp_id' => $empSubcontractor->emp_id,
            'active' => $empSubcontractor->active,
            'status' => $status
        ], "Employee has been {$status} successfully.");
    }

    public function getAssignedProjects(Request $request)
    {
        $subcontractor = Auth::user();

        // Get all company associations for this subcontractor
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('del', '0')
            ->get();

        if ($subcontractorCompanies->isEmpty()) {
            return $this->withCount(collect([]), 'No company associations found for this subcontractor.');
        }

        // Collect all project IDs from all company associations
        $allProjectIds = [];
        $projectCompanyMap = []; // Map project_id => [customer_id, workspace_id, required_docs_status] for reference

        foreach ($subcontractorCompanies as $subcontractorCompany) {
            $projectIds = $subcontractorCompany->project_ids ?? [];
            foreach ($projectIds as $projectId) {
                $allProjectIds[] = $projectId;
                // Only set if not already set (use first company association for each project)
                if (!isset($projectCompanyMap[$projectId])) {
                    // Calculate induction status for this subcontractor-company relationship
                    $inductionStatusData = $this->calculateInductionStatus(
                        $subcontractor->id,
                        $subcontractorCompany->customer_id,
                        $subcontractorCompany->workspace_id
                    );

                    $projectCompanyMap[$projectId] = [
                        'customer_id' => $subcontractorCompany->customer_id,
                        'workspace_id' => $subcontractorCompany->workspace_id,
                        'required_docs_status' => $subcontractorCompany->required_docs_status,
                        'required_docs_status_label' => $subcontractorCompany->required_docs_status_label,
                        'induction_status' => $inductionStatusData['induction_status'],
                        'induction_status_label' => $inductionStatusData['induction_status_label'],
                    ];
                }
            }
        }

        if (empty($allProjectIds)) {
            return $this->withCount(collect([]), 'No projects assigned to this subcontractor.');
        }

        // Remove duplicates
        $allProjectIds = array_unique($allProjectIds);

        // Query projects that are assigned to this subcontractor across all companies
        $query = Project::whereIn('id', $allProjectIds)
            ->where('is_deleted', 0)
            ->where('active', 1)
            ->latest('id');

        // Filter by active status if provided
        if ($request->filled('active')) {
            $query->where('active', $request->active == 1 || $request->active === '1' ? 1 : 0);
        }

        // Apply search filter across multiple fields
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', '%' . $searchTerm . '%')
                    ->orWhere('description', 'like', '%' . $searchTerm . '%')
                    ->orWhere('address', 'like', '%' . $searchTerm . '%')
                    ->orWhere('state', 'like', '%' . $searchTerm . '%')
                    ->orWhereHas('company', function ($companyQuery) use ($searchTerm) {
                        $companyQuery->where('name', 'like', '%' . $searchTerm . '%')
                            ->orWhere('company_name', 'like', '%' . $searchTerm . '%');
                    })
                    ->orWhereHas('site', function ($siteQuery) use ($searchTerm) {
                        $siteQuery->whereHas('siteDetails', function ($siteDetailsQuery) use ($searchTerm) {
                            $siteDetailsQuery->where('title', 'like', '%' . $searchTerm . '%')
                                ->orWhere('name', 'like', '%' . $searchTerm . '%')
                                ->orWhere('street_address', 'like', '%' . $searchTerm . '%');
                        });
                    });
            });
        }

        // Filter by company/workspace if provided
        if ($request->filled('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }
        if ($request->filled('workspace_id')) {
            $query->where('workspace_id', $request->workspace_id);
        }

        // Include related data with nested site details (similar to ProjectController)
        $query->with([
            'company',
            'ProjectMeta',
            'site' => function ($query) {
                // Filter ProjectSite records to only include those with active sites
                $query->whereHas('siteDetails', function ($siteQuery) {
                    $siteQuery->where('active', '1')->where('del', '0');
                });
            },
            'site.siteDetails' => function ($query) {
                // Filter Sites to only load active ones
                $query->where('active', '1')->where('del', '0');
            }
        ]);

        $projects = $query->get();

        // Collect all unique customer_id and workspace_id pairs to fetch company_name from settings in bulk
        $customerWorkspacePairs = [];
        foreach ($projects as $project) {
            $customerId = null;
            $workspaceId = null;
            
            if (isset($projectCompanyMap[$project->id])) {
                $customerId = $projectCompanyMap[$project->id]['customer_id'];
                $workspaceId = $projectCompanyMap[$project->id]['workspace_id'];
            } else {
                $customerId = $project->customer_id ?? null;
                $workspaceId = $project->workspace_id ?? null;
            }
            
            if ($customerId && $workspaceId) {
                $key = $customerId . '_' . $workspaceId;
                if (!isset($customerWorkspacePairs[$key])) {
                    $customerWorkspacePairs[$key] = [
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId
                    ];
                }
            }
        }

        // Fetch company_name from adminsettings for all customer/workspace pairs in bulk
        $companyNamesMap = [];
        if (!empty($customerWorkspacePairs)) {
            foreach ($customerWorkspacePairs as $key => $pair) {
                $companyName = DB::table('adminsettings')
                    ->where('customer_id', $pair['customer_id'])
                    ->where('workspace', $pair['workspace_id'])
                    ->where('key', 'company_company_name')
                    ->value('value');
                
                // If not found with workspace, try without workspace filter
                if (empty($companyName)) {
                    $companyName = DB::table('adminsettings')
                        ->where('customer_id', $pair['customer_id'])
                        ->where('key', 'company_company_name')
                        ->value('value');
                }
                
                // Fallback to User model company_name or name if not found in settings
                if (empty($companyName)) {
                    $customer = User::find($pair['customer_id']);
                    if ($customer) {
                        $companyName = $customer->company_name ?? $customer->name ?? null;
                    }
                }
                
                $companyNamesMap[$key] = $companyName;
            }
        }

        // Process sites to add site_name and site_details (similar to ProjectController)
        // Also add required_docs_status from the project company map
        $projects = $projects->map(function ($project) use ($projectCompanyMap, $companyNamesMap, $subcontractor) {
            // Filter out any inactive sites that might have been loaded
            if ($project->site) {
                $project->site = $project->site->filter(function ($siteRelation) {
                    // Only keep sites where siteDetails exists and is active
                    return $siteRelation->siteDetails &&
                        $siteRelation->siteDetails->active == '1' &&
                        $siteRelation->siteDetails->del == '0';
                })->values(); // Re-index the collection

                foreach ($project->site as $siteRelation) {
                    // Use the already loaded siteDetails or fetch if not loaded
                    $siteDetails = $siteRelation->siteDetails;
                    if (!$siteDetails) {
                        // Fallback: fetch site details if not loaded (shouldn't happen with eager loading)
                        $siteDetails = \App\Models\Sites::where('id', $siteRelation->site_id)
                            ->where('active', '1')
                            ->where('del', '0')
                            ->first();
                    }
                    // Add site_name property
                    $siteRelation->site_name = $siteDetails ? ($siteDetails->name ?? $siteDetails->title ?? 'Unknown Site') : 'Unknown Site';
                    $siteRelation->site_details = $siteDetails;
                }
            }

            // Add required_docs_status, customer_id, and workspace_id from the project company map
            $customerId = null;
            $workspaceId = null;
            if (isset($projectCompanyMap[$project->id])) {
                $project->required_docs_status = $projectCompanyMap[$project->id]['required_docs_status'];
                $project->required_docs_status_label = $projectCompanyMap[$project->id]['required_docs_status_label'];
                $project->induction_status = $projectCompanyMap[$project->id]['induction_status'];
                $project->induction_status_label = $projectCompanyMap[$project->id]['induction_status_label'];
                $project->project_customer_id = $projectCompanyMap[$project->id]['customer_id'];
                $project->project_workspace_id = $projectCompanyMap[$project->id]['workspace_id'];
                $customerId = $projectCompanyMap[$project->id]['customer_id'];
                $workspaceId = $projectCompanyMap[$project->id]['workspace_id'];
            } else {
                $project->required_docs_status = 0;
                $project->required_docs_status_label = 'not_uploaded';
                $project->induction_status = 0;
                $project->induction_status_label = 'not_signed';
                // Fallback to project's own customer_id and workspace_id if not in map
                $project->project_customer_id = $project->customer_id ?? null;
                $project->project_workspace_id = $project->workspace_id ?? null;
                $customerId = $project->customer_id ?? null;
                $workspaceId = $project->workspace_id ?? null;
            }

            // Calculate overall status for required documents
            $project->required_docs_overall_status = $this->calculateRequiredDocumentsOverallStatus(
                $subcontractor->id,
                $customerId,
                $workspaceId
            );

            // Calculate overall status for induction
            $project->induction_overall_status = $this->calculateInductionOverallStatus(
                $subcontractor->id,
                $customerId,
                $workspaceId
            );

            // Add company_name from settings
            $companyName = null;
            if ($customerId && $workspaceId) {
                $key = $customerId . '_' . $workspaceId;
                $companyName = $companyNamesMap[$key] ?? null;
            }
            $project->company_name = $companyName;

            return $project;
        });

        // Filter by approved status if provided (only return projects with approved documents AND induction)
        if ($request->filled('approved') && ($request->approved == 1 || $request->approved === '1')) {
            $projects = $projects->filter(function ($project) use ($subcontractor) {
                // Check both required documents AND induction status if project has customer_id and workspace_id
                if (isset($project->project_customer_id) && isset($project->project_workspace_id)) {
                    // Use the comprehensive method that checks both required docs and induction
                    return $this->areAllDocumentsAndInductionComplete(
                        $subcontractor->id,
                        $project->project_customer_id,
                        $project->project_workspace_id
                    );
                }

                // Fallback: check required documents status only if customer/workspace not available
                return isset($project->required_docs_status) &&
                    $project->required_docs_status == SubcontractorCompany::STATUS_COMPLETE;
            })->values(); // Re-index the collection after filtering
        }

        // Format projects by company if format parameter is filled
        if ($request->filled('format')) {
            // Group projects by project_customer_id
            $groupedProjects = $projects->groupBy('project_customer_id');

            // Get all unique customer IDs
            $customerIds = $groupedProjects->keys()->filter()->toArray();

            // Fetch customer details (User records) for all companies
            $customers = collect();
            if (!empty($customerIds)) {
                $customers = User::whereIn('id', $customerIds)
                    ->select('id', 'name', 'email', 'company_name')
                    ->get()
                    ->keyBy('id');
            }

            // Format response with companies as headings
            $formattedData = [];
            foreach ($groupedProjects as $customerId => $companyProjects) {
                $customer = $customers->get($customerId);

                $formattedData[] = [
                    'company_id' => $customerId,
                    'company_name' => $customer ? ($customer->name ?? $customer->company_name ?? 'Unknown Company') : 'Unknown Company',
                    'company_email' => $customer ? $customer->email : null,
                    'projects' => $companyProjects->values()->toArray(),
                    'projects_count' => $companyProjects->count(),
                ];
            }

            // Sort by company name for better organization
            usort($formattedData, function ($a, $b) {
                return strcmp($a['company_name'], $b['company_name']);
            });

            return $this->withCount(collect($formattedData), 'Projects retrieved successfully');
        }

        return $this->withCount($projects, 'Projects retrieved successfully');
    }

    public function getAssignedProjectsById($id)
    {
        $subcontractor = Auth::user();
        if (!$subcontractor || $subcontractor->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized. Only subcontractors can access this endpoint.', 403);
        }
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('del', '0')
            ->get();
        if ($subcontractorCompanies->isEmpty()) {
            return $this->error('No company associations found for this subcontractor.', 404);
        }
        $allProjectIds = [];
        $projectCompanyMap = [];
        foreach ($subcontractorCompanies as $subcontractorCompany) {
            $projectIds = $subcontractorCompany->project_ids ?? [];
            foreach ($projectIds as $projectId) {
                $allProjectIds[] = $projectId;
                if (!isset($projectCompanyMap[$projectId])) {
                    $projectCompanyMap[$projectId] = [
                        'customer_id' => $subcontractorCompany->customer_id,
                        'workspace_id' => $subcontractorCompany->workspace_id,
                        'required_docs_status' => $subcontractorCompany->required_docs_status,
                        'required_docs_status_label' => $subcontractorCompany->required_docs_status_label,
                    ];
                }
            }
        }
        if (!in_array($id, $allProjectIds)) {
            return $this->error('Project not found or not assigned to this subcontractor.', 404);
        }
        $project = Project::where('id', $id)
            ->where('is_deleted', 0)
            ->where('active', 1)
            ->with([
                'company',
                'ProjectMeta',
                'site' => function ($query) {
                    $query->whereHas('siteDetails', function ($siteQuery) {
                        $siteQuery->where('active', '1')->where('del', '0');
                    });
                },
                'site.siteDetails' => function ($query) {
                    $query->where('active', '1')->where('del', '0');
                }
            ])
            ->first();

        if (!$project) {
            return $this->error('Project not found.', 404);
        }
        if ($project->site) {
            $project->site = $project->site->filter(function ($siteRelation) {
                return $siteRelation->siteDetails &&
                    $siteRelation->siteDetails->active == '1' &&
                    $siteRelation->siteDetails->del == '0';
            })->values(); // Re-index the collection
            foreach ($project->site as $siteRelation) {
                // Use the already loaded siteDetails or fetch if not loaded
                $siteDetails = $siteRelation->siteDetails;
                if (!$siteDetails) {
                    // Fallback: fetch site details if not loaded (shouldn't happen with eager loading)
                    $siteDetails = \App\Models\Sites::where('id', $siteRelation->site_id)
                        ->where('active', '1')
                        ->where('del', '0')
                        ->first();
                }
                // Add site_name property
                $siteRelation->site_name = $siteDetails ? ($siteDetails->name ?? $siteDetails->title ?? 'Unknown Site') : 'Unknown Site';
                $siteRelation->site_details = $siteDetails;
                
                // Load site documents for this site that are assigned to this subcontractor only
                if ($siteDetails && $siteDetails->id) {
                    $subcontractorId = $subcontractor->id;
                    // Get all documents for this site and filter to only those assigned to this subcontractor
                    $allSiteDocuments = SiteDocument::where('site_id', $siteDetails->id)
                        ->orderBy('created_at', 'desc')
                        ->get();
                    
                    // Filter documents to only include those assigned to this subcontractor
                    $siteDocuments = $allSiteDocuments->filter(function ($doc) use ($subcontractorId) {
                        // Check if document has subcontractors assigned
                        if (!$doc->subcontractors || !is_array($doc->subcontractors) || empty($doc->subcontractors)) {
                            return false; // No subcontractors assigned, skip this document
                        }
                        // Check if this subcontractor's ID is in the array
                        return in_array($subcontractorId, $doc->subcontractors);
                    })->values(); // Re-index the collection
                    
                    // Load additional details for each document
                    foreach ($siteDocuments as $doc) {
                        // Get uploaded_by user information
                        if ($doc->uploaded_by_type == 'customer') {
                            $user = User::find($doc->uploaded_by);
                            if ($user) {
                                $doc->uploaded_by_name = $user->name;
                            }
                        } else {
                            $emp = EmpPersonalDetails::where('emp_id', $doc->uploaded_by)->first();
                            if ($emp) {
                                $doc->uploaded_by_name = trim(($emp->first_name ?? '') . " " . ($emp->middle_name ?? '') . " " . ($emp->last_name ?? ''));
                            }
                        }
                        
                        // Load roles from role_ids JSON array
                        if ($doc->role_ids && is_array($doc->role_ids)) {
                            $doc->roles = Role::whereIn('id', $doc->role_ids)
                                ->where('del', '0')
                                ->select('id', 'title', 'code')
                                ->get();
                        } else {
                            $doc->roles = collect([]);
                        }
                        
                        // Load document type relationship
                        if ($doc->document_type) {
                            $doc->documentTypeDetail = DocumentType::find($doc->document_type);
                        }
                        
                        // Load subcontractor relationships
                        if ($doc->subcontractors && is_array($doc->subcontractors)) {
                            $subcontractorUsers = User::whereIn('id', $doc->subcontractors)
                                ->where('user_type', config('constants.user_types.subcontractor'))
                                ->where('del', '0')
                                ->select('id', 'name', 'company_name', 'email', 'mobile_number')
                                ->get();
                            
                            $doc->subcontractorsDetail = $subcontractorUsers->map(function ($user) {
                                return [
                                    'id' => $user->id,
                                    'title' => $user->company_name ?? $user->name,
                                    'name' => $user->name,
                                    'company_name' => $user->company_name,
                                    'email' => $user->email,
                                    'mobile_number' => $user->mobile_number,
                                ];
                            });
                        } else {
                            $doc->subcontractorsDetail = collect([]);
                        }
                        
                        // Load signature for this subcontractor
                        $signature = SiteDocumentSignature::where('site_document_id', $doc->id)
                            ->where('employee_id', $subcontractorId)
                            ->where('site_id', $siteDetails->id)
                            ->where('user_type', 'subcontractor')
                            ->first();
                        
                        if ($signature) {
                            $doc->signature = [
                                'id' => $signature->id,
                                'signature' => $signature->signature,
                                'signature_url' => strpos($signature->signature, '/') !== false 
                                    ? asset('/' . $signature->signature) 
                                    : null,
                                'signed_at' => $signature->signed_at,
                                'user_type' => $signature->user_type,
                            ];
                            $doc->is_signed = true;
                        } else {
                            $doc->signature = null;
                            $doc->is_signed = false;
                        }
                    }
                    
                    // Add documents to site relation
                    $siteRelation->site_documents = $siteDocuments;
                } else {
                    $siteRelation->site_documents = collect([]);
                }
            }
        }

        // Add required_docs_status, customer_id, and workspace_id from the project company map
        if (isset($projectCompanyMap[$project->id])) {
            $project->required_docs_status = $projectCompanyMap[$project->id]['required_docs_status'];
            $project->required_docs_status_label = $projectCompanyMap[$project->id]['required_docs_status_label'];
            $project->project_customer_id = $projectCompanyMap[$project->id]['customer_id'];
            $project->project_workspace_id = $projectCompanyMap[$project->id]['workspace_id'];
        } else {
            $project->required_docs_status = 0;
            $project->required_docs_status_label = 'not_uploaded';
            // Fallback to project's own customer_id and workspace_id if not in map
            $project->project_customer_id = $project->customer_id ?? null;
            $project->project_workspace_id = $project->workspace_id ?? null;
        }

        return $this->success($project, 'Project retrieved successfully');
    }

    public function getEmployeeManagementData($id)
    {
        $subcontractor = Auth::user();


        // Get tab filter (default to 'projects')
        $tabFilter = request()->get('tab', 'projects');
        $validTabs = ['projects', 'defects', 'documents', 'induction'];
        if (!in_array($tabFilter, $validTabs)) {
            return $this->error('Invalid tab filter. Valid options: ' . implode(', ', $validTabs), 400);
        }

        // Verify that the employee belongs to this subcontractor
        $employeeMeta = EmployeeSubcontractorMeta::where('emp_id', $id)
            ->where('subcontractor_id', $subcontractor->id)
            // ->where('active', 1)
            ->first();

        if (!$employeeMeta) {
            return $this->error('Employee not found or you do not have permission to view it.', 404);
        }

        // Get employee details
        $employee = EmployeeSubcontractor::find($id);
        if (!$employee) {
            return $this->error('Employee not found.', 404);
        }

        // Get all company associations for this subcontractor to get customer/workspace contexts
        $subcontractorCompanies = SubcontractorEmployeeInvitation::where('subcontractor_id', $subcontractor->id)
            ->where('employee_id', $id)
            ->where('invitation_status', 'accepted')
            // ->where('status', 1)
            ->get();

        // if ($subcontractorCompanies->isEmpty()) {
        //     return $this->error('No company associations found for this subcontractor.', 404);
        // }

        // Prepare base employee data
        $employeeData = [
            'id' => $employee->id,
            'first_name' => $employee->first_name,
            'middle_name' => $employee->middle_name ?? null,
            'last_name' => $employee->last_name,
            'email' => $employee->email,
            'phone' => $employee->phone ?? null,
            'profile_image' => $employee->profile_image ? url($employee->profile_image) : null,
            'active' => $employeeMeta->active ?? true,
        ];

        // Get employee's assigned project IDs from meta
        $employeeProjectIds = $employeeMeta->project_ids ?? [];

        // Calculate summary counts across all company associations
        $summary = [
            'assigned_projects' => 0,
            'defects' => 0,
            'documents_uploaded' => 0,
            'documents_pending' => 0,
            'induction_completed' => false,
        ];

        // Collect all customer/workspace IDs for queries
        $allCustomerWorkspacePairs = [];
        foreach ($subcontractorCompanies as $company) {
            $allCustomerWorkspacePairs[] = [
                'customer_id' => $company->customer_id,
                'workspace_id' => $company->workspace_id,
            ];
        }

        // Calculate summary counts
        if (!empty($employeeProjectIds)) {
            $summary['assigned_projects'] = Project::whereIn('id', $employeeProjectIds)
                ->where('is_deleted', 0)
                ->count();
        }

        // Count defects assigned to this employee across all companies
        foreach ($allCustomerWorkspacePairs as $pair) {
            $defectsCount = Defect::where('assigned_subcontractor_emp_id', $id)
                ->where('assignment_type', 'subcontractor_employee')
                ->where('customer_id', $pair['customer_id'])
                ->where('workspace_id', $pair['workspace_id'])
                ->where('del', '0')
                ->count();
            $summary['defects'] += $defectsCount;
        }

        // Count documents
        foreach ($allCustomerWorkspacePairs as $pair) {
            $uploadedDocs = SubcontractorEmployeeDocument::where('employee_id', $id)
                ->where('customer_id', $pair['customer_id'])
                ->where('workspace_id', $pair['workspace_id'])
                ->where('del', '0')
                ->count();
            $summary['documents_uploaded'] += $uploadedDocs;
        }

        // Check induction completion status
        $inductionComplete = true;
        foreach ($allCustomerWorkspacePairs as $pair) {
            $inductionDocuments = InductionDocument::where('customer_id', $pair['customer_id'])
                ->where('workspace_id', $pair['workspace_id'])
                ->where('del', '0')
                ->where('is_active', true)
                ->get();

            $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
                $roleTypes = $doc->role_types ?? [];
                return in_array('all', $roleTypes) || in_array('external', $roleTypes) || in_array('subcontractor', $roleTypes);
            });

            if ($filteredInductionDocuments->isNotEmpty()) {
                foreach ($filteredInductionDocuments as $doc) {
                    $hasSigned = InductionDocumentSignature::where('induction_document_id', $doc->id)
                        ->where('document_version', $doc->version)
                        ->where('employee_id', $id)
                        ->where('signature_type', 'subcontractor_employee')
                        ->where('customer_id', $pair['customer_id'])
                        ->where('workspace_id', $pair['workspace_id'])
                        ->where('is_valid', true)
                        ->where('del', '0')
                        ->exists();
                    if (!$hasSigned) {
                        $inductionComplete = false;
                        break 2; // Break out of both loops
                    }
                }
            }
        }
        $summary['induction_completed'] = $inductionComplete;

        // Prepare base response data
        $responseData = [
            'employee' => $employeeData,
            'summary' => $summary,
        ];

        // Get tab-specific data
        switch ($tabFilter) {
            case 'projects':
                $responseData = $this->getEmployeeManagementProjectsData($responseData, $employee, $employeeMeta, $subcontractorCompanies);
                break;
            case 'defects':
                $responseData = $this->getEmployeeManagementDefectsData($responseData, $employee, $subcontractorCompanies);
                break;
            case 'documents':
                $responseData = $this->getEmployeeManagementDocumentsData($responseData, $employee, $subcontractorCompanies);
                break;
            case 'induction':
                $responseData = $this->getEmployeeManagementInductionData($responseData, $employee, $subcontractorCompanies);
                break;
        }

        return $this->success($responseData, 'Employee management data retrieved successfully');
    }
    public function assignProjectsToEmployee(Request $request)
    {
        $subcontractor = Auth::user();

        // Validate request
        $validator = Validator::make($request->all(), [
            'emp_id' => 'required|integer|exists:employees_subcontractors,id',
            'project_ids' => 'required|array',
            'project_ids.*' => 'integer',
        ], [
            'emp_id.required' => 'Employee ID is required.',
            'emp_id.integer' => 'Employee ID must be an integer.',
            'emp_id.exists' => 'Employee not found.',
            'project_ids.required' => 'Project IDs are required.',
            'project_ids.array' => 'Project IDs must be an array.',
            'project_ids.*.integer' => 'Each project ID must be an integer.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $empId = $request->emp_id;
        $projectIds = $request->project_ids;
        $subcontractorId = $subcontractor->id;

        // Verify that the employee exists in employees_subcontractors table
        $employee = EmployeeSubcontractor::find($empId);
        if (!$employee) {
            return $this->error('Employee not found.', 404);
        }

        // Check if meta record exists, if not create it
        $empSubcontractor = EmployeeSubcontractorMeta::where('emp_id', $empId)
            ->where('subcontractor_id', $subcontractorId)
            ->first();

        if (!$empSubcontractor) {
            // Create the meta record if it doesn't exist
            // This handles cases where employee exists but wasn't properly associated
            $empSubcontractor = EmployeeSubcontractorMeta::create([
                'emp_id' => $empId,
                'subcontractor_id' => $subcontractorId,
                'company_id' => null, // Optional - employee can work across multiple companies
                'project_ids' => [],
                'active' => true,
            ]);

            if (!$empSubcontractor) {
                return $this->error('Failed to create employee association. Please try again.', 500);
            }
        }

        // Get all company associations for this subcontractor (needed for validation and email)
        $subcontractorCompanies = SubcontractorCompany::where('user_id', $subcontractorId)
            ->where('del', '0')
            ->get();

        if ($subcontractorCompanies->isEmpty()) {
            return $this->error('No company associations found for your subcontractor account.', 404);
        }

        // Validate that all projects exist and are assigned to the subcontractor across any company
        if (!empty($projectIds)) {

            // Collect all project IDs assigned to this subcontractor across all companies
            $allAssignedProjectIds = [];
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                $assignedProjectIds = $subcontractorCompany->project_ids ?? [];
                $allAssignedProjectIds = array_merge($allAssignedProjectIds, $assignedProjectIds);
            }

            // Remove duplicates
            $allAssignedProjectIds = array_unique($allAssignedProjectIds);

            // Check if all requested project IDs are in the assigned projects
            $invalidProjectIds = array_diff($projectIds, $allAssignedProjectIds);
            if (!empty($invalidProjectIds)) {
                return $this->error('Some project IDs are not assigned to your subcontractor account: ' . implode(', ', $invalidProjectIds), 422);
            }

            // Verify that all projects exist and are not deleted
            $existingProjects = Project::whereIn('id', $projectIds)
                ->where('is_deleted', 0)
                ->pluck('id')
                ->toArray();

            $missingProjectIds = array_diff($projectIds, $existingProjects);
            if (!empty($missingProjectIds)) {
                return $this->error('Some project IDs do not exist or are deleted: ' . implode(', ', $missingProjectIds), 422);
            }
        }

        // Get previously assigned project IDs before updating
        $previouslyAssignedProjectIds = $empSubcontractor->project_ids ?? [];

        // Update the employees_subcontractors_metas record with the project_ids
        // Note: company_id is optional now - employee can work across multiple companies
        $empSubcontractor->update([
            'project_ids' => $projectIds,
        ]);

        // Calculate newly assigned projects (projects in new list but not in old list)
        $newlyAssignedProjectIds = array_diff($projectIds, $previouslyAssignedProjectIds);

        // Create invitation records for newly assigned projects
        if (!empty($newlyAssignedProjectIds)) {
            // Get the first accepted invitation for this employee from the same subcontractor to any customer
            // We'll use this to copy the statuses for new projects from the same customer
            $firstAcceptedInvitation = SubcontractorEmployeeInvitation::where('employee_id', $empId)
                ->where('subcontractor_id', $subcontractorId)
                ->where('invitation_status', 'accepted')
                ->where('status', 1)
                ->orderBy('id', 'asc') // Get the first (oldest) accepted invitation
                ->first();

            // Process each newly assigned project
            foreach ($newlyAssignedProjectIds as $projectId) {
                // Get project details to find customer_id and workspace_id
                $project = Project::find($projectId);
                
                if (!$project) {
                    continue; // Skip if project doesn't exist
                }

                $customerId = $project->customer_id;
                $workspaceId = $project->workspace_id;

                // Check if invitation already exists for this project
                $existingInvitation = SubcontractorEmployeeInvitation::where('employee_id', $empId)
                    ->where('subcontractor_id', $subcontractorId)
                    ->where('project_id', $projectId)
                    ->first();

                if ($existingInvitation) {
                    // Invitation already exists, skip
                    continue;
                }

                // Check if there's an accepted invitation for this employee from the same subcontractor to the same customer
                $customerAcceptedInvitation = SubcontractorEmployeeInvitation::where('employee_id', $empId)
                    ->where('subcontractor_id', $subcontractorId)
                    ->where('customer_id', $customerId)
                    ->where('invitation_status', 'accepted')
                    ->where('status', 1)
                    ->orderBy('id', 'asc') // Get the first (oldest) accepted invitation for this customer
                    ->first();

                // Determine invitation status and statuses
                $invitationStatus = 'pending';
                $requiredDocsStatus = SubcontractorEmployeeInvitation::DOCS_STATUS_NOT_UPLOADED;
                $inductionStatus = SubcontractorEmployeeInvitation::INDUCTION_STATUS_NOT_SIGNED;
                $status = 1; // Active
                $respondedAt = null;

                if ($customerAcceptedInvitation) {
                    // Employee has already accepted invitation for this customer
                    // Copy statuses from the first accepted invitation for this customer
                    $invitationStatus = 'accepted';
                    $requiredDocsStatus = $customerAcceptedInvitation->required_docs_status ?? SubcontractorEmployeeInvitation::DOCS_STATUS_NOT_UPLOADED;
                    $inductionStatus = $customerAcceptedInvitation->induction_status ?? SubcontractorEmployeeInvitation::INDUCTION_STATUS_NOT_SIGNED;
                    $respondedAt = $customerAcceptedInvitation->responded_at ?? now();
                } elseif ($firstAcceptedInvitation && $firstAcceptedInvitation->customer_id == $customerId) {
                    // Fallback: use first accepted invitation if it's for the same customer
                    $invitationStatus = 'accepted';
                    $requiredDocsStatus = $firstAcceptedInvitation->required_docs_status ?? SubcontractorEmployeeInvitation::DOCS_STATUS_NOT_UPLOADED;
                    $inductionStatus = $firstAcceptedInvitation->induction_status ?? SubcontractorEmployeeInvitation::INDUCTION_STATUS_NOT_SIGNED;
                    $respondedAt = $firstAcceptedInvitation->responded_at ?? now();
                }

                // Create invitation record for this project
                try {
                    SubcontractorEmployeeInvitation::create([
                        'employee_id' => $empId,
                        'subcontractor_id' => $subcontractorId,
                        'project_id' => $projectId,
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId,
                        'invitation_status' => $invitationStatus,
                        'status' => $status,
                        'required_docs_status' => $requiredDocsStatus,
                        'induction_status' => $inductionStatus,
                        'responded_at' => $respondedAt,
                    ]);
                } catch (\Exception $e) {
                    // Log error but don't fail the request if invitation creation fails
                    // This could happen if there's a unique constraint violation (duplicate)
                    Log::error('Error creating invitation record for newly assigned project: ' . $e->getMessage(), [
                        'employee_id' => $empId,
                        'subcontractor_id' => $subcontractorId,
                        'project_id' => $projectId,
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId,
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }
        }

        // Send email notification to employee only about newly assigned projects
        if (!empty($newlyAssignedProjectIds) && !empty($employee->email)) {
            try {
                // Get employee full name
                $employeeName = trim(
                    ($employee->first_name ?? '') . ' ' .
                        ($employee->middle_name ?? '') . ' ' .
                        ($employee->last_name ?? '')
                );
                if (empty($employeeName)) {
                    $employeeName = $employee->email;
                }

                // Get names of newly assigned projects only
                $newProjects = Project::whereIn('id', $newlyAssignedProjectIds)
                    ->where('is_deleted', 0)
                    ->pluck('title', 'id')
                    ->toArray();
                $newProjectNames = array_values($newProjects);

                // Get first subcontractor company for customer_id and workspace_id (for email template)
                $firstSubcontractorCompany = $subcontractorCompanies->first();
                $customerId = $firstSubcontractorCompany->customer_id ?? null;
                $workspaceId = $firstSubcontractorCompany->workspace_id ?? null;

                // Get subcontractor name
                $subcontractorName = $subcontractor->name ?? $subcontractor->company_name ?? 'Your Subcontractor';

                // Prepare email content
                $subject = 'New Project Assignment Notification - ' . env('APP_NAME', 'WMS');
                $emailContent = view('Emails.project-assignment-notification', [
                    'subject' => $subject,
                    'employee_name' => $employeeName,
                    'employee_email' => $employee->email,
                    'subcontractor_name' => $subcontractorName,
                    'project_names' => $newProjectNames,
                    'customer_id' => $customerId,
                    'workspace_id' => $workspaceId,
                ])->render();

                // Send email
                $emailParams = [
                    'to' => $employee->email,
                    'subject' => $subject,
                    'msg' => $emailContent,
                ];

                $this->SendInstantEmail($emailParams);
            } catch (\Exception $e) {
                // Log error but don't fail the request if email fails
                Log::error('Error sending project assignment email to employee: ' . $e->getMessage(), [
                    'employee_id' => $empId,
                    'newly_assigned_project_ids' => $newlyAssignedProjectIds,
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return $this->success([
            'emp_id' => $empSubcontractor->emp_id,
            'project_ids' => $empSubcontractor->project_ids,
        ], 'Projects assigned to employee successfully.');
    }

    public function getRequiredDocumentsForCompany(Request $request)
    {
        $subcontractor = Auth::user();

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer',
            'workspace_id' => 'required|integer',
        ], [
            'customer_id.required' => 'Customer ID is required.',
            'workspace_id.required' => 'Workspace ID is required.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $customerId = $request->customer_id;
        $workspaceId = $request->workspace_id;

        // Verify that the subcontractor is associated with this company
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if (!$subcontractorCompany) {
            return $this->error('You are not associated with this company.', 403);
        }

        // Get all required documents for subcontractors from this company
        $requiredDocuments = RequiredDocument::where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->with(['requiredDocumentActiveField'])
            ->get();

        // Get already uploaded documents for this subcontractor
        $uploadedDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractor->id)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->get()
            ->groupBy('required_document_id');

        // Merge uploaded data with required documents
        $documentsWithStatus = $requiredDocuments->map(function ($doc) use ($uploadedDocuments, $subcontractor, $customerId, $workspaceId) {
            $docId = $doc->id;
            $uploaded = $uploadedDocuments->get($docId) ?? collect();

            // Map uploaded values to fields
            $uploadedFieldValues = [];
            foreach ($uploaded as $uploadedDoc) {
                $uploadedFieldValues[$uploadedDoc->required_document_field_id] = [
                    'id' => $uploadedDoc->id,
                    'value' => $uploadedDoc->value,
                    'updated_at' => $uploadedDoc->updated_at,
                ];
            }

            // Check if document is complete
            $isComplete = SubcontractorRequiredDocument::isDocumentComplete(
                $subcontractor->id,
                $customerId,
                $workspaceId,
                $docId
            );

            // Calculate document-level approval status
            $docApprovalStatus = null;
            $docRejectionReason = null;
            $docApprovedBy = null;
            $docApprovedAt = null;

            if ($uploaded->isNotEmpty()) {
                $allApproved = $uploaded->every(fn($d) => $d->approval_status === 'approved');
                $anyRejected = $uploaded->contains(fn($d) => $d->approval_status === 'rejected');

                if ($allApproved) {
                    $docApprovalStatus = 'approved';
                    $firstApproved = $uploaded->first();
                    $docApprovedBy = $firstApproved->approved_by;
                    $docApprovedAt = $firstApproved->approved_at;
                } elseif ($anyRejected) {
                    $docApprovalStatus = 'rejected';
                    $rejectedDoc = $uploaded->firstWhere('approval_status', 'rejected');
                    $docRejectionReason = $rejectedDoc->rejection_reason;
                    $docApprovedBy = $rejectedDoc->approved_by;
                    $docApprovedAt = $rejectedDoc->approved_at;
                } else {
                    $docApprovalStatus = 'pending';
                }
            }

            return [
                'id' => $doc->id,
                'title' => $doc->title,
                'is_complete' => $isComplete,
                'approval_status' => $docApprovalStatus,
                'rejection_reason' => $docRejectionReason,
                'approved_by' => $docApprovedBy,
                'approved_at' => $docApprovedAt,
                'fields' => $doc->requiredDocumentActiveField->map(function ($field) use ($uploadedFieldValues) {
                    $fieldId = $field->id;
                    $uploadedValue = $uploadedFieldValues[$fieldId] ?? null;

                    return [
                        'id' => $field->id,
                        'field_name' => $field->field_name,
                        'field_type' => $field->field_type,
                        'field_required' => $field->field_required,
                        'priority' => $field->priority,
                        'uploaded_value' => $uploadedValue ? $uploadedValue['value'] : null,
                        'uploaded_id' => $uploadedValue ? $uploadedValue['id'] : null,
                        'last_updated' => $uploadedValue ? $uploadedValue['updated_at'] : null,
                    ];
                }),
            ];
        });

        // Get overall status
        $allComplete = $documentsWithStatus->every(function ($doc) {
            return $doc['is_complete'];
        });

        // Calculate document counts by status (separate logic to avoid affecting existing functionality)
        $documentCounts = [
            'approved' => 0,
            'pending' => 0,
            'rejected' => 0,
            'not_uploaded' => 0,
            'total' => $documentsWithStatus->count(),
        ];

        foreach ($documentsWithStatus as $doc) {
            $status = $doc['approval_status'];
            
            if ($status === null) {
                // Document has not been uploaded yet
                $documentCounts['not_uploaded']++;
            } elseif ($status === 'approved') {
                $documentCounts['approved']++;
            } elseif ($status === 'pending') {
                $documentCounts['pending']++;
            } elseif ($status === 'rejected') {
                $documentCounts['rejected']++;
            }
        }

        return $this->success([
            'required_docs_status' => $subcontractorCompany->required_docs_status,
            'required_docs_status_label' => $subcontractorCompany->required_docs_status_label,
            'rejection_reason' => $subcontractorCompany->rejection_reason,
            'all_documents_complete' => $allComplete,
            'document_counts' => $documentCounts,
            'documents' => $documentsWithStatus,
        ], 'Required documents retrieved successfully');
    }

    public function storeRequiredDocuments(Request $request)
    {

        $subcontractor = Auth::user();
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|integer',
            'workspace_id' => 'required|integer',
        ], [
            'customer_id.required' => 'Customer ID is required.',
            'workspace_id.required' => 'Workspace ID is required.',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422);
        }

        $customerId = $request->customer_id;
        $workspaceId = $request->workspace_id;

        // Verify that the subcontractor is associated with this company
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->where('del', '0')
            ->first();

        if (!$subcontractorCompany) {
            return $this->error('You are not associated with this company.', 403);
        }

        try {
            // Get all required documents for subcontractors from this company
            $required_documents = RequiredDocument::with('requiredDocumentActiveField')
                ->where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('for_subcontractor', true)
                ->where('del', '0')
                ->get();

            // Ensure upload directory exists
            $uploadDirectory = public_path('SubcontractorDocuments');
            if (!file_exists($uploadDirectory)) {
                mkdir($uploadDirectory, 0755, true);
            }

            $i = 0;

            foreach ($required_documents as $required_document) {
                foreach ($required_document->requiredDocumentActiveField as $required_document_field) {
                    // Check if field is required and not provided
                    if (
                        $required_document_field->field_required == "1" &&
                        !$request->filled('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id) &&
                        !$request->hasFile('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id)
                    ) {

                        // Check if there's an existing record for this required field
                        $existingRecord = SubcontractorRequiredDocument::where([
                            'subcontractor_id' => $subcontractor->id,
                            'customer_id' => $customerId,
                            'workspace_id' => $workspaceId,
                            'required_document_id' => $required_document_field->doc_id,
                            'required_document_field_id' => $required_document_field->id,
                            'del' => '0'
                        ])->first();

                        // Only return error if no existing record exists
                        if (!$existingRecord) {
                            return $this->error($required_document_field->field_name . ' field is required', 422);
                        }
                    }

                    // Validate file uploads
                    if ($required_document_field->field_type == "image" || $required_document_field->field_type == "file") {
                        if ($request->hasFile('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id)) {
                            $allowedExtensions = ['jpg', 'jpeg', 'png', 'docx', 'pdf', 'doc', 'xls', 'xlsx'];
                            $document_file = $request->file('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id);

                            if (!$document_file || !$document_file->isValid()) {
                                Log::error('Invalid file uploaded', [
                                    'subcontractor_id' => $subcontractor->id,
                                    'doc_id' => $required_document_field->doc_id,
                                    'field_id' => $required_document_field->id
                                ]);
                                continue;
                            }

                            $extension = $document_file->getClientOriginalExtension();
                            $mimeType = $document_file->getMimeType();

                            // Validate extension and MIME type
                            $allowedMimeTypes = [
                                'image/jpeg',
                                'image/png',
                                'application/pdf',
                                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                                'application/msword',
                                'application/vnd.ms-excel',
                                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
                            ];

                            if (!in_array($extension, $allowedExtensions) || !in_array($mimeType, $allowedMimeTypes)) {
                                return $this->error($required_document_field->field_name . ' file must be a file of type: jpg, jpeg, png, docx, pdf, doc, xls, xlsx.', 422);
                            }

                            // Check file size (10MB max)
                            if ($document_file->getSize() > 10485760) {
                                return $this->error($required_document_field->field_name . ' file must not exceed 10MB.', 422);
                            }
                        }
                    }
                }
                $i++;
            }

            // Process and save documents
            $i = 0;
            foreach ($required_documents as $required_document) {
                foreach ($required_document->requiredDocumentActiveField as $required_document_field) {

                    // Handle file uploads (image or file type)
                    if ($required_document_field->field_type == "image" || $required_document_field->field_type == "file") {

                        // Check if a new file is being uploaded
                        if ($request->hasFile('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id)) {

                            // NEW FILE: Delete old record and create new one
                            $oldRecord = SubcontractorRequiredDocument::where([
                                'subcontractor_id' => $subcontractor->id,
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                                'required_document_id' => $required_document_field->doc_id,
                                'required_document_field_id' => $required_document_field->id,
                                'del' => '0'
                            ])->first();

                            // Delete old physical file if exists
                            if ($oldRecord && $oldRecord->value && file_exists(public_path($oldRecord->value))) {
                                unlink(public_path($oldRecord->value));
                            }

                            // Delete old record
                            if ($oldRecord) {
                                $oldRecord->delete();
                            }

                            $document_file = $request->file('document_array.' . $required_document_field->doc_id . '.' . $required_document_field->id);

                            if ($document_file && $document_file->isValid()) {
                                $documentName = $i . time() . '.' . $document_file->getClientOriginalExtension();

                                if ($document_file->move($uploadDirectory, $documentName)) {
                                    $filePath = 'SubcontractorDocuments/' . $documentName;

                                    SubcontractorRequiredDocument::create([
                                        'subcontractor_id' => $subcontractor->id,
                                        'customer_id' => $customerId,
                                        'workspace_id' => $workspaceId,
                                        'required_document_id' => $required_document_field->doc_id,
                                        'required_document_field_id' => $required_document_field->id,
                                        'value' => $filePath,
                                        'approval_status' => SubcontractorRequiredDocument::STATUS_PENDING,
                                        'rejection_reason' => null,
                                        'approved_by' => null,
                                        'approved_at' => null,
                                        'del' => '0',
                                    ]);
                                } else {
                                    Log::error('Failed to move uploaded file', [
                                        'subcontractor_id' => $subcontractor->id,
                                        'doc_id' => $required_document_field->doc_id,
                                        'field_id' => $required_document_field->id,
                                    ]);
                                }
                            }
                        } else {
                            // NO NEW FILE: Keep existing record, don't delete anything
                            // This preserves the existing image/file
                        }
                    }

                    // Handle text fields (text, date, number, etc.)
                    if ($required_document_field->field_type != "image" && $required_document_field->field_type != "file") {
                        if (isset($request->document_array[$required_document_field->doc_id][$required_document_field->id])) {
                            $textValue = $request->document_array[$required_document_field->doc_id][$required_document_field->id];

                            // Delete existing record first for text fields
                            SubcontractorRequiredDocument::where([
                                'subcontractor_id' => $subcontractor->id,
                                'customer_id' => $customerId,
                                'workspace_id' => $workspaceId,
                                'required_document_id' => $required_document_field->doc_id,
                                'required_document_field_id' => $required_document_field->id,
                                'del' => '0'
                            ])->delete();

                            if ($textValue !== null && $textValue !== '') {
                                SubcontractorRequiredDocument::create([
                                    'subcontractor_id' => $subcontractor->id,
                                    'customer_id' => $customerId,
                                    'workspace_id' => $workspaceId,
                                    'required_document_id' => $required_document_field->doc_id,
                                    'required_document_field_id' => $required_document_field->id,
                                    'value' => $textValue,
                                    'approval_status' => SubcontractorRequiredDocument::STATUS_PENDING,
                                    'rejection_reason' => null,
                                    'approved_by' => null,
                                    'approved_at' => null,
                                    'del' => '0',
                                ]);
                            }
                        }
                    }
                }
                $i++;
            }

            // Check if all documents are now complete and update status to pending approval
            if (SubcontractorRequiredDocument::areAllDocumentsComplete($subcontractor->id, $customerId, $workspaceId)) {
                // Set status to pending (4) - waiting for admin approval
                // Only update if current status is not already approved (1) or rejected (3)
                if ($subcontractorCompany->required_docs_status != SubcontractorCompany::STATUS_COMPLETE) {
                    $subcontractorCompany->update(['required_docs_status' => SubcontractorCompany::STATUS_PENDING]);
                }
            }

            // Refresh to get updated status
            $subcontractorCompany->refresh();

            return $this->success([
                'required_docs_status' => $subcontractorCompany->required_docs_status,
                'required_docs_status_label' => $subcontractorCompany->required_docs_status_label,
                'all_documents_complete' => SubcontractorRequiredDocument::areAllDocumentsComplete(
                    $subcontractor->id,
                    $customerId,
                    $workspaceId
                ),
            ], 'Documents saved successfully');
        } catch (\Exception $e) {
            Log::error('Error saving subcontractor required documents: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return $this->error('Failed to save documents. Please try again.', 500);
        }
    }

    /**
     * Get dashboard data for subcontractor
     * Returns summary cards, recent invitations, and urgent defects
     */
    public function getDashboard(Request $request)
    {
        $user = Auth::user();
        
        if (!$user || $user->user_type != config('constants.user_types.subcontractor')) {
            return $this->error('Unauthorized. Only subcontractors can access this endpoint.', 403);
        }

        try {
            // Get system date format from adminsettings (using authenticated user's ID as customer_id, no workspace filter)
            $dateFormatSetting = DB::table('adminsettings')
                ->where('customer_id', $user->id)
                ->where('key', 'system_date_format')
                ->first();
            
            // Convert system format to Carbon format
            $dateFormat = 'd-m-Y'; // Default format
            if ($dateFormatSetting && $dateFormatSetting->value) {
                $formatMap = [
                    'mm-dd-yyyy' => 'm-d-Y',
                    'dd-mm-yyyy' => 'd-m-Y',
                    'yyyy-mm-dd' => 'Y-m-d',
                    'mm/dd/yyyy' => 'm/d/Y',
                    'dd/mm/yyyy' => 'd/m/Y',
                    'yyyy/mm/dd' => 'Y/m/d',
                ];
                $dateFormat = $formatMap[strtolower($dateFormatSetting->value)] ?? 'd-m-Y';
            }
            
            // Get all company associations for this subcontractor
            $subcontractorCompanies = SubcontractorCompany::where('user_id', $user->id)
                ->where('del', '0')
                ->get();

            // Collect all project IDs from all company associations
            $allProjectIds = [];
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                $projectIds = $subcontractorCompany->project_ids ?? [];
                $allProjectIds = array_merge($allProjectIds, $projectIds);
            }
            $allProjectIds = array_unique($allProjectIds);

            // ==================== SUMMARY CARDS ====================
            
            // 1. Active Tenders: Count of tenders with status='open' and invite_status='invited'
            $activeTendersCount = Tender::whereHas('participants', function ($q) use ($user) {
                    $q->where('user_id', $user->id)
                      ->where('invite_status', 'invited');
                })
                ->where('is_published', 1)
                ->where('status', 'open')
                ->count();

            // 2. Assigned Projects: Count of unique active projects assigned to subcontractor
            $assignedProjectsCount = 0;
            if (!empty($allProjectIds)) {
                $assignedProjectsCount = Project::whereIn('id', $allProjectIds)
                    ->where('active', 1)
                    ->where('is_deleted', '0')
                    ->count();
            }

            // 3. Open Defects: Count of defects assigned to subcontractor with status=0 (open)
            // Defects can be assigned via subcontractor_id directly OR via defect_ids array in subcontractor_companies
            $defectIdsFromCompanies = [];
            foreach ($subcontractorCompanies as $subcontractorCompany) {
                $defectIds = $subcontractorCompany->defect_ids ?? [];
                $defectIdsFromCompanies = array_merge($defectIdsFromCompanies, $defectIds);
            }
            $defectIdsFromCompanies = array_unique($defectIdsFromCompanies);
            
            $openDefectsQuery = Defect::where('status', 0) // Open status
                ->where('del', '0');
            
            if (!empty($defectIdsFromCompanies)) {
                $openDefectsQuery->where(function ($q) use ($user, $defectIdsFromCompanies) {
                    $q->where('subcontractor_id', $user->id)
                      ->orWhereIn('id', $defectIdsFromCompanies);
                });
            } else {
                $openDefectsQuery->where('subcontractor_id', $user->id);
            }
            
            $openDefectsCount = $openDefectsQuery->count();

            // 4. Team Members: Count of active subcontractor employees
            $teamMembersCount = EmployeeSubcontractorMeta::where('subcontractor_id', $user->id)
                ->where('active', 1)
                ->distinct('emp_id')
                ->count('emp_id');

            // ==================== RECENT INVITATIONS ====================
            
            // Helper function to parse dates with multiple format support
            // Uses the user's preferred date format first to handle ambiguous formats correctly
            $parseDate = function ($dateValue) use ($dateFormat) {
                if (empty($dateValue)) {
                    return null;
                }
                
                // If already a Carbon instance, return it
                if ($dateValue instanceof Carbon) {
                    return $dateValue;
                }
                
                // If not a string, convert to string
                if (!is_string($dateValue)) {
                    $dateValue = (string) $dateValue;
                }
                
                $dateValue = trim($dateValue);
                
                // Build formats to try - user's preferred format FIRST
                $formatsToTry = [$dateFormat]; // Try user's format first
                
                // Add other common formats (excluding the user's format to avoid duplicates)
                $allFormats = [
                    'd-m-Y',    // dd-mm-yyyy
                    'm-d-Y',    // mm-dd-yyyy
                    'Y-m-d',    // yyyy-mm-dd
                    'd/m/Y',    // dd/mm/yyyy
                    'm/d/Y',    // mm/dd/yyyy
                    'Y/m/d',    // yyyy/mm/dd
                ];
                
                // Add formats that are not the user's preferred format
                foreach ($allFormats as $format) {
                    if ($format !== $dateFormat && !in_array($format, $formatsToTry)) {
                        $formatsToTry[] = $format;
                    }
                }
                
                // Try each format
                foreach ($formatsToTry as $format) {
                    try {
                        $parsed = Carbon::createFromFormat($format, $dateValue);
                        // Validate that the parsed date makes sense (e.g., for ambiguous formats)
                        // For mm-dd-yyyy vs dd-mm-yyyy, we trust the user's format preference
                        return $parsed;
                    } catch (\Exception $e) {
                        // Continue to next format
                    }
                }
                
                // If all formats fail, try Carbon::parse() as fallback
                try {
                    return Carbon::parse($dateValue);
                } catch (\Exception $e) {
                    Log::warning('Failed to parse date: ' . $dateValue . ' with format: ' . $dateFormat);
                    return null;
                }
            };
            
            // Get recent tenders where invite_status='invited' for this subcontractor
            $recentInvitations = Tender::with(['project:id,title', 'participants' => function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                }])
                ->whereHas('participants', function ($q) use ($user) {
                    $q->where('user_id', $user->id)
                      ->where('invite_status', 'invited');
                })
                ->where('is_published', 1)
                ->where('status', 'open')
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get()
                ->map(function ($tender) use ($user, $parseDate, $dateFormat) {
                    $participant = $tender->participants->first();
                    
                    // Check if user has submitted a bid
                    $hasSubmittedBid = TenderLineItemsBid::where('tender_id', $tender->id)
                        ->where('user_id', $user->id)
                        ->where('status', 'submitted')
                        ->exists();

                    // Extract reference number (e.g., "001" from "PT-01-2025-001")
                    $reference = $tender->reference ?? '';
                    $referenceNumber = '';
                    if (preg_match('/-(\d{3})$/', $reference, $matches)) {
                        $referenceNumber = $matches[1];
                    }

                    // Safely parse closing_date and format using system date format
                    $dueDate = null;
                    if ($tender->closing_date) {
                        try {
                            $parsedDate = $parseDate($tender->closing_date);
                            if ($parsedDate) {
                                $dueDate = $parsedDate->format($dateFormat);
                            }
                        } catch (\Exception $e) {
                            Log::warning('Failed to parse tender closing_date: ' . ($tender->closing_date ?? 'null'));
                        }
                    }

                    return [
                        'id' => $tender->id,
                        'reference' => $referenceNumber ?: $reference,
                        'title' => $tender->title,
                        'project_name' => $tender->project ? $tender->project->title : null,
                        'status' => $hasSubmittedBid ? 'Submitted' : 'Invited',
                        'due_date' => $dueDate,
                        'closing_date' => $tender->closing_date,
                    ];
                });

            // ==================== URGENT DEFECTS ====================
            
            // Get urgent defects (priority='high' or 'medium') with status=0 (open)
            // Defects can be assigned via subcontractor_id directly OR via defect_ids array in subcontractor_companies
            $urgentDefectsQuery = Defect::with(['project:id,title'])
                ->whereIn('priority', ['high', 'medium'])
                ->where('status', 0) // Open status
                ->where('del', '0');
            
            if (!empty($defectIdsFromCompanies)) {
                $urgentDefectsQuery->where(function ($q) use ($user, $defectIdsFromCompanies) {
                    $q->where('subcontractor_id', $user->id)
                      ->orWhereIn('id', $defectIdsFromCompanies);
                });
            } else {
                $urgentDefectsQuery->where('subcontractor_id', $user->id);
            }
            
            $urgentDefects = $urgentDefectsQuery
                ->orderByRaw("CASE WHEN priority = 'high' THEN 1 WHEN priority = 'medium' THEN 2 ELSE 3 END")
                ->orderBy('due_date', 'asc')
                ->limit(10)
                ->get()
                ->map(function ($defect) use ($parseDate, $dateFormat) {
                    // Safely parse due_date and format using system date format
                    $dueDate = null;
                    if ($defect->due_date) {
                        try {
                            $parsedDate = $parseDate($defect->due_date);
                            if ($parsedDate) {
                                $dueDate = $parsedDate->format($dateFormat);
                            }
                        } catch (\Exception $e) {
                            Log::warning('Failed to parse defect due_date: ' . ($defect->due_date ?? 'null'));
                        }
                    }

                    return [
                        'id' => $defect->id,
                        'priority' => ucfirst($defect->priority ?? 'medium'),
                        'description' => $defect->title ?? $defect->description,
                        'project_name' => $defect->project ? $defect->project->title : null,
                        'due_date' => $dueDate,
                    ];
                });

            // ==================== RESPONSE ====================
            
            $dashboardData = [
                'summary' => [
                    'active_tenders' => $activeTendersCount,
                    'assigned_projects' => $assignedProjectsCount,
                    'open_defects' => $openDefectsCount,
                    'team_members' => $teamMembersCount,
                ],
                'recent_invitations' => $recentInvitations,
                'urgent_defects' => $urgentDefects,
            ];

            return $this->success($dashboardData, 'Dashboard data retrieved successfully');
            
        } catch (\Exception $e) {
            Log::error('Error fetching subcontractor dashboard data: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'user_id' => $user->id ?? null
            ]);
            return $this->error('Failed to retrieve dashboard data. Please try again.', 500);
        }
    }
    public function updateProfile(Request $request)
    {
        return $this->updateSubcontractorProfile($request);
    }
}
