<?php

namespace App\Http\Controllers\Subcontractor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tender;
use App\Models\TenderLineItem;
use App\Models\TenderParticipant;
use App\Models\TenderLineItemsBid;
use App\Models\TenderAttachment;
use App\Models\TenderRfi;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Carbon\Carbon;

class SubcontractorTenderController extends Controller
{
    /**
     * Get all invited tenders for authenticated subcontractor
     */
    public function getInvitedTenders(Request $request)
    {
        $user = Auth::user();
        // First, update tenders with passed closing dates (if not allowing late submissions)
        // Get tender IDs first, then query with raw date to avoid format issues
        $tenderIds = Tender::whereHas('participants', function ($q) use ($user) {
            $q->where('user_id', $user->id)
                ->where('invite_status', 'invited');
        })
            ->where('is_published', 1)
            ->where('status', 'open')
            ->pluck('id');
        
        // Query with raw date field to get actual database value
        $tendersToUpdate = DB::table('tenders')
            ->whereIn('id', $tenderIds)
            ->whereNotNull('closing_date')
            ->select('id', 'closing_date', 'allow_late_submissions', 'status')
            ->get();
            
        foreach ($tendersToUpdate as $tenderRow) {
            // Get the full tender model for saving
            $tender = Tender::find($tenderRow->id);
            if (!$tender || !$tenderRow->closing_date) {
                continue; // Skip if no closing date
            }
            // Parse closing date - database should return YYYY-MM-DD format
            try {
                $closingDate = null;
                $rawClosingDate = $tenderRow->closing_date;
                
                // Database date column should be in YYYY-MM-DD format
                if (is_string($rawClosingDate)) {
                    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $rawClosingDate)) {
                        // Format: YYYY-MM-DD (standard database format)
                        $closingDate = Carbon::createFromFormat('Y-m-d', $rawClosingDate);
                    } elseif (preg_match('/^\d{2}-\d{2}-\d{4}$/', $rawClosingDate)) {
                        // Try both MM-DD-YYYY and DD-MM-YYYY formats
                        $dateMMDD = null;
                        $dateDDMM = null;
                        try {
                            $dateMMDD = Carbon::createFromFormat('m-d-Y', $rawClosingDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        try {
                            $dateDDMM = Carbon::createFromFormat('d-m-Y', $rawClosingDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        
                        // Use the date that makes sense (prefer past date if checking for past)
                        if ($dateMMDD && $dateDDMM) {
                            // If both parse successfully, prefer the one that's in the past (more likely correct)
                            $closingDate = $dateMMDD->isPast() ? $dateMMDD : ($dateDDMM->isPast() ? $dateDDMM : $dateMMDD);
                        } elseif ($dateMMDD) {
                            $closingDate = $dateMMDD;
                        } elseif ($dateDDMM) {
                            $closingDate = $dateDDMM;
                        } else {
                            throw new \Exception('Could not parse date in either MM-DD-YYYY or DD-MM-YYYY format');
                        }
                    } else {
                        // Try default parse
                        $closingDate = Carbon::parse($rawClosingDate);
                    }
                } else {
                    continue; // Skip if not a valid date format
                }
                
                if (!$closingDate) {
                    continue; // Skip if parsing failed
                }
            } catch (\Exception $e) {
                Log::warning('Failed to parse closing_date for tender', [
                    'tender_id' => $tenderRow->id,
                    'closing_date' => $rawClosingDate ?? $tenderRow->closing_date,
                    'error' => $e->getMessage()
                ]);
                continue; // Skip this tender if date parsing fails
            }
            
            // If closing date has passed, late submissions are not allowed (0 or null), and status is open, close the tender
            // allow_late_submissions: 0 = not allowed, 1 = allowed, null = not allowed (default)
            // Set closing date to end of day for proper comparison
            $closingDateEndOfDay = $closingDate->copy()->endOfDay();
            $isPast = $closingDateEndOfDay->isPast();
            $lateSubmissionsNotAllowed = ($tenderRow->allow_late_submissions == 0 || $tenderRow->allow_late_submissions === null);
            $isOpen = ($tenderRow->status == 'open');
            
       
            
            if ($isPast && $lateSubmissionsNotAllowed && $isOpen) {
                $tender->status = 'close';
                $tender->save();
            }
        }
        $query = Tender::with(['project', 'lineItems', 'participants'])
            ->whereHas('participants', function ($q) use ($user) {
                $q->where('user_id', $user->id)
                    ->where('invite_status', 'invited');
            })
            ->where('is_published', 1);
        // Filter by status if provided
        if ($request->filled('status')) {
            if ($request->status === 'awarded') {
                // For awarded status, only show if this subcontractor is assigned
                $query->where('status', 'awarded')
                    ->whereHas('participants', function ($participantQ) use ($user) {
                        $participantQ->where('user_id', $user->id)
                            ->where('invite_status', 'invited')
                            ->where('is_assigned', true);
                    });
            } else {
                // For other statuses (like 'open'), show normally
                $query->where('status', $request->status);
            }
        } else {
            // If no status filter, show open tenders OR awarded tenders (only if assigned to this user)
            $query->where(function ($q) use ($user) {
                // Show open tenders to all invited subcontractors
                $q->where('status', 'open')
                    // OR show awarded tenders only if this subcontractor is assigned
                    ->orWhere(function ($subQ) use ($user) {
                        $subQ->where('status', 'awarded')
                            ->whereHas('participants', function ($participantQ) use ($user) {
                                $participantQ->where('user_id', $user->id)
                                    ->where('invite_status', 'invited')
                                    ->where('is_assigned', true);
                            });
                    });
            });
        }
        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                    ->orWhere('reference', 'like', '%' . $search . '%')
                    ->orWhereHas('project', function ($subquery) use ($search) {
                        $subquery->where('title', 'like', '%' . $search . '%');
                    });
            });
        }
        $tenders = $query->orderBy('created_at', 'desc')->get();
        // Collect all unique customer_id and workspace_id pairs from the tenders
        $customerWorkspacePairs = [];
        foreach ($tenders as $tender) {
            $customerId = $tender->customer_id;
            $workspaceId = $tender->workspace_id;
            if ($customerId && $workspaceId) {
                $key = $customerId . '_' . $workspaceId;
                if (!isset($customerWorkspacePairs[$key])) {
                    $customerWorkspacePairs[$key] = [
                        'customer_id' => $customerId,
                        'workspace_id' => $workspaceId
                    ];
                }
            }
        }
        // Fetch company_name from adminsettings for all customer/workspace pairs in bulk
        $companyNamesMap = $this->getCompanyNamesBulk($customerWorkspacePairs);
        // Add assignment status and company name for each tender
        $tenders = $tenders->map(function ($tender) use ($user, $companyNamesMap) {
            $participant = $tender->participants->where('user_id', $user->id)->first();
            $isAssigned = $participant ? $participant->is_assigned : false;
            $tender->is_assigned = $isAssigned;
            $tender->is_awarded = $isAssigned; // Flag indicating if tender is awarded to this subcontractor
            $tender->assigned_at = $participant ? $participant->assigned_at : null;
            // Check if user has submitted a bid
            $hasBid = TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $user->id)
                ->where('status', 'submitted')
                ->exists();
            $tender->has_submitted_bid = $hasBid;
            // Add company name
            $companyName = null;
            if ($tender->customer_id && $tender->workspace_id) {
                $key = $tender->customer_id . '_' . $tender->workspace_id;
                $companyName = $companyNamesMap[$key] ?? null;
            }
            $tender->company_name = $companyName;
            return $tender;
        });
        return $this->success($tenders, 'Invited tenders retrieved successfully');
    }

    /**
     * Get invited tender by ID with documents/attachments
     */
    public function getInvitedTenderById($id)
    {
        $user = Auth::user();
        // Get tender with relations
        $tender = Tender::with(['project', 'lineItems'])
            ->where('id', $id)
            ->where('is_published', 1)
            ->whereIn('status', ['open', 'awarded'])
            ->first();

        if (!$tender) {
            return $this->message('Tender not found or you are not invited', 404);
        }

        // Check if user has submitted a bid
        $hasBid = TenderLineItemsBid::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('status', 'submitted')
            ->exists();
        $tender->has_submitted_bid = $hasBid;

        // Check if tender is awarded to this subcontractor
        $participant = TenderParticipant::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->first();
        $isAwarded = $participant ? $participant->is_assigned : false;
        $tender->is_awarded = $isAwarded; // Flag indicating if tender is awarded to this subcontractor
        $tender->is_assigned = $isAwarded;
        $tender->assigned_at = $participant ? $participant->assigned_at : null;

        // Get bid prices for this user (both draft and submitted)
        $bidPrices = TenderLineItemsBid::with('lineItem')
            ->where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->orderBy('tender_line_items_id')
            ->get();

        // Format bid prices with line item details
        $formattedBidPrices = $bidPrices->map(function ($bid) {
            return [
                'id' => $bid->id,
                'line_item_id' => $bid->tender_line_items_id,
                'line_item_description' => $bid->lineItem->description ?? null,
                'line_item_unit' => $bid->lineItem->unit ?? null,
                'line_item_quantity' => $bid->lineItem->quantity ?? null,
                'bid_price' => $bid->bid_price,
                'status' => $bid->status,
                'total_base_bid' => $bid->total_base_bid,
                'exclusions_clarifications' => $bid->exclusions_clarifications,
                'is_late_submission' => $bid->is_late_submission ?? false,
                'submitted_at' => $bid->submitted_at,
                'created_at' => $bid->created_at,
                'updated_at' => $bid->updated_at,
            ];
        });

        // Get summary bid information (from first bid record if exists)
        $firstBid = $bidPrices->first();
        $tender->bid_prices = $formattedBidPrices;
        $tender->bid_summary = $firstBid ? [
            'total_base_bid' => $firstBid->total_base_bid ?? 0,
            'exclusions_clarifications' => $firstBid->exclusions_clarifications ?? null,
            'status' => $firstBid->status ?? null,
            'is_late_submission' => $firstBid->is_late_submission ?? false,
            'submitted_at' => $firstBid->submitted_at ?? null,
        ] : null;

        // Get admin uploaded documents/attachments (entity_type = 'customer')
        $documents = TenderAttachment::where('entity_type', 'customer')
            ->where('entity_id', $tender->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $tender->documents = $documents;

        // Get bid attachments (if user has submitted a bid)
        if ($hasBid) {
            $entityId = $tender->id . '_' . $user->id;
            $bidAttachments = TenderAttachment::where('entity_type', 'tender_bid')
                ->where('entity_id', $entityId)
                ->orderBy('created_at', 'desc')
                ->get();
            $tender->bid_attachments = $bidAttachments;
        } else {
            $tender->bid_attachments = [];
        }

        // Get customer user information (the user who created the tender)
        $customerUser = null;
        if ($tender->customer_id) {
            $customerUser = User::where('id', $tender->customer_id)
                ->select('id', 'name', 'email', 'mobile_number')
                ->first();
        }

        // Get customer company information from adminsettings
        $companyName = null;
        $companyLogo = null;

        if ($tender->customer_id && $tender->workspace_id) {
            $settings = DB::table('adminsettings')
                ->where('customer_id', $tender->customer_id)
                ->where('workspace', $tender->workspace_id)
                ->whereIn('key', ['company_company_name', 'brand_logo_dark'])
                ->get();

            // Transform settings into key-value pairs
            $settingsData = [];
            foreach ($settings as $setting) {
                $settingsData[$setting->key] = $setting->value;
            }

            $companyName = $settingsData['company_company_name'] ?? null;
            $companyLogo = $settingsData['brand_logo_dark'] ?? null;
        }

        // Add customer info to tender response
        $tender->customer_info = [
            // Customer user information
            'customer' => $customerUser ? [
                'id' => $customerUser->id,
                'name' => $customerUser->name,
                'email' => $customerUser->email,
                'mobile_number' => $customerUser->mobile_number,
                'company_name' => $companyName,
                'company_logo' => $companyLogo,
            ] : null,
        ];

        return $this->success($tender, 'Invited tender retrieved successfully');
    }

    /**
     * Add/Update bid prices for line items
     */
    public function addBidPrices(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'line_items' => 'required|array|min:1',
            'line_items.*.line_item_id' => 'required|integer|exists:tender_line_items,id',
            'line_items.*.bid_price' => 'required|numeric|min:0',
            'total_base_bid' => 'nullable|numeric|min:0',
            'exclusions_clarifications' => 'nullable|string',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $user = Auth::user();
        // Verify tender is published and open
        $tender = Tender::find($request->tender_id);
        if (!$tender || $tender->is_published != 1 || $tender->status != 'open') {
            return $this->message('Tender not found or not available for bidding', 404);
        }
        // Verify user is invited
        $participant = TenderParticipant::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('invite_status', 'invited')
            ->first();
        if (!$participant) {
            return $this->message('You are not invited to this tender', 403);
        }
        DB::beginTransaction();
        try {
            $ids = $this->getCustomerAndWorkspaceIds();
            if (!$ids) {
                return $this->message('Unable to determine customer or workspace', 400);
            }
            
            // Load line items with quantities for calculating total_base_bid
            $lineItemsMap = TenderLineItem::where('tender_id', $tender->id)
                ->get()
                ->keyBy('id');
            
            // Calculate total if not provided - multiply bid_price by quantity for each line item
            if ($request->has('total_base_bid')) {
                $totalBaseBid = $request->total_base_bid;
            } else {
                $totalBaseBid = 0;
                foreach ($request->line_items as $item) {
                    $lineItem = $lineItemsMap->get($item['line_item_id']);
                    $quantity = $lineItem ? ($lineItem->quantity ?? 1) : 1;
                    $totalBaseBid += (float)$item['bid_price'] * $quantity;
                }
            }
            
            // Update or create bid prices for each line item
            foreach ($request->line_items as $item) {
                TenderLineItemsBid::updateOrCreate(
                    [
                        'tender_id' => $tender->id,
                        'tender_line_items_id' => $item['line_item_id'],
                        'user_id' => $user->id,
                    ],
                    [
                        'bid_price' => $item['bid_price'],
                        'total_base_bid' => $totalBaseBid,
                        'exclusions_clarifications' => $request->exclusions_clarifications,
                        'status' => 'draft',
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]
                );
            }
            DB::commit();
            return $this->success(null, 'Bid prices saved successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving bid prices: ' . $e->getMessage());
            return $this->message('Failed to save bid prices', 500);
        }
    }

    /**
     * Submit bid with attachments
     */
    public function submitBid(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'line_items' => 'nullable|array|min:1',
            'line_items.*.line_item_id' => [
                'required_with:line_items',
                'integer',
                Rule::exists('tender_line_items', 'id')->where(function ($query) use ($request) {
                    return $query->where('tender_id', $request->tender_id);
                }),
            ],
            'line_items.*.bid_price' => 'required_with:line_items|numeric|min:0',
            'total_base_bid' => 'nullable|numeric|min:0',
            'exclusions_clarifications' => 'nullable|string',
            'attachments' => 'nullable|array',
        ]);
        // Custom validation for nested file attachments
        if ($request->has('attachments') && is_array($request->attachments)) {
            foreach ($request->attachments as $index => $attachment) {
                // Validate title
                if (!isset($attachment['title']) || empty(trim($attachment['title']))) {
                    $validator->errors()->add("attachments.{$index}.title", "The attachments.{$index}.title field is required.");
                } elseif (strlen($attachment['title']) > 255) {
                    $validator->errors()->add("attachments.{$index}.title", "The attachments.{$index}.title may not be greater than 255 characters.");
                }
                // Validate file - check if file exists in the nested structure
                $fileKey = "attachments.{$index}.file";
                if (!$request->hasFile($fileKey)) {
                    $validator->errors()->add("attachments.{$index}.file", "The attachments.{$index}.file must be a file.");
                } else {
                    $file = $request->file($fileKey);
                    if (!$file || !$file->isValid()) {
                        $validator->errors()->add("attachments.{$index}.file", "The attachments.{$index}.file is not valid.");
                    } else {
                        // Validate file type
                        $allowedMimes = ['pdf', 'doc', 'docx', 'xlsx', 'jpg', 'jpeg', 'png'];
                        $extension = strtolower($file->getClientOriginalExtension());
                        if (!in_array($extension, $allowedMimes)) {
                            $validator->errors()->add("attachments.{$index}.file", "The attachments.{$index}.file must be a file of type: pdf, doc, docx, xlsx, jpg, jpeg, png.");
                        }
                        // Validate file size (10240 KB = 10 MB)
                        if ($file->getSize() > 10240 * 1024) {
                            $validator->errors()->add("attachments.{$index}.file", "The attachments.{$index}.file may not be greater than 10240 kilobytes.");
                        }
                    }
                }
            }
        }
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $user = Auth::user();
        $tender = Tender::find($request->tender_id);
        if (!$tender || $tender->is_published != 1 || $tender->status != 'open') {
            return $this->message('Tender not found or not available for bidding', 404);
        }
        // Check if tender is closed
        if ($tender->closing_date) {
            try {
                $closingDate = null;
                if ($tender->closing_date instanceof Carbon) {
                    $closingDate = $tender->closing_date;
                } elseif (is_string($tender->closing_date)) {
                    // Try different date formats
                    if (preg_match('/^\d{2}-\d{2}-\d{4}$/', $tender->closing_date)) {
                        // Format: MM-DD-YYYY
                        $closingDate = Carbon::createFromFormat('m-d-Y', $tender->closing_date);
                    } elseif (preg_match('/^\d{4}-\d{2}-\d{2}$/', $tender->closing_date)) {
                        // Format: YYYY-MM-DD
                        $closingDate = Carbon::parse($tender->closing_date);
                    } else {
                        // Try default parse
                        $closingDate = Carbon::parse($tender->closing_date);
                    }
                }
                if ($closingDate && $closingDate->isPast()) {
                    if (!$tender->allow_late_submissions) {
                        return $this->message('Tender has closed and late submissions are not allowed', 400);
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Failed to parse closing_date for tender validation', [
                    'tender_id' => $tender->id,
                    'closing_date' => $tender->closing_date,
                    'error' => $e->getMessage()
                ]);
                // Continue if date parsing fails
            }
        }
        // Verify user is invited
        $participant = TenderParticipant::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('invite_status', 'invited')
            ->first();
        if (!$participant) {
            return $this->message('You are not invited to this tender', 403);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        
        // Load line items with quantities for calculating total_base_bid
        $lineItemsMap = TenderLineItem::where('tender_id', $tender->id)
            ->get()
            ->keyBy('id');
        
        // Helper function to calculate total_base_bid by multiplying bid_price by quantity
        $calculateTotalBaseBid = function($bids) use ($lineItemsMap) {
            return $bids->sum(function($bid) use ($lineItemsMap) {
                $lineItem = $lineItemsMap->get($bid->tender_line_items_id);
                $quantity = $lineItem ? ($lineItem->quantity ?? 1) : 1;
                return (float)$bid->bid_price * $quantity;
            });
        };
        
        // Helper function to calculate total from request line_items
        $calculateTotalFromRequest = function($requestLineItems) use ($lineItemsMap) {
            $total = 0;
            foreach ($requestLineItems as $item) {
                $lineItem = $lineItemsMap->get($item['line_item_id']);
                $quantity = $lineItem ? ($lineItem->quantity ?? 1) : 1;
                $total += (float)$item['bid_price'] * $quantity;
            }
            return $total;
        };
        
        // Check if submitted bid exists (last submitted bid)
        $existingSubmittedBids = TenderLineItemsBid::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('status', 'submitted')
            ->get();
        // Check if draft bid prices exist
        $bidPrices = TenderLineItemsBid::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('status', 'draft')
            ->get();
        // Check if submission is late
        $isLateSubmission = false;
        if ($tender->closing_date) {
            try {
                $closingDate = null;
                if ($tender->closing_date instanceof Carbon) {
                    $closingDate = $tender->closing_date;
                } elseif (is_string($tender->closing_date)) {
                    // Try different date formats
                    if (preg_match('/^\d{2}-\d{2}-\d{4}$/', $tender->closing_date)) {
                        // Format: MM-DD-YYYY
                        $closingDate = Carbon::createFromFormat('m-d-Y', $tender->closing_date);
                    } elseif (preg_match('/^\d{4}-\d{2}-\d{2}$/', $tender->closing_date)) {
                        // Format: YYYY-MM-DD
                        $closingDate = Carbon::parse($tender->closing_date);
                    } else {
                        // Try default parse
                        $closingDate = Carbon::parse($tender->closing_date);
                    }
                }
                $isLateSubmission = $closingDate && $closingDate->isPast();
            } catch (\Exception $e) {
                Log::warning('Failed to parse closing_date for late submission check', [
                    'tender_id' => $tender->id,
                    'closing_date' => $tender->closing_date,
                    'error' => $e->getMessage()
                ]);
                // Default to false if parsing fails
                $isLateSubmission = false;
            }
        }
        $exclusionsClarifications = $request->exclusions_clarifications;
        // If there are existing submitted bids, update them instead of creating new ones
        if (!$existingSubmittedBids->isEmpty()) {
            // Calculate total if not provided
            if ($request->has('line_items') && !empty($request->line_items)) {
                $totalBaseBid = $request->total_base_bid ?? $calculateTotalFromRequest($request->line_items);
                // Update or create bid prices for each line item
                foreach ($request->line_items as $item) {
                    TenderLineItemsBid::updateOrCreate(
                        [
                            'tender_id' => $tender->id,
                            'tender_line_items_id' => $item['line_item_id'],
                            'user_id' => $user->id,
                        ],
                        [
                            'bid_price' => $item['bid_price'],
                            'total_base_bid' => $totalBaseBid,
                            'exclusions_clarifications' => $exclusionsClarifications,
                            'status' => 'submitted',
                            'is_late_submission' => $isLateSubmission,
                            'submitted_at' => now(),
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]
                    );
                }
            } else {
                // If no line_items provided, just update existing submitted bids with new exclusions_clarifications
                // Calculate total by multiplying bid_price by quantity for each line item
                $totalBaseBid = $calculateTotalBaseBid($existingSubmittedBids);
                TenderLineItemsBid::where('tender_id', $tender->id)
                    ->where('user_id', $user->id)
                    ->where('status', 'submitted')
                    ->update([
                        'total_base_bid' => $totalBaseBid,
                        'exclusions_clarifications' => $exclusionsClarifications ?? $existingSubmittedBids->first()->exclusions_clarifications,
                        'is_late_submission' => $isLateSubmission,
                        'submitted_at' => now(),
                    ]);
            }
        } else {
            // No existing submitted bid, check for draft bids or create new ones
            if ($bidPrices->isEmpty() && $request->has('line_items') && !empty($request->line_items)) {
                // Calculate total if not provided - multiply bid_price by quantity for each line item
                $totalBaseBid = $request->total_base_bid ?? $calculateTotalFromRequest($request->line_items);
                // Create bid prices for each line item
                foreach ($request->line_items as $item) {
                    TenderLineItemsBid::create([
                        'tender_id' => $tender->id,
                        'tender_line_items_id' => $item['line_item_id'],
                        'user_id' => $user->id,
                        'bid_price' => $item['bid_price'],
                        'total_base_bid' => $totalBaseBid,
                        'exclusions_clarifications' => $exclusionsClarifications,
                        'status' => 'draft',
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                }
                // Refresh bid prices after creation
                $bidPrices = TenderLineItemsBid::where('tender_id', $tender->id)
                    ->where('user_id', $user->id)
                    ->where('status', 'draft')
                    ->get();
            }
            // If still no bid prices exist, return error
            if ($bidPrices->isEmpty()) {
                return $this->message('Please add bid prices before submitting', 400);
            }
            // Calculate total by multiplying bid_price by quantity for each line item
            $totalBaseBid = $calculateTotalBaseBid($bidPrices);
            $exclusionsClarifications = $exclusionsClarifications ?? $bidPrices->first()->exclusions_clarifications;
            // Update all draft bid prices to submitted status
            TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $user->id)
                ->where('status', 'draft')
                ->update([
                    'status' => 'submitted',
                    'total_base_bid' => $totalBaseBid,
                    'exclusions_clarifications' => $exclusionsClarifications,
                    'is_late_submission' => $isLateSubmission,
                    'submitted_at' => now(),
                ]);
        }
        // Handle attachments - delete old attachments if updating existing bid
        $entityId = $tender->id . '_' . $user->id;
        if (!$existingSubmittedBids->isEmpty()) {
            // Delete old attachments when updating existing bid
            $oldAttachments = TenderAttachment::where('entity_type', 'tender_bid')
                ->where('entity_id', $entityId)
                ->get();
            foreach ($oldAttachments as $oldAttachment) {
                $oldFilePath = public_path($oldAttachment->file_path);
                if (file_exists($oldFilePath)) {
                    @unlink($oldFilePath);
                }
            }
            TenderAttachment::where('entity_type', 'tender_bid')
                ->where('entity_id', $entityId)
                ->delete();
        }
        // Add new attachments
        if ($request->has('attachments') && is_array($request->attachments)) {
            $uploadPath = 'tender_bid_attachments';
            $fullUploadPath = public_path($uploadPath);
            if (!file_exists($fullUploadPath)) {
                mkdir($fullUploadPath, 0755, true);
            }
            foreach ($request->attachments as $index => $attachment) {
                if (isset($attachment['file']) && $request->hasFile("attachments.{$index}.file")) {
                    $file = $request->file("attachments.{$index}.file");
                    if ($file->isValid()) {
                        $filename = time() . '_' . uniqid() . '_' . $file->getClientOriginalName();
                        $file->move($fullUploadPath, $filename);
                        $filePath = $uploadPath . '/' . $filename;
                        $title = $attachment['title'] ?? $file->getClientOriginalName();
                        TenderAttachment::create([
                            'entity_type' => 'tender_bid',
                            'entity_id' => $entityId,
                            'title' => $title,
                            'file_name' => $filename,
                            'file_path' => $filePath,
                            'uploaded_by' => $user->id,
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);
                    }
                }
            }
        }
        return $this->message('Bid submitted successfully');
    }

    /**
     * Get my submitted bids
     */
    public function getMyBids(Request $request)
    {
        $user = Auth::user();
        $query = Tender::with(['project', 'lineItems'])
            ->whereHas('bids', function ($q) use ($user) {
                $q->where('user_id', $user->id)
                    ->where('status', 'submitted');
            });
        if ($request->filled('tender_id')) {
            $query->where('id', $request->tender_id);
        }
        $tenders = $query->orderBy('created_at', 'desc')->get();
        $tenders = $tenders->map(function ($tender) use ($user) {
            $bid = TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $user->id)
                ->where('status', 'submitted')
                ->first();
            // Check if tender is awarded to this subcontractor
            $participant = TenderParticipant::where('tender_id', $tender->id)
                ->where('user_id', $user->id)
                ->first();
            $isAwarded = $participant ? $participant->is_assigned : false;
            $tender->bid_total = $bid->total_base_bid ?? 0;
            $tender->bid_submitted_at = $bid->submitted_at ?? null;
            $tender->is_late_submission = $bid->is_late_submission ?? false;
            $tender->is_awarded = $isAwarded; // Flag indicating if tender is awarded to this subcontractor
            $tender->assigned_at = $participant ? $participant->assigned_at : null;
            // Get attachments
            $tender->attachments = TenderAttachment::where('entity_type', 'tender_bid')
                ->where('entity_id', $tender->id . '_' . $user->id)
                ->get();
            return $tender;
        });
        return $this->success($tenders, 'My bids retrieved successfully');
    }

    /**
     * Add RFI question
     */
    public function addRfiQuestion(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'question' => 'required|string',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $user = Auth::user();
        $tender = Tender::find($request->tender_id);
        if (!$tender || $tender->is_published != 1) {
            return $this->message('Tender not found', 404);
        }
        // Check RFI cut-off date
        if ($tender->rfi_cut_off_date) {
            try {
                $rfiCutOffDate = null;
                if ($tender->rfi_cut_off_date instanceof Carbon) {
                    $rfiCutOffDate = $tender->rfi_cut_off_date;
                } elseif (is_string($tender->rfi_cut_off_date)) {
                    // Try different date formats
                    if (preg_match('/^\d{2}-\d{2}-\d{4}$/', $tender->rfi_cut_off_date)) {
                        // Format: MM-DD-YYYY
                        $rfiCutOffDate = Carbon::createFromFormat('m-d-Y', $tender->rfi_cut_off_date);
                    } elseif (preg_match('/^\d{4}-\d{2}-\d{2}$/', $tender->rfi_cut_off_date)) {
                        // Format: YYYY-MM-DD
                        $rfiCutOffDate = Carbon::parse($tender->rfi_cut_off_date);
                    } else {
                        // Try default parse
                        $rfiCutOffDate = Carbon::parse($tender->rfi_cut_off_date);
                    }
                }
                if ($rfiCutOffDate && $rfiCutOffDate->isPast()) {
                    return $this->message('RFI cut-off date has passed', 400);
                }
            } catch (\Exception $e) {
                Log::warning('Failed to parse rfi_cut_off_date for tender', [
                    'tender_id' => $tender->id,
                    'rfi_cut_off_date' => $tender->rfi_cut_off_date,
                    'error' => $e->getMessage()
                ]);
                // Continue if date parsing fails (don't block RFI submission)
            }
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        $rfi = TenderRfi::create([
            'tender_id' => $tender->id,
            'user_id' => $user->id,
            'question' => $request->question,
            'status' => 'pending',
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
        ]);
        return $this->success($rfi, 'RFI question submitted successfully');
    }

    /**
     * Get RFIs for a tender (subcontractor sees only their RFIs)
     */
    public function getRfisForTender($tenderId)
    {
        $tender = Tender::find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found', 404);
        }
        $user = Auth::user();
        $rfis = TenderRfi::with('user')
            ->where('tender_id', $tenderId)
            ->where('user_id', $user->id) // Only show subcontractor's RFIs
            ->orderBy('created_at', 'desc')
            ->get();
        return $this->success($rfis, 'RFIs retrieved successfully');
    }
}
