<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Tender;
use App\Models\TenderLineItem;
use App\Models\TenderParticipant;
use App\Models\TenderLineItemsBid;
use App\Models\TenderAttachment;
use App\Models\TenderRfi;
use App\Models\User;
use App\Models\SubcontractorCompany;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;


class TenderController extends Controller
{
    /**
     * Get user's preferred date format (from BaseModel)
     * Uses Tender model to access BaseModel's getUserDateFormat method
     */
    private function getUserDateFormat()
    {
        $tenderModel = new Tender();
        return $tenderModel->getUserDateFormat();
    }

    /**
     * Get user's preferred time format (from BaseModel)
     * Uses Tender model to access BaseModel's getUserTimeFormat method
     */
    private function getUserTimeFormat()
    {
        $tenderModel = new Tender();
        return $tenderModel->getUserTimeFormat();
    }

    /**
     * Step 1: Store/Update basic tender information
     * Fields: project_id, title, closing_date, reference (auto-generated)
     */
    public function step1(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'nullable|integer|exists:tenders,id',
            'project_id' => 'required|integer|exists:projects,id',
            'title' => 'required|string|max:255',
            'closing_date' => 'required|date',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Get customer_id and workspace_id
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        $data = [
            'project_id' => $request->project_id,
            'title' => $request->title,
            'closing_date' => $request->closing_date,
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
            'status' => 'draft',
        ];
        if ($request->filled('id')) {
            // Update existing tender
            $tender = Tender::find($request->id);
            $tender->update($data);
            $message = 'Tender updated successfully';
        } else {
            // Create new tender (reference will be auto-generated in boot method)
            $tender = Tender::create($data);
            $message = 'Tender created successfully';
        }
        return $this->success($tender, $message);
    }

    /**
     * Step 2: Store/Update scope of work and line items
     * Fields: scope_of_work, line_items (array with description, unit, quantity)
     */
    public function step2(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'id' => 'required|integer|exists:tenders,id',
                'scope_of_work' => 'nullable|string',
                'line_items' => 'required|array|min:1',
                'line_items.*.description' => 'required|string',
                'line_items.*.unit' => 'required|string|max:50',
                'line_items.*.quantity' => 'required|integer|min:1',
            ],
            [
                'id.required' => 'Tender ID is required.',
                'id.integer' => 'Tender ID must be a valid number.',
                'id.exists' => 'The selected tender does not exist.',
        
                'line_items.required' => 'At least one line item is required.',
                'line_items.array' => 'Line items must be provided in a valid format.',
                'line_items.min' => 'Please add at least one line item.',
        
                'line_items.*.description.required' => 'Each line item must have a description.',
                'line_items.*.unit.required' => 'Each line item must have a unit.',
                'line_items.*.unit.max' => 'The unit field may not be greater than 50 characters.',
                'line_items.*.quantity.required' => 'Each line item must have a quantity.',
                'line_items.*.quantity.integer' => 'Quantity must be a valid number.',
                'line_items.*.quantity.min' => 'Quantity must be at least 1.',
            ]
        );
        
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        
        // Get customer_id and workspace_id and verify tender ownership
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        // Update scope of work
        $tender->scope_of_work = $request->scope_of_work;
        $tender->save();
        // Delete existing line items for this tender
        TenderLineItem::where('tender_id', $tender->id)->delete();
        // Create new line items
        $lineItems = [];
        foreach ($request->line_items as $item) {
            $lineItems[] = TenderLineItem::create([
                'tender_id' => $tender->id,
                'description' => $item['description'],
                'unit' => $item['unit'],
                'quantity' => $item['quantity'],
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ]);
        }
        // Reload tender with line items
        $tender->load('lineItems');
        return $this->success($tender, 'Tender line items saved successfully');
    }

    /**
     * Step 3: Store/Update tender settings
     * Fields: reminder_days_before, allow_late_submissions, allow_bidder_document_uploads
     */
    public function step3(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:tenders,id',
            'reminder_days_before' => 'nullable|integer|min:0',
            'allow_late_submissions' => 'nullable|boolean',
            'allow_bidder_document_uploads' => 'nullable|boolean',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        // Get customer_id and workspace_id and verify tender ownership
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }

        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->id);

        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $data = [];
        if ($request->has('reminder_days_before')) {
            $data['reminder_days_before'] = $request->reminder_days_before;
        }
        if ($request->has('allow_late_submissions')) {
            $data['allow_late_submissions'] = $request->allow_late_submissions;
        }
        if ($request->has('allow_bidder_document_uploads')) {
            $data['allow_bidder_document_uploads'] = $request->allow_bidder_document_uploads;
        }
        $tender->update($data);
        return $this->success($tender, 'Tender settings saved successfully');
    }



    /**
     * Get step data for tender form
     * Accepts step (step1, step2, step3) and tender_id (optional)
     * Returns data according to the step
     */
    public function getStepData(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'step' => 'required|string|in:step0,step1,step2',
            'tender_id' => 'nullable|integer|exists:tenders,id',
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $step = $request->step;
        $tenderId = $request->tender_id;

        // Apply customer workspace filter
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);

        // If tender_id is provided, verify ownership and get tender
        $tender = null;
        if ($tenderId) {
            $tender = $query->find($tenderId);
            if (!$tender) {
                return $this->message('Tender not found or you do not have access', 404);
            }
        }

        $data = [];

        switch ($step) {
            case 'step0':
                // Step 0: Basic information
                if ($tender) {
                    $data = [
                        'project_id' => $tender->project_id,
                        'title' => $tender->title,
                        'closing_date' => $tender->closing_date,
                        'reference' => $tender->reference,
                        'status' => $tender->status,
                    ];
                } else {
                    // Return empty structure for new tender
                    $data = [
                        'project_id' => null,
                        'title' => null,
                        'closing_date' => null,
                        'reference' => null,
                        'status' => 'draft',
                    ];
                }
                break;

            case 'step1':
                // Step 1: Scope of work and line items
                if ($tender) {
                    $tender->load('lineItems');
                    $data = [
                        'scope_of_work' => $tender->scope_of_work,
                        'line_items' => $tender->lineItems->map(function ($item) use ($tender) {
                            return [
                                'id' => $item->id,
                                'description' => $item->description,
                                'unit' => $item->unit,
                                'quantity' => $item->quantity,
                                'closing_date' => $tender->closing_date,
                            ];
                        })->toArray(),
                    ];
                } else {
                    // Return empty structure for new tender
                    $data = [
                        'scope_of_work' => null,
                        'line_items' => [],
                    ];
                }
                break;

            case 'step2':
                // Step 2: Settings
                if ($tender) {
                    $data = [
                        'reminder_days_before' => $tender->reminder_days_before,
                        'allow_late_submissions' => $tender->allow_late_submissions ?? false,
                        'allow_bidder_document_uploads' => $tender->allow_bidder_document_uploads ?? false,
                        'closing_date' => $tender->closing_date,
                    ];
                } else {
                    // Return default values for new tender
                    $data = [
                        'reminder_days_before' => null,
                        'allow_late_submissions' => false,
                        'allow_bidder_document_uploads' => false,
                        'closing_date' => $tender->closing_date,
                    ];
                }
                break;

            default:
                return $this->message('Invalid step provided', 400);
        }
        $data['step'] = $step;
        $data['tender_id'] = $tenderId;
        return $this->success($data, ucfirst($step) . ' data retrieved successfully');
    }

    /**
     * Get tender by ID
     */
    public function show($id)
    {
        $query = Tender::with(['lineItems', 'project', 'participants.user', 'rfis']);
        // Apply customer workspace filter
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($id);

        if (!$tender) {
            return $this->message('Tender not found', 404);
        }

        // Get all participants with their bid submission status and bid details
        $participantsWithBids = [];
        foreach ($tender->participants as $participant) {
            $userId = $participant->user_id;

            // Check if this participant has submitted a bid
            $hasSubmittedBid = TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $userId)
                ->where('status', 'submitted')
                ->exists();

            $participantData = [
                'id' => $participant->id,
                'user_id' => $participant->user_id,
                'user' => $participant->user,
                'invite_status' => $participant->invite_status,
                'is_assigned' => $participant->is_assigned ?? false,
                'assigned_at' => $participant->assigned_at,
                'has_submitted_bid' => $hasSubmittedBid,
                'bid_details' => null,
            ];

            // If participant has submitted a bid, get their bid details
            if ($hasSubmittedBid) {
                $bidPrices = TenderLineItemsBid::with('lineItem')
                    ->where('tender_id', $tender->id)
                    ->where('user_id', $userId)
                    ->where('status', 'submitted')
                    ->orderBy('tender_line_items_id')
                    ->get();

                // Format bid prices with line item details
                $formattedBidPrices = $bidPrices->map(function ($bid) {
                    $quantity = $bid->lineItem->quantity ?? 1;
                    $unitPrice = (float)$bid->bid_price;
                    $totalPrice = $unitPrice * $quantity;
                    
                    return [
                        'id' => $bid->id,
                        'line_item_id' => $bid->tender_line_items_id,
                        'line_item_description' => $bid->lineItem->description ?? null,
                        'line_item_unit' => $bid->lineItem->unit ?? null,
                        'line_item_quantity' => $quantity,
                        'bid_price' => $unitPrice,
                        'total_price' => $totalPrice,
                        'status' => $bid->status,
                        'total_base_bid' => $bid->total_base_bid,
                        'exclusions_clarifications' => $bid->exclusions_clarifications,
                        'is_late_submission' => $bid->is_late_submission ?? false,
                        'submitted_at' => ($rawSubmittedAt = $bid->getRawOriginal('submitted_at')) ? Carbon::parse($rawSubmittedAt)->format('Y-m-d H:i:s') : null,
                        'created_at' => $bid->created_at ? (is_string($bid->created_at) ? $bid->created_at : $bid->created_at->toDateTimeString()) : null,
                        'updated_at' => $bid->updated_at ? (is_string($bid->updated_at) ? $bid->updated_at : $bid->updated_at->toDateTimeString()) : null,
                    ];
                });

                // Get summary bid information
                $firstBid = $bidPrices->first();
                $entityId = $tender->id . '_' . $userId;

                // Get bid attachments
                $bidAttachments = TenderAttachment::where('entity_type', 'tender_bid')
                    ->where('entity_id', $entityId)
                    ->orderBy('created_at', 'desc')
                    ->get();

                // Parse submitted_at once for both date and time
                $rawSubmittedAt = $firstBid ? $firstBid->getRawOriginal('submitted_at') : null;
                $submittedAtCarbon = $rawSubmittedAt ? Carbon::parse($rawSubmittedAt) : null;
                
                // Get date and time formats from settings
                $dateFormat = $this->getUserDateFormat();
                $timeFormat = $this->getUserTimeFormat();
                
                $participantData['bid_details'] = [
                    'bid_prices' => $formattedBidPrices->toArray(),
                    'bid_summary' => $firstBid ? [
                        'total_base_bid' => $firstBid->total_base_bid ?? 0,
                        'exclusions_clarifications' => $firstBid->exclusions_clarifications ?? null,
                        'status' => $firstBid->status ?? null,
                        'is_late_submission' => $firstBid->is_late_submission ?? false,
                        'submitted_at' => $submittedAtCarbon ? $submittedAtCarbon->format($dateFormat) : null,
                        'submitted_time' => $submittedAtCarbon ? $submittedAtCarbon->format($timeFormat) : null,
                    ] : null,
                    'attachments' => $bidAttachments->map(function ($attachment) {
                        return [
                            'id' => $attachment->id,
                            'title' => $attachment->title,
                            'file_name' => $attachment->file_name,
                            'file_path' => $attachment->file_path,
                            'uploaded_by' => $attachment->uploaded_by,
                            'created_at' => $attachment->created_at ? (is_string($attachment->created_at) ? $attachment->created_at : $attachment->created_at->toDateTimeString()) : null,
                        ];
                    })->toArray(),
                ];
            }
            // Convert user object to array if it exists
            if (isset($participantData['user']) && is_object($participantData['user'])) {
                $participantData['user'] = $participantData['user']->toArray();
            }

            $participantsWithBids[] = $participantData;
        }
        // Convert tender to array and add our custom fields
        $tenderArray = $tender->toArray();
        $tenderArray['participants'] = $participantsWithBids;
        return $this->success($tenderArray, 'Tender retrieved successfully');
    }

    /**
     * List all tenders
     */
    public function index(Request $request)
    {
        // First, update tenders with passed closing dates (if not allowing late submissions)
        // Get tender IDs first, then query with raw date to avoid format issues
        $baseQuery = Tender::query();
        $baseQuery = $this->applyCustomerWorkspaceFilter($baseQuery);
        $tenderIds = $baseQuery->where('status', 'open')
            ->pluck('id');
        
        // Query with raw date field to get actual database value
        $tendersToUpdate = DB::table('tenders')
            ->whereIn('id', $tenderIds)
            ->whereNotNull('closing_date')
            ->select('id', 'closing_date', 'allow_late_submissions', 'status')
            ->get();
            
        foreach ($tendersToUpdate as $tenderRow) {
            // Get the full tender model for saving
            $tender = Tender::find($tenderRow->id);
            if (!$tender || !$tenderRow->closing_date) {
                continue; // Skip if no closing date
            }
            // Parse closing date - database should return YYYY-MM-DD format
            try {
                $closingDate = null;
                $rawClosingDate = $tenderRow->closing_date;
                
                // Database date column should be in YYYY-MM-DD format
                if (is_string($rawClosingDate)) {
                    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $rawClosingDate)) {
                        // Format: YYYY-MM-DD (standard database format)
                        $closingDate = Carbon::createFromFormat('Y-m-d', $rawClosingDate);
                    } elseif (preg_match('/^\d{2}-\d{2}-\d{4}$/', $rawClosingDate)) {
                        // Try both MM-DD-YYYY and DD-MM-YYYY formats
                        $dateMMDD = null;
                        $dateDDMM = null;
                        try {
                            $dateMMDD = Carbon::createFromFormat('m-d-Y', $rawClosingDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        try {
                            $dateDDMM = Carbon::createFromFormat('d-m-Y', $rawClosingDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        
                        // Use the date that makes sense (prefer past date if checking for past)
                        if ($dateMMDD && $dateDDMM) {
                            // If both parse successfully, prefer the one that's in the past (more likely correct)
                            $closingDate = $dateMMDD->isPast() ? $dateMMDD : ($dateDDMM->isPast() ? $dateDDMM : $dateMMDD);
                        } elseif ($dateMMDD) {
                            $closingDate = $dateMMDD;
                        } elseif ($dateDDMM) {
                            $closingDate = $dateDDMM;
                        } else {
                            throw new \Exception('Could not parse date in either MM-DD-YYYY or DD-MM-YYYY format');
                        }
                    } else {
                        // Try default parse
                        $closingDate = Carbon::parse($rawClosingDate);
                    }
                } else {
                    continue; // Skip if not a valid date format
                }
                
                if (!$closingDate) {
                    continue; // Skip if parsing failed
                }
            } catch (\Exception $e) {
                Log::warning('Failed to parse closing_date for tender', [
                    'tender_id' => $tenderRow->id,
                    'closing_date' => $rawClosingDate ?? $tenderRow->closing_date,
                    'error' => $e->getMessage()
                ]);
                continue; // Skip this tender if date parsing fails
            }
            
            // If closing date has passed, late submissions are not allowed (0 or null), and status is open, close the tender
            // allow_late_submissions: 0 = not allowed, 1 = allowed, null = not allowed (default)
            // Set closing date to end of day for proper comparison
            $closingDateEndOfDay = $closingDate->copy()->endOfDay();
            $isPast = $closingDateEndOfDay->isPast();
            $lateSubmissionsNotAllowed = ($tenderRow->allow_late_submissions == 0 || $tenderRow->allow_late_submissions === null);
            $isOpen = ($tenderRow->status == 'open');
            
            if ($isPast && $lateSubmissionsNotAllowed && $isOpen) {
                $tender->status = 'close';
                $tender->save();
            }
        }
        
        $query = Tender::with(['project', 'lineItems', 'participants.user']);
        // Apply customer workspace filter if needed
        $query = $this->applyCustomerWorkspaceFilter($query);
        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                    ->orWhere('reference', 'like', '%' . $search . '%')
                    ->orWhereHas('project', function ($subquery) use ($search) {
                        $subquery->where('title', 'like', '%' . $search . '%');
                    });
            });
        }
        // Filter by project
        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        $query->orderBy('created_at', 'desc');
        return $this->withCount($query, 'Tenders retrieved successfully');
    }

    /**
     * Delete tender
     */
    public function destroy($id)
    {
        $query = Tender::query();
        // Apply customer workspace filter
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($id);
        if (!$tender) {
            return $this->message('Tender not found', 404);
        }
        // Delete related records
        TenderLineItem::where('tender_id', $tender->id)->delete();
        $tender->delete();
        return $this->message('Tender deleted successfully', 200);
    }



    /**
     * Invite subcontractors to a tender
     * Accepts tender_id and subcontractor_ids (array or single integer)
     * Creates TenderParticipant records and sends invitation emails
     */
    public function inviteSubcontractors(Request $request)
    {
        // Normalize subcontractor_ids to always be an array
        $subcontractorIds = $request->subcontractor_ids;
        if (!is_array($subcontractorIds)) {
            // If single value, convert to array
            if (is_numeric($subcontractorIds)) {
                $subcontractorIds = [(int)$subcontractorIds];
            } else {
                return $this->message('subcontractor_ids must be an array or integer', 400);
            }
        }
        $validator = Validator::make([
            'tender_id' => $request->tender_id,
            'subcontractor_ids' => $subcontractorIds,
        ], [
            'tender_id' => 'required|integer|exists:tenders,id',
            'subcontractor_ids' => 'required|array|min:1',
            'subcontractor_ids.*' => 'required|integer|exists:users,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Get customer_id and workspace_id and verify tender ownership
        $ids = $this->getCustomerAndWorkspaceIds();
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->with(['project', 'lineItems'])->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        if ($tender->is_published != 1) {
            return $this->message('Tender is not published please first publish the tender', 400);
        }
        // Verify all subcontractors belong to this customer/workspace
        $subcontractors = $this->getSubcontractorsForCustomer();
        $validSubcontractorIds = $subcontractors->pluck('id')->toArray();
        $invalidIds = array_diff($subcontractorIds, $validSubcontractorIds);
        if (!empty($invalidIds)) {
            return $this->message('Some subcontractors are not associated with your customer/workspace', 400);
        }
        $invitedCount = 0;
        $emailSentCount = 0;
        $errors = [];
        foreach ($subcontractorIds as $subcontractorId) {
            // Check if participant already exists
            $existingParticipant = TenderParticipant::where('tender_id', $tender->id)
                ->where('user_id', $subcontractorId)
                ->first();
            if ($existingParticipant) {
                // Update invite status if already exists
                $existingParticipant->update([
                    'invite_status' => 'invited',
                ]);
                $invitedCount++;
            } else {
                // Create new participant record
                TenderParticipant::create([
                    'tender_id' => $tender->id,
                    'user_id' => $subcontractorId,
                    'invite_status' => 'invited',
                    'customer_id' => $ids['customer_id'],
                    'workspace_id' => $ids['workspace_id'],
                ]);
                $invitedCount++;
            }
            // Get subcontractor details
            $subcontractor = User::find($subcontractorId);
            if (!$subcontractor || !filter_var($subcontractor->email, FILTER_VALIDATE_EMAIL)) {
                $errors[] = "Invalid email for subcontractor ID: {$subcontractorId}";
                continue;
            }
            // Format closing date - handle multiple date formats
            $closingDate = 'N/A';
            if ($tender->closing_date) {
                try {
                    $rawDate = $tender->closing_date;
                    $parsedDate = null;
                    
                    // Try different date formats
                    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $rawDate)) {
                        // Format: YYYY-MM-DD (standard database format)
                        $parsedDate = Carbon::createFromFormat('Y-m-d', $rawDate);
                    } elseif (preg_match('/^\d{2}-\d{2}-\d{4}$/', $rawDate)) {
                        // Try both MM-DD-YYYY and DD-MM-YYYY formats
                        $dateMMDD = null;
                        $dateDDMM = null;
                        try {
                            $dateMMDD = Carbon::createFromFormat('m-d-Y', $rawDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        try {
                            $dateDDMM = Carbon::createFromFormat('d-m-Y', $rawDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        
                        // Use the one that parses successfully
                        if ($dateMMDD) {
                            $parsedDate = $dateMMDD;
                        } elseif ($dateDDMM) {
                            $parsedDate = $dateDDMM;
                        }
                    }
                    
                    // If still not parsed, try default parse
                    if (!$parsedDate) {
                        $parsedDate = Carbon::parse($rawDate);
                    }
                    
                    $closingDate = $parsedDate->format('d/m/Y');
                } catch (\Exception $e) {
                    // If parsing fails, use raw date or N/A
                    $closingDate = is_string($tender->closing_date) ? $tender->closing_date : 'N/A';
                }
            }
            // Get project name
            $projectName = $tender->project ? $tender->project->title : 'N/A';
            // Prepare email subject
            $subject = "Tender Invitation - {$tender->title}";
            $subject .= " | " . env('APP_NAME', 'WMS');
            // Prepare email content
            $emailContent = view('Emails.tender-invitation', [
                'subcontractor_name' => $subcontractor->name ?? 'there',
                'tender_title' => $tender->title,
                'tender_reference' => $tender->reference,
                'project_name' => $projectName,
                'closing_date' => $closingDate,
                'scope_of_work' => $tender->scope_of_work ?? '',
                'line_items' => $tender->lineItems ?? collect([]),
                'subject' => $subject,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ])->render();
            // Send email using EmailTrait
            $params = [
                'to' => $subcontractor->email,
                'subject' => $subject,
                'msg' => $emailContent,
            ];
            if ($this->SendInstantEmail($params)) {
                $emailSentCount++;
            } else {
                $errors[] = "Failed to send email to: {$subcontractor->email}";
            }
        }
        $response = [
            'invited_count' => $invitedCount,
            'email_sent_count' => $emailSentCount,
            'total_requested' => count($subcontractorIds),
        ];
        if (!empty($errors)) {
            $response['errors'] = $errors;
        }
        $message = "Successfully invited {$invitedCount} subcontractor(s). {$emailSentCount} email(s) sent.";
        if (!empty($errors)) {
            $message .= " Some errors occurred.";
        }
        return $this->success($response, $message);
    }

    /**
     * Send reminders to invited subcontractors for a tender
     * Accepts tender_id (required) and user_ids (optional array)
     * If user_ids is provided, sends reminders only to those specific users
     * If not provided, sends reminders to all invited participants
     */
    public function sendReminders(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'user_ids' => 'nullable|array',
            'user_ids.*' => 'required|integer|exists:users,id',
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        // Get customer_id and workspace_id and verify tender ownership
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }

        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->with(['project', 'lineItems'])->find($request->tender_id);

        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }

        // Get invited participants for this tender
        $participantsQuery = TenderParticipant::where('tender_id', $tender->id)
            ->where('invite_status', 'invited')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id']);

        // If specific user_ids provided, filter by them
        if ($request->filled('user_ids') && is_array($request->user_ids) && !empty($request->user_ids)) {
            $participantsQuery->whereIn('user_id', $request->user_ids);
        }

        $participants = $participantsQuery->with('user')->get();

        if ($participants->isEmpty()) {
            return $this->message('No invited participants found for this tender', 404);
        }

        $emailSentCount = 0;
        $errors = [];

        foreach ($participants as $participant) {
            $subcontractor = $participant->user;

            if (!$subcontractor || !filter_var($subcontractor->email, FILTER_VALIDATE_EMAIL)) {
                $errors[] = "Invalid email for subcontractor ID: {$participant->user_id}";
                continue;
            }

            // Format closing date - handle multiple date formats
            $closingDate = 'N/A';
            if ($tender->closing_date) {
                try {
                    $rawDate = $tender->closing_date;
                    $parsedDate = null;
                    
                    // Try different date formats
                    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $rawDate)) {
                        // Format: YYYY-MM-DD (standard database format)
                        $parsedDate = Carbon::createFromFormat('Y-m-d', $rawDate);
                    } elseif (preg_match('/^\d{2}-\d{2}-\d{4}$/', $rawDate)) {
                        // Try both MM-DD-YYYY and DD-MM-YYYY formats
                        $dateMMDD = null;
                        $dateDDMM = null;
                        try {
                            $dateMMDD = Carbon::createFromFormat('m-d-Y', $rawDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        try {
                            $dateDDMM = Carbon::createFromFormat('d-m-Y', $rawDate);
                        } catch (\Exception $e) {
                            // Ignore
                        }
                        
                        // Use the one that parses successfully
                        if ($dateMMDD) {
                            $parsedDate = $dateMMDD;
                        } elseif ($dateDDMM) {
                            $parsedDate = $dateDDMM;
                        }
                    }
                    
                    // If still not parsed, try default parse
                    if (!$parsedDate) {
                        $parsedDate = Carbon::parse($rawDate);
                    }
                    
                    $closingDate = $parsedDate->format('d/m/Y');
                } catch (\Exception $e) {
                    // If parsing fails, use raw date or N/A
                    $closingDate = is_string($tender->closing_date) ? $tender->closing_date : 'N/A';
                }
            }

            // Get project name
            $projectName = $tender->project ? $tender->project->title : 'N/A';

            // Prepare email subject with "Reminder" prefix
            $subject = "Reminder: Tender Invitation - {$tender->title}";
            $subject .= " | " . env('APP_NAME', 'WMS');

            // Prepare email content using the same template as invitation
            $emailContent = view('Emails.tender-invitation', [
                'subcontractor_name' => $subcontractor->name ?? 'there',
                'tender_title' => $tender->title,
                'tender_reference' => $tender->reference,
                'project_name' => $projectName,
                'closing_date' => $closingDate,
                'scope_of_work' => $tender->scope_of_work ?? '',
                'line_items' => $tender->lineItems ?? collect([]),
                'subject' => $subject,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ])->render();

            // Send email using EmailTrait
            $params = [
                'to' => $subcontractor->email,
                'subject' => $subject,
                'msg' => $emailContent,
            ];

            if ($this->SendInstantEmail($params)) {
                $emailSentCount++;
            } else {
                $errors[] = "Failed to send email to: {$subcontractor->email}";
            }
        }

        $response = [
            'reminder_sent_count' => $emailSentCount,
            'total_participants' => $participants->count(),
        ];

        if (!empty($errors)) {
            $response['errors'] = $errors;
        }

        $message = "Successfully sent {$emailSentCount} reminder(s) to invited subcontractor(s).";
        if (!empty($errors)) {
            $message .= " Some errors occurred.";
        }

        return $this->success($response, $message);
    }

    /**
     * Publish tender
     * Updates is_published to 1 and changes status from draft to open
     */
    public function publish(Request $request)
    {
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $updateData = [
            'is_published' => 1,
        ];
        if ($tender->status == 'draft') {
            $updateData['status'] = 'open';
        }
        $tender->update($updateData);
        return $this->success($tender, 'Tender published successfully');
    }



    /**
     * Upload document to tender (Admin only)
     */
    public function uploadTenderDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'title' => 'required|string|max:255',
            'file' => 'required|file|mimes:pdf,doc,docx,xlsx,jpg,jpeg,png|max:10240',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }

        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);

        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $uploadPath = 'tender_documents';
        $fullUploadPath = public_path($uploadPath);
        if (!file_exists($fullUploadPath)) {
            mkdir($fullUploadPath, 0755, true);
        }
        $file = $request->file('file');
        $filename = time() . '_' . uniqid() . '_' . $file->getClientOriginalName();
        $file->move($fullUploadPath, $filename);
        $filePath = $uploadPath . '/' . $filename;
        $attachment = TenderAttachment::create([
            'entity_type' => 'customer',
            'entity_id' => $tender->id,
            'title' => $request->title,
            'file_name' => $filename,
            'file_path' => $filePath,
            'uploaded_by' => Auth::id(),
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
        ]);
        return $this->success($attachment, 'Document uploaded successfully');
    }

    /**
     * Get all documents for a tender (Admin)
     */
    public function getTenderDocuments($tenderId)
    {
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $documents = TenderAttachment::where('entity_type', 'customer')
            ->where('entity_id', $tenderId)
            ->with('uploadedBy')
            ->orderBy('created_at', 'desc')
            ->get();

        return $this->success($documents, 'Tender documents retrieved successfully');
    }

    /**
     * Delete tender document (Admin)
     */
    public function deleteTenderDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'document_id' => 'required|integer|exists:tender_attachments,id',
            'tender_id' => 'required|integer|exists:tenders,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $document = TenderAttachment::where('id', $request->document_id)
            ->where('entity_type', 'customer')
            ->where('entity_id', $tender->id)
            ->first();
        if (!$document) {
            return $this->message('Document not found or you do not have access', 404);
        }
        // Delete file from storage
        if ($document->file_path && file_exists(public_path($document->file_path))) {
            unlink(public_path($document->file_path));
        }
        $document->delete();
        return $this->message('Document deleted successfully', 200);
    }

    /**
     * Get all bids for a tender (Admin)
     */
    public function getAllBidsForTender($tenderId)
    {
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        // Get all unique users who submitted bids
        $userIds = TenderLineItemsBid::where('tender_id', $tenderId)
            ->where('status', 'submitted')
            ->distinct()
            ->pluck('user_id');
        $bids = [];
        foreach ($userIds as $userId) {
            $user = User::find($userId);
            $lineItemBids = TenderLineItemsBid::where('tender_id', $tenderId)
                ->where('user_id', $userId)
                ->where('status', 'submitted')
                ->with('lineItem')
                ->get();
            $firstBid = $lineItemBids->first();
            $bids[] = [
                'user_id' => $userId,
                'user_name' => $user->name ?? 'N/A',
                'user_email' => $user->email ?? 'N/A',
                'total_base_bid' => $firstBid->total_base_bid ?? 0,
                'exclusions_clarifications' => $firstBid->exclusions_clarifications ?? null,
                'is_late_submission' => $firstBid->is_late_submission ?? false,
                'submitted_at' => $firstBid->submitted_at ?? null,
                'line_items' => $lineItemBids->map(function ($bid) {
                    $quantity = $bid->lineItem->quantity ?? 1;
                    $unitPrice = (float)$bid->bid_price;
                    $totalPrice = $unitPrice * $quantity;
                    
                    return [
                        'line_item_id' => $bid->tender_line_items_id,
                        'description' => $bid->lineItem->description ?? 'N/A',
                        'unit' => $bid->lineItem->unit ?? 'N/A',
                        'quantity' => $quantity,
                        'unit_price' => $unitPrice,
                        'bid_price' => $totalPrice,
                    ];
                }),
                'attachments' => TenderAttachment::where('entity_type', 'tender_bid')
                    ->where('entity_id', $tenderId . '_' . $userId)
                    ->get(),
            ];
        }
        return $this->success($bids, 'All bids retrieved successfully');
    }

    /**
     * Compare bids for a tender (Admin)
     * Returns structured data matching the bid comparison UI
     */
    public function compareBids(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->with(['project', 'lineItems'])->find($request->tender_id);

        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }

        // Get the assigned participant (if any)
        $assignedParticipant = TenderParticipant::where('tender_id', $tender->id)
            ->where('is_assigned', true)
            ->first();

        $assignedUserId = $assignedParticipant ? $assignedParticipant->user_id : null;

        // Get all users who have submitted bids for this tender
        $submittedBidUserIds = TenderLineItemsBid::where('tender_id', $tender->id)
            ->where('status', 'submitted')
            ->distinct()
            ->pluck('user_id')
            ->toArray();

        // If tender is assigned to someone who hasn't submitted a bid, include them too
        if ($assignedUserId && !in_array($assignedUserId, $submittedBidUserIds)) {
            $submittedBidUserIds[] = $assignedUserId;
        }

        if (count($submittedBidUserIds) < 1) {
            return $this->message('No submitted bids found for this tender', 404);
        }

        // Get all users with their bids
        $users = User::whereIn('id', $submittedBidUserIds)->get()->keyBy('id');

        // Prepare bid data for each user
        $bidsData = [];
        $chartData = [
            'labels' => [],
            'base_prices' => [],
            'adjustments' => [],
        ];

        $lowestBidderId = null;
        $lowestLevelledPrice = null;

        foreach ($submittedBidUserIds as $userId) {
            $user = $users->get($userId);
            if (!$user) continue;

            // Get all line item bids for this user
            $lineItemBids = TenderLineItemsBid::with('lineItem')
                ->where('tender_id', $tender->id)
                ->where('user_id', $userId)
                ->where('status', 'submitted')
                ->get();

            // If user has no bids but is assigned, still include them with empty bid data
            $isAssigned = ($userId == $assignedUserId);
            if ($lineItemBids->isEmpty() && !$isAssigned) continue;

            // Get the first bid to get summary data (total_base_bid, exclusions, etc.)
            $firstBid = $lineItemBids->first();

            // Use total_base_bid from database, or calculate from line items if not set
            // If no bids exist (assigned but no bid), set to 0
            if ($firstBid && $firstBid->total_base_bid) {
                $baseBidTotal = $firstBid->total_base_bid;
            } else {
                // Calculate total by multiplying unit price by quantity for each line item
                $baseBidTotal = $lineItemBids->sum(function ($bid) {
                    $quantity = $bid->lineItem->quantity ?? 1;
                    return (float)$bid->bid_price * $quantity;
                });
            }
            if (!$firstBid) {
                $baseBidTotal = 0;
            }

            // For now, adjustments are empty (can be added later via separate API)
            // Adjustments structure: [{description: string, amount: float}]
            $adjustments = []; // TODO: Fetch from adjustments table if exists
            $adjustmentsTotal = array_sum(array_column($adjustments, 'amount'));

            // Calculate final levelled price
            $finalLevelledPrice = $baseBidTotal + $adjustmentsTotal;

            // Track lowest bidder (only if they have submitted a bid)
            if (!$lineItemBids->isEmpty()) {
                if ($lowestLevelledPrice === null || $finalLevelledPrice < $lowestLevelledPrice) {
                    $lowestLevelledPrice = $finalLevelledPrice;
                    $lowestBidderId = $userId;
                }
            }

            // Prepare line items with prices
            // Format: description with quantity/unit, then price
            $lineItems = [];
            foreach ($tender->lineItems as $lineItem) {
                $bid = $lineItemBids->firstWhere('tender_line_items_id', $lineItem->id);

                // Format description with quantity and unit (e.g., "Sub-Mains Cabling (150 m)")
                $description = $lineItem->description;
                $quantityUnit = '';
                if ($lineItem->quantity && $lineItem->unit) {
                    $quantityUnit = " ({$lineItem->quantity} {$lineItem->unit})";
                }

                // Calculate total price: bid_price (unit price) * quantity
                $unitPrice = $bid ? (float)$bid->bid_price : 0;
                $totalPrice = $unitPrice * ($lineItem->quantity ?? 1);
                
                $lineItems[] = [
                    'line_item_id' => $lineItem->id,
                    'description' => $description,
                    'unit' => $lineItem->unit,
                    'quantity' => $lineItem->quantity,
                    'unit_price' => $unitPrice,
                    'price' => $totalPrice,
                ];
            }

            // If no bids exist, still create empty line items structure
            if ($lineItems === [] && $tender->lineItems->isNotEmpty()) {
                foreach ($tender->lineItems as $lineItem) {
                    $description = $lineItem->description;
                    $quantityUnit = '';
                    if ($lineItem->quantity && $lineItem->unit) {
                        $quantityUnit = " ({$lineItem->quantity} {$lineItem->unit})";
                    }
                    $lineItems[] = [
                        'line_item_id' => $lineItem->id,
                        'description' => $description . $quantityUnit,
                        'unit' => $lineItem->unit,
                        'quantity' => $lineItem->quantity,
                        'price' => 0,
                    ];
                }
            }

            // Check if this user is assigned
            $isAssigned = ($userId == $assignedUserId);

            // Get participant data for this user
            $participant = TenderParticipant::where('tender_id', $tender->id)
                ->where('user_id', $userId)
                ->first();

            // Build bid data
            $bidData = [
                'bidder_id' => $userId,
                'bidder_name' => $user->name ?? $user->company_name ?? 'N/A',
                'bidder_email' => $user->email ?? null,
                'bidder_company_name' => $user->company_name ?? null,
                'bidder_mobile' => $user->mobile_number ?? null,
                'is_assigned' => $isAssigned,
                'assigned_at' => $isAssigned && $assignedParticipant ? $assignedParticipant->assigned_at : null,
                'compliance_percentage' => 0, // TODO: Calculate compliance percentage if needed
                'line_items' => $lineItems,
                'base_bid_total' => (float)$baseBidTotal,
                'noted_exclusions' => $firstBid ? ($firstBid->exclusions_clarifications ?? 'No exclusions') : 'No exclusions',
                'adjustments' => $adjustments, // Array of {description, amount}
                'adjustments_total' => $adjustmentsTotal,
                'final_levelled_price' => (float)$finalLevelledPrice,
                'is_lowest_bid' => false, // Will be set after all bids are processed
                'has_submitted_bid' => !$lineItemBids->isEmpty(), // Flag to indicate if bid was submitted
            ];

            $bidsData[] = $bidData;

            // Add to chart data
            $chartData['labels'][] = $user->name ?? $user->company_name ?? 'N/A';
            $chartData['base_prices'][] = (float)$baseBidTotal;
            $chartData['adjustments'][] = (float)$adjustmentsTotal;
        }

        // Mark the lowest bidder
        foreach ($bidsData as &$bid) {
            if ($bid['bidder_id'] == $lowestBidderId) {
                $bid['is_lowest_bid'] = true;
            }
        }

        // Prepare recommendation
        $lowestBidder = $users->get($lowestBidderId);
        $recommendation = [
            'recommended_bidder_name' => $lowestBidder ? ($lowestBidder->name ?? $lowestBidder->company_name ?? 'N/A') : 'N/A',
            'recommended_bidder_levelled_total' => (float)$lowestLevelledPrice,
        ];

        // Format closing date
        $closingDate = $tender->closing_date ? date('Y-m-d', strtotime($tender->closing_date)) : null;

        // Get assigned user details if assigned
        $assignedUserData = null;
        if ($assignedUserId) {
            $assignedUser = $users->get($assignedUserId);
            if ($assignedUser) {
                $assignedUserData = [
                    'user_id' => $assignedUser->id,
                    'name' => $assignedUser->name ?? $assignedUser->company_name ?? 'N/A',
                    'email' => $assignedUser->email ?? null,
                    'company_name' => $assignedUser->company_name ?? null,
                    'mobile_number' => $assignedUser->mobile_number ?? null,
                    'assigned_at' => $assignedParticipant ? $assignedParticipant->assigned_at : null,
                ];
            }
        }

        // Build final response
        $response = [
            'tender_metadata' => [
                'tender_id' => $tender->id,
                'tender_name' => $tender->title,
                'tender_reference' => $tender->reference,
                'status' => $tender->status,
                'project_name' => $tender->project ? $tender->project->title : null,
                'closing_date' => $closingDate,
                'is_assigned' => $assignedUserId !== null,
                'assigned_user' => $assignedUserData,
            ],
            'chart_data' => $chartData,
            'recommendation' => $recommendation,
            'bids' => $bidsData,
        ];

        return $this->success($response, 'Bid comparison retrieved successfully');
    }


    /**
     * Get RFIs for a tender (Admin - shows all RFIs)
     */
    public function getRfisForTender($tenderId)
    {
        $tender = Tender::find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found', 404);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        // Admin sees all RFIs
        $rfis = TenderRfi::with('user')
            ->where('tender_id', $tenderId)
            ->orderBy('created_at', 'desc')
            ->get();
        return $this->success($rfis, 'RFIs retrieved successfully');
    }


    /**
     * Reply to RFI question (Admin)
     */
    public function replyToRfi(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'rfi_id' => 'required|integer|exists:tender_rfis,id',
            'answer' => 'required|string',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }

        $rfi = TenderRfi::with('tender')
            ->where('id', $request->rfi_id)
            ->whereHas('tender', function ($query) use ($ids) {
                $query->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id']);
            })
            ->first();

        if (!$rfi) {
            return $this->message('RFI not found or you do not have access', 404);
        }
        $rfi->update([
            'answer' => $request->answer,
            'status' => 'answered',
            'answered_at' => now(),
        ]);
        return $this->success($rfi, 'RFI answered successfully');
    }

    /**
     * Assign tender to subcontractor (Admin)
     */
    public function assignTender(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'user_id' => 'required|integer|exists:users,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $participant = TenderParticipant::where('tender_id', $tender->id)
            ->where('user_id', $request->user_id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();
        if (!$participant) {
            return $this->message('Subcontractor is not a participant in this tender', 404);
        }
        // Unassign all other participants
        TenderParticipant::where('tender_id', $tender->id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->update(['is_assigned' => false, 'assigned_at' => null]);
        // Assign to selected participant
        $participant->update([
            'is_assigned' => true,
            'assigned_at' => now(),
        ]);
        $tender->status = 'awarded';
        $tender->save();
        $subcontractorCompany = SubcontractorCompany::where('user_id', $request->user_id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();
        // Get existing project_ids or initialize empty array
        $projectIds = $subcontractorCompany ? ($subcontractorCompany->project_ids ?? []) : [];
        // Add tender's project_id to the array if it exists and is not already in the array
        if ($tender->project_id && !in_array($tender->project_id, $projectIds)) {
            $projectIds[] = $tender->project_id;
        }
        // Update or create subcontractor_companies record
        if ($subcontractorCompany) {
            $subcontractorCompany->update([
                'project_ids' => $projectIds,
            ]);
        } else {
            SubcontractorCompany::create([
                'user_id' => $request->user_id,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
                'project_ids' => $projectIds,
                'required_docs_status' => SubcontractorCompany::STATUS_NOT_UPLOADED,
                'del' => '0',
            ]);
        }
        return $this->success($participant, 'Tender assigned successfully');
    }
}
