<?php

namespace App\Http\Controllers\Traits;

use App\Models\Permission;
use App\Models\PermissionsModules;
use App\Models\PermissionsSubModules;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

trait PermissionTrait
{
    /**
     * Validation rules for permission creation
     *
     * @param Request $request
     * @param bool $isUpdate
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function permissionValidationRequest(Request $request, $isUpdate = false)
    {
        $rules = [
            'title' => 'required|string|max:255',
            'module_id' => 'nullable|integer',
            'module_type' => 'nullable|string|in:module,sub_module',
            'status' => 'nullable|string|in:active,inactive',
        ];

        if ($isUpdate) {
            $rules['id'] = 'required|integer|exists:permissions,id';
            // Make title optional for update, but if provided, check uniqueness
            $rules['title'] = 'sometimes|required|string|max:255';
        }

        $validator = Validator::make($request->all(), $rules);

        // Add custom validation after initial validation
        $validator->after(function ($validator) use ($request) {
            $moduleId = $request->input('module_id');
            $moduleType = $request->input('module_type');

            // If module_id is provided, module_type must also be provided
            if ($moduleId && !$moduleType) {
                $validator->errors()->add('module_type', 'module_type is required when module_id is provided');
                return;
            }

            // If module_type is provided, module_id must also be provided
            if ($moduleType && !$moduleId) {
                $validator->errors()->add('module_id', 'module_id is required when module_type is provided');
                return;
            }

            // Validate module_id exists in the correct table based on module_type
            if ($moduleId && $moduleType) {
                if ($moduleType === 'sub_module') {
                    $exists = PermissionsSubModules::where('id', $moduleId)
                        ->where('del', '0')
                        ->exists();
                    if (!$exists) {
                        $validator->errors()->add('module_id', 'The selected module id is invalid. (Sub-module not found)');
                    }
                } else {
                    // module_type is 'module' or not specified
                    $exists = PermissionsModules::where('id', $moduleId)
                        ->where('del', '0')
                        ->exists();
                    if (!$exists) {
                        $validator->errors()->add('module_id', 'The selected module id is invalid. (Module not found)');
                    }
                }
            }
        });

        return $validator;
    }

    /**
     * Validate module_id and module_type combination
     *
     * @param int|null $moduleId
     * @param string|null $moduleType
     * @return array ['valid' => bool, 'message' => string]
     */
    protected function validateModuleAndType($moduleId, $moduleType)
    {
        if (!$moduleId) {
            return ['valid' => true, 'message' => null];
        }

        if (!$moduleType) {
            return ['valid' => false, 'message' => 'module_type is required when module_id is provided'];
        }

        if ($moduleType === 'module') {
            $module = PermissionsModules::find($moduleId);
            if (!$module) {
                return ['valid' => false, 'message' => 'Module not found'];
            }
        } elseif ($moduleType === 'sub_module') {
            $subModule = PermissionsSubModules::find($moduleId);
            if (!$subModule) {
                return ['valid' => false, 'message' => 'Sub-module not found'];
            }
        }

        return ['valid' => true, 'message' => null];
    }

    /**
     * Get permission with module/sub-module relationship
     *
     * @param int $id
     * @return Permission|null
     */
    protected function getPermissionWithModule($id)
    {
        $permission = Permission::find($id);
        
        if ($permission && $permission->module_id) {
            $permission->load('moduleOrSubModule');
        }

        return $permission;
    }
}
